// DMAPLAY.C
// This program reads 8 bit PCM sound data from a file.
// The 8 bit sound data is then played back using DMA.
//
// The buffers that are used in this program are allocated 
// using cvxBufferAlloc. Each buffer is set to 32K. 

#include <stdio.h>
#include <dos.h>
#include <io.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define _BUFFER_SIZE  0x8000     // Size of each DMA buffer.
#define _BUFFER_COUNT 4          // Number of DMA buffers.
#define _PCM_FILE     "TEST1.V8" // File containing sound data.

extern _dmaInProgressFlag;          
extern _dmaDevice;

VOID main( VOID )
{
   HANDLE  pcmFileHandle;
   LPSTR   pcmBuffer[ _BUFFER_COUNT ];
   LONG    bytesAvailable;
   LONG    bytesRead;
   WORD    portAddress;
   BYTE    dmaRate;
   WORD    phase       = 0;
   WORD    repeatCount = 1;
   WORD    initError;
   WORD    i;


   // Allocate memory for buffers used to read 8 bit PCM
   // data from file.
   for( i = 0; i < _BUFFER_COUNT; i++ )
   {
      pcmBuffer[ i ] = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );

      if( pcmBuffer == _NULL )
      {
         printf( "ERROR : Cannot allocate memory!\n" );
         exit( 0 );
      }
   }


   // Initialize DMA. Setting each parameter to _AUTODETECT 
   // causes dmaInit to perform a search for the Port,
   // DMA channel, and IRQ setting respectively.
   initError = dmaInit( _AUTODETECT, _AUTODETECT, _AUTODETECT,
                        &portAddress );

   // If the variable _dmaDevice equals 0 then the DMA
   // sub-system was not initialized correctly.
   if( _dmaDevice == 0 )
      printf( "ERROR = %d : dmaInit failed\n", initError );
   else
   {
      // Open a file containing 8 bit PCM data.
      if( cvxFileOpen( ( LPSTR )_PCM_FILE, _OPEN_R_ONLY, &pcmFileHandle ) )
         printf( "ERROR : %s not found.\n", _PCM_FILE );
      else
      {
         // Prompt user to hit a key to begin playback.
         printf( "Hit any key to begin playback.\n" );
         
         // Wait for keystroke.
         while( !kbhit() );
         
         // Clear keystroke(s) from keyboard buffer.
         while( kbhit() )
            getch();
         
         // Notify user that playback has begun.
         printf( "Playing sound file ..." );
 
         // Get header information, if it exists, from file.
         cvxFileRead( pcmFileHandle, pcmBuffer[ phase ],
                      ( LONG )_HEADER_LENGTH, &bytesRead );

         // Get rate from header.
         dmaRate = pcmBuffer[ phase ][ _HEADER_RATE_OFFSET ];

         //  Loop here until we have queued the entire file.
         while( bytesRead )
         {
            // Fill buffer with sound data.
            cvxFileRead( pcmFileHandle, pcmBuffer[ phase ],
                         ( LONG )_BUFFER_SIZE, &bytesRead );

            // Insert buffer into DMA queue.
            dmaPlay( pcmBuffer[ phase ], bytesRead,
                     dmaRate, repeatCount );

            // Toggle phase so next buffer is accessed.
            if( phase == ( _BUFFER_COUNT - 1 ) )
               phase = 0;
            else
               phase++;

            // Loop until a spot in the queue opens up.
            while( dmaNumberInQueue() == _BUFFER_COUNT );

         }

         // Loop until DMA has completed.
         while( _dmaInProgressFlag );


         // Notify user that playback has completed.
         printf( " complete.\n" );
 
      }
   }

   // Clear all requests from the Play and Record Queue.
   if( dmaFlushQueue() )
      printf( "DMA uninit failure : could not flush queue.\n" );

   // Uninitialize the DMA system.   
   if( dmaUninit() )
      printf( "DMA uninit failure.\n" );

   // Close the sound file.
   cvxFileClose( pcmFileHandle );

   // Free memory used by 8 bit PCM buffer.
   for( i = ( _BUFFER_COUNT - 1 ); i != -1; i-- )
   {
      if( cvxBufferFree( pcmBuffer[ i ] ) )
         printf( "ERROR : Cannot free memory!\n" );
   }
}




