/**************************************************************************
*
*     filename :  ldshell.c
*
* COPYRIGHT (c) 1989, 1990, 1991  Matrox Electronic Systems Ltd.
* All Rights Reserved
*
***************************************************************************/

/* Matrox stuff:                                                          */
#include "imseries.h"            /* The OS being used.                    */
#include "imbind.h"              /* Stuff internal to Matrox.             */
#include "proto.h"               /* Function prototypes.                  */
#include "i_head.h"              /* User-known defines.                   */

/* Standard stuff:                                                        */
#ifndef I_UNIX
#include <stdlib.h>
#endif

#include <string.h>
#include <stdio.h>
#include <ctype.h>

#ifdef I_MS_DOS
#include <sys\types.h>
#include <sys\timeb.h>
#include <time.h>
#endif

/* COFF files definition */

typedef struct {
           unsigned short   magic;
           unsigned short   nheaders;
           unsigned long    timedate;
           unsigned long    symbt_off;
           unsigned long    nsym;
           unsigned short   optheadlen;
           unsigned short   flags;
           } COFF_HEADER;

typedef struct {
           char    name[8];
           unsigned long physaddr;
           unsigned long virtaddr;
           unsigned long sizebits;
           unsigned long data_off;
           unsigned long reloc_off;
           unsigned long lnum_off;
           unsigned short nreloc;
           unsigned short nlnum;
           unsigned long flags;
           } SECTION_HEADER;

typedef struct {
           unsigned short magic; /* magic number (0x0108) */
           unsigned short version;
           unsigned long textbits;
           unsigned long databits;
           unsigned long bssbits;
           unsigned long addrexec;
           unsigned long addrbss;
           unsigned long addrdata;
           } OPT_HEADER;

typedef struct {
            int   retcode      ;
   unsigned int   verify     :1;
   unsigned int   bssdownl   :1;
   unsigned int   coff_info  :1;
   unsigned int   cache_used :1;
   unsigned int   reset      :1;
   unsigned int   shell_ID   :1;
   unsigned int   no_mute    :1;
   } OPTIONS;

/* this enables to boot the GSP with  cache unused.  */
static OPTIONS     options;
static COFF_HEADER coffhead;
static OPT_HEADER  opthead;
static SECTION_HEADER secthead;

/* SHey 94/03/07 Changed window_base and window_end for WATCOM compiler support */
/* It was defined as 'unsigned long before' */
/* hardware locations used */
static unsigned char FAR * window_end;
static unsigned char FAR * window_base;
static short dummy;           /* use for dummy read */

/* handle to IM1280 */
static int fd;

/* buffer to read coff files */
static unsigned short multibuf[MULTIBUF_SIZE];

/*************************************************************************/
/* prototype */
#ifdef ANSI
static FILE _I_PTYPE *coff_parse( char _I_PTYPE *filename );
static void           coff_head(COFF_HEADER _I_PTYPE *coffh);
static int            coff_shell_ID( FILE _I_PTYPE *coff, long header_seek );
static int            coff_load( FILE _I_PTYPE *coff, long header_seek );

static short gsp_fix_endian( long device );
static short gsp_download(FILE _I_PTYPE *in, SECTION_HEADER _I_PTYPE *sh);
static short gsp_verify(FILE _I_PTYPE *in, SECTION_HEADER _I_PTYPE *sh);
static short gsp_exec(long addr);
static short gsp_halt(void);

static void gsp_set_LC_int( void );
static short fileseek(FILE _I_PTYPE *f, long pos);
static short fileread(FILE _I_PTYPE *f, char _I_PTYPE *buf, short size);
static void host_wait(void);
static void init_image(void);
static int ImagePresent(void);
static FILE _I_PTYPE *path_fopen ( char _I_PTYPE *name, char _I_PTYPE *path,
                                   char _I_PTYPE *type );
#ifdef I_FPGA_IN_SHELL
static double get_version ( FILE _I_PTYPE *in, SECTION_HEADER _I_PTYPE *sh );
#endif
#else
static FILE _I_PTYPE *coff_parse();
static void           coff_head();      
static int            coff_shell_ID();
static int            coff_load();

static short gsp_fix_endian();
static short gsp_halt();
static short gsp_download();
static short gsp_verify();
static short gsp_exec();

static void gsp_set_LC_int();
static short fileseek();
static short fileread();
static void host_wait();
static init_image();
static ImagePresent();
static FILE _I_PTYPE *path_fopen();
#ifdef I_FPGA_IN_SHELL
static double get_version ();
#endif
#endif

/****************/
#define DELAY 100L
/****************/

/***************************************************************************
*
*   Function: ioldshell()
*   Date:     91.04.18
*   Parameters: char *filename : name of shell file to download
*               short flags : download options specified by ANDing
*                             the following values:
*
*                 I_LOADSHELL_S: Skip download verification
*                 I_LOADSHELL_B: download .BSS section
*                 I_LOADSHELL_P: Print coff file information on sections size
*                 I_LOADSHELL_A: Alternate address: board is located at CC000H
*                                instead of C4000H (strap 12 is installed)
*                 I_LOADSHELL_2: Register Area 2 is used instead of Register
*                                Area 1  (strap 4 is not installed)
*                 I_LOADSHELL_C: boot with code Cache disabled
*                 I_LOADSHELL_R: do not Reset the board before download
*                 I_LOADSHELL_Q: Quiet. Do not print downloaded file header
*                 I_LOADSHELL_M: Mute all print messages
*
*   Description: Downloads specified file to GSP local RAM
*   Returns:
*                 0: No errors
*                 1: Unable to open device or memory channel 
*                 2: Unable to open file
*                 3: Board not present
*                 4: Invalid COFF file
*                 5: GSP not halted
*                 6: GSP exec failed
*                 7: Problem during GSP verify
*                 8: Data read failure
*
***************************************************************************/


#ifdef ANSI
short FTYPE ioldshell (char _I_PTYPE *filename, short flags)
#else
short FTYPE ioldshell (filename, flags)
char _I_PTYPE *filename;
short flags;
#endif
{
FILE _I_PTYPE *coff;
long header_seek;

unsigned short decode_RA2, low_map_swap;
unsigned long  device;

/* ************************** */
/* Parse flags into structure */
/* ************************** */

options.verify     = flags & I_LOADSHELL_S ? 0 : 1;
options.bssdownl   = flags & I_LOADSHELL_B ? 1 : 0;
options.coff_info  = flags & I_LOADSHELL_P ? 1 : 0;
options.cache_used = flags & I_LOADSHELL_C ? 0 : 1;
options.reset      = flags & I_LOADSHELL_R ? 0 : 1;
options.shell_ID   = flags & I_LOADSHELL_Q ? 0 : 1;
options.no_mute    = flags & I_LOADSHELL_M ? 0 : 1;
   
/* ************************** */
/* Select proper device.      */
/* ************************** */

low_map_swap = flags & I_LOADSHELL_A ? 1 : 0;
decode_RA2   = flags & I_LOADSHELL_2 ? 0 : 1;

if      (!decode_RA2 && !low_map_swap) device = I_LOUT_AOUT_LOW;
else if ( decode_RA2 && !low_map_swap) device = I_LOUT_AIN_LOW;
else if (!decode_RA2 &&  low_map_swap) device = I_LIN_AOUT_LOW;
else                                   device = I_LIN_AIN_LOW;

/* ************************** */
/* Parse file name parameter. */
/* ************************** */

if ( ( coff = coff_parse( filename ) ) != NULL ) {

#ifdef I_MS_DOS
/* ***************************************************** */
/* When PC boot, the ROMs may try accessing the IM-Board */
/* thus putting it in wrong-endian.  This fixes it !     */
/* ***************************************************** */
   if ( gsp_fix_endian( device )) {
	   return( options.retcode );
      }
#endif /* Fix endian on I_MS_DOS */

/* ************************** */
/* Open the IM-Series device. */
/* ************************** */

   if ( ( fd = ioopdevice( device, 0 ) ) != -1) {
      if (     iosldevice( fd )          != -1) {
         if (  ioslmemdevice( fd )       != -1) {
            if ( ImagePresent()               ) {

               /* ************************** */
               /* Start of HW messing about. */
               /* ************************** */

               /* SHey 94/03/07 WATCOM compiler support */
               /* It was defined as 'unsigned long before' */
               window_base = (unsigned char FAR *) &(i_glob.vrammap -> haa);
               window_end  = window_base + WINDOW_SIZE;

               /* reset IMAGE if no reset option is not used */
               if (options.reset)
                  iorsdevice( fd, I_COLD );

               /* min. initialization needed to download code */
               init_image();

               if ( gsp_halt() == (short) NULL ) 
					   {

                  if ( options.coff_info && options.no_mute )
                     coff_head ( &coffhead );

                  header_seek = ftell ( coff );

                  /* ************************** */
                  /* Print out Shell ID string. */
                  /* ************************** */
                  if ( options.shell_ID && options.no_mute ) 
							coff_shell_ID( coff, header_seek );

                  /* ************************** */
                  /* Load the GSP code.         */
                  /* ************************** */
                  if ( coff_load( coff, header_seek ) != (int) NULL ) 
						   {
   
                  	/* ***************************** */
                  	/* Ask driver for interrupt used */
                  	/* then program the LC board.    */
                  	/* ***************************** */

                  	gsp_set_LC_int();

                  	/* ************************** */
                  	/* Boot Shell running on GSP. */
                  	/* ************************** */

                  	if ( gsp_exec( opthead.addrexec ) == (short) NULL ) 
							   {
#ifdef I_UNIX
                     	sleep ( 1 );
#endif /* I_UNIX */

                      	/* Poll until Shell signals boot completion. */
                        while (( i_glob.imghra->hstctll ) & 0x10 );

                  		/* set registers, for drivers using interrupts */
#ifdef I_UNIX
                  		ioctl ( fd, IMG_SETREGS, NULL );
#endif /* I_UNIX */
                        } 
							else 
							   { /* GSP boot failed. */
								options.retcode = 6;
                        } /* Coff_load */

                     } /* Loading & verifying the GSP code. */
                  } 
					else 
					   { /* GSP Halt failed. */
                  options.retcode = 5;
                  if (options.no_mute) 
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
                     {printf("\nIM-Series: GSP does not send an halt ");
                      printf("acknowledge.  Download aborted.\n"); }
# else
                     {fprintf( stderr, "\nIM-Series: GSP does not send an halt ");
                      fprintf( stderr, "acknowledge.  Download aborted.\n"); }
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		               ;
#endif  /* I_NOPRINTF */
                     } /* GSP Halt */

                  } 
						else { /* Image not present */
               options.retcode = 3;
               if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
                  printf("\nIM-Series: board not present.\n");
# else
                  fprintf( stderr, "\nIM-Series: board not present.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
                  ;
#endif  /* I_NOPRINTF */
               } /* Checked for Image presence. */

            iorlmemdevice(fd);

            } else { /* ioslmemdevice() failed */
            options.retcode = 1;
            if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
               printf("\nIM-Series: GSP local memory not available.\n");
# else
               fprintf( stderr, "\nIM-Series: GSP local memory not available.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		         ;
#endif  /* I_NOPRINTF */
            } /* Attempt ioslmemdevice() */

         } else { /* iosldevice() failed */
         options.retcode = 1;
         if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
            printf("\nIM-Series: device unavailable.\n");
# else
            fprintf(stderr, "\nIM-Series: device unavailable.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
	 	      ;
#endif  /* I_NOPRINTF */
         } /* Attempt iosldevice() */

      iocldevice(fd);

      } else { /* ioopdevice() failed */
      options.retcode = 1;
      if (options.no_mute) 
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
         printf("\nIM-Series: unable to open device.\n");
# else
         fprintf(stderr, "\nIM-Series: unable to open device.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		   ;
#endif  /* I_NOPRINTF */
      } /* Attempt ioopdevice() */

   fclose(coff);
   } /* Parse name */

return( options.retcode );

} /* ioldshell () */

/*                        ####    ####   ######  ######
                         #    #  #    #  #       #
                         #       #    #  #####   #####
                         #       #    #  #       #
                         #    #  #    #  #       #
                          ####    ####   #       #
 */

/* ***************************************************************************
 * Function: static FILE _I_PTYPE *coff_parse( char _I_PTYPE *filename )
 *
 * Synopsis: Given a string, this function attempts to return a handle to
 *           a valid COFF file.
 *
 *           To do this, the function checks for the presence of an extension
 *           or adds one if missing. It then tries to open such a file first
 *           in the current directory, then according
 *
 * Returns:  Handle to an valid COOF file, NULL if no file opened.
 *           option.retcode will contain the exception code.
 */

#ifdef ANSI
static FILE _I_PTYPE *coff_parse( char _I_PTYPE *filename )
#else
static FILE _I_PTYPE *coff_parse( filename )
char _I_PTYPE *filename;
#endif
{
char name[80], _I_PTYPE *p;
FILE _I_PTYPE *coff;

p = name;
strcpy(p, filename);

/* Add .out extension when ommited */
while (*p == '.')
   p++;

if (!strchr(p, '.'))
   strcat(p, ".out");

if ( (( coff = (FILE _I_PTYPE *) (fopen ( name, "rb"))) == NULL )
#ifndef I_OS2
&& ((coff=path_fopen(name,(char _I_PTYPE *)getenv(SHELLPATHENVAR),"rb"))==NULL )
&& ((coff=path_fopen(name,SHELLPATHDEFAULT, "rb" )) == NULL )
#endif /* I_OS2 */
   ) {
   if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "\nIM-Series: unable to open file %s.\n", name );
# else
      fprintf( stderr, "\nIM-Series: unable to open file %s.\n", name );
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		;
#endif  /* I_NOPRINTF */
   options.retcode = 2;
   return( NULL );
   }

/* To be valid COFF, check : File header, optionnal header, magics */

if ( fileread ( coff, (char _I_PTYPE *)&coffhead, sizeof(coffhead) )
||   fileread ( coff, (char _I_PTYPE *)&opthead,  coffhead.optheadlen )
   ) {
   if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "\nIM-Series: error reading %s.\n", name );
# else
      fprintf( stderr, "\nIM-Series: error reading %s.\n", name );
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   options.retcode = 8;
   return( NULL );
   }

if (coffhead.magic!=0x0090||opthead.magic!=0x0108||coffhead.optheadlen!=28) {
   if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "\nIM-Series: %s not a COFF file.\n", name );
# else
      fprintf( stderr, "\nIM-Series: %s not a COFF file.\n", name );
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   options.retcode = 4;
   return( NULL );
   }

return( coff );

} /* coff_parse() */

/* ***************************************************************************
 * Function: static void coff_head(COFF_HEADER _I_PTYPE *coffh)
 *
 * Synopsis: Given a pointer to a COFF header, this function displays some
 *           pertinent informations.
 */

#ifdef ANSI
static void coff_head(COFF_HEADER _I_PTYPE *coffh)
#else
static void coff_head(coffh)
COFF_HEADER _I_PTYPE *coffh;
#endif
{
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
printf("File header:\n");
printf("Number of sections = %d\n", coffh->nheaders);
printf("Number of entries in symbol table = %ld\n", coffh->nsym);
printf("File header flags = 0x%04x\n\n", coffh->flags);
# else
fprintf(stderr, "File header:\n");
fprintf(stderr, "Number of sections = %d\n", coffh->nheaders);
fprintf(stderr, "Number of entries in symbol table = %ld\n", coffh->nsym);
fprintf(stderr, "File header flags = 0x%04x\n\n", coffh->flags);
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
coffh = coffh;       /* Avoid not used warning */
#endif  /* I_NOPRINTF */
} /* coff_head() */

/* ***************************************************************************
 * Function: static int coff_shell_ID( FILE _I_PTYPE *coff, long header_seek );
 *
 * Synopsis: This function display the IM-Series Shell ID string.
 */

#ifdef ANSI
static int  coff_shell_ID( FILE _I_PTYPE *coff, long header_seek )
#else
static int  coff_shell_ID( coff, header_seek )
FILE _I_PTYPE *coff;
long header_seek;
#endif
{
int  read_c;
unsigned short i = 1;

for ( i = 0; i < coffhead.nheaders; i++ ) {

   if ( fileseek ( coff, header_seek )  /* Load section header. */
     || fileread ( coff, (char _I_PTYPE *) &secthead, sizeof(secthead) ) ) {
      options.retcode = 8;
      return ( (int) NULL );
      }

   header_seek = ftell(coff);

   /* coff_shell_ID() specific. */
   if ( !strcmp( secthead.name,".text") ) {

      if (fileseek(coff, secthead.data_off)) {
         options.retcode = 8;
         return ( (int) NULL );
         }

#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "Downloaded file identification:\n");
# else
      fprintf( stderr, "Downloaded file identification:\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		;
#endif  /* I_NOPRINTF */

      while (( read_c = fgetc ( coff )) != '\0' ) {
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
         printf( "%c", (char)read_c );
# else
         fprintf( stderr, "%c", (char)read_c );
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  read_c = read_c;            /* Avoid not used warning */
#endif  /* I_NOPRINTF */
         }
      }
   /* coff_shell_ID specific. */
   }
return ( -1 );
} /* coff_shell_ID() */

/* ***************************************************************************
 * Function: static int coff_load( FILE _I_PTYPE *coff, long header_seek )
 *
 * Synopsis: This function display the IM-Series Shell ID string.
 */

#ifdef ANSI
static int  coff_load( FILE _I_PTYPE *coff, long header_seek )
#else
static int  coff_load( coff, header_seek )
FILE _I_PTYPE *coff;
long header_seek;
#endif
{
unsigned short i = 1;

for (i = 0; i < coffhead.nheaders; i++) {

   if ( fileseek ( coff, header_seek )  /* Load section header. */
     || fileread ( coff, (char _I_PTYPE *) &secthead, sizeof(secthead) ) ) {
      options.retcode = 8;
      return ( (int) NULL );
      }

   header_seek = ftell(coff);

   /* coff_load() specific. */
   if ( options.coff_info && options.no_mute ) {
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "Section %-8.8s (%6ld bytes, starts at 0x%08lX) ", 
               secthead.name, secthead.sizebits >> 3, secthead.physaddr);
# else
      fprintf( stderr, "Section %-8.8s (%6ld bytes, starts at 0x%08lX) ", 
               secthead.name, secthead.sizebits >> 3, secthead.physaddr);
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
      }

   /* download only initialized sections, all if bssdownl is set */
   if ( options.bssdownl || !(secthead.flags & 0x0083) ) {

      if ( options.coff_info && options.no_mute )
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
	 printf( "downloaded.\n");
# else
	 fprintf( stderr, "downloaded.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */

      if ( gsp_download ( coff, &secthead ) ) {
		   options.retcode = 6;
         return ( (int) NULL );
         }

      if ( options.verify && gsp_verify ( coff, &secthead ) ) {
         options.retcode = 7;
         return ( (int) NULL );
         }
      } else
      if ( options.coff_info && options.no_mute )
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
         printf("not downloaded.\n");
# else
         fprintf( stderr, "not downloaded.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   /* coff_load() specific. */
   } 
return( -1 );
} /* coff_load() */

/*
                         #####   #####  ######
                        #     # #     # #     #
                        #       #       #     #
                        #  ####  #####  ######
                        #     #       # #
                        #     # #     # #
                         #####   #####  #
 */

/* ***************************************************************************
 * Function: static int gsp_fix_endian( long device )
 *
 * Synopsis: Corrects the uncommon occasion where the GSP is set to big-endian
 *           by the boot ROM.
 *
 * Returns:  NULL if exception occured.
 *           option.retcode will contain the exception code.
 */

#ifdef ANSI
static short gsp_fix_endian( long device )
#else
static short gsp_fix_endian( device )
long device;
#endif
{
int handle;

/* Force open of device to update C-BINDING variables for RESET */
   handle = ioopdevice( device + 0x80000000L, 0 );  
/* Select the device */
   iosldevice (handle);
/* reset IMAGE in little-endian mode if the no-reset option is not used */
if (options.reset) {
   *(char far *) &(i_glob.imghra->hstctlh) = (char)0x80; /* Reset GSP */
   host_wait();                                    		/* wait */
   dummy = i_glob.imghra->hstctlh; /* dummy read to fix GSP io latency */

   /* second reset try for the cases were GSP is put in big endian mode */
   /* by PC boot sequence */
   *(char far *) &(i_glob.imghra->intenb) = (char)0x80;
   host_wait();                                    /* wait */
   dummy = i_glob.imghra->hstctlh; /* dummy read to fix GSP io latency */
   }

iocldevice(handle); /* Close device */
return( 0 );
} /* gsp_fix_endian() */

/* ***************************************************************************
 * Function: static short gsp_halt( void )
 *
 * Synopsis: Corrects the uncommon occasion where the GSP is set to big-endian
 *           by the boot ROM.
 *
 * Returns:  NULL if exception occured.
 *           option.retcode will contain the exception code.
 */

#ifdef ANSI
static short gsp_halt(void)
#else
static short gsp_halt()
#endif
{
/* halt and flush cache */
i_glob.vramhra -> hstctlh = HALT_HSTCTLH;

/* wait for halt acknowledge */
host_wait();

if ( i_glob.vramhra -> hstctlh != HALT_CHECK)
   return ( 1 ); /* Did not halt. */
else
   return ( 0 );

} /* gsp_halt() */

/* ***************************************************************************
 * Function: static short gsp_download( FILE _I_PTYPE *in, 
 *                                      SECTION_HEADER _I_PTYPE *sh
 *                                    )
 *
 * Synopsis: Given a pointer to a section whitin a COFF file, this code
 *           takes care of transfering data to the IM-Series GSP LRAM.
 */

#ifdef ANSI
static short gsp_download(FILE _I_PTYPE *in, SECTION_HEADER _I_PTYPE *sh)
#else
static short gsp_download(in, sh)
FILE _I_PTYPE *in;
SECTION_HEADER _I_PTYPE *sh;
#endif
{
unsigned long       totwords;

/* SHey 94/03/08 Changed ptr for WATCOM compiler support */
/* It was defined without FAR before */
unsigned short FAR  *hst_ptr, FAR *ptr;

unsigned short      page_value, nwords;

/* Condition for valid .out file: Data to be loaded should map to LRAM */
if ( (sh->physaddr < GSP_MIN_LOAD_ADDR) && (sh->sizebits > 0) ) {

   if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "\nIM-Series: load address is not in local RAM.\n");
# else
      fprintf( stderr, "\nIM-Series: load address is not in local RAM.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
	   ;
#endif  /* I_NOPRINTF */

   return ( 1 );
   }

/* if section is empty, do nothing */
if ( sh->sizebits == 0 )
   return ( 0 );

page_value = (unsigned short) ( (sh->physaddr>>16) & 0xfff );
i_glob.vramhra -> gcdapr = page_value;
ioFlushIndexInADR((short _I_HRDWTYPE *)&(i_glob.vramhra -> gcdapr));
                                               /* bit to byte */
hst_ptr = (unsigned short FAR *) ( window_base + ((sh->physaddr>>3L) & 0x1fffL) );

totwords = sh->sizebits >> 4;
if (fileseek(in, sh->data_off))
   return (8);

while (totwords > 0) {
   /* SHey 94/03/07 WATCOM compiler support */
   /* Calculate with char FAR * for hst_ptr: was casted in unsigned long before */
   nwords = ((short)(window_end + 1 - (unsigned char FAR *)hst_ptr) >> 1);
   nwords = totwords > (unsigned long) nwords ?
   nwords : (unsigned short)totwords;

   /* Read nwords from COFF file into a buffer. */
   if (fileread(in, (char _I_PTYPE *)multibuf, nwords << 1))
      return (8);

   totwords -= nwords;
   ptr = (unsigned short FAR *)multibuf;

   while ( nwords-- )
      *hst_ptr++ = *ptr++;

   /* Setup for next page: Increment page, reset pointer to window */
   i_glob.vramhra -> gcdapr = ++page_value;
   ioFlushIndexInADR((short _I_HRDWTYPE *)&(i_glob.vramhra -> gcdapr));
   hst_ptr = (unsigned short FAR *) window_base;
   }

return ( 0 );

} /* gsp_download() */

/* ***************************************************************************
 * Function: static short gsp_verify( FILE _I_PTYPE *in, 
 *                                      SECTION_HEADER _I_PTYPE *sh
 *                                    )
 *
 * Synopsis: Given a pointer to a section whitin a COFF file, this code
 *           takes care of verifying data transferred to the IM-Series GSP LRAM.
 */

#ifdef ANSI
static short gsp_verify(FILE _I_PTYPE *in, SECTION_HEADER _I_PTYPE *sh)
#else
static short gsp_verify(in, sh)
FILE *in;
SECTION_HEADER *sh;
#endif
{
unsigned long       totwords;
unsigned short FAR  *hst_ptr;
unsigned short      page_value, nwords;

unsigned short i;
short           ret_val = 0;
unsigned long physaddr;
unsigned short      val_read;

/* Condition for valid .out file: Data to be loaded should map to LRAM */
if ( (sh->physaddr < GSP_MIN_LOAD_ADDR) && (sh->sizebits > 0) ) {
   if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
      printf( "\nIM-Series: load address is not in local RAM.\n");
# else
      fprintf( stderr, "\nIM-Series: load address is not in local RAM.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   return ( 1 );
   }

/* if section is empty, do nothing */
if ( sh->sizebits == 0 )
   return (0);

page_value = (unsigned short) ( (sh->physaddr>>16) & 0xfff );
i_glob.vramhra -> gcdapr = page_value;
ioFlushIndexInADR((short _I_HRDWTYPE *)&(i_glob.vramhra -> gcdapr));
                                            /* bit to byte */
hst_ptr = (unsigned short FAR *) ( window_base + ((sh->physaddr>>3) & 0x1fff) );

totwords = sh->sizebits >> 4;
if (fileseek(in, sh->data_off))
   return (1);
physaddr = sh->physaddr;

while (totwords > 0) {
   /* SHey 94/03/07 WATCOM compiler support */
   /* Calculate with char FAR * for hst_ptr: was casted in unsigned long before */
   nwords = ((short)(window_end + 1 - (unsigned char FAR *)hst_ptr) >> 1);
   nwords = totwords > (unsigned long) nwords ?
   nwords : (unsigned short)totwords;

   if (fileread(in, (char _I_PTYPE *)multibuf, nwords << 1))
      return (1);

   for (i = 0; i < nwords; i++) {
      if ( (val_read = *hst_ptr) != multibuf[i] ) {
         if (options.no_mute) 
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
            printf("\nIM-Series: Bad transfer at %.8lxh: read: %.4xh, expected: %.4xh\n",
                   physaddr+((unsigned long)i << 4L), val_read, multibuf[i] );
# else
            fprintf( stderr, "\nIM-Series: Bad transfer at %.8lxh: read: %.4xh, expected: %.4xh\n",
                     physaddr+((unsigned long)i << 4L), val_read, multibuf[i] );
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  val_read = val_read;        /* Avoid not used warning */
#endif  /* I_NOPRINTF */
         ret_val = 1;
         }

      hst_ptr++;
      }

   physaddr+= ((unsigned long)nwords<<4L);

   /* adjust number of words to be checked, page register to next 8K, ... */
   totwords -= nwords;

   /* reset page value and host pointer */
   i_glob.vramhra -> gcdapr = ++page_value;
   ioFlushIndexInADR((short _I_HRDWTYPE *)&(i_glob.vramhra -> gcdapr));
   hst_ptr = (unsigned short FAR *) window_base;
   }
return (ret_val);

} /* gsp_verify() */

/* ***************************************************************************
 * Function: static short gsp_exec( long addr )
 *
 * Synopsis: This code boots the GSP code at a given LRAM address.
 */

#ifdef ANSI
static short gsp_exec(long addr)
#else
static short gsp_exec(addr)
long addr;
#endif
{
unsigned short FAR *reset_ptr;
short dummy = 0;
int      error_nmi;

#if ! ( defined( I_OS2) || defined( I_UNIX ) )
struct timeb time_loop0, time_loop1;
signed long difference; 
#endif /* I_OS2 */

/* top of memory.  (Max value in GCDAPR) */
i_glob.vramhra -> gcdapr = EXEC_VECTPAGE;
ioFlushIndexInADR((short _I_HRDWTYPE *)&(i_glob.vramhra -> gcdapr));

/* System configuration  word is reload at reset time */
reset_ptr = (unsigned short FAR *) RESET_VECTOR;
*(reset_ptr++) = (short) (addr + SYSTEM_CONF_WORD);
*reset_ptr = (short)(addr >> 16);

reset_ptr = (unsigned short FAR *) NMI_VECTOR;        /* NMI */
*(reset_ptr++) = (short) (addr + SYSTEM_CONF_WORD);
*reset_ptr = (short)(addr >> 16);

if ( options.cache_used ) /* disable halt and set NMI */
   i_glob.vramhra -> hstctlh = EXEC_HSTCTLH;
else
   /* disable halt and set NMI and disable cache */
   i_glob.vramhra -> hstctlh = 0x4000 | EXEC_HSTCTLH;

/* ******************************* */
/* This actually waits for a while */
/* this code his differrent on the */
/* many OS we support !            */
/* ******************************* */

#ifdef  I_MS_DOS
   error_nmi = 1;
   ftime( &time_loop0);

   while ( 1 ) {

      /* Exit when NMI gets set ! */

      dummy = i_glob.vramhra -> hstctlh;
      if ( (dummy & 0x0100) == 0x0000 ) {
         error_nmi = 0;
         break;
         }
        
      ftime( &time_loop1 );
      difference  = ((signed long)time_loop1.time)-((signed long)time_loop0.time);
      difference *= (signed long)1000L;
      difference += ((signed long)time_loop1.millitm)-((signed long)time_loop0.millitm);

      /* 18.2 =  1 seconde */
      if ( difference >= (signed long)DELAY )
         break; /* while (1) */

      } /* while (1) */
#endif /* MS_DOS */


#ifdef I_UNIX
   sleep(1);
   error_nmi = 1;
   dummy = i_glob.vramhra -> hstctlh;
   if ( (dummy & 0x0100) == 0x0000 )
      error_nmi = 0;
#endif  /* I_UNIX */

#ifdef I_OS2
   DosSleep(200L);
   error_nmi = 1;
   dummy = i_glob.vramhra -> hstctlh;
   if ( (dummy & 0x0100) == 0x0000 )
      error_nmi = 0;
#endif /* I_OS2 */

if ( error_nmi != 0 && options.no_mute ) {
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
   printf("\nIM-Series: Code had been transfered to local RAM" );
   printf(" , but GSP does not boot.\n");
   printf("             NMI bit does not reset itself.\n");
# else
   fprintf( stderr, "\nIM-Series: Code had been transfered to local RAM" );
   fprintf( stderr, " , but GSP does not boot.\n");
   fprintf( stderr, "             NMI bit does not reset itself.\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   return (1);
   }

return (0);

} /* gsp_exec() */
/* ***************************************************************************
 * Function: static void gsp_set_LC_int( void ) 
 *
 * Synopsis: Set the IRQ line used for interrupt on a LC.
 */

#ifdef ANSI
static void gsp_set_LC_int( void )
#else
static void gsp_set_LC_int() 
#endif
{
#ifdef I_OS2
USHORT int_no = 0;  /*  Interrupt level */
PUSHORT pint_no = &int_no;
#endif

#ifdef I_UNIX
short int_no = 0;
#endif

#ifdef I_UNIX
ioctl ( fd, IMG_IQINT, &int_no );
#endif /* I_UNIX */

#ifdef I_OS2
if ( DosDevIOCtl ( (PUSHORT) pint_no, (PUSHORT) NULL, (USHORT ) I_GET_INT,
                   (USHORT ) 0x80,    (HFILE  ) fd) != 0)

if (options.no_mute)
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
   printf("Unable to get int vector\n\n\n ");
# else
   fprintf( stderr, "Unable to get int vector\n\n\n ");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */

#endif /* I_OS2 */

#if (defined ( I_UNIX )) || (defined ( I_OS2 )) 
i_glob.imghra->tappt = int_no;
#endif
} /* gsp_set_LC_int() */

/*
                     #     #   ###    #####   #####
                     ##   ##    #    #     # #     #
                     # # # #    #    #       #
                     #  #  #    #     #####  #
                     #     #    #          # #
                     #     #    #    #     # #     #
                     #     #   ###    #####   #####
*/



#ifdef ANSI
static short fileseek(FILE _I_PTYPE *f, long pos)
#else
static short fileseek(f, pos)
FILE _I_PTYPE *f;
long pos;
#endif
{
if (fseek(f, pos, 0) != 0 ) {
   if ( options.no_mute ) 
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
	   printf("\nIM-Series: Data seek failure\n");
# else
	   fprintf( stderr, "\nIM-Series: Data seek failure\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */
   return (1);
   }
return (0);
} /* fileseek() */

#ifdef ANSI
static short fileread(FILE _I_PTYPE *f, char _I_PTYPE *buf, short size)
#else
static short fileread(f, buf, size)
FILE _I_PTYPE *f;
char _I_PTYPE *buf;
short size;
#endif
{
   if (fread(buf, size, 1, f) != 1) {
      if (options.no_mute) 
#ifndef  I_NOPRINTF
# ifdef I_INTEL		/* fprintf buggy under INTEL Vr4 */
		   printf("\nIM-Series: Data read failure\n");
# else
		   fprintf( stderr, "\nIM-Series: Data read failure\n");
# endif /* I_INTEL    */
#else	  /* !I_NOPRINTF */
		  ;
#endif  /* I_NOPRINTF */

      return (1);
      }
   return (0);
} /* fileread() */

#ifdef ANSI
static void host_wait(void)
#else
static void host_wait()
#endif
{
#ifdef  I_MS_DOS
   struct timeb time0, time1;
   signed long   difference;

   ftime(&time0);
   do {
      ftime(&time1);
      difference  = ((signed long)time1.time   ) - ((signed long)time0.time);
      difference *= (signed long)1000L;
      difference += ((signed long)time1.millitm) - ((signed long)time0.millitm);
      } while (difference <= (signed long) DELAY);  /* 18.2 =  1 seconde */
#endif

#ifdef  I_UNIX
   sleep(1);
#endif

#ifdef  I_OS2
   DosSleep(500L);
#endif
} /* host_wait() */


#ifdef ANSI
static void init_image(void)
#else
static init_image()
#endif
{
   /* set proper RR and RCA bus configuration */
   i_glob.vramhra -> config = CONFIG_WORD;

   /* set ADR-1024 Host access area */
   i_glob.vramhra -> hctrl |= INIT_HCTRL;

   /* MSB of mouse x & y are mapped on GSP fifo count. */
   i_glob.imghra->mouse_x = 0;
   i_glob.imghra->mouse_y = 0;

} /* init_image() */

/* ***************************************************************************
 * Function: static int ImagePresent( void )
 *
 * Synopsis: Tickles some HW location to make sure that an Image is present
 *           for the opened device.
 *
 * Returns:  NULL if board not present.
 *           option.retcode will contain the exception code.
 */

#ifdef ANSI
static int ImagePresent(void)
#else
static int ImagePresent()
#endif
{
   static unsigned short dummy;

   i_glob.vramhra -> hstadrh = TSTV1;            /* hstadrh */

   /* SHey 94/03/08 Changed to avoid truncation of ushort to short on WATCOM */
   i_glob.vramhra -> hstadrl = (short)TSTV2;            /* hstadrl */

   if ( (dummy = i_glob.vramhra -> hstadrh ) != TSTV1)
      return( (int) NULL );

   /* SHey 94/03/08 Changed to avoid truncation of ushort to short on WATCOM */
   i_glob.vramhra -> hstadrh = (short)TSTV2;            /* hstadrh */

   i_glob.vramhra -> hstadrl = TSTV1;            /* hstadrl */
   if ( (dummy = i_glob.vramhra -> hstadrh ) != TSTV2)
      return( (int) NULL );

   return( -1 );

} /* ImagePresent() */

#ifdef ANSI
static FILE _I_PTYPE *path_fopen ( char _I_PTYPE *name, char _I_PTYPE *path,
 char _I_PTYPE *type )
#else
static FILE _I_PTYPE *path_fopen (name, path, type)
char _I_PTYPE *name;
char _I_PTYPE *path;
char _I_PTYPE *type;
#endif
{
   char _I_PTYPE *p, pathname[256];
   FILE _I_PTYPE *fp;

   if ( path == NULL )
      return ( NULL );

   /* check name is specified with an absolute path */
#ifdef I_UNIX
   if (( *name == '.' ) || ( *name == '/' ) || ( *name == '~' ))
#else
   if (( *name == '.' ) || ( *name == '\\' ) || ( name[1] == ':' ))
#endif
      return ( NULL );

   /* check for an environment variable for the search path */
   while ( *path  != 0 ) {

      /* skip all separators */
      while ( *path == PATHSEPARATOR )
         path++;

      p = pathname;
      while (( *path != 0 ) && ( *path != PATHSEPARATOR ))
         *p++ = *path++;

#ifdef  I_UNIX
      if ( *( p - 1 ) != '/' )
         *p++ = '/';
#else
      if ( *( p - 1 ) != '\\' )
         *p++ = '\\';
#endif

      strcpy ( p, name );

      if ((( fp = fopen ( pathname, type )) != NULL ))
         return ( fp );
      }

   /* when every thing else failed... */
   return ( NULL );

} /* path_fopen () */


