/*                         PROGRAM SERVO
                      Code conversion by Eugene Klein

  Control R/C Model Servo Motors from a parallel printer port.
  A Servo Motor is a DC motor with feedback.
  All IBM / clone 8253 timer generates frequencies up to 1,193,180 Hertz
  For R/C model servos, pulse width should be 0.5ms to 2.5ms.
  Motors may oscillate due to delay increase cause by timer and other
   internal interrupt calls.  The effect is more noticeable on slower machines.
  Many thanks to my good friend, Mike Mench, who developed this program
*/


#include <dos.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <bios.h>
#include "My_TPU.h"

#define PIA_8255B    0x61;    // 8255 port B address
#define TIM_8253C    0x43;    // 8253 timer control register
#define TIM_8253T2   0x42;    // 8253 timer 2 register

const int ESC = 0x1B;
const unsigned int  Time_10_ms    =  0x2e9a;    // 10 milliseconds
const unsigned int  time_2_ms     =  0x0952;  // computed time for 2 millisecond delay
const unsigned int  time_1_ms     =  0x04a9;  // computed time for 1 millisecond delay
const unsigned int  time_half_ms  =  0x0255;  // computed time for half a millisecond delay

char Servo =0; // byte = unsigned char
char Ch;
unsigned char PBbits, r,y ,motor, Lpt_Num;
unsigned int LPTx, Delay_Value;

void Do_Pulse( unsigned int Delay_Value)
{
 /*
 Send three copies of bit at three-phase zero crossings
 Waits for next zero crossing and restarts timer
 Gets bit value from global bitval
 Mashes BL and DX

 NOTE: Jumps to labels L1, L2, L3, and L4 are not needed on XT machines
       XT machines do not have a pre-fetch.
 */
 asm      {
	   PUSH    bx
	   MOV     bx,Delay_Value       // load the contents of Delay_Value into Bx
	   PUSH    bp
	   MOV     bp,sp
	   CLI
	   IN      AL,0x61              //   get current port B bit
	   OR      AL,0x01              //   turn on timer gate
	   JMP     L1                   //   flush CPU's internal buffer
	   }
 L1:
 asm      {
	   OUT     0x61,AL              //   update timer gate bits
	   STI                          //   get 16 bit timer count
	   }
 Reget:
 asm      {
	   MOV     AL,0x80              //   latch Timer 2
	   JMP     L2                   //   flush CPU's internal buffer
	  }
 L2:
 asm      {
	   CLI
	   OUT     0x43,AL              //   load 8255 control register w/AL
	   JMP     L3                   //   flush CPU's internal buffer
	  }
 L3:
 asm      {
	   IN      AL,0x42              //   get LSB from timer
	   MOV     AH,AL
	   JMP     L4                   //   flush CPU's internal buffer
	   }
 L4:
 asm      {
	   IN      AL,0x42              //   get MSB from timer
	   STI
	   XCHG    AH,AL                //   swap bits around,
	   NOT     AX                   //   and flip bits to count upward
	   CMP     AX,BX                // compare until count ( in AX ) >=  Delay_Value
	   JB      Reget
	   POP     BP
	   POP     BX
	  }
}

void Init_Time(void)
{
/*
 Delay times measured in Timer-2 ticks These are found by computing
   (time/54.9ms)*64*1024 Delay required on AT after each I/O operation
 Sets up Timer2 Halts timer counting,
 sets FFFF reload value
 Sets Mode 2
 Clears "speaker enable" bit
 Does not start timer, leaves "gate speaker" bit low
*/

// NOTE: Jumps to labels L1, L2, and L3 are not needed on XT machines
//       XT machines do not have a pre-fetch.

 asm      {
	   CLI                           //  clear control bits
	   IN      AL,0x61               //  get current port B bits
	   AND     AL, 0xFC              //  turn off timer gate and speaker
	   JMP     L1                    //  flush CPU's internal buffer
	  }
 L1:
 asm      {
	   OUT     0x61,AL               //  load Port B bits
	   STI
	   CLI                           //  set Timer 2 to LSB/MSB mode 2
	   MOV     AL,0xB4               //  10110100B
	   OUT     0x43,AL               //  send control byte to 8253
	   MOV     AL,0xFF               //  set reload value to FFFF
	   JMP     L2                    //  flush CPU's internal buffer
	  }
 L2:
 asm      {
	   OUT     0x42,AL
	   JMP     L3                    //  flush CPU's internal buffer
	  }
 L3:
 asm      {
	   OUT     0x42,AL
	   STI
	  }
}


void Delay_Between(void)
/*
 set up for a ten millisecond delay between pulses sent to servo motor.
 may be eliminated if you are controlling only one motor
*/
{
 Init_Time();
 Do_Pulse(Time_10_ms);
}

void Pulse(unsigned char r, unsigned char motor,unsigned int Delay_Value)

 //converts r value, (0 to 255), to appropriate pulse width, (0.5 to 2.5 ms)

{
 Delay_Value = time_1_ms + ( 10 * r ) + 75;
 Init_Time();
 Do_Pulse(time_half_ms);
 outport(LPTx, motor);        // set printer port active bits HIGH
 Do_Pulse(Delay_Value);
 outport(LPTx,0x00);          // set printer port bits LOW
}

void main()
{
 char cmd[10] = " ";

 Lpt_Num = Select_Printer_Port();
 LPTx = Init_Printer_Port(Lpt_Num);
 clrscr();
 Ch = 0x32;
 r = 128;
 gotoxy(10,10);
 printf("Ch = %s  :   r value = %i",cmd,r);
 while(Ch != ESC)
 {
  if(kbhit())
  {
   Ch = getch();
   if(Ch==0)
    Ch=getch();
   if(Ch=='6'|| Ch=='M')
   {
    r++;
    strcpy(cmd,"right arrow");
   }
   if(Ch=='4' || Ch=='K')
   {
    r--;
    strcpy(cmd,"left arrow");
   }
   if(Ch=='8'|| Ch=='H')
   {
    if ( r > 250 )
     r = 255;
    else
     r += 5;
    strcpy(cmd,"Up arrow");
   }
   if(Ch=='2'|| Ch=='P')
   {
    if ( r > 4 )
     r -= 5;
    else
     r = 0;
    strcpy(cmd,"Down arrow");
   }
   if(Ch=='+')
   {
    r = 255;
    strcpy(cmd,"Plus");
   }
   if(Ch=='-')
   {
    r = 1;
    strcpy(cmd,"Minus");
   }
   clrscr();
   gotoxy(10,10);
   printf("Ch = %s  :   r value = %i",cmd,r);
  }
  for(y = 0;y<=2;y++)
  {
   motor = 0xff;
   Pulse(r,motor, Delay_Value);
   Delay_Between();
  }
 }
}

