{$D+,L+} { to be sure complete debug information is generated }

PROGRAM LCD_Demo;
{ Demonstrates how to control an LCD device from your parallel printer port.
  You can control any LCD display using an Hatachi 44780 controller chip.
  Demonstrates how to generate 5 by 7 graphics, 5 by 10 graphics, and use
  display's internal 5 by 10 character set.

  Written in Turbo Pascal 6.0.
}
USES DOS, CRT, My_Tpu;
TYPE String30 = String[80];
CONST
T = 1;
All_High = 4;     { 00000100, all control line outputs HIGH, inverted or not }
All_Low = 11;     { 00001011, all control line outputs LOW, inverted or not  }
Strobe = $FF01;   { 11110001,   strobe = D0, Base + 2, bit 1                 }
AutoFeed = $FF02; { 00000010, AutoFeed = D1, Base + 2, bit 2                 }

VAR                          Temp : INTEGER;
                        Tab, Data : BYTE;
                Use_Large_Letters : BOOLEAN;
                          Message : String30;
    RS, Lpt_Num, Lpt_Port_Address : WORD;
                             Regs : REGISTERS;

PROCEDURE Pulse_Strobe_Line(Strobe : WORD);
{ provide the LCD module with an ACTIVE LOW strobe                          }
{ in other words, pulse inverted strobe line so output goes HIGH, then LOW  }
BEGIN
Strobe := LO(Strobe);
{ pulse strobe line HIGH }
 PORT[  Lpt_Port_Address + 2] :=
   (255 - Strobe ) AND PORT[ Lpt_Port_Address + 2];
{ pulse strobe line LOW }
PORT[  Lpt_Port_Address + 2] := Strobe OR PORT[  Lpt_Port_Address + 2];
delay(t);  { turn strobe LOW }
END; { pulse strobe }

PROCEDURE Set_Control_Bit(Lpt_Port_Address, Bit : WORD);
BEGIN
IF Hi(Bit) = $FF THEN { process inverted bit }
  BEGIN
  Bit := LO(Bit);
  PORT[Lpt_Port_Address + 2] := All_Low; DELAY(T); { clear I/O lines,00000100 }
  IF Bit = 2 THEN       { turn RS, ( = D1, inverted AutoFeed line) HIGH       }
    Temp := PORT[ Lpt_Port_Address + 2] AND (255 - Bit)
  ELSE               { turn RS, ( = D1, inverted AutoFeed line ) LOW          }
    Temp := PORT[ Lpt_Port_Address + 2] OR Bit;
  PORT[Lpt_Port_Address + 2] := Temp; delay(t);      { read value at Lpt1 + 2 }
  END; { if HI bit }
END; { send control bit }

PROCEDURE Transmit_Byte(Lpt_Port_Address, Bit : WORD; Data : Byte);
{ RS = AutoFeed = Bit 2 of Base + 2.  This is an INVERTED bit                 }
VAR Temp : BYTE;
BEGIN
Set_Control_Bit(Lpt_Port_Address, RS);
PORT[ Lpt_Port_Address ] := Data; delay(t);       { put data on data lines }
Pulse_Strobe_Line(Strobe);
END;

PROCEDURE Write_Character(VAR Lpt_Port_Address : WORD; Data : BYTE);
{ RS must be high to send LCD display character data }
BEGIN
RS := $FF02;
Transmit_Byte( Lpt_Port_Address, RS, Data);
END;

PROCEDURE Write_Instruction(VAR Lpt_Port_Address : WORD; Data : BYTE);
{ RS must be LOW to to send LCD display control instruction }
VAR E : INTEGER;
BEGIN
RS := $FF00;
Transmit_Byte(Lpt_Port_Address, Rs, Data);
END;

PROCEDURE Init_LCD_Module(Lpt_Port_Address : WORD);
{ Transmit Control Word three times to tell module you are initializing set-up.}
{ Just follow sequence set forth in LCD's data manual: to initialize module:   }
{   -send control number three times with small delay between transmissions.   }
{   -Delay provids time for LCD chip to digest information.  Delay is          }
{     usually not required for PASCAL, C, or BASIC compiled programs, BUT,     }
{   -Delay is a must for Embedded controller / LCD module control.             }
VAR E : INTEGER;
BEGIN
PORT[ Lpt_Port_Address + 2] := All_Low; Delay(T); { set all control lines LOW  }
Data := $34;     { 30hex = 48dec,  set for 8-bit interface, one display line   }
PORT[ Lpt_Port_Address] := Data; delay(T);
FOR E := 1 to 3 do                             { set 8 bit interface 3 times   }
  BEGIN
  Pulse_Strobe_Line(Strobe);
  DELAY(T);
  END; { for e }
END; { init LCD module }

PROCEDURE Set_Desired_Configuration(Large_Letters : BOOLEAN);
{ many more options are listed in LCD data manual                              }
BEGIN
{ Set 8 bit interface, 1 line display, & 5 by 10 character set = $34           }
  IF Large_Letters = TRUE THEN Data := $34 ELSE Data := $30;
  Write_Instruction(Lpt_Port_Address, DATA);
{ display on, cursor off  }
  Data := $0C; Write_Instruction(Lpt_Port_Address, DATA);
{ clear display }
  Data := $01; Write_Instruction(Lpt_Port_Address, DATA);
{ move cursor to beginning of line }  { for beginning of 2ed line, use $40     }
  Data := $02; Write_Instruction(Lpt_Port_Address, DATA);
{ increment addr counter after each write, to keep cursor moving to right }
  Data := $06; Write_Instruction(Lpt_Port_Address, DATA);
END; { set configuration }

PROCEDURE Send_Message(Tab : BYTE; Message : String30);
{   For each byte of data sent:                                              }
{   1) Tell module you are going to write to CG memory                       }
{   2) Send data to that address                                             }

VAR E : BYTE;
BEGIN
{ clear display }
   Data := $01; Write_Instruction(Lpt_Port_Address, DATA);
{ move cursor to beginning of line }
   Data := $02; Write_Instruction(Lpt_Port_Address, DATA);
{ tab to center text }
   Data := 127 + Tab; Write_Instruction(Lpt_Port_Address, DATA);
{ send message }
FOR E := 1 to LENGTH(Message)  DO
   BEGIN
   Data := ORD(Message[E]);
   Write_Character(Lpt_Port_Address, Data);
   END;
Delay(1500);
END; { send message }

 PROCEDURE Load_User_Defined_Character(Redefined_Character : String30);
{ Send redefined character info to CG memory }
VAR       Instr, E : BYTE;
BEGIN
Instr := $40;
FOR E := 1 TO   LENGTH(Redefined_Character)  DO
   BEGIN
{ tell LCD module you are going to write to CG memory, at address Instr }
   Write_Instruction(Lpt_Port_Address, Instr);
{ write character definition BYTE to CG memory }
   Data := ORD(Redefined_Character[E]);
   Write_Character(Lpt_Port_Address, DATA);
   Instr := Instr + 1;
   END; {  for e }
END; { load user defined character }

PROCEDURE Define_And_Load_Airplane;
{ Define airplane.  This is one case where BASIC had a better idea.  DATA
  statement support would be very useful here.
}
VAR Instr, E : BYTE;
Char_0, Char_A, Char_B, Char_C : String[10];
    Airplane : String[80];
BEGIN
Char_C := ConCat( CHAR($00),
                  CHAR($00),
                  CHAR($00),
                  CHAR($1C),   { 000* **00 }
                  CHAR($1F),   { 000* **** }
                  CHAR($00),
                  CHAR($00),
                  CHAR($00));

 char_B  := Concat(CHAR($10),  { 000* 0000 }
                   CHAR($0C),  { 0000 **00 }
                   Char($06),  { 0000 *0*0 }
                   CHAR($1F),  { 000* **** }
                   char($1F),  { 000* **** }
                   Char($06),  { 0000 *0*0 }
                   Char($0C),  { 0000 **OO }
                   char($10)); { 000* 0000 }

 char_A  := Concat(CHAR($18),  { 000* *000 }
                   CHAR($18),  { 000* *000 }
                   CHAR($18),  { 000* *000 }
                   CHAR($1F),  { 000* **** }
                   CHAR($1F),  { 000* **** }
                   Char($00),
                   CHAR($00),
                   CHAR($00));

 Char_0  := Concat(CHAR($00),
                   CHAR($00),
                   CHAR($00),
                   CHAR($00),
                   CHAR($00),
                   Char($00),
                   CHAR($00),
                   char($00));

Airplane :=  concat(Char_0 + Char_A + Char_B + Char_C);

Load_User_Defined_Character(Airplane);
END; { define airplane }

PROCEDURE Define_And_Load_Happy_Face;
VAR Happy_Face : String[80];

BEGIN
{ this is the closest thing to BASIC's DATA statement }
Happy_Face := CONCAT(
     { char A }      CHAR($03),  { 0000 00** }
                     CHAR($08),  { 0000 *000 }
                     CHAR($10),  { 000* 0000 }
                     CHAR($12),  { 000* 00*0 }
                     CHAR($10),  { 000* 0000 }
                     CHAR($10),  { 000* 0000 }
                     CHAR($14),  { 000* 0*00 }
                     CHAR($03),  { 0000 00** }
                     CHAR($08),  { 0000 *000 }
                     CHAR($03),  { 0000 00** }
                     CHAR($00),  { 0000 0000 }

     { spaces }      CHAR($00),CHAR($00), CHAR($00), CHAR($00), CHAR($00),

     { char B }      CHAR($18),  { 000* *000 }
                     CHAR($02),  { 0000 00*0 }
                     CHAR($01),  { 0000 000* }
                     CHAR($09),  { 0000 *00* }
                     CHAR($01),  { 0000 000* }
                     CHAR($01),  { 0000 000* }
                     CHAR($05),  { 0000 0*0* }
                     CHAR($18),  { 000* *000 }
                     CHAR($02),  { 0000 00*0 }
                     CHAR($18),  { 000* *000 }
                     CHAR($00)); { 0000 0000 }

Load_User_Defined_Character(Happy_Face);
END; { load happy face }

BEGIN  { main }
Select_Printer_Port( Lpt_Num, Lpt_Port_Address); { in My_Tpu }
Init_Printer_Port(Lpt_Num, Lpt_Port_Address);
  REPEAT
  writeln('-=[ 5 by 10 characters with user defined happy face ]=-');
  Init_LCD_Module(Lpt_Port_Address);
  Use_large_Letters := TRUE;
  Set_Desired_Configuration(Use_Large_Letters);
  Define_And_Load_Happy_Face;
  Message := CONCAT(CHAR($01), CHAR($02),        { happy smile  pattern }
             ' HELLO! ',                         { ASCII characters     }
             CHAR($01), CHAR($02));              { happy smile  pattern }
  Tab := 3; Send_Message(Tab, Message);
  IF KEYPRESSED THEN EXIT;
    WRITELN('-=[ send ASCII using 5 by 10 character set ]=-');
  Init_LCD_Module(Lpt_Port_Address);
  Use_Large_Letters := TRUE;
  Set_Desired_Configuration(Use_Large_Letters);
  Message := ConCat('Paul Ber', CHAR($E7), 'sman');
  Tab := 2; Send_Message(Tab, Message);
  WRITELN;
  Writeln('-=[ 5 by 7 characters with user defined airplane ]=-');
  Init_LCD_Module(Lpt_Port_Address);
  Use_large_Letters := FALSE;
  Set_Desired_Configuration(Use_Large_Letters);
  Define_And_Load_Airplane;
  Message :=  CHAR($01) + CHAR($02) + CHAR($03)   + { airplane characters  }
              ' HELLO! ' +                          { normal ASCII letters }
              CHAR($01) + CHAR($02) + char($03);    { airplane characters  }
  Tab := 2; Send_Message(Tab, Message);
  IF KEYPRESSED THEN EXIT;
  WRITELN('-=[ send ASCII using 5 by 10 character set ]=-');
  Init_LCD_Module(Lpt_Port_Address);
  Use_Large_Letters := TRUE;
  Set_Desired_Configuration(Use_Large_Letters);
  Message := ConCat('Paul Ber', CHAR($E7), 'sman');
  Tab := 2; Send_Message(Tab, Message);
  WRITELN;
  UNTIL KEYPRESSED;
END. { main }

