{*********************************************************************}
{*                            D I S K M O N P                        *}
{*-------------------------------------------------------------------*}
{*    Task           : DISKMON is a small disk monitor based on      *}
{*                     the functions of the BIOS diskette            *}
{*                     interrupt 13(h)                               *}
{*-------------------------------------------------------------------*}
{*    Author          : MICHAEL TISCHER                              *}
{*    developed on    : 7/9/87                                       *}
{*    last update     : 5/19/89                                      *}
{*********************************************************************}

program DISKMON;


Uses Crt, Dos;                            { adds Crt and Dos features }

type BufferTyp = array [1..1] of char;
     FormatTyp = record         { BIOS requires this information for  }
                   Track,                          { every sector of  }
                   Side,                    { a track to be formatted }
                   Sector,
                   Length  : byte;
                 end;

var ErrCode     : byte;       { Error status after diskette operation }
    Command     : string[1];             { Command input by the user  }
    FTyp,                     { Diskette type for formatting function }
    DriveNum,                               { Number of current drive }
    Side       : integer;       { Number of the current diskette side }
    Dummy       : integer;                           { Dummy variable }
    AT          : boolean;                   { is the computer an AT? }

{*********************************************************************}
{* RESETDISK: Reset for all attached disk drives                     *}
{* Input    : none                                                   *}
{* Output   : error status                                           *}
{*********************************************************************}

function ResetDisk : integer;

var Regs : Registers;          { Register variable for interrupt call }

begin
 Regs.ah := 0;                       { Function number for reset is 0 }
 intr($13, Regs);                          { Call BIOS disk interrupt }
 ResetDisk := Regs.ah;                            { Read error status }
end;

{*********************************************************************}
{* GETDISKSTATUS: reads the error status                             *}
{* Input : none                                                      *}
{* Output : the error status                                         *}
{*********************************************************************}

function GetDiskStatus : integer;

var Regs : Registers;          { Register variable for interrupt call }

begin
 Regs.ah := 1;                { Function number for error status is 1 }
 intr($13, Regs);                          { Call BIOS disk interrupt }
 GetDiskStatus := Regs.ah;                        { Read error status }
end;

{*********************************************************************}
{* READSECTORS: read a certain number of sectors                     *}
{* Input : see below                                                 *}
{* Output : error status                                             *}
{*********************************************************************}

function ReadSectors(DriveNum,              { Disk drive for reading  }
                     Side,           { Side or read/write head number }
                     Track,                       { track to be read  }
                     Sector,            { The first sector to be read }
                     Number,          { Number of sectors to be read  }
                     SegAdr,          { Segment address of the buffer }
                     OfsAdr : integer; { Offset address of the buffer }
                     var NumRead : integer) : integer;

var Regs : Registers;         { Register variable for interrupt call }

begin
 Regs.ah := 2;                    { Function number for reading is 2 }
 Regs.al := Number;              { Set number of sectors for reading }
 Regs.dh := Side;                                  { Set side number }
 Regs.dl := DriveNum;                             { Set drive number }
 Regs.ch := Track;                                { Set track number }
 Regs.cl := Sector;                              { Set sector number }
 Regs.es := SegAdr;                             { Set buffer address }
 Regs.bx := OfsAdr;
 intr($13, Regs);                         { Call BIOS disk interrupt }
 NumRead := Regs.al;                       { Number of sectors read  }
 ReadSectors := Regs.ah;                         { Read error status }
end;

{*********************************************************************}
{* WRITESECTORS: Write a certain number of sectors                   *}
{* Input  : see below                                                *}
{* Output : error status                                             *}
{*********************************************************************}

function WriteSectors(DriveNum,                         { Disk drive  }
                      Side,                 { Side or read/write head }
                      Track,                   { Track to be written  }
                      Sector,            { First sector to be written }
                      Number,      { Number of sectors to be written  }
                      SegAdr,         { Segment address of the buffer }
                      OfsAdr : integer;{ Offset address of the buffer }
                      var NumWritten : integer) : integer;

var Regs : Registers;          { Register variable for interrupt call }

begin
 Regs.ah := 3;                     { Function number for writing is 3 }
 Regs.al := Number;                { Set number of sectors to be read }
 Regs.dh := Side;                                   { Set side number }
 Regs.dl := DriveNum;                             { Set drive number  }
 Regs.ch := Track;                                 { Set track number }
 Regs.cl := Sector;                               { Set sector number }
 Regs.es := SegAdr;                              { Set buffer address }
 Regs.bx := OfsAdr;
 intr($13, Regs);                          { Call BIOS disk interrupt }
 NumWritten := Regs.al;                  { Number of sectors written  }
 WriteSectors := Regs.ah;                         { Read error status }
end;

{*********************************************************************}
{* SETDASD: must be called for an AT before formatting to indicate   *}
{*          if it should be formatted with 360 KB                    *}
{*          or with 1.2 MB                                           *}
{* Input : see below                                                 *}
{* Output : none                                                     *}
{*********************************************************************}

procedure SetDasd(DiskFormat : integer);

var Regs : Registers;          { Register variable for interrupt call }

begin
 Regs.ah := $17;                                    { Function number }
 Regs.al := DiskFormat;                                      { Format }
 Regs.dl := DriveNum;                                  { Drive number }
 intr($13, Regs);                          { Call BIOS disk interrupt }
end;

{*********************************************************************}
{* FORMATTRACK:  formats a track                                     *}
{* Input : see below                                                 *}
{* Output : the error status                                         *}
{*********************************************************************}

function FormatTrack(DriveNum,             { Number of the disk drive }
                      Side,                { the side or head number  }
                      Track,                  { Track to be formatted }
                      Number,      { Number of sectors in this track  }
                      Bytes   :  integer) :  integer;

var Regs : Registers;        { Register variable for interrupt call }
    DataField : array [1..15] of FormatTyp;     { maximum 15 sectors }
    LoopCnt   : integer;                              { Loop counter }

begin
 for LoopCnt := 1 to Number do             { Create sector descriptor }
  begin
   DataField[LoopCnt].Track   := Track;         { Number of the track }
   DataField[LoopCnt].Side  := Side;                  { Diskette side }
   DataField[LoopCnt].Sector := LoopCnt;       { Number of the sector }
   DataField[LoopCnt].Length := Bytes;{ Number of bytes in the sector }
  end;
 Regs.ah := 5;
 Regs.al := Number;                         { Function number, Number }
 Regs.es := seg(DataField[1]);         { Address of the data field in }
 Regs.bx := ofs(DataField[1]);                  { registers ES and BX }
 Regs.dh := Side;                                       { Side number }
 Regs.dl := DriveNum;                                    { Drive unit }
 Regs.ch := Track;                                 { Set track number }
 intr($13, Regs);                          { Call BIOS disk interrupt }
 FormatTrack := Regs.ah;                          { Read error status }
end;

{*********************************************************************}
{* WRITEERROR: Output error message according to error value         *}
{* Input : the error number                                          *}
{* Output : none                                                     *}
{*********************************************************************}

procedure WriteError(ErrorNumber : integer);

begin
 case ErrorNumber of
  $00 : ;                                         { 0 means no error  }
  $01 : writeln('ERROR: Invalid function number');
  $02 : writeln('ERROR: Address marking not found');
  $03 : writeln('ERROR: Write attempt on protected disk');
  $04 : writeln('ERROR: Sector not found');
  $06 : writeln('ERROR: Diskette changed');
  $08 : writeln('ERROR: DMA overrun');
  $09 : writeln('ERROR: Data transmission beyond segment border');
  $10 : writeln('ERROR: Read error');
  $20 : writeln('ERROR: Disk controller error');
  $40 : writeln('ERROR: Track not found');
  $80 : writeln('ERROR: Time out error');
  else  writeln('ERROR: Error ',ErrorNumber,' unknown');
 end;
 if (ErrorNumber <> 0) then ErrorNumber:=ResetDisk; { Reset performed }
end;

{*********************************************************************}
{* CONSTANTS:  Input of the two constants and                        *}
{*             diskette side or head number, as well as diskette     *}
{*             type for AT                                           *}
{* Input : none                                                      *}
{* Output : none                                                     *}
{*********************************************************************}

procedure Constants;

begin
 write('Unit-Number (0-3)   : ');
 readln(DriveNum);                             { Read unit number }
 write('Diskette side (0 or 1): ');
 readln(Side);                                 { Read head number }
 if AT then                                         { only for AT }
  begin
   writeln('Format-Parameter:');
   writeln('  1 = 320/360-KB-Diskette in 320/360-KB drive');
   writeln('  2 = 320/360-KB-Diskette in 1.2-MB drive');
   write('  3 = 1.2-MB-Diskette in 1.2-MB-drive -- Please input: ');
   readln(FTyp)
  end;
end;

{*********************************************************************}
{* HELP: Display help text on the screen                             *}
{* Input : none                                                      *}
{* Output : none                                                     *}
{*********************************************************************}

procedure Help;

begin
 writeln(#13#10'C O M M A N D   O V E R V I E W');
 writeln('-------------------------------');
 writeln('e = End');
 writeln('g = Get (Read)');
 writeln('s = Sector fill');
 writeln('r = Reset');
 writeln('f = Format');
 writeln('c = Constants');
 writeln('? = Help'#13#10);
end;

{*********************************************************************}
{* READSEC: Read a diskette sector and display it on the screen      *}
{* Input : none                                                      *}
{* Output : none                                                     *}
{*********************************************************************}

procedure READSEC;

var DataBuffer : array [1..512] of char;       { the characters read }
    Track,                            { the track from which to read }
    Sector : integer;                            { Sector to be read }

begin
 write('Track : ');
 readln(Track);                           { Read track from keyboard }
 write('Sector: ');
 readln(Sector);                    { Read sector from the keyboard }
 ErrCode := ReadSectors(DriveNum, Side, Track, Sector, 1,
                         seg(DataBuffer), ofs(DataBuffer), Dummy);
 if (ErrCode = 0) then              { Error occurred during reading? }
  begin
   write('----------------------------------------'+
         '----------------------------------------');
   for Dummy:=1 to 512 do                 { output the 512 characters }
    begin
     case DataBuffer[Dummy] of
      #00 : write('<NUL>');    { treat control characters separately }
      #07 : write('<BEL>');
      #08 : write('<BS>');
      #09 : write('<TAB>');
      #10 : write('<LF>');
      #13 : write('<CR>');
      #27 : write('<ESC>');
      else  write(DataBuffer[Dummy]);   { output normal character }
     end;
    end;
   write(#13#10'----------------------------------------'+
               '----------------------------------------');
  end
  else WriteError(ErrCode);                    { output error message }
end;

{*********************************************************************}
{* FORMATIT:    format a certain number of sectors of a              *}
{*              track with 512 bytes each                            *}
{* Input : none                                                      *}
{* Output : none                                                     *}
{*********************************************************************}

procedure FormatIt;

var Track,                                  { Track to be formatted  }
    Sector : integer;                            { Number of sectors }

begin
  write('Track : ');
  readln(Track);                { Read number of tracks from keyboard }
  write('Sector: ');
  readln(Sector);         { Read number of sectors from the keyboard  }
  if AT then SetDasd(FTyp);               { if AT then diskette type }
  WriteError(FormatTrack(DriveNum, Side, Track, Sector, 2));
end;

{*********************************************************************}
{* FILLSECTOR: Fill a sector with a character                        *}
{* Input : none                                                      *}
{* Output : none                                                     *}
{*********************************************************************}

procedure FillSector;

var DataBuffer : array [1..512] of char;  { Content of sector to fill }
    LoopCnt,                                           { Loop counter }
    Track,                    { Track in which the sector is located  }
    Sector : integer;                 { Number of sector to be filled }
    FillChar : char;                             { the fill character }

begin
 write('Track    : ');
 readln(Track);                            { Read track from keyboard }
 write('Sector   : ');
 readln(Sector);                         { Read sector from keyboard  }
 write('Character: ');
 readln(FillChar);        { Read the fill character from the keyboard }
 for LoopCnt := 1 to 512 do
  DataBuffer[LoopCnt] := FillChar;      { Fill buffer with characters }
 WriteError(WriteSectors(DriveNum, Side, Track, Sector, 1,
                          seg(DataBuffer), ofs(DataBuffer), Dummy));
end;

{*********************************************************************}
{**                           MAIN PROGRAM                          **}
{*********************************************************************}

begin
 clrscr;                                               { Clear screen }
 textbackground(7);                                { light background }
 textcolor(0);                                      { dark characters }
 writeln(' DISKMON: (c) 1987 by Michael Tischer   '+       { Headline }
         '                              ? = Help ');
 textbackground(0);                                 { dark background }
 textcolor(7);                                           { light text }
 window(1, 2, 80, 25);    { only first line does not belong to window }
 DriveNum := 0;                        { Indicate unit 0 as constant  }
 Side := 0;                                      { Side 0 as constant }
 FTyp := 3;                          { 1.2 MB diskette in 1.2 MB unit }
 if mem[$F000:$FFFE] = $FC then AT := true            { test if AT or }
                           else AT := false;               { PC or XT }
 WriteError(ResetDisk);                               { perform Reset }
 repeat
  repeat
   write('DISKMON>');                                 { output prompt }
   readln(Command);                      { Read command from keyboard }
  until (Command <> '');
  case (Command [1]) of
  '?'  : Help;                                   {? display help text }
  'r' : WriteError(ResetDisk);                       {r perform reset }
  's' : FillSector;                                  {s fill a sector }
  'f' : FormatIt;                                   {f format a track }
  'g' : READSEC;                                     {g read a sector }
  'c': Constants;                                  {c input constants }
 else if Command <> 'e' then writeln('unknown command');
  end;
 until (Command = 'e');                                {e end program }
end.
