
;
;                            A20 Test
;
;
;    Check if A20 is toggled by keyboard controller
;
;    For all computers equipped with a 8042 keyboard controller
;    (not for PC/XT).
;
;    (c) Copyright 1994, 1996  Frank van Gilluwe
;    All Rights Reserved.
;
;    V2.00 - Improve A20 detection routine, CHECK_A20
;            Switched to smaller cpu386 detector

include undocpc.inc


cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stack1

keyview         proc near

message1 db     CR, LF
         db     '  A20 CONTROL TEST                       v2.00 (c) 1994, 1996 FVG', CR, LF
         db     '  ', CR, LF
         db     '  Tests the A20 gate state and keyboard controller status and the', CR, LF
         db     '  ability of the controller to change the A20 gate.   When A20 is', CR, LF
         db     '  enabled, memory is accessable above 1MB.    Otherwise, when A20', CR, LF
         db     '  is disabled, the system acts like an 8088.', CR, LF
         db     CR, LF
         db     '   Test Results ', CR, LF
         db     CR, LF
         db     '                          Verified     Keyboard Controller ', CR, LF
         db     '  A20 Test                A20 State   Returned State     D0 Value', CR, LF
         db     '              ', CR, LF
         db     '$'
msg1     db     '  Initial State           $'
msg2_ena db     '  D1 Command enable       $'
msg3_dis db     '  D1 Command disable      $'
msg4_dis db     '  DD Command disable      $'
msg5_ena db     '  DF Command enable       $'

         db     CR, LF, '$'

enabled  db     'enabled     $'
disabled db     'disabled    $'

hexline  db     '       '
hexout   db     'xxh', CR, LF, '$'


v86msg   db     CR, LF
         db     '  System is in V86 mode, and changing the A20 state would cause a'
         db     CR, LF
         db     '  a system crash.  State change tests skipped.'
         db     CR, LF, CR, LF, '$'


init_state dw   0
init_port  db   0                  ; state from I/O port
temp_port  db   0                  ; temp state from controller D0 command
cpu_val    db   0                  ; current CPU state

old_int6_seg dw 0                  ; temp storage for old int 6
old_int6_off dw 0                  ;  vector (bad opcode)
badoff       dw 0                  ; temp return offset if bad offset
                                   ;  interrupt 6 called

;

start:
        push    cs
        pop     ds
        OUTMSG  message1           ; display initial message

        call    check_A20          ; test A20 state
        mov     [init_state], cx   ; save state
        OUTMSG  msg1               ; initial test
        call    display_states     ; display results from test & keyboard
        mov     al, [temp_port]
        mov     [init_port], al    ; save initial value

        call    far ptr cpu386
        mov     [cpu_val], al
        call    cpumode
        cmp     al, 1              ; V86/protected mode ?
        jbe     mode_ok            ; jump if not
        OUTMSG  v86msg             ; can't safely test in V86 mode
        jmp     exit2

; now use keyboard coontroller commands to disabled/enable

mode_ok:
        cmp     [init_state], 0    ; was it disabled ?
        je      a20_skip1
        OUTMSG  msg3_dis
        jmp     a20_skip2
a20_skip1:
        OUTMSG  msg2_ena

a20_skip2:
        mov     al, [init_port]
        and     al, 0FDh           ; clear A20 bit
        cmp     [init_state], 0    ; was it disabled ?
        jne     a20_skip3          ; jump if not
        or      al, 2              ; switch to enabled
a20_skip3:
        cli
        mov     bl, 0D1h           ; command to change controller port
        push    ax
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display & exit if error (ah=1)
        pop     ax
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display

        call    boop               ; slight delay
        call    display_states

; now try to return to original state

        cmp     [init_state], 0    ; was it disabled ?
        jne     a20_skip4
        OUTMSG  msg3_dis
        jmp     a20_skip5
a20_skip4:
        OUTMSG  msg2_ena

a20_skip5:
        mov     al, [init_port]
        and     al, 0FDh           ; clear A20 bit
        cmp     [init_state], 0    ; was it disabled ?
        je      a20_skip6          ; jump if so
        or      al, 2              ; switch to enabled
a20_skip6:
        cli
        mov     bl, 0D1h           ; command to change controller port
        push    ax
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display & exit if error (ah=1)
        pop     ax
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display

        call    boop               ; slight delay
        call    display_states

; now try the direct keyboard controller commands for enable & disable

        cmp     [init_state], 0    ; was it disabled ?
        je      a20_skip7
        OUTMSG  msg4_dis
        jmp     a20_skip8
a20_skip7:
        OUTMSG  msg5_ena

a20_skip8:
        mov     bl, 0DDh           ; assumed disable
        cmp     [init_state], 0    ; was it disabled ?
        jne     a20_skip9          ; jump if not
        mov     bl, 0DFh           ; enable
a20_skip9:
        cli
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display & exit if error (ah=1)
        call    boop               ; slight delay
        call    display_states

; now try to return to original state

        cmp     [init_state], 0    ; was it disabled ?
        jne     a20_skip10
        OUTMSG  msg4_dis
        jmp     a20_skip11
a20_skip10:
        OUTMSG  msg5_ena

a20_skip11:
        mov     bl, 0DDh           ; assumed disable
        cmp     [init_state], 0    ; was it disabled ?
        je      a20_skip12         ; jump if so
        mov     bl, 0DFh           ; enable
a20_skip12:
        cli
        mov     bl, 0D1h           ; command to change controller port
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display & exit if error (ah=1)
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display

        call    boop               ; slight delay
        call    display_states

exit:
exit2:
        mov     ah, 4Ch
        int     21h                ; exit
keyview endp

;
;                         SUBROUTINES



;
;    Display A20 State
;       Check the A20 state directly and from the keyboard
;       controller and display the results
;
;       Called with:    nothing
;
;       Regs Used:      al

display_states  proc    near
        call    check_A20          ; test A20 state
        jcxz    now_disabled
        OUTMSG  enabled
        jmp     disp_skip1
now_disabled:
        OUTMSG  disabled

; read state from keyboard controller

disp_skip1:
        cli                        ; disable interrupts
        mov     bl, 0D0h           ; read port command
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display & exit if error (ah=1)
        call    keyboard_read      ; read value
        call    error_read         ; display & exit if error (ah=1)
        sti
        mov     [temp_port], al    ; save controller port state
        push    ax
        mov     bx, offset hexout
        call    hex
        pop     ax
        test    al, 2              ; A20 enabled or disabled ?
        jz      disp_skip2         ; bit 1=0 if disabled
        OUTMSG  enabled            ; bit 1=1 if enabled
        jmp     disp_skip3

disp_skip2:
        OUTMSG  disabled           ; disabled

disp_skip3:
        OUTMSG  hexline
        ret
display_states endp


;
;    KEYBOARD_READ
;       read a byte from the keyboard into al (port 60h).
;
;       Called with:    nothing
;
;       Returns:        if ah=0, al=byte read from keyboard
;                       if ah=1, no byte ready after timeout
;
;       Regs Used:      al

keyboard_read   proc    near
        push    cx
        push    dx

        xor     cx, cx             ; counter for timeout (64K)
key_read_loop:
        in      al, 64h            ; keyboard controller status
        IODELAY
        test    al, 1              ; is a data byte ready ?
        jnz     key_read_ready     ; jump if now ready to read
        loop    key_read_loop

        mov     ah, 1              ; return status - bad
        jmp     key_read_exit

key_read_ready:
        push    cx                 ; delay routine needed for
        mov     cx, 16             ;   MCA Type 1 controller.
key_read_delay:                    ;   Insures a 7 uS or longer
        IODELAY
        loop    key_read_delay     ;   Assumes CPU is 80486,
        pop     cx                 ;   66Mhz or slower

        in      al, 60h            ; now read the byte
        IODELAY
        xor     ah, ah             ; return status - ok
key_read_exit:
        pop     dx
        pop     cx
        ret
keyboard_read   endp


;
;    KEYBOARD_WRITE
;       Send byte AL to the keyboard controller (port 60h).
;       Assumes no BIOS interrupt 9 handler active.
;
;       If the routine times out due to the buffer remaining
;       full, ah is non-zero.
;
;       Called with:    al = byte to send
;                       ds = cs
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
;
;       Regs Used:      ax

keyboard_write  proc    near
        push    cx
        push    dx

        mov     dl, al             ; save data for keyboard
        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop:
        in      al, 64h            ; get keyboard status
        IODELAY
        jz      kbd_wrt_ok
        loop    kbd_wrt_loop       ; try again
                                   ; fall through, still busy
        mov     ah, 1              ; return status - failed
        jmp     kbd_wrt_exit

kbd_wrt_ok:
        mov     al, dl
        out     60h, al            ; data to controller/keyboard
        IODELAY
        xor     ah, ah             ; return status ok

kbd_wrt_exit:
        pop     dx
        pop     cx
        ret
keyboard_write  endp


;
;    KEYBOARD_CMD
;       Send a command in register BL to the keyboard controller
;       (port 64h).
;
;       If the routine times out due to the buffer remaining
;       full, ah is non-zero.
;
;       Called with:    bl = command byte
;                       ds = cs
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
;
;       Regs Used:      ax, cx


keyboard_cmd    proc    near
        xor     cx, cx             ; counter for timeout (64K)
cmd_wait:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_send           ; ready to accept command ?
        loop    cmd_wait           ; jump if not
                                   ; fall through, still busy
        jmp     cmd_error

cmd_send:                          ; send command byte
        mov     al, bl
        out     64h, al            ; send command
        IODELAY

        xor     cx, cx             ; counter for timeout (64K)
cmd_accept:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_ok             ; jump if command accepted
        loop    cmd_accept         ; try again
                                   ; fall through, still busy
cmd_error:
        mov     ah, 1              ; return status - failed
        jmp     cmd_exit
cmd_ok:
        xor     ah, ah             ; return status ok
cmd_exit:
        ret
keyboard_cmd    endp


;
;    CHECK_A20
;       Check if A20 enabled
;
;       Returns:        cx = 0, if disabled
;                            1, if enabled
;
;       Regs Used:      ax, cx

check_A20     proc    near
        push    di
        push    si
        push    ds
        push    es

        mov     di, 10h
        mov     ax, 0FFFFh
        mov     es, ax             ; es:di = FFFF:10h
        xor     ax, ax             ; default ax=0
        mov     si, ax
        mov     ds, ax             ; ds:si = 0:0
        mov     cx, ax

        cli                        ; disable interrupts
        mov     ax, es:[di]        ; get value at FFFF:10h
        cmp     ax, [si]           ; same as 0:0 ?
        je      getA20_skip1       ; jump if unknown state
        inc     cx                 ; A20 enabled (compare fails)
        jmp     getA20_exit

getA20_skip1:
        not     word ptr ds:[si]   ; invert word
        mov     ax, es:[di]        ; get value at FFFF:10h
        cmp     ax, [si]           ; same as 0:0 ?
        je      getA20_skip2       ; jump if disabled
        inc     cx                 ; A20 enabled (compare fails)
getA20_skip2:
        not     word ptr ds:[si]   ; restore word back to original

getA20_exit:
        sti                        ; enable interrupts
        pop     es
        pop     ds
        pop     si
        pop     di
        ret
check_A20     endp


;
;    ERROR_WRITE
;       Check if ah=0, as returned from the keyboard_write
;       routine and display message that a keyboard write failed
;       if ah not zero. A real error handler might replace this
;       routine.
;
;       Regs Used:      ah

error_write     proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_w_exit       ; exit if not
        OUTMSG  kbd_wrt_errmsg     ; error handler goes here
        call    boop
        mov     ah, 4Fh
        int     21h                ; exit, wer're done
error_w_exit:
        ret
error_write     endp

kbd_wrt_errmsg  db      'Keyboard_write - Input buffer full'
                db       CR, LF, '$'



;
;    ERROR_READ
;       Check if ah=0, as returned from the keyboard_read
;       routine and display message that a keyboard read failed
;       if ah not zero. A real error handler might replace this
;       routine.
;
;       Regs Used:      ah

error_read      proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_r_exit       ; exit if not
        OUTMSG  kbd_rd_errmsg      ; error handler goes here
        call    boop
        mov     ah, 4Fh
        int     21h                ; exit, wer're done
error_r_exit:
        ret
error_read      endp

kbd_rd_errmsg   db      'Keyboard_read - Timeout error'
                db       CR, LF, '$'


;
;    ERROR_CMD
;       Check if ah=0, as returned from the keyboard_cmd routine
;       and display message that a keyboard command failed if
;       ah not zero. A real error handler might replace this
;       routine.
;
;       Regs Used:      ah

error_cmd       proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_c_exit       ; exit if not
        OUTMSG  kbd_cmd_errmsg     ; error handler goes here
        call    boop
        mov     ah, 4Fh
        int     21h                ; exit, wer're done

error_c_exit:
        ret
error_cmd       endp

kbd_cmd_errmsg  db      'Keyboard_cmd - Input buffer full'
                db      CR, LF, '$'


;
;    BOOP ERROR SOUND SUBROUTINE
;       Send a short tone to the speaker
;
;       Called with:    nothing
;
;       Regs used:      none

boop    proc    near
        push    ax
        push    bx
        push    cx
        in      al, 61h            ; read 8255 port B
        and     al, 0FEh           ; turn off the 8253 timer bit
        mov     bx, 150            ; loop bx times

; begin loops of tone on and tone off to create frequency

bbcycle:
        or      al, 2              ; turn on speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; on cycle time duration
bbeepon:
        loop    bbeepon            ; delay
        and     al, 0FDh           ; turn off the speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; off cycle time duration
bbeepoff:
        loop    bbeepoff           ; delay
        dec     bx
        jnz     bbcycle            ; loop
        pop     cx
        pop     bx
        pop     ax
        ret
boop    endp


;
;   HEX SUBROUTINE
;       convert the hex number in al into two ascii characters
;       at ptr ds:bx.
;
;       Called with:    al = input hex number
;                       ds:bx = ptr where to put ascii
;
;       Regs Used:      al, bx

hex     proc    near
        push    bx
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al
        mov     ax, bx
        pop     bx
        mov     [bx], ax           ; transfer ascii bytes
        ret
hex     endp


;
;    CPU 386 IDENTIFICATION SUBROUTINE
;       Identify the CPU type, from 8088 to 386+.  This is
;       subset of the more extensive CPUVALUE program.  It is
;       used when identification of CPUs above the 386 is
;       not necessary (i.e. 32-bit support or not)
;
;       Called with:    nothing
;
;       Returns:        al = CPU type
;                             0 if 8088/8086 or V20/V30
;                             1 if 80186/80188
;                             2 if 80286
;                             3 if 80386 or better
;
;       Regs used:      ax, bx
;                       eax (32-bit CPU only)
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler

.8086   ; all instructions 8088/8086 unless overridden later

cpu386  proc    far
        push    cx
        push    dx
        push    ds
        push    es

; 8088/8086 test - Use rotate quirk - All later CPUs mask the CL
;   register with 0Fh, when shifting a byte by cl bits.  This
;   test loads CL with a large value (20h) and shifts the AX
;   register right.  With the 8088, any bits in AX are shifted
;   out, and becomes 0.  On all higher level processors, the
;   CL value of 20h is anded with 0Fh, before the shift.  This
;   means the effective number of shifts is 0, so AX is
;   unaffected.

        mov     cl, 20h            ; load high CL value
        mov     ax, 1              ; load a non-zero value in AX
        shr     ax, cl             ; do the shift
        cmp     ax, 0              ; if zero, then 8088/86
        jne     up186              ; jump if not 8088/86
        jmp     uP_Exit

; 80186/80188 test - Check what is pushed onto the stack with a
;   PUSH SP instruction.  The 80186 updates the stack pointer
;   before the value of SP is pushed onto the stack.  With all
;   higher level processors, the current value of SP is pushed
;   onto the stack, and then the stack pointer is updated.

up186:
        mov     bx, sp             ; save the current stack ptr
        push    sp                 ; do test
        pop     ax                 ; get the pushed value
        cmp     ax, bx             ; did SP change ?
        je      up286              ; if not, it's a 286+
        mov     ax, 1              ; set 80186 flag
        jmp     uP_Exit

; 80286 test A - We'll look at the top four bits of the EFLAGS
;   register.  On a 286, these bits are always zero.  Later
;   CPUs allow these bits to be changed.  During this test,
;   We'll disable interrupts to ensure interrupts do not change
;   the flags.

up286:
        cli                        ; disable interrupts
        pushf                      ; save the current flags

        pushf                      ; push flags onto stack
        pop     ax                 ; now pop flags from stack
        or      ax, 0F000h         ; try and set bits 12-15 hi
        push    ax
        popf                       ; set new flags
        pushf
        pop     ax                 ; see if upper bits are 0

        popf                       ; restore flags to original
        sti                        ; enable interrupts
        test    ax, 0F000h         ; were any upper bits 1 ?
        jnz     up386plus          ; if so, not a 286

; 80286 test B - If the system was in V86 mode, (386 or higher)
;   the POPF instruction causes a protection fault, and the
;   protected mode software must emulate the action of POPF. If
;   the protected mode software screws up, as occurs with a
;   rarely encountered bug in Windows 3.1 enhanced mode, the
;   prior test may look like a 286, but it's really a higher
;   level processor. We'll check if the protected mode bit is
;   on.  If not, it's guaranteed to be a 286.

.286P                              ; allow a 286 instruction
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      is286              ; jump if not (must be 286)

; 80286 test C - It's very likely a 386 or greater, but it is
;   not guaranteed yet.  There is a small possibility the system
;   could be in 286 protected mode so we'll do one last test. We
;   will try out a 386 unique instruction, after vectoring the
;   bad-opcode interrupt vector (int 6) to ourselves.

        call    hook_int6          ; do it!
        mov     [badoff], offset upbad_op3  ; where to go if bad
.386
        xchg    eax, eax           ; 32 bit nop (bad on 286)

        call    restore_int6       ; restore vector
        jmp     up386plus          ; only gets here if 386
                                   ;  or greater!

; Interrupt vector 6 (bad opcode) comes here if system is a
;   80286 (assuming the 286 protected mode interrupt 6 handler
;   will execute the bad-opcode interrupt).

upbad_op3:
        call    restore_int6
is286:
        mov     ax, 2              ; set 80286 flag
        jmp     uP_Exit

up386plus:
        mov     ax, 3              ; 32-bit CPU (386 or later)

up_Exit:
        pop     es
        pop     ds
        pop     dx
        pop     cx
        ret
cpu386  endp
.8086                              ; return to 8086 instructions


;
;    CPU MODE
;       Check if the 286 or later CPU is in real, protected or
;       V86 mode.  It is assumed that if the 80386 or later
;       processor is in protected mode, we must be in V86 mode.
;
;       Call with:      ds:[cpu_val] set
;
;       Returns:        al = 0 protected mode not supported
;                            1 if real mode
;                            2 if protected mode
;                            3 if V86 mode
;                       ah = privilege level 0 to 3
;
;       Regs used:      ax

.386P                              ; allow 286/386 instructions

cpumode proc    near
        push    cx
        xor     cx, cx             ; assume no protected mode
        cmp     [cpu_val], 2       ; 286 CPU or later ?
        jb      cpum_Exit          ; jump if not
        mov     cx, 1              ; assume real mode flag
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      cpum_Exit          ; jump if not (real mode)

cpu_not_real:
        mov     cl, 2              ; protected mode
        pushf
        pop     ax                 ; get flags
        and     ax, 3000h          ; get I/O privilege level
        shr     ax, 12
        mov     ch, al             ; save privilege
        cmp     [cpu_val], 2       ; if 286, then protected
        je      cpum_Exit          ; jump if so

; On 386 or later, we have to assume V86 mode.  Note that the
;  next four lines of code (commented out) might seem the
;  correct way to detect V86 mode.  It will not work, since the
;  PUSHFD instruction clears the VM bit before placing it on the
;  stack.  This is undocumented on the 386 and 486, but
;  documented on the Pentium/Pentium Pro.

;        pushfd                     ; save flags on stack
;        pop     eax                ; get extended flags
;        test    eax, 20000h        ; V86 mode ?
;        jz      cpum_out_mode      ; jump if not

        mov     cl, 3              ; return V86 mode

cpum_Exit:
        mov     ax, cx             ; return status
        pop     cx
        ret
cpumode endp
.8086



;
;    HOOK INTERRUPT 6
;       Save the old interrupt 6 vector and replace it with
;       a new vector to the bad_op_handler.  Vectors are handled
;       directly without using DOS.
;
;       Called with:    nothing
;
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Regs used:      none

hook_int6 proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     ax, es:[6*4]       ; get offset of int 6
        mov     cx, es:[6*4+2]     ; get segment
        mov     es:[6*4], offset bad_op_handler
        mov     word ptr es:[6*4+2], seg bad_op_handler
        sti                        ; enable interrupts
        mov     [old_int6_seg], cx ; save original vector
                mov     [old_int6_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_int6 endp


;
;    RESTORE INTERRUPT 6
;       Restore the previously saved old interrupt 6 vector.
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Returns:        vector restored
;
;       Regs used:      none

restore_int6 proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_int6_seg] ; get original vector
        mov     dx, [old_int6_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[6*4], dx       ; restore original int 6
        mov     es:[6*4+2], cx
        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_int6 endp


;
;    BAD OFFSET INTERRUPT HANDLER
;       If a bad opcode occurs (80286 or later) will come here.
;       The saved BADOFF offset is used to goto the routine
;       previously stored in BADOFF.
;
;       In a few cases, it is also used for double faults. A few
;       instructions (RDMSR & WRMSR) can issue a double fault if
;       not supported, so well come here as well.
;
;       Called with:    cs:[badoff] previously set
;
;       Returns:        returns to address stored in badoff


bad_op_handler proc far
        push    ax
        push    bp
        mov     ax, cs:[badoff]
        mov     bp, sp
        mov     ss:[bp+4], ax      ; insert new return offset
        pop     bp
        pop     ax
        iret
bad_op_handler endp


cseg    ends

;================================================= stack1 ======

stack1  segment para stack

        db      192 dup (0)

stack1  ends


        end     start

