
;
;                           BIOSVIEW
;
;
;    Display the contents of BIOS values in the 40h segment
;    with names.  See book for reserved functions, which vary in
;    function depending on manufacturer.
;
;    Functions are labeled based on the machine the program is
;    run on, and will be different depending on the platform
;    used.  Bit descriptions describe the case when the bit
;    is set to "1".
;
;    (c) Copyright 1996  Frank van Gilluwe  All Rights Reserved.

include undocpc.inc

BMSG  struc
	b_addr		dw	?	; RAM address offset (segment 40h)
					;  -1 indicates end of list
					;  -2 indicates extra description
					;      line(s) and ignore other
					;      fields
	b_length        db	?	; number of bytes
					;  number of bits if b_addr = -2
	b_type		db	?	; type of data,
					;   0=hex, 1=binary, 2=seg:off
					;   starting bit # if b_addr = -2
	b_platform	db	?	; 0 = PC only
					; 1 = PC & XT only
					; 2 = All systems
					; 3 = XT+ only
					; 4 = AT+ only
					; 5 = XT only
					; 6 = EGA+
					; 7 = VGA+
	b_msg		db	36 dup(?)	; description string
ends


cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stacka

biosview  proc  far
        db      'BIOSVIEW v1.00 '
        db      '(c) 1996 Frank van Gilluwe',0

	; BIOS data array (see BMSG structure)
	;      addr len typ plat description

msg0    BMSG   <00h,  2,  0,  2, 'Serial I/O address, port 1          '>
        BMSG   <02h,  2,  0,  2, 'Serial I/O address, port 2          '>
        BMSG   <04h,  2,  0,  2, 'Serial I/O address, port 3          '>
        BMSG   <06h,  2,  0,  2, 'Serial I/O address, port 4          '>
        BMSG   <08h,  2,  0,  2, 'Parallel I/O address, port 1        '>
        BMSG   <0Ah,  2,  0,  2, 'Parallel I/O address, port 2        '>
        BMSG   <0Ch,  2,  0,  2, 'Parallel I/O address, port 3        '>
        BMSG   <0Eh,  2,  0,  1, 'Parallel I/O address, port 4        '>
        BMSG   <0Eh,  2,  0,  4, 'Extended BIOS data area segment     '>
        BMSG   <10h,  2,  1,  2, 'Equipment flags                     '>
        BMSG   <-2 ,  2, 15,  4, '  number of parallel ports detected '>
        BMSG   <-2 ,  1, 13,  4, '  internal modem present            '>
        BMSG   <-2 ,  1, 12,  4, '  unused                            '>
        BMSG   <-2 ,  3, 11,  4, '  number of serial ports detected   '>
        BMSG   <-2 ,  1,  8,  4, '  unused                            '>
        BMSG   <-2 ,  2,  7,  4, '  diskette drives (00=1 drive)      '>
        BMSG   <-2 ,  2,  5,  4, '  video adapter type (11=mono)      '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  1,  2,  4, '  motherboard mouse support         '>
        BMSG   <-2 ,  1,  1,  4, '  math coprocessor installed        '>
        BMSG   <-2 ,  1,  0,  4, '  boot diskette drive present       '>
        BMSG   <-2 ,  2, 15,  1, '  number of parallel ports detected '>
        BMSG   <-2 ,  1, 13,  1, '  unused                            '>
        BMSG   <-2 ,  1, 12,  1, '  game port attached                '>
        BMSG   <-2 ,  3, 11,  1, '  number of serial ports detected   '>
        BMSG   <-2 ,  1,  8,  1, '  unused                            '>
        BMSG   <-2 ,  2,  7,  1, '  diskette drives (00=1 drive)      '>
        BMSG   <-2 ,  2,  5,  1, '  video adapter type (11=mono)      '>
        BMSG   <-2 ,  2,  3,  1, '  motherboard memory size 16-64KB   '>
        BMSG   <-2 ,  1,  1,  5, '  math coprocessor installed        '>
        BMSG   <-2 ,  1,  1,  0, '  unused                            '>
        BMSG   <-2 ,  1,  0,  1, '  boot diskette drive present       '>
        BMSG   <12h,  1,  0,  2, 'Manufacturing test                  '>
        BMSG   <13h,  2,  0,  2, 'Main memory size in 1 KB blocks     '>
        BMSG   <15h,  2,  0,  1, 'Adapter memory size                 '>
        BMSG   <15h,  2,  0,  4, 'Error codes or vendor specific info '>
        BMSG   <17h,  1,  1,  2, 'Keyboard, shift flags 1             '>
        BMSG   <-2 ,  1,  7,  2, '  insert on                         '>
        BMSG   <-2 ,  1,  6,  2, '  caps lock on                      '>
        BMSG   <-2 ,  1,  5,  2, '  num lock on                       '>
        BMSG   <-2 ,  1,  4,  2, '  scroll lock on                    '>
        BMSG   <-2 ,  1,  3,  2, '  alt key down (left or right)      '>
        BMSG   <-2 ,  1,  2,  2, '  control key down (left/right)     '>
        BMSG   <-2 ,  1,  1,  2, '  left shift key down               '>
        BMSG   <-2 ,  1,  0,  2, '  right shift key down              '>
        BMSG   <18h,  1,  1,  2, 'Keyboard, shift flags 2             '>
        BMSG   <-2 ,  1,  7,  2, '  insert key down                   '>
        BMSG   <-2 ,  1,  6,  2, '  caps lock key down                '>
        BMSG   <-2 ,  1,  5,  2, '  num lock key down                 '>
        BMSG   <-2 ,  1,  4,  2, '  scroll lock key down              '>
        BMSG   <-2 ,  1,  3,  2, '  pause activated                   '>
        BMSG   <-2 ,  1,  2,  2, '  sys req key down                  '>
        BMSG   <-2 ,  1,  1,  2, '  left alt key down                 '>
        BMSG   <-2 ,  1,  0,  2, '  left ctrl key down                '>
        BMSG   <19h,  1,  0,  2, 'Keyboard, Alt-Numpad work area      '>
        BMSG   <1Ah,  2,  0,  2, 'Keyboard, head of buffer pointer    '>
        BMSG   <1Ch,  2,  0,  2, 'Keyboard, end of buffer pointer     '>
        BMSG   <1Eh,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <20h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <22h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <24h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <26h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <28h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <2Ah,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <2Ch,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <2Eh,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <30h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <32h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <34h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <36h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <38h,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <3Ah,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <3Ch,  2,  0,  2, 'Keyboard buffer                     '>
        BMSG   <3Eh,  1,  1,  2, 'Diskette, recalibrate status        '>
        BMSG   <-2 ,  1,  7,  2, '  diskette interrupt occurred       '>
        BMSG   <-2 ,  2,  6,  2, '  unused                            '>
        BMSG   <-2 ,  1,  3,  2, '  drive 3 calibrated (PC/XT only)   '>
        BMSG   <-2 ,  1,  2,  2, '  drive 2 calibrated (PC/XT only)   '>
        BMSG   <-2 ,  1,  1,  2, '  drive 1 calibrated                '>
        BMSG   <-2 ,  1,  0,  2, '  drive 0 calibrated                '>
        BMSG   <3Fh,  1,  1,  2, 'Diskette, motor status              '>
        BMSG   <-2 ,  1,  7,  2, '  current operation requires delay  '>
        BMSG   <-2 ,  1,  6,  2, '  unused                            '>
        BMSG   <-2 ,  2,  5,  2, '  drive select                      '>
        BMSG   <-2 ,  1,  3,  2, '  drive 3 motor on (PC/XT only)     '>
        BMSG   <-2 ,  1,  2,  2, '  drive 2 motor on (PC/XT only)     '>
        BMSG   <-2 ,  1,  1,  2, '  drive 1 motor on                  '>
        BMSG   <-2 ,  1,  0,  2, '  drive 0 motor on                  '>
        BMSG   <40h,  1,  0,  2, 'Diskette, motor timeout             '>
        BMSG   <41h,  1,  0,  2, 'Diskette, controller return code    '>
        BMSG   <42h,  1,  1,  2, 'Diskette, controller status reg 0   '>
        BMSG   <-2 ,  2,  7,  2, '  interrupt code                    '>
        BMSG   <-2 ,  1,  5,  2, '  commanded seek completed          '>
        BMSG   <-2 ,  1,  4,  2, '  drive fault                       '>
        BMSG   <-2 ,  1,  3,  2, '  drive not ready                   '>
        BMSG   <-2 ,  1,  2,  2, '  head state                        '>
        BMSG   <-2 ,  2,  1,  2, '  drive selected at interrupt       '>
        BMSG   <43h,  1,  1,  2, 'Diskette, controller status reg 1   '>
        BMSG   <-2 ,  1,  7,  2, '  access beyond last cylinder       '>
        BMSG   <-2 ,  1,  6,  2, '  unused                            '>
        BMSG   <-2 ,  1,  5,  2, '  CRC error detected on read        '>
        BMSG   <-2 ,  1,  4,  2, '  DMA overrun                       '>
        BMSG   <-2 ,  1,  3,  2, '  unused                            '>
        BMSG   <-2 ,  1,  2,  2, '  data problem                      '>
        BMSG   <-2 ,  1,  1,  2, '  write protection problem          '>
        BMSG   <-2 ,  1,  0,  2, '  missing address mark              '>
        BMSG   <44h,  1,  1,  2, 'Diskette, controller status reg 2   '>
        BMSG   <-2 ,  1,  7,  2, '  unused                            '>
        BMSG   <-2 ,  1,  6,  2, '  deleted address mark found        '>
        BMSG   <-2 ,  1,  5,  2, '  CRC error in data field           '>
        BMSG   <-2 ,  1,  4,  2, '  wrong cylinder                    '>
        BMSG   <-2 ,  1,  3,  2, '  equal condition satisfied         '>
        BMSG   <-2 ,  1,  2,  2, '  can not find sector               '>
        BMSG   <-2 ,  1,  1,  2, '  bad cylinder                      '>
        BMSG   <-2 ,  1,  0,  2, '  address mark can not be found     '>
        BMSG   <45h,  1,  0,  2, 'Diskette, controller cylinder number'>
        BMSG   <46h,  1,  0,  2, 'Diskette, controller head number    '>
        BMSG   <47h,  1,  0,  2, 'Diskette, controller sector number  '>
        BMSG   <48h,  1,  0,  2, 'Diskette, controller bytes written  '>
        BMSG   <49h,  1,  0,  2, 'Video mode                          '>
        BMSG   <4Ah,  2,  0,  2, 'Video, number of columns            '>
        BMSG   <4Ch,  2,  0,  2, 'Video, bytes per page               '>
        BMSG   <4Eh,  2,  0,  2, 'Video, current page offset          '>
        BMSG   <50h,  2,  0,  2, 'Video, cursor position, page 0      '>
        BMSG   <52h,  2,  0,  2, 'Video, cursor position, page 1      '>
        BMSG   <54h,  2,  0,  2, 'Video, cursor position, page 2      '>
        BMSG   <56h,  2,  0,  2, 'Video, cursor position, page 3      '>
        BMSG   <58h,  2,  0,  2, 'Video, cursor position, page 4      '>
        BMSG   <5Ah,  2,  0,  2, 'Video, cursor position, page 5      '>
        BMSG   <5Ch,  2,  0,  2, 'Video, cursor position, page 6      '>
        BMSG   <5Eh,  2,  0,  2, 'Video, cursor position, page 7      '>
        BMSG   <60h,  2,  0,  2, 'Video, cursor shape                 '>
        BMSG   <62h,  1,  0,  2, 'Video, active display page          '>
        BMSG   <63h,  2,  0,  2, 'Video, I/O port number base         '>
        BMSG   <65h,  1,  1,  2, 'Video, internal mode register       '>
        BMSG   <-2 ,  2,  7,  2, '  unused                            '>
        BMSG   <-2 ,  1,  5,  2, '  screen attribute bit 7 for blink  '>
        BMSG   <-2 ,  1,  4,  2, '  mode 6 graphics operation         '>
        BMSG   <-2 ,  1,  3,  2, '  enable video signal               '>
        BMSG   <-2 ,  1,  2,  2, '  monochrome operation (0=color)    '>
        BMSG   <-2 ,  1,  1,  2, '  modes 4 or 5 graphics operation   '>
        BMSG   <-2 ,  1,  0,  2, '  text modes 2 or 3                 '>
        BMSG   <66h,  1,  1,  2, 'Video, color palette                '>
        BMSG   <-2 ,  2,  7,  2, '  unused                            '>
        BMSG   <-2 ,  1,  5,  2, '  mode 5 color control              '>
        BMSG   <-2 ,  1,  4,  2, '  background uses intensified colors'>
        BMSG   <-2 ,  1,  3,  2, '  intensified border color          '>
        BMSG   <-2 ,  1,  2,  2, '  red                               '>
        BMSG   <-2 ,  1,  1,  2, '  green                             '>
        BMSG   <-2 ,  1,  0,  2, '  blue                              '>
        BMSG   <67h,  2,  0,  0, 'Cassette, time count                '>
        BMSG   <67h,  2,  0,  3, 'General use offset                  '>
        BMSG   <69h,  2,  0,  0, 'Cassette, CRC register              '>
        BMSG   <69h,  2,  0,  3, 'General use segment                 '>
        BMSG   <6Bh,  1,  0,  0, 'Cassette, last value read           '>
        BMSG   <6Bh,  1,  1,  3, 'Last interrupt                      '>
        BMSG   <-2 ,  1,  7,  3, '  IRQ 7 occurred                    '>
        BMSG   <-2 ,  1,  6,  3, '  IRQ 6 occurred                    '>
        BMSG   <-2 ,  1,  5,  3, '  IRQ 5 occurred                    '>
        BMSG   <-2 ,  1,  4,  3, '  IRQ 4 occurred                    '>
        BMSG   <-2 ,  1,  3,  3, '  IRQ 3 occurred                    '>
        BMSG   <-2 ,  1,  2,  4, '  IRQ 2 occurred (really IRQ 8-15)  '>
        BMSG   <-2 ,  1,  2,  5, '  IRQ 2 occurred                    '>
        BMSG   <-2 ,  1,  1,  3, '  IRQ 1 occurred                    '>
        BMSG   <-2 ,  1,  0,  3, '  IRQ 0 occurred                    '>
        BMSG   <6Ch,  4,  0,  2, 'Timer ticks count                   '>
        BMSG   <70h,  1,  0,  2, 'Timer ticks rollover flag           '>
        BMSG   <71h,  1,  0,  2, 'Keyboard, control-break flag        '>
        BMSG   <-2 ,  1,  7,  2, '  Control-Break was pressed         '>
        BMSG   <-2 ,  7,  6,  2, '  unused (and can be any value)     '>
        BMSG   <72h,  2,  0,  2, 'Warm boot flag                      '>
        BMSG   <74h,  1,  0,  3, 'Hard Disk, status of last operation '>
        BMSG   <75h,  1,  0,  3, 'Hard Disk, number attached          '>
        BMSG   <76h,  1,  1,  3, 'Hard Disk, control byte             '>
        BMSG   <-2 ,  2,  7,  3, '  disable retry on error (non-zero) '>
        BMSG   <-2 ,  2,  5,  3, '  unused                            '>
        BMSG   <-2 ,  1,  3,  3, '  drive has more than 8 heads       '>
        BMSG   <-2 ,  3,  2,  3, '  unused                            '>
        BMSG   <77h,  1,  0,  5, 'Hard Disk, port offset              '>
        BMSG   <78h,  1,  0,  3, 'Parallel Printer 1, timeout         '>
        BMSG   <79h,  1,  0,  3, 'Parallel Printer 2, timeout         '>
        BMSG   <7Ah,  1,  0,  3, 'Parallel Printer 3, timeout         '>
        BMSG   <7Bh,  1,  0,  5, 'Parallel Printer 4, timeout         '>
        BMSG   <7Bh,  1,  1,  4, 'VDS support (Virtual DMA services)  '>
        BMSG   <-2 ,  2,  7,  4, '  unused                            '>
        BMSG   <-2 ,  1,  5,  4, '  VDS services supported            '>
        BMSG   <-2 ,  1,  4,  4, '  unused                            '>
        BMSG   <-2 ,  1,  3,  4, '  chaining required on int 4Bh      '>
        BMSG   <-2 ,  3,  2,  4, '  unused                            '>
        BMSG   <7Ch,  1,  0,  3, 'Serial port 1, timeout              '>
        BMSG   <7Dh,  1,  0,  3, 'Serial port 2, timeout              '>
        BMSG   <7Eh,  1,  0,  3, 'Serial port 3, timeout              '>
        BMSG   <7Fh,  1,  0,  3, 'Serial port 4, timeout              '>
        BMSG   <80h,  2,  0,  3, 'Keyboard, start of buffer offset    '>
        BMSG   <82h,  2,  0,  3, 'Keyboard, end of buffer offset      '>
        BMSG   <84h,  1,  0,  6, 'Video, number of rows, less 1       '>
        BMSG   <85h,  2,  0,  6, 'Video, pixels per character         '>
        BMSG   <87h,  1,  1,  6, 'Video, options                      '>
        BMSG   <-2 ,  1,  7,  6, '  no buffer clear on mode change    '>
        BMSG   <-2 ,  2,  6,  6, '  minimum memory on video card      '>
        BMSG   <-2 ,  1,  4,  6, '  unused                            '>
        BMSG   <-2 ,  1,  3,  6, '  video adapter active              '>
        BMSG   <-2 ,  1,  2,  6, '  wait for display enable           '>
        BMSG   <-2 ,  1,  1,  6, '  monochrome monitor (0=color)      '>
        BMSG   <-2 ,  1,  0,  6, '  no CGA emulation of cursor        '>
        BMSG   <88h,  1,  1,  6, 'Video, switches                     '>
        BMSG   <-2 ,  1,  7,  6, '  state of feature 1 line           '>
        BMSG   <-2 ,  1,  6,  6, '  state of feature 0 line           '>
        BMSG   <-2 ,  2,  5,  6, '  unused                            '>
        BMSG   <-2 ,  4,  3,  6, '  adapter switch settings           '>
        BMSG   <89h,  1,  1,  7, 'Video, save area 1                  '>
        BMSG   <-2 ,  1,  7,  7, '  scan lines setting (with bit 4)   '>
        BMSG   <-2 ,  1,  6,  7, '  unused (mostly)                   '>
        BMSG   <-2 ,  1,  5,  7, '  unused                            '>
        BMSG   <-2 ,  1,  4,  7, '  scan lines setting (with bit 7)   '>
        BMSG   <-2 ,  1,  3,  7, '  do not change colors on mode set  '>
        BMSG   <-2 ,  1,  2,  7, '  monochrome monitor (0=color)      '>
        BMSG   <-2 ,  1,  1,  7, '  use gray scales (0=normal colors) '>
        BMSG   <-2 ,  1,  0,  7, '  all modes allowable on monitor    '>
        BMSG   <8Ah,  1,  0,  7, 'Video, save area 2                  '>
        BMSG   <8Bh,  1,  1,  4, 'Diskette, configuration data        '>
        BMSG   <-2 ,  2,  7,  4, '  last data rate sent to controller '>
        BMSG   <-2 ,  2,  5,  4, '  last stepping rate sent           '>
        BMSG   <-2 ,  2,  3,  4, '  data rate at start of operation   '>
        BMSG   <-2 ,  2,  1,  4, '  unused (mostly)                   '>
        BMSG   <8Ch,  1,  0,  4, 'Hard Disk, status                   '>
        BMSG   <8Dh,  1,  0,  4, 'Hard Disk, error register           '>
        BMSG   <8Eh,  1,  0,  4, 'Hard Disk, task complete flag       '>
        BMSG   <8Fh,  1,  1,  4, 'Diskette, controller information    '>
        BMSG   <-2 ,  1,  7,  4, '  unused                            '>
        BMSG   <-2 ,  1,  6,  4, '  drive 1 type has been determined  '>
        BMSG   <-2 ,  1,  5,  4, '  drive 1 is multi-rate             '>
        BMSG   <-2 ,  1,  4,  4, '  drive 1 can detect changed floppy '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  1,  2,  4, '  drive 0 type has been determined  '>
        BMSG   <-2 ,  1,  1,  4, '  drive 0 is multi-rate             '>
        BMSG   <-2 ,  1,  0,  4, '  drive 0 can detect changed floppy '>
        BMSG   <90h,  1,  1,  4, 'Diskette 0, media state             '>
        BMSG   <-2 ,  2,  7,  4, '  data transfer rate 250-1000 Kbit/s'>
        BMSG   <-2 ,  1,  5,  4, '  double stepping required          '>
        BMSG   <-2 ,  1,  4,  4, '  known media in diskette drive     '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  3,  2,  4, '  determination of last access      '>
        BMSG   <91h,  1,  1,  4, 'Diskette 1, media state             '>
        BMSG   <-2 ,  2,  7,  4, '  data transfer rate 250-1000 Kbit/s'>
        BMSG   <-2 ,  1,  5,  4, '  double stepping required          '>
        BMSG   <-2 ,  1,  4,  4, '  known media in diskette drive     '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  3,  2,  4, '  determination of last access      '>
        BMSG   <92h,  1,  1,  4, 'Diskette 0, starting state          '>
        BMSG   <-2 ,  2,  7,  4, '  data transfer rate 250-1000 Kbit/s'>
        BMSG   <-2 ,  1,  5,  4, '  unknown function                  '>
        BMSG   <-2 ,  1,  4,  4, '  unknown function                  '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  1,  2,  4, '  drive type has been determined    '>
        BMSG   <-2 ,  1,  1,  4, '  drive is multi-rate               '>
        BMSG   <-2 ,  1,  0,  4, '  can detect a changed diskette     '>
        BMSG   <93h,  1,  1,  4, 'Diskette 1, starting state          '>
        BMSG   <-2 ,  2,  7,  4, '  data transfer rate 250-1000 Kbit/s'>
        BMSG   <-2 ,  1,  5,  4, '  unknown function                  '>
        BMSG   <-2 ,  1,  4,  4, '  unknown function                  '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  1,  2,  4, '  drive type has been determined    '>
        BMSG   <-2 ,  1,  1,  4, '  drive is multi-rate               '>
        BMSG   <-2 ,  1,  0,  4, '  can detect a changed diskette     '>
        BMSG   <94h,  1,  0,  4, 'Diskette 0, current cylinder        '>
        BMSG   <95h,  1,  0,  4, 'Diskette 1, current cylinder        '>
        BMSG   <96h,  1,  1,  4, 'Keyboard, status flags 3            '>
        BMSG   <-2 ,  1,  7,  4, '  reading two byte keyboard ID      '>
        BMSG   <-2 ,  1,  6,  4, '  last byte was 1st ID byte         '>
        BMSG   <-2 ,  1,  5,  4, '  force num lock on after ID        '>
        BMSG   <-2 ,  1,  4,  4, '  101/102 keyboard (0=83/84 key)    '>
        BMSG   <-2 ,  1,  3,  4, '  right alt key depressed           '>
        BMSG   <-2 ,  1,  2,  4, '  right ctrl key depressed          '>
        BMSG   <-2 ,  1,  1,  4, '  E0h scan code just received       '>
        BMSG   <-2 ,  1,  0,  4, '  E1h scan code just received (most)'>
        BMSG   <97h,  1,  1,  4, 'Keyboard, status flags 4            '>
        BMSG   <-2 ,  1,  7,  4, '  keyboard transmit error           '>
        BMSG   <-2 ,  1,  6,  4, '  LED update in progress            '>
        BMSG   <-2 ,  1,  5,  4, '  keyboard sent a Resend byte       '>
        BMSG   <-2 ,  1,  4,  4, '  keyboard sent an Acknowledge byte '>
        BMSG   <-2 ,  1,  3,  4, '  unused                            '>
        BMSG   <-2 ,  1,  2,  4, '  caps lock LED on                  '>
        BMSG   <-2 ,  1,  1,  4, '  num lock LED on                   '>
        BMSG   <-2 ,  1,  0,  4, '  scroll lock LED on                '>
        BMSG   <98h,  4,  2,  4, 'Users wait flag pointer             '>
        BMSG   <9Ch,  4,  0,  4, 'Users wait count                    '>
        BMSG  <0A0h,  1,  1,  4, 'Wait flag                           '>
        BMSG   <-2 ,  1,  7,  4, '  wait time elapsed                 '>
        BMSG   <-2 ,  6,  6,  4, '  unused                            '>
        BMSG   <-2 ,  1,  0,  4, '  wait in progress                  '>
        BMSG  <0A1h,  1,  0,  4, 'Local area network area 1           '>
        BMSG  <0A2h,  1,  0,  4, 'Local area network area 2           '>
        BMSG  <0A3h,  1,  0,  4, 'Local area network area 3           '>
        BMSG  <0A4h,  1,  0,  4, 'Local area network area 4           '>
        BMSG  <0A5h,  1,  0,  4, 'Local area network area 5           '>
        BMSG  <0A6h,  1,  0,  4, 'Local area network area 6           '>
        BMSG  <0A7h,  1,  0,  4, 'Local area network area 7           '>
        BMSG  <0A8h,  4,  2,  6, 'Video, parameter control block ptr  '>
        BMSG  <0CEh,  1,  0,  4, 'Clock, days since 1980 (some)       '>
        BMSG  <100h,  1,  0,  2, 'Print screen status                 '>

	dw	-1      	    ; end marker


msg_BIOS_head   db      CR, LF
                db      'BIOS RAM MEMORY CONTENTS (segment 40h)', CR, LF
                db      CR, LF
                db      'Address  Size       Value       Description', CR, LF
		db	'   '
crlf            db      CR, LF, '$'

msg_BIOS_text   db      '40:'
msg_BIOS_addr   db         '      '
msg_BIOS_size   db               'bit 15-14  '
msg_BIOS_valu   db                          '00110011b   '
msg_BIOS_desc   db	'                                    '
msg_BIOS_end    db      CR, LF, '$'


byte_msg	db	'byte '
word_msg	db	'word '
dword_msg	db	'dword'


old_int6_seg 	dw 	0	   ; temp storage for old int 6
old_int6_off 	dw 	0          ;  vector (bad opcode)
badoff       	dw	0          ; temp return offset if bad offset
				   ;  interrupt 6 called
system_type  	db 	0          ; temp storage
video_adapter	db	0	   ; 4=EGA, 5+=VGA

last_addr	dw	0	   ; last address processed

;

start:
	push	cs
	pop	ds
        call    sysvalue           ; get system type
	mov	[system_type], al  ; save system type

	call	video_type
	mov	[video_adapter], al ; save video adapter type

        call    display_BIOS       ; display BIOS values


        mov     ax,4C00h
        int     21h                ; exit with al return code

biosview  endp


;
;    DISPLAY_BIOS
;       display the BIOS RAM memory contents which may be
;       redirected to a file.
;
;       Called with:    nothing
;
;       Returns:        BIOS values sent to std output
;
;       Regs used:      all
;
;       Subs used:      hex

display_BIOS    proc    near
        push    cs
        pop     ds
	mov	ax, 40h
	mov	es, ax

        OUTMSG  msg_BIOS_head      ; output header string

	mov	si, offset msg0

	; process each line

disp_BIOS_item:
	mov	ax, [si].b_addr	   ; get address
	cmp     ax, -1             ; done?
	jne	disp_BIOS_ok
	jmp     disp_BIOS_done

disp_BIOS_ok:
	mov	al, [si].b_platform
	mov	ah, [system_type]
	cmp	al, 0
	jne	disp_BIOS_try1	   ; jump if description not PC
	cmp	ah, 0		   ; running on a PC ?
	je	disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try1:
	cmp	al, 1
	jne	disp_BIOS_try2	   ; jump if description not PC or XT
	cmp	ah, 8		   ; running on a PC or XT
	jbe     disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try2:
	cmp	al, 2
	je	disp_BIOS_output   ; output for every system type

	cmp	al, 3
	jne	disp_BIOS_try4	   ; jump if description not XT+
	cmp	ah, 8		   ; running on a XT or better
	jae     disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try4:
	cmp	al, 4
	jne	disp_BIOS_try5	   ; jump if description not AT+
	cmp	ah, 10h 	   ; running on a AT+
	jae     disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try5:
	cmp	al, 5
	jne	disp_BIOS_try6	   ; jump if description not XT
	cmp     ah, 1		   ; running on a XT
	je      disp_BIOS_output   ; jump if so
	cmp     ah, 8		   ; running on a XT
	je      disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try6:
	cmp	al, 6
	jne	disp_BIOS_try7	   ; jump if description not EGA only
	cmp     [video_adapter], 4 ; running on a EGA
	je      disp_BIOS_output   ; jump if so
	jmp     disp_BIOS_next

disp_BIOS_try7:
	cmp	al, 7
	jne	disp_BIOS_next2    ; jump if description not VGA
	cmp     [video_adapter], 5 ; running on a VGA or better
	jae     disp_BIOS_output   ; jump if so
disp_BIOS_next2:
	jmp     disp_BIOS_next

	; clear line for output

disp_BIOS_output:
	mov	cx, offset msg_BIOS_end - offset msg_BIOS_text
	mov	al, ' '
	mov	di, offset msg_BIOS_text
	push	es
	push	ds
	pop	es
	cld
	rep	stosb		   ; clear current string
	pop	es

	cmp	[si].b_addr, -2    ; output description only?
	jne	disp_BIOS_addr	   ; jump if not

	; insert bit values into line for multiple lines

	cmp	[si].b_length, 0   ; no length field?
	je	disp_BIOS_skp4	   ; jump if so

	mov	bx, offset msg_BIOS_size
	mov	word ptr [bx], 'ib'
	add	bx, 2
	mov	word ptr [bx], ' t' ; insert "bit "
	add	bx, 2
	mov	al, [si].b_type	   ; get bit number
	mov	ah, al
	cmp	al, 10
	jb	disp_BIOS_skp1
	mov	byte ptr [bx], '1'
	inc	bx
	sub	al, 10
disp_BIOS_skp1:
	add	al, 30h		   ; covert to ascii
	mov	[bx], al	   ; insert bit number
	mov	al, [si].b_length
	cmp	al, 1          	   ; more than 1 bit?
	jbe	disp_BIOS_skp3	   ; jump if not
	inc	bx
	mov	byte ptr [bx], '-' ; insert dash (i.e. bit 10-8)
	inc	bx
	sub	ah, al
	inc	ah
	mov	al, ah		   ; al is the lower bit number
	cmp	al, 10
	jb	disp_BIOS_skp2
	mov	byte ptr [bx], '1'
	inc	bx
	sub	al, 10
disp_BIOS_skp2:
	add	al, 30h		   ; covert to ascii
	mov	[bx], al	   ; insert bit number

disp_BIOS_skp3:
	mov	di, [last_addr]
	mov	ax, es:[di] 	   ; get last value
	mov	bx, offset msg_BIOS_valu + 2
	mov	cl, [si].b_type	   ; get bit number
	ror	ax, cl
	mov	cl, [si].b_length  ; xfer binary bits
	xor	ch, ch

disp_BIOS_loop0:
	mov	byte ptr [bx], '0' ; assume zero bit
	test	al, 1
	jz	disp_BIOS_not1a
	mov	byte ptr [bx], '1' ; assume zero bit
disp_BIOS_not1a:
	inc	bx
	rol	al, 1
	loop	disp_BIOS_loop0

disp_BIOS_skp4:
	jmp	disp_BIOS_desc	   ; output description


	; insert address into output string

disp_BIOS_addr:
	mov	word ptr [msg_BIOS_text], '04'
	mov	[msg_BIOS_text+2], ':' ; insert "40:"

	mov	bx, offset msg_BIOS_addr
	mov	ax, [si].b_addr
	xchg    al, ah
	call    hex
	xchg	al, ah
	call	hex		   ; output 4 hex digits

	; insert size into output string

	mov	bx, offset msg_BIOS_size
	mov	al, [si].b_length
	cmp	al, 1
	jne	disp_BIOS_skp5
	mov	di, offset byte_msg
	jmp	disp_BIOS_size
disp_BIOS_skp5:
	cmp	al, 2
	jne	disp_BIOS_skp6
	mov	di, offset word_msg
	jmp	disp_BIOS_size
disp_BIOS_skp6:
	mov	di, offset dword_msg
	jmp	disp_BIOS_size
disp_BIOS_size:
	mov	cx, 5

disp_BIOS_loop1:
	mov	al, [di]
	mov	[bx], al
	inc	di
	inc	bx
	loop	disp_BIOS_loop1    ; move size string

	; insert value into output string

	mov	al, [si].b_length
	mov	ah, [si].b_type
	mov	di, [si].b_addr
	mov	[last_addr], di
	mov	bx, offset msg_BIOS_valu

	cmp	al, 1		   ; byte ?
	jne	disp_BIOS_skp7
	mov	al, es:[di]	   ; get byte
	cmp     ah, 0
	je	disp_BIOS_hex
	mov	cx, 8		   ; xfer binary bits
disp_BIOS_loop2:
	mov	byte ptr [bx], '0' ; assume zero bit
	test	al, 80h
	jz	disp_BIOS_not1
	mov	byte ptr [bx], '1' ; assume zero bit
disp_BIOS_not1:
	inc	bx
	rol	al, 1
	loop	disp_BIOS_loop2
	mov	byte ptr [bx], 'b' ; end binary string with 'b'
	jmp	disp_BIOS_desc

disp_BIOS_hex:
	mov	al, es:[di]	   ; get byte
	call	hexh    	   ; insert hex byte
	jmp	disp_BIOS_desc

disp_BIOS_skp7:
	cmp	al, 2		   ; word ?
	jne	disp_BIOS_skp8
	cmp	word ptr es:[di], 9 ; 9 or less?
	jbe	disp_BIOS_hex	   ; jump if so
	mov	al, es:[di+1]	   ; get byte
	call	hex		   ; insert hex word
	mov	al, es:[di]	   ; get byte
	call	hex
	jmp	disp_BIOS_h

disp_BIOS_skp8:                    ; assumed dword
	cmp	ah, 2
	je      disp_BIOS_skp9     ; jump if seg:off type dword
	cmp	word ptr es:[di+2], 0
	jne	disp_BIOS_skp9	   ; jump if > 9
	cmp	word ptr es:[di], 9
	jbe	disp_BIOS_hex	   ; output 0-9
disp_BIOS_skp9:
	mov	al, es:[di+3]	   ; get byte
	call	hex		   ; insert hex dword
	mov	al, es:[di+2]	   ; get byte
	call	hex
	cmp	ah, 2
	jne	disp_BIOS_dword
	mov	byte ptr [bx], ':' ; insert seg:off separator
	inc	bx
disp_BIOS_dword:
	mov	al, es:[di+1]	   ; get byte
	call	hex
	mov	al, es:[di]	   ; get byte
	call	hex
disp_BIOS_h:
	mov	byte ptr [bx], 'h' ; end hex value with 'h'

	; insert description into output string

disp_BIOS_desc:
	mov	bx, offset msg_BIOS_desc
	lea	di, [si].b_msg 	   ; pointer to description
	mov	cx, 36
disp_BIOS_loop3:
	mov	al, [di]
	mov	[bx], al
	inc	di
	inc	bx
	loop	disp_BIOS_loop3    ; move desciption string

	; display line

        OUTMSG  msg_BIOS_text      ; output info line

	; move to next item
disp_BIOS_next:
	add	si, SIZE BMSG
	jmp	disp_BIOS_item

disp_BIOS_done:
        ret
display_BIOS    endp



;
;    HEX
;       convert the hex number in al into two ascii characters
;       in [bx] and increment bx by 2
;
;       Called with:    al = input hex number
;                       bx = ptr where to put ascii
;
;       Regs Used:      bx

hex     proc    near
	push	ax
        push    bx
        mov     bl, al
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al
        mov     ax, bx
        pop     bx
        mov     [bx], ax           ; output ascii bytes
	add	bx, 2
	pop	ax
        ret
hex     endp


;
;    HEXH
;       convert the hex number in al into ascii characters
;	  values 0-9 appear in decimal
;         A to FFh appear in hex followed by a "h"
;         ascii is placed in [bx]
;
;       Called with:    al = input hex number
;                       bx = ptr where to put ascii
;
;       Regs Used:      bx

hexh    proc    near
	cmp	al, 9
	jbe	hexh_skp1
	call	hex		   ; output 2 hex digits
	mov	byte ptr [bx], 'h' ; insert "h"
	jmp	hexh_done

hexh_skp1:
	add	al, 30h		   ; convert to digit 0-9
	mov	[bx], al

hexh_done:
	ret
hexh    endp


;
;    SYSTEM TYPE DETECTION SUBROUTINE
;       Determine the type of system the software is running
;       on.
;
;       Called with:    nothing
;
;       Returns:        al = System type
;                             0 if PC (8088 based)
;                             1 if XT (8088 based)
;                             2 if PC convertible (8088 based)
;                             3 if PC jr (8088 based)
;                             4 other pre-80286 based machine
;                             8 if XT (80286 based)
;                            10h if AT or ISA
;                            20h if EISA
;                            40h if MCA
;
;       Regs used:      ax, bx
;                       eax, ebx (386 or later)
;
;       Subs called:    cpuvalue

sysvalue proc    near
        push    cx
        push    dx
        push    es

        call    far ptr cpuvalue   ; get the cpu type in al
        mov     cl, al             ; save cpu number (0 to 5)

; Avoid directly reading BIOS ROM, because a few memory managers
; like 386MAX alter bytes at the end of the BIOS.

        push    cx                 ; save cpu number on stack
        mov     ah, 0C0h
        int     15h                ; get BIOS config data es:bx
        pop     cx
        jc      sys_skp1           ; jump if no config support
        mov     dl, es:[bx+2]      ; get model byte
        mov     dh, es:[bx+3]      ; get sub-model byte

        mov     al, 40h            ; assume MCA
        test    byte ptr es:[bx+5], 2
        jnz     sys_Exit           ; exit if MCA
        jmp     sys_skp2

; we only get hear on older PCs in which a memory manager
;  can not be run

sys_skp1:                          ; ok, get BIOS model directly
        mov     ax, 0F000h
        mov     es, ax             ; point into system BIOS
        mov     dx, es:[0FFFEh]    ; get model & sub-model byte

; now use the model and submodel bytes to determine machine

sys_skp2:
        xor     al, al             ; assume PC (al=0)
        cmp     dl, 0FFh
        je      sys_Exit           ; jump if PC
        inc     al                 ; assume XT (al=1)
        cmp     dl, 0FEh
        je      sys_Exit           ; jump if XT
        cmp     dl, 0FBh
        je      sys_Exit           ; jump if XT
        inc     al                 ; assume PC convertible (al=2)
        cmp     dl, 0F9h
        je      sys_Exit           ; jump if convertible
        inc     al                 ; assume PCjr (al=3)
        cmp     dl, 0FDh
        je      sys_Exit           ; jump if PCjr
        inc     al                 ; assume other pre-286 (al=4)
        cmp     cl, 2              ; cpu pre-286 ?
        jb      sys_Exit           ; jump so
        ja      sys_skp3           ; jump if 386 or above

; possible a 286 XT - use the model and sub-model bytes to
;  determine

        mov     al, 8              ; assumption for 286XT
        cmp     dx, 02FCh          ; model code for 286XT ?
        je      sys_exit           ; jump if so

; check if EISA system by looking for the "EISA" string at
;  address F000:FFD9

sys_skp3:
        mov     ax, 0F000h
        mov     es, ax
        mov     al, 10h            ; assume a standard AT/ISA
        cmp     word ptr es:[0FFD9h], 'IE'
        jne     sys_exit           ; jump if not EISA
        cmp     word ptr es:[0FFDBh], 'AS'
        jne     sys_exit           ; jump if not EISA
        mov     al, 20h            ; EISA machine
sys_Exit:
        pop     es
        pop     dx
        pop     cx
        ret
sysvalue endp


;
;    VIDEO TYPE DETECT
;
;       Find the video type, type attributes and possible
;       vendor string.  This routine assumes the display is
;       in a text mode to determine the attribute byte CL.
;
;       The attributes option indicates what attributes a
;       program should use, color, monochrome, or grayscales.
;
;       Called with:    nothing
;
;       Returns:        al = video type
;                             0 = MDA
;                             1 = HGA
;                             2 = CGA
;                             3 = MCGA
;                             4 = EGA
;                             5 = VGA
;                             6 = SVGA
;                             7 = XGA
;                             8 = VESA XGA
;                       ch = attribute type
;                             0 = color
;                             1 = monochrome
;                             2 = gray scale (some MCGA or VGA+)
;                       cl = vendor string present in es:bx
;                             0 = no vendor string
;                             1 = vendor string
;                       es:si = vendor string, zero terminated
;                               (if cl = 1)
;
;       Regs used:      ax, cx, si, es

infobuf db      256 dup (0)        ; buffer for video info
hercstr db      'Hercules', 0      ; Vendor string if Hercules

video_type proc   near
        push    bx
        push    dx
        push    di
        push    bp
        mov     bp, 4              ; bp = temp video type, 4=EGA

; --- check if EGA or later using get video information function

        mov     ah, 12h            ; get video info function
        mov     bh, 5Ah            ; test value
        mov     bl, 10h            ; subfunction EGA+ info
        int     10h
        cmp     bh, 1              ; must be 0, color or 1, mono
        ja      below_EGA          ; jump if not EGA+

; --- it is an EGA or later, so now test for VGA

        push    bx                 ; save color info for later
        mov     ax, 1A00h          ; get display code
        int     10h
        cmp     al, 1Ah            ; is function supported ?
        je      vid_VGA            ; if so, at least a VGA
        jmp     type_found         ; jump if not (must be EGA)

; --- at least a VGA, now test for SVGA

vid_VGA:
        inc     bp                 ; assume VGA (5)
        push    cs
        pop     es
        mov     di, offset infobuf ; buffer for video info
        mov     ax, 4F00h          ; return SVGA info
        int     10h
        cmp     al, 4Fh            ; is function supported ?
        jne     XGA_test           ; jump if not SVGA
        inc     bp                 ; assume SVGA (6)

; --- at least a VGA/SVGA, now test for XGA/VESA XGA

XGA_test:
        mov     ax, 1F00h          ; get XGA information size
        int     10h
        cmp     al, 1Fh            ; is function supported ?
        jne     type_found         ; jump if not, is VGA or SVGA
        mov     bp, 7              ; set to XGA

; --- at least a XGA, now test for VESA XGA

        mov     di, offset infobuf ; buffer for video info
        mov     ax, 4E00h          ; return VESA XGA info
        int     10h
        cmp     ax, 004Eh          ; is function supported ?
        jne     type_found         ; jump if not
        inc     bp                 ; VESA XGA (8)
        jmp     type_found

; --- arrives here if adapter is below an EGA

below_EGA:
        mov     bp, 3              ; assume MCGA (3)
        mov     ax, 1A00h          ; get display code
        int     10h
        cmp     al, 1Ah            ; is function supported ?
        jne     vid_not_MCGA       ; jump if not MCGA
        mov     ah, 0Fh
        int     10h                ; get video mode
        mov     cx, 100h           ; assume mono, no vendor
        cmp     al, 7
        je      vid_chk_gray       ; if mono, check gray
        mov     ch, 0              ; looks like color!
        je      vid_chk_gray       ; check gray scales

; --- not MCGA, test for CGA or MDA/HGA

vid_not_MCGA:
        dec     bp                 ; assume CGA (2)
        mov     ah, 0Fh            ; get video mode
        int     10h
        cmp     al, 7              ; mode 7 monochrome ?
        je      vid_mono_type      ; jump if so
        xor     cx, cx             ; return color, no vendor
        jne     vid_mono_type
        jmp     vid_mono_chk       ; must be CGA

; --- Must be MDA or HGA, so find out which.  The HGA
;     (Hercules Graphics Adapter) toggles an undefined bit on
;     the MDA. Check to see ifthis bit changes state 10 times
;     or more.

vid_mono_type:
        dec     bp                 ; assume HGA (1)
        xor     bl, bl             ; start count at 0
        mov     dx, 3BAh           ; status port on HGA/MDA
        xor     ah, ah             ; ah used for prior status
        mov     cx, 0FFFFh         ; test for a long time
vid_loop:
        in      al, dx             ; read status port
        IODELAY
        and     al, 80h            ; isolate HGA toggle bit
        cmp     al, ah             ; has it changed ?
        je      vid_no_toggle      ; jump if not
        inc     bl                 ; bit changed, increment
        cmp     bl, 10             ; more than 10 toggles ?
        jae     vid_herc           ; if so, it is a Hercules card
vid_no_toggle:
        loop    vid_loop           ; read again until cx 0

; --- falls through if MDA (bit does not toggle)

        dec     bp                 ; set to MDA (0)
        mov     cx, 100h           ; mono attribute, no vendor
        jmp     vid_exit

; --- adapter is Hercules type

vid_herc:
        mov     cx, 101h           ; mono attribute, vendor ok
        push    cs
        pop     es
        mov     si, offset hercstr ; set string to Hercules
        jmp     vid_exit

; --- For MCGA/EGA/VGA/XGA put the attribute type in CL

type_found:
        pop     bx                 ; get attribute type (0 or 1)
        mov     ch, bh
vid_chk_gray:
        mov     ax, 1A00h          ; read display code
        int     10h
        cmp     al, 1Ah            ; check if supported
        jne     vid_string         ; jump if can't be grayscale
        cmp     bl, 7              ; grayscale monitor?
        je      vid_is_gray        ; jump if so
        cmp     bl, 0Bh            ; grayscale monitor?
        je      vid_is_gray        ; jump if so
        cmp     bp, 5              ; VGA or later ?

        mov     ax, 40h            ; BIOS data area
        mov     es, ax
        test    byte ptr es:[89h], 2  ; grayscale summing on ?
        jz      vid_string         ; not grayscale
vid_is_gray:
        mov     ch, 2              ; set ch to grayscale

vid_string:
        xor     cl, cl             ; assume no vendor string
        cmp     bp, 6              ; SVGA ?
        je      vid_skp1           ; jump if so
        cmp     bp, 8              ; VESA XGA ?
        jne     vid_mono_chk       ; jump if not
vid_skp1:
        mov     di, offset infobuf ; get buffer of SVGA info
        mov     si, cs:[di+6]      ; get offset and segment to
        mov     ax, cs:[di+8]      ;   the vendor string
        mov     es, ax             ; es:bx points to string
        inc     cl                 ; vendor string valid

; The last check is made to see if video mode 2 is set,
; indicating monochrome attributes should be used (if we
; haven't already detected monochrome operation)

vid_mono_chk:
        cmp     ch, 0              ; set to color ?
        jne     vid_exit           ; jump if not
        mov     ah, 0Fh
        int     10h                ; get video mode
        and     al, 7Dh
        cmp     al, 0              ; video mode 0 or 2 ?
        jne     vid_exit           ; jump if not
        mov     ch, 1              ; use monochrome attributes

vid_exit:
        mov     ax, bp             ; return adapter type in al
        pop     bp
        pop     di
        pop     dx
        pop     bx
        ret
video_type endp


;
;    CPU IDENTIFICATION SUBROUTINE
;       Identify the CPU type, from 8088 to the Pentium.  Works
;       even if the 386 or later CPU is in V86 mode.  Note that
;       interrupts are enabled at exit, even if they were
;       disabled on entry.  If it is necessary to run this
;       routine with interrupts disabled, just remove all CLI
;       and STI instructions, so long as interrupts are
;       always disabled before running.
;
;       Called with:    nothing
;
;       Returns:        al = CPU type
;                             0 if 8088/8086 or V20/V30
;                             1 if 80186/80188
;                             2 if 80286
;                             3 if 80386
;                             4 if 80486
;                             5 if Pentium
;                       ah =  bit 0 = 0 if CPUID unavailable
;                                     1 if CPUID ok
;                             bit 1 = 0 if not V20/V30
;                                     1 if NEC V20/V30
;
;       Regs used:      ax, bx (all)
;                       eax, ebx (386 or later)
;
;       Subs called:    hook_int6, restore_int6, bad_op_handler

.8086   ; all instructions 8088/8086 unless overridden later

cpuvalue proc    far
        push    cx
        push    dx
        push    ds
        push    es

; 8088/8086 test - Use rotate quirk - All later CPUs mask the CL
;   register with 0Fh, when shifting a byte by cl bits.  This
;   test loads CL with a large value (20h) and shifts the AX
;   register right.  With the 8088, any bits in AX are shifted
;   out, and becomes 0.  On all higher level processors, the
;   CL value of 20h is anded with 0Fh, before the shift.  This
;   means the effective number of shifts is 0, so AX is
;   unaffected.

        mov     cl, 20h            ; load high CL value
        mov     ax, 1              ; load a non-zero value in AX
        shr     ax, cl             ; do the shift
        cmp     ax, 0              ; if zero, then 8088/86
        jne     up186              ; jump if not 8088/86

; V20/V30 test - It is now either a V20/V30 or a 8088.  I'll use
;   another undocumented trick to find out which.  On the 8088,
;   0Fh performs a POP CS.  On the V20/V30, it is the start of
;   a number of multi-byte instructions.  With the byte string
;   0Fh, 14h, C3h the CPU will perform the following:
;               8088/8086               V20/V30
;             pop     cs              set1   bl, cl
;             adc     al, 0C3h

        xor     al, al             ; clear al and carry flag
        push    cs
        db      0Fh, 14h, 0C3h     ; instructions (see above)
        cmp     al, 0C3h           ; if al is C3h then 8088/8086
        jne     upV20
        mov     ax, 0              ; set 8088/8086 flag
        jmp     uP_Exit

upV20:
        pop     ax                 ; correct for lack of pop cs
        mov     ax, 200h           ; set V20/V30 flag
        jmp     uP_Exit

; 80186/80188 test - Check what is pushed onto the stack with a
;   PUSH SP instruction.  The 80186 updates the stack pointer
;   before the value of SP is pushed onto the stack.  With all
;   higher level processors, the current value of SP is pushed
;   onto the stack, and then the stack pointer is updated.

up186:
        mov     bx, sp             ; save the current stack ptr
        push    sp                 ; do test
        pop     ax                 ; get the pushed value
        cmp     ax, bx             ; did SP change ?
        je      up286              ; if not, it's a 286+
        mov     ax, 1              ; set 80186 flag
        jmp     uP_Exit

; 80286 test A - We'll look at the top four bits of the EFLAGS
;   register.  On a 286, these bits are always zero.  Later
;   CPUs allow these bits to be changed.  During this test,
;   We'll disable interrupts to ensure interrupts do not change
;   the flags.

up286:
        cli                        ; disable interrupts
        pushf                      ; save the current flags

        pushf                      ; push flags onto stack
        pop     ax                 ; now pop flags from stack
        or      ax, 0F000h         ; try and set bits 12-15 hi
        push    ax
        popf                       ; set new flags
        pushf
        pop     ax                 ; see if upper bits are 0

        popf                       ; restore flags to original
        sti                        ; enable interrupts
        test    ax, 0F000h         ; were any upper bits 1 ?
        jnz     up386              ; if so, not a 286

; 80286 test B - If the system was in V86 mode, (386 or higher)
;   the POPF instruction causes a protection fault, and the
;   protected mode software must emulate the action of POPF. If
;   the protected mode software screws up, as occurs with a
;   rarely encountered bug in Windows 3.1 enhanced mode, the
;   prior test may look like a 286, but it's really a higher
;   level processor. We'll check if the protected mode bit is
;   on.  If not, it's guaranteed to be a 286.

.286P                              ; allow a 286 instruction
        smsw    ax                 ; get machine status word
        test    ax, 1              ; in protected mode ?
        jz      is286              ; jump if not (must be 286)

; 80286 test C - It's very likely a 386 or greater, but it is
;   not guaranteed yet.  There is a small possibility the system
;   could be in 286 protected mode so we'll do one last test. We
;   will try out a 386 unique instruction, after vectoring the
;   bad-opcode interrupt vector (int 6) to ourselves.

        call    hook_int6          ; do it!
        mov     [badoff], offset upbad_op  ; where to go if bad
.386
        xchg    eax, eax           ; 32 bit nop (bad on 286)

        call    restore_int6       ; restore vector
        jmp     up386              ; only gets here if 386
                                   ;  or greater!

; Interrupt vector 6 (bad opcode) comes here if system is a
;   80286 (assuming the 286 protected mode interrupt 6 handler
;   will execute the bad-opcode interrupt).

upbad_op:
        call    restore_int6
is286:
        mov     ax, 2              ; set 80286 flag
        jmp     uP_Exit

; 80386 test - Bit 18 in EFLAGS is not settable on a 386, but is
;   changeable on the 486 and later CPUs.  Bit 18 is used to
;   flag alignment faults. During this test, we'll disable
;   interrupts to ensure no interrupt will change any flags.

.386                               ; allow 386 instructions

up386:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 40000h        ; toggle bit 18
        push    eax
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed
        jnz     up486              ; changed, so 486 or later
        mov     ax, 3              ; set 80386 flag
        jmp     uP_Exit

; 80486 test - Bit 21 in EFLAGS is not settable on a 486, but is
;   changeable on the Pentium CPU.  If bit 21 is changeable, it
;   indicates the CPU supports the CPUID instruction.  It's
;   amazing it's only taken 10 years to implement the CPUID
;   instruction, which should have been included from the start!
;   During this test, we'll disable interrupts to ensure no
;   interrupt will change any flags.

up486:
        cli                        ; disable interrupts
        pushfd                     ; push flags to look at
        pop     eax                ; get eflags
        mov     ebx, eax           ; save for later
        xor     eax, 200000h       ; toggle bit 21
        push    eax
        popfd                      ; load modified eflags to CPU
        pushfd                     ; push eflags to look at
        pop     eax                ; get current eflags
        push    ebx                ; push original onto stack
        popfd                      ; restore original flags
        sti                        ; enable interrupts
        xor     eax, ebx           ; check if bit changed
        jnz     upPentium          ; changed, it's a Pentium
        mov     ax, 4              ; set 80486 flag
        jmp     uP_Exit

; Pentium - It's possible the CPUID instruction may appear on
;   other CPU chips, so run the CPUID instruction to see what
;   CPU type it indicates.  The CPUID returns a family number
;   0 to 5 for the processor type.  As of this date, only the
;   Pentium supports the CPUID instruction and it is assigned
;   type 5.

upPentium:
        push    ecx                ; CPUID changes eax to edx
        push    edx
        mov     eax, 1             ; get family info function
        CPUID                      ; macro for CPUID instruction
        and     eax, 0F00h         ; find family info
        shr     eax, 8             ; move to al
        mov     ah, 1              ; set flag that CPUID ok
        pop     edx
        pop     ecx

up_Exit:
        pop     es
        pop     ds
        pop     dx
        pop     cx
        ret
cpuvalue endp
.8086                              ; return to 8086 instructions


;
;    HOOK INTERRUPT 6
;       Save the old interrupt 6 vector and replace it with
;       a new vector to the bad_op_handler.  Vectors are handled
;       directly without using DOS.
;
;       Called with:    nothing
;
;       Returns:        vector hooked
;                       old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Regs used:      none

hook_int6 proc    near
        push    ax
        push    cx
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     ax, es:[6*4]       ; get offset of int 6
        mov     cx, es:[6*4+2]     ; get segment
        mov     es:[6*4], offset bad_op_handler
        mov     word ptr es:[6*4+2], seg bad_op_handler
        sti                        ; enable interrupts
        mov     [old_int6_seg], cx ; save original vector
        mov     [old_int6_off], ax
        pop     es
        pop     cx
        pop     ax
        ret
hook_int6 endp


;
;    RESTORE INTERRUPT 6
;       Restore the previously saved old interrupt 6 vector.
;       Vectors handled directly without using DOS.
;
;       Called with:    old vector stored at
;                         ds:[old_int6_seg]
;                         ds:[old_int6_off]
;
;       Returns:        vector restored
;
;       Regs used:      none

restore_int6 proc    near
        push    ax
        push    cx
        push    dx
        mov     cx, [old_int6_seg] ; get original vector
        mov     dx, [old_int6_off]
        push    es
        xor     ax, ax
        mov     es, ax
        cli                        ; disable interrupts
        mov     es:[6*4], dx       ; restore original int 6
        mov     es:[6*4+2], cx
        sti                        ; enable interrupts
        pop     es
        pop     dx
        pop     cx
        pop     ax
        ret
restore_int6 endp


;
;    BAD OFFSET INTERRUPT HANDLER
;       If a bad opcode occurs (80286 or later) will come here.
;       The saved BADOFF offset is used to goto the routine
;       previously stored in BADOFF.
;
;       In a few cases, it is also used for double faults. A few
;       instructions (RDMSR & WRMSR) can issue a double fault if
;       not supported, so well come here as well.
;
;       Called with:    cs:[badoff] previously set
;
;       Returns:        returns to address stored in badoff


bad_op_handler proc far
        push    ax
        push    bp
        mov     ax, cs:[badoff]
        mov     bp, sp
        mov     ss:[bp+4], ax      ; insert new return offset
        pop     bp
        pop     ax
        iret
bad_op_handler endp


cseg    ends

;================================================== stack ======

stacka  segment para stack

        db      192 dup (0)

stacka  ends

        end     start

