
;
;                            KEYSCAN
;
;
;    This program demonstrates the lowest level interactions
;    with the keyboard and keyboard controller.  It also
;    shows the Kscan codes between the keyboard and controller
;    and the scan codes translated by the controller and fed
;    normally to the BIOS interrupt 9 handler.
;
;    KEYSCAN performs the following actions:
;       Slows the keyboard repeat rate to the slowest allowed
;       Turns on all the keyboard LEDs (just to show how
;         it's done)
;       Displays the command byte we write for translation mode
;       Shows one of two types of scan codes when any key is
;         pressed.  The default (no command line option) shows
;         the typical scan code when a key is pressed (this
;         is what is sent to INT 9 normally). Use the command
;         line option -k to show the Kscan codes which come
;         from the keyboard, before translated by the keyboard
;         controller.
;       Escape by pressing the Esc key or 50 key presses
;       Normal keyboard operation is restored
;       The LED state is restored
;       Keyboard repeat rate is set to 24 cps, with a 500 ms
;         delay before repeat starts
;
;    For all computers equipped with a 8042 style keyboard
;    controller (not for PC/XT).
;
;    (c) Copyright 1994, 1996  Frank van Gilluwe
;    All rights reserved.
;
;    V2.00 - Adds the option for Kscan codes or Scan codes.
;            Keyboard repeat rate slowed during operation.
;            Restores keyboard LED state on exit
;            Shows the Escape key scan code when pressed
;            Exit if 50 keys pressed and released or Escape
;            Improved KEYBOARD_WRITE to work with latest PCs

include undocpc.inc


cseg    segment para public
        assume  cs:cseg, ds:cseg, ss:stacka

keyscan         proc near

message1 db     CR, LF
         db     'KEYBOARD INFORMATION AND SCAN CODES'
         db     '            v2.00 (c) 1994, 1996 FVG'
         db     CR, LF
         db     ''
         db     ''
         db     CR, LF
         db     '  To exit, press Escape', CR, LF, CR, LF, '$'

message2 db     '  Displays the normal scan code for each'
         db     ' key pressed.', CR, LF
         db     '  Use command line option -k to see the'
         db     ' untranslated scan codes.'
         db     CR, LF, CR, LF, '$'

message3 db     '  Displays the untranslated Kscan code for'
         db     ' each key pressed.', CR, LF, CR, LF, '$'

message4 db     'New keyboard command byte = '
commandb db     '  ', CR, LF, CR, LF, '$'

message5 db     'Returned scan code = '
scanbytN db     '  ', CR, LF, '$'

message6 db     'Returned Kscan code = '
scanbytK db     '  ', CR, LF, '$'

message7 db     'Keyboard read timeout error', CR, LF, '$'

message8 db     CR, LF
         db     'Complete - 50 keys pressed and released'
crlf     db     CR, LF, '$'

message9 db     CR, LF
         db     'Complete - Escape key pressed', CR, LF, '$'

cmd_line db     0                  ; 'K' if no translation

start:
        xor     bl, bl
        cmp     byte ptr ds:[80h], 1 ; any characters on line ?
        jbe     no_options         ; jump if not
        mov     bl, ds:[83h]       ; get option on cmd line
no_options:
        mov     ax, cs
        mov     ds, ax
        mov     es, ax
        and     bl, 0DFh           ; convert to upper case
        mov     [cmd_line], bl     ; save option
        OUTMSG  message1           ; display initial message
        cmp     [cmd_line], 'K'    ; Translation or not?
        je      no_translate
        OUTMSG  message2           ; normal translation
        jmp     slow_rate

no_translate:

        OUTMSG  message3           ; show Kscan codes

; although not necessary, we'll change the repeat rate to the
; slowest rate so it is easy to see the key pressed and
; release codes

slow_rate:
        mov     ax, 305h           ; set typematic rate
        mov     bx, 031Fh          ; use 1 sec delay, 2 cps
        int     16h                ; do it!

set_LEDs_on:
        mov     bl, 7              ; turn all LEDs on
        call    setLEDs

; Send command to turn off keyboard interrupt IRQ1 (bit 0=0) so
;   the BIOS will not handle anything from the keyboard.
;   If Kscan mode, set keyboard translation off (bit 6=0)

        cli                        ; disable interrupts
        mov     bl, 60h            ; set command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        mov     al, 24h            ; turn off use of IRQ 1
                                   ;  (int 9), no translation
        cmp     [cmd_line], 'K'    ; Kscan mode ?
        je      load_cmd           ; jump if so
        mov     al, 64h            ; translation ok
load_cmd:
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display

; flush any remaining keys out of the buffer

flush_all:
        in      al, 64h            ; get status
        test    al, 1              ; any keys to read ?
        jz      read_command
        call    keyboard_read      ; read until buffer empty
        jmp     flush_all


; Now read the just programmed command byte and display it

read_command:
        mov     bl, 20h            ; get command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        call    keyboard_read      ; read command byte into al
        or      ah, ah             ; valid ?
        jnz     error              ; jump if not
        mov     bx, offset commandb
        call    hex                ; convert to ascii
        OUTMSG  message4           ; display new command byte

        mov     dx, 100            ; maximum key press and
                                   ;   releases before exit

; Display each key pressed scan code screen or the release of
;   the escape key exits program

next_key:
        call    keyboard_read      ; read command byte into al
        or      ah, ah             ; valid ?
        jnz     next_key           ; loop if no key

display_key:
        push    ax                 ; save key for later
        push    dx
        cmp     [cmd_line], 'K'    ; Kscan type?
        je      display_kscan

        mov     bx, offset scanbytN
        call    hex                ; convert to ascii
        OUTMSG  message5           ; display scan information
        jmp     display_next

display_kscan:
        mov     bx, offset scanbytK
        call    hex                ; convert to ascii
        OUTMSG  message6           ; display Kscan information

display_next:
        pop     dx
        pop     ax

        mov     bl, 76h            ; Escape key Kscan code
        cmp     [cmd_line], 'K'
        je      check_if_esc
        mov     bl, 1              ; Escape key scan code
check_if_esc:
        cmp     al, bl             ; Escape key ?
        je      escape             ; exit if so
        dec     dx
        jnz     next_key           ; if under 100, get next
        OUTMSG  message8           ; display done
        jmp     done

; Escape pressed - so restore controller to normal operation

escape:
        OUTMSG  message9           ; display done
done:
        mov     bl, 60h            ; set command byte function
        call    keyboard_cmd       ; activate
        call    error_cmd          ; display if error (ah=1)
        mov     al, 45h            ; reset to normal
        call    keyboard_write     ; send command to keyboard
        call    error_write        ; if error, display
        jmp     exit


; Display keyboard read timeout error message

error:
        OUTMSG  message7           ; display error message

exit:
        sti
        mov     bl, 0AEh           ; enable keyboard
        call    keyboard_cmd       ; do-it
        call    error_cmd          ; display if error (ah=1)

; return the keyboard LEDs to the proper state

        mov     ax, 40h
        mov     es, ax
        mov     bl, es:[97h]       ; get the real LED flags
        call    setLEDs

; reset the repeat rate to something closer to normal

        mov     ax, 305h           ; set typematic rate
        mov     bx, 102h           ; use 500 msec delay, 24 cps
        int     16h                ; do it!

        mov     ah, 4Ch
        int     21h                ; exit
keyscan endp


;
;    SET LEDs
;       Send 3 bits from BL to the keyboard LEDs (handy for
;       debugging TSRs). Does NOT update the keyboard flags. The
;       next LED update will restore the 3 keyboard LEDs. In the
;       unlikely event that an LED update is in progress, the
;       update is skipped.
;
;           40:97h = keyboard_flags2
;                 bit 7 = 1 if keyboard transmit error occurred
;                 bit 6 = 1 if LED update is in progress
;                 bit 5 = 1 if resend received
;                 bit 4 = 1 if acknowledgment received
;
;       This routine also assumes the standard int 9 BIOS
;       keyboard handler maintains keyboard status in
;       keyboard_flags2 at 40:97h.
;
;       Called with:    bl, bit 0=1 to turn on Scroll Lock LED
;                               1=1 to turn on Num Lock LED
;                               2=1 to turn on Caps Lock LED
;                               other bits 3-7 ignored
;
;       Regs Used:      ax

setLEDs proc    near
        push    es
        mov     ax, 40h
        mov     es, ax                  ; es points to BIOS data
        cli                             ; ints off while update
        test    byte ptr es:[97h], 40h  ; already updating ?
        jnz     setLED_return3          ; return if so

        or      byte ptr es:[97h], 40h  ; set update in-progress
        mov     al, 0EDh                ; Update LED command
        call    keyboard_write          ; send keyboard command
        test    byte ptr es:[97h], 80h  ; xmit error ?
        jnz     setLED_return1          ; exit if xmit error
        mov     al, bl
        and     al, 7                   ; only send 3 LED bits
        call    keyboard_write          ; LED data to keyboard
        test    byte ptr es:[97h], 80h  ; xmit error ?
        jz      setLED_return2          ; jump if not
setLED_return1:
        mov     al, 0F4h                ; enable keyboard
        call    keyboard_write          ;  since error occurred
setLED_return2:
        and     byte ptr es:[97h], 3Fh  ; error off & update
setLED_return3:
        sti                             ; enable interrupts
        pop     es
        ret
setLEDs endp


;
;    KEYBOARD_READ
;       read a byte from the keyboard into al (port 60h).
;
;       Called with:    nothing
;
;       Returns:        if ah=0, al=byte read from keyboard
;                       if ah=1, no byte ready after timeout
;
;       Regs Used:      al

keyboard_read   proc    near
        push    cx
        push    dx

        xor     cx, cx             ; counter for timeout (64K)
key_read_loop:
        in      al, 64h            ; keyboard controller status
        IODELAY
        test    al, 1              ; is a data byte ready ?
        jnz     key_read_ready     ; jump if now ready to read
        loop    key_read_loop

        mov     ah, 1              ; return status - bad
        jmp     key_read_exit

key_read_ready:
        push    cx                 ; delay routine needed for
        mov     cx, 16             ;   MCA Type 1 controller.
key_read_delay:                    ;   Insures a 7 uS or longer
        IODELAY
        loop    key_read_delay     ;   Assumes CPU is 80486,
        pop     cx                 ;   66Mhz or slower

        in      al, 60h            ; now read the byte
        IODELAY
        xor     ah, ah             ; return status - ok
key_read_exit:
        pop     dx
        pop     cx
        ret
keyboard_read   endp



;
;    KEYBOARD_WRITE
;       Send byte AL to the keyboard controller (port 60h).
;       Assumes no BIOS interrupt 9 handler active.
;
;       If the routine times out due to the buffer remaining
;       full, ah is non-zero.
;
;       Called with:    al = byte to send
;                       ds = cs
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
;
;       Regs Used:      ax

keyboard_write  proc    near
        push    cx
        push    dx
        mov     dl, al             ; save data for keyboard

        ; wait until keyboard receive timeout is clear (usually is)

        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop1:
        in      al, 64h            ; get keyboard status
        IODELAY
        test    al, 20h            ; receive timeout occurred?
        jz      kbd_wrt_ok1        ; jump if not
        loop    kbd_wrt_loop1      ; try again
                                   ; fall through
        mov     ah, 1              ; return status - failed
        jmp     kbd_wrt_exit

kbd_wrt_ok1:
        in      al, 60h            ; dispose of anything in buffer

        ; wait for input buffer to clear (usually is)

        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop:
        in      al, 64h            ; get keyboard status
        IODELAY
        test    al, 2              ; check if buffer in use
        jz      kbd_wrt_ok         ; jump if not in use
        loop    kbd_wrt_loop       ; try again
                                   ; fall through, still busy
        mov     ah, 1              ; return status - failed
        jmp     kbd_wrt_exit

        ; write data (temporally stored in DL)

kbd_wrt_ok:
        mov     al, dl
        out     60h, al            ; data to controller/keyboard
        IODELAY

        ; wait until input buffer clear (usually is)

        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop3:
        in      al, 64h            ; get keyboard status
        IODELAY
        test    al, 2              ; check if buffer in use
        jz      kbd_wrt_ok3        ; jump if not in use
        loop    kbd_wrt_loop3      ; try again
                                   ; fall through, still busy
        mov     ah, 1              ; return status - failed
        jmp     kbd_wrt_exit

        ; wait until output buffer clear

kbd_wrt_ok3:
        mov     ah, 8              ; larger delay loop (8 * 64K)
kbd_wrt_loop4:
        xor     cx, cx             ; counter for timeout (64K)
kbd_wrt_loop5:
        in      al, 64h            ; get keyboard status
        IODELAY
        test    al, 1              ; check if buffer in use
        jnz     kbd_wrt_ok4        ; jump if not in use
        loop    kbd_wrt_loop5      ; try again
                                   ; fall through, still busy
        dec     ah
        jnz     kbd_wrt_loop4
kbd_wrt_ok4:
        xor     ah, ah             ; return status ok

kbd_wrt_exit:
        pop     dx
        pop     cx
        ret
keyboard_write  endp


;
;    KEYBOARD_CMD
;       Send a command in register BL to the keyboard controller
;       (port 64h).
;
;       If the routine times out due to the buffer remaining
;       full, ah is non-zero.
;
;       Called with:    bl = command byte
;                       ds = cs
;
;       Returns:        if ah = 0, successful
;                       if ah = 1, failed
;
;       Regs Used:      ax, cx


keyboard_cmd    proc    near
        xor     cx, cx             ; counter for timeout (64K)
cmd_wait:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_send           ; ready to accept command ?
        loop    cmd_wait           ; jump if not
                                   ; fall through, still busy
        jmp     cmd_error

cmd_send:                          ; send command byte
        mov     al, bl
        out     64h, al            ; send command
        IODELAY

        xor     cx, cx             ; counter for timeout (64K)
cmd_accept:
        in      al, 64h            ; get controller status
        IODELAY
        test    al, 2              ; is input buffer full?
        jz      cmd_ok             ; jump if command accepted
        loop    cmd_accept         ; try again
                                   ; fall through, still busy
cmd_error:
        mov     ah, 1              ; return status - failed
        jmp     cmd_exit
cmd_ok:
        xor     ah, ah             ; return status ok
cmd_exit:
        ret
keyboard_cmd    endp


;
;    ERROR_WRITE
;       Check if ah=0, as returned from the keyboard_write
;       routine and display message that a keyboard write failed
;       if ah not zero. A real error handler might replace this
;       routine.
;
;       Regs Used:      ah

error_write     proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_w_exit       ; exit if not
        push    dx
        OUTMSG  kbd_wrt_errmsg     ; error handler goes here
        call    boop
        pop     dx
error_w_exit:
        ret
error_write     endp

kbd_wrt_errmsg  db      'Keyboard_write - Input buffer full'
                db       CR, LF, '$'


;
;    ERROR_CMD
;       Check if ah=0, as returned from the keyboard_cmd routine
;       and display message that a keyboard command failed if
;       ah not zero. A real error handler might replace this
;       routine.
;
;       Regs Used:      ah

error_cmd       proc    near
        cmp     ah, 0              ; did an error occur ?
        je      error_c_exit       ; exit if not
        push    dx
        OUTMSG  kbd_cmd_errmsg     ; error handler goes here
        call    boop
        pop     dx
error_c_exit:
        ret
error_cmd       endp

kbd_cmd_errmsg  db      'Keyboard_cmd - Input buffer full'
                db      CR, LF, '$'


;
;   HEX SUBROUTINE
;       convert the hex number in al into two ascii characters
;       at ptr ds:bx.
;
;       Called with:    al = input hex number
;                       ds:bx = ptr where to put ascii
;
;       Regs Used:      al, bx

hex     proc    near
        push    bx
        mov     bl, al
        and     al, 0Fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bh, al

        mov     al, bl             ; upper nibble
        shr     al, 1
        shr     al, 1
        shr     al, 1
        shr     al, 1
        and     al, 0Fh
        add     al, 90h
        daa
        adc     al, 40h
        daa
        mov     bl, al
        mov     ax, bx
        pop     bx
        mov     [bx], ax           ; transfer ascii bytes
        ret
hex     endp


;
;    BOOP ERROR SOUND SUBROUTINE
;       Send a short tone to the speaker
;
;       Called with:    nothing
;
;       Regs used:      none

boop    proc    near
        push    ax
        push    bx
        push    cx
        in      al, 61h            ; read 8255 port B
        and     al, 0FEh           ; turn off the 8253 timer bit
        mov     bx, 150            ; loop bx times

; begin loops of tone on and tone off to create frequency

bbcycle:
        or      al, 2              ; turn on speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; on cycle time duration
bbeepon:
        loop    bbeepon            ; delay
        and     al, 0FDh           ; turn off the speaker bit
        out     61h, al            ; output to port B
        mov     cx, 300            ; off cycle time duration
bbeepoff:
        loop    bbeepoff           ; delay
        dec     bx
        jnz     bbcycle            ; loop
        pop     cx
        pop     bx
        pop     ax
        ret
boop    endp


cseg    ends

;================================================== stack ======

stacka  segment para stack

        db      192 dup (0)

stacka  ends


        end     start

