	include	mmclock.mac
	title	MM58167A I/O buffer routines -- Copyright 1990 Wales

; ======================================================================
;
; I/O buffer interface for MM58167A clock/calendar.
; (C) Copyright 1990 Richard B. Wales.  All Rights Reserved.
;
; Global variables:
;
;	None.
;
; Global routines:
;
;	ClkToBuf
;		Set a buffer from the real-time clock.
;
;	BufToClk
;		Set the real-time clock from a buffer.

; ======================================================================
;
; Global symbols.

	public	ClkToBuf, BufToClk

; ======================================================================
;
; Routines from other modules called here.

	extrn	GetDate:proc, GetTime:proc, SetDate:proc, SetTime:proc
	extrn	SetBIOS:proc

; ======================================================================
;
; Programming note:
;
;	Since the external routines called from this module make no
;	attempt to save and restore registers, anything of value in a
;	register must be pushed onto the stack before a CALL, and then
;	popped afterwards.  Sometimes -- often because the next routine
;	to be called requires arguments to be in specific registers --
;	a value will be restored back into a different register from the
;	one it was previously.  To help the reader keep track of all
;	this, the comments periodically show which values are in which
;	registers.

; ======================================================================
;
; ClkToBuf
;
;	Reads the date and time from the real-time clock, and copies the
;	clock data to a six-byte I/O buffer.
;
;	The date is read first, then the time.  But if the time turns out
;	to be midnight (00:00), the date is immediately re-read, in case
;	it advanced just after the first reading and before the time was
;	read.
;
;	If the date has not been reset today, it is reset now (same date,
;	but in "canonical" form).  Assuming the system refers to the clock
;	at least once every eight months (!), this should suffice to keep
;	the "month/day" portion of the clock chip (used to track the time
;	since the "long count" date was written to the RAM) within bounds.
;	However, if the current time is 23:59 (just before midnight), this
;	resetting of the date is skipped, so as to avoid any problems with
;	the time turning over to midnight before the date can be rewritten.
;
;	Arguments:
;
;		DX	Initial I/O port address for the clock.
;		ES:DI	Initial address of the buffer.
;
;	Returned values:
;
;		Carry	Set to 0 if successful; 1 if not.

ClkToBuf proc near

	; REGISTERS at this point:
	;   DX = I/O port number
	;   ES = buffer address (segment)
	;   DI = buffer address (offset)

	; We won't need the buffer address for some time yet.
	PUSHM	es, di

	; Get the date.
	push	dx
	call	GetDate			; returns CX (date),
					;         DX (days since reset)
	pop	bp

	; REGISTERS at this point:
	;   CX = date (long count)
	;   DX = # days since date last set
	;   BP = I/O port number

	; Get the time.
	PUSHM	cx, dx, bp
	mov	dx, bp			; I/O port number
	call	GetTime			; returns CX (hours/minutes),
					;         DX (seconds)
	POPM	bp, bx, ax

	; REGISTERS at this point:
	;   AX = date (long count)
	;   BX = # days since date last set
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	;   BP = I/O port number

	; If it's midnight, read the date again, just in case
	; it advanced just after we read it the first time.
	and	cx, cx			; see if it's midnight (00:00)
	jnz	short maybe_reset_date
	PUSHM	cx, dx, bp
	mov	dx, bp
	call	GetDate			; returns CX (date),
					;         DX (days since reset)
	mov	ax, cx
	mov	bx, dx
	POPM	bp, dx, cx

	; REGISTERS at this point:
	;   AX = date (long count)
	;   BX = # days since date last set
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	;   BP = I/O port number

maybe_reset_date:
	; If it's been at least a day since the date was last reset,
	; go ahead and reset it now.  Don't do this if it's almost
	; midnight, though, lest a new day should start in the middle
	; of this routine (resulting in the clock being a day off).
	and	bx, bx			; date reset today?
	jz	short write_buffer	;     yes -- don't reset now
	cmp	cx, (23 shl 8) + 59	; is the time 23:59?
	je	short write_buffer	;     yes -- don't reset now
	PUSHM	ax, cx, dx
	mov	cx, ax			; date
	mov	dx, bp			; port number
	call	SetDate
	POPM	dx, cx, ax

	; REGISTERS at this point:
	;   AX = date (long count)
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	; We don't care about I/O port number or
	; # days since date last set any more.

write_buffer:
	; Write the information to the buffer.
	POPM	di, es
	STOSWR	ax			; date
	STOSWR	cx			; hours/minutes
	STOSWR	dx			; seconds
	ret

ClkToBuf endp

; ======================================================================
;
; BufToClk
;
;	Reads a new date and time from a six-byte I/O buffer, and stores
;	this information in the real-time clock.
;
;	The date is taken from the buffer (and written to the clock RAM)
;	in "long count" form -- i.e., number of days since 1/1/1980.
;
;	If the time to be set is just before midnight (23:59), special
;	care is taken to prevent the time from rolling over to midnight
;	before the date has been set.  This is done by first setting the
;	time to 23:59:00 (with the "seconds" explicitly set to zero);
;	setting the correct date; and finally resetting the time with the
;	correct "seconds" information.
;
;	The new time of day is also given to the BIOS, so that the clock
;	interrupt count can be set properly for IBM compatibility.
;
;	Arguments:
;
;		DX	Initial I/O port address for the clock.
;		DS:SI	Initial address of the buffer.
;
;	Returned values:
;
;		Carry	Set to 0 if successful; 1 if not.

BufToClk proc near

	; REGISTERS at this point:
	;   DX = I/O port number
	;   DS = buffer address (segment)
	;   SI = buffer address (offset)

	; Read the information from the I/O buffer.
	mov	bp, dx			; save I/O port
	LODSWR	bx			; date
	LODSWR	cx			; hours/minutes
	LODSWR	dx			; seconds

	; REGISTERS at this point:
	;   BX = date (long count)
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	;   BP = I/O port number

	; If it's 23:59, take special precautions to make sure the
	; time doesn't roll over before the date can be set.
	cmp	bx, (23 shl 8) + 59	; is the time 23:59?
	je	short careful_write	; if so, be careful

regular_write:
	; Set the time.
	PUSHM	bp, bx, dx, cx
	call	SetTime
	POPM	cx, dx

	; REGISTERS at this point:
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	; The date and I/O port number are still on the stack.

	; Set the time in the BIOS data area.
	call	SetBIOS
	POPM	cx, dx

	; REGISTERS at this point:
	;   CX = date (long count)
	;   DX = I/O port number
	; We don't care about the time any more.

	; Set the date.
	call	SetDate

	; All done.
	ret

careful_write:
	; It's 23:59.  We must avoid a situation where the time of day
	; rolls over to 00:00 before we have finished writing the date;
	; otherwise, the clock would lose a whole day.  To do this, we
	; first set the time (but with seconds equal to zero).  Then we
	; set the date.  Finally, we go back and set the time again --
	; but with the correct seconds information this time.

	; REGISTERS at this point:
	;   BX = date (long count)
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	;   BP = I/O port number

	; Set the time (with seconds = 0).
	PUSHM	bp, bx, cx, dx
	xor	dx, dx			; seconds = 0 for now
	call	SetTime
	POPM	ax, bx, cx, dx

	; REGISTERS at this point:
	;   AX = time (seconds and 1/100 second)
	;   BX = time (hours/minutes)
	;   CX = date (long count)
	;   DX = I/O port number

	; Set the date.
	PUSHM	ax, bx, dx
	call	SetDate
	POPM	bp, cx, dx

	; REGISTERS at this point:
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)
	;   BP = I/O port number
	; We don't care about the date any more.

	; Set the time (with correct seconds this time).
	PUSHM	dx, cx
	call	SetTime
	POPM	cx, dx

	; REGISTERS at this point:
	;   CX = time (hours/minutes)
	;   DX = time (seconds and 1/100 second)

	; Set the time in the BIOS data area.
	call	SetBIOS

	; All done.
	ret

BufToClk endp

code	ends

	end