WITH Ada.Numerics; USE Ada.Numerics;
PACKAGE BODY Geometry IS
------------------------------------------------------------------
--| Body of abstract data type package for geometric figures.
--| Author: Michael B. Feldman, The George Washington University 
--| Last Modified: September 1995                                     
------------------------------------------------------------------

-- Body of abstract data type package for geometric figures.

  -- internal functions, not exported to client. ComputePerimeter
  -- and ComputeArea are used to ensure that all figures are
  -- constructed with these attributes automatically inserted.
  -- The exported selectors Perimeter and Area assume that these
  -- fields have been set by the internal functions.

  FUNCTION ComputePerimeter (OneFig : Figure) RETURN NonNegFloat IS
  -- Pre : The discriminant and characteristics of OneFig are defined.
  -- Post: Returns Perimeter of OneFig.

  BEGIN -- ComputePerimeter   
    
    CASE OneFig.FigShape IS
      WHEN Rectangle =>
        RETURN 2.0 * (OneFig.Width + OneFig.Height);
      WHEN Square =>
        RETURN 4.0 * OneFig.Width;
      WHEN Circle =>
        RETURN 2.0 * Pi * OneFig.Radius;
    END CASE;
    
  END ComputePerimeter;

  FUNCTION ComputeArea (OneFig : Figure) RETURN NonNegFloat IS
  -- Pre : The discriminant and characteristics of OneFig are defined.
  -- Post: Returns Area of OneFig.
    
  BEGIN -- ComputeArea   
    
    CASE OneFig.FigShape IS
      WHEN Rectangle =>
        RETURN OneFig.Width * OneFig.Height;
      WHEN Square =>
        RETURN OneFig.Width ** 2;
      WHEN Circle =>
        RETURN Pi * OneFig.Radius ** 2 ;
    END CASE;

  END ComputeArea;

  -- Exported Operations  

  FUNCTION MakeRectangle (Width, Height : NonNegFloat) RETURN Figure IS

    Result : Figure(FigShape => Rectangle);

  BEGIN -- MakeRectangle

    Result.Height    := Height;
    Result.Width     := Width; 
    Result.Area      := ComputeArea(Result);
    Result.Perimeter := ComputePerimeter(Result);

    RETURN Result;

  END MakeRectangle;

  FUNCTION MakeCircle (Radius : NonNegFloat) RETURN Figure IS

    Result: Figure (FigShape => Circle);

  BEGIN -- MakeCircle

    Result.Radius    := Radius;
    Result.Area      := ComputeArea(Result);
    Result.Perimeter := ComputePerimeter(Result);

    RETURN Result;

  END MakeCircle;

  FUNCTION MakeSquare (Side : NonNegFloat) RETURN Figure IS

    Result: Figure (FigShape => Square);

  BEGIN -- MakeSquare

    Result.Height    := Side;
    Result.Width     := Side;
    Result.Area      := ComputeArea(Result);
    Result.Perimeter := ComputePerimeter(Result);

    RETURN Result;

  END MakeSquare;

  FUNCTION Shape (OneFig : Figure) RETURN FigKind IS

  BEGIN -- Perimeter   
    RETURN OneFig.FigShape; 
  END Shape;    

  FUNCTION Perimeter (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Perimeter   
    RETURN OneFig.Perimeter;
  END Perimeter;

  FUNCTION Area (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Area   
    RETURN OneFig.Area;
  END Area;

  FUNCTION Height (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Height
    CASE OneFig.FigShape IS
      WHEN Rectangle | Square =>
        RETURN OneFig.Height;
      WHEN OTHERS =>
        RAISE ShapeError;
    END CASE;
  END Height;

  FUNCTION Width (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Width 
    CASE OneFig.FigShape IS
      WHEN Rectangle | Square =>
        RETURN OneFig.Width; 
      WHEN OTHERS =>
        RAISE ShapeError;
    END CASE;
  END Width; 

  FUNCTION Side (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Side  
    CASE OneFig.FigShape IS
      WHEN  Square =>
        RETURN OneFig.Height;
      WHEN OTHERS =>
        RAISE ShapeError;
    END CASE;
  END Side;  

  FUNCTION Radius (OneFig : Figure) RETURN NonNegFloat IS

  BEGIN -- Radius
    CASE OneFig.FigShape IS
      WHEN Circle =>
        RETURN OneFig.Radius;
      WHEN OTHERS =>
        RAISE ShapeError;
    END CASE;
  END Radius;

END Geometry;
