; Size of traps in 8086 mode
; stack generates a GPF
;
;
; EMS system implemented by
;
; David Lindauer
;
; gclind01@ulkyvx.louisville.edu
;
; August, 1995 
;
; As part of the FREE-DOS project
;
; v86.asm
;  Function: 8086 emulation within a V86 window.
;    This code assumes it will be run with IPL = 0.  In actual fact
;	I am running it with IPL=3.  This is faster.  The only thing
;	emulated at IPL 3 is the INT instructions, because in idt.asm
;	the I set the interrupt gates to run at ring 0 and this generates
;	an exception
;    LOCK is not fully supported, it should sometimes cause an exception
;    but if it's part of a trapped instruction it won't.
;    HLT has two bugs: first it changes the OCW3 of both PICS
;	without restoring it, second the actual interrupt the halt
;	is waiting on may not occur until the instruction after the halt
;	completes, I haven't read my processor book
;    I don't support OPSIZ on anything but pushf & popf
;    Control ops have to use the out num,al or in al,num form.
;    This only affects ports 84h and 85h, which only deal with
;    communicating with the emulator from V86 mode.
;    This file only supports INT, IRET, pushf(d), popf(d), CLI, STI, HLT.
;    See vm386 for emulation of 386-class protected instructions.
;
;    If you happen to run the stack seg out of bounds you will get
;    an exception 13 instead of an exception 12.
;
	.386P

include segs.asi
include tss.asi
include vm86.asi
include idt.asi
include emuinter.asi

	assume cs:dgroup,ds:dgroup, es:dgroup, fs: ABSDATA

	public bootv86, transfer, emulate, EmuFlags, GFTrapFrame
	public reflect, intframe2
SEG386	segment
	extrn settss:PROC, setidt: PROC, endprotection: PROC, waitforint: PROC
	extrn emuin:PROC, emuout:PROC, emu386: PROC
SEG386	ends
StackSeg segment
	extrn	TOS: BYTE
StackSeg ends

SEG386DATA segment
	extrn userIDT : dword
EmuFlags dd	0	
IPLstate	dd	IPL3		; Set the default IPL state here,
					; this is also the IPL state for
					; emulation purposes
SEG386DATA ends

;
; Return us to dos when we are in VM86 mode
;
SEG8086	segment
backtodos	proc
	mov	edx,offset dgroup:TOS	; TOS is end of program
	add	dx,15			; Make sure we get any trailing bytes
	shr	dx,4			; Calculate len in segs
	mov	ah,31h			; Keep the program.  At this
	int	21h			; point we are returning to DOS
					; while in a VM86 window
backtodos	endp
SEG8086 ends

;
; Emulator starts here
;
SEG386	segment
;
; Prefixes we look for
;
prefixtabstart LABEL BYTE
	db	26h,2eh,36h,3eh,64h,65h,66h,67h,0f0h,0f2h,0f3h
prefixtabend	LABEL BYTE
PREFIXTABSIZE	EQU PrefixTabEnd - PrefixTabStart
;
; Instructions we are trapping
;
EmuTabStart 	LABEL BYTE
	db	06ch,06dh,06eh,06fh
	db	0e4h,0e5h,0e6h,0e7h,0ech,0edh,0eeh,0efh
	db	9ch,9dh,0cch,0cdh,0ceh,0cfh,0f4h,0fah,0fbh
EmuTabEnd	LABEL BYTE
EMUTABSIZE	EQU EmuTabEnd - EmuTabStart
;
; The run-time routines corresponding to the instructions we trap
;
	align
EmuRoutines	LABEL DWORD
	dd	dgroup:endprotection, dgroup:endprotection, dgroup:endprotection
	dd	dgroup:endprotection, dgroup:emu_inal, dgroup:endprotection
	dd	dgroup:emu_outal, dgroup:endprotection, dgroup:endprotection
	dd	dgroup:endprotection, dgroup:endprotection, dgroup:endprotection
	dd	dgroup:emu_pushf, dgroup:emu_popf, dgroup:emu_int3
	dd	dgroup:emu_intx, dgroup:emu_into, dgroup:emu_iret
	dd	dgroup:emu_hlt,dgroup:emu_cli, dgroup:emu_sti
;
; Hardware interrupts come here and get reflected into V86 mode
;  Note that hardware interrupts for the GPF trap are disabled in the
;  IDT so we'll never encounter this while we are emulating.
;
;  Also a variety of 8086 traps which don't have error codes get
;  trapped here, see idt.asm
;
transfer proc
	push	ebp	; Get the EMU stack frame
			;  Note that in IDT.ASM we push the vector
			; number on the stack in the place that the GPF
			; error code would go, thus we can use GPF routines
	mov	ebp,esp ;
	push	eax	; Save regs
	push	edi	;
	push	DSABS	; Point us at 8086 data
	pop	fs	;
	push	DS386	;
	pop	ds	;
	call	intframe2; Push the interrupt frame on the 8086 stack
	mov	al,[GFERR]; Get the vector number
	call	reflect	;  Reflect the vector into 8086 mode
oo2:
	pop	edi	; Pop regs
	pop	eax	;
	pop	ebp	;
	add	esp,4	; Clear the vector num from the stack
	iretd
transfer	endp
;
; Subroutine to set our return address to one of the int vectors
;
reflect	proc
	movzx	eax,al	; Zero extend vector num
	shl	eax,2	; * 4
	add	eax,[userIDT] ; Get base of IDT for emulation
	mov	di,fs:[eax]	; Get IP from real mode vector table
	mov	[GFEIP],di	; 
	mov	di,fs:[eax+2]	; Get CS from real mode vector table
	mov	[GFCS],di	;
	btr	dword ptr [GFFLAGS],IFL; Interrupts always run with IF clear
	btr	dword ptr [GFFLAGS],NT; Nested task bit gets cleared
	btr	dword ptr [GFFLAGS],TF; Trap bit gets cleared
	ret
reflect	endp
;
; Report a GP trapF to the 8086 software
;
GFTrapFrame	proc
	GETFRAME	GFESP,GFSS,edi	; Get stack frame
	mov	ax,[GFFLAGS]		; Get flags
	mov	fs:[edi-2],ax		; Push them
	mov	ax,[GFCS]		; Get CS
	mov	fs:[edi-4],ax		; Push it
	mov	eax,[GFEIP]		; Get EIP
	mov	fs:[edi-6],ax		; Push it
	mov	ax,[GFERR]             ; Get the err code
	mov	fs:[edi-8],ax		; Push it
	RSUB	GFESP,8			; MAke room ( so emulation not exact)
	mov	al,GFTRAP		; GPF trap
	jmp	emu_reflect		; Reflect it & get out
GFTrapFrame	endp
;
; Make an interrupt frame on the 8086 stack
;
intframe	proc
	RADD	GFEIP,dx		; Update EIP for return adr
intframe2:
					; Hardware ints come here
	RSUB	GFESP,6			; Make space
	GETFRAME	GFESP,GFSS,edi	; Get stack frame
	mov	ax,[GFFLAGS]		; Push flags
	and	ax,NOT IPL3 		; Get the IPL state
	or	ax,word ptr [IPLstate]		;
	mov	word ptr fs:[edi+4],ax	;
	mov	ax,[GFCS]		; Push CS
	mov	word ptr fs:[edi+2],ax	;
	mov	ax,[GFEIP]		; Push IP
	mov	word ptr fs:[edi],ax	;
	ret
intframe	endp
;
; Get the emulated flags state to IOPL and the real flags state to
; GFFLAGS
;
save_flags	proc
	push	eax			; Save flags
	and	ax,IPL3			; Get IPL bits
	mov	[IPLstate],eax		; Save user version
	pop	eax			; Restore flags
	and	ax,NOT IPL3		; Get rid of IPL bits
	and	word ptr [GFFLAGS],IPL3	; Get rid of all but IPL bits from flags
	or	[GFFLAGS],ax		; Or the two together
	ret
save_flags	endp
;
; Emulate a pushf
;
emu_pushf	proc
	RADD	GFEIP,dx		; Update EIP for return adr
	GETFRAME	GFESP,GFSS,edi	; Get stack frame
	mov	ax,[GFFLAGS]		; Get flags
	and	ax,NOT IPL3		; Get rid of system version of IPL
	or	eax,[IPLstate]		; Get user version
	bt	[EmuFlags],BOPSIZE	; Check for OPSIZ
	jc	emu_pushfd		; Branch if so
	mov	word ptr fs:[edi-2],ax	; Else push low word of flags
	RSUB	GFESP,2			; Make space
	jmp	emu_exit		; Return to 8086
emu_pushfd:
	mov	word ptr fs:[edi-2],0
	mov	word ptr fs:[edi-4],ax	; Push lo word of flags
	RSUB	GFESP,4			; make space
	jmp	emu_exit		; Return to 8086
emu_pushf	endp
;
; Emulate a popf
;
emu_popf	proc
	RADD	GFEIP,dx		; Update return address
	GETFRAME	GFESP,GFSS,edi	; Get stack frame
	bt	[EmuFlags],BOPSIZE	; Check for OPSIZ
	jc	emu_popfd		; Branch if so
	mov	ax,word ptr fs:[edi]	; Get word flags from stack
	call	save_flags		; Save the flags state
	RADD	GFESP,2			; Clear stack
	jmp	emu_exit		;
emu_popfd:
	mov	eax,dword ptr fs:[edi]	; Get DWorD flags from stack
	call	save_flags		; Save the flags state
	RADD	GFESP,4			; Clear stack
	jmp	emu_exit		;
emu_popf	endp
;
; Emulate debug int
;
emu_int3	proc
	call	intframe		; Make an int frame
	mov	al,3			; Reflect int 3
	jmp	emu_reflect		;
emu_int3	endp
;
; Emulate a numbered int
;
emu_intx	proc
	db	64h	; FS:		; Get the number
	lodsb
	inc	edx			; Update instruction len
	push	eax			; Save number
	call	intframe		; Create an int frame
	pop	eax			; Restore number
	jmp	emu_reflect		; Reflect it
emu_intx	endp
;
; Emulate into
;
emu_into	proc
	bt	word ptr [GFFLAGS],OVFLAG	; Check for overflow
	jnc	noovf			; If not just continue
	call	intframe		; Else make an intframe
	mov	al,4                    ; Reflect int 4
	jmp	emu_reflect
noovf:
	RADD	GFEIP,dx		; Point at next instruction
	jmp	emu_exit		; Return to 8086
emu_into	endp
;
; Emulate an iret
;
emu_iret	proc
	GETFRAME	GFESP,GFSS,edi	; Get stack frame
	mov	ax,word ptr fs:[edi+4]	; Get flags
	call	save_flags		; Save the flags state
	mov	ax,word ptr fs:[edi+2]	; Get CS
	mov	[GFCS],ax
	mov	ax,word ptr fs:[edi]	; Get ip
	mov	[GFEIP],ax
	RADD	GFESP,6			; Empty stack
	jmp	emu_exit
emu_iret	endp
;
; Emulate a cli
;
emu_cli	proc
	btr	dword ptr [GFFLAGS],IFL	; Clear the interrupt flag
	RADD	GFEIP,dx		; Point to next instruction
	jmp	emu_exit		; Exit
emu_cli	endp
;
; Emulate an sti
;
emu_sti	proc
	bts	dword ptr [GFFLAGS],IFL	; Set the interrupt flag
	RADD	GFEIP,dx		; Point to next instruction
	jmp	emu_exit		; Exit
emu_sti	endp
;
; Emulate a halt
;
emu_hlt	proc
	bt	dword ptr [GFFLAGS],IFL
	jnc	emu_end
	call	waitforint		; Wait for int
	RADD	GFEIP,dx		; Step past this
	jmp	emu_exit		; Exit
emu_end:
	hlt
emu_hlt	endp
;
; The next two are for control purposes.  At the moment the only
; trap will be on ports 84h and ports 85h, we'll use these to control
; the emulator system
;
; Emulate in al,portnum
;
emu_inal	proc
	inc	edx		; Inc past instruction
	RADD	GFEIP,dx	;
	db	64h		; FS:
	lodsb        		; Get prot num
	mov	dl,al		; dl = port number
	call	emuin		; Emulate the in
	jmp	emu_exit	; Exit to V86
emu_inal	endp
;
; Emulate out portnum,al
;
emu_outal	proc
	inc	edx		; Inc past instruction
	RADD	GFEIP,dx	;
	db	64h		; FS:
	lodsb        		; Get prot num
	mov	dl,al		; dl = port number
	mov	al,[GFEAX]	; al = data	
	call	emuout		;emulate the out
	jmp	emu_exit	; Exit to V86
emu_outal	endp
;
; GPF fault... we emulate from here
;
emulate	proc
	push	ebp    			; Get EMU stack frame
	mov	ebp,esp
	push	eax			; Push regs.  Order is important
	push	ebx			;
	push	ecx			;
	push	edx			;
	push	esi			;
	push	edi			;
	mov	ax,DS386		; Set segs up reasonably
	mov	ds,ax                   ;
	mov	es,ax			;
	push	DSABS			; Point at 8086 data
	pop	fs			;
	mov	[EmuFlags],(1 SHL BINEMU); Clear flags & put us in emulator
	bt	dword ptr [GFFLAGS],TF  ; Check if we have to single step
	jnc	nosingle		; Branch if not
	bts	[EmuFlags],XTF		; Set a flag so we'll know
nosingle:
	GETFRAME	GFEIP,GFCS,esi	; Get the code frame
	sub	edx,edx			; Bytes in instruction = 0
prefixlp:
	cld				; One line of code = 30 hrs of debugging...
	db	64h	; FS:		; Get a byte
	lodsb				;
	inc	edx			; Count it
	mov	ecx,PREFIXTABSIZE	; Length of prefix tab
	mov	edi,offset DGROUP:PrefixTabEnd-1; End of prefix tab
	std				; Direction = down
	repne	scasb			; Scan table for a match
	jnz	checkop			; No match, check opcode table
	bts	[EmuFlags],ecx		; Else set the bit corresponding
					; To that prefix and get the next byte
	jmp	prefixlp	

checkop:
	mov	ecx,EMUTABSIZE		; Emulation table size
	mov	edi,offset DGROUP:EmuTabEnd-1	; Table end
 	repne	scasb			; Scan it, dir is still down
	jnz	GFTrapFrame		; Trap resulted from something other
					; than an instruction
	cld				; Some ops read another byte
	shl	ecx,2			; Get offset into the EMU routines table
	jmp	[ECX+EmuRoutines]	; Branch to the emulation routine
;
; Emulation routines jump to one of these exit points
;
emu_reflect:
	call	reflect			; Replace CS:IP with an int vector
emu_exit:
	btr	[EmuFlags],XTF		; Check if we have to take int 1
	jnc	emu_done		; Done if not
	call	intframe2		; Else make an int frame for it
	mov	al,1			; Take the int 1 trap
	call	reflect
emu_done:
	pop	edi			; Pop regs
	pop	esi
	pop	edx
	pop	ecx
	pop	ebx
	pop	eax
	pop	ebp
	add	esp,4			; Get rid of error code
	btr	[EmuFlags],BINEMU	; Not in emulator any more
	iretd				; back to V86
emulate	endp
;
; Put is in VM86 mode the first time and run a program
;
bootv86	proc
	call	settss			; Set up TSS with a ring 0 stack
					; and appropriate IO bitmap
	call	setidt			; Set up IDT
	sub	eax,eax			; Push 0 for seg regs,
	push	eax			; 8086 prog will have to init
	push	eax			;
	push	eax			;
	push	eax			;
	mov	ebx,DGROUP		; SS:SP = our local stack
	push	ebx			;
	mov	eax,offset DGROUP:TOS	;
	push	eax			;
	pushfd				; Clear NT and RF bits and
	pop	eax			; put as at the base IPL
	btr	eax,NT			;
	btr	eax,RF			;
	and	eax, NOT IPL3		;
	or	eax, [IPLstate]		;
	push	eax			; These are system flags
	popfd				;
	bts	eax,VM			; Prog is going to run in V86 mode
	push	eax			;
	push	ebx			; Code seg
	mov	eax,offset DGROUP:backtodos; Routine to run
	push	eax			;
	iretd				; Put us in V86 mode
bootv86	endp
SEG386	ends
	end