; sqrt86.asm
comment ^

        This is a sample program to calculate the integral portion of a
        square root of a 32-bit positive integer.  It runs on any 80XXX
        processor.

        written on Sat  09-16-1995  by Ed Beroset
                original 80386 version

        modified on Fri  09-22-1995  by Ed Beroset
                ported to 8086

        released to the public domain by the author
^
        IDEAL
        MODEL small
        p8086

        STACK   400h

        DATASEG
alpha   dd      100000                  ; an arbitrary number
result  dw      ?

        CODESEG
proc  main
        STARTUPCODE                     ;
        mov     ax,[word alpha]         ; load 'em up
        mov     dx,[word alpha+2]       ;
        call    sqrt                    ; call our sqrt function
        mov     [result],bx             ; demonstrate how to save result
        EXITCODE                        ; bail
endp  main

;/***************************************************************************
;
;   Name:
;       sqrt
;
;   Purpose:
;       Calculate the integral portion of the square root of a
;       positive integer.
;
;   Algorithm:
;
;       The way this is done is rather simple.  First, use a series of
;       32-bit shifts to scan for the bit number of the most significant
;       bit.  The result of this operation is the log (base 2) of our
;       original.  We use this information to generate a good first
;       guess for our algorithm, since 2**((log[2](X))/2) = sqrt(X).
;       That is, if we halve the log of a number and raise the result to
;       the base of our log, we get the square root of the original.
;
;       Once we have that number, we know that it must be the most
;       significant digit in our answer.  Since this is true, we can
;       successively approximate the answer by "guessing" each of the
;       bits and testing by squaring the guess.
;
;       This is not meant to be an optimal solution, but more of an
;       intellectual curiosity.  If I were actually programming a sqrt
;       function on any computer with an NPU, I'd use FSQRT and be done
;       with it. I haven't actually timed them both yet; totalling
;       execution (e.g. data sheet) timings can lead to incorrect
;       conclusions.
;
;   Entry:
;
;     DX:AX = the number whose square root we're seeking
;
;   Register usage within the routine:
;
;        AX = scratch register (for multiplications)
;        BX = current guess
;        CX = current bit number
;        DX = scratch register (for multiplications)
;        SI = bit mask for determining successive guesses
;     DI:BP = copy of original number
;
;   Exit:
;
;        BX = calculated square root
;     DX:AX = square of number in AX
;     DI:BP = target value
;
;   Trashed:
;       CX, SI
;
;***************************************************************************/
proc sqrt
        mov     di,dx                   ; save dx:ax in di:bp
        mov     bp,ax                   ;
        mov     cx,31                   ; max bit count FindSetBit:
        shl     ax,1                    ; do 32-bit shift left
        rcl     dx,1                    ;   of DX:AX
        jc      GotSetBit               ; if we found MSB, we're done
        loop    FindSetBit              ; keep going
GotSetBit:
        shr     cx,1                    ; log2(alpha)/2
        mov     bx,1                    ; set up our initial guess
        shl     bx,cl                   ; now shift bit into position
        mov     si,bx                   ; make a copy for our next guess
        inc     cx                      ; to get the right number of loops
top:
        shr     si,1                    ; slide our bit mask to next bit
        or      bx,si                   ; set bit in our answer
        mov     ax,bx                   ; save copy in ax
        mul     bx                      ; DX:AX = BX*BX
        cmp     dx,di                   ; test high half of answer
        jb      over                    ; if over, no need for more tests
        jnz     SkipLoTest              ; if not equal, skip low test
        cmp     ax,bp                   ; test low half of answer
        jz      done                    ;   if equal, we're done
        jb      over                    ;   if guess low, leave bit set
SkipLoTest:
        xor     bx,si                   ;   guess too big, clear bit
over:
        loop    top                     ; keep going for more
done:
        ret
        endp sqrt

        END
