/****************************************************************

   	gr_mgr.c   	Implementation of Graphics Interface 
              		for Bellcore MGR (tm)

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

****************************************************************/

#include "stdio.h"
#include "bw.h"
#include "gr.h"
#include "mgrterm.h"      /* for Bellcore MGR (tm) */

#ifdef __STDC__
#include "malloc.h"
#else
extern char * malloc();
#endif

#define X_OFFSET   12
#define Y_OFFSET   12
#define MAX_X      640
#define MAX_Y      450

#define	CHECK_PARAMS	TRUE

#define MGR_BLSIZE  32

#define MGR_FONT0	10
#define MGR_FONT1	11
#define MGR_FONT2	12
#define MGR_FONT3	13
#define MGR_FONT4	14

#define	MGR_DRAWMAIN	0		/* mgr primary screen */
#define MGR_DRAWHIDE	1		/* mgr secondary (hidden) screen */
#define MGR_DRAWGRID	2		/* mgr grid screen */
#define MGR_DRAWSTART   6		/* start assignable screens */

#define	TINYFONT			/* try to use tiny font */

int   gr_screens = 2;
int   gr_colors = 2;
int   gr_pxsize = 287;
int   gr_pysize = 428;
int   gr_ismouse = TRUE;
int   gr_clipping = FALSE;
int   gr_blitting = TRUE;
int   gr_saving = TRUE;

static int   mgr_wno;

static int   mgr_cltarget;
static int   mgr_border;
static int   mgr_hidden = FALSE;
static int   mgr_curwind = MGR_DRAWMAIN;
static int   mgr_xblocks, mgr_yblocks;
static int   mgr_clx1, mgr_cly1, mgr_clx2, mgr_cly2;
struct gr_window *mgr_grwind;

extern tam_imsize();

/****************************************************************

   gr_init()

****************************************************************/

gr_init( grwindow, font_path )
   struct gr_window *grwindow;
   char *font_path;
   {
   register int x, y, i;
   int target_x, target_y;
   static xmax, ymax;
   static char _host[ 32 ];

   mgr_grwind = grwindow;

   /* set up MGR system */

   m_setup( M_FLUSH );
   m_ttyset();

   /* get maximum sizes from MGR system */

   get_param( _host, &xmax, &ymax, &mgr_border );

   /* calculate target size for window, x axis */

   if ( ( xmax - ( X_OFFSET * 2) ) < MAX_X )
      {
      target_x = xmax - ( X_OFFSET * 2 );
      }
   else
      {
      target_x = MAX_X;
      }

   /* calculate target size for window, y axis */

   if ( ( ymax - ( Y_OFFSET * 2) ) < MAX_Y )
      {
      target_y = ymax - ( Y_OFFSET * 2 );
      }
   else
      {
      target_y = MAX_Y;
      }

   /* get new window for this application */

   mgr_wno = m_makewindow( X_OFFSET, Y_OFFSET, target_x, target_y );
   m_selectwin( mgr_wno );
   mgr_grwind->xmax = target_x - ( ( mgr_border * 2 ) + 1 );
   mgr_grwind->ymax = target_y - ( ( mgr_border * 2 ) + 1 );

   /* load fonts */


   x = m_loadfont( MGR_FONT4, "gal12x20r.fnt" );
#ifdef	OLD_DEBUG
      fprintf( stderr, "load font %d returned %d \n", MGR_FONT4, x );
      getchar();
#endif
   x = m_loadfont( MGR_FONT3, "cour8x16r.fnt" );
#ifdef	OLD_DEBUG
      fprintf( stderr, "load font %d returned %d \n", MGR_FONT3, x );
      getchar();
#endif
   x = m_loadfont( MGR_FONT2, "cour7x14r.fnt" );
#ifdef	OLD_DEBUG
      fprintf( stderr, "load font %d returned %d \n", MGR_FONT2, x );
      getchar();
#endif
   x = m_loadfont( MGR_FONT1, "cour6x12r.fnt" );
#ifdef	OLD_DEBUG
      fprintf( stderr, "load font %d returned %d \n", MGR_FONT1, x );
      getchar();
#endif
   x = m_loadfont( MGR_FONT0, "tiny4x7r.fnt" );
#ifdef	OLD_DEBUG
      fprintf( stderr, "load font %d returned %d \n", MGR_FONT0, x );
      getchar();
#endif


   gr_font( GR_PRIMARY, F_DEFAULT, 10 );

   /* set to absolute window coordinate mode */

   m_setmode( M_ABS );
/*   m_setmode( M_NOWRAP ); */

   /* calculate number of blocks necessary for fill patterns */

   mgr_xblocks = ( ( target_x ) / MGR_BLSIZE ) + 1;
   mgr_yblocks = ( ( target_y ) / MGR_BLSIZE ) + 1;

#ifdef OLD_DEBUG
   fprintf( stderr, "x size = %d, x blocks = %d \n", mgr_grwind->xmax, mgr_xblocks );
   fprintf( stderr, "y size = %d, y blocks = %d \n", mgr_grwind->ymax, mgr_yblocks );
   kb_rx();
#endif

   /* get scratchpad bitmaps */

   m_bitcreate( MGR_DRAWGRID, mgr_xblocks * MGR_BLSIZE, 
      mgr_yblocks * MGR_BLSIZE );

#ifdef USEFILL
   m_bitcreate( MGR_DRAWFILL, mgr_xblocks * MGR_BLSIZE, 
      mgr_yblocks * MGR_BLSIZE );
#endif

   m_bitcreate( MGR_DRAWHIDE, target_x, target_y );

#ifdef	OLD_DEBUG
   fprintf( stderr, "Bitmaps created. \n" );
#endif

   /* signon message */

   fprintf( stderr, "Implementation of Bywater Graphics (gr) Standard\n" );
   fprintf( stderr, "for Bellcore MGR (tm). This implementation copyright\n" );
   fprintf( stderr, "(c) 1990, Ted A. Campbell, Bywater Software \n" );
   fprintf( stderr, "\nInitializing: " );

#ifdef USEFILL

   /* fill first block of fill bitmap with solid pattern */

#ifdef OLD_DEBUG
   fprintf( stderr, "Initializing SOLID fill: " );
#endif
   m_func( B_SET );
   for ( y = 0; y < MGR_BLSIZE; ++y )
      {
      fprintf( stderr, "." );
      m_lineto( MGR_DRAWFILL, 0, y, MGR_BLSIZE, y );
      }

   /* now replicate the block pattern throughout the fill bitmap */

#ifdef OLD_DEBUG
   fprintf( stderr, "\nCopying FILL images: " );
#endif
   m_func( B_SRC );
   for ( y = 0; y < mgr_yblocks; ++y )
      {
#ifdef OLD_DEBUG
      fprintf( stderr, "\nFILL line %d: ", y );
#endif
      fprintf( stderr, "." );
      for ( x = 0; x < mgr_xblocks; ++x )
         {
         if ( ( x == 0 ) && ( y == 0 ))
            {
            }
         else
            {
            m_bitcopyto( x * MGR_BLSIZE, y * MGR_BLSIZE, 
               MGR_BLSIZE, MGR_BLSIZE, 
               0, 0, MGR_DRAWFILL, MGR_DRAWFILL );
            }
         }
      }

#endif		/* USEFILL */

   /* fill grid bitmap with solid pattern */

#ifdef OLD_DEBUG
   fprintf( stderr, "\nInitializing GRID fill: " );
#endif
   for ( y = 0; y < MGR_BLSIZE; y += 2 )
      {

#ifdef OLD_DEBUG
      fprintf( stderr, "\nLine %d: ", y );
#endif
      for ( x = 0; x < MGR_BLSIZE; x += 2 )
         {
         fprintf( stderr, "." );
         m_func( B_SET );
         m_lineto( MGR_DRAWGRID, x, y, x, y );
         m_func( B_CLEAR );
         m_lineto( MGR_DRAWGRID, x + 1, y, x + 1, y );
         m_func( B_CLEAR );
         m_lineto( MGR_DRAWGRID, x, y + 1, x, y + 1 );
         m_func( B_SET );
         m_lineto( MGR_DRAWGRID, x + 1, y + 1, x + 1, y + 1 );
         }
      }

   /* now replicate the block pattern throughout the fill bitmap */

#ifdef OLD_DEBUG
   fprintf( stderr, "\nCopying GRID images: " );
#endif
   m_func( B_SRC );
   for ( y = 0; y < mgr_yblocks; ++y )
      {
#ifdef OLD_DEBUG
      fprintf( stderr, "\nGRID line %d: ", y );
#else
      fprintf( stderr, "." );
#endif
      for ( x = 0; x < mgr_xblocks; ++x )
         {
         if ( ( y == 0 ) && ( x == 0 ))
            {
            }
         else
            {
#ifdef OLD_DEBUG
            fprintf( stderr, "." );
#endif
            m_bitcopyto( x * MGR_BLSIZE, y * MGR_BLSIZE, 
               MGR_BLSIZE, MGR_BLSIZE, 
               0, 0, MGR_DRAWGRID, MGR_DRAWGRID );
            }
         }
      }

#ifdef OLD_DEBUG
   fprintf( stderr, "\nGRID bitmap filled. \n" );
#endif

   /* turn cursor off */

   m_setcursor( CS_INVIS );

   /* move cursor to 0,0 to be safe */

   m_move( 0, 0 );

   /* note that system is initailized */

   mgr_grwind->initialized = TRUE;
   return TRUE;
   }

/****************************************************************

   gr_deinit()

****************************************************************/

gr_deinit( )
   {
   m_bitdestroy( MGR_DRAWGRID );
   m_bitdestroy( MGR_DRAWHIDE );
   m_destroywin( mgr_wno );
   m_ttyreset();
   }

/****************************************************************

   gr_cls()

****************************************************************/

gr_cls( screen )
   int screen;
   {
   gr_rectangle( screen, 0, 0, mgr_grwind->xmax, mgr_grwind->ymax, 
      BLACK, SOLID );
   }


/****************************************************************

   gr_pixel()

****************************************************************/

gr_pixel( screen, x, y, color )
   int screen;
   int x, y;
   int color;
   {

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_pixel(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   mgr_color( color );
   if ( screen == MGR_DRAWMAIN )
      {
       m_line( x, mgr_y( y ), x, mgr_y( y ) );
      }
   else
      {
      m_lineto( screen, x, mgr_y( y ), x, mgr_y( y ) );
      }
   }
         
/****************************************************************

   gr_line()

****************************************************************/

gr_line( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {

#if	CHECKPARAMS
   if ( ( x1 < 0 ) || ( x1 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_line(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   mgr_color( color );
   if ( screen == MGR_DRAWMAIN )
      {
      m_line( x1, mgr_y( y1 ),
         x2, mgr_y( y2 ) );
      }
   else
      {
      m_lineto( screen, x1, mgr_y( y1 ),
         x2, mgr_y( y2 ) );
      }
   }

/****************************************************************

   gr_text()

****************************************************************/

gr_text( screen, x, y, string, foreground, background )
   int screen;
   int x, y;
   int foreground, background;
   char *string;
   {

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > mgr_grwind->xmax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): x value is %d", x );
      kb_rx();
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > mgr_grwind->ymax ))
      {
      fprintf( stderr, "ERROR: [pr:] gr_text(): y value is %d", y );
      kb_rx();
      return BW_ERROR;
      }
#endif

   m_func( B_SET );
/*   m_move( 0, 0 ); */

   if ( foreground == WHITE )
      {
      m_clearmode( M_STANDOUT );
      }
   else
      {
      m_setmode( M_STANDOUT );
      }
   if ( screen == MGR_DRAWMAIN )
      {
      m_moveprint( x, mgr_y( y ), string );
      }
   else
      {
      m_stringto( screen, x, mgr_y( y ), string );
      }
   }

/****************************************************************

   gr_strlen()

****************************************************************/

unsigned int
gr_strlen( string )
   char *string;
   {
   return ( strlen( string ) * mgr_grwind->fxsize );
   }

/****************************************************************

   gr_rectangle()

****************************************************************/

gr_rectangle( screen, x1, y1, x2, y2, color, style )
   int screen;
   int x1, y1, x2, y2;
   int color, style;
   {

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_rectangle(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   switch( style )
      {
      case SOLID:
         if ( color == WHITE )
            {
            m_func( B_SET );
            }
         else
            {
            m_func( B_CLEAR );
            }
         m_bitcopyto( x1, mgr_y( y2 ), 
            abs( x2 - x1 ), abs( y2 - y1 ),
            x1, mgr_y( y2 ),
            screen, MGR_DRAWGRID ); 
         break;
      case GRID:
      case HATCH:
         m_func( B_SRC );
         m_bitcopyto( x1, mgr_y( y2 ),
            abs( x2 - x1 ), abs( y2 - y1 ),
            x1, mgr_y( y2 ),
            screen, MGR_DRAWGRID ); 
         break;
      case HOLLOW:
      default:
         mgr_color( color );
         if ( screen == MGR_DRAWMAIN )
            {
            m_line( x1, mgr_y( y1 ), 
               x2, mgr_y( y1 ) );
            m_line( x2, mgr_y( y1 ), x2, 
               mgr_y( y2 ) );
            m_line( x1, mgr_y( y2 ), 
               x2, mgr_y( y2 ) );
            m_line( x1, mgr_y( y2 ), 
               x1, mgr_y( y1 ) );
            }
         else
            {
            m_lineto( screen, x1, mgr_y( y1 ), 
               x2, mgr_y( y1 ) );
            m_lineto( screen, x2, mgr_y( y1 ), x2, 
               mgr_y( y2 ) );
            m_lineto( screen, x1, mgr_y( y2 ), 
               x2, mgr_y( y2 ) );
            m_lineto( screen, x1, mgr_y( y2 ), 
               x1, mgr_y( y1 ) );
            }
         break;
      }
   }

/****************************************************************

   gr_circle()

****************************************************************/

gr_circle( screen, x, y, radius, color, style )
   int screen;
   int x, y, radius;
   int color, style;
   {
   register int c;

#if	CHECK_PARAMS
   if ( ( x < 0 ) || ( x > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): x value is %d", x );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y < 0 ) || ( y > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_circle(): y value is %d", y );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   mgr_color( color );
   switch( style )
      {
      case SOLID:
      case HATCH:
      case GRID:
         for ( c = 0; c <= radius; ++c )
            {
            if ( screen == MGR_DRAWMAIN )
               {
               m_ellipse( x, mgr_y( y ), 
                  c, ( c * gr_pxsize ) / gr_pysize );
               }
            else
               {
               m_ellipseto( screen, x, mgr_y( y ), 
                  c, ( c * gr_pxsize ) / gr_pysize );
               }
            }
         break;
      case HOLLOW:
      default:
         if ( screen == MGR_DRAWMAIN )
            {
            m_ellipse( x, mgr_y( y ), 
               radius, ( radius * gr_pxsize ) / gr_pysize );
            }
         else
            {
            m_ellipseto( screen, x, mgr_y( y ), 
               radius, ( radius * gr_pxsize ) / gr_pysize );
            }
         break;
      }
   }

/****************************************************************

   gr_ellipse()

****************************************************************/

gr_ellipse( screen, x, y, x_radius, y_radius, mode, color, style )
   int screen;
   int x, y, x_radius, y_radius;
   int mode, color, style;
   {
   }
         
/****************************************************************

   gr_clip()

****************************************************************/

gr_clip( screen, mode, x1, y1, x2, y2 )
   int screen;
   int mode;
   int x1, y1, x2, y2;
   {

#ifdef  DEBUG
   if ( ( x1 < 0 ) || ( x1 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_clip(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   return BW_ERROR;
   }

/****************************************************************

   gr_font()

****************************************************************/

gr_font( screen, type, rq_height )
   int screen;
   int type, rq_height;
   {
   static int x_fontno;

   if ( rq_height > 18 )
      {
      x_fontno = MGR_FONT4;
      mgr_grwind->fysize = 20;
      mgr_grwind->fxsize = 12;
      }
   else if ( rq_height > 15 )
      {
      x_fontno = MGR_FONT3;
      mgr_grwind->fysize = 16;
      mgr_grwind->fxsize = 8;
      }
   else if ( rq_height > 13 )
      {
      x_fontno = MGR_FONT2;
      mgr_grwind->fysize = 14;
      mgr_grwind->fxsize = 7;
      }
#ifdef	TINYFONT
    else if ( rq_height > 9 )
      {
      x_fontno = MGR_FONT1;
      mgr_grwind->fysize = 12;
      mgr_grwind->fxsize = 6;
      }
   else 
      {
      x_fontno = MGR_FONT0;
      mgr_grwind->fysize = 7;
      mgr_grwind->fxsize = 4;
      }
#else
   else
      {
      x_fontno = MGR_FONT1;
      mgr_grwind->fysize = 12;
      mgr_grwind->fxsize = 6;
      }
#endif      

   mgr_curspark();			/* get cursor out of the way */
   m_font( x_fontno );

   }

/****************************************************************

   gr_blit()

****************************************************************/

gr_blit( src, dst, x1, y1, x2, y2 )
   int src, dst;
   int x1, y1, x2, y2;
   {

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( src == dst )
      {
      sprintf( bw_ebuf, "[pr:] gr_blit(): src == dst" );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   if ( src == dst )
      {
      return BW_ERROR;
      }

   m_func( B_SRC );
   m_bitcopyto( x1, mgr_y( y2 ), x2 - x1, y2 - y1, 
      x1, mgr_y( y2 ), dst, src );
   }

/****************************************************************

   gr_imsave()

****************************************************************/

gr_imsave( screen, mode, x1, y1, x2, y2, image )
   int screen; 
   int mode, x1, y1, x2, y2;
   int *image;
   {

#if	CHECK_PARAMS
   if ( ( x1 < 0 ) || ( x1 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): x1 value is %d", x1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( x2 < 0 ) || ( x2 > mgr_grwind->xmax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): x2 value is %d", x2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y1 < 0 ) || ( y1 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): y1 value is %d", y1 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
   if ( ( y2 < 0 ) || ( y2 > mgr_grwind->ymax ))
      {
      sprintf( bw_ebuf, "[pr:] gr_save(): y2 value is %d", y2 );
      bw_error( bw_ebuf );
      return BW_ERROR;
      }
#endif

   if ( mode == TRUE )
      {
      *image = mgr_getbno();                /* assign a number */

#ifdef OLD_DEBUG
      fprintf( stderr, "\nDEBUG: Window number is %d", *i );
      kb_rx();
#endif

      m_bitcreate( *image, abs( x2 - x1 ), abs( y2 - y1 ) );

      m_func( B_SRC );
      m_bitcopyto( x1, mgr_y( y2 ), 
         abs( x2 - x1 ), abs( y2 - y1 ),
	 x1, mgr_y( y2 ), *image, MGR_DRAWMAIN );
      }
   else
      {
#ifdef OLD_DEBUG
      fprintf( stderr, "\nDEBUG: Window number is %d", *image );
      kb_rx();
#endif
      m_func( B_SRC );
      m_bitcopyto( x1, mgr_y( y2 ),
         abs( x2 - x1 ), abs( y2 - y1 ),
	 x1, mgr_y( y2 ), MGR_DRAWMAIN, *image );
      }
   }

/****************************************************************

	gr_imfree()

****************************************************************/

gr_imfree( image )
   int image;
   {
   m_bitdestroy( image );
   }

/****************************************************************

   gr_mouse()

****************************************************************/

gr_mouse( mode, x, y, buttons )
   int mode;
   int *x, *y;
   int *buttons;
   {
   static int m_pending = FALSE, x_save, y_save;
   static int m_prevr;
   int c, r;

   switch( mode )
      {
      case SAMPLE:
         if ( m_pending == TRUE )
            {
            *x = x_save;
            *y = y_save;
            return TRUE;
            }
         r = mgr_mouse( x, y );
         *y = mgr_y( *y );
         if ( r == TRUE )
            {
            m_pending = TRUE;
            x_save = *x;
            y_save = *y;
            }
         return r;
         break;
      case WAIT:
         if ( m_pending == TRUE )
            {
            m_pending = FALSE;
            *x = x_save;
            *y = y_save;
            return TRUE;
            }
         c = FALSE;
         while( c == FALSE )
            {
            c = r = mgr_mouse( x, y );
            *y = mgr_y( *y );
            }
         m_pending = FALSE;
         m_prevr = r;
         return TRUE;
         break;
      default:
#ifdef	DEBUG
         sprintf( bw_ebuf, "gr_mouse() received mode %d", mode );
         bw_error( bw_ebuf );
#endif
         return BW_ERROR;
         break;
      }
   
   }

mgr_mouse( x, y )
   int *x, *y;
   {
   static int r = 0, x_prevr = 0;

   r = get_mouse( x, y );
#ifdef	OLD_DEBUG
   sprintf( bw_ebuf, "r = %d", r );
   gr_text( GR_PRIMARY, 12, 12, bw_ebuf, WHITE, BLACK );
#endif
   if ( r == x_prevr )
      {
      return FALSE;
      }
   x_prevr = r;
   return TRUE;
   }

mgr_y( oldyval )
   int oldyval;
   {
   return (( mgr_grwind->ymax  ) - oldyval );
   }

mgr_color( color )
   int color;
   {
   if ( color == WHITE )
      {
      m_func( B_SET );
      }
   else
      {
      m_func( B_CLEAR );
      }
   }

mgr_getbno()
   {
   static int x = MGR_DRAWSTART;

   ++x;
   return x;
   }

mgr_curspark()
   {
   m_movecursor( 0, 85 );
   }

