/********* Listing 1 *************************** PALETTE.C ********** 
 *                                                                  * 
 * DESCRIPTION:  Functions for manipulating EGA/VGA color palettes  * 
 * COMPILER(s):  Borland 2.0+, MS 5.0+, Zortech C/C++ 2.0+          * 
 * AUTHOR:       Marv Luse, Autumn Hill Software, Inc.              * 
 * COPYRIGHT:    Use freely but acknowledge source.                 * 
 *******************************************************************/

#include "stdlib.h"
#include "string.h"
#include "dos.h"

#include "palette.h"

/*------------------------------------------------------------------*/
/* Local variables...                                               */
/*------------------------------------------------------------------*/

/* address of ROM save table */
static unsigned long old_save_ptr;

/* area in RAM to establish a new table */
static unsigned long save_ptr_tbl[7];

/* palette register values for the default EGA palette */
static unsigned char ega_default_pal[16] =
{
     0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x14, 0x07,
     0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F
};

/* area in RAM for parameter save area - first 17 bytes contains */
/* a copy of the 16 palette registers plus the overscan register */
static unsigned char ega_save_area[256];

/*------------------------------------------------------------------*/
/* Get or Set address of Video Save Pointer Table...                */
/*------------------------------------------------------------------*/

/* save pointer is at 0x40:0xA8 */
#define SAVE_POINTER ((void far *) (0x004000A8))

void far * get_save_ptr( void )
{
    return( SAVE_POINTER );
}

void set_save_ptr( void far * ptbl )
{
    *((unsigned long far *) SAVE_POINTER) = (unsigned long) ptbl;
}

/*------------------------------------------------------------------*/
/* Setup our own Save Table and Parameter Save Area...              */
/*------------------------------------------------------------------*/

void init_save_area( int init_palette )
{
    int i;
    char *s;
    char far *fs;

    /* save address of ROM table */
    old_save_ptr = (unsigned long) get_save_ptr();

    /* copy save ptr table from ROM to RAM */
    for (s = (char *)save_ptr_tbl, fs = get_save_ptr(), i = 0; i < 28; i++)
        *s++ = *fs++;

    /* update save pointer to point to our copy */
    set_save_ptr( save_ptr_tbl );

    /* the 2nd table entry points to parameter save area */
    save_ptr_tbl[1] =
        (unsigned long) ((unsigned char far *) ega_save_area);

    /* zero out the new parameter save area */
    memset( ega_save_area, 0, 256 );

    /* if requested, initialize the default palette */
    if( init_palette )
        for( i=0; i<16; i++ )
            set_ega_pal_register( i, ega_default_pal[i] );
}

/*------------------------------------------------------------------*/
/* Restore address of EGA Save Table to default ROM table...        */
/*------------------------------------------------------------------*/

void restore_save_area( int reinit_palette )
{
    int i;

    /* restore address of default ROM table */
    set_save_ptr( (void far *) old_save_ptr );

    /* if requested, reinitialize the default palette */
    if( reinit_palette )
        for( i=0; i<16; i++ )
            set_ega_pal_register( i, ega_default_pal[i] );
}

/*------------------------------------------------------------------*/
/* Returns nonzero if active display is VGA color...                */
/*------------------------------------------------------------------*/

int is_vga_color( void )
{
    union REGS r;
    static int already_called = 0;
    static int is_vga = 0;

    /* Have we been called before?                             */
    if ( already_called )
        return( is_vga );

    /*  Int 10h function 1Ah is supported on the VGA, but not  */
    /*  the EGA.  If supported, a known value will be returned */
    /*  in the BL register, which will be 08h for VGA color.   */
    /*  To determine that the function is truly supported,     */
    /*  BL must be initialized to a known invalid value.       */

    r.h.ah = 0x1A; /* function 1Ah */
    r.h.al = 0x00; /* subfunction 00h */
    r.h.bl = 0x99; /* invalid return value */

    int86( 0x10, &r, &r );

    already_called = 1;
    is_vga = (r.h.bl == 0x08) ? 1 : 0;

    return( is_vga );
}

/*------------------------------------------------------------------*/
/* Returns nonzero if active display is EGA color...                */
/*------------------------------------------------------------------*/

int is_ega_color( void )
{
    union REGS r;
    static int already_called = 0;
    static int is_ega = 0;

    /*  Since the VGA BIOS is a superset of the EGA BIOS, we   */
    /*  must first rule out the possibility of a VGA display.  */

    if( is_vga_color() )
        return( 0 );

    /* Have we been called before?                             */
    if ( already_called )
        return( is_ega );

    /* We now attempt to invoke an EGA-only BIOS function to   */
    /* rule out the CGA, MDA, and other display hardware.      */

    r.h.ah = 0x12; /* function 12h */
    r.h.bl = 0x10; /* sub function 10h */

    int86( 0x10, &r, &r );

    /* If the function is unsupported, BL remains unchanged    */

    if( r.h.bl == 0x10 )
        return( 0 );

    /* The function is supported, so this is an EGA display.   */
    /* If the value returned in BH is zero, then it's color.   */

    already_called = 1;
    is_ega = (r.h.bh == 0) ? 1 : 0 ;

    return( is_ega );
}

/*------------------------------------------------------------------*/
/* Get contents of specified EGA palette register...                */
/*------------------------------------------------------------------*/

int get_ega_pal_register( int reg_no )
{
    /* get copy of register maintained by the BIOS */
    return( (int) ega_save_area[reg_no] );
}

/*------------------------------------------------------------------*/
/* Set contents of specified EGA palette register...                */
/*------------------------------------------------------------------*/

void set_ega_pal_register( int reg_no, int reg_val )
{
    union REGS r;

    r.h.ah = 0x10;                    /* function 10h */
    r.h.al = 0x00;                    /* subfunction 00h */
    r.h.bl = (unsigned char) reg_no;  /* register number */
    r.h.bh = (unsigned char) reg_val; /* color value */

    int86( 0x10, &r, &r );
}

/*------------------------------------------------------------------*/
/* Get contents of specified VGA palette register...                */
/*------------------------------------------------------------------*/

int get_vga_pal_register( int reg_no )
{
    union REGS r;

    r.h.ah = 0x10;                   /* function 10h */
    r.h.al = 0x07;                   /* subfunction 07h */
    r.h.bl = (unsigned char) reg_no; /* register number */

    int86( 0x10, &r, &r );

    return( (int) r.h.bh );
}

/*------------------------------------------------------------------*/
/* Set contents of specified VGA palette register...                */
/*------------------------------------------------------------------*/

void set_vga_pal_register( int reg_no, int reg_val )
{
    union REGS r;

    r.h.ah = 0x10;                    /* function 10h */
    r.h.al = 0x00;                    /* subfunction 00h */
    r.h.bl = (unsigned char) reg_no;  /* register number */
    r.h.bh = (unsigned char) reg_val; /* color value */

    int86( 0x10, &r, &r );
}

/*------------------------------------------------------------------*/
/* Get contents of specified VGA DAC register...                    */
/*------------------------------------------------------------------*/

void get_vga_dac_register( int reg_no, unsigned char rgb[] )
{
    union REGS r;

    r.h.ah = 0x10;   /* function 10h */
    r.h.al = 0x15;   /* subfunction 15h */
    r.x.bx = reg_no; /* register number */

    int86( 0x10, &r, &r );

    rgb[0] = r.h.dh; /* 6-bit red component */
    rgb[1] = r.h.ch; /* 6-bit grn component */
    rgb[2] = r.h.cl; /* 6-bit blu component */
}

/*------------------------------------------------------------------*/
/* Set contents of specified VGA DAC register...                    */
/*------------------------------------------------------------------*/

void set_vga_dac_register( int reg_no, unsigned char rgb[] )
{
    union REGS r;

    r.h.ah = 0x10;   /* function 10h */
    r.h.al = 0x10;   /* subfunction 10h */
    r.x.bx = reg_no; /* register number */
    r.h.dh = rgb[0]; /* 6-bit red component */
    r.h.ch = rgb[1]; /* 6-bit grn component */
    r.h.cl = rgb[2]; /* 6-bit blu component */

    int86( 0x10, &r, &r );
}