
%DEPTH 0	;no include-numbers
%TABSIZE 4
%PAGESIZE 64,118	;col=20(line+code)+78(text)+20(tasm-bug)
%NOTRUNC
;============================================================================
;
; PSEUDOCD.ASM
; cdrom-drive simulating device driver, reads a cdrom image on a hard disk
; This file is part of the
;
; PseudoCD package (version 02)
; Copyright (C) 1997  C.Kulms
;
;   This is free software; you can redistribute it and/or modify it
;   under the terms of the GNU General Public License as published by
;   the Free Software Foundation; version 2 of the License.
;
;   This software is distributed in the hope that it will be useful,
;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;   GNU General Public License for more details.
;
;   You should have received a copy of the GNU General Public License
;   along with this software; if not, write to the Free Software
;   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;
%NOLIST
;
; This file contains two versions: one uses the BIOS to read the cdrom-image
; (prefered method for drives supported by the BIOS); the other uses the
; disk's device-driver (needed for BIOS-less SCSI drives etc).
; The symbol USEDRIVER decides which version is assembled.
; Default is the BIOS-version (USEDRIVER undefined).
;
IFDEF USEDRIVER
DISPLAY "generate the driver-version"
NAME PSCDDDRV
%TITLE "PSCDDDRV - cdrom-simulating device-driver"
%LIST
;
; Usage of driver-version (in config.sys):
; DEVICE=[path]PSCDDDRV.SYS /devicename /disk /sector
; where: devname = device-name to use (legal filename, max. 8 characters)
;        disk = DOS-disk where the cd-image file is stored
;        sector = first sector of cd-image (hex, relative to start of disk)
;
ELSE
DISPLAY "generate the bios-version"
NAME PSCDBIOS
TITLE "PSCDBIOS - cdrom-simulating device-driver"
%LIST
;
; Usage of BIOS-version (in config.sys):
; DEVICE=[path]PSCDBIOS.SYS /devicename /driveparameter /sector
; where: devname = device-name to use (legal filename, max. 8 characters)
;        driveparameter = low-level information about the physical drive
;                         the image-file is stored on
;                         (bios-drive<<16|heads<<8|sec/track, hex)
;        sector = first sector of cd-image
;                 (hex, relative to start of phys. drive)
;
ENDIF
;============================================================================

IDEAL		;sorry, but I prefer this
%NOINCL		;don't list any include files
%NOMACS		;avoid macro-expansion in listing
%NOCONDS	;skip conditional blocks not done in listing
NOJUMPS		;no automatic jump-constructs
LOCALS @@	;use "@@" as local-variable prefix

P386		;use 386-opcodes (shifts and e-regs)

;Please note this code isn't optimized anyway - neither for size nor for speed.

STACKSIZE	EQU 256		;size of stack in words

INCLUDE "DRIVER.INC"	;strucs/equs for device drivers
INCLUDE "CDROM.INC"		;strucs/equs for cdrom drivers

IFDEF USEDRIVER
INCLUDE "DPB.INC"	;Disk Parameter Block definition
ELSE
;we need some constants not defined in the .INC files
BIOS_READ	EQU 02h		;int 13h,'read sectors'
BIOS_WRITE	EQU 03h		;int 13h,'write sectors'
ENDIF

MACRO mcPUTS str
; displays a '$'-terminated string at stdout
; ds:str->string
; modified: flags
	push	ax
	push	dx
	mov		dx,OFFSET str
	mov		ah,09h
	int		21h
	pop		dx
	pop		ax
ENDM

%SUBTTL "data area"
%NEWPAGE
;----------------------------------------------------------------------------
; driver's segment
;----------------------------------------------------------------------------
SEGMENT CODE PARA PRIVATE USE16
		ASSUME CS:CODE		;tasm needs this ...
		ASSUME DS:NOTHING	;they changes too often to assume anything
		ASSUME ES:NOTHING	;

;device-header (extended cdrom version), initialized with
;drv.Attrib = character,IOCTL-support,OPEN/CLOSE/RM
;drv.Strat = offset of 'strategy'-routine
;drv.Intr = offset of 'interrupt'-routine
;drv.Name = empty string (overwritten by initialisation)
;drv.Rsvd = 0
;drv.Drive = 0 (will be set by MSCDEX)
;drv.Units = 1
MyDrvHdr	sCDDRVHDR <-1,0C800h,strategy,entry,"        ",0,0,1>

;CDROM driver signature (MSCDEX does not check for it, but Win95 does)
DB "MSCD00"

;copyright notice
DB "PseudoCD v02  (C) Copyright 1997  C.Kulms"

;pointer to request-header
;(values given at strategy-routine call)
LABEL	ReqHdrAdr DWORD
ReqHdrOff	DW 0	;offset of request-header
ReqHdrSeg	DW 0	;segment of request-header

;data where and how cd-image is found
ImgSector	dd ?	;first sector of image
IFDEF USEDRIVER
;use the disk's device-driver, so some information about it is needed
DiskNumber	DB ?			;DOS disk-number
LABEL	DiskDrvHdr DWORD	;pointer to disk driver's header
DiskDrvOff	DW ?
DiskDrvSeg	DW ?
LABEL	DiskDrvStr DWORD	;disk driver's strategy entry point
DiskStrOff	DW ?
DiskStrSeg	DW ?
LABEL	DiskDrvInt DWORD	;disk driver's interrupt entry point
DiskIntOff	DW ?
DiskIntSeg	DW ?
RWREQ	sRWREQ ?			;request-header used for reading sectors
DiskMedia	DB ?			;disk's media byte (from DPB)
DiskUnit	DB ?			;disk's unit number (from DPB)
ELSE
;the BIOS is used, so some drive-specific information is needed
Drive		DB ?			;drive-number as needed by bios (hd>=80h)
DriveSPT	DB ?			;sectors per track
DriveHeads	DB ?			;heads
DriveHSPT	DW ?			;heads*sectors_per_track (sector-size of track)
ENDIF

;data of the cd-rom
HeadLoc		dd 0									;current head-location
DevStatus	rCDSTAT <0,0,0,0,0,0,1,0,0,0,0,0,1,0>	;current device-status
			;(initial: closed,unlocked,cooked only,read only,no audio play,
			;no interleaving,prefetching,no audio channel,hsg only,
			;disk_present,no r-w sub-channels)
;----------------------------------------------------------------------------
		DW STACKSIZE-1 DUP(5353h)
toss	DW 5353h		;top of own stack

LABEL	MyStack DWORD	;used to load my ss:sp in a save way (lss)
MySP	DW toss
MySS	DW ?

LABEL	OldStack DWORD	;holds callers stack
OldSP	DW ?
OldSS	DW ?

;----------------------------------------------------------------------------
%SUBTTL "read-sector function"
%NEWPAGE
PROC rdsec NEAR
;Reads a number of consecutive hd-sectors.
;call:	es:di->buffer
;		edx=first sector
;		cl=number of sectors to read
;return: carry on error
;modified: ax,flags
%NOLIST
;
; This function is the main difference of the two versions. The bios-version
; calls int 13h, function 02h to read the sectors of the image; while the
; driver-version makes a read-request to the disk's device-driver.
; All other differences are in the initialisation routine, because different
; data has to be setup for this two types of disk-read.
; The driver-version is more general: each kind of disk-drive could be
; accessed this way (in theory). But it is not clear if it is save - the MSPRM
; does not say anything about re-usage of block-device drivers (the re-usage of
; character-device drivers is explicitly noted). Also a DOS-call which is not
; allowed for a device-driver's initialisation-routine by the MSPRM has to done
; to get the parameters of the disk (see below).
; NOTE: THE DRIVER-VERSION MAY CAUSE YOUR SYSTEM TO CRASH AT BOOT IF USED WITH
; DRIVES MANAGED BY DOS'S INTERNAL DRIVERS!
;
%LIST
IFDEF USEDRIVER

	push	ds						;save ds
	push	cx						;save cx
	mov		ax,cs					;ensure ds=cs
	mov		ds,ax					;

	;build request-header for 'read sectors'
	mov		[RWREQ.rwFunction],DRV_READ	;set function READ
	mov		al,[DiskUnit]			;get unit number
	mov		[RWREQ.rwUnit],al		;set unit number
	mov		al,[DiskMedia]			;get media-byte
	mov		[RWREQ.rwMedia],al		;set media-byte
	cmp		edx,0FFFFh				;sector >= 0FFFFh?
	jl short @@lowsector
	mov		[RWREQ.rwStartSec],0FFFFh	;indicates huge sector number
	mov		[RWREQ.rwHugeSec],edx	;set (32bit) start-sector
	jmp short @@secok
@@lowsector:
	mov		[RWREQ.rwStartSec],dx	;set (16bit) start-sector
@@secok:
	xor		ch,ch					;cx=number of sectors to read
	mov		[RWREQ.rwCount],cx		;set number of sectors
	mov		[RWREQ.rwBufferSeg],es	;store pointer to buffer (segment)
	mov		[RWREQ.rwBufferOff],di	;store offset

	;call the strategy-routine with es:bx->request block
	push	bx						;save bx
	push	es						;save es
	mov		bx,cs					;ensure es=cs
	mov		es,bx
	mov		bx,OFFSET RWREQ			;es:bx->request header for read
	call [DiskDrvStr]				;far call to disk's driver strategy routine
	call [DiskDrvInt]				;far call to disk's driver interrupt routine
	pop		es						;restore es
	pop		bx						;restore bx

	;check error- and done-bit
	mov		ax,[RWREQ.rwStatus]		;get status
	pop		cx						;restore cx
	pop		ds						;restore ds
	or		ax,ax					;error bit set?
	js short @@error
	cmp		ah,1					;done bit set? (and not busy)
	jnz		@@error
	clc								;clear carry = success
	ret
@@error:
	stc								;set carry = error
	ret

ELSE	;bios-version

	push	edx						;save edx
	push	bx						;save bx
	push	cx						;save cx

	;setup regs for int 13h, 'read sectors'
	;ah=BIOS_READ, al=number of sectors to read
	;dl=drive, dh=head
	;cl7,6,ch=track, cl5..0=first sector on track under head
	mov		bl,cl					;save cl in bl
	mov		ax,dx					;ax=bit15...0 of sector-number
	shr		edx,16					;dx,ax=number of first sector
	div 	[cs:DriveHSPT]			;ax=track,dx=head*sector
	mov		cx,ax					;cx=track
	xchg	cl,ch					;ch=bit7...0 of track
	shl		cl,6					;cl7.6=bit9.8 of track
	mov		ax,dx					;ax=head*sector
	div		[cs:DriveSPT]			;al=head,ah=sector
	mov		dh,al					;dh=head
	inc		ah						;ah=sector counted from 1
	or		cl,ah					;cl5...0=sector
	mov		dl,[cs:Drive]			;dl=drive
	mov		al,bl					;al=saved cl=number of sectors to read

	mov		bx,di					;es:bx->buffer
	mov		ah,BIOS_READ			;function: read sectors
	int		13h						;(stc on error, clc on success)
	;no re-try done - some docs say we have to reset the conroller and try at
	;least 5 times. IMHO a read-fault reported by a modern hdd is serious
	;enough to accept it.

	pop		cx						;restore cx
	pop		bx						;restore bx
	pop		edx						;restore edx
	ret

ENDIF
ENDP rdsec
;----------------------------------------------------------------------------
%NOLIST
;
; From here on there are no more differences between the two versions until
; the initialisation-routine.
;
%LIST
%SUBTTL "general cdrom-driver functions"
%NEWPAGE
PROC UnknownCmd NEAR
;called for each unknown/unsupported function code
;return: ax=ERR_UNKNOWNCMD

	mov		ax,ERR_UNKNOWNCMD
	ret

ENDP UnknownCmd
;----------------------------------------------------------------------------
PROC InputFlush NEAR
PROC OpenDevice NEAR
PROC CloseDevice NEAR
PROC PlayAudio NEAR
PROC StopAudio NEAR
PROC ResumeAudio NEAR
;these functions do nothing
;return: ax=STAT_DONE

	mov		ax,STAT_DONE
	ret

;the audio functions are present here, since MSPRM says these requests
;should be 'ignored' if not supported.
ENDP ResumeAudio
ENDP StopAudio
ENDP PlayAudio
ENDP CloseDevice
ENDP OpenDevice
ENDP InputFlush
;----------------------------------------------------------------------------
PROC IOCTLinput NEAR
;handles all IOCTL-READ requests
;call: ds:bx->request-header
;return: ax=status-code
;modifies: ds, bx, eax, flags

	lds		bx,[(sREQIOCTL bx).ioBuffer]	;ds:bx->ioctl command
	mov		al,[bx]			;al=command
	inc		bx				;ds:bx->transfer buffer
	or		al,al			;cmd 0?
	jz short @@hdradr		;return address of device-header
	jns short @@swcmd
@@unknown:
	mov		ax,ERR_UNKNOWNCMD
	ret
@@hdradr:
	mov		[WORD bx],0			;store pointer to my driver-header
	mov		[WORD bx+2],cs
@@done:
	mov		ax,STAT_DONE		;return success
	ret

@@hdloc:
	mov		eax,[cs:HeadLoc]	;get head-location
	mov		[BYTE bx],HSGADR	;always as hsg-address (=sector-number)
	mov		[bx+1],eax			;store it
	jmp		@@done

@@secsize:
	cmp		[BYTE bx],COOKED	;which sector-size?
	jnz		@@rawsecsize
	mov		[WORD bx+1],BPCDS	;return bytes per sector - cooked
	jmp		@@done				;
@@rawsecsize:
	mov		[WORD bx+1],BPCDRAW	;return bytes per sector - raw
	jmp		@@done				;

@@devstatus:
	mov		eax,[cs:DevStatus]	;get device-status
	mov		[bx],eax			;store it
	jmp		@@done

@@diskinfo:
	mov		al,[cs:LowTrack]	;get lowest track-number (from header)
	mov		[bx],al				;set lowest track number
	mov		al,[cs:HighTrack]	;get highest track-number (from header)
	mov		[bx+1],al			;set highest track number
	mov		eax,[cs:LeadOut]	;get lead-out track start (from header)
	mov		[bx+2],eax			;set lead-out track address
	jmp		@@done

@@mediachg:
	mov		[BYTE bx],1			;return 'media not changed'
	jmp		@@done				;(we do this even if the door is open ;-))

@@volsize:
	mov		eax,[cs:VolSize]	;get volume size (from header)
	mov		[bx],eax			;store it
	jmp		@@done

@@swcmd:
	dec		al				;cmd 1?
	jz		@@hdloc			;return location of head
	dec		al				;cmd 2?
	jz		@@unknown		;reserved
	dec		al				;cmd 3?
	jz		@@unknown		;error statistics
	dec		al				;cmd 4?
	jz		@@unknown		;audio channel info
	dec		al				;cmd 5?
	jz		@@done			;read drive bytes (doc says we have to do this)
	dec		al				;cmd 6?
	jz		@@devstatus	;device status
	dec		al				;cmd 7?
	jz		@@secsize		;sector size
	dec		al				;cmd 8?
	jz		@@volsize		;volume size
	dec		al				;cmd 9?
	jz		@@mediachg		;media changed
	dec		al				;cmd 10?
	jz		@@diskinfo		;audio disk info
	dec		al				;cmd 11? (audio track info)
	jnz		@@unknown

	mov		al,[bx]				;al=track-number
	cmp		al,[cs:LowTrack]	;check track-number against boundaries
	jae short @@tnook1			;of the cdrom-image
@@tnoerr:
	mov		ax,ERR_GENFAILURE	;'general failure' if invalid track-number
	ret							;(MSPRM says nothing about this case)
@@tnook1:						
	cmp		al,[cs:HighTrack]
	ja		@@tnoerr
	dec		al					;I count them from 0
	mov		ah,SIZE sTRACKINFO	;
	mul		ah					;ax=index in sTRACKINFO-array
	mov		si,OFFSET TrackInfo	;cs:si->TRACKINFO[0]
	add		si,ax				;cs:si->sTRACKINFO of reqeusted track
	mov		eax,[cs:(sTRACKINFO si).tiStart]
	mov		[bx+1],eax			;set starting point of track
	mov		al,[cs:(sTRACKINFO si).tiCtrl]
	mov		[bx+5],al			;set track control information
	jmp		@@done

ENDP IOCTLinput
;----------------------------------------------------------------------------
PROC IOCTLoutput NEAR
;handles all IOCTL-WRITE requests
;call: ds:bx->request-header
;return: ax=status-code
;modifies: ds, bx, eax, flags

	lds		bx,[(sREQIOCTL bx).ioBuffer]	;ds:bx->ioctl command
	mov		al,[bx]			;al=command
	inc		bx				;ds:bx->transfer buffer
	or		al,al			;cmd 0?
	jz short @@eject		;eject disk
	jns short @@swcmd
@@unknown:
	mov		ax,ERR_UNKNOWNCMD
	ret

@@eject:
	or		[cs:DevStatus],MASK DOOR OR MASK DISKP	;mark door open (no disk)
			;also unlock the door

@@unlock:
	or		[cs:DevStatus],MASK LCK	;mark door unlocked

@@done:
	mov		ax,STAT_DONE	;return success
	ret

@@lockunlock:
	cmp		[BYTE bx],LOCKDOOR	;lock the door?
	jnz		@@unlock
	and		[cs:DevStatus],NOT MASK LCK	;mark door locked
	jmp		@@done

@@swcmd:
	dec		al				;cmd 1?
	jz		@@lockunlock	;lock/unlock door
	dec		al				;cmd 2?
	jz		@@reset			;reset drive
	dec		al				;cmd 3?
	jz		@@unknown		;audio channel control
	dec		al				;cmd 4?
	jz		@@done			;write device control string (no need for this)
	dec		al				;cmd 5?
	jnz		@@unknown		;(close tray ...)

@@reset:
	mov		[cs:HeadLoc],0	;reset head location
	and		[cs:DevStatus],NOT (MASK DOOR OR MASK DISKP) ;mark door closed
	jmp		@@unlock		;also unlock the door (disk present)

ENDP IOCTLoutput
;----------------------------------------------------------------------------
%SUBTTL "read related functions"
%NEWPAGE
PROC Seek NEAR
PROC ReadLongPref NEAR
;these functions 'move' the head to given location
;call:	ds:bx->request header
;return: ax=STAT_DONE

	cmp		[(sREADL bx).rlAdrMode],REDBOOK	;red book addressing mode?
	jz short @@error
	mov		eax,[(sREADL bx).rlSector]		;eax=sector-number
	mov		[cs:HeadLoc],eax	;store it
	mov		ax,STAT_DONE		;nothing more to do
	ret

@@error:
	mov		ax,ERR_SECNOTFOUND	;or ERR_GENFAILURE
	ret

ENDP ReadLongPref
ENDP Seek
;----------------------------------------------------------------------------
PROC ReadLong NEAR
;handles the READ-LONG request
;call: ds:bx->request-header
;return: ax=status-code
;modifies: eax, flags

	cmp		[(sREADL bx).rlDatMode],RAWDATA	;raw mode?
	jnz short @@cooked						;(support only cooked)
@@generr:
	mov		ax,ERR_GENFAILURE
	ret
@@error:	
	mov		ax,ERR_READFAULT
	ret
@@secnoerr:
	mov		ax,ERR_SECNOTFOUND
	ret

@@cooked:
	cmp		[(sREADL bx).rlAdrMode],REDBOOK	;red book addressing mode?
	jz		@@secnoerr	;or @@generr
	mov		eax,[(sREADL bx).rlSector]		;eax=sector-number
	cmp		eax,[cs:TotSize]	;check against cd-sector-size of cd-image
	jae		@@secnoerr

	;convert it to a sector number of the disk
	shl		eax,SPCDSsh				;eax=sector offset in cd-image
	add		eax,[cs:ImgSector]		;eax=drive-sector of first sector to read
	mov		edx,eax					;edx=first drive-sector to read

	;normalize buffer-pointer
	xor		eax,eax					;eax=0
	mov		edi,eax					;edi=0
	les		di,[(sREADL bx).rlBuffer]	;es:di->transfer-buffer
	mov		ax,es					;ax=seg of transfer-buffer
	shl		eax,4
	add		eax,edi					;eax=address of transfer-buffer
	mov		di,ax
	and		di,000Fh				;di=normalized offset
	shr		eax,4					;ax=normalized seg
	mov		es,ax					;es:di=normalized pointer

	;initialize sector counters
	mov		ax,[(sREADL bx).rlSize]	;ax=cd-sectors to read
	mov		cx,ax					;cx=-"-
	shr		cx,5					;cx=-"- / 32
	and		ax,001Fh				;ax=cd-sectors mod32
	push	ax						;save this value

	;read sectors (32 per time)
	jcxz	@@readn32
@@read32:
	push	cx						;save counter
	shl		cx,32 SHL SPCDSsh		;cx=number of hd-sectors to read (=128)
	call	rdsec					;read cd-sectors
	pop		cx						;restore counter
	jc		@@error					;(returns carry on error)
	mov		ax,es					;increment 64K-segment
	add		ax,1000h				;next segment
	mov		es,ax
	add		edx,32 SHL SPCDSsh		;next start-sector
	loop	@@read32

@@readn32:
	;read sectors (mod32)
	pop		cx						;cx=sectorcount mod 32
	shl		cx,SPCDSsh				;cx=number of hd-sectors to read
	call	rdsec					;read cd-sectors
	jc		@@error					;(returns carry on error)

	xor		ecx,ecx
	mov		cx,[(sREADL bx).rlSize]	;ecx=cd-sectors read
	add		[HeadLoc],ecx			;adjust head-location
	mov		ax,STAT_DONE			;return success
	ret

ENDP ReadLong

;----------------------------------------------------------------------------
%SUBTTL "function table"
%NEWPAGE
;table of functions (codes <0Fhh; >=0Fh,<80h;>88h mapped to 00h)
CmdTable	DW init			;00 (all >=0Fh,<80h or >82h)
							;set to UnknownCmd after INIT
			DW UnknownCmd	;01 (block devices only: MEDIA CHECK)
			DW UnknownCmd	;02 (block devices only: BUILD BPB)
			DW IOCTLinput	;03=IOCTL READ
			DW UnknownCmd	;04=READ
			DW UnknownCmd	;05=NONDESTRUCTIVE READ
			DW UnknownCmd	;06=INPUT STATUS
			DW InputFlush	;07=INPUT FLUSH
			DW UnknownCmd	;08=WRITE
			DW UnknownCmd	;09=WRITE WITH VERIFY
			DW UnknownCmd	;0A=OUTPUT STATUS
			DW UnknownCmd	;0B=OUTPUT FLUSH
			DW IOCTLoutput	;0C=IOCTL WRITE
			DW OpenDevice	;0D=OPEN DEVICE
			DW CloseDevice	;0E=CLOSE DEVICE

;table of functions (codes >=80h, <=88h)
NewCmdTable	DW ReadLong		;80h=READ LONG
			DW UnknownCmd	;81h=RESERVED
			DW ReadLongPref	;82h=READ LONG PREFETCH
			DW Seek			;83h=SEEK
			DW PlayAudio	;84h=PLAY AUDIO
			DW StopAudio	;85h=STOP AUDIO
			DW UnknownCmd	;86h=WRITE LONG
			DW UnknownCmd	;87h=WRITE LONG VERIFY
			DW ResumeAudio	;88h=RESUME AUDIO

;----------------------------------------------------------------------------
%SUBTTL "general service"
%NEWPAGE
PROC driver FAR
;really two functions: 'strategy'- and 'interrupt'-routine

strategy:	;far-call entry for passing request-header address
	mov	[cs:ReqHdrOff],bx	;store pointer to caller's request-header
	mov [cs:ReqHdrSeg],es	;(why does DOS use this method?)
	ret						;ret far

entry:	;far-call entry for service
	pushf			;save regs
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	ds
	push	es

	;get the function requested
	lds		bx,[cs:ReqHdrAdr]		;DS:BX->request-header
	mov		al,[(sREQHDR bx).rqCmd]	;al=request-header cmd-byte
	or		al,al					;
	jns short lowcmd				;<80h?
	mov		si,OFFSET NewCmdTable	;cs:si->function table
	and		al,7Fh					;mask out cmd
	cmp		al,08h					;a valid number?
	jle short cmdnook				;<=08h
badcmd:
	xor		al,al					;return unknown command
	jmp short cmdnook
lowcmd:
	mov		si,OFFSET CmdTable		;cs:si->function table
	cmp		al,0Eh					;a valid command?
	ja		badcmd
cmdnook:
	xor		ah,ah					;ax=command (number)
	shl		ax,1					;ax=index in table

	;setup own stack
	mov		[cs:OldSS],ss			;save caller's stack
	mov		[cs:OldSP],sp
	jz		nostack					;if cmd=0 it is UNKNOWN or INIT: no own
									;stack resp. skip invalid one
	lss		sp,[cs:MyStack]			;load ss:sp

	;call the requested function
	add		si,ax					;cs:si->ip of function
nostack:
	call	[WORD cs:si]			;near call requested function
									;(still ds:bx->request header)

	;when function returns all regs may be modified
	;ax contains status-code
exit:
	lss		sp,[cs:OldStack]		;restore old stack
	lds		bx,[cs:ReqHdrAdr]		;DS:BX->request-header
	mov		[(sREQHDR bx).rqStatus],ax	;set request-header status-word
	pop		es		;restore regs
	pop		ds
	pop		di
	pop		si
	pop		dx
	pop		cx
	pop		bx
	pop		ax
	popf
	ret	;ret far

;----------------------------------------------------------------------------
%SUBTTL "image-header"
%NEWPAGE
%INCL
INCLUDE "IMAGHEAD.INC"	;the header of the cd-image
%NOINCL
;----------------------------------------------------------------------------
%SUBTTL "initialisation - secondary functions"
%NEWPAGE
; all below this point will be removed after initialisation because it is never
; used again (the unused part of the sTRACKINFO-array will be removed too).

MACRO mcERROR str			;macro for init-abort
	mov		dx,OFFSET str	;cs:dx->error message
	jmp		initerror		;see below
ENDM

PROC skipspace NEAR
;call: ds:si->buffer
;return: ds:si->first character after '/'
;modified: al, si, flags

@@lop:
	lodsb					;get next character,si++
	cmp		al,' '
	jz		@@lop			;skip spaces
	cmp		al,'/'			;parameter 'sign'
	jnz short @@error		;no, but space(s) and then an other parameter
	ret						;expected

@@error:
	mcERROR	TxtFailParm		;abort initialisation

ENDP skipspace
;----------------------------------------------------------------------------
PROC getnumber NEAR
;input: ds:si->first digit of hex-number string - no skip of whitespace!
;output: edx=number given by string
;		 if no number found aborts
;stops at first non hex-digit character
;if string contains more than 8 digits only the lower part is stored
;modifies:	ax, edx, flags
;			si->character stopped on

	push	bx
	mov		bx,si	;save position we start
	xor		edx,edx	;edx=0 (holds the number)
@@lop:
	lodsb			;get character
	or		al,20h	;convert lower case to upper (4xh to 6xh)
	cmp		al,39h	;<='9'
	ja short @@letter
	add		al,27h	;move 30h-39h to 57h-60h (so we got 57h='0'-66h='F')
@@letter:
	sub		al,57h	;if character was hex-digit got the nibble
	test	al,0F0h	;otherwhise got anything <0 or >15
	jnz short @@done	;..so stop here
	shl		edx,4	;edx*=16 ('shift digits up', so can ..)
	or		dl,al	;insert new least significant nibble (the digit)
	jmp		@@lop	;go on with next character
@@done:
	cmp		bx,si	;any digits found?
	pop		bx		;restore bx
	je short @@error	;if no digit found abort
	ret

@@error:
	mcERROR	TxtFailParm		;abort initialisation

ENDP getnumber
;----------------------------------------------------------------------------
%SUBTTL "initialisation - command processing"
%NEWPAGE
initerror:						;cs:dx->error message
	mov		ax,cs				;ensure ds=cs
	mov		ds,ax				;ds:dx->error message
	mov		ah,09h				;DOS-call 'display string'
	int		21h
	mov		dx,OFFSET TxtNotInst
	int		21h					;display failure message

	;setup the request-header for failed installation
	lds		bx,[cs:ReqHdrAdr]				;DS:BX->request-header
	mov		[(sREQINIT bx).irEndAdrOff],0	;set break-address (->free mem)
	mov		[(sREQINIT bx).irEndAdrSeg],cs	;(free all=remove driver)
	mov		[(sREQINIT bx).irParmAdr],0		;param-field=0 (MSPRM says so)
	mov		[(sREQINIT bx).irMsgFlag],1		;message-flag=1 (DOS-msg also)
	mov		ax,ERR_GENFAILURE				;return error
	jmp		exit

;----------------------------------------------------------------------------
init:							;here we start the initialization
	mov		ax,cs				;setup seg-regs
	mov		ds,ax				;ds=cs
	mov		es,ax				;es=cs

	;setup stack pointers
	mov		[MySS],ax			;set my stack segment
	lss		sp,[MyStack]		;load own stack

	;reset 'cmd 0' table-entry to 'unknown cmd'
	mov		[WORD CmdTable],OFFSET UnknownCmd

	;display a 'hello'
	mcPUTS	TxtInit

	;get the parameters
	lds		bx,[ReqHdrAdr]					;ds:bx->request header
	lds		si,[(sREQINIT bx).irParmAdr]	;ds:si->config.sys command line

	;skip path in the config.sys-line
	cld						;step forward
lop0:
	lodsb					;get character, si++
	cmp		al,'/'			;sync on first space or '/'
	jz		getparm_		;(skip 'skipspace' - there's none)
	cmp		al,' '
	ja		lop0
	jnz short failparm

getparm:					;get the device-name
	call	skipspace		;skip spaces, sync on '/'
getparm_:
	mov		di,OFFSET MyDrvHdr.drvName	;es:di->name field of my driver-header
	mov		cx,8+1			;max. 8 characters (the ninth should be a space
getname:					;or a '/')
	lodsb					;get next character,si++
	cmp		al,' '			;a space?
	jle short gotname		;..so we got the name
	cmp		al,'/'			;next paramter?
	jz short gotname_		;..so we got the name
stoname:
	stosb					;store character,di++
	loop	getname			;no check if the name is valid done
							;(sorry; but - for example - the DOS-call to
							;get the table of filename-characters is not
							;allowed at this time %-()
failparm:
	mcERROR	TxtFailParm

gotname_:
	dec		si				;adjust 'parse'-pointer
gotname:
	cmp		di,OFFSET MyDrvHdr.drvName
	jz		failparm

IFDEF USEDRIVER
	;get the disk
	call	skipspace			;sync on '/'
	lodsb						;get drive letter
	or		al,20h				;convert upper-case to lower-case
	cmp		al,'z'				;a valid drive letter?
	ja		failparm			;
	cmp		al,'c'				;is in [C-Z]?
	jl		failparm			;(no floppy accepted)
	sub		al,'a'				;al=disk number
	mov		[cs:DiskNumber],al	;store it for later use
ELSE	;bios version
	;get the diskparameter
	call	skipspace			;skip spaces, sync on '/'
	call	getnumber			;get the drive-paramter 'number'
	or		dl,dl				;the low-byte is sectors_per_track
	jz		failparm			;..which must be >0
	cmp		dl,3Fh				;..and <=63
	ja		failparm
	mov		[cs:DriveSPT],dl	;store sectors-per-track
	or		dh,dh				;second byte is heads
	jz		failparm			;..which must be >0
	mov		[cs:DriveHeads],dh	;store heads
	shr		edx,8				;dh=drive
	or		dh,dh				;the drive must be >= 80h (no floppy)
	jns		failparm			;(it is a bios drive number)
	mov		[cs:Drive],dh		;store drive-number
	mov		al,[cs:DriveSPT]	;calculate sector-size for a track
	mov		ah,[cs:DriveHeads]	;(needed by rdsec)
	mul		ah					;ax=heads*sectors_per_track
	mov		[cs:DriveHSPT],ax	;store sector-size of track
ENDIF
	;get the start-sector
	call	skipspace			;skip spaces, sync on '/'
	call	getnumber			;expect a hex-number
	or		edx,edx				;<>0? (0 is MBR)
	jz		failparm
	mov		[cs:ImgSector],edx	;store start-sector

;----------------------------------------------------------------------------
%SUBTTL "initialisation - main function"
%NEWPAGE
;we got the config-line parameters

	;setup seg-regs
	mov		ax,cs
	mov		ds,ax				;ds=cs

IFDEF USEDRIVER
	;now get the DPB of disk to get disk's device-driver entry
	;(this int is not allowed by MS-PRM, but it works - and why not: it
	;simply returns the address of a table)
	;naturally the disk-driver has to be loaded before this driver ;-)
	push	ds					;save ds
	mov		dl,[cs:DiskNumber]	;get disk-number
	inc		dl					;this function counts from 1 (A=1,B=2,C=3,...)
	mov		ah,32h				;DOS-funtion 'get DPB'
	int		21h					;on success: ds:bx->disk's DPB
	cmp		al,00h				;error?
	jz short gotdpb
	mcERROR	TxtFailure

gotdpb:									;now copy the relevant data
	mov		ax,[(sDPB bx).dpbDrvOff]	;get address of disk driver's header
	mov		[cs:DiskDrvOff],ax
	mov		ax,[(sDPB bx).dpbDrvSeg]	;get driver-header's segment
	mov		[cs:DiskDrvSeg],ax
	mov		[cs:DiskStrSeg],ax			;the segment is the same for strategy-
	mov		[cs:DiskIntSeg],ax			;and interrupt-routine
	mov		al,[(sDPB bx).dpbMedia]		;get media byte
	mov		[cs:DiskMedia],al
	mov		al,[(sDPB bx).dpbUnit]		;get driver's unit number
	mov		[cs:DiskUnit],al

	;now some more checks for the sector-number could be made
	mov		eax,[cs:ImgSector]			;get first sector of image-file
	xor		ecx,ecx
	mov		cx,[(sDPB bx).dpbFirstSec]	;ecx=first sector of disk's data-area
	cmp		eax,ecx						;image has to be in disk's data-area
	jl		failparm
	mov		cl,[(sDPB bx).dpbSPCshift]	;check if sector's cluster-number 
	shr		eax,cl						;is less than 0FFF8h
	cmp		ax,0FFF8h
	ja		failparm
	shr		eax,16
	or		ax,ax						;ax should be 0
	jnz		failparm

	;get disk-driver's entry-points
	lds		bx,[cs:DiskDrvHdr]			;ds:bx->disk driver's header
	mov		ax,[(sDRVHDR bx).drvStrat]	;ax=offset of strategy-routine
	mov		[cs:DiskStrOff],ax			;store it
	mov		ax,[(sDRVHDR bx).drvIntr]	;ax=offset of interrupt-routine
	mov		[cs:DiskIntOff],ax			;store it
	pop		ds							;restore ds to cs
ENDIF

	;now read the header
	xor		ecx,ecx
	mov		cl,IMGHDRSEC			;ecx=header size in hd-sectors
	mov		edx,[ImgSector]			;first sector of image-header
	add		[ImgSector],ecx			;ImgSector=first sector of cd-image
	mov		di,OFFSET ImageHeader	;es:di->buffer
	call	rdsec					;read file header
	jnc short gothdr				;rdsec set carry on error
	mcERROR	TxtFailure

gothdr:	;check signature (the only validation done)
	mov		si,OFFSET DefSignat
	mov		di,OFFSET Signature
	mov		cx,LENGTH DefSignat
	rep cmpsb
	jz short hdrok
	mcERROR	TxtFailHdr

hdrok:
	;display some information
	mov		si,OFFSET MyDrvHdr.drvName	;copy the given name
	mov		di,OFFSET TxtName			;into success message
	mov		cx,8
	rep		movsb
	mcPUTS	ImgComment					;display image-comment
	mcPUTS	TxtSuccess					;display success-message

	;setup the request-header for succesfull installation
	mov		al,[HighTrack]
	mov		ah,SIZE sTRACKINFO
	mul		ah								;ax=index in TrackInfo-array
	add		ax,OFFSET TrackInfo				;cs:ax->end of driver
	lds		bx,[ReqHdrAdr]					;DS:BX->request-header
	mov		[(sREQINIT bx).irEndAdrOff],ax	;cut off all after
	mov		[(sREQINIT bx).irEndAdrSeg],cs	;trackinfos
	mov		[(sREQINIT bx).irParmAdr],0		;parm-field=0 (MSPRM says so)
	mov		ax,STAT_DONE					;status-word=done
	jmp		exit							;return success

ENDP driver

;----------------------------------------------------------------------------
%SUBTTL "initialisation - data area"
%NEWPAGE
;some texts used while init
DefSignat	DB SIGNATURE
TxtInit		DB 0Dh,0Ah,"PseudoCD v02  Copyright (C) 1997 C.Kulms",0Dh,0Ah,'$'
TxtSuccess	DB 0Dh,0Ah,"PSCD"
IFDEF USEDRIVER
			DB "DDRV"
ELSE
			DB "BIOS"
ENDIF
			DB " installed as "
TxtName		DB 8 DUP(?),0Dh,0Ah,'$'
TxtFailParm	DB "invalid parameter$"
TxtFailHdr	DB "invalid image$"
TxtFailure	DB "general failure$"
TxtNotInst	DB 0Dh,0Ah,"PSCD"
IFDEF USEDRIVER
			DB "DDRV"
ELSE
			DB "BIOS"
ENDIF
			DB " NOT INSTALLED",0Ah,0DH,'$'

ENDS CODE

END

;=============================================================================
;END OF FILE 'PSEUDOCD.ASM'
;=============================================================================

