
/*============================================================================

	CDROMF.C
	Some CDROM-functions (MSCDEX interface).
  This file is part of the

  PseudoCD package (version 02)
  Copyright (C) C.Kulms, 1997

  This is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This software is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this software; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

============================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>

#include "driver.h"
#include "cdrom.h"

/*
	MSCDEX_DriverRequest
	Launches MSCDEX's 'Driver Send Request' function, checks for error.
	return: status code of request header, exits on error
*/
unsigned MSCDEX_DriverRequest( int iCDno, t_REQHDR *ptReqHdr )
{
unsigned	uStatus,		/* status */
					uCmd;				/* command code */
char			*pcStatus,	/* holds status description */
					*pcCmd;			/* holds command description */
int				retry;			/* retry counter */

	/* send request */
	for( retry = 5; retry; retry-- )
	{
		ptReqHdr->uStatus = 0;
		uStatus = MSCDEX_DriverSendRequest( iCDno, (t_REQHDR far *)ptReqHdr );

		/* check for error */
#if 0
		if( uStatus & ERRORBIT )	/* error bit set? */
		/* this is the correct way, but some drivers don't set the error-bit some
		times (e.g. request 'play audio' on a data-track;
		then error-code is 'general failure', status = 0x030C). */
#else
		if( uStatus & ERRORCOND )	/* error-bit and/or error-code set? */
#endif
		{
			uCmd = (unsigned)ptReqHdr->ucCmd;
			if( uCmd == DRV_IOCTLI || uCmd == DRV_IOCTLO )
				uCmd = (uCmd << 8) | (unsigned)*((t_REQIOCTL *)ptReqHdr)->fpBuffer;

			switch( uCmd )
			{
				case RDLONG:
					pcCmd = "read long";
				break;
				case RDLONGP:
					pcCmd = "read long prefetch";
				break;
				case SEEK:
					pcCmd = "seek";
				break;
				case PLAY:
					pcCmd = "play audio";
				break;
				case STOP:
					pcCmd = "stop audio";
				break;
				case WRLONG:
					pcCmd = "write long";
				break;
				case WRLONGV:
					pcCmd = "write long verify";
				break;
				case RESUME:
					pcCmd = "resume audio";
				break;
				case IOCTL_DRVHDR:
					pcCmd = "return device-driver header";
				break;
				case IOCTL_HEADLOC:
					pcCmd = "location of head";
				break;
				case IOCTL_ERRSTAT:
					pcCmd = "error statistics";
				break;
				case IOCTL_ACINFO:
					pcCmd = "audio channel info";
				break;
				case IOCTL_RBYTES:
					pcCmd = "read drive bytes";
				break;
				case IOCTL_DEVSTAT:
					pcCmd = "return device status";
				break;
				case IOCTL_SECSIZE:
					pcCmd = "return sector size";
				break;
				case IOCTL_VOLSIZE:
					pcCmd = "return volume size";
				break;
				case IOCTL_MEDCHG:
					pcCmd = "media changed";
				break;
				case IOCTL_DISKINFO:
					pcCmd = "audio disk info";
				break;
				case IOCTL_TRACKINFO:
					pcCmd = "audio track info";
				break;
				case IOCTL_QCINFO:
					pcCmd = "audio Q-channel info";
				break;
				case IOCTL_SCINFO:
					pcCmd = "audio sub-channel info";
				break;
				case IOCTL_UPC:
					pcCmd = "UPC code";
				break;
				case IOCTL_ASINFO:
					pcCmd = "audio status info";
				break;
				case IOCTL_EJECT:
					pcCmd = "eject disk";
				break;
				case IOCTL_LOCK:
					pcCmd = "lock/unlock door";
				break;
				case IOCTL_RESET:
					pcCmd = "reset drive";
				break;
				case IOCTL_ACCTRL:
					pcCmd = "audio channel control";
				break;
				case IOCTL_WBYTES:
					pcCmd = "write device control string";
				break;
				case IOCTL_CLOSE:
					pcCmd = "close tray";
				break;
				default:
					pcCmd = "(unknown)";
			}
			switch( uStatus & ERRORMASK )
			{
				case DEV_BUSY:
					pcStatus = "device busy";
				break;
				case ERR_WRITEPROT:
					pcStatus = "write protection violation";
				break;
				case ERR_UNKNOWNUNIT:
					pcStatus = "invalid sub-unit";
				break;
				case ERR_NOTREADY:
					pcStatus = "device not ready";
				break;
				case ERR_UNKNOWNCMD:
					pcStatus = "unknown command";
				break;
				case ERR_CRCERROR:
					pcStatus = "crc-error";
				break;
				case ERR_BADRQLENGTH:
					pcStatus = "bad request-header length";
				break;
				case ERR_SEEKERROR:
					pcStatus = "seek error";
				break;
				case ERR_UNKNOWNMED:
					pcStatus = "unknown media byte";
				break;
				case ERR_SECNOTFOUND:
					pcStatus = "sector not found";
				break;
				case ERR_OUTOFPAPER:
					pcStatus = "out of paper";
				break;
				case ERR_WRITEFAULT:
					pcStatus = "write fault";
				break;
				case ERR_READFAULT:
					pcStatus = "read fault";
				break;
				case ERR_GENFAILURE:
					pcStatus = "general failure";
				break;
				case ERR_INVCHANGE:
					pcStatus = "invalid media change";
				break;
				default:
					pcStatus = "(unknown)";
			}
			fprintf( stderr, "Driver request '%s' (%04X) failed: '%s' (%04X).\n",
								pcCmd, uCmd, pcStatus, uStatus );
			if( (uStatus & ERRORMASK) == ERR_NOTREADY && retry )
			{
				sleep( 1 );	/* wait a second */
				continue;		/* then retry */
			}
			exit( -1 );	/* any other than 'device not ready' aborts */
		}
		break;	/* no error no retry */
	}	/* for( retry = 5; retry; retry-- ) */

	return( uStatus );
}

/*
	MSCDEX_IoctlRequest
	Launches an IOCTL request.
	return: status code of request header
*/
unsigned MSCDEX_IoctlRequest( int iCDno, unsigned uCmd, char *pcBuffer )
{
t_REQIOCTL	tReqIoctl;	/* IOCTL request header */
unsigned 		uStatus;		/* status of request */

	*pcBuffer = (char)uCmd;	/* set IOCTL sub-command */
	tReqIoctl.tReqHdr.ucCmd = (unsigned char)(uCmd >> 8);	/* IOCTL command */
	tReqIoctl.fpBuffer = (char far *)pcBuffer;	/* set pointer to buffer */

	/* launch request */
	return( MSCDEX_DriverRequest( iCDno, (t_REQHDR *)&tReqIoctl ) );
}

/*
	ReadCDSectors
	Reads sectors of CDROM. Has to be done by a 'READ LONG' request since
	MSCDEX's 'read absolute sector' function does not read at the very beginning
	of the CDROM (like DOS's 'read absolute sector' which does not read any
	sector before the data area of a disk).
	return: 1, exits on error
*/
int ReadCDSectors( int iCDno, char *pcBuffer, unsigned long ulSector, unsigned uCount )
{
t_READL	tReadLong;
unsigned	uStatus;

	/* setup (extened) request header */
	tReadLong.cAdrMode = HSGADR;								/* use HSG address */
	tReadLong.fpBuffer = (char far *)pcBuffer;	/* target buffer */
	tReadLong.uSize = uCount;										/* number of sectors */
	tReadLong.sSector.ulNumber = ulSector;			/* first sector */
	tReadLong.cDatMode = COOKED;								/* 'cooked' data please */
	tReadLong.cIlSize = 0;											/* no interleave */
	tReadLong.cIlSkip = 0;                      /* no interleave */
	tReadLong.tReqHdr.ucCmd = RDLONG;						/* command */

	/* launch request */
	uStatus = MSCDEX_DriverRequest( iCDno, (t_REQHDR *)&tReadLong );
	if( uStatus != REQDONE )
	{
		fprintf( stderr, "ReadLong (%lu,%d) failed (status %04X).\n",
							ulSector, uCount, uStatus );
		exit( -1 );
	}

	return( 1 );
}

/* end of file 'CDROMF.C' */

