/* 
 * 
 * ------------------------------------------------------------------
 * Home Libarian 1.1 by Deepwoods Software
 * ------------------------------------------------------------------
 * hlcgimain.cc - HL CGI Interface
 * Created by Robert Heller on Sat Mar  2 17:34:33 1996
 * ------------------------------------------------------------------
 * Modification History: 
 * ------------------------------------------------------------------
 * Contents:
 * ------------------------------------------------------------------
 *  
 *     Home Librarian Database -- a program for maintaining a database
 *                                for a home library
 *     Copyright (C) 1991-1996  Robert Heller D/B/A Deepwoods Software
 * 			51 Locke Hill Road
 * 			Wendell, MA 01379-9728
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *  
 */

#include <iostream.h>
#include <fstream.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <cgi.h>

#include <vBTree.h>
#include <ctype.h>
#include <CardRecord.h>
#include <ListRecord.h>
#include <PTree.h>

extern Tree* ParseOnly(char*);
extern Boolean EvalExpr(const Card*, const Tree*);
static vBTree *TreeFile;
static Tree* onlyexpr = 0;
static char *tfilename;
static ifstream tfile;

enum PrintFields {PrintType, PrintAuthor, PrintTitle, PrintPublisher,
		  PrintCity, PrintVolume, PrintYear, PrintIdent,
		  PrintDescription, PrintSubject, PrintFileName, 
		  PrintSearchString, PrintSearchMode, PrintServerName, 
		  PrintRemoteHost, PrintRemoteAddr, PrintRemoteUser, IllField };

static const struct {
	const char* n;
	PrintFields f;
} BodyFieldNames[] = {
	{ "TYPE", PrintType },
	{ "AUTHOR", PrintAuthor },
	{ "TITLE", PrintTitle },
	{ "PUBLISHER", PrintPublisher },
	{ "CITY", PrintCity },
	{ "VOLUME", PrintVolume },
	{ "YEAR", PrintYear },
	{ "ID", PrintIdent },
	{ "DESCRIPTION", PrintDescription },
	{ "SUBJECT", PrintSubject },
}, OtherFieldNames[] = {
	{ "FILENAME", PrintFileName },
	{ "SEARCHSTRING", PrintSearchString },
	{ "SEARCHMODE", PrintSearchMode },
	{ "SERVER_NAME", PrintServerName },
	{ "REMOTE_HOST", PrintRemoteHost },
	{ "REMOTE_ADDR", PrintRemoteAddr },
	{ "REMOTE_USER", PrintRemoteUser },
};

const NumBFields = sizeof(BodyFieldNames) / sizeof(BodyFieldNames[0]);
const NumOFields = sizeof(OtherFieldNames) / sizeof(OtherFieldNames[0]);

static Boolean streqp(const char* s1,const char* s2)
{
	register char c1,c2;
	do {
		c1 = *s1++;
		if (islower(c1)) c1 = toupper(c1);
		c2 = *s2++;
		if (islower(c2)) c2 = toupper(c2);
		if (c1 != c2) return(false);
	} while (c1 != 0);
	return(true);
}

void QuoteStringBuffer(char* buffer,char* string)
{
	*buffer++ = '"';
	while (*string != 0) {
		if (*string == '"') *buffer++ = '\\';
		*buffer++ = *string++;
	}
	*buffer++ = '"';
	*buffer = 0;
}

static char *ptr;
static Key   subjkey;
static void ExamineSubj(CoreItem* item,int)
{
	if (item->data.size <= 0) return;
	ListRecord rec(&item->data);
	int icount = rec.ElementCount();
	for (int i = 0; i < icount; i++) {
		if (streqp(subjkey,rec[i])) {
			QuoteStringBuffer(ptr,item->key);
			ptr += strlen(ptr);
			*ptr++ = ',';
			return;
		}
	}
}

void FetchSubjs(char* subjbuffer,const Key key)
{
	ptr = subjbuffer;
	strcpy(subjkey,key);
	TreeFile->TraverseSubj(ExamineSubj);
	if (ptr > subjbuffer && *(ptr-1) == ',') ptr--;
	*ptr = '\0';
}

void PrintCardRecord(CoreItem* item,int)
{
	if (item->data.size <= 0) return;
	CardRecord rec(&item->data);
	const DescrSize = 4096;
	static char SubjList[DescrSize];
	Boolean havesl = false;
	register const Card* c = &rec;
	if (onlyexpr != 0) {
		if (!EvalExpr(c,onlyexpr)) return;
	}
	tfile.open(tfilename,ios::in);
	if (!tfile.is_open()) {
		int error = errno;
		perror("hlcgimain: template file open");
		cerr << "hlcgimain: could not open " << tfilename << "\n";
		exit(error);
	}

	//cerr << "*** card: " << item->key << "\n";
	//cerr << "*** tfile.eof() = " << tfile.eof() <<
	//	", tfile.fail() = " << tfile.fail() << "\n";
	while (!tfile.eof() && !tfile.fail()) {
		char ch;
		tfile.get(ch);
		//cerr << "*** ch = " << ((int) ch) << "\n";
		if (ch == EOF) break;
		else if (ch != '%') cout << ch;
		else {
			tfile.get(ch);
			if (ch == '%') cout << ch;
			else if (ch == EOF) break;
			else if (isalpha(ch)) {
				static char word[256];
				register char* p = word;
				register PrintFields field = IllField;
				do {
					if (islower(ch)) ch = toupper(ch);
					*p++ = ch;
					tfile.get(ch);
				} while (isalpha(ch));
				if (ch != EOF) tfile.putback(ch);
				*p = 0;
				for (int i = 0; i < NumBFields; i++) {
					if (strcmp(word,BodyFieldNames[i].n) == 0) {
						field = BodyFieldNames[i].f;
						break;
					}
				}
				if (field != IllField) 
				{
					switch (field) {
						case PrintType:
							cout << TypeName(c->type);
							break;
						case PrintAuthor:
							cout << c->author;
							break;
						case PrintTitle:
							cout << c->title;
							break;
						case PrintPublisher:
							cout << c->publisher;
							break;
						case PrintCity:
							cout << c->city;
							break;
						case PrintVolume:
							cout << c->vol;
							break;
						case PrintYear:
							cout << c->year;
							break;
						case PrintIdent:
							cout << item->key;
							break;
						case PrintDescription:
							cout << c->description;
							break;
						case PrintSubject:
							if (!havesl)
							{
								FetchSubjs(SubjList,item->key);
								havesl = true;
							}
							cout << SubjList;
							break;
						default: break;
					}
				}
			}
		}
	}
	tfile.close();
}

void PrintListRecord(CoreItem *item,int level = 0)
{
	if (item->data.size > 0) {
		ListRecord rec(&item->data);
		int numitems = rec.ElementCount();
#ifdef ILLEGAL_INSTR
		static CoreItem temp;
		for (int i = 0;i < numitems; i++) {
			if (TreeFile->SearchId(rec[i],&temp) &&
			    strlen(rec[i]) == strlen(temp.key)) 
			    	PrintCardRecord(&temp,level);
		}
#else
		CoreItem* temp = new CoreItem;
		for (int i = 0;i < numitems; i++) {
			if (TreeFile->SearchId(rec[i],temp) &&
			    strlen(rec[i]) == strlen(temp->key)) 
			    	PrintCardRecord(temp,level);
		}
		delete temp;
#endif
	}
}

static String htmlQuote(const String string)
{
	char * str = (char*) string;
	String result = "";
	while (*str != '\0')
	{
		switch (*str)
		{
			case '&': result = result + "&amp;"; break;
			case '<': result = result + "&lt;"; break;
			case '>': result = result + "&gt;"; break;
			case '"': result = result + "&quot;"; break;
			default: result = result + *str; break;
		}
		str++;
	}
	return result;
}

void PrintHeaderOrTrailer(String templatefile,const CGI &cgi)
{
	ifstream tfile(templatefile);
	if (!tfile.is_open()) {
		int error = errno;
		perror("hlcgimain: template file open");
		cerr << "hlcgimain: could not open " << tfilename << "\n";
		exit(error);
	}
	while (!tfile.eof() && !tfile.fail()) {
		char ch;
		tfile.get(ch);
		//cerr << "*** ch = " << ((int) ch) << "\n";
		if (ch == EOF) break;
		else if (ch != '%') cout << ch;
		else {
			tfile.get(ch);
			if (ch == '%') cout << ch;
			else if (ch == EOF) break;
			else if (isalpha(ch)) {
				static char word[256];
				register char* p = word;
				register PrintFields field = IllField;
				do {
					if (islower(ch)) ch = toupper(ch);
					*p++ = ch;
					tfile.get(ch);
				} while (isalpha(ch));
				if (ch != EOF) tfile.putback(ch);
				*p = 0;
				for (int i = 0; i < NumOFields; i++) {
					if (strcmp(word,OtherFieldNames[i].n) == 0) {
						field = OtherFieldNames[i].f;
						break;
					}
				}
				if (field != IllField) 
				{
					switch (field) {
						case PrintFileName:
							cout << cgi.GetVal("LibraryFile");
							break;
						case PrintSearchString:
							cout << htmlQuote(cgi.GetVal("SearchString"));
							break;
						case PrintSearchMode:
							cout << cgi.GetVal("SearchMode");
							break;
						case PrintServerName:
							cout << cgi.GetVal("SERVER_NAME");
							break;
						case PrintRemoteHost:
							cout << cgi.GetVal("REMOTE_HOST");
							break;
						case PrintRemoteAddr:
							cout << cgi.GetVal("REMOTE_ADDR");
							break;
						case PrintRemoteUser:
							cout << cgi.GetVal("REMOTE_USER");
							break;
						default: break;
					}
				}
			}
		}
	}
	tfile.close();
}

#ifndef DEFAULTDIR
#define DEFAULTDIR "/usr/local/lib/hlcgi"
#endif

static 	String BaseDirectory = DEFAULTDIR;

static String CheckFile(const String FileName)
{
	String Result = FileName;
	if (FileName[0] != '/') /* not abs path? */
	{
		Result = BaseDirectory + "/" + FileName;
	}
	if (access((char*)Result,R_OK) == 0) return Result;
	else return "";
}	


static void ErrorHandler(ErrKind err,char* msg)
{
	if (err == sysErr) {
		int error = errno;
		cerr.form("Error: System:%d %s %s\n",error,strerror(error),
				msg);
	} else {
		cerr.form("Error: %s %s\n",
			(err == termErr ? "Terminal" : "Memory"),
			msg);
	}
}
	

int main(int argc,char **argv)
{
	if (argc > 1) BaseDirectory = argv[1];
	CGI cgi;
	String LibraryFile = CheckFile(cgi.GetVal("LibraryFile"));
	if (LibraryFile.length() == 0)
	{
		cout << "Content-type: text/html\n\n";
		cout << "LibraryFile not set or LibraryFile not found!\n";
		exit(0);
	}
	vBTree vtree((char*)LibraryFile,(OpenMode)(ReadOnly));
	if (vtree.OpenStat() == failure) {
		int error = errno;
		cerr << "Could not open " << LibraryFile
		     << ": " << strerror(error) << "\n";
		exit(error);
	}
	vtree.ErrorFun() = ErrorHandler;
	TreeFile = &vtree;
	cout << "Content-type: text/html\n\n";
	String HeaderTemplate = CheckFile(cgi.GetVal("HeaderTemplate"));
	if (HeaderTemplate.length() == 0)	/* no header template? use a default one */
	{
		cout << "<HTML><HEAD><TITLE>Result of Search</TITLE></HEAD><BODY>\n";
	} else
	{
		PrintHeaderOrTrailer(HeaderTemplate,cgi);
	}
	String BodyTemplate = CheckFile(cgi.GetVal("BodyTemplate"));
	if (BodyTemplate.length() != 0)	/* skip body if no body template */
	{
		tfilename = (char*) BodyTemplate;
		String SearchMode = cgi.GetVal("SearchMode");
		String SearchString = cgi.GetVal("SearchString");
		if (SearchMode.length() == 0) SearchMode = "All";
		if (SearchMode == "All") vtree.TraverseAuthor(PrintListRecord);
		else if (SearchMode == "Author")
		{
			Key searchkey;
			CoreItem item;
			memset(searchkey,0,sizeof(searchkey));
			strncpy(searchkey,(char*)SearchString,sizeof(searchkey)-1);
			if (vtree.SearchAuthor(searchkey,&item))
			{
				do {
					PrintListRecord(&item);
				} while (vtree.SearchAuthorAgain(&item));
			}
		} else if (SearchMode == "Title")
		{
			Key searchkey;
			CoreItem item;
			memset(searchkey,0,sizeof(searchkey));
			strncpy(searchkey,(char*)SearchString,sizeof(searchkey)-1);
			if (vtree.SearchTitle(searchkey,&item))
			{
				do {
					PrintListRecord(&item);
				} while (vtree.SearchTitleAgain(&item));
			}
		} else if (SearchMode == "Subject")
		{
			Key searchkey;
			CoreItem item;
			memset(searchkey,0,sizeof(searchkey));
			strncpy(searchkey,(char*)SearchString,sizeof(searchkey)-1);
			if (vtree.SearchSubj(searchkey,&item))
			{
				do {
					PrintListRecord(&item);
				} while (vtree.SearchSubjAgain(&item));
			}
		} else if (SearchMode == "Id")
		{
			Key searchkey;
			CoreItem item;
			memset(searchkey,0,sizeof(searchkey));
			strncpy(searchkey,(char*)SearchString,sizeof(searchkey)-1);
			if (vtree.SearchId(searchkey,&item))
			{
				do {
					PrintListRecord(&item);
				} while (vtree.SearchIdAgain(&item));
			}
		} else if (SearchMode == "Expression")
		{
			onlyexpr = ParseOnly((char*)SearchString);
			if (onlyexpr != 0)
			{
				vtree.TraverseId(PrintCardRecord);
			}
		} else vtree.TraverseAuthor(PrintListRecord);
	}
	String TrailerTemplate = CheckFile(cgi.GetVal("TrailerTemplate"));
	if (TrailerTemplate.length() == 0)       /*no trailer template? use a default one */
	{
		cout << "</BODY></HTML>\n";
	} else
	{
		PrintHeaderOrTrailer(TrailerTemplate,cgi);
	}
	exit(0);
}
