Unit Menu;
(* Version 1.05: 8 Mar 1995                                                 *)

(* Copyright (C) 1995 Tapirsoft, Harald Selke                               *)
(* A collection of procedures for easily creating simple menus              *)
(* See Menu.Doc for full documentation.                                     *)
(*                                                                          *)
(* This programme is free software; you can redistribute it and/or modify   *)
(* it under the terms of the GNU General Public License (version 1) as      *)
(* published by the Free Software Foundation.                               *)
(*                                                                          *)
(* This programme is distributed in the hope that it will be useful,        *)
(* but WITHOUT ANY WARRANTY; without even the implied warranty of           *)
(* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *)
(* GNU General Public License for more details.                             *)
(*                                                                          *)
(* You should have received a copy of the GNU General Public License        *)
(* along with this programme; if not, write to the Free Software            *)
(* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                *)

Interface

Const MessLength   = 60;                     (* length of a message         *)
      MLLength     = 40;                     (* length of a menu line       *)
      FILength     = 30;                     (* length of a form item/entry *)
      MaxLines     = 10;                     (* maximum number of lines in  *)
      MaxItems     = 20;                     (* a menu or items in a form   *)

Type MessageString = String [MessLength];           (* the type of messages *)
     MenuLine      = String [MLLength];      (* menu lines are of this type *)
     HotkeyType    = String [MaxItems];            (* hotkeys for selecting *)

     FileName      = String [12];
     FileExtension = String [3];

(* A Menu is simply an array of menu lines. *)
     MenuType      = Array [0..MaxLines] Of MenuLine;           (* the menu *)

(* A Form consists of form items, which in turn consist of two strings, a   *)
(* position, where the item will appear, and four pointers to the items,    *)
(* which are the neighbours above, below, to the right, and to the left.    *)
(* A FormItem is displayed only if its Active component is true.            *)
     FormItem      = Record
                       Item, Entry      : String [FILength];
                       x, y, u, d, r, l : Byte;
                       Active           : Boolean
                     End;
     FormType      = Array [0..MaxItems] Of FormItem;           (* the form *)

(* The colours are defined by the following variables. They are set to      *)
(* initial values in the initialization part at the end of the unit.        *)
(* MessColour is the colour for messages, LowColour the colour for entries  *)
(* and items that are not highlighted, HighColour is for the highlighted    *)
(* entry or item, BackColour is the colour of the background. HeadColour    *)
(* and HeadBackColour define the colours of the text in the headline and of *)
(* the background of the headline. FrameColour is used for drawing frames.  *)
Var MessColour, LowColour, HighColour, MenuBackColour,
    HeadColour, HeadBackColour, FrameColour        : Byte;

    MessDelay : Integer;                         (* delay time for messages *)

Procedure ClearMessLine;
(* The message line is cleared. *)

Procedure ErrorMess (Message : MessageString);
(* The message is displayed below the menu, the program halts then. *)

Procedure TitleMess (Message : MessageString);
(* The message is displayed on the right side in the headline. *)

Procedure DisplayMess (Message : MessageString);
(* The message is displayed in the middle below the menu. *)

Procedure Prompt (Message : MessageString);
(* The message is displayed on the left side below the menu. *)

Function GetInt (Message : MessageString; old, min, max : Integer) : Integer;
(* The message and the old value are displayed like a prompt and an integer *)
(* is read from the keyboard. If a number less than min or greater than max *)
(* is read, the old value is returned. The same is true if the input wasn't *)
(* an integer.                                                              *)

Function GetFileName (Message : MessageString; old : FileName;
                      ext : FileExtension) : FileName;
(* The message and the old filename are displayed like a prompt. If the in- *)
(* put doesn't contain a dot, the extension ext is added automatically. If  *)
(* the input was <Return>, the old file name is returned.                   *)

Procedure ShowHeadline (Logo, MenuTitle : MenuLine);
(* The logo is displayed to the left of the menutitle in the headline. *)

Procedure MakeFrame (Var x, y : Byte; w, h : Byte);
(* A frame is drawn around a window, which has its upper left corner in *)
(* (x,y), a width of w and a height of h characters.                    *)

Function StdMenu (Logo : MenuLine; Menu : MenuType;
                  Hotkeys : HotkeyType; PreSel : Byte) : Byte;
(* The menu is displayed together with the logo. Hotkeys may be empty, *)
(* PreSel is the initially highlighted item.                           *)

Function PopMenu (Menu : MenuType; Hotkeys : HotkeyType;
                  PreSel, XPos, YPos : Byte) : Byte;
(* The menu is displayed with its first column in column XPos and first line *)
(* line in line YPos. Hotkeys may be empty, PreSel is highlighted initially. *)

Function StdForm (Logo : MenuLine; Form : FormType;
                  Hotkeys : HotkeyType; PreSel : Byte) : Byte;
(* The form is displayed together with the logo. Hotkeys may be empty, *)
(* PreSel is the initially highlighted item.                           *)


Implementation
Uses Crt;

Const Ret  = #13;                      (* <Return>      *)
      Esc  = #27;                      (* <Escape>      *)
      Back = #8;                       (* <Back Space>  *)

(* First come the utilities which are used by the menu and form procedures. *)

Procedure Burp;
(* Emits a sound. *)
Begin                                                            (* Burp    *)
  Sound (50); Delay (200); NoSound                               (* Private *)
End;   (* Burp *)

Procedure WriteLine (MLine : MessageString;
                     XPos, YPos, Colour, BackColour : Byte);
(* Writes the text MLine starting at position (XPos, YPos) in colour Colour *)
(* on background BackColour.                                                *)
Var OldTextAttr : Byte;
Begin                                                          (* WriteLine *)
  OldTextAttr := TextAttr;                                     (* Private   *)
  TextAttr := Colour + 16 * BackColour;
  GotoXY (XPos, YPos);
  ClrEol; Write (MLine);
  TextAttr := OldTextAttr
End;   (* WriteLine *)

Procedure ClearMessLine;
Begin                                                      (* ClearMessLine *)
  GotoXY (Lo(WindMin)+1, Hi(WindMax));                     (* Public        *)
  ClrEol
End;   (* ClearMessLine *)

Procedure ErrorMess;
Begin                                                          (* ErrorMess *)
  Burp;                                                        (* Public    *)
  WriteLine (Message, (Lo(WindMax) - Length (Message)) Div 2,
             Hi(WindMax), MessColour, MenuBackColour);
  Burp;
  Delay (5000);
  Halt;
End;   (* ErrorMess *)

Procedure TitleMess;                                           (* TitleMess *)
Begin                                                          (* Public    *)
  WriteLine (Message, Lo(WindMax) - Length (Message) - 1,
             Hi(WindMin)+1, HeadColour, HeadBackColour)
End;   (* TitleMess *)

Procedure DisplayMess;                                       (* DisplayMess *)
Begin                                                        (* Public      *)
  WriteLine (Message, (Lo(WindMax) - Length (Message)) Div 2,
             Hi(WindMax), MessColour, MenuBackColour);
  Delay (MessDelay)
End;   (* DisplayMess *)

Procedure Prompt;                                                 (* Prompt *)
Begin                                                             (* Public *)
  WriteLine (Message, Lo(WindMax) Div 2 - Length (Message),
             Hi(WindMax), MessColour, MenuBackColour)
End;   (* Prompt *)

Function GetString : MessageString;
(* If <Return> is typed, an empty string is returned, else the current line *)
(* is cleared and the typed characters are echoed. The input is terminated  *)
(* by <Return>. <Back Space> may be used to edit the input.                 *)
Var s : MessageString;
    c : Char;
    OldTextAttr : Byte;
Begin                                                          (* GetString *)
  OldTextAttr := TextAttr;                                     (* Private   *)
  TextAttr := MessColour + 16 * MenuBackColour;
  s := '';
  Repeat Until KeyPressed;
  c := ReadKey;
  If c <> Ret Then
  Begin
    ClrEol;
    Repeat
      If c = Back Then
      Begin
        GotoXY (WhereX-1, WhereY); Write (' ');
        GotoXY (WhereX-1, WhereY);
        Delete (s, Length (s), 1);
      End
      Else
      Begin
        Write (c);
        s := s + c;
      End;
      c := ReadKey
    Until c = Ret
  End;
  TextAttr := OldTextAttr;
  GetString := s
End;   (* GetString *)

Function GetInt;
Var s : MessageString;
    new, res : Integer;
Begin                                                             (* GetInt *)
  Str (old, s);                                                   (* Public *)
  WriteLine (Message + s, Lo(WindMax) Div 2 - Length (Message),
             Hi(WindMax), MessColour, MenuBackColour);
  GotoXY (Lo(WindMax) Div 2, Hi(WindMax));
  s := GetString;
  new := old;
  If s <> '' Then
  Begin
    Val (s, new, res);
    If res <> 0 Then
    Begin
      ClearMessLine;
      DisplayMess ('Oops, this was not a number!');
      new := old
    End
    Else
    Begin
      If new > max Then
      Begin
        Str (max, s);
        ClearMessLine;
        DisplayMess ('Maximum value allowed here is: ' + s);
        new := old
      End
      Else If new < min Then
      Begin
        Str (min, s);
        ClearMessLine;
        DisplayMess ('Minimum value allowed here is: ' + s);
        new := old
      End
    End
  End;
  GetInt := new
End;   (* GetInt *)

Function GetFileName;
Var new : FileName;                                          (* GetFileName *)
Begin                                                        (* Public      *)
  WriteLine (Message + old, Lo(WindMax) Div 2 - Length (Message),
             Hi(WindMax), MessColour, MenuBackColour);
  GotoXY (Lo(WindMax) Div 2, Hi(WindMax));
  new := GetString;
  If new <> '' Then
  Begin
    If Pos ('.', new) = 0 Then new := new + '.' + ext;
    GetFileName := new
  End
  Else GetFileName := old
End;   (* GetFileName *)

Procedure ShowHeadline;                                     (* ShowHeadline *)
Begin                                                       (* Public       *)
  WriteLine (Logo, Lo(WindMin+1), Hi(WindMin)+1, HeadColour, HeadBackColour);
  WriteLine (MenuTitle, (Lo(WindMax) - Length (MenuTitle)) Div 2,
             Hi(WindMin)+1, HeadColour, HeadBackColour)
End;   (* ShowHeadline *)

Procedure MakeFrame;
Var i, OldTextAttr : Byte;
Begin                                                          (* MakeFrame *)
  (* Move the frame so that it fits in the current window. *)  (* Public    *)
  If (x-2 <= Lo (WindMin)) Then x := Lo (WindMin)+3;
  If (y-1 <= Hi (WindMin)) Then y := Hi (WindMin)+2;
  If (x+w+1 > Lo (WindMax)) Then x := Lo (WindMax)-w-1;
  If (y+h > Hi (WindMax)) Then y := Hi (WindMax)-h;
  If (x-2 <= Lo (WindMin)) Or (y-1 <= Hi (WindMin)) Then
    ErrorMess ('The frame is too large!')
  Else
  Begin
    OldTextAttr := TextAttr;
    TextAttr := FrameColour + 16 * MenuBackColour;
    GotoXY (x-2,y-1);                  (* draw the top line *)
    Write (#218);
    For i := 1 To w+2 Do Write (#196);
    Write (#191);
    For i := 0 To h-1 Do               (* draw the left and *)
    Begin                              (* right line and    *)
      GotoXY (x-2,y+i); Write (#179);  (* clean the window  *)
      Write (' ':w+2);
      Write (#179)
    End;
    GotoXY (x-2,y+h);
    Write (#192);                      (* draw the bottom line *)
    For i := 1 To w+2 Do Write (#196);
    Write (#217);
    TextAttr := OldTextAttr
  End
End;   (* MakeFrame *)

Function UpCase (ch : Char) : Char;
(* Uppercase a letter.                                             *)
(* Adapted from Arne Schaepers, TurboPascal 4.0 - Tips und Tricks. *)
Inline($58/$3C/$61/$72/$39/$3C/$7A/$76/$33/$3C/$84/$75/$02/$B0/$8E
      /$3C/$94/$75/$02/$B0/$99/$3C/$81/$75/$02/$B0/$9A
      /$3C/$87/$75/$02/$B0/$80/$3C/$86/$75/$02/$B0/$8F
      /$3C/$82/$75/$02/$B0/$90/$3C/$91/$75/$02/$B0/$92
      /$3C/$A4/$75/$02/$B0/$A5/$EB/03/90/$2C/$20);

Function OrdKey (Var Hotkeys : HotkeyType; c: Char;
                 PreSel, Last: Byte; Var Selected: Boolean) : Byte;
(* Checks, which ordinary key has been pressed and if this can be           *)
(* associated to a menu line or form item.                                  *)
Const NOff = 48;                       (* Ord ('0')     *)
Var i : Integer;
Begin                                                            (* OrdKey  *)
  If c = Ret Then Selected := True                               (* Private *)
  Else If c = Esc Then
  Begin
    Selected := True;
    PreSel := Last
  End
  Else
  Begin
    i := Ord (c) - NOff;
    If (i >= 0) And (i <= Last) Then
    Begin
      If i = 0 Then i := Last;
      Selected := True;
      PreSel := i
    End
    Else
    Begin
      c := UpCase (c);
      i := Pos (c, Hotkeys);
      If (i <= Last) And (i > 0) Then
      Begin
        Selected := True;
        PreSel := i
      End
      Else Burp
    End
  End;
  OrdKey := PreSel
End;   (* OrdKey *)

(* Next come the procedures to be used by the menu procedures only.         *)

Procedure PrepareMenu (Var Menu : MenuType; Var Hotkeys : HotkeyType;
                       Var NrLines, MaxLength : Byte);
(* Prepares the menu and discovers (hopefully) all errors.                  *)
(* If Hotkeys is empty, the first character of each line is written to its  *)
(* corresponding position. If Hotkeys contains more or less characters than *)
(* the menu has lines, an error is reported and the program halts. If       *)
(* everything was fine, letters have been capitalized in Hotkeys.           *)
(* In NrLines and MaxLength the number of menu lines and the length of the  *)
(* longest menu line are returned. The program halts, if the menu is empty. *)
Var HKLength, i : Byte;
Begin                                                        (* PrepareMenu *)
  NrLines := 0;                                              (* Private     *)
  MaxLength := Length (Menu [0]);
  While (NrLines < MaxLines) And (Length (Menu [NrLines+1]) > 0) Do
  Begin
    Inc (NrLines);                     (* count the lines and *)
    i := Length (Menu [NrLines]);      (* store the length    *)
    If i > MaxLength Then MaxLength := i;
  End;
  If Length (Hotkeys) = 0 Then
    For i := 1 To NrLines Do Hotkeys := Hotkeys + Menu [i,1];
  HKLength := Length (Hotkeys);
  If NrLines = 0 Then ErrorMess ('There are no items in the menu!');
  If NrLines <> HKLength Then
    ErrorMess ('The number of hotkeys doesn''t fit!');
  For i := 1 To HKLength Do Hotkeys [i] := UpCase (Hotkeys [i])
End;   (* PrepareMenu *)

Procedure ShowMenu (Var Menu : MenuType; PreSel, NrLines, x, y, dist : Byte);
(* Display the menu with item PreSel highlighted. The items appear in col-  *)
(* umn x, the first in line y with the others following with distance dist. *)
(* Coordinates are not checked for correctness!                             *)
Var i : Byte;
Begin                                                           (* ShowMenu *)
  For i := 1 To NrLines Do                                      (* Private  *)
  Begin
    GotoXY (x, y+dist*(i-1));
    If i = PreSel Then TextColor (HighColour);
    Write (Menu [i]);
    If i = PreSel Then TextColor (LowColour)
  End
End;   (* ShowMenu *)

Function MenuChoice (Var Menu : MenuType; Var Hotkeys : HotkeyType;
                     PreSel, x, y, dist : Byte) : Byte;
(* PreSel is the item, which the cursor points to. x, y , dist are used as  *)
(* in ShowMenu above. MenuChoice waits until an item is selected.           *)
(* Coordinates are not checked for correctness!                             *)
Var Selected : Boolean;
    NrLines : Byte;
    c : Char;

Function ExtKey (c: Char; PreSel: Byte) : Byte;
(* Checks, which extended key has been pressed. c contains the extended key. *)
Const Up   = #72;                      (* <cursor up>   *)
      Down = #80;                      (* <cursor down> *)
      Home = #71;                      (* <Home>        *)
      Nd   = #79;                      (* <End>         *)
Begin                                                             (* ExtKey *)
  Case c Of
    Up   : If PreSel = 1 Then PreSel := NrLines
           Else Dec (PreSel);
    Down : If PreSel = NrLines Then PreSel := 1
           Else Inc (PreSel);
    Home : PreSel := 1;
    Nd   : PreSel := NrLines
    Else Burp
  End;
  ExtKey := PreSel
End;   (* ExtKey *)

Begin                                                         (* MenuChoice *)
  Selected := False;                                          (* Private    *)
  NrLines := Length (Hotkeys);
  Repeat
    Repeat Until KeyPressed;
    GotoXY (x, y+dist*(PreSel-1));
    Write (Menu [PreSel]);
    c := ReadKey;
    If c = #0 Then
    Begin
      c := ReadKey;
      PreSel := ExtKey (c, PreSel)
    End
    Else PreSel := OrdKey (Hotkeys, c, PreSel, NrLines, Selected);
    GotoXY (x, y+dist*(PreSel-1));
    TextColor (HighColour);
    Write (Menu [PreSel]);
    TextColor (LowColour)
  Until Selected;
  MenuChoice := PreSel
End;   (* MenuChoice *)

(* And here are the procedures that manage the menus.                       *)
(* First, the standard menu procedure:                                      *)

Function StdMenu;
Var NrLines, MaxLength, XPos, YPos, OldTextAttr : Byte;
Begin                                                            (* StdMenu *)
  PrepareMenu (Menu, Hotkeys, NrLines, MaxLength);               (* Public  *)
  XPos := (Lo(WindMax) - Lo(WindMin) - MaxLength) Div 2;
  YPos := Hi(WindMin) + 4;
  If PreSel < 1 Then PreSel := 1
  Else If PreSel > NrLines Then PreSel := NrLines;
  OldTextAttr := TextAttr;
  TextAttr := LowColour + 16 * MenuBackColour;
  ClrScr;
  ShowHeadline (Logo, Menu [0]);
  MakeFrame (XPos, YPos, MaxLength, 2*NrLines-1);
  ShowMenu (Menu, PreSel, NrLines, XPos, YPos, 2);
  PreSel := MenuChoice (Menu, Hotkeys, PreSel, XPos, YPos, 2);
  TextAttr := OldTextAttr;
  StdMenu := PreSel
End;   (* StdMenu *)

(* Next, the small pop-up menu procedure:                                   *)

Function PopMenu;
Var NrLines, MaxLength, OldTextAttr : Byte;
Begin                                                            (* PopMenu *)
  PrepareMenu (Menu, Hotkeys, NrLines, MaxLength);               (* Public  *)
  If PreSel < 1 Then PreSel := 1
  Else If PreSel > NrLines Then PreSel := NrLines;
  OldTextAttr := TextAttr;
  TextAttr := LowColour + 16 * MenuBackColour;
  MakeFrame (XPos, YPos, MaxLength, NrLines);
  If Length (Menu [0]) > 0 Then
  Begin
    GotoXY (XPos, YPos-1);
    Write (Menu [0])
  End;
  ShowMenu (Menu, PreSel, NrLines, XPos, YPos, 1);
  PreSel := MenuChoice (Menu, Hotkeys, PreSel, XPos, YPos, 1);
  TextAttr := OldTextAttr;
  PopMenu := PreSel;
End;   (* PopMenu *)

(* Here come the procedures to be used with forms.                          *)

Procedure PrepareForm (Var Form : FormType; Var Hotkeys : HotkeyType;
                       Var NrItems : Byte);
(* Prepares the menu and discovers (hopefully) all errors.                  *)
(* If Hotkeys is empty, the first character of each form item is written to *)
(* its corresponding position. If Hotkeys contains more or less characters  *)
(* than the form has items, an error is reported and the program halts.     *)
(* If everything was fine, letters have been capitalized in Hotkeys. In     *)
(* NrItems the number of form items is returned. The program halts, if the  *)
(* form is empty. Form entries are not cared for.                           *)
Var HKLength, i : Byte;
Begin                                                        (* PrepareForm *)
  NrItems := 0;                                              (* Private     *)
  While (NrItems < MaxItems)
    And (Length (Form [NrItems+1].Item) > 0) Do
      Inc (NrItems);                       (* count the items *)
  If Length (Hotkeys) = 0 Then
    For i := 1 To NrItems Do Hotkeys := Hotkeys + Form [i].Item [1];
  HKLength := Length (Hotkeys);
  If NrItems = 0 Then ErrorMess ('There are no items in the form!');
  If NrItems <> HKLength Then
    ErrorMess ('The number of hotkeys doesn''t fit!');
  For i := 1 To HKLength Do Hotkeys [i] := UpCase (Hotkeys [i])
End;   (* PrepareForm *)

Procedure ShowForm (Var Form : FormType; PreSel, NrItems : Byte);
(* Display the form with item PreSel highlighted. *)
(* Coordinates are not checked for correctness!   *)
Var i : Byte;
Begin                                                           (* ShowForm *)
  For i := 1 To NrItems Do                                      (* Private  *)
  Begin
    With Form [i] Do
    Begin
      If Active Then
      Begin
        GotoXY (x, y);
        If i = PreSel Then TextColor (HighColour);
        Write (Item, Entry);
        If i = PreSel Then TextColor (LowColour)
      End
    End
  End
End;   (* ShowForm *)

Function FormChoice (Var Form : FormType; Var Hotkeys : HotkeyType;
                     PreSel : Byte) : Byte;
(* PreSel is the item, which the cursor points to. FormChoice waits until   *)
(* an item is selected. Coordinates are not checked for correctness!        *)
Var Selected : Boolean;
    NrItems : Byte;
    c : Char;

Function ExtKey (c: Char; PreSel: Byte) : Byte;
(* Checks, which extended key has been pressed. c contains the extended key. *)
(* Uses the pointer components of Form, which is not passed as parameter.    *)
Const Up    = #72;                      (* <cursor up>    *)
      Down  = #80;                      (* <cursor down>  *)
      Right = #77;                      (* <cursor right> *)
      Left  = #75;                      (* <cursor left>  *)
      Home  = #71;                      (* <Home>         *)
      Nd    = #79;                      (* <End>          *)
Begin                                                             (* ExtKey *)
  Case c Of
    Up    : If Form [PreSel].u < 0 Then Burp
            Else PreSel := Form [PreSel].u;
    Down  : If Form [PreSel].d < 0 Then Burp
            Else PreSel := Form [PreSel].d;
    Right : If Form [PreSel].r < 0 Then Burp
            Else PreSel := Form [PreSel].r;
    Left  : If Form [PreSel].l < 0 Then Burp
            Else PreSel := Form [PreSel].l;
    Home  : PreSel := 1;
    Nd    : PreSel := NrItems
    Else Burp
  End;
  ExtKey := PreSel
End;   (* ExtKey *)

Begin                                                         (* FormChoice *)
  Selected := False;                                          (* Private    *)
  NrItems := Length (Hotkeys);
  Repeat
    Repeat Until KeyPressed;
    With Form [PreSel] Do
    Begin
      GotoXY (x, y);
      TextColor (LowColour);
      Write (Item, Entry)
    End;
    c := ReadKey;
    If c = #0 Then
    Begin
      c := ReadKey;
      PreSel := ExtKey (c, PreSel)
    End
    Else PreSel := OrdKey (Hotkeys, c, PreSel, NrItems, Selected);
    With Form [PreSel] Do
    Begin
      GotoXY (x, y);
      TextColor (HighColour);
      Write (Item, Entry);
      TextColor (LowColour)
    End
  Until Selected;
  FormChoice := PreSel
End;   (* FormChoice *)


(* And here is the procedure that manages the forms.                        *)

Function StdForm;
Var NrItems, OldTextAttr : Byte;
Begin                                                            (* StdForm *)
  PrepareForm (Form, Hotkeys, NrItems);                          (* Public  *)
  If PreSel < 1 Then PreSel := 1
  Else If PreSel > NrItems Then PreSel := NrItems;
  OldTextAttr := TextAttr;
  TextAttr := LowColour + 16 * MenuBackColour;
  ClrScr;
  ShowHeadline (Logo, Form [0].Item);
  ShowForm (Form, PreSel, NrItems);
  PreSel := FormChoice (Form, Hotkeys, PreSel);
  TextAttr := OldTextAttr;
  StdForm := PreSel
End;   (* StdForm *)


Begin                                             (* Initialization of Menu *)
  MessColour := LightGray;
  LowColour := LightGray;
  HighColour := White;
  MenuBackColour := Black;
  HeadColour := Black;
  HeadBackColour := LightGray;
  FrameColour := LightGray;
  MessDelay := 1000;
End.
