/******************************************************************************
Fractal Infinity - HARDWARE SPECIFIC FUNCTIONS - fr_hware.c.
This file Copyright (C) 1999 Graeme Fenwick- see documentation for details.
******************************************************************************/

#define  alleg_mouse_unused                         /* save compilation time */
#define  alleg_timer_unused
#define  alleg_joystick_unused
#define  alleg_flic_unused
#define  alleg_sound_unused
#define  alleg_gui_unused
#include <allegro.h>
#include <limits.h>
#include <unistd.h>
#include <sys/vfs.h>
#include "fr_gen.h"
#include "fr_hware.h"

PALETTE                    fake_true;
int                        sxdim = 0, sydim = 0, sdeep = 0;

static unsigned long int   black, white;

/********** HARDWARE_SETUP: Initialize Allegro handlers, and set up screen.
                            Successful only if screen ok & values stored ok. */

int hardware_setup(int gfx_x, int gfx_y, int gfx_depth)
{
   allegro_init();                                  /* (required by Allegro) */
   install_keyboard();                       /* replace standard key handler */
   set_color_depth(sdeep = gfx_depth);
   if (set_gfx_mode(GFX_VESA2L, sxdim = gfx_x, sydim = gfx_y, 0, 0) == -1)
      return 0;                                      /* couldn't open screen */
   if (gfx_depth == 8) {
      generate_332_palette(fake_true);           /* pseudo truecolor palette */
      set_palette(fake_true);
   }
   black = makecol(0, 0, 0);
   white = makecol(255, 255, 255);
   clear_to_color(screen, black);                            /* cls to black */
   text_mode(black);                          /* solid black text background */
   return 1;
}

/********** TOGGLE_DRAW_MODE: Switch between solid and exor drawing */

int toggle_draw_mode(void)
{
   static int   mode = 0;

   mode = !mode;                              /* toggle current drawing mode */
   drawing_mode((mode == 0) ? DRAW_MODE_SOLID : DRAW_MODE_XOR, NULL, 0, 0);
   return mode;                           
}

/********** PLOT_POINT: Screen independent color pixel plot. */

unsigned long int plot_point(int sx, int sy, int maxval, int maxit, int it)
{
   float                     h, s, v;
   int                       col, r, g, b;
   
   iteration_color(it, maxit, maxval, &h, &s, &v);
   hsv_to_rgb(h, s, v, &r, &g, &b);
   putpixel(screen, sx, sy, col = makecol(r, g, b));
   return col;                                       /* actual color plotted */
}

/********** INFO_DISPLAY: Display Fractal Information at bottom of screen
                          (Resolution/ Depth independent)                    */

void info_display(long double mag, long double range,
                  long double xc,  long double yc)
{
   textprintf(screen, font, 0, sydim - 16, white, "X  : %-+12.6Le ", xc);
   textprintf(screen, font, sxdim / 3 - 1, sydim - 16, white,
      "Y: %-+12.6Le ", yc);
   textprintf(screen, font, (int) (sxdim * (2.0 / 3) -1), sydim - 16, white,
      "RNG: %-+12.6Le ", range);
   textprintf(screen, font, 0, sydim - 8, white, "MAG: %-+5.5Lf ", mag);
}

/********** DRAW_XHAIR: (at physical screen coordinates) */

int draw_xhair(int sx, int sy)
{
   hline(screen, 0, sy, sxdim - 1, white);
   vline(screen, sx, 0, sydim - 17, white);
   return 1;
}

/********** SAVE_DISPLAY: Saves only visible (physical) portion of screen */

int save_display(char *fname)
{
   BITMAP   *temp;                      /* temp sub-bitmap of visible screen */
   int      rstat;                                          /* return-status */

   temp = create_sub_bitmap(screen, 0, 0, sxdim, sydim);
   if (fname == NULL)
      return 0;                                      /* empty filename check */
   else
      rstat = !save_bmp(fname, temp, (sdeep == 8) ? fake_true : NULL);
   destroy_bitmap(temp);
   return rstat;                                    /* nonzero if successful */
}

/********** DETECT_KEY: Custom routine only detects relevant keys. Some keys
                        have aliases (e.g. numpad '+' matches '=')           */

int detect_key(int flush)
{
   unsigned int   key;

   if (flush) {
      clear_keybuf();                       /* double up as key-buffer flush */
      return FLUSH;
   }
   if (!keypressed())
      return NO_KEY;                                         /* buffer empty */
   key = (unsigned) readkey() >> 8;
   switch (key) {                         /* note aliases and lack of breaks */
   case KEY_UP:
   case KEY_8:
      return UP;
   case KEY_DOWN:
   case KEY_2:
      return DOWN;
   case KEY_LEFT:
   case KEY_4:
      return LEFT;
   case KEY_RIGHT:
   case KEY_6:
      return RIGHT;
   case KEY_PLUS_PAD:
   case KEY_EQUALS:
      return PLUS;
   case KEY_MINUS:
   case KEY_MINUS_PAD:
      return MINUS;
   case KEY_1:
      return ONE;
   case KEY_ENTER:
      return ENTER;
   case KEY_S:
   case KEY_SLASH:
   case KEY_SLASH2:
      return SAVE;
   case KEY_SPACE:
      return SPACE;
   case KEY_Q:
      return QUIT;
   default:
      return NONVALID;                               /* nonvalid key pressed */
   }
}

/********** SLEEP_WRAP: Exactly the same as common sleep() function, but
                        using a wrapper to keep other files ANSI */

unsigned sleep_wrap(unsigned seconds)
{
   return sleep(seconds);
}

/********** MESSAGE_DIALOG: 2 Line dialog box routine- disappears after key
                            pressed. Returns False if text is too wide */

int message_dialog(char *mes1, char *mes2)
{
   BITMAP   *temp;
   char     *press = "PRESS ANY KEY TO CONTINUE";
   int      width, sxc, syc, left_x, right_x, top_y, bot_y;

   width = (strlen(mes1) > strlen(mes2)) ? strlen(mes1) : strlen(mes2);
   width = (strlen(press) > width) ? strlen(press) : width;
   width = (width + 1) * 8;
   if (width > sxdim)                                /* does dialog box fit? */
      return 0;
   sxc = (sxdim / 2) - 1;                           /* find center of screen */
   syc = ((sydim - 16) / 2) - 1;
   left_x = sxc - width / 2 + 1;                    /* corners of dialog box */
   right_x = sxc + width / 2;
   top_y = syc - (2.5 * 8) + 1;
   bot_y = syc + (2.5 * 8);
   temp = create_bitmap(width, 40);               /* backup center of screen */
   blit(screen, temp, left_x, top_y, 0, 0, width, 40);
   rectfill(screen, left_x, top_y, right_x, bot_y, black);       /* dlg. box */
   rect(screen, left_x, top_y, right_x, bot_y, white);
   textout_centre(screen, font, mes1, sxc, syc - 15, white);
   textout_centre(screen, font, mes2, sxc, syc - 7,  white);
   textout_centre(screen, font, press, sxc, syc + 9, white);
   detect_key(1);                                 /* wait for *new* keypress */
   while (detect_key(0) == NO_KEY)
      ;
   blit(temp, screen, 0, 0, left_x, top_y, width, 40);     /* restore center */
   destroy_bitmap(temp);
   return 1;
}

/********** CRT0_GLOB_FUNCTION : Existence of this function stops DJGPP
                                 doing Unix-style globbing of CLI args */

char **__crt0_glob_function(char *_arg)
{
   return NULL;
}

/********** VERIFY_IMAGESPACE : Checks if there is enough space to save current
                                screen to disk */

long verify_imagespace(void)
{
   struct statfs   cur_vol;                       /* current directory stats */
   long            s_req, s_avail;

   if (sdeep == 8)
      s_req = sxdim * sydim + 1024 + 54;
   else
      s_req = 3 * sxdim * sydim + 54;
   statfs(NULL, &cur_vol);                       /* get current volume stats */
   s_avail = cur_vol.f_bsize * cur_vol.f_bfree;
   return s_avail - s_req;             /* if negative, not enough free space */
}


