;*************************************************************************
;* MODULE INFORMATION*
;*********************
;*     FILE     NAME:       clocka.asm
;*     SYSTEM   NAME:       CLOCK
;*     ORIGINAL AUTHOR(S):  Alfred Kayser
;*     VERSION  NUMBER:     1.2
;*     CREATION DATE:       91/01/30
;*
;* DESCRIPTION: Micro-, and milliseconds timer for x86 systems.
;*              Measures time with a granularity of 0.838574 seconds.
;*              Relative error is 0.000125. (or 0.01%)
;*              (- (/ 54965.0 65536) (/ 1.0 1.1925)) => 0.000124917
;**********************************************************************
;* CHANGES INFORMATION **
;************************
;* REVISION:    $Revision:   1.1  $
;* CHANGER:     $Author:   etstjan  $
;* WORKFILE:    $Workfile:   CLOCK.ASM  $
;* LOGFILE:     $Logfile:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/CLOCK.ASV  $
;* LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/NPD/DP/VCS/CLOCK.ASV  $
;*              
;*                 Rev 1.1   01 Feb 1991 14:21:24   etstjan
;*              
;*                 Rev 1.0   26 Oct 1990 12:28:14   etstjan
;*              Initial revision.
;*                 Rev 1.1   26 Oct 1990 12:28:14   etstjan
;*              Initial revision.
;*                 Rev 1.2   30 Jan 1991 12:28:14   alfred
;*              Removed overflow bug.
;*              Now even less code! Also less time interrupts disabled!
;*********************************************************************/
IFDEF MODEL_L
.model large
ENDIF
IFDEF MODEL_S
.model small
ENDIF
IFDEF MODEL_M
.model medium
ENDIF

.code
T1      equ     54957
        public  _clock_init
        public  _clock_us
        public  _clock_ms

;*****************************************************************
;* NAME:        clock_init
;* SYNOPSIS:    void clock_init(void);
;* DESCRIPTION: Initializes clock system.
;* RETURNS:     void.
;*****************************************************************
_clock_init proc
        mov   al,34h          ; control word
        cli                   ; disable interrupts
        out   43h,al          ; outp(0x43,0x34)  /* read latches */
        xor   al,al
        out   40h,al          ; outp(0x43,0)  /* set latches */
        out   40h,al          ; outp(0x43,0)  /* set latches */
        sti                   ; renable them
        ret
_clock_init endp

;***************************************************************
;* NAME:        clock_us
;* SYNOPSIS:    clock_t clock_us(void);
;* DESCRIPTION: Measures time ellapses in microseconds.
;*              Time granularity is in 0.8386 microsecs.
;*              Return value is a long integer containing the 
;*              number of seconds since last 0:00 am.
;*              Therefore the maximum time that can be measured 
;*              is about one hour. (2^32 => 4.000.000.000 secs 
;*              => 4000 secs => 1 hr, 6 mins, 40 secs.)
;* PERFORMANCE: Total count of states is 133+m. This is on a
;*              386 in real mode, excluding the call. <m> is the
;*              number of bytes of the instruction after the call.
;*              25 Mhz. timing: 5.32 microsecs. 
;* RETURNS:     Nr of microsecs since last 0:00 am
;*****************************************************************
_clock_us proc                               ;386 timing (real mode);
        xor   al,al                                                 ; 2   2
        mov   dx,40H          ; select timer control register       ; 2   4
        mov   es,dx           ; es = 40h                            ; 2   5

        cli                   ; disable interrupts                  ; 3   8
        out   43h,al          ; outp(0x43,0)  /* read latches */    ;10  18
        mov   dx,40H          ; Select timer data register          ; 2  20
        in    al,dx           ; al = inp(0x40)                      ;13  33
        mov   ah,al           ; ax = al << 8;                       ; 2  35
        in    al,dx           ; ax = (ax & 0xff00) + inp(0x40)      ;13  48
        mov   bx,es:[6ch]     ; read lower word of bios_cnt         ; 4  52
        sti                   ; renable them                        ; 3  55

        xchg  ah,al           ; wrong way aroung, LSB is read first.; 3  58
        not   ax              ; timer counts down                   ; 2  60
        mov   cx,T1           ;                                     ; 2  62
        mul   cx              ; ax:dx = timer * T1                  ;22  84
        xchg  cx,dx           ; store high word in CX, T1 in AX     ; 3  87
        mov   ax, bx          ; lower word of bios_cnt in AX        ; 2  89
        mul   dx              ; ax:dx = bios_cnt * T1               ;22 111
        add   ax,cx           ; add higher word (timer*T1)          ; 2 113
        adc   dx,0            ; add carry                           ; 2 115
        ret                                                         ;18 133
_clock_us endp                                                      ;+m +m
   
;*****************************************************************
;* NAME:        clock_ms
;* SYNOPSIS:    clock_t clock_ms(void);
;* DESCRIPTION: Measures time ellapses in milliseconds.
;*              Time granularity is in 0.8386 microsecs.
;*              Remember: at 0:00.00 the counters are reset.
;* PERFORMANCE: Total count of states is 221+m. This is on a
;*              386 in real mode, excluding the call. <m> is the
;*              number of bytes of the instruction after the call.
;*              25 Mhz. timing: 8.84 microsecs.
;* RETURNS:     Nr of millisecs since last 0:00 am.
;*****************************************************************
_clock_ms proc
        xor   al,al           ;                                         2   2
        mov   dx,40H          ; Select timer data register              2   4
        mov   es,dx           ; es = 40h (same as port number!!)        2   6

        cli                   ; disable interrupts                      3   9
        out   43h,al          ; outp(0x43,0)  /* read latches */       10  19
        in    al,dx           ; al = inp(0x40)                         13  32
        mov   ah,al           ; ax = al << 8;                           2  34
        in    al,dx           ; ax = (ax & 0xff00) + inp(0x40)         13  47
        mov   bx,es:[6ch]     ; read lower word of bios_cnt (A)         4  51
        mov   cx,es:[6eh]     ; read higher word of bios_cnt (B)        4  55
        sti                   ; renable them                            3  58

        xchg  ah,al           ; wrong way around, LSB is read first.    3  61
        not   ax              ; timer counts down                       2  63
        mov   dx,T1           ;                                         2  65
        mul   dx              ; ax:dx = timer * T1                     22  87

        mov   ax,bx           ; lower word of bios_cnt                  2  89
        mov   bx,dx           ; store high word of (timer*T1) in bx     2  91
        mov   dx,T1           ;                                         2  93
        mul   dx              ; ax:dx = bios_cnt*T1                    22 115
        add   ax,bx           ; add high word of timer*T1 to A*T1       2 117  
        adc   dx,0            ; carry...                                2 119

        mov   bx,dx           ; higher word of A*T1 in BX               2 121
        xchg  ax,cx           ; high part of bios_cnt (B) in AX,        3 124
                              ; lower word of A*T1 in CX
        mov   dx,T1           ;                                         2 126
        mul   dx              ; dx:ax = B*T1                           22 148
        add   ax,bx           ; add high word of A*T1 to B*T1           2 150
        adc   dx,0            ; carry...                                2 152

        mov   bx,1000         ; divide by 1000.                         2 154
        div   bx              ; quo in AX, rem in DX                   22 176
        xchg  cx,ax           ; store (B*T1)/1000 in CX                 3 179
                              ; lower word of A*T1 in AX
                              ; remainder stays in DX (DX=(B*T1)%1000)
        div   bx              ; divide total (DX:AX) by 1000           22 201
        mov   dx,cx           ; dx=(B*T1)/1000,                         2 203
                              ; ax=((B*T1)%1000)<<16+(A*T1)+(T*T1)>>16)/1000
        ret                   ; exit                                   18 221
_clock_ms endp

         end

