/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       SOCKET.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/6/27
**
** DESCRIPTION: Contains Application Level Socket Library routines.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.2  $
** WORKFILE:    $Workfile:   SOCKET.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/SOCK/VCS/SOCKET.C_V  $
**              
**                 Rev 1.2   01 Feb 1991 11:24:38   etstjan
**              No explicit note
**              
**                 Rev 1.1   21 Nov 1990 14:32:18   etstjan
**              No explicit note
**              
**                 Rev 1.0   21 Nov 1990 12:27:34   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/SOCK/VCS/SOCKET.C_V   1.2   01 Feb 1991 11:24:38   etstjan  $";
#endif
#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <dnpap.h>
 
#include      "ip.h"                   /* general ip header             */
#include      "ipcodes.h"              /* and ip return codes           */
#include      "iplib.h"                /* include ip library            */
#include      "ipif.h"                 /* interface layer               */
#include      "ipevents.h"             /* event processing              */
#include      "iplayer.h"              /* general ip layer header file  */
#include      "socklib.h"              /* socket library header file    */
#include      "socket.h"

/* prototypes */
int SetPortAdd(SOCKET s, struct sockaddr *name, int namelen,
               int Command);

/**************************************************************
** NAME:        socket                                    [API]
** SYNOPSIS:    SOCKET socket(int Domain,
**                            int Type,
**                            int Protocol);
** DESCRIPTION: Creates a new socket. At this moment, the
**              following constraints apply:
**              Type should be SOCK_DGRAM or SOCK_RAW;
**              Domain should be AF_INET;
**              Protocol should be 0 (autodetect) or UDP.
** RETURNS:     NULL  --> Failed to create socket:
**                        - parameter error;
**                        - no buffer space.     
**              else  --> Pointer to the newly created
**                        socket.
**************************************************************/
SOCKET socket(int Domain, int Type, int Protocol)
{
  /* check validity of parameters */
  if (((Type != SOCK_DGRAM) && (Type != SOCK_RAW)) ||
      (Domain != AF_INET) ||
      ((Protocol != 0) && (Protocol != UDP))) {
    return NULL;
    }

  return CreateSocket(Type, Domain, Protocol);
}

/**************************************************************
** NAME:        closesocket                               [API]
** SYNOPSIS:    int closesocket(SOCKET thissocket);
**           
** DESCRIPTION: Releases thissocket. The protocol is requested
**              to detach the socket. If no error occures in
**              detaching the socket, the memory space for
**              the socket is freed as well.
** RETURNS:     NO_ERR -->   no error
**              else         error code
**************************************************************/
int closesocket(SOCKET thissocket)
{
  return _RelSocket(thissocket);
}

/**************************************************************
** NAME:        bind                                      [API]
** SYNOPSIS:    int bind(SOCKET s,
**                       struct sockaddr *name,
**                       int namelen)
** DESCRIPTION: Binds the address in name to the socket s.
**              If the address family is AF_UNSPEC, the
**              socket is unbound.
**              AF_INET is the only supported address family.
**              If Port = 0, the protocol will assign a port,
**              selected out of the non priviliged user
**              space. If s_addr = 0, the address wil not be
**              bound.
** RETURNS:     NOERR      --> no error
**              ILLFAMILY  --> illegal address family
**              ILLADDSIZE --> wrong address size
**************************************************************/
int bind(SOCKET s, struct sockaddr *name, int namelen)
{
  switch (((struct sockaddr_in *)name)->sin_family) {
    case AF_UNSPEC :
      (*(((struct _socket *)s)->So_Prot->UserReq))
       (PRU_DEBIND, s, NULL, NULL, NULL);
      return NOERR;
    case AF_INET :
      return SetPortAdd(s, name, namelen, PRU_BIND);
    default :
      return ILLFAMILY;
    }
}


/**************************************************************
** NAME:        connect                                   [API]
** SYNOPSIS:    int connect(SOCKET s,
**                          struct sockaddr *name,
**                          int namelen)
** DESCRIPTION: Connects socket s to the name in name.
**              If the address family is AF_UNSPEC, the
**              socket is disconnected.
**              AF_INET is the only supported address family.
**              If Port = 0, the remote port will be
**              disconnected.If s_addr = 0, the address
**              wil be disconnected.
** RETURNS:     NOERR      --> no error
**              ILLFAMILY  --> illegal address family
**              ILLADDSIZE --> wrong address size
**************************************************************/
int connect(SOCKET s, struct sockaddr *name, int namelen)
{
  switch (((struct sockaddr_in *)name)->sin_family) {
    case AF_UNSPEC :
      (*(((struct _socket *)s)->So_Prot->UserReq))
       (PRU_DISCONNECT, s, NULL, NULL, NULL);
      return NOERR;
    case AF_INET :
      return SetPortAdd(s, name, namelen, PRU_CONNECT);
    default :
      return ILLFAMILY;
    }
}

/**************************************************************
** NAME:        SetPortAdd
** SYNOPSIS:    int SetPortAdd(SOCKET s,
**                             struct sockaddr *name,
**                             int namelen,
**                             int Command)
**              
** DESCRIPTION: Calls the Protocol User Service routine,
**              with Command as request. 
** RETURNS:     NOERR      --> Everything just fine
**              ILLADDSIZE --> wrong address size
**              else       --> Protocol Request return code
**************************************************************/
int SetPortAdd(SOCKET s, struct sockaddr *name, int namelen,
               int Command)
{
  SO_ADDRESS          ThisAddress;
  struct sockaddr_in  *InAdPnt = (struct sockaddr_in *)name;
  
  if (namelen != sizeof(struct sockaddr_in)) {
    return ILLADDSIZE;
    }
  ThisAddress.Port = ntohs(InAdPnt->sin_port);
  ThisAddress.IPAddress = InAdPnt->sin_addr.s_addr;
  return (*(((struct _socket *)s)->So_Prot->UserReq))
          (Command, s, NULL, &ThisAddress, NULL);
}

/**************************************************************
** NAME:        send                                      [API]
** SYNOPSIS:    int send(SOCKET s, char *msg,
**                       int len, int flags)
** DESCRIPTION: Send message indicated by msg with length len
**              through socket s. The socket must be connected
**              before using this call.
**              The flags parameter is present for
**              compatability reasons, should be 0.
** RETURNS:     NOERR      -->   no error
**              NOTBOUND   -->   socket not bound                      
**              NOTCONNECT -->   socket not connected                  
**              else       -->   general error code
**************************************************************/
int send(SOCKET s, char *msg, int len, int flags)
{
  struct sockaddr_in  ToAddress;

  if (!LOCALBOUND(((struct _socket *)s))) {
    return NOTBOUND;
    }
  if (!REMOTEBOUND(((struct _socket *)s))) {
    return NOTCONNECT;
    }
  ToAddress.sin_family = AF_INET;
  ToAddress.sin_port = htons(((struct _socket *)s)->So_RemAdd.Port);
  ToAddress.sin_addr.s_addr = ((struct _socket *)s)->So_RemAdd.IPAddress;
  return sendto(s, msg, len, flags,
                (struct sockaddr *)&ToAddress, sizeof(struct sockaddr_in));
}

/**************************************************************
** NAME:        sendto                                    [API]
** SYNOPSIS:    int sendto(SOCKET s, char *msg,
**                         int len, int flags,
**                         struct sockaddr *to, int tolen);
** DESCRIPTION: Send message indicated by msg with length len
**              through socket s. The socket may be connected
**              but the local port must be bound.
**              The address specified in to is used as the
**              destination address.
**              The flags parameter is present for
**              compatability reasons, should be 0.
** RETURNS:     NOERR      -->   no error
**              NOTBOUND   -->   socket not bound                      
**              else       -->   general error code
**************************************************************/
int sendto(SOCKET s, char *msg, int len, int flags,
           struct sockaddr *to, int tolen)
{
  SO_ADDRESS  ToAddress;
  DATALINK    *DataBuf;
  int         ErrCode;

  if (!LOCPORTBOUND(((struct _socket *)s))) {
    return NOTBOUND;
    }

  if (tolen != sizeof(struct sockaddr_in)) {
    return ILLADDSIZE;
    }
  if ((ErrCode = ((struct _socket *)s)->So_Error) != NOERR) {
    ((struct _socket *)s)->So_Error = NOERR;
    return ErrCode;
    }

  if ((DataBuf = IPDataLinkGet(len)) == NULL) {
    IPReport(NOSPACE, NULL, NULL);
    return NOBUFS;
    }

  DataBuf->ThisType = DATAHD;
  memcpy(DataBuf->DataStart, msg, len);   /* copy message from user space */

  ToAddress.Port = ntohs(((struct sockaddr_in *)to)->sin_port);
  ToAddress.IPAddress = ((struct sockaddr_in *)to)->sin_addr.s_addr;

  return (*(((struct _socket *)s)->So_Prot->UserReq))
          (PRU_SENDTO, s, DataBuf,
           &(((struct _socket *)s)->So_LocAdd), &ToAddress);
}

/**************************************************************
** NAME:        recv                                      [API]
** SYNOPSIS:    int recv(SOCKET s, char *buf,
**                       int len, int flags);
** DESCRIPTION: Receives data from socket s. The data is
**              truncated at len bytes and placed in buf.
**              If flags = MSG_PEEK, data is not removed
**              from the socket.
** RETURNS:     >= 0      --> number of bytes received.
**              NOTBOUND  --> socket not bound
**              NOMESSAGE --> no message received yet
**              else      --> general error code
**************************************************************/
int recv(SOCKET s, char *buf, int len, int flags)
{
  struct _socket *ThisSocket = s;
  int            i;
  SO_PACKLIST    *ThisPack;

  if (!LOCPORTBOUND(ThisSocket)) {
    return NOTBOUND;
    }
  if ((ThisPack = ThisSocket->RecQueue) == NULL) {
    return NOMESSAGE;
    }
  for (i = 0; (i < len) && (i < (int)(ThisPack->Packet->Length)); i++) {
    buf[i] = ThisPack->Packet->DataStart[i]; 
    }
  if (!(flags & MSG_PEEK)) {
    ThisSocket->RecQueue = ThisPack->NextPack;
    FreeSoPack(ThisPack);
    ThisSocket->So_RCurLen--;
    }    
  return i;
}

/**************************************************************
** NAME:        recvfrom                                  [API]
** SYNOPSIS:    int recvfrom(SOCKET s, char *buf,
**                       int len, int flags,
**                       struct sockaddr *from, int *fromlen);
** DESCRIPTION: Receives data from socket s. The data is
**              truncated at len bytes and placed in buf.
**              If from is non zero, the source address is
**              filled in and the length of the address is
**              placed in fromlen.
**              If flags = MSG_PEEK, data is not removed
**              from the socket.
** RETURNS:     >= 0      --> number of bytes received.
**              NOTBOUND  --> socket not bound
**              NOMESSAGE --> no message received yet
**              else      --> general error code
**************************************************************/
int recvfrom(SOCKET s, char *buf, int len, int flags,
             struct sockaddr *from, int *fromlen)
{
  struct _socket *ThisSocket = s;
  SO_PACKLIST    *ThisPack;

  if ((ThisPack = ThisSocket->RecQueue) == NULL) {
    return NOMESSAGE;
    }
  if (*fromlen < sizeof(struct sockaddr_in)) {
    return ILLADDSIZE;
    }
  ((struct sockaddr_in *)from)->sin_family = AF_INET;
  ((struct sockaddr_in *)from)->sin_addr.s_addr = ThisPack->Source->Source;
  ((struct sockaddr_in *)from)->sin_port = *(int *)(ThisPack->Packet->DataStart - 8);
  *fromlen = sizeof(struct sockaddr_in);
  return recv(s, buf, len, flags);
}

/**************************************************************
** NAME:        getsockopt                                [API]
** SYNOPSIS:    int getsockopt(SOCKET s, int level,
**                             int optname,
**                             char *optval, int *optlen);
**
** DESCRIPTION: Delivers a socket option from protocol level.
**              The name of the option must be in optname,
**              the protocol in level.
**              The option value is copied in optval, the 
**              length is returned in optlen.
** RETURNS:     NOERR      --> no error
**              ILLPROT    --> illegal protocol level
**              ILLOPTNAME --> illegal option name
**              ILLOPTVAL  --> illegal option value
**************************************************************/
int getsockopt(SOCKET s, int level, int optname,
               char *optval, int *optlen)
{
  struct _socket   *Soc = (struct _socket *)s;

  switch (level) {
    case SOL_SOCKET :
      switch (optname) {
        case SO_ERROR :
          *(int *)optval = Soc->So_Error;
          Soc->So_Error = NOERR;
          *optlen = sizeof(int);
          return NOERR;
        default :
          return ILLOPTNAME;
        }
    default :
      return ILLPROTOC;
    }
}

/**************************************************************
** NAME:        setsockopt                                [API]
** SYNOPSIS:    int setsockopt(SOCKET s, int level,
**                             int optname,
**                             char *optval, int optlen);
**
** DESCRIPTION: Delivers a socket option to protocol level.
**              The name of the option must be in optname,
**              the protocol in level.
**              The option value is copied in optval, the 
**              length is returned in optlen.
** RETURNS:     NOERR      --> no error
**              ILLPROT    --> illegal protocol level
**              ILLOPTNAME --> illegal option name
**              ILLOPTVAL  --> illegal option value
**************************************************************/
int setsockopt(SOCKET s, int level, int optname,
               char *optval, int optlen)
{
  switch (level) {
    case 0 :
      switch (optname) {
        case IP_OPTIONS :
          return SetIPOptions((struct _socket *)s, optval, optlen);
        default :
          return ILLOPTNAME;
        }
    default :
      return ILLPROTOC;
    }
}

/**************************************************************
** NAME:        getsockname                               [API]
** SYNOPSIS:    int getsockname(SOCKET s,
**                              struct sockaddr *name,
**                              int *namelen);
**  
** DESCRIPTION: Returns the current name for socket 's'.
**              'namelen' should indicate the bufferspace
**              for 'name'. On return, 'namelen' is set
**              to the actual size.
** RETURNS:     NOERR       -->  no error
**              ILLADDSIZE  -->  not enough space for address
**************************************************************/
int  getsockname(SOCKET s, struct sockaddr *name, int *namelen)
{
  struct _socket      *Soc = (struct _socket *)s;
  SO_ADDRESS          *ThisAddress = &(Soc->So_LocAdd);
  struct sockaddr_in  *InAdPnt = (struct sockaddr_in *)name;
  
  if (*namelen < sizeof(struct sockaddr_in)) {
    return ILLADDSIZE;
    }
  *namelen = sizeof(struct sockaddr_in);

  InAdPnt->sin_port = (Soc->So_State & SS_LOCPORT) ?
                      htons(ThisAddress->Port) : 0;
  InAdPnt->sin_addr.s_addr = (Soc->So_State & SS_LOCALBOUND) ?
                             ThisAddress->IPAddress : 0l;

  return NOERR;
}

/**************************************************************
** NAME:        shutdown                                  [API]
** SYNOPSIS:    int shutdown(SOCKET s, int howto);
**  
** DESCRIPTION: Disconnects and debinds socket 's'.
**              Removes all packets waiting on this socket.
**              The parameter 'howto' is implemented only
**              for compatability reasons. The actual action
**              performed matches 'howto' == 2.
** RETURNS:     NOERR       -->  no error
**************************************************************/
int shutdown(SOCKET s, int howto)
{
  (*(((struct _socket *)s)->So_Prot->UserReq))(PRU_SHUTDOWN, s, NULL, NULL, NULL);
  return NOERR;
}

/**************************************************************
** NAME:        inet_addr                                 [API]
** SYNOPSIS:    unsigned long inet_addr(char *cp);
**        
** DESCRIPTION: Converts a dotted decimal IP Addressto an
**              unsigned long. The address returned is in
**              network order.
** RETURNS:     INADDR_NONE  -->   conversion error
**              else         -->   converted address
**************************************************************/
unsigned long inet_addr(char *cp)
{
  int           j, Tmp[4];
  unsigned long AddStore = INADDR_NONE;

  j = sscanf(cp, "%i.%i.%i.%i", Tmp, Tmp + 1, Tmp + 2, Tmp + 3);
  if (j < 4) 
    return INADDR_NONE;
  for (j = 0; j < 4; j++) 
    if ((*((BYTE *)(&AddStore) + j) = (BYTE)Tmp[j]) > 255) 
      return INADDR_NONE;
  return AddStore;
}

/**************************************************************
** NAME:        inet_ntoa                                 [API]
** SYNOPSIS:    char *inet_ntoa(unsigned long cp);
**
** DESCRIPTION: Prints cp in a local character buffer.
**              The default dotted notation is used.
** RETURNS:     A pointer to the local character buffer. 
**************************************************************/
char *inet_ntoa(unsigned long cp)
{
  static char  Buffer[16];
  int          j, NrDig = 0;

  for (j = 0; j < 4; j++) {
    NrDig += sprintf(Buffer + NrDig, "%d%s", ((BYTE *)&cp)[j], (j == 3) ? "" : ".");
    }
  Buffer[NrDig] = '\0';
  return Buffer;
}

