/*
 * MultiMail offline mail reader
 * ANSI image/text viewer

 Copyright (c) 1998 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "interfac.h"
#include "mysystem.h"
#include "isoconv.h"

#ifdef SIGWINCH
extern Interface	*interface;
#endif

AnsiLine::AnsiLine(int space, AnsiLine *parent)
{
	next = NULL;
	prev = parent;
	if (space) {
		text = new chtype[space];
		for (int i = 0; i < (space - 1); i++)
			text[i] = ' ' | C_SBACK;
		text[space - 1] = 0;
	} else
		text = NULL;
	length = 0;
}

AnsiLine::~AnsiLine()
{
	delete text;
}

AnsiLine *AnsiLine::getnext(int space)
{
	if (!next)
		if (space)
			next = new AnsiLine(space, this);
	return next;
}

AnsiLine *AnsiLine::getprev()
{
	return prev;
}

void AnsiWindow::Clear(WINDOW *text)
{
	wbkgdset(text, ' ' | C_SBACK);
	werase(text);
	wbkgdset(text, ' ');
	wattrset(text, 0);
}

void AnsiWindow::Scroll(WINDOW *text, int i)
{
	scrollok(text, TRUE);
	wscrl(text, i);
	scrollok(text, FALSE);
}

void AnsiWindow::DestroyChain()
{
	while (NumOfLines)
		delete linelist[--NumOfLines];
	delete linelist;
}

int AnsiWindow::getparm()
{
	char *parm;
	int value;

	if (escparm[0]) { 
		for (parm = escparm; (*parm != ';') && *parm; parm++);
		if (*parm == ';') {
			*parm++ = '\0';
			value = atoi(escparm);
			strcpy(escparm, parm);
		} else {
			value = atoi(escparm);
			escparm[0] = '\0';
		}
	} else
		value = 1;

	return value;
}

void AnsiWindow::colreset()
{
	cfl = crv = cbr = 0;
	ccf = COLOR_WHITE;
	ccb = COLOR_BLACK;
}

void AnsiWindow::colorset()
{
	static const int colortable[8] = {COLOR_BLACK, COLOR_RED,
		COLOR_GREEN, COLOR_YELLOW, COLOR_BLUE, COLOR_MAGENTA,
		COLOR_CYAN, COLOR_WHITE};
	int tmp;

	while(escparm[0]) {
		tmp = getparm();
		switch (tmp) {
		case 0:
			colreset();
			break;
		case 1:
			cbr = 1;
			break;
		case 5:
			cfl = 1;
			break;
		case 7:
			crv = 1;
			break;
		default:
			if ((tmp > 29) && (tmp < 38))
				ccf = colortable[tmp - 30];
			else if ((tmp > 39) && (tmp < 48))
				ccb = colortable[tmp - 40];
		}
	}

	if (!anim || ((ccb | ccf) || !(oldcolorx | oldcolory)))
#ifdef __PDCURSES__
		attrib = (crv ? COL(ccb, ccf) : COL(ccf, ccb));
	else
		attrib = (crv ? COL(oldcolory, oldcolorx) :
			COL(oldcolorx, oldcolory));
#else
		attrib = COL(ccf, ccb) | (crv ? A_REVERSE : 0);
	else
		attrib = COL(oldcolorx, oldcolory) | (crv ? A_REVERSE : 0);
#endif
	attrib |= (cbr ? A_BOLD : 0) | (cfl ? A_BLINK : 0);

	if (!anim)
#ifdef __PDCURSES__
		if (crv)
			colorsused[(ccb << 3) + ccf] = 1;
		else
#endif
			colorsused[(ccf << 3) + ccb] = 1;
}

unsigned char *AnsiWindow::escfig(unsigned char *escode)
{
	char a[2];

	a[0] = *escode;
	a[1] = '\0';

	switch (a[0]) {
	case 'A':
		cpy -= getparm();
		if (cpy < 0)
			cpy = 0;
		break;
	case 'B':
		cpy += getparm();
		if (anim && (cpy > (LINES - 2)))
			cpy = LINES - 2;
		break;
	case 'C':
		cpx += getparm();
		if (cpx > (COLS - 1))
			cpx = COLS - 1;
		break;
	case 'D':
		cpx -= getparm();
		if (cpx < 0)
			cpx = 0;
		break;
	case 'J':
		if (getparm() == 2) {
			if (anim) {
				Clear(animtext);
				posreset();
			} else {
				AnsiLine *tmp;
				curr = head->getnext();
				while (curr) {
					tmp = curr;
					curr = curr->getnext();
					delete tmp;
				}
				delete head;
				ResetChain();
			}
		}
		break;
	case 'H':
	case 'f':
		cpy = getparm() - 1;
		cpx = getparm() - 1;
		break;
	case 's':
		spx = cpx;
		spy = cpy;
		break;
	case 'u':
		cpx = spx;
		cpy = spy;
		break;
	case 'm':
		colorset();
		break;
	default:
		if (((a[0] > 47) && (a[0] < 60)) && (a[0] != 58)) {
			strcat(escparm, a);
			return escfig(++escode);
		}
	}
	return escode;
}

void AnsiWindow::posreset()
{
	cpx = cpy = lpy = spx = spy = 0;
}

void AnsiWindow::checkpos()
{
	if (cpy > lpy) {
		for (; lpy != cpy; lpy++)
			curr = curr->getnext(COLS + 1);
	} else
		if (cpy < lpy) {
			for (; lpy != cpy; lpy--)
				curr = curr->getprev();
		}
	if ((cpy + 1) > NumOfLines)
		NumOfLines = cpy + 1;
}

void AnsiWindow::update(unsigned char c)
{
	chtype ouch = attrib | (isoconvert ? (unsigned char)
			dos2isotab[c] : c);
	int limit = LINES - 2;

	if (anim) {
		if (cpy > limit) {
			Scroll(animtext, cpy - limit);
			cpy = limit;
		}
		mvwaddch(animtext, cpy, cpx++, ouch);
		wrefresh(animtext);
		ansiAbort |= (wgetch(animtext) != ERR);
	} else {
		checkpos();
		curr->text[cpx++] = ouch;
		if (cpx > (int) curr->length)
			curr->length = cpx;
	}
	if (cpx == COLS) {
		cpx = 0;
		cpy++;
	}
}

void AnsiWindow::ResetChain()
{
	head = new AnsiLine();
	curr = head;
	posreset();
	curr = curr->getnext(COLS + 1);
	NumOfLines = 1;
}

void AnsiWindow::MakeChain(unsigned char *message)
{
	ansiAbort = 0;
	if (!anim)
		ResetChain();
	attrib = C_SBACK;
	colreset();

	while (*message && !ansiAbort) {
		switch (*message) {
		case 10:
			cpy++;
		case 13:
			cpx = 0;
		case 7:
			break;
		case '`':
		case 27:
			message++;
			if (*message == '[') {
				escparm[0] = '\0';
				message = escfig(++message);
			} else {
				message--;
				update('`');
			}
			break;
		default:
			update(*message);
		}
		message++;
	}

	if (!anim) {
		linelist = new AnsiLine *[NumOfLines];
		curr = head->getnext();
		int i = 0;
		while (curr) {
			linelist[i++] = curr;
			curr = curr->getnext();
		}
		delete head;
	}
}

void AnsiWindow::statupdate()
{
	char format[80];
	sprintf(format, " ANSI View: %%-%ds", COLS - 12);

	mvwprintw(statbar, 0, 0, format, title);
}

void AnsiWindow::animate()
{
	int c;
	char format[80];

	animtext = newwin(LINES -1, COLS, 0, 0);
	keypad(animtext, TRUE);
	nodelay(animtext, TRUE);
	Clear(animtext);

	posreset();
	colreset();
	anim = 1;

	sprintf(format, " Animating: %%-%ds", COLS - 12);
	mvwprintw(statbar, 0, 0, format, title);
	wrefresh(statbar);

	MakeChain(source);

	sprintf(format, "      Done: %%-%ds", COLS - 12);
	mvwprintw(statbar, 0, 0, format, title);
	wrefresh(statbar);

	if (!ansiAbort)
		do
			c = wgetch(animtext);
		while (!ansiAbort && (c == ERR));
	nodelay(animtext, FALSE);
 
	anim = 0;
	delwin(animtext);
	statupdate();
	touchwin(header);
	touchwin(text);
	wnoutrefresh(text);
	DrawHeader();
}

void AnsiWindow::Draw()
{
	DrawHeader();
	DrawBody();
}

void AnsiWindow::DrawHeader()
{
	wnoutrefresh(header);
	wnoutrefresh(statbar);
}

void AnsiWindow::oneLine(int i)
{
#ifdef __PDCURSES__
	mvwaddchnstr(text, i, 0, linelist[position + i]->text, COLS);
#else
	mvwaddchstr(text, i, 0, linelist[position + i]->text);
#endif
}

void AnsiWindow::DrawBody()
{
	int i, j;

	for (i = 0; i < y; i++)
		if ((position + i) < NumOfLines)
			oneLine(i);
		else
			for (j = 0; j < x; j++)
				mvwaddch(text, i, j, ' ' | C_SBACK);

	wnoutrefresh(text);
}

void AnsiWindow::ReDraw()
{
	header = newwin(1, COLS, 0, 0);
	text = newwin(LINES - 2, COLS, 1, 0);
	statbar = newwin(1, COLS, LINES - 1, 0);
	leaveok(header, TRUE);
	leaveok(text, TRUE);
	keypad(text, TRUE);

	Clear(text);

	wattrset(header, C_LBOTTSTAT);

	wprintw(header, " " MM_TOPHEADER, MM_NAME, MM_MAJOR, MM_MINOR);
	for (int i = 0; i < COLS-29; i++)
		waddch(header, ' ');

	getmaxyx(text, y, x);

	wattrset(statbar, C_LBOTTSTAT);
	statupdate();

	position = 0;
	Draw();
	doupdate();
}

AnsiWindow::AnsiWindow(unsigned char *ansiSource, const char *winTitle)
{
	unsigned i, end;
	int j;

	for (i = 0; i < 64; i++)
		colorsused[i] = 0;

	colorsused[PAIR_NUMBER(C_LBOTTSTAT)] = 1;  //don't remap stat bars

	oldcolorx = oldcolory = 0;

	init_pair(((COLOR_WHITE << 3) + COLOR_WHITE),
		COLOR_WHITE, COLOR_WHITE);

	source = ansiSource;
	anim = 0;
	MakeChain(source);

	// This deals with the unavailability of color pair 0:

	if (colorsused[0]) {	// assumes COLOR_BLACK == 0
		for (end = 0, i = 1; i < 64 && !end; i++)
			if (!colorsused[i]) {
				end = 1;
				oldcolorx = i >> 3;
				oldcolory = i & 7;
				init_pair(i, COLOR_BLACK, COLOR_BLACK);
			}
		if (end) {
			for (j = 0; j < NumOfLines; j++) {
				curr = linelist[j];
				for (i = 0; i < curr->length; i++)
					if (!PAIR_NUMBER(curr->text[i])) {
						curr->text[i] &= ~A_COLOR;
						curr->text[i] |=
						  COL(oldcolorx, oldcolory);
					}
			}
		}
	}

	title = winTitle;
	ReDraw();
#ifdef SIGWINCH
	currAnsi = this;
#endif
}

void AnsiWindow::Move(direction dir)
{
	switch (dir) {
	case UP:
		if (position > 0) {
			position--;
			Scroll(text, -1);
			oneLine(0);
			wnoutrefresh(text);
		}
		break;
	case DOWN:
		if (position < NumOfLines - y) {
			position++;
			Scroll(text, 1);
			oneLine(y - 1);
			wnoutrefresh(text);
		}
		break;
	case HOME:
		position = 0;
		DrawBody();
		break;
	case END:
		if (NumOfLines > y) {
			position = NumOfLines - y;
			DrawBody();
		}
		break;
	case PGUP:
		position -= ((y < position) ? y : position);
		DrawBody();
		break;
	case PGDN:
		if (position < NumOfLines - y) {
			position += y;
			if (position > NumOfLines - y)
				position = NumOfLines - y;
			DrawBody();
		}
	}
}

AnsiWindow::~AnsiWindow()
{
	if (oldcolorx + oldcolory)
		init_pair(((oldcolorx << 3) + oldcolory),
			oldcolorx, oldcolory);
	init_pair(((COLOR_WHITE << 3) + COLOR_WHITE),
		COLOR_BLACK, COLOR_BLACK);
	DestroyChain();
	delwin(header);
	delwin(text);
#ifdef SIGWINCH
	currAnsi = NULL;
#endif
}

void AnsiWindow::KeyHandle()
{
	int c, end = 0;

	while (!end) {
		do
			c = wgetch(text);
		while (c == ERR);
		switch (c) {
		case 'q':
		case 'Q':
		case '\033':	//the escape key
		case KEY_BACKSPACE:
			end = 1;
			break;
		case KEY_DOWN:
			Move(DOWN);
			break;
		case KEY_UP:
			Move(UP);
			break;
		case KEY_HOME:
			Move(HOME);
			break;
		case KEY_END:
			Move(END);
			break;
		case 'b':
		case 'B':
		case KEY_PPAGE:
			Move(PGUP);
			break;
		case 'f':
		case 'F':
		case ' ':
		case KEY_NPAGE:
			Move(PGDN);
			break;
		case 'v':
		case 'V':
		case 'a':
		case 'A':
		case 1:
		case 22:
			animate();
		default:;
		}
		if (!end)
			doupdate();
	}
}

#ifdef SIGWINCH
void AnsiWindow::sigwinch()
{
	ansiAbort = anim;
	delwin(header);
	delwin(text);
	interface->sigwinch();
	init_pair(((COLOR_WHITE << 3) + COLOR_WHITE),
		COLOR_WHITE, COLOR_WHITE);
	if (oldcolorx + oldcolory)
		init_pair(((oldcolorx << 3) + oldcolory),
			COLOR_BLACK, COLOR_BLACK);
	ReDraw();
}
#endif
