
;==========================================================================
;
;   FILE:           forcext.asm
;
;   DESCRIPTION:    This is a resident program that will force textmode on
;                   VGA's to have 50 lines, and on EGA's (if there are any
;                   left) to have 43 lines whenever a program terminates.
;
;                   It can also be set up to monitor all calls to the BIOS
;                   `Set Textmode'-function, and force 50/43 lines when
;                   programs try to change textmode.
;
;                   The main part of the program is a general frame I've
;                   written for TSRs following the AMIS-standard. The few
;                   lines that actually make the program, are from a
;                   Norwegian (non-AMIS) version i wrote in 1990.
;
;   WRITTEN BY:     Sverre H. Huseby
;                   Bjoelsengt. 17
;                   N-0468 Oslo
;                   Norway
;
;                   Internet: sverrehu@ifi.uio.no
;
;                   !!! This is _my_ work: please don't remove my name !!!
;
;                   Feel free to modify the source for your own needs,
;                   but don't redistribute it with your changes.
;
;                   I've looked at Ralf Brown's AMIS-routines (AMISL091.ZIP)
;                   to figure out how this standard works, and parts of the
;                   uninstall-routines are borrowed with small changes.
;
;   TOOLS:          Borland Turbo Assembler (TASM) (I use v3.1)
;                   Borland Turbo Linker (TLINK) (I have v5.22b)
;
;==========================================================================

            IDEAL
            MODEL TINY
            ASSUME cs: @code, ds: @code, es: @code



PROG_NAME   EQU "FORCEXT"       ; Name of program.
PROG_VER    EQU "v3.1"          ; Textual program version.
PROG_INTVER EQU 0310h           ; Program version as an integer. 1234h=v12.34
PROG_DATE   EQU "11/5/94"       ; Textual date of program.



            CODESEG
            ORG     0100h

EntryPoint:
        jmp Main

;**************************************************************************
;*                                                                        *
;*                    R E S I D E N T    P A R T                          *
;*                                                                        *
;**************************************************************************

;==========================================================================
;
;                            V A R I A B L E S
;
;==========================================================================

MultiplexID DB  0               ; Multiplex ID used for installation check.
IDString    DB  "SHH     "      ; Multiplex installation check string.
  pnme      DB  PROG_NAME
  IF SIZE pnme NE 8
            DB  (8 - SIZE pnme) DUP (' ')
  ENDIF
            DB  0

Enabled     DB  1               ; Program functions enabled?
Always      DB  0               ; Force extended at each textmode call?

; ### List of chained interrupts. The entry of interrupt 2Dh marks the
; end of the list!
IntList     DB  10h
            DW  OFFSET Int10
            DB  20h
            DW  OFFSET Int20
            DB  21h
            DW  OFFSET Int21
            DB  2Dh
            DW  OFFSET Int2D

Norwegian   DB  0               ; Enable Norwegian characters?
;
;  And here they are, the 8x8 versions of Norwegian/Danish letters o-slash
;  and O-slash. They replace cent and yen respectively. Seems that IBM
;  forgot us when defining their character set. :-)
;
Lower_oe    DB  00000000b
            DB  00000000b
            DB  01111100b
            DB  11001110b
            DB  11010110b
            DB  11100110b
            DB  01111100b
            DB  00000000b

Upper_oe    DB  00111010b
            DB  01101100b
            DB  11001110b
            DB  11010110b
            DB  11100110b
            DB  01101100b
            DB  10111000b
            DB  00000000b





;==========================================================================
;
;                           P R O C E D U R E S
;
;==========================================================================

;--------------------------------------------------------------------------
;
;   NAME:           ExtendedTextMode
;
;   DESCRIPTION:    Switch to extended textmode (50 lines on VGA, or 43
;                   lines on EGA), possibly redefining cent and yen to
;                   Norwegian letters.
;
;   DESTROYS:       Nothing
;
;
PROC    ExtendedTextMode

        push    ax
        push    bx
        push    cx
        push    dx
        push    si
        push    di
        push    bp
        push    es

    ; Change to 8x8 character set.
        mov     ax, 1112h
        xor     bl, bl
        int     10h

    ; Should we redefine Norwegian letters?
        cmp     [cs: Norwegian], 0
        je      @@skip_norwegian

    ; Yep! Do the lowercase oe...
        mov     bx, 0800h
        mov     cx, 1
        mov     dx, 155         ; ''
        push    cs
        pop     es
        mov     bp, OFFSET Lower_oe
        mov     ax, 1110h
        int     10h

    ; ... and the uppercase one.
        mov     dx, 157         ; ''
        mov     bp, OFFSET Upper_oe
        mov     ax, 1110h
        int     10h

  @@skip_norwegian:
    ; Set correct cursor type. This depends on wethere we have a VGA
    ; or an EGA.
        xor     ax, ax
        mov     es, ax
        cmp     [BYTE es: 0484h], 42        ; 42 + 1
        jne     @@not_ega_43

    ; We have EGA and 43 lines. Set the cursor.
        mov     bl, [es: 0487h]
        or      [BYTE es: 0487h], 1
        mov     ah, 1
        mov     cx, 0600h
        int     10h
        mov     [es: 0487h], bl

  @@not_ega_43:
    ; VGA handles the cursor itself. Just fix the underscore-line.
        mov     dx, 03D4h
        mov     ax, 0714h
        out     dx, ax

        pop     es
        pop     bp
        pop     di
        pop     si
        pop     dx
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    ExtendedTextMode



;--------------------------------------------------------------------------
;
;   NAME:           ExtendIfNot
;
;   DESCRIPTION:    Switch to extended textmode (50 lines on VGA, or 43
;                   lines on EGA), but only if it isn't already extended.
;
;   DESTROYS:       Nothing
;
;
PROC    ExtendIfNot

        push    ax
        push    bx
        push    es

    ; Only do this in textmode, to avoid screwing up Windows.
        mov     ah, 0Fh         ; Get Video Mode
        int     10h
    ; Is it textmode? (Except monochrome)
        cmp     al, 3
        ja      @@ret

    ; It's textmode. Check if already in 50/43-line mode.
        xor     ax, ax
        mov     es, ax
        mov     al, [es: 0484h] ; Rows on screen minus one.
        cmp     al, 43 - 1
        je      @@ret
        cmp     al, 50 - 1
        je      @@ret

    ; Do it!
        call    ExtendedTextMode

  @@ret:
        pop     es
        pop     bx
        pop     ax

        ret

ENDP    ExtendIfNot



;--------------------------------------------------------------------------
;
;   NAME:           Int10
;
;   DESCRIPTION:
;
;   IN/OUT:
;
;
PROC    Int10 FAR

    ; Start with the header specified by IBM, to make it possible to
    ; uninstall the program even if other programs has hooked the
    ; vectors afterwards.
    ; Any other interrupt-function should follow this scheme.
            DB  0EBh, 10h       ; Short jmp to skip the header.
OldInt10    DD  0               ; Address of old multiplex interrupt.
            DW  424Bh           ; ISP signature.
EOIflag10   DB  0               ; Software or secondary hardware int.
        jmp     SHORT @@iret    ; Short jmp to hardware reset routine.
            DB  7 DUP (0)       ; Reserved.

    ; Check if Set Video Mode.
        or      ah, ah
        jz      @@set_vid_mode

  @@jmp_old:
    ; Chain to the original interrupt vector.
        jmp     [DWORD FAR cs: OldInt10]

  @@set_vid_mode:
    ; Check if our routine is currently enabled.
        cmp     [cs: Enabled], 0
        je      @@jmp_old

    ; Check if we should intercept, or just doit at program exit.
        cmp     [cs: Always], 0
        je      @@jmp_old

        sti

    ; Does the user want textmode? (Except monochrome)
        cmp     al, 3
        ja      @@jmp_old

    ; First do the mode-change.
        pushf
        call    [DWORD FAR cs: OldInt10]

    ; Set Extended Textmode (50/43 lines).
        call    ExtendedTextMode

  @@iret:
        iret

ENDP    Int10



;--------------------------------------------------------------------------
;
;   NAME:           Int20
;
;   DESCRIPTION:
;
;   IN/OUT:
;
;
PROC    Int20 FAR

    ; Start with the header specified by IBM, to make it possible to
    ; uninstall the program even if other programs has hooked the
    ; vectors afterwards.
    ; Any other interrupt-function should follow this scheme.
            DB  0EBh, 10h       ; Short jmp to skip the header.
OldInt20    DD  0               ; Address of old multiplex interrupt.
            DW  424Bh           ; ISP signature.
EOIflag20   DB  0               ; Software or secondary hardware int.
        jmp     SHORT @@iret    ; Short jmp to hardware reset routine.
            DB  7 DUP (0)       ; Reserved.

  @@terminate:
    ; Check if our routine is currently enabled.
        cmp     [cs: Enabled], 0
        je      @@jmp_old

        sti

    ; Possibly set Extended Textmode (50/43 lines).
        call    ExtendIfNot

  @@jmp_old:
    ; Chain to the original interrupt vector.
        jmp     [DWORD FAR cs: OldInt20]

  @@iret:
        iret


ENDP    Int20



;--------------------------------------------------------------------------
;
;   NAME:           Int21
;
;   DESCRIPTION:
;
;   IN/OUT:
;
;
PROC    Int21 FAR

    ; Start with the header specified by IBM, to make it possible to
    ; uninstall the program even if other programs has hooked the
    ; vectors afterwards.
    ; Any other interrupt-function should follow this scheme.
            DB  0EBh, 10h       ; Short jmp to skip the header.
OldInt21    DD  0               ; Address of old multiplex interrupt.
            DW  424Bh           ; ISP signature.
EOIflag21   DB  0               ; Software or secondary hardware int.
        jmp     SHORT @@iret    ; Short jmp to hardware reset routine.
            DB  7 DUP (0)       ; Reserved.

    ; Check if Terminate Process With Return Code. (The preferred one).
        cmp     ah, 4Ch
        je      @@terminate

    ; Check if Terminate Process.
        or      ah, ah
        jz      @@terminate

  @@jmp_old:
    ; Chain to the original interrupt vector.
        jmp     [DWORD FAR cs: OldInt21]

  @@terminate:
    ; Check if our routine is currently enabled.
        cmp     [cs: Enabled], 0
        je      @@jmp_old

        sti

    ; Possibly set Extended Textmode (50/43 lines).
        call    ExtendIfNot

    ; Do the real program terminate.
        jmp     @@jmp_old

  @@iret:
        iret

ENDP    Int21



;--------------------------------------------------------------------------
;
;   NAME:           Int2D
;
;   DESCRIPTION:    Multiplex interrup used to state that this program
;                   is installed.
;
;   IN/OUT:         See AMIS, the Alternate Multiplex Interrupt Specification.
;                   Can be found in Ralf Brown's interrupt list.
;
;
PROC    Int2D FAR

    ; Start with the header specified by IBM, to make it possible to
    ; uninstall the program even if other programs has hooked the
    ; vectors afterwards.
    ; Any other interrupt-function should follow this scheme.
            DB  0EBh, 10h       ; Short jmp to skip the header.
OldInt2D    DD  0               ; Address of old multiplex interrupt.
            DW  424Bh           ; ISP signature.
EOIflag2D   DB  0               ; Software or secondary hardware int.
        jmp     SHORT @@iret    ; Short jmp to hardware reset routine.
            DB  7 DUP (0)       ; Reserved.

    ; Check if this is a call to our multiplex number.
        cmp     ah, [cs: MultiplexID]
        je      @@our_id

    ; Chain to the original interrupt vector.
        jmp     [DWORD FAR cs: OldInt2D]

  @@our_id:
    ; Someone is calling for us. Check what function is wanted,
    ; and act according to it.
        sti
        or      al, al          ; Installation check?
        je      @@installation_check
        cmp     al, 2           ; Uninstall?
        je      @@uninstall_request
        cmp     al, 4           ; Determine chained interrupts?
        je      @@chained_int_request

    ; A function not supported by us. Return "not implemented".
        xor     al, al
  @@iret:
        iret

  @@installation_check:
    ; Return what is required for an installation check.
        mov     cx, PROG_INTVER ; Version number.
        mov     dx, cs          ; DX:DI points to ID-string.
        mov     di, OFFSET IDString
  @@return_alFF:
        mov     al, 0FFh        ; Multiplex number in use.
        iret

  @@uninstall_request:
    ; Return what is required for an uninstall request. We don't uninstall,
    ; just return a code stating that it is safe to do so.
        inc     al              ; Increased from 2 to 3; Safe to remove.
        mov     bx, cs          ; Segment of memory block with resident code.
        iret

  @@chained_int_request:
    ; Return list of hooked interrupts. AL is kept at 4.
        mov     dx, cs          ; DX:BX points to list.
        mov     bx, OFFSET IntList
        iret

ENDP    Int2D





;**************************************************************************
;*                                                                        *
;*                   N O N R E S I D E N T    P A R T                     *
;*                                                                        *
;**************************************************************************

            LABEL NonResidentPart BYTE

;==========================================================================
;
;                            V A R I A B L E S
;
;==========================================================================

TsrSeg      DW  ?               ; Segment of resident instance of program.
ExitFunc    DB  4Ch             ; Function used to terminate the program.
DoInstall   DB  0               ; Make program go resident?
ResPara     DW  0               ; Number of paragraphs to keep resident.

ArgPtr      DW  ?               ; Pointer running through command line.
ArgCurrOpt  DW  ?               ; Pointer to start of command line option
                                ;   currently being interpreted.

DoQuiet     DB  0               ; Suppress status messages? (-q)

TmpIntStr   DB  6 DUP(0)        ; For printing integers.

FatalTxt1   DB  13, 10, PROG_NAME, ": ", 0
FatalTxt2   DB  13, 10, 0

MsgHeader   DB  PROG_NAME, " ", PROG_VER, "  --  (C) ", PROG_DATE, " - "
            DB  "Sverre H. Huseby, Norway.    Add -? for help.", 13, 10, 0

MsgUsage    DB  10
            DB  "This program will force extended textmode (50 lines on VGA "
            DB  "or 43 lines ", 13, 10
            DB  "on EGA) when other programs terminate. It can also be set to "
            DB  "force"
            DB  13, 10
            DB  "extended mode when programs change textmode through the BIOS."
            DB  13, 10
            DB  10
            DB  "Usage: ", PROG_NAME, " [options]", 13, 10
            DB  10
            DB  "       Switch options:", 13, 10
            DB  "         -a[-]  Enable [disable] always forcing extended mode."
            DB                   13, 10
            DB  "         -n[-]  Enable [disable] Norwegian letters oe/OE."
            DB                   13, 10
            DB  "         -off   Temporarily disable program.", 13, 10
            DB  "         -on    Reenable program.", 13, 10
            DB  "         -q     Quiet. Don't show status messages.", 13, 10
            DB  "         -u     Uninstall if possible.", 13, 10
            DB  10
            DB  0

MsgInst     DB  "Now installed in memory.", 13, 10, 0
MsgUninst   DB  "Now removed from memory.", 13, 10, 0

SttCrLf     DB  13, 10, 0
SttMargin   DB  "    ", 0
SttSpace    DB  " ", 0
SttOff      DB  "off", 13, 10, 0
SttOn       DB  "on", 13, 10, 0
SttEnabled  DB  "enabled", 13, 10, 0
SttDisabled DB  "disabled", 13, 10, 0
SttNorw     DB  "Norwegian letters is", 0
SttAlways   DB  "Force 50 (or 43) lines always is", 0
SttStatus   DB  "The program is", 0

ErrVersion  DB  "Another version of the program is already resident.", 13, 10, 0
ErrEgaVga   DB  "This program requires EGA or VGA.", 0
ErrArgument DB  "Unknown argument: ", 0
ErrUninst1  DB  "Nothing to uninstall.", 0
ErrUninst2  DB  "Unable to uninstall.", 0
ErrNumExp   DB  "Missing number.", 13, 10, 0
ErrTooBigN  DB  "Number out of range.", 13, 10, 0


; ### Options. If one option is a substring of another, the longer
; must be located earlier in the list.

Options     DB  "?", 0
            DW  OFFSET OptHelp

            DB  "h", 0
            DW  OFFSET OptHelp

            DB  "a-", 0
            DW  OFFSET OptAlwaysOff

            DB  "a", 0
            DW  OFFSET OptAlwaysOn

            DB  "n-", 0
            DW  OFFSET OptNorwOff

            DB  "n", 0
            DW  OFFSET OptNorwOn

            DB  "off", 0
            DW  OFFSET OptTurnOff

            DB  "on", 0
            DW  OFFSET OptTurnOn

            DB  "q", 0
            DW  OFFSET OptQuiet

            DB  "u", 0
            DW  OFFSET OptUninstall

            DB  0               ; End of options list.





;==========================================================================
;
;                           P R O C E D U R E S
;
;==========================================================================

;--------------------------------------------------------------------------
;
;   NAME:           StrLen
;
;   DESCRIPTION:    Get length of string.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            AX      Length of string, not including the 0.
;
;   DESTROYS:       None.
;
;
PROC    StrLen

        push    cx
        push    dx
        push    di
        push    es

        mov     cx, ds
        mov     es, cx
        mov     di, dx
        xor     al, al          ; Look for the 0-byte.
        mov     cx, 0FFFFh

        cld
        repne   scasb

        mov     ax, di
        dec     ax
        sub     ax, dx

  @@ret:
        pop     es
        pop     di
        pop     dx
        pop     cx

        ret

ENDP    StrLen



;--------------------------------------------------------------------------
;
;   NAME:           IsSpace
;
;   DESCRIPTION:    Check if a character is space, tab or carriage return.
;
;   IN:             AL   Character to test.
;
;   OUT:            Z-flag: 1 if it is, 0 if not.
;
;   DESTROYS:       Nothing
;
;
PROC    IsSpace

        cmp     al, ' '
        je      @@ret
        cmp     al, 9
        je      @@ret
        cmp     al, 13

  @@ret:
        ret

ENDP    IsSpace



;--------------------------------------------------------------------------
;
;   NAME:           Exit
;
;   DESCRIPTION:    Exit the program.
;
;   IN:             AX   Exit code to return to DOS.
;
;   OUT:            Nothing. Never returns.
;
;   DESTROYS:       N/A
;
;
PROC    Exit

        mov     ah, 4Ch         ; Terminate with Exit Code.
        int     21h

ENDP    Exit



;--------------------------------------------------------------------------
;
;   NAME:           Print
;
;   DESCRIPTION:    Outputs a string to stdout.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    Print

        push    ax
        push    bx
        push    cx

        call    StrLen
        mov     cx, ax
        mov     ah, 40h         ; Write File or Device
        mov     bx, 1           ; stdout
        int     21h

  @@ret:
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    Print



;--------------------------------------------------------------------------
;
;   NAME:           PrintWord
;
;   DESCRIPTION:    Outputs an unsigned integer to stdout.
;
;   IN:             AX   Number to output.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    PrintWord

        push    ax
        push    cx
        push    dx
        push    di

        mov     di, OFFSET TmpIntStr + 5

  @@more_left:
        dec     di
        xor     dx, dx
        mov     cx, 10
        div     cx
        add     dl, '0'
        mov     [di], dl
        or      ax, ax
        jnz     @@more_left

        mov     dx, di
        call    Print

  @@ret:
        pop     di
        pop     dx
        pop     cx
        pop     ax

        ret

ENDP    PrintWord



;--------------------------------------------------------------------------
;
;   NAME:           PrintErr
;
;   DESCRIPTION:    Outputs a string to stderr.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    PrintErr

        push    ax
        push    bx
        push    cx

        call    StrLen
        mov     cx, ax
        mov     ah, 40h         ; Write File or Device
        mov     bx, 2           ; stderr
        int     21h

  @@ret:
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    PrintErr



;--------------------------------------------------------------------------
;
;   NAME:           Fatal
;
;   DESCRIPTION:    Outputs a string to stderr, then aborts the program.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            Nothing. Never returns.
;
;   DESTROYS:       N/A
;
PROC    Fatal

    ; Encapsulate the string.
        push    dx
        mov     dx, OFFSET FatalTxt1
        call    PrintErr
        pop     dx

        call    PrintErr

        mov     dx, OFFSET FatalTxt2
        call    PrintErr

        mov     ax, -1
        call    Exit

    ; We never get here . . .

ENDP    Fatal



;--------------------------------------------------------------------------
;
;   NAME:           Fatal2
;
;   DESCRIPTION:    Outputs two strings to stderr, then aborts the program.
;
;   IN:             DS:DX   Pointer to first zero terminated string.
;                   DS:BX   Pointer to second string.
;
;   OUT:            Nothing. Never returns.
;
;   DESTROYS:       N/A
;
PROC    Fatal2

    ; Encapsulate the string.
        push    dx
        mov     dx, OFFSET FatalTxt1
        call    PrintErr
        pop     dx

        call    PrintErr
        mov     dx, bx
        call    PrintErr

        mov     dx, OFFSET FatalTxt2
        call    PrintErr

        mov     ax, -1
        call    Exit

    ; We never get here . . .

ENDP    Fatal2



;--------------------------------------------------------------------------
;
;   NAME:           CondPrint
;
;   DESCRIPTION:    Outputs a string to stdout if not DoQuiet.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    CondPrint

        cmp     [DoQuiet], 1
        je      @@ret

        call    Print

  @@ret:
        ret

ENDP    CondPrint



;--------------------------------------------------------------------------
;
;   NAME:           CondPrintWord
;
;   DESCRIPTION:    Outputs an unsigned integer to stdout if not DoQuiet.
;
;   IN:             AX   Integer to print.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    CondPrintWord

        cmp     [DoQuiet], 1
        je      @@ret

        call    PrintWord

  @@ret:
        ret

ENDP    CondPrintWord



;--------------------------------------------------------------------------
;
;   NAME:           ICondPrint
;
;   DESCRIPTION:    Outputs some spaces followed by a string to stdout
;                   if not DoQuiet. I is for Indent.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    ICondPrint

        push    dx
        mov     dx, OFFSET SttMargin
        call    CondPrint
        pop     dx

        call    CondPrint

  @@ret:
        ret

ENDP    ICondPrint



;--------------------------------------------------------------------------
;
;   NAME:           StatOnOff
;
;   DESCRIPTION:    Outputs some spaces, a string, "on" or "off" and CrLf
;                   to stdout if not DoQuiet.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;                   AL      0-off, 1-on.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    StatOnOff

        call    ICondPrint

        push    dx
        mov     dx, OFFSET SttSpace
        call    CondPrint
        mov     dx, OFFSET SttOn
        or      al, al
        jne     @@print_it
        mov     dx, OFFSET SttOff
  @@print_it:
        call    CondPrint
        pop     dx

  @@ret:
        ret

ENDP    StatOnOff



;--------------------------------------------------------------------------
;
;   NAME:           StatEnblDsbl
;
;   DESCRIPTION:    Outputs some spaces, a string, "enabled" or "disabled"
;                   and CrLf to stdout if not DoQuiet.
;
;   IN:             DS:DX   Pointer to zero terminated string.
;                   AL      0-disabled, 1-enabled.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    StatEnblDsbl

        call    ICondPrint

        push    dx
        mov     dx, OFFSET SttSpace
        call    CondPrint
        mov     dx, OFFSET SttEnabled
        or      al, al
        jne     @@print_it
        mov     dx, OFFSET SttDisabled
  @@print_it:
        call    CondPrint
        pop     dx

  @@ret:
        ret

ENDP    StatEnblDsbl



;--------------------------------------------------------------------------
;
;   NAME:           UnhookInterrupts
;
;   DESCRIPTION:    Try to unhook interrupt vectors used by a resident
;                   instance of this program. Used as a part of Uninstall.
;
;                   If unable to unhook, an error message is displayed,
;                   and the program is aborted.
;
;                   Note that most of this procedure runs with DS != CS.
;
;   IN:             DX:BX   Pointer to hooked interrupts table.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
PROC    UnhookInterrupts

        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    ds
        push    es

    ; Search through all the hooked interrupts to check if all
    ; can be unhooked.
        mov     ds, dx
        mov     si, bx
        push    si              ; Will be recalled later.

        cld
  @@check_next:
        lodsb                   ; Interrupt number in AL.
        mov     dx, [si]        ; Offset of int-handler.
        inc     si
        inc     si
        cmp     al, 2Dh         ; Last entry in the chain?
        je      @@all_unhookable

    ; To determine if a vector can be unhooked, we must have one of the
    ; following:
    ;   1. Our interrupt is the last one in the chain.
    ;   2. All `later' program conform to the ISP-standard.

        mov     ah, 35h         ; Get Interrupt Vector
        int     21h

        mov     ax, es
        mov     cx, ds
        cmp     ax, cx          ; Same segment as our vector?
        jne     @@check_next_isp

        cmp     dx, bx          ; Does offsets match too?
        je      @@check_next    ; Yes, this can be unhooked according to 1.

  @@check_next_isp:
    ; Our interupt vector is not the last one in the chain. Now check that
    ; all vectors hooked after our are `nice' ISP-interrupts.
        cmp     [WORD es: bx], 10EBh    ; Is the JMP SHORT $+12 there?
        jne     @@not_unhookable
        cmp     [WORD es: bx + 6], 424Bh; Valid signature?
        jne     @@not_unhookable
        cmp     [BYTE es: bx + 9], 0EBh ; Hardware reset JMP SHORT ?
        jne     @@not_unhookable
        cmp     cx, [WORD es: bx + 4]   ; Next segment ours?
        jne     @@jmp_check_next_isp
        cmp     dx, [WORD es: bx + 2]   ; Next offset ours?
        je      @@check_next            ; Can be unhooked according to 2.

  @@jmp_check_next_isp:
        les     bx, [es: bx + 2]; Advance to next ISP header.
        jmp     @@check_next_isp

  @@not_unhookable:
    ; For some reason we are unable to unhook the vectors. Display
    ; an error message and abort the program.
        pop     es
        pop     ds
        mov     dx, OFFSET ErrUninst2
        call    Fatal

    ; We never get here . . .

  @@all_unhookable:
        pop     si              ; Get back start of hook list.

  @@unhook_next:
    ; We loop through the chain to find our vector.
        lodsb                   ; Interrupt number in AL.
        mov     dx, [si]        ; Offset of int-handler.
        inc     si
        inc     si

        push    ds
        push    ax
        mov     ah, 35h         ; Get Interrupt Vector.
        int     21h

        mov     ax, es
        mov     cx, ds
        cmp     ax, cx          ; Same segment as our vector?
        jne     @@search_succ_main

        cmp     dx, bx          ; Does offsets match too?
        jne     @@search_succ_main

    ; Our vector is found to be the last one in the chain. We must thus
    ; reset the interrupt-vector to what it was before our program.
        lds     dx, [bx + 2]    ; Get our pointer to old routine.
        pop     ax
        push    ax
        mov     ah, 25h         ; Set Interrupt Vector.
        int     21h

        jmp     SHORT @@jmp_unhook_next

  @@search_succ_next:
        les     bx, [es: bx + 2]; Advance to next ISP header.

  @@search_succ_main:
    ; Our interrupt is not the last one in the chain. Find it's
    ; successor.
        cmp     cx, [es: bx + 4]; Segment of 'previous' ours?
        jne     @@search_succ_next
        cmp     dx, [es: bx + 2]; Offset of 'previous' ours?
        jne     @@search_succ_next

    ; We found the procedure that was installed after ours. Now
    ; unlink ours from the chain by entering our old vector in
    ; our successors old vector.
        xchg    bx, dx
        lds     bx, [bx + 2]
        xchg    bx, dx          ; ES:BX -> previous ISP.
                                ; DS:DX -> next ISP.
        mov     [es: bx + 2], dx; prev->next = curr->next
        mov     [es: bx + 4], ds

  @@jmp_unhook_next:
        pop     ax
        pop     ds
        cmp     al, 2Dh         ; Last entry in the chain?
        jne     @@unhook_next

  @@ret:
        pop     es
        pop     ds
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    UnhookInterrupts



;--------------------------------------------------------------------------
;
;   NAME:           ReleaseEnvironment
;
;   DESCRIPTION:    Free the memory block used by the environment
;                   variables.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
PROC    ReleaseEnvironment

        push    ax
        push    es

        mov     ax, [002Ch]     ; Segment address of environment
        or      ax, ax
        jz      @@ret

        mov     es, ax
        mov     ah, 49h         ; Release Memory Block
        int     21h

        mov     [WORD 002Ch], 0

  @@ret:
        pop     es
        pop     ax

        ret

ENDP    ReleaseEnvironment



;--------------------------------------------------------------------------
;
;   NAME:           GetTsrSegment
;
;   DESCRIPTION:    Check if the program is already installed. If it is,
;                   store the resident instance's segment. If not, store
;                   the segment of this instance. The segment is stored
;                   in TsrSeg.
;
;                   The intention is to fetch the segment where the
;                   variables to update according to command line options
;                   are.
;
;                   MultiplexID is updated to contain the multiplex number
;                   used by the resident instance of the program, or the
;                   next free number if not already installed.
;
;                   Nothing is done to install the program. A flag is set
;                   to indicate whether PossiblyInstall should install it
;                   or not.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
PROC    GetTsrSegment

        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    es

    ; Search the Alternate Multiplex Interrupt Chain for an instance
    ; of this program. The first free multiplex number is stored in
    ; MultiplexID to be used if the program is installed.
        mov     cx, 0100h
        xor     ax, ax          ; Installation check, multiplex #0.
        xor     si, si          ; Free multiplex number found?
  @@next_id:
        push    cx
        push    ax
        push    si
        int     2Dh             ; Alternate Multiplex Interrupt (AMIS).
        mov     bx, cx          ; Version number.
        pop     si
        or      al, al          ; Multiplex number free?
        pop     ax
        pop     cx
        jnz     @@check_match

    ; A free multiplex number is found. If this is the first one, store it.
        or      si, si          ; Check if we already have one.
        jnz     @@loop_next

    ; Store this number.
        mov     [MultiplexID], ah
        inc     si              ; Mark that a number is found.
        jmp     SHORT @@loop_next

  @@check_match:
    ; Check if the multiplex number just found belongs to a resident
    ; instance of this program. It does if the string returned in DX:DI
    ; matches our pattern IDString.
        push    cx
        push    si
        mov     es, dx
        mov     si, OFFSET IDString
        mov     cx, 16
        cld
        repe    cmpsb
        pop     si
        pop     cx
        je      @@already_installed

  @@loop_next:
        inc     ah              ; Next multiplex number.
        loop    @@next_id

  @@install:
    ; The program is not already installed. Set the flag so it will be
    ; installed later.
        mov     [DoInstall], 1
        mov     [TsrSeg], cs

        jmp     SHORT @@ret

  @@already_installed:
    ; An instance of this program is already running. Check if the
    ; version is compatible. If not, abort the program.
        mov     [TsrSeg], dx
        mov     [MultiplexID], ah

        cmp     bx, PROG_INTVER ; Version stored in bx above.
        je      @@ret

        mov     dx, OFFSET ErrVersion
        call    Fatal

  @@ret:
        pop     es
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    GetTsrSegment



;--------------------------------------------------------------------------
;
;   NAME:           PossiblyInstall
;
;   DESCRIPTION:    If the variable DoInstall indicates that the program
;                   should be installed, this function does it by hooking
;                   the interrupt vectors, and setting the exit-function
;                   to Terminate and Stay Resident.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
PROC    PossiblyInstall

        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    es

    ; Check if the program should go resident.
        cmp     [DoInstall], 0
        je      @@ret

    ; The program is not already installed, so we install it here.
    ; Make sure the program terminates and stays resident.
        mov     [ExitFunc], 31h

    ; Calculate the number of paragraphs needed.
        mov     ax, OFFSET NonResidentPart

    ; ### Add any extra bytes...


    ; Convert to paragraphs, and store.
        mov     cl, 4
        add     ax, 15
        shr     ax, cl
        mov     [ResPara], ax

    ; Save some bytes by releasing the environment block.
        call    ReleaseEnvironment

    ; Set up a loop that installs all interrupts accoring to the
    ; hooked interrupts table.
        mov     si, OFFSET IntList
  @@hook_next:
        cld
        lodsb                   ; Get interrupt number to AL.
        mov     ah, 35h         ; Get Interrupt Vector.
        int     21h

        mov     dx, bx          ; ES:DX is previous handler.
        mov     bx, [si]        ; Offset of interrupt handler
        inc     si
        inc     si

        mov     [bx + 2], dx    ; Set 'previous' pointer in ISP header.
        mov     [bx + 4], es

        mov     dx, bx          ; DS:DX is our handler.
        mov     ah, 25h         ; Set Interrupt Vector. AL still int#.
        int     21h

        cmp     al, 2Dh         ; INT 2Dh is last in hook list.
        jne     @@hook_next

    ; Let the user know that we have installed the program.
        mov     dx, OFFSET MsgInst
        call    ICondPrint

  @@ret:
        pop     es
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    PossiblyInstall



;--------------------------------------------------------------------------
;
;   NAME:           DisplayStatus
;
;   DESCRIPTION:    ### Display the state of all variables, etc.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    DisplayStatus

        push    ax
        push    dx
        push    es

        mov     es, [TsrSeg]

    ; Give us some space!
        mov     dx, OFFSET SttCrLf
        call    CondPrint

    ; Should 50/43 lines always be forced?
        mov     dx, OFFSET SttAlways
        mov     al, [es: Always]
        call    StatEnblDsbl

    ; Do we use Norwegian letters?
        mov     dx, OFFSET SttNorw
        mov     al, [es: Norwegian]
        call    StatOnOff

    ; Is the program enabled or disabled?
        mov     dx, OFFSET SttStatus
        mov     al, [es: Enabled]
        call    StatEnblDsbl

  @@ret:
        pop     es
        pop     dx
        pop     ax

        ret

ENDP    DisplayStatus



;--------------------------------------------------------------------------
;
;   NAME:           GetArgWord
;
;   DESCRIPTION:    Get an unsigned integer from the argument line,
;                   skipping any whitespace-characters in front.
;
;                   Aborts if no number found, or if the number is
;                   too big.
;
;   IN:             Nothing.
;
;   OUT:            AX   the number.
;
;   DESTROYS:       AX
;
;
PROC    GetArgWord

        push    dx
        push    si

        mov     si, [ArgPtr]

  @@skip_whitespace:
    ; First skip whitespace, and check if we have reached the end.
        mov     al, [si]
        cmp     al, 13          ; CR marks the end of the line.
        je      @@no_number
        call    IsSpace
        jne     @@whitespace_skipped
        inc     si
        jmp     @@skip_whitespace

  @@no_number:
        mov     dx, OFFSET ErrNumExp
        call    Fatal

  @@too_big:
        mov     dx, OFFSET ErrTooBigN
        call    Fatal

  @@whitespace_skipped:
        mov     dl, al
        xor     ax, ax
        cmp     dl, '0'
        jb      @@no_number
        cmp     dl, '9'
        ja      @@no_number

  @@add_digit:
        push    dx
        mov     dx, 10
        mul     dx
        or      dx, dx
        pop     dx
        jnz     @@too_big
        sub     dl, '0'
        xor     dh, dh
        add     ax, dx
        jc      @@too_big
        inc     si
        mov     dl, [si]
        cmp     dl, '0'
        jb      @@finished
        cmp     dl, '9'
        jbe     @@add_digit

  @@finished:
        mov     [ArgPtr], si

  @@ret:
        pop     si
        pop     dx
        ret

ENDP    GetArgWord



;--------------------------------------------------------------------------
;
;   NAME:           GeneralArgument
;
;   DESCRIPTION:    ### Respond to command line arguments that does not
;                   start with '-' or '/'.
;
;                   The procedure must update ArgPtr to after the last
;                   character used from the argument.
;
;                   If ArgPtr is left pointing to non-whitespace, an error
;                   message will be displayed.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    GeneralArgument

  @@ret:
        ret

ENDP    GeneralArgument



;--------------------------------------------------------------------------
;
;   NAME:           OptHelp
;
;   DESCRIPTION:    Respond to options for displaying program usage.
;                   The program is not made resident when this option
;                   is given.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptHelp

        push    dx

        mov     dx, OFFSET MsgUsage
        call    Print

        mov     [DoInstall], 0
        mov     [DoQuiet], 1

  @@ret:
        pop     dx
        ret

ENDP    OptHelp



;--------------------------------------------------------------------------
;
;   NAME:           OptAlwaysOn
;
;   DESCRIPTION:    Respond to option for enabling always forcing letters.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptAlwaysOn

        push    es

        mov     es, [TsrSeg]
        mov     [es: Always], 1

  @@ret:
        pop     es
        ret

ENDP    OptAlwaysOn



;--------------------------------------------------------------------------
;
;   NAME:           OptAlwaysOff
;
;   DESCRIPTION:    Respond to option for disabling always forcing letters.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptAlwaysOff

        push    es

        mov     es, [TsrSeg]
        mov     [es: Always], 0

  @@ret:
        pop     es
        ret

ENDP    OptAlwaysOff



;--------------------------------------------------------------------------
;
;   NAME:           OptNorwOn
;
;   DESCRIPTION:    Respond to option for enabling Norwegian letters.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptNorwOn

        push    es

        mov     es, [TsrSeg]
        mov     [es: Norwegian], 1

  @@ret:
        pop     es
        ret

ENDP    OptNorwOn



;--------------------------------------------------------------------------
;
;   NAME:           OptNorwOff
;
;   DESCRIPTION:    Respond to option for disabling Norwegian letters.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptNorwOff

        push    es

        mov     es, [TsrSeg]
        mov     [es: Norwegian], 0

  @@ret:
        pop     es
        ret

ENDP    OptNorwOff



;--------------------------------------------------------------------------
;
;   NAME:           OptTurnOn
;
;   DESCRIPTION:    Respond to option for reenabling the program.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptTurnOn

        push    es

        mov     es, [TsrSeg]
        mov     [es: Enabled], 1

  @@ret:
        pop     es
        ret

ENDP    OptTurnOn



;--------------------------------------------------------------------------
;
;   NAME:           OptTurnOff
;
;   DESCRIPTION:    Respond to option for disabling the program.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptTurnOff

        push    es

        mov     es, [TsrSeg]
        mov     [es: Enabled], 0

  @@ret:
        pop     es
        ret

ENDP    OptTurnOff



;--------------------------------------------------------------------------
;
;   NAME:           OptQuiet
;
;   DESCRIPTION:    Respond to option to enter Quiet mode (no status
;                   messages).
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    OptQuiet

        mov     [DoQuiet], 1

  @@ret:
        ret

ENDP    OptQuiet



;--------------------------------------------------------------------------
;
;   NAME:           OptUninstall
;
;   DESCRIPTION:    Respond to option to uninstall the program. We just
;                   uninstall and quit the program. No need to go on.
;
;   IN:             Nothing.
;
;   OUT:            Nothing. Never returns.
;
;   DESTROYS:       N/A
;
;
PROC    OptUninstall

    ; Check if the program is resident. If not, there's nothing
    ; to uninstall.
        mov     ax, [TsrSeg]
        mov     dx, cs
        cmp     ax, dx
        jne     @@ok_to_uninstall

        mov     dx, OFFSET ErrUninst1
        call    Fatal

    ; We never get here . . .

  @@ok_to_uninstall:
    ; Get pointer to the hooked interrupt table, as needed by the
    ; UnhookInterrupts-procedure.
        mov     ah, [MultiplexID]
        mov     al, 4           ; Determain Chained Interrupts.
        int     2Dh

    ; Our program will always return status 4, DX:BX is pointer
    ; to hooked interrupt chain. Unhook the interrupts, and
    ; release the memory.
        call    UnhookInterrupts

        mov     es, [TsrSeg]
        mov     ah, 49h         ; Release Memory Block.
        int     21h

    ; Display a message.
        mov     dx, OFFSET MsgUninst
        call    ICondPrint

    ; Quit the program.
        xor     ax, ax
        call    Exit

    ; We never get here . . .

ENDP    OptUninstall



;--------------------------------------------------------------------------
;
;   NAME:           HandleArguments
;
;   DESCRIPTION:    Interpret the command line arguments, if any. Arguments
;                   starting with '-' or '/' are matched against the
;                   Options-table. Any match causes a call to the table-
;                   specified procedure.
;
;                   Arguments that does not start with '-' or '/' are
;                   passed to GeneralArgument.
;
;                   Any argument handling procedure can test ArgPtr
;                   (address of next character in command line) to find
;                   any additional info following the argument. In that
;                   case ArgPtr should be advanced to the first character
;                   after the ones used.
;
;   IN:             Nothing.
;
;   OUT:            Nothing.
;
;   DESTROYS:       None.
;
;
PROC    HandleArguments

        push    ax
        push    bx
        push    cx
        push    dx
        push    di
        push    si
        push    es

    ; Start at the beginning of the command line buffer.
        mov     [ArgPtr], 81h   ; First character in command line.

  @@get_next_arg:
        mov     si, [ArgPtr]

  @@skip_whitespace:
    ; First skip whitespace, and check if we have reached the end.
        mov     al, [si]
        cmp     al, 13          ; CR marks the end of the line.
        je      @@ret
        call    IsSpace
        jne     @@whitespace_skipped
        inc     si
        jmp     @@skip_whitespace

  @@whitespace_skipped:
    ; Update pointers.
        mov     [ArgPtr], si
        mov     [ArgCurrOpt], si

    ; Check if this is an option (starting with '-' or '/').
        cmp     al, '-'
        je      @@option_found
        cmp     al, '/'
        je      @@option_found

    ; This is not an option, just a plain argument. Give it to
    ; the general argument procedure.
        call    GeneralArgument

  @@check_if_completed:
    ; Check if the entire argument was consumed. If not, it's an error.
        mov     si, [ArgPtr]
        mov     al, [si]
        cmp     al, 13
        je      @@ret
        call    IsSpace
        je      @@get_next_arg

  @@unknown_argument:
    ; The entire argument was not used. Display an error message and abort.
    ; First we zero-terminate the current argument.
        mov     si, [ArgCurrOpt]
        mov     dx, OFFSET ErrArgument
        mov     bx, si

  @@find_whitespace:
        mov     al, [si]
        call    IsSpace
        je      @@whitespace_found
        inc     si
        jmp     @@find_whitespace

  @@whitespace_found:
        mov     [BYTE si], 0
        call    Fatal2

    ; We never get here . . .

  @@option_found:
    ; We have an option. Compare this with all options in Options.
        mov     di, OFFSET Options
        push    ds
        pop     es

  @@compare_next:
    ; Skip the '-' or '/'.
        mov     si, [ArgCurrOpt]
        inc     si

    ; Now do the compare. At the same time check if we've reached
    ; the end of possible options.
        mov     dx, di
        call    StrLen
        mov     cx, ax
        jcxz    @@unknown_argument
        cld
        repe    cmpsb
        je      @@found_option

    ; No match on this one. Move on to the next. First find the zero-
    ; byte terminating the option string.
  @@get_zero:
        mov     al, [di]
        or      al, al
        je      @@found_zero
        inc     di
        jmp     @@get_zero

  @@found_zero:
    ; Then skip the zero and the procedure address.
        add     di, 3

    ; Look for more!
        jmp     @@compare_next

  @@found_option:
    ; Update the argument pointer to after what we have compared.
        mov     [ArgPtr], si

    ; Skip the terminating zero-byte to get to the function, and
    ; call it.
        inc     di
        call    [WORD NEAR di]

    ; Check if entire option was used.
        jmp     @@check_if_completed

  @@ret:
        pop     es
        pop     si
        pop     di
        pop     dx
        pop     cx
        pop     bx
        pop     ax

        ret

ENDP    HandleArguments



Main:
    ; Display program header.
        mov     dx, OFFSET MsgHeader
        call    Print

    ; Check if EGA/VGA is available.
        mov     ah, 12h
        mov     bl, 10h
        int     10h
        cmp     bl, 10h
        jne     @@EGA_found
        mov     dx, OFFSET ErrEgaVga
        call    Fatal

    ; We never get here... Fatal doesn't return.

  @@EGA_found:
    ; Activate page 0.
        mov     ax, 0500h
        int     10h

    ; Check if already installed, and get segment of resident instance.
        call    GetTsrSegment

    ; Interpret any command line parameters.
        call    HandleArguments

    ; Possibly install the program's interrupt vectors.
        call    PossiblyInstall

    ; We enter extended textmode each time.
        call    ExtendedTextMode

    ; Possibly display any status messages.
        call    DisplayStatus

    ; Exit the program with exitcode 0. Possibly Terminate and Stay Resident.
        mov     ah, [ExitFunc]
        xor     al, al
        mov     dx, [ResPara]
        int     21h



            ENDS
            END EntryPoint
