{ EFLIB | Extended Function Library (C) Johan Larsson, 1992 - 1997
          All rights reserved. E-mail to jola@ts.umu.se.

          EXAMPLE PROGRAM                  [x] Real mode
        | Advanced/Arrayadt.pas            [x] Protected mode

  This unit demonstrates how you can create your own ADT from
  the abstract ADT object delivered with EFLIB (tADT). The
  unit implements a standard Pascal array as an EFLIB ADT.

  EFLIB IS PROTECTED BY THE COPYRIGHT LAW AND MAY NOT BE MANIPULATED,
  DISTRIBUTED OR COPIED. THIS DEMONSTRATION PROGRAM MAY FREELY BE USED
  AND DISTRIBUTED.                                                        }

unit ARRAYADT;


INTERFACE


uses EFDEF, EFKERNEL, EFADT;


const NumberOfElements = 1000;

type { The elements that the array will contain (must be defined
       at compile time). Exercise; modify this element to a pointer
       and try to implement a generic element array with a fixed
       size! }
     ElementType = real;


     pPascalArray = ^tPascalArray;
     tPascalArray = object (tADT)

       { Implementation of a standard Pascal array with a size defined at
         compile-time.

         Because this object is inherited from EFLIBs parent object for
         data types, it has features such as sorting, searching and stream
         storage. These methods are inherited from the EFLIB ADT parent. }

       private

         { Fields }
         fBaseArray     : array [1 .. NumberOfElements] of ElementType;
         fLastUsed      : word;

       public

         { Miscellaneous methods }
         procedure   Clear; virtual;

         { Methods for handling of elements }
         procedure   Add (var Data); virtual;
         procedure   Insert (var Data; Index : word); virtual;
         procedure   Update (Index : word; var Data); virtual;
         procedure   Element (Index : word; var Data); virtual;                                  { Retrieves an element }
         procedure   Erase (Index : word); virtual;

         function    Compare (Index1, Index2 : word) :
                     shortint; virtual;
         function    CompareContent (Index : word; var Data) :
                     shortint; virtual;

         { Methods for stream storage }
         constructor StreamLoad (Stream : pStream);             { Loads from a stream }

         { Methods for direct element access }
         function    ElementSize (Index : word) :
                     word; virtual;
         function    ElementPointer (Index : word) :
                     pointer; virtual;

         { Status methods }
         function    Elements : word; virtual;                  { Number of elements }
         function    Capacity : word; virtual;                  { Capacity of elements }

     end;


IMPLEMENTATION


{ Clears data type (ie. erases all elements). }
procedure tPascalArray.Clear;
begin
     FillChar (fBaseArray, SizeOf(fBaseArray), 0);
     fLastUsed := 0;
end;

{ Adds data into data type in a new element. }
procedure tPascalArray.Add (var Data);
begin
     if fLastUsed < Capacity then begin
        Inc (fLastUsed); fBaseArray[fLastUsed] := ElementType(Data);
     end else Error (Error_Forbidden) { Error; array is full } ;
end;

{ Inserts data to data type in a new element that follows specified indexed
  element in order. If index is zero, element is inserted first in the
  data type. }
procedure tPascalArray.Insert (var Data; Index : word);
var Count : word;
begin
     if Capacity > Elements then begin
        { Pull elements inside array to make space for a new element }
        for Count := Elements downto Succ(Index) do
            fBaseArray[Succ(Count)] := fBaseArray[Count];
        Inc (fLastUsed); fBaseArray[Index] := ElementType (Data);
     end else Error (Error_Forbidden) { Error; array is full } ;
end;

{ Updates an element in the data type. }
procedure tPascalArray.Update (Index : word; var Data);
begin
     if (Index >= 1) and (Index <= Elements) then
        fBaseArray[Index] := ElementType(Data)
     else Error (Error_Forbidden) { Error; range check error; not a valid element index } ;
end;

{ Returns the data in an indexed element in the data type. }
procedure tPascalArray.Element (Index : word; var Data);
begin
     if IsValid (Index) then
        Move (fBaseArray[Index], Data, ElementSize(Index))
     else Error (Error_Forbidden) { Error; range check error; not a valid element index } ;
end;

{ Erases an element from the data type. This is a method that must be
  overridden by all descendants. }
procedure tPascalArray.Erase (Index : word);
var Count : word;
begin
     if IsValid(Index) then begin
        { Pull elements inside array to make space for a new element }
        for Count := Index to Pred(Elements) do
            fBaseArray[Count] := fBaseArray[Succ(Count)];
        Dec (fLastUsed);
     end else Error (Error_Forbidden) { Error; range check error; not a valid element index } ;
end;

{ Compares two indexed elements inside the data type and returns
  1, 0 or -1, depending on if the first element is bigger, equal
  or smaller than the second element. }
function tPascalArray.Compare (Index1, Index2 : word) : shortint;
begin
     if fBaseArray[Index1] > fBaseArray[Index2] then Compare := 1
        else if fBaseArray[Index1] < fBaseArray[Index2] then Compare := -1
             else Compare := 0;
end;

{ Compares the content of an elements with some data and returns
  1, 0 or -1, depending on if the element is bigger, equal or smaller
  than the data. }
function tPascalArray.CompareContent (Index : word; var Data) : shortint;
begin
     if fBaseArray[Index] > ElementType(Data) then CompareContent := 1
        else if fBaseArray[Index] < ElementType(Data) then CompareContent := -1
             else CompareContent := 0;
end;


{ Constructs and loads the object from a stream. This is an abstract
  constructor that must be overridden by all descendants that support
  stream storage. }
constructor tPascalArray.StreamLoad (Stream : pStream);
begin
     if IsValidStream (Stream) then with Stream^ do begin

        { Load object data }
        Initialize;
        Inherited StreamLoad (Stream);

     end else Error (Error_Stream) { Error; failed to load object } ;
end;


{ Returns the size of elements inside the data type. }
function tPascalArray.ElementSize (Index : word) : word;
begin
     ElementSize := SizeOf(ElementType);
end;

{ Returns a pointer to a specified elements data region. }
function tPascalArray.ElementPointer (Index : word) : pointer;
begin
     ElementPointer := @fBaseArray [Index];
end;


{ Returns the number of elements inside the data type. }
function tPascalArray.Elements : word;
begin
     Elements := fLastUsed;
end;

{ Returns the number of elements that can be stored inside the data
  type. }
function tPascalArray.Capacity : word;
begin
     Capacity := SizeOf(fBaseArray) div ElementSize (0);
end;



begin

     with Classes^ do

          { All new classes must be registered to
            take advantage of EFLIB's advanced OO mechanisms. }

           Register ( 1000, 'tPascalArray', TypeOf(tPascalArray), TypeOf(tADT),
                            @tPascalArray.StreamLoad, @tPascalArray.StreamStore );

end. { unit }