{$F+} { Compiler Directive: Generate far calls: On } { DO NOT CHANGE! }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  Draw
    Version 1.0

    This unit holds a nice assortment of graphical drawing routines for
      drawing various objects on screen.

    Purpose:
      This unit combines an assortment of rasper graphic drawing routines
      that are designed to make it easy to draw various graphical objects
      on the screen.  Included are routines to draw arcs, ellipse, circles,
      boxes and many other objects.

    How it works:
      Each routine uses the fastest method to perform the various tasks.

    Features:
      The routines are designed to be closely compatable with the graphic
        routines included with Turbo Pascal (R).  These routines, however are
        more versitile and by default use the system interrupts to achieve
        maximum versitility.
      All coordinates are taken in pixels from the upper left hand corner and
        proceeding downwards and rightwards as the values increase.  The X
        values represent the pixels to the right of the left side of the
        screen, while the Y values represent the pixels to the bottom of the
        top side of the screen.
      All angles are taken from a reference line proceeding to the right of
        the center point and traveling counter clockwise.  Angles are measured
        in degrees giving a point directly upwards a value of 90, directly to
        the left a value of 180 and directly downwards a value of 270.
      Some of the routines have been optimized with assembly language for use
        with Turbo Pascal version 6.

  Copyright 1991, 1993, All rights reserved.
    Paul R. Renaud

  Compiler:
    Turbo Pascal Versions 5.0 to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit Draw;

  Interface

    Uses
      DOS;

(***********************************************************

  Y_Range
    Defines the range of values for the Y coordinate limit.
    This range determines the maximum resulution allowed
    by the routines for the height of the screen.  The
    larger this range is, the more memory the system will
    need to use.

***********************************************************)

    Type
      Y_Range = 0 .. 768;

(***********************************************************

  Y_Limit
    This value is the current limit to the Y coordinate
    limit.  This value is allowed to change to reflect the
    current video mode.

***********************************************************)

    Const
      Y_Limit: Y_Range = 350;

(***********************************************************

  Colors
    These values hold the current colors used by the various
    graphical routines.  Drawing_Color is the color used by
    the line and curve routines.  Filling_Color is used by
    the filled polygon and object routines.
    Background_Color is the currently defined blanking
    color.

***********************************************************)

    Var
      Drawing_Color,
      Filling_Color,
      Background_Color: Word;

(***********************************************************

  These variable routines are the core subroutines used to
    perform all of the screen drawing.  They may be
    substituted with other routines so that the drawing can
    be redirected to some other device or data structure.

***********************************************************)

      Set_Pixel: Procedure( X, Y: Integer; Color: Word );
      Get_Pixel: Function( X, Y: Integer ): Word;
      Fill_Line: Procedure( X1, X2, Y: Integer; Color: Word );

(***********************************************************

  Procedure: Clear Defaults.

    This procedure clears the default colors to their
    origional values and moves the current pointer to the
    upper left hand corner.

***********************************************************)

    Procedure Clear_Defaults;

(***********************************************************

  Procedure: Draw Arc.

    This procedure draws an arc on the screen centered at
    the given coordinates in CX, CY.  Radius is the number
    of pixels to the top or bottom of the arc, while the
    radius is adjusted to the left and right to compensate
    for the density of the screen.

***********************************************************)

    Procedure Draw_Arc( CX, CY: Integer; Start_Angle, End_Angle, Radius: Word );

(***********************************************************

  Procedure: Draw Box.

    This procedure draws a vertical rectangle on the screen
    at the given coordinates.  Left defines the left
    coordinate of the box, Top the top coordinate, Right the
    right coordinate and Bottom the bottom.  To give the box
    the appearance of depth, a value of depth should be
    supplied.  The right side of the box is drawn to match
    the supplied depth.  When Top_On is true, the top side
    of the box is also drawn.

***********************************************************)

    Procedure Draw_Box( Left, Top, Right, Bottom: Integer; Depth: Word; Top_On: Boolean );

(***********************************************************

  Procedure: Draw Bezier Spline.

    This procedure draws a curvey line that extends around
    from the point defined by E1_X, E1_Y to the point
    defined by E2_X, E2_Y.  The Points C1_X, C1_Y and C2_X,
    C2_Y define the centers of the curve and are used as the
    focal points upon which the lines curve.

***********************************************************)

    Procedure Draw_Bezier_Spline( E1_X, E1_Y, C1_X, C1_Y, E2_X, E2_Y, C2_X, C2_Y: Integer );

(***********************************************************

  Procedure: Draw Circle.

    This procedure draws a circle on the screen centered at
    the coordinates defined by X, Y.  R defines the radius
    of the circle measured from top to bottom.  The radius
    is adjusted widthwise to compensate for the density of
    the screen.

***********************************************************)

    Procedure Draw_Circle( X, Y: Integer; Radius: Word );

(***********************************************************

  Procedure: Draw Elliptical Arc.

    This procedure draws a horizonially aligned ellipical
    arc on the screen.  The center is defined by CX and CY,
    The arc starts at Start_Angle and ends at End_Angle.
    XRadius defines the width of the arc while YRadius
    defines it's height.

***********************************************************)

    Procedure Draw_Elliptical_Arc( CX, CY: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Ellipse.

    This procedure draws a horizonially aligned ellipse on
    the screen.  The center is defined by CX and CY.
    XRadius defines the width of the arc while YRadius
    defines it's height.

***********************************************************)

    Procedure Draw_Ellipse( CX, CY: Integer; X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Filled Circle.

    This procedure draws a filled circle at the coordinates
    given by X and Y.  The height of the circle is defined
    by Radius while the width is automatically adjusted to 
    make the circle appear roundest.

***********************************************************)

    Procedure Draw_Filled_Circle( X, Y: Integer; Radius: Word );

(***********************************************************

  Procedure: Draw Filled Ellipse.

    This procedure draws a filled ellipse at the coordinates
    given by X and Y.  The height of the circle is defined
    by YRadius, while the width is defined by XRadius.

***********************************************************)

    Procedure Draw_Filled_Ellipse( CX, CY: Integer; X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Filled Polygon.

    Unfinished?
    This procedure draws a filled polygon using the same
    mannor as the Turbo Pascal's graphical unit.  Amount is
    a word indicating how many points are in the data
    structure while Data is an array of points ( X followed
    by Y as integers ) which define the edges of the polygon
    in a linked fashion.

***********************************************************)

    Procedure Draw_Filled_Polygon( Amount: Word; Var Data );

(***********************************************************

  Procedure Draw Filled Rectangle.

    This procedure draws a vertical rectangle on the screen
    at the given coordinates.  Left defines the left
    coordinate of the box, Top the top coordinate, Right the
    right coordinate and Bottom the bottom.

***********************************************************)

    Procedure Draw_Filled_Rectangle( Left, Top, Right, Bottom: Integer );

(***********************************************************

  Procedure: Draw Filled Rounded Box.

    This procedure draws a box on the screen with rounded
    edges at the given coordinates.  X1, Y1 define the
    center of the left top arc while X2, Y2 defines the
    center of the right bottom arc.

***********************************************************)

    Procedure Draw_Filled_Rounded_Box( X1, Y1, X2, Y2: Integer; Radius: Word );

(***********************************************************

  Procedure: Draw Line.

    This procedure draws a line from the coordinates defined
    by X1, Y1 to the coordinates defined by X2, Y2 in the
    current drawing color.

***********************************************************)

    Procedure Draw_Line( X1, Y1, X2, Y2: Integer );

(***********************************************************

  Procedure: Draw Line Relative.

    This procedure draws a line from the current pointer to
    a point that is defined by DX, DY as a distance from the
    current pointer.  The current pointer is afterwards at
    the endpoint.

***********************************************************)

    Procedure Draw_Line_Relative( DX, DY: Integer );

(***********************************************************

  Procedure: Draw Line To.

    This procedure draws a line from the current pointer to
    the absolute point defined by X, Y.  The current pointer
    is then at the new endpoint.

***********************************************************)

    Procedure Draw_Line_To( X, Y: Integer );

(***********************************************************

  Procedure: Draw Pie Slice.
    This procedure draws a segment of a circle, centered on
    the screen at the given coordinates X1, Y1.  Radius
    defines the height and the Width is automatically
    adjusted.

***********************************************************)

    Procedure Draw_Pie_Slice( X1, Y1: Integer; Start_Angle, End_Angle, Radius: Word );

(***********************************************************

  Procedure: Draw Polygon.

    This procedure draws a polygon using the same manner as
    the Turbo Pascal's graphical unit.  Amount is a word
    indicating how many points are in the data structure
    while Data is an array of points ( X followed by Y as
    integers ) which define the edges of the polygon in a
    linked fashion.  The last point should match the first
    to close the polygon.

***********************************************************)

    Procedure Draw_Polygon( Amount: Word; Var Data );

(***********************************************************

  Procedure: Draw Rectangle.

    This procedure draws a vertical rectangle outline on the
    screen at the given coordinates.  Left defines the left
    coordinate of the box, Top the top coordinate, Right the
    right coordinate and Bottom the bottom.

***********************************************************)

    Procedure Draw_Rectangle( Left, Top, Right, Bottom: Integer );

(***********************************************************

  Procedure: Draw Rounded Box.

    This procedure draws the outline of a box on the screen
    with rounded edges at the given coordinates.  X1, Y1
    define the center of the left top arc while X2, Y2
    defines the center of the right bottom arc.

***********************************************************)

    Procedure Draw_Rounded_Box( X1, Y1, X2, Y2: Integer; Radius: Word );

(***********************************************************

  Procedure: Draw Sector.

    This procedure draws a segment of a filled ellipical
    arc, centered on the screen at the given coordinates X1,
    Y1.  Y_Radius defines the height, while X_radius defines
    the width.

***********************************************************)

    Procedure Draw_Sector( X1, Y1: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Get Aspect Ratio.

    This procedure returns the current values for the X
    aspect ratio and the Y aspect ratio.

***********************************************************)

    Procedure Get_Aspect_Ratio( Var X_Ratio, Y_Ratio: Word );

(***********************************************************

  Procedure: Set Aspect Ratio.

    This procedure sets the values for the X aspect ratio
    and the Y aspect ratio.

***********************************************************)

    Procedure Set_Aspect_Ratio( X_Ratio, Y_Ratio: Word );

(***********************************************************

  Procedure: Move Relative.

    This procedure moves the current pointer from the
    current position to a point that is defined by DX, DY as
    a distance from the current location.  Nothing is
    displayed on the screen.

***********************************************************)

    Procedure Move_Relative( DX, DY: Integer );

(***********************************************************

  Procedure: Move To.

    This procedure moves the current pointer from the
    current position to the absolute point that defined by
    X, Y.  Nothing is displayed on the screen.

***********************************************************)

    Procedure Move_To( X, Y: Integer );

(***********************************************************

  Procedure: Set Background Color.

    This procedure is the prefered way to change the
    background working color.  The currently displayed
    color on the screen is unaffected.

***********************************************************)

    Procedure Set_Background_Color( Color: Word );

(***********************************************************

  Procedure: Set Filling Color.

    This procedure is the prefered way to change the
    current filling color.  The currently displayed color on
    the screen is unaffected.

***********************************************************)

    Procedure Set_Filling_Color( Color: Word );

(***********************************************************

  Procedure: Set Color.

    This procedure is the prefered way to change the
    current drawing color.  The currently displayed color on
    the screen is unaffected.

***********************************************************)

    Procedure Set_Color( Color: Word );

(***********************************************************

  Procedure: Roll Point.

    This procedure rolls a point around the X,Y axis by the
    amount given.

***********************************************************)

    Procedure Roll_Point( Var X, Y: Real; Amount: Real );

(***********************************************************

  Procedure: Draw Disjoint Ellipse.

    This procedure draws seperate segments of a single
    ellipse in a disjointed manner.  When used properly,
    this routine can make drawing multicurved objects much
    quicker.  See Draw_Rounded_Box.

***********************************************************)

    Procedure Draw_Disjoint_Ellipse( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Ellipse Box.

    This procedure draws the outline of a box on the screen
    with rounded edges at the given coordinates.  X1, Y1
    define the center of the left top arc while X2, Y2
    defines the center of the right bottom arc.

***********************************************************)

    Procedure Draw_Ellipse_Box( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Filled Ellipse Box.

    This procedure draws a box on the screen with rounded
    edges at the given coordinates.  X1, Y1 define the
    center of the left top arc while X2, Y2 defines the
    center of the right bottom arc.

***********************************************************)

    Procedure Draw_Filled_Ellipse_Box( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Elliptical Arc Job.

    This procedure plots an ellipical arc in the screen
    buffer.  See Draw_Elliptical_Arc.

***********************************************************)

    Procedure Draw_Elliptical_Arc_Job( CX, CY: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );

(***********************************************************

  Procedure: Draw Filled Ellipse Job.

    This procedure plots a filled ellipse in the screen
    buffer.  See Draw_Filled_Ellipse.

***********************************************************)

    Procedure Draw_Filled_Ellipse_Job( CX, CY: Integer; X_Radius, Y_Radius: Word );

{----------------------------------------------------------------------------}

  Implementation

    Uses
      FillCode;

    Const
     { This value defines the limit to the resolution of the Bezier Spline. }
      Bezier_Spline_Time_Limit = 100;

    Type
     { Definition of the standard Integer point data structure. }
      Point =  Record
                 X,
                 Y: Integer;
               End;
     { Definition of an array of standard Integer points. }
      Points = array[ 1 .. 16383 ] of Point;
     { Record to hold the data of the arc coordinates. }
      Arc_Coordinates_Type = Record
                               Center_X_Coordinate,
                               Center_Y_Coordinate,
                               Start_X_Coordinate,
                               Start_Y_Coordinate,
                               Finish_X_Coordinate,
                               Finish_Y_Coordinate: Integer;
                             End;
     { Data type used to hold values for circle plotting calcuations. }
      DType = array[ 1 .. 2 ] of LongInt;
      TType = array[ 1 .. 9 ] of LongInt;

    Var
     { The current pointer holding variables. }
      Current_Pointer_X,
      Current_Pointer_Y: Integer;
     { Holds the data from the last arc operation. }
      Arc_Coordinate: Arc_Coordinates_Type;
     { Holds the current ratio data. }
      X_Aspect_Ratio,
      Y_Aspect_Ratio: Word;
      Aspect_Ratio: Real;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Line Forward.
    This procedure draws a line in the forward
    mannor from the X, Y point to the A, B point.
    This procedure uses the quickest method to
    draw the line.

*************************************************)

    Procedure Line_Forward( X, Y, A, B: Integer );
     {$IFDEF Ver60 }
      Assembler;
      Asm
        Mov Ax, B          { Ax = B }
        Cmp Ax, A          { Compare Ax to A }
        JG  @Part2         { Do Part2 if A > B }
        Mov Bx, B          { Put B in Bx }
        Sub Bx, A          { Put B - A in Bx }
        Mov Ax, A          { Put A in Ax }
        SAR Ax, 1          { Div A by 2 }
        Mov Dx, B          { Put B in Dx }
        Sub Dx, Ax         { Subtract A Div 2 From B }
        Mov Cx, A          { Mov A Into Cx }
        JCxZ @Out          { If Cz is Zero then leave }
        Inc CX             { Prepair CX for loop }
        @Big_Loop1:
        Push Bx            { Save register }
        Push Cx            { Save register }
        Push Dx            { Save register }
        Push X             { Push parameter }
        Push Y             { Push parameter }
        Push Drawing_Color { Push parameter }
        Call Set_Pixel     { Call Set_Pixel }
        Pop Dx             { Restore register }
        Pop Cx             { Restore register }
        Pop Bx             { Restore register }
        Inc X              { Increment X }
        Cmp Dx, 0          { Is Dx less than zero? }
        JGE @Over1         { If not then skip }
        Add Dx, B          { Add B to Dx }
        Loop @Big_Loop1    { Redo all }
        Jmp @Out           { or leave }
        @Over1:
        Inc Y              { Increment Y }
        Add Dx, Bx         { Add Bx to Dx }
        Loop @Big_Loop1    { Redo all }
        Jmp @Out           { or leave }
        @Part2:
        Mov Bx, A          { Put A in Bx }
        Sub Bx, B          { Put A - B in Bx }
        Mov Ax, B          { Put B in Ax }
        SAR Ax, 1          { Div B by 2 }
        Mov Dx, A          { Put A in Dx }
        Sub Dx, Ax         { Subtract B div 2 From A }
        Mov Cx, B          { Mov B Into Cx }
        JCxZ @Out          { If Cx is zero then leave }
        Inc CX             { Prepair Cx for loop }
        @Big_Loop2:
        Push Bx            { Save register }
        Push Cx            { Save register }
        Push Dx            { Save register }
        Push X             { Push parameter }
        Push Y             { Push parameter }
        Push Drawing_Color { Push parameter }
        Call Set_Pixel     { Call Set_Pixel }
        Pop Dx             { Restore register }
        Pop Cx             { Restore register }
        Pop Bx             { Restore register }
        Inc Y              { Increment Y }
        Cmp Dx, 0          { Is Dx less than zero? }
        JGE @Over2         { If not then skip }
        Add Dx, A          { Add A to Dx }
        Loop @Big_Loop2    { Redo all }
        Jmp @Out           { or leave }
        @Over2:
        Inc X              { Increment X }
        Add Dx, Bx         { Add Bx to Dx }
        Loop @Big_Loop2    { Redo all }
        @Out:
      End;
     {$else}
      Var
        I,
        T,
        D: Integer;
      Begin
        If ( A > B )
          then
            Begin
              T := ( B - A );
              D := B - A div 2;
              For I := 0 to A do
                Begin
                  Set_Pixel( X, Y, Drawing_Color );
                  Inc( X );
                  If ( D < 0 )
                    then
                      D := ( D + B )
                    else
                      Begin
                        Inc( Y );
                        D := ( D + T );
                      End;
                End;
            End
          else
            Begin
              T := ( A - B );
              D := A - B div 2;
              For I := 0 to B do
                Begin
                  Set_Pixel( X, Y, Drawing_Color );
                  Inc( Y );
                  If ( D < 0 )
                    then
                      D := ( D + A )
                    else
                      Begin
                        Inc( X );
                        D := ( D + T );
                      End;
                End;
            End;
      End;
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Line Backward.
    This procedure draws a line in the backward
    mannor from the X, Y point to the A, B point.
    This procedure uses the quickest method to
    draw the line.

*************************************************)

    Procedure Line_Backward( X, Y, A, B: Integer );
     {$IFDEF VER60 }
      Assembler;
      Asm
        Mov Ax, B          { Ax = B }
        Cmp Ax, A          { Compare Ax to A }
        JG  @Part2         { do part2 if A > B }
        Mov Bx, B          { Put B in Bx }
        Sub Bx, A          { Put B - A in Bx }
        Mov Ax, A          { Put A in Ax }
        SAR Ax, 1          { Div A by 2 }
        Mov Dx, B          { Put B in Dx }
        Sub Dx, Ax         { Subtract A div 2 From B }
        Mov Cx, A          { Mov A into Cx }
        JCxZ @Out          { If CX is zero then leave }
        Inc CX             { Prepair Cx for loop }
        @Big_Loop1:
        Push Bx            { Save register }
        Push Cx            { Save register }
        Push Dx            { Save register }
        Push X             { Push parameter }
        Push Y             { Push parameter }
        Push Drawing_Color { Push parameter }
        Call Set_Pixel     { Call Set_Pixel }
        Pop Dx             { Restore register }
        Pop Cx             { Restore register }
        Pop Bx             { Restore register }
        Inc X              { Increment X }
        Cmp Dx, 0          { Is Dx less than zero? }
        JGE @Over1         { If not then skip }
        Add Dx, B          { Add B to Dx }
        Loop @Big_Loop1    { Redo all }
        Jmp @Out           { or leave }
        @Over1:
        Dec Y              { Decrement Y }
        Add Dx, Bx         { Add Bx to Dx }
        Loop @Big_Loop1    { Redo all }
        Jmp @Out
        @Part2:
        Mov Bx, A          { Put A in Bx }
        Sub Bx, B          { Put A - B in Bx }
        Mov Ax, B          { Put B in Ax }
        SAR Ax, 1          { Div B by 2 }
        Mov Dx, A          { Put A in Dx }
        Sub Dx, Ax         { Subtract B div 2 From A }
        Mov Cx, B          { Mov B into Cx }
        JCxZ @Out          { If CX is Zero then Leave }
        Inc CX             { Prepair CX For Loop }
        @Big_Loop2:
        Push Bx            { Save register }
        Push Cx            { Save register }
        Push Dx            { Save register }
        Push X             { Push parameter }
        Push Y             { Push parameter }
        Push Drawing_Color { Push parameter }
        Call Set_Pixel     { Call Set_Pixel }
        Pop Dx             { Restore register }
        Pop Cx             { Restore register }
        Pop Bx             { Restore register }
        Dec Y              { Decrement Y }
        Cmp Dx, 0          { Is Dx less than zero? }
        JGE @Over2         { If not then skip }
        Add Dx, A          { Add A to Dx }
        Loop @Big_Loop2    { Redo all }
        Jmp @Out           { or leave }
        @Over2:
        Inc X              { Increment X }
        Add Dx, Bx         { Add Bx to Dx }
        Loop @Big_Loop2    { Redo all }
        @Out:
      End;
     {$else}
      Var
        I,
        T,
        D: Integer;
      Begin
        If ( A > B )
          then
            Begin
              T := ( B - A );
              D := B - A div 2;
              For I := 0 to A do
                Begin
                  Set_Pixel( X, Y, Drawing_Color );
                  Inc( X );
                  If ( D < 0 )
                    then
                      D := ( D + B )
                    else
                      Begin
                        Dec( Y );
                        D := ( D + T );
                      End;
                End;
            End
          else
            Begin
              T := ( A - B );
              D := A - B div 2;
              For I := 0 to B do
                Begin
                  Set_Pixel( X, Y, Drawing_Color );
                  Dec( Y );
                  If ( D < 0 )
                    then
                      D := ( D + A )
                    else
                      Begin
                        Inc( X );
                        D := ( D + T );
                      End;
                End;
            End;
      End;
     {$ENDIF}

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Line.
    As previously defined.

*************************************************)

    Procedure Draw_Line( X1, Y1, X2, Y2: Integer );
      Var
        A,
        B: Integer;
      Begin
        A := ( X2 - X1 );
        B := ( Y2 - Y1 );
        If ( A > 0 )
          then
            If ( B > 0 )
              then
                Line_Forward( X1, Y1, A, B )
              else
                Line_Backward( X1, Y1, A, -B )
          else
            If ( B > 0 )
              then
                Line_Backward( X2, Y2, -A, B )
              else
                Line_Forward( X2, Y2, -A, -B );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Prepair for Circle.
    This procedure sets up the data arrays with
    the initial calculations for plotting an
    ellipical arc.

*************************************************)

    Procedure Prepair_For_Circle( Var X, Y: Integer; XRadius, YRadius: Word; Var T: TType; Var D: DType );
      Begin
        X := XRadius;
        Y := 0;
        T[ 1 ] := Sqr( XRadius );
        T[ 2 ] := ( 2 * T[ 1 ] );
        T[ 3 ] := ( 2 * T[ 2 ] );
        T[ 4 ] := Sqr( YRadius );
        T[ 5 ] := ( 2 * T[ 4 ] );
        T[ 6 ] := ( 2 * T[ 5 ] );
        T[ 7 ] := ( XRadius * T[ 5 ] );
        T[ 8 ] := ( 2 * T[ 7 ] );
        T[ 9 ] := 0;
        D[ 1 ] := ( T[ 2 ] - T[ 7 ] + ( T[ 4 ] div 2 ) );
        D[ 2 ] := ( ( T[ 1 ] div 2 ) - T[ 8 ] + T[ 5 ] );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Circle Calculate1.
    This procedure performs the first set of
    calculations for plotting an ellipical arc.

*************************************************)

    Procedure Circle_Calculate1( Var X, Y: Integer; Var T: TType; Var D: DType );
      Begin
        Inc( Y );
        T[ 9 ] := ( T[ 9 ] + T[ 3 ] );
        If ( D[ 1 ] < 0 )
          then
            Begin
              D[ 1 ] := ( D[ 1 ] + T[ 9 ] + T[ 2 ] );
              D[ 2 ] := ( D[ 2 ] + T[ 9 ] );
            End
          else
            Begin
              Dec( X );
              T[ 8 ] := ( T[ 8 ] - T[ 6 ] );
              D[ 1 ] := ( D[ 1 ] - T[ 8 ] + T[ 9 ] + T[ 2 ] );
              D[ 2 ] := ( D[ 2 ] - T[ 8 ] + T[ 9 ] + T[ 5 ] );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Circle Calculate2.
    This procedure performs the second set of
    calculations for plotting an ellipical arc.

*************************************************)

    Procedure Circle_Calculate2( Var X, Y: Integer; Var T: TType; Var D: DType );
      Begin
        Dec( X );
        T[ 8 ] := ( T[ 8 ] - T[ 6 ] );
        If ( D[ 2 ] < 0 )
          then
            Begin
              Inc( Y );
              T[ 9 ] := ( T[ 9 ] + T[ 3 ] );
              D[ 2 ] := ( D[ 2 ] - T[ 8 ] + T[ 5 ] + T[ 9 ] );
            End
          else
            D[ 2 ] := ( D[ 2 ] - T[ 8 ] + T[ 5 ] );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Part Circle1.
    This procedure performs part of the
    calculations for plotting an ellipical arc.

*************************************************)

    Procedure Part_Circle1( Var X: Integer; Var T: TType );
      Begin
        Dec( X );
        T[ 8 ] := ( T[ 8 ] - T[ 6 ] );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Part Circle2.
    This procedure performs part of the
    calculations for plotting an ellipical arc.

*************************************************)

    Procedure Part_Circle2( Var Y: Integer; Var T: TType; Var D: DType );
      Begin
        Inc( Y );
        T[ 9 ] := ( T[ 9 ] + T[ 3 ] );
        D[ 2 ] := ( D[ 2 ] - T[ 8 ] + T[ 5 ] + T[ 9 ] );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Part Circle3.
    This procedure performs part of the
    calculations for plotting an ellipical arc.

*************************************************)

    Procedure Part_Circle3( Var T: TType; Var D: DType );
      Begin
        D[ 2 ] := ( D[ 2 ] - T[ 8 ] + T[ 5 ] );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Ellipse.
    As previously defined.

*************************************************)

    Procedure Draw_Ellipse( CX, CY: Integer; X_Radius, Y_Radius: Word );
      Var
        X,
        Y: Integer;
        T: TType;
        D: DType;
      Begin
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            Set_Pixel( ( CX + X ), ( CY + Y ), Drawing_Color );
            Set_Pixel( ( CX - X ), ( CY + Y ), Drawing_Color );
            Set_Pixel( ( CX + X ), ( CY - Y ), Drawing_Color );
            Set_Pixel( ( CX - X ), ( CY - Y ), Drawing_Color );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Set_Pixel( ( CX + X ), ( CY + Y ), Drawing_Color );
          Set_Pixel( ( CX - X ), ( CY + Y ), Drawing_Color );
          Set_Pixel( ( CX + X ), ( CY - Y ), Drawing_Color );
          Set_Pixel( ( CX - X ), ( CY - Y ), Drawing_Color );
          Circle_Calculate2( X, Y, T, D );
        Until ( X < 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Disjoint Ellipse.
    As previously defined.

*************************************************)

    Procedure Draw_Disjoint_Ellipse( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );
      Var
        X,
        Y,
        Value: Integer;
        T: TType;
        D: DType;
      Begin
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            Set_Pixel( ( X2 + X ), ( Y2 + Y ), Drawing_Color );
            Set_Pixel( ( X1 - X ), ( Y2 + Y ), Drawing_Color );
            Set_Pixel( ( X2 + X ), ( Y1 - Y ), Drawing_Color );
            Set_Pixel( ( X1 - X ), ( Y1 - Y ), Drawing_Color );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Set_Pixel( ( X2 + X ), ( Y2 + Y ), Drawing_Color );
          Set_Pixel( ( X1 - X ), ( Y2 + Y ), Drawing_Color );
          Set_Pixel( ( X2 + X ), ( Y1 - Y ), Drawing_Color );
          Set_Pixel( ( X1 - X ), ( Y1 - Y ), Drawing_Color );
          Circle_Calculate2( X, Y, T, D );
        Until ( X < 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Ellipse Box.
    As previously defined.

*************************************************)

    Procedure Draw_Ellipse_Box( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );
      Var
        Value: Integer;
      Begin
        Draw_Disjoint_Ellipse( X1, Y1, X2, Y2, X_Radius, Y_Radius );
        Value := ( Y1 - Y_Radius );
        Draw_Line( X1, Value, X2, Value );
        Value := ( X1 - X_Radius );
        Draw_Line( Value, Y1, Value, Y2 );
        Value := ( X2 + X_Radius );
        Draw_Line( Value, Y1, Value, Y2 );
        Value := ( Y2 + Y_Radius );
        Draw_Line( X1, Value, X2, Value );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Ellipse.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Ellipse( CX, CY: Integer; X_Radius, Y_Radius: Word );
      Var
        X,
        Y: Integer;
        T: TType;
        D: DType;
      Begin
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            Fill_Line( ( CX + X ), ( CX - X ), ( CY + Y ), Filling_Color );
            Fill_Line( ( CX + X ), ( CX - X ), ( CY - Y ), Filling_Color );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Part_Circle1( X, T );
          If ( D[ 2 ] < 0 )
            then
              Begin
                Fill_Line( ( CX + X ), ( CX - X ), ( CY + Y ), Filling_Color );
                Fill_Line( ( CX + X ), ( CX - X ), ( CY - Y ), Filling_Color );
                Part_Circle2( Y, T, D );
              End
            else
              Part_Circle3( T, D );
        Until ( X < 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Ellipse Box.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Ellipse_Box( X1, Y1, X2, Y2: Integer; X_Radius, Y_Radius: Word );
      Var
        X,
        Y: Integer;
        T: TType;
        D: DType;
      Begin
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            Fill_Line( ( X2 + X ), ( X1 - X ), ( Y2 + Y ), Filling_Color );
            Fill_Line( ( X2 + X ), ( X1 - X ), ( Y1 - Y ), Filling_Color );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Part_Circle1( X, T );
          If ( D[ 2 ] < 0 )
            then
              Begin
                Fill_Line( ( X2 + X ), ( X1 - X ), ( Y2 + Y ), Filling_Color );
                Fill_Line( ( X2 + X ), ( X1 - X ), ( Y1 - Y ), Filling_Color );
                Part_Circle2( Y, T, D );
              End
            else
              Part_Circle3( T, D );
        Until ( X < 0 );
        For Y := Y2 downto Y1 do
          Fill_Line( ( X1 - X_Radius ), ( X2 + X_Radius ), Y, Filling_Color );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Circle.
    As previously defined.

*************************************************)

    Procedure Draw_Circle( X, Y: Integer; Radius: Word );
      Begin
        Draw_Ellipse( X, Y, Radius, Round( Radius * Aspect_Ratio ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Circle.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Circle( X, Y: Integer; Radius: Word );
      Begin
        Draw_Filled_Ellipse( X, Y, Radius, Round( Radius * Aspect_Ratio ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Rounded Box.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Rounded_Box( X1, Y1, X2, Y2: Integer; Radius: Word );
      Begin
        Draw_Filled_Ellipse_Box( X1, Y1, X2, Y2, Radius, Round( Radius * Aspect_Ratio ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Rounded Box.
    As previously defined.

*************************************************)

    Procedure Draw_Rounded_Box( X1, Y1, X2, Y2: Integer; Radius: Word );
      Begin
        Draw_Ellipse_Box( X1, Y1, X2, Y2, Radius, Round( Radius * Aspect_Ratio ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Polygon.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Polygon( Amount: Word; Var Data );
      Var
        Index: Word;
        Last_Point,
        First_Point,
        Previous_Point: Point;
        Info: Points absolute Data;
      Begin
        Clear_List;
        Previous_Point := Info[ Amount ];
        First_Point := Info[ 1 ];
        For Index := 2 to Amount do
          Begin
            Last_Point := Info[ Index ];
            Filled_Line_Job( First_Point.X, First_Point.Y, Last_Point.X, Last_Point.Y );
            If ( ( ( Previous_Point.Y > First_Point.Y ) and ( First_Point.Y > Last_Point.Y ) ) or
                 ( ( Previous_Point.Y < First_Point.Y ) and ( First_Point.Y < Last_Point.Y ) ) )
              then
                Filled_Pixel_Job( First_Point.X, First_Point.Y );
            Previous_Point := First_Point;
            First_Point := Last_Point;
          End;
        First_Point := Info[ 1 ];
        Filled_Line_Job( First_Point.X, First_Point.Y, Last_Point.X, Last_Point.Y );
        Draw_List;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Polygon.
    As previously defined.

*************************************************)

    Procedure Draw_Polygon( Amount: Word; Var Data );
      Var
        Index: Word;
        Last_Point,
        First_Point: Point;
        Info: Points absolute Data;
      Begin
        Clear_Y_Coordinate_List;
        First_Point := Info[ 1 ];
        For Index := 2 to Amount do
          Begin
            Last_Point := Info[ Index ];
            Draw_Line( First_Point.X, First_Point.Y, Last_Point.X, Last_Point.Y );
            First_Point := Last_Point;
          End;
        First_Point := Info[ 1 ];
        Draw_Line( First_Point.X, First_Point.Y, Last_Point.X, Last_Point.Y );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Rectangle.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Rectangle( Left, Top, Right, Bottom: Integer );
      Begin
        Clear_Y_Coordinate_List;
        If ( Left > Right )
          then
            Swap( Left, Right );
        If ( Top < Bottom )
          then
            Swap( Top, Bottom );
        Filled_Line( Left, Top, Left, Bottom );
        Filled_Line( Right, Top, Right, Bottom );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Box.
    As previously defined.

*************************************************)

    Procedure Draw_Box( Left, Top, Right, Bottom: Integer; Depth: Word; Top_On: Boolean );
      Const
        Change_Value = 0.75;
      Var
        TopY,
        LeftX,
        RightX,
        BottomY: Integer;
      Begin
        Clear_Y_Coordinate_List;
        If ( Left > Right )
          then
            Swap( Left, Right );
        If ( Top < Bottom )
          then
            Swap( Top, Bottom );
        Filled_Line( Left, Top, Left, Bottom );
        Filled_Line( Right, Top, Right, Bottom );
        RightX := ( Right + Depth );
        TopY := ( Top + Round( Depth * Change_Value ) );
        BottomY := ( Bottom + Round( Depth * Change_Value ) );
        LeftX := ( Left + Depth );
        Draw_Line( Right, Bottom, RightX, BottomY );
        Draw_Line( RightX, BottomY, RightX, TopY );
        Draw_Line( RightX, TopY, Right, Top );
        If Top_On
          then
            Begin
              Draw_Line( RightX, TopY, LeftX, TopY );
              Draw_Line( LeftX, TopY, Left, Top );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Which Quad.
    This function is used for determining if the
    scan between the Start_Angle and the
    Finish_Angle fall inside the given section of
    the circle.  A return value of 0 means no
    overlap, a value of 1 means it starts or stops
    in that sector, and a value of 2 means it
    passes entirely through.

*************************************************)

    Function Which_Quad( Start_Angle, Finish_Angle, Quad_Start, Quad_Finish: Word; Var Start, Stop: Integer ): Byte;
      Var
        First,
        Second,
        Third,
        Ending,
        Beginning: Boolean;
      Begin
        While ( Start_Angle > 360 ) do
          Start_Angle := ( Start_Angle - 360 );
        While ( Finish_Angle > 360 ) do
          Finish_Angle := ( Finish_Angle - 360 );
        First := ( Start_Angle <= Quad_Start );
        Second := ( Finish_Angle >= Quad_Finish );
        Third := ( Start_Angle > Finish_Angle );
        If ( First and Second ) or
           ( ( Start_Angle >= Quad_Finish ) and Second and Third ) or
           ( First and ( Finish_Angle <= Quad_Start ) and Third )
          then
            Which_Quad := 2
          else
            Begin
              Beginning := ( ( Start_Angle > Quad_Start ) and ( Start_Angle < Quad_Finish ) );
              Ending := ( ( Finish_Angle > Quad_Start ) and ( Finish_Angle < Quad_Finish ) );
              If Beginning or Ending
                then
                  Begin
                    Which_Quad := 1;
                    If Beginning
                      then
                        Start := Arc_Coordinate.Start_Y_Coordinate;
                    If Ending
                      then
                        Stop := Arc_Coordinate.Finish_Y_Coordinate;
                  End
                else
                  Which_Quad := 0;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Between.
    This function returns true if the given value
    falls between Low_Limit and High_Limit.

*************************************************)

    Function Between( Low_Limit, Value, High_Limit: Integer ): Boolean;
      Begin
        If ( Low_Limit < High_Limit )
          then
            Between := ( ( Low_Limit <= Value ) and ( Value <= High_Limit ) )
          else
            Between := ( ( Low_Limit < Value ) or ( Value < High_Limit ) );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Elliptical Arc.
    As previously defined.

*************************************************)

    Procedure Draw_Elliptical_Arc( CX, CY: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );
      Var
        T: TType;
        D: DType;
        Quad1,
        Quad2,
        Quad3,
        Quad4: Byte;
        Value: Real;
        X,
        Y,
        Quad1_End,
        Quad2_End,
        Quad3_End,
        Quad4_End,
        Quad1_Start,
        Quad2_Start,
        Quad3_Start,
        Quad4_Start: Integer;
      Begin
        While ( Start_Angle > 360 ) do
          Start_Angle := ( Start_Angle - 360 );
        While End_Angle > 360 do
          End_Angle := ( End_Angle - 360 );
        If End_Angle < Start_Angle
          then
            End_Angle := ( End_Angle + 360 );
        Arc_Coordinate.Center_X_Coordinate := CX;
        Arc_Coordinate.Center_Y_Coordinate := CY;
        Value := ( Start_Angle * ( Pi / 180 ) );
        Arc_Coordinate.Start_X_Coordinate := Trunc( X_Radius * Cos( Value ) );
        Arc_Coordinate.Start_Y_Coordinate := -Trunc( Y_Radius * Sin( Value ) );
        Value := ( End_Angle * ( Pi / 180 ) );
        Arc_Coordinate.Finish_X_Coordinate := Trunc( X_Radius * Cos( Value ) );
        Arc_Coordinate.Finish_Y_Coordinate := -Trunc( Y_Radius * Sin( Value ) );
        Quad1_Start := 0;
        Quad1_End := Y_Radius;
        Quad2_Start := Y_Radius;
        Quad2_End := 0;
        Quad3_Start := 0;
        Quad3_End := -Y_Radius;
        Quad4_Start := -Y_Radius;
        Quad4_End := 0;
        Quad1 := Which_Quad( Start_Angle, End_Angle, 0, 90, Quad1_Start, Quad1_End );
        Quad2 := Which_Quad( Start_Angle, End_Angle, 90, 180, Quad2_Start, Quad2_End );
        Quad3 := Which_Quad( Start_Angle, End_Angle, 180, 270, Quad3_Start, Quad3_End );
        Quad4 := Which_Quad( Start_Angle, End_Angle, 270, 360, Quad4_Start, Quad4_End );
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            If ( Quad1 = 2 ) or ( ( Quad1 = 1 ) and Between( Quad1_End, -Y, Quad1_Start ) )
              then
                Set_Pixel( ( CX + X ), ( CY - Y ), Drawing_Color );
            If ( Quad2 = 2 ) or ( ( Quad2 = 1 ) and Between( Quad2_Start, -Y, Quad2_End ) )
              then
                Set_Pixel( ( CX - X ), ( CY - Y ), Drawing_Color );
            If ( Quad3 = 2 ) or ( ( Quad3 = 1 ) and Between( Quad3_Start, Y, Quad3_End ) )
              then
                Set_Pixel( ( CX - X ), ( CY + Y ), Drawing_Color );
            If ( Quad4 = 2 ) or ( ( Quad4 = 1 ) and Between( Quad4_End, Y, Quad4_Start ) )
              then
                Set_Pixel( ( CX + X ), ( CY + Y ), Drawing_Color );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          If ( Quad1 = 2 ) or ( ( Quad1 = 1 ) and Between( Quad1_End, -Y, Quad1_Start ) )
            then
              Set_Pixel( ( CX + X ), ( CY - Y ), Drawing_Color );
          If ( Quad2 = 2 ) or ( ( Quad2 = 1 ) and Between( Quad2_Start, -Y, Quad2_End ) )
            then
              Set_Pixel( ( CX - X ), ( CY - Y ), Drawing_Color );
          If ( Quad3 = 2 ) or ( ( Quad3 = 1 ) and Between( Quad3_Start, Y, Quad3_End ) )
            then
              Set_Pixel( ( CX - X ), ( CY + Y ), Drawing_Color );
          If ( Quad4 = 2 ) or ( ( Quad4 = 1 ) and Between( Quad4_End, Y, Quad4_Start ) )
            then
              Set_Pixel( ( CX + X ), ( CY + Y ), Drawing_Color );
          Circle_Calculate2( X, Y, T, D );
        Until ( X < 0 );
        Arc_Coordinate.Start_X_Coordinate := ( CX + Arc_Coordinate.Start_X_Coordinate );
        Arc_Coordinate.Start_Y_Coordinate := ( CY + Arc_Coordinate.Start_Y_Coordinate );
        Arc_Coordinate.Finish_X_Coordinate := ( CX + Arc_Coordinate.Finish_X_Coordinate );
        Arc_Coordinate.Finish_Y_Coordinate := ( CY + Arc_Coordinate.Finish_Y_Coordinate );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Elliptical Arc Job.
    As previously defined.

*************************************************)

    Procedure Draw_Elliptical_Arc_Job( CX, CY: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );
      Var
        T: TType;
        D: DType;
        Quad1,
        Quad2,
        Quad3,
        Quad4: Byte;
        Value: Real;
        X,
        Y,
        Quad1_End,
        Quad2_End,
        Quad3_End,
        Quad4_End,
        Quad1_Start,
        Quad2_Start,
        Quad3_Start,
        Quad4_Start: Integer;
      Begin
        While ( Start_Angle > 360 ) do
          Start_Angle := ( Start_Angle - 360 );
        While End_Angle > 360 do
          End_Angle := ( End_Angle - 360 );
        If End_Angle < Start_Angle
          then
            End_Angle := ( End_Angle + 360 );
        Arc_Coordinate.Center_X_Coordinate := CX;
        Arc_Coordinate.Center_Y_Coordinate := CY;
        Value := ( Start_Angle * ( Pi / 180 ) );
        Arc_Coordinate.Start_X_Coordinate := Trunc( X_Radius * Cos( Value ) );
        Arc_Coordinate.Start_Y_Coordinate := -Trunc( Y_Radius * Sin( Value ) );
        Value := ( End_Angle * ( Pi / 180 ) );
        Arc_Coordinate.Finish_X_Coordinate := Trunc( X_Radius * Cos( Value ) );
        Arc_Coordinate.Finish_Y_Coordinate := -Trunc( Y_Radius * Sin( Value ) );
        Quad1_Start := 0;
        Quad1_End := Y_Radius;
        Quad2_Start := Y_Radius;
        Quad2_End := 0;
        Quad3_Start := 0;
        Quad3_End := -Y_Radius;
        Quad4_Start := -Y_Radius;
        Quad4_End := 0;
        Quad1 := Which_Quad( Start_Angle, End_Angle, 0, 90, Quad1_Start, Quad1_End );
        Quad2 := Which_Quad( Start_Angle, End_Angle, 90, 180, Quad2_Start, Quad2_End );
        Quad3 := Which_Quad( Start_Angle, End_Angle, 180, 270, Quad3_Start, Quad3_End );
        Quad4 := Which_Quad( Start_Angle, End_Angle, 270, 360, Quad4_Start, Quad4_End );
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            If ( Y > 0 )
              then
                Begin
                  If ( Quad1 = 2 ) or ( ( Quad1 = 1 ) and Between( Quad1_End, -Y, Quad1_Start ) )
                    then
                      Filled_Pixel_Job( ( CX + X ), ( CY - Y ) );
                  If ( Quad3 = 2 ) or ( ( Quad3 = 1 ) and Between( Quad3_Start, Y, Quad3_End ) )
                    then
                      Filled_Pixel_Job( ( CX - X ), ( CY + Y ) );
                End;
            If ( Quad2 = 2 ) or ( ( Quad2 = 1 ) and Between( Quad2_Start, -Y, Quad2_End ) )
              then
                Filled_Pixel_Job( ( CX - X ), ( CY - Y ) );
            If ( Quad4 = 2 ) or ( ( Quad4 = 1 ) and Between( Quad4_End, Y, Quad4_Start ) )
              then
                Filled_Pixel_Job( ( CX + X ), ( CY + Y ) );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Part_Circle1( X, T );
          If ( D[ 2 ] < 0 )
            then
              Begin
                If ( Quad1 = 2 ) or ( ( Quad1 = 1 ) and Between( Quad1_End, -Y, Quad1_Start ) )
                  then
                    Filled_Pixel_Job( ( CX + X ), ( CY - Y ) );
                If ( Quad2 = 2 ) or ( ( Quad2 = 1 ) and Between( Quad2_Start, -Y, Quad2_End ) )
                  then
                    Filled_Pixel_Job( ( CX - X ), ( CY - Y ) );
                If ( Quad3 = 2 ) or ( ( Quad3 = 1 ) and Between( Quad3_Start, Y, Quad3_End ) )
                  then
                    Filled_Pixel_Job( ( CX - X ), ( CY + Y ) );
                If ( Quad4 = 2 ) or ( ( Quad4 = 1 ) and Between( Quad4_End, Y, Quad4_Start ) )
                  then
                    Filled_Pixel_Job( ( CX + X ), ( CY + Y ) );
                Part_Circle2( Y, T, D );
              End
            else
              Part_Circle3( T, D );
        Until ( X < 0 );
        Arc_Coordinate.Start_X_Coordinate := ( CX + Arc_Coordinate.Start_X_Coordinate );
        Arc_Coordinate.Start_Y_Coordinate := ( CY + Arc_Coordinate.Start_Y_Coordinate );
        Arc_Coordinate.Finish_X_Coordinate := ( CX + Arc_Coordinate.Finish_X_Coordinate );
        Arc_Coordinate.Finish_Y_Coordinate := ( CY + Arc_Coordinate.Finish_Y_Coordinate );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Filled Ellipse Job.
    As previously defined.

*************************************************)

    Procedure Draw_Filled_Ellipse_Job( CX, CY: Integer; X_Radius, Y_Radius: Word );
      Var
        X,
        Y: Integer;
        T: TType;
        D: DType;
      Begin
        Prepair_For_Circle( X, Y, X_Radius, Y_Radius, T, D );
        While ( D[ 2 ] < 0 ) do
          Begin
            If ( Y > 0 )
              then
                Begin
                  Filled_Pixel_Job( ( CX + X ), ( CY - Y ) );
                  Filled_Pixel_Job( ( CX - X ), ( CY + Y ) );
                End;
            Filled_Pixel_Job( ( CX + X ), ( CY + Y ) );
            Filled_Pixel_Job( ( CX - X ), ( CY - Y ) );
            Circle_Calculate1( X, Y, T, D );
          End;
        Repeat;
          Part_Circle1( X, T );
          If ( D[ 2 ] < 0 )
            then
              Begin
                Filled_Pixel_Job( ( CX + X ), ( CY + Y ) );
                Filled_Pixel_Job( ( CX - X ), ( CY + Y ) );
                Filled_Pixel_Job( ( CX + X ), ( CY - Y ) );
                Filled_Pixel_Job( ( CX - X ), ( CY - Y ) );
                Part_Circle2( Y, T, D );
              End
            else
              Part_Circle3( T, D );
        Until ( X < 0 );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Sector.
    As previously defined.

*************************************************)

    Procedure Draw_Sector( X1, Y1: Integer; Start_Angle, End_Angle, X_Radius, Y_Radius: Word );
      Begin
        Clear_List;
        Draw_Elliptical_Arc_Job( X1, Y1, Start_Angle, End_Angle, X_Radius, Y_Radius );
        Filled_Line_Job( Arc_Coordinate.Start_X_Coordinate, Arc_Coordinate.Start_Y_Coordinate, X1, Y1 );
        Filled_Line_Job( Arc_Coordinate.Finish_X_Coordinate, Arc_Coordinate.Finish_Y_Coordinate, X1, Y1 );
        Draw_List;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Draw Arc.
    As previously defined.

*************************************************)

    Procedure Draw_Arc( CX, CY: Integer; Start_Angle, End_Angle, Radius: Word );
      Begin
        Draw_Elliptical_Arc( CX, CY, Start_Angle, End_Angle, Radius, Round( Radius * Aspect_Ratio ) );
      End;

{-----------------------------------------------------------------------------}

  {$I Draw2.Pas }

{-----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.

*************************************************)

    Begin
      Set_Pixel := Default_Set_Pixel;
      Get_Pixel := Default_Get_Pixel;
      Fill_Line := Default_Fill_Line;
      Background_Color := 0;
      Drawing_Color := 1;
      Filling_Color := 1;
      Current_Pointer_X := 0;
      Current_Pointer_Y := 0;
    End.

