{$F+} { Compiler Directive: Generate far procedure calls: On }
{$O+} { Compiler Directive: Generate overlay code: On }

(*****************************************************************************

  TextExtras
    Version 1.0

  Purpose:
    Contains several extra feature extentions for the TextEdit unit.

  Features:
    Indent block - ^KI - indents all the lines in the block.
    Unindent block - ^KU - unindents all the lines in the block.
    Flip block - ^KF - flips all the lines in the block.
    Sort block - ^KS - sorts all the lines in the block.
    Border block - ^KZ - puts a frame around the given block.

  Limitations:
    Sorting very large blocks can overload the stack and cause the program
    to crash.

  CopyRight 1994, All rights reserved.
    By Paul Renaud.

  Compiler:
    Turbo Pascal Versions xxx to 6.0

  System:
    MS-DOS, MDOS

*****************************************************************************)

Unit TextExtras;

  Interface

    Uses
      CRT,
      Core,
      KeyBoard,
      TextLine,
      TextEdit,
      TextLink,
      String_Utilities;

(***********************************************************

  This variable constant alter the indention and unindenting
    amount.

***********************************************************)

    Const
      Indent_Amount: Byte = 1;

(***********************************************************

  These variable procedures are available for replacement by
  the main program.  They are all indulgences to the default
  procedures which allow maximum versatility, but can still
  be altered to make a more consistent and elegant user
  interface.  Study the code thoroughly before attempting to
  replace it, because it may also perform a operation that
  isn't so obvious at first.

    To replace one of these routines, define a replacement
    routine that mimics the default code in functions.

      For example...

        Procedure Replacement( parameters ); Far;
          Begin
            do something here.
          End;

    Then, somewhere in the initialization section of the
    main program, substitute your new routine for the old
    one.

      For example...

        Old_Routine := Replacement;

   That's all there is to it.  Now the new routine will
   be called in place of the old one.

***********************************************************)

    Var
     { This is the default sort verifier. }
      Verify_Sort: Function: Boolean;

     { This is the default frame pattern displayer. }
      Get_Frame_Style: Function( Var Style: Byte ): Boolean;

(***********************************************************

  Error codes passed to Write_Error.

    500 - Indent block failure.
    501 - Unindent block failure.
    502 - Flip block failure.
    503 - Sort block failure.
    504 - Frame block failure.

***********************************************************)

{----------------------------------------------------------------------------}

  Implementation

    Var
     { Used to link the new help routine to the old one. }
      Old_Help: Procedure;
     { Used to link the routines to the editing system. }
      Old_K_Link: Procedure( Var All: All_Type; Data: Char; Var Start, Finish: Point_Type; Var Done: Boolean );

{----------------------------------------------------------------------------}

(*************************************************

  Function: Flip the block.

    This function attempts to flip the block in
    Text marked off by Start and Finish.  It
    returns true only if it's successful.

*************************************************)

    Function Flip_Block( Var All: All_Type; Start, Finish: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Limit,
        Point: LongInt;
      Begin
        Okay := Valid_Block_Operation( Start, Finish, Finish );
        If Show_Status
          then
            Begin
              Limit := ( ( Finish.Row - Start.Row ) div 2 );
              If ( Limit < 1 )
                then
                  Limit := 1;
              Point := 1;
            End;
        While Okay and ( Start.Row < Finish.Row ) do
          Begin
            Okay := Swap_Line( All.Text, Start.Row, Finish.Row );
            Dec( Finish.Row );
            Inc( Start.Row );
            If ( Okay and Show_Status )
              then
                Begin
                  If Odd( Point )
                    then
                      Process_Status( Point, Limit, Okay );
                  Inc( Point );
                End;
          End;
        Flip_Block := Okay;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Compare low.
    This function returns true if the given row
    in Text is less than the given text line.

*************************************************)

    Function Compare_Low( Var All: All_Type; Row: LongInt; Var Data: Line_Type ): Boolean;
      Begin
        Look_Line( All.Text, Row, Other_Buffer^ );
        Compare_Low := Less_Then( Other_Buffer^, Data );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Compare high.
    This function returns true if the given row in
    Text is greater than the given text row.

*************************************************)

    Function Compare_High( Var All: All_Type; Row: LongInt; Var Data: Line_Type ): Boolean;
      Begin
        Get_Line( All.Text, Row, Other_Buffer^ );
        Compare_High := Less_Then( Data, Other_Buffer^ );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Quick Sort.
    This function attempts to sort the given block
    of lines in Text defined by Low and High.
    It uses the quick sort recursion method to
    quickly process the data.

*************************************************)

    Function QSort( Var All: All_Type; Var Other: Line_Type; Low, High: LongInt; Var Point, Limit: LongInt ): Boolean;
      Var
        Okay: Boolean;
        New_Low,
        New_High: LongInt;
      Begin
        Okay := True;
        New_Low := Low;
        New_High := High;
        If Show_Status
          then
            Begin
              If Odd( Point )
                then
                  Process_Status( Point, Limit, Okay );
              Inc( Point );
            End;
        Get_Line( All.Text, ( ( Low + High ) div 2 ), Other );
        Repeat
          While ( Compare_Low( All, New_Low, Other ) and ( New_Low < High ) ) do
            Inc( New_Low );
          While ( Compare_High( All, New_High, Other ) and ( New_High > Low ) ) do
            Dec( New_High );
          If ( New_Low <= New_High )
            then
              Begin
                Okay := Swap_Line( All.Text, New_Low, New_High );
                If Okay
                  then
                    Begin
                      Inc( New_Low );
                      Dec( New_High );
                    End
              End;
        Until ( ( New_Low > New_High ) or ( not Okay ) );
        If ( Okay and ( Low < New_High ) and ( New_High <> High ) )
          then
            Okay := QSort( All, Other, Low, New_High, Point, Limit );
        If ( Okay and ( New_Low < High ) and ( New_Low <> Low ) )
          then
            Okay := QSort( All, Other, New_Low, High, Point, Limit );
        QSort := Okay;
      End;

{--------------------------------------------------------------------------------}

(*************************************************

  Function Sort the block.
    This function attempts to sort the lines of
    the block in text that is marked off by
    Start and Finish.  It uses the quick sort
    method and returns true only if it's
    successful.  Because this method uses
    recursion, it is possible, although unlikely,
    that a stack overflow can occur with large
    blocks.

*************************************************)

    Function Sort_Block( Var All: All_Type; Var Other: Line_Type; Start, Finish: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Point,
        Limit: LongInt;
      Begin
        Point := 0;
        Limit := Pred( Finish.Row - Start.Row );
        If ( Limit < 1 )
          then
            Limit := 1;
        Okay := Valid_Block_Operation( Start, Finish, Finish );
        If Okay
          then
            Okay := QSort( All, Other, Start.Row, Finish.Row, Point, Limit );
        Sort_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Indent block.
    This function attempts to indent the block in
    Text marked off by Start and Finish.  It
    returns true only if it is successful, other-
    wise, only part of the block will have been
    indented.

*************************************************)

    Function Indent_Block( Var All: All_Type; Var Start, Finish: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Index: Byte;
        Count,
        Point,
        Limit: LongInt;
      Begin
        Okay := Valid_Block_Operation( Start, Finish, Finish );
        If Okay
          then
            Begin
              If Show_Status
                then
                  Begin
                    Limit := ( Finish.Row - Start.Row );
                    If ( Limit < 1 )
                      then
                        Limit := 1;
                    Point := 1;
                  End;
              For Count := Start.Row to Finish.Row do
                If Okay
                  then
                    Begin
                      Get_Line( All.Text, Count, Other_Buffer^ );
                      For Index := 1 to Indent_Amount do
                        Insert_Line( Other_Buffer^, 1, ' ' );
                      Okay := Put_Line( All.Text, Count, Other_Buffer^, True );
                      If ( Okay and Show_Status )
                        then
                          Begin
                            If Odd( Point )
                              then
                                Process_Status( Point, Limit, Okay );
                            Inc( Point );
                          End;
                    End;
            End;
        If Okay
          then
            Inc( Finish.Column );
        Indent_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Unindent block.
    This function attempts to unindent the block
    in Text marked off by Start and Finish.  It
    returns true only if it is successful, other-
    wise, only part of the block will have been
    unindented.  A line must have a preceding
    blank space to be affected.

*************************************************)

    Function UnIndent_Block( Var All: All_Type; Var Start, Finish: Point_Type ): Boolean;
      Var
        Okay: Boolean;
        Index: Byte;
        Count,
        Limit,
        Point: LongInt;
      Begin
        Okay := Valid_Block_Operation( Start, Finish, Finish );
        If Okay
          then
            Begin
              If Show_Status
                then
                  Begin
                    Limit := ( Finish.Row - Start.Row );
                    If ( Limit < 1 )
                      then
                        Limit := 1;
                    Point := 1;
                  End;
              For Count := Start.Row to Finish.Row do
                If Okay
                  then
                    Begin
                      Get_Line( All.Text, Count, Other_Buffer^ );
                      If ( Other_Buffer^.Size > 0 ) and ( Other_Buffer^.Data[ 1 ] = ' ' )
                        then
                          Begin
                            For Index := 1 to Indent_Amount do
                              If ( ( Other_Buffer^.Size > 0 ) and ( Other_Buffer^.Data[ 1 ] = ' ' ) )
                                then
                                  Delete_Line( Other_Buffer^, 1, 1 );
                            Okay := Put_Line( All.Text, Count, Other_Buffer^, True );
                          End;
                      If ( Okay and Show_Status )
                        then
                          Begin
                            If Odd( Point )
                              then
                                Process_Status( Point, Limit, Okay );
                            Inc( Point );
                          End;
                    End;
            End;
        If Okay
          then
            Dec( Finish.Column );
        UnIndent_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Make block.
    This function attempts to draw a block around
    the current block markers.  The block
    beginning marks the top left corner and the
    block ending marks the bottom right corner
    of the block.  Spaces are added if the line
    needs them, but the text is pushed off to the
    side or left in the center.

*************************************************)

    Function Make_Block( Var All: All_Type; Var Line: Line_Type; Start, Finish: Point_Type; Var Frame: Frame_Type ): Boolean;
      Var
        Okay: Boolean;
        Point: Word;
        Count: LongInt;
        Limit,
        Pointer: LongInt;
      Begin
        Okay := ( ( Start.Row < Finish.Row ) and ( Start.Column < Pred( Finish.Column ) ) );
        If Okay
          then
            Begin
              If Show_Status
                then
                  Begin
                    Limit := Pred( Finish.Row - Start.Row );
                    If ( Limit < 1 )
                      then
                        Limit := 1;
                    Pointer := 1;
                  End;
              { Draw top bar. }
              Get_Line( All.Text, Start.Row, Line );
              For Point := Line.Size to Start.Column do
                Append_Line( Line, ' ' );
              Insert_Line( Line, Start.Column, Frame.Data[ 1 ] );
              For Point := Succ( Start.Column ) to Pred( Pred( Finish.Column ) ) do
                Insert_Line( Line, Point, Frame.Data[ 2 ] );
              Insert_Line( Line, Pred( Finish.Column ), Frame.Data[ 3 ] );
              Okay := Put_Line( All.Text, Start.Row, Line, True );
              { Draw side bars. }
              If Okay
                then
                  For Count := Succ( Start.Row ) to Pred( Finish.Row ) do
                    If Okay
                      then
                        Begin
                          Get_Line( All.Text, Count, Line );
                          For Point := Line.Size to Start.Column do
                            Append_Line( Line, ' ' );
                          Insert_Line( Line, Start.Column, Frame.Data[ 4 ] );
                          For Point := Line.Size to Finish.Column do
                            Append_Line( Line, ' ' );
                          Insert_Line( Line, Pred( Finish.Column ), Frame.Data[ 6 ] );
                          Okay := Put_Line( All.Text, Count, Line, True );
                          If ( Okay and Show_Status )
                            then
                              Begin
                                If Odd( Pointer )
                                  then
                                    Process_Status( Pointer, Limit, Okay );
                                Inc( Pointer );
                              End;
                        End;
              { Draw bottom bar. }
              If Okay
                then
                  Begin
                    Get_Line( All.Text, Finish.Row, Line );
                    For Point := Line.Size to Start.Column do
                      Append_Line( Line, ' ' );
                    Insert_Line( Line, Start.Column, Frame.Data[ 7 ] );
                    For Point := Succ( Start.Column ) to Pred( Pred( Finish.Column ) ) do
                      Insert_Line( Line, Point, Frame.Data[ 8 ] );
                    Insert_Line( Line, Pred( Finish.Column ), Frame.Data[ 9 ] );
                    Okay := Put_Line( All.Text, Finish.Row, Line, True );
                  End;
            End;
        Make_Block := Okay;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Indent the block
    This procedure tries to move the entire block
    of text one space to the right.  If it fails,
    it displays the error message.

*************************************************)

    Procedure Indent_The_Block( Var All: All_Type; Var Start, Finish: Point_Type );
      Begin
        Write_Wait;
        If not Indent_Block( All, Start, Finish )
          then
            Write_Error( 500 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: UnIndent the block
    This procedure tries to move the entire block
    of text one space to the left.  If it fails,
    it displays the error message.

*************************************************)

    Procedure UnIndent_The_Block( Var All: All_Type; Var Start, Finish: Point_Type );
      Begin
        Write_Wait;
        If not UnIndent_Block( All, Start, Finish )
          then
            Write_Error( 501 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Flip the block
    This procedure reverses the order of the lines
    of text in the block so that the last one is
    first and the first is last.  If it fails, it
    displays the error message.

*************************************************)

    Procedure Flip_The_Block( Var Text: All_Type; Var Start, Finish: Point_Type );
      Begin
        Write_Wait;
        If not Flip_Block( Text, Start, Finish )
          then
            Write_Error( 502 );
        Write_Complete;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Sort the block
    This procedure alters the order of the lines
    of text in the block so that they are sorted
    in alphabetical order.  If it fails, it
     displays the error message.

*************************************************)

    Procedure Sort_The_Block( Var All: All_Type; Var Start, Finish: Point_Type );
      Begin
        If Verify_Sort
          then
            Begin
              Write_Wait;
              If not Sort_Block( All, Working_Buffer^, Start, Finish )
                then
                  Write_Error( 503 );
              Write_Complete;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: Make the block.
    This procedure tries to create a frame around
    the current block in the text editor.  A list
    of options is introduced from which the user
    can easily select a frame to use.

*************************************************)

    Procedure Make_The_Block( Var All: All_Type; Var Start, Finish: Point_Type );
      Var
        Frame: Frame_Type;
        Style: Byte;
      Begin
        If Get_Frame_Style( Style )
          then
            Begin
              Set_Window_Frame( Frame, 0, Style );
              Write_Wait;
              If ( not Make_Block( All, Working_Buffer^, Start, Finish, Frame ) )
                then
                  Write_Error( 504 );
              Write_Complete;
            End;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: K link.
    This procedure links into the control K
    interface to add the new functions to the
    TextEdit editing routine.

*************************************************)

    Procedure K_Link( Var All: All_Type; Character: Char; Var Block_Start, Block_Finish: Point_Type; Var Done: Boolean ); Far;
      Begin
        Case Character of
          'I': Indent_The_Block( All, Block_Start, Block_Finish );
          'U': UnIndent_The_Block( All, Block_Start, Block_Finish );
          'F': Flip_The_Block( All, Block_Start, Block_Finish );
          'S': Sort_The_Block( All, Block_Start, Block_Finish );
          'Z': Make_The_Block( All, Block_Start, Block_Finish );
          else Old_K_Link( All, Character, Block_Start, Block_Finish, Done );
        End; { Case }
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Get frame style default.
    This function presents the different frame
    styles available to the user and allows one
    to be selected.

*************************************************)

    Function Get_Frame_Style_Default( Var Style: Byte ): Boolean;
      Const
        High_Stack = 4;
        Wide_Stack = 6;
      Var
        Top,
        Left,
        Right,
        Bottom,
        Width,
        Height,
        Count1,
        Count2,
        Start_Top: Byte;
        Frame: Frame_Type;
      Begin
        Width := Pred( Right_Of_Window^ - Left_Of_Window^ );
        ClrScr;
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, Center( 'Frame the block', Width, ' ' ) );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, Center( 'Please select a frame pattern to use', Width, ' ' ) );
        TextAttr := Message_Normal;
        WriteLn( Screen );
        TextAttr := Message_HighLight;
        Style := 1;
        Height := Pred( Bottom_Of_Window^ - WhereY );
        Start_Top := WhereY;
        For Count1 := 1 to High_Stack do
          Begin
            Top := Start_Top + ( Pred( Count1 ) * ( Height div High_Stack ) );
            Bottom := Pred( Start_Top + ( Count1 * ( Height div High_Stack ) ) );
            For Count2 := 1 to Wide_Stack do
              Begin
                Left := Succ( Pred( Count2 ) * ( Width div Wide_Stack ) );
                Right := Pred( Count2 * ( Width div Wide_Stack ) );
                Set_Window_Frame( Frame, Message_Normal, Style );
                Draw_Window_Frame( Frame, Left, Top, Right, Bottom );
                GotoXY( Pred( ( Right + Left ) div 2 ), ( ( Bottom + Top ) div 2 ) );
                Write( Screen, Style:2 );
                Inc( Style );
              End;
          End;
        Repeat
          GotoXY( 1, Pred( Bottom_Of_Window^ ) );
          TextAttr := Message_HighLight;
          Write( Screen, 'Enter choice number ( 0 to exit ): ' );
          TextAttr := Message_Normal;
          {$I-}
          ReadLn( Keys, Style )
          {$I+}
        Until ( IoResult = 0 );
        Get_Frame_Style_Default := ( Style < 25 ) and ( Style > 0 );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Function: Verify sort default.
    This function allows the user to verify that
    the block is to be sorted.  It allows a route
    of escape since sorting can destroy the order
    of text if it is accidentally activated.

*************************************************)

    Function Verify_Sort_Default: Boolean;
      Begin
        GotoXY( 1, 1 );
        TextAttr := Message_HighLight;
        Write( Screen, 'Sort the block? ' );
        Verify_Sort_Default := ( Get_Answer = 'Y' );
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Procedure: New help.
    This function adds a new page to the help
    routine explaining the new functions.

*************************************************)

    Procedure New_Help; Far;
      Var
        Width: Byte;
      Begin
        Old_Help;
        Width := Pred( Right_Of_Window^ - Left_Of_Window^ );
        ClrScr;
        TextAttr := Message_HighLight;
        WriteLn( Screen, Center( 'Help Screen', Width, ' ' ) );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, Center( 'Extra block functions', Width, ' ' ) );
        TextAttr := Message_Normal;
        WriteLn( Screen, Expand( '-', Width ) );
        TextAttr := Message_Normal;
        Write( Screen, Push_To( 'Flip block', '^KF', 37, '.' ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '^KF' );
        TextAttr := Message_Normal;
        Write( Screen, Push_To( 'Sort block lines', '^KS', 37, '.' ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '^KS' );
        TextAttr := Message_Normal;
        Write( Screen, Push_To( 'Indent block', '^KI', 37, '.' ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '^KI' );
        TextAttr := Message_Normal;
        Write( Screen, Push_To( 'Unindent block', '^KU', 37, '.' ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '^KU' );
        TextAttr := Message_Normal;
        Write( Screen, Push_To( 'Border block', '^KZ', 37, '.' ) );
        TextAttr := Message_HighLight;
        WriteLn( Screen, '^KZ' );
        TextAttr := Message_Normal;
        WriteLn( Screen );
        Pause_For_Key;
      End;

{----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    This section initializes the variable
    procedures and functions.

*************************************************)

  Begin
    Old_K_Link := Control_K_Link;
    Control_K_Link := K_Link;
    Old_Help := Help;
    Help := New_Help;
    Verify_Sort := Verify_Sort_Default;
    Get_Frame_Style := Get_Frame_Style_Default;
  End.

