{----------------------------------------------------------}
{      CCLOCK.PAS - TClock Delphi visual component         }
{    by Michael Kochiashvili   Kochini@iberiapac.ge        }
{----------------------------------------------------------}
{ This unit is public domain. You can add/rewrite code, but}
{if you do it, please, let me know. If you have any        }
{suggestions for improvements, or if you find any bugs,    }
{please notify the author                                  }
{----------------------------------------------------------}
unit CClock;

interface
uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  ExtCtrls;

type
  TClock = class(TGraphicControl)
  private
    Ticker : TTimer; { }

    FPen: TPen;          { Arrows pen}
    FBitMap : TBitMap;   { Background BitMap}
    { OnSecond, OnMinute, OnHour events}
    FOnSecond, FOnMinute, FOnHour : TNotifyEvent;

    { Clock variables}
    CenterPoint : TPoint; { Center of arrows}
    Radius : integer;     { Radius of clock}
    LapStepW : integer;   { Width of lap steps rectangles}
    PrevTime : TDateTime;
    ShowSecond : boolean; { Show or not second arrow}
    FArrowColor : TColor; { Arrows color}
    FFaceColor : TColor;  { Clock face color}
    CHourDiff, CMinDiff : integer; { Hour & Min difference between system time}

    procedure SetFaceColor( Value : TColor);
    procedure SetArrowColor( Value : TColor);
    procedure SetShowSecond( Value : boolean);
    procedure SetHourDiff( Value : integer);
    procedure SetMinDiff( Value : integer);


    function MinuteAngle( Min : word) : real;    { Minute arrow angle}
    function HourAngle( Hou, Min : word) : real; { Hour arrow angle}
    procedure CalcClockSettings;
    procedure DrawClockBkg; { Draw clock background on FBitMap}
    procedure DrawArrows; { Draw clock arrows}
    { Time difference ( GTime - DTime) }
    procedure TimeDiff( GTime, DTime : TDateTime; var dHour, dMin : integer);
    procedure DecodeCTime( CTime : TDateTime; var H, M, S : word); { Decode Clk. time}

  protected
    procedure Paint; override;
    procedure TickerCall(Sender: TObject); { Timer event}

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure ClkTime( var H, M, S : word); { Get clock time}
    procedure SetClkTime( CTime : TDateTime); { Set clock time}

  published
    property Align;
    property Enabled;
    property ParentShowHint;
    property ShowHint;
    property Visible;
    {}
    property ClkArrowColor : TColor read FArrowColor write SetArrowColor default clBlack;
    property ClkFaceColor : TColor read FFaceColor write SetFaceColor default clBtnFace;
    property DifHour : integer read CHourDiff write SetHourDiff default 0;
    property DifMinute : integer read CMinDiff write SetMinDiff default 0;
    property SecArrow : boolean read ShowSecond write SetShowSecond;
    property OnSecond : TNotifyEvent read FOnSecond write FOnSecond;
    property OnMinute : TNotifyEvent read FOnMinute write FOnMinute;
    property OnHour   : TNotifyEvent read FOnHour   write FOnHour;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents( 'Samples', [TClock]);
end;

const
  SecScale = 1;      SecThick = 1;
  MinScale = 0.95;   MinThick = 3;
  HouScale = 0.60;   HouThick = 6;

constructor TClock.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 66;  Height := 66;

  { Initialize private variables}

  FFaceColor := clBtnFace;
  FArrowColor := clBlack;

  PrevTime := 0;
  ShowSecond := true;
  CHourDiff := 0;
  CMinDiff := 0;

  { Create Pen}
  FPen := TPen.Create;
  { Create background BitMap}
  FBitMap := TBitMap.Create;
  FBitMap.Width := Width;
  FBitMap.Height := Height;
  { Create and set timer}
  Ticker := TTimer.Create( Self);
  Ticker.Interval := 1000;
  Ticker.OnTimer := TickerCall;
  Ticker.Enabled := true;
end;

procedure TClock.SetFaceColor( Value : TColor);
begin FFaceColor := Value; invalidate; end;

procedure TClock.SetArrowColor( Value : TColor);
begin FArrowColor := Value; invalidate; end;

procedure TClock.SetShowSecond( Value : boolean);
begin
  ShowSecond := Value;
  PrevTime := 0;
  Invalidate;
end;

procedure TClock.SetHourDiff( Value : integer);
begin
  CHourDiff := Value MOD 12;
  DrawArrows;
end;

procedure TClock.SetMinDiff( Value : integer);
begin
  CMinDiff := Value MOD 60;
  DrawArrows;
end;


procedure TClock.Paint;
begin
  CalcClockSettings;
  DrawClockBkg;
  PrevTime := 0;
  DrawArrows;
end;

destructor TClock.Destroy;
begin
  FPen.Free;
  FBitMap.Free;
  Ticker.Free;
  inherited Destroy;
end;

procedure TClock.TimeDiff( GTime, DTime : TDateTime; var dHour, dMin : integer);
var MinDiff : integer; GTMin, DTMin : integer;

  function Time2Min( Tim : TDateTime) : integer;
    var cH, cM, cS, cmS : word;
  begin
    DecodeTime( Tim, cH, cM, cS, cmS);  Result := cH * 60 + cM;
  end;

begin
  GTMin := Time2Min( GTime);
  DTMin := Time2Min( DTime);
  MinDiff :=  GTMin - DTMin;
  dHour := MinDiff DIV 60;
  dMin  := MinDiff MOD 60;
end;

{ Decode current clock time}
procedure TClock.DecodeCTime( CTime : TDateTime; var H, M, S : word);
var
  cH, cM, cS, cmS : word;
  SysMinAft0, TotMinDiff, ClkMinAft0 : integer;
begin
  DecodeTime( CTime, cH, cM, cS, cmS);
  SysMinAft0 := cH * 60 + cM;
  TotMinDiff := CHourDiff * 60 + CMinDiff;
  ClkMinAft0 := SysMinAft0 + TotMinDiff;
  if ClkMinAft0 < 0 then ClkMinAft0 := 24 * 60 + ClkMinAft0;
  H := ClkMinAft0 DIV 60;
  M := ClkMinAft0 MOD 60;
  S := cS;
end;

procedure TClock.ClkTime( var H, M, S : word); { Get clock time}
begin
  DecodeCTime( Time, H, M, S);
end;

{ Set clock time. Seconds are set to system clock}
procedure TClock.SetClkTime( CTime : TDateTime);
begin
  TimeDiff( CTime, Time, CHourDiff, CMinDiff);
  invalidate;
end;

function TClock.MinuteAngle( Min : word) : real;
begin
  MinuteAngle := Min * 2 * Pi / 60;
end; { MinuteAngle}

function TClock.HourAngle( Hou, Min : word) : real;
begin
 HourAngle := ( Hou MOD 12) * 2 * Pi / 12 + MinuteAngle( Min) / 12;
end; { HourAngle}

procedure TClock.TickerCall(Sender: TObject);
var
  H, M, S, pH, pM, pS : word;
begin
  { Don't do enithing in design time}
  if csDesigning in ComponentState then exit;

  DecodeCTime( Time, H, M, S);
  DecodeCTime( PrevTime, pH, pM, pS);

  { Handle published timer events}
  if Assigned( FOnSecond) then FOnSecond( Self); { OnSecond event}
  if Assigned( FOnMinute) AND ( pS > S) then FOnMinute( Self); { OnMinute event}
  if Assigned( FOnHour) AND ( pM > M) then FOnHour( Self); { OnHour event}
  PrevTime := Time;

  { Don't draw arrows if minute not changed when second arrow is disabled}
  if ( NOT ShowSecond) AND ( pS <= S) then exit;

  { Draw clock arrows}
  DrawArrows; {}
end;

procedure TClock.DrawArrows;
var
  H, M, S : word;
  CurPoint : TPoint;
  CTime : TDateTime;
  ABitMap : TBitMap;

  procedure DrawArrow( Angle, Scale : real; AWidth : integer);
  var SR : real;
  begin
    with ABitMap.Canvas do begin
      Pen.Width := AWidth;
      MoveTo( CenterPoint.X, CenterPoint.Y);
      SR := Scale *  Radius;
      LineTo( trunc(  SR * sin( Angle)) + CenterPoint.X,
              trunc( -SR * cos( Angle)) + CenterPoint.Y);
      end;
  end;

begin
  { Create offscreen bitmap}
  ABitMap := TBitMap.Create;
  FPen.Color := ClkArrowColor;

  try { draw arrows on offscreen image}
    { Set offscreen bitmap attributes}
    ABitMap.Width := Width;
    ABitMap.Height := Height;
    with ABitMap.Canvas do begin
      Pen := FPen;
      Brush.Color := ClkFaceColor;
      end;
    { Copy background bitmap in to offscreen bitmap}
    ABitMap.Canvas.CopyMode := cmSrcCopy;
    ABitMap.Canvas.CopyRect(
        ABitMap.Canvas.ClipRect,
        FBitMap.Canvas,
        FBitMap.Canvas.ClipRect
        );
    { Draw new arrows}
    DecodeCTime( Time, H, M, S);
    if ShowSecond then
      DrawArrow( MinuteAngle( S),  SecScale, SecThick); { second}
    DrawArrow( MinuteAngle( M),  MinScale, MinThick);   { minute}
    DrawArrow( HourAngle( H, M), HouScale, HouThick);   { hour}

    { Draw offscreen image on clock face}
    Canvas.CopyMode := cmSrcCopy;
    Canvas.Draw( 0, 0, ABitMap);
  finally
    ABitMap.Free;
    end;
end; { DrawArrows}

procedure TClock.CalcClockSettings;
begin
  {Create new background image}
  FBitMap.Free;
  FBitMap := TBitMap.Create;
  FBitMap.Width := Width;
  FBitMap.Height := Height;
  { Calc Center of clock arrows}
  CenterPoint := Point( Width DIV 2, Height DIV 2 );
  { Calc Radius of clock}
  with CenterPoint do
    if X <= Y then Radius := X
    else           Radius := Y;

  LapStepW := Radius DIV 8;
  if LapStepW < 6 then LapStepW := 6;

  dec( Radius, LapStepW + 2);
end; { CalcClockSettings}

{ Draw clock background on FBitMap}
procedure TClock.DrawClockBkg;

  procedure Draw3dRect( ARect : TRect);
  begin
    Frame3D( FBitMap.Canvas, ARect, clBtnHighlight, clBtnShadow, 1);
  end; { Draw3dRect}

  { Draw minute points on FBitMap}
  procedure DrawMinSteps;
  var
    CPen1, CPen2 : TPen;
    OfsX, OfsY : integer;
    MinCou : word;
    CurPt : TPoint;
    TmpRect : TRect;
    SR, Ang : real;
  begin
    OfsX := LapStepW DIV 2; OfsY := OfsX;
    MinCou := 0;
    while MinCou < 56 do begin
      SR := Radius + OfsX;
      Ang := MinuteAngle( MinCou);
      CurPt := Point(
                 trunc(  SR * sin( Ang)) + CenterPoint.X,
                 trunc( -SR * cos( Ang)) + CenterPoint.Y);
      if MinCou MOD 15 = 0 then
        TmpRect := Rect( CurPt.X - OfsX, CurPt.Y - OfsY,
                         CurPt.X + OfsX, CurPt.Y + OfsY)
      else
        TmpRect := Rect( CurPt.X - 2, CurPt.Y - 2,
                         CurPt.X + 2, CurPt.Y + 2);

      Draw3dRect( TmpRect);

      inc( MinCou, 5);
      end; { while MinCou < 56}
  end; { DrawMinSteps}

begin
  with FBitMap.Canvas do begin
    Brush.Style := bsSolid;
    Brush.Color := ClkFaceColor;
    FillRect( ClipRect);
    end;
  DrawMinSteps;
end; { DrawClockBkg}

end.
