/*

   Sample Converter Library for Apple II Oasis :: Disk (image) Manager
   -------------------------------------------------------------------
                   Copyright (c) BG Universal Software, Inc. 1997-2002


   mailto:apl24win@yahoo.com
   http://www.geocities.com/apl24win/

   NOTE: This is 16-bit DLL
*/

#ifdef __cplusplus
#error Disable _C++_ compiler option!
#endif

#define    STRICT
#define    WIN31
#include   <windows.h>
#include   <stdio.h>
#include   <share.h>
#include   <mem.h>
#include   "adm_conv.h"
#include   "conv_lib.h"

HINSTANCE  hInst;

char       szSecADM_Conv[]   = "ADM_Conv.DLL";  // Section in ADMANAGR.INI
char       szEntDiskName[]   = "DiskName";    
char       szEntAlways[]     = "Always";
char       szEntAutoRename[] = "AutoRename";

char       szSecApple[]      = "Apple II";  // See .A4W file format
char       szSecOptions[]    = "Options";
char       szEntType[]       = "Type";    
char       szEntFormat[]     = "Format";
char       szTypeDisk2[]     = "Disk II";
char       szTypeHDisk[]     = "Fixed Disk";

LPSTR String( UINT uID )
{
  static char sz[256];

  LoadString( hInst, uID, sz, sizeof sz );

  return sz;
}

BOOL IsTYP( LPSTR filename, char ** szTYP )
{
  int i, j;

  for ( j = lstrlen( filename ); filename[j] != '.'; j-- )
    if ( j == 0 ) return FALSE;

  for ( i = 0; *szTYP[i] != '*'; i++ )
    if ( !lstrcmpi( filename + j, szTYP[i] ) )
      return TRUE;

  return FALSE;
}

BOOL IsTXT( LPSTR filename )
{
  static char * szTXT[] = { ".TXT", ".ASC", "*" };

  return IsTYP( filename, szTXT );
}

BOOL IsA4W( LPSTR filename )
{
  static char * szA4W[] = { ".A4W", "*" };

  return IsTYP( filename, szA4W );
}

BOOL IsDSK( LPSTR filename )
{
  static char * szDSK[] = { ".DSK", ".DO", "*" };

  return IsTYP( filename, szDSK );
}

BOOL IsPO( LPSTR filename )
{
  static char * szPO[] = { ".PO", "*" };

  return IsTYP( filename, szPO );
}

BOOL IsHDV( LPSTR filename )
{
  static char * szHDV[] = { ".HDV", "*" };

  return IsTYP( filename, szHDV );
}

// Similar function to GetPrivateProfileString
// Differences:
//   (1) all features are not supported;
//   (2) does not work automatically in WINDOWS directory;
//   (3) uses file sharing flags (_fsopen);

int _GetPrivateProfileString( LPSTR szSection, LPSTR szEntry, LPSTR szDefault, LPSTR szRetBuf, UINT cbRetBuf, LPSTR szFile )
{
  FILE FAR * fp;
  BOOL       fSectionIn;
  int        c, i, n;
  char       sz[144];

  cbRetBuf--;
  AnsiToOem( szFile, sz );
  fp         = _fsopen( sz, "rt", SH_DENYNO );
  n          = -2;
  fSectionIn = FALSE;
  if ( fp )
  do
  {
    LPSTR p1, p2;
    for ( i = 0; i < sizeof sz - 1 && ( c = getc( fp ) ) != EOF && c != 10; sz[i++] = (char)c );
    sz[i] = 0;
    p1 = sz;
    while ( *p1 == 32 || *p1 == 9 ) p1++;
    p2 = p1;
    if ( *p1 == '[' )
    {
      while ( *p2 && *p2 != ']' ) p2++;
      *p2 = 0;
      fSectionIn = !lstrcmpi( p1 + 1, szSection );
    }
    else
      if ( fSectionIn )
      {
        while ( *p2 && *p2 != '=' ) p2++;
        *p2 = 0;
        if ( !lstrcmpi( p1, szEntry ) )
        {                              
          for ( i = 0; i < cbRetBuf && ( szRetBuf[i] = *++p2 ) != 0; i++ );
          szRetBuf[n = i] = 0;
        }
      }
  }
  while ( n == -2 && c != EOF );

  if ( n == -2 )
  {
    LPSTR p = szDefault;

    for ( i = 0; i < cbRetBuf && ( szRetBuf[i] = *p++ ) != 0; i++ );
    szRetBuf[n = i] = 0;
  }
  fclose( fp );

  return n;
}

WINAPI LibMain( HINSTANCE hInstance, WORD wDataSeg, WORD cbHeapSize, LPSTR lpCmdLine )
{
  hInst = hInstance;
  if ( cbHeapSize != 0 ) UnlockData( 0 );

  return 1;
}

// About Dialog
BOOL CALLBACK _export InfoDlg( HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{
  switch ( msg )
  {
    case WM_INITDIALOG:
    break;

    case WM_COMMAND:
    if ( wParam == IDOK || wParam == IDCANCEL )
      EndDialog( hwndDlg, 0 );
    break;

    default: return FALSE;
  }
  return TRUE;
}

// Setup Dialog Box for Disk Image converter
BOOL CALLBACK _export SetupDlg( HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{    
  char sz[64];

  switch ( msg )
  {
    case WM_INITDIALOG:
    // All optional settings are stored in ADMANAGR.INI file / ADM_Conv.DLL section
    GetPrivateProfileString( szSecADM_Conv, szEntDiskName, "", sz, sizeof sz, String( IDS_PROFILE ) );
    SetDlgItemText( hwndDlg, IDD_DISKNAME, sz );
    CheckDlgButton( hwndDlg, IDD_CBALWAYS, GetPrivateProfileInt( szSecADM_Conv, szEntAlways, 0, String( IDS_PROFILE ) ) );
    break;

    case WM_COMMAND:
    switch ( wParam )
    {
      case IDOK:
      {
        int a = IsDlgButtonChecked( hwndDlg, IDD_CBALWAYS );
        GetDlgItemText( hwndDlg, IDD_DISKNAME, sz, sizeof sz );
        WritePrivateProfileString( szSecADM_Conv, szEntDiskName, sz, String( IDS_PROFILE ) );
        WritePrivateProfileString( szSecADM_Conv, szEntAlways, a ? "1" : "0", String( IDS_PROFILE ) );
        EndDialog( hwndDlg, a );
      }
      break;

      case IDCANCEL:
      EndDialog( hwndDlg, -1 );
      break;

      case IDD_PBHELP:
      {
        char sz[256];
        int  i;

        GetModuleFileName( hInst, sz, sizeof sz ); // get ADM_CONV.DLL's directory
        for ( i = lstrlen( sz ); i; i-- )             // ADMANAGR.HLP/EXE & ADM_CONV.DLL
          if ( sz[i] == '\\' )                        // must be in the same directory
          {
            sz[i + 1] = 0;
            break;
          }
        lstrcat( sz, String( IDS_HELPFILE ) );

        WinHelp( hwndDlg, sz, HELP_CONTEXT, HCS_DICONV_SETUP );
      }
      break;
    }
    break;

    default: return FALSE;
  }
  return TRUE;
}

// Setup Dialog Box for .PO Disk Image converter
BOOL CALLBACK _export Setup2Dlg( HWND hwndDlg, UINT msg, WPARAM wParam, LPARAM lParam )
{    
  switch ( msg )
  {
    case WM_INITDIALOG:
    CheckDlgButton( hwndDlg, IDD_CBAUTORENAME, GetPrivateProfileInt( szSecADM_Conv, szEntAutoRename, 0, String( IDS_PROFILE ) ) );
    break;

    case WM_COMMAND:
    switch ( wParam )
    {
      case IDOK:
      {
        int a = IsDlgButtonChecked( hwndDlg, IDD_CBAUTORENAME );
        WritePrivateProfileString( szSecADM_Conv, szEntAutoRename, a ? "1" : "0", String( IDS_PROFILE ) );
        EndDialog( hwndDlg, a );
      }
      break;

      case IDCANCEL:
      EndDialog( hwndDlg, -1 );
      break;
    }
    break;

    default: return FALSE;
  }
  return TRUE;
}

// Info Button calls About Dialog Box
void CALLBACK _export InfoButton( HWND hwndParent, WORD iConv, LP_DATA pvData )
{
  DialogBox( hInst, MAKEINTRESOURCE( ID_INFO_DIALOG ), hwndParent, InfoDlg );
}   

void CALLBACK _export SetupButton( HWND hwndParent, WORD iConv, LP_DATA pvData )
{
  if ( iConv == CV_DI || iConv == CV_DI2 )
  {
    int r = DialogBox( hInst, MAKEINTRESOURCE( ID_SETUP_DIALOG ), hwndParent, SetupDlg );

    if ( r != -1 ) pvData->fAlways = r;
  }
  if ( iConv == CV_DI3 )
  {
    int r = DialogBox( hInst, MAKEINTRESOURCE( ID_SETUP_DIALOG2 ), hwndParent, Setup2Dlg );

    if ( r != -1 ) pvData->fAutoRename = r;
  }
}

WORD CALLBACK _export Init( LPWORD pwVer, LPWORD pwBufSize )
{
  if ( *pwVer < 2 ) return 0;

  *pwVer     = 2; // API Version
  *pwBufSize = sizeof (CL_DATA) + min( *pwBufSize, 32000 ) * 2;

  return NCONV;
}

void CALLBACK _export Info( WORD iConv, LPCL_INFO pCL_Info, LP_DATA pvData )
{
  pCL_Info->wFlags = 0;

  switch ( iConv )
  {
    case CV_DI:
    case CV_DI2:
    case CV_DI3:
    pCL_Info->wFlags = CLF_SETUP | CLF_CONFIRM;

    case CV_TF:
    pCL_Info->wFlags |= CLF_ENABLE | CLF_INFO;
    pCL_Info->wFlags ^= CLF_CONFIRM;
    lstrcpyn( pCL_Info->szName, String( IDS_CONVNAME + iConv ), sizeof pCL_Info->szName );
    lstrcpyn( pCL_Info->szDescr, String( IDS_CONVDESCR + iConv ), sizeof pCL_Info->szDescr );
    break;

    default:
    lstrcpyn( pCL_Info->szName, String( IDS_LIBNAME ), sizeof pCL_Info->szName );
    lstrcpyn( pCL_Info->szDescr, String( IDS_LIBDESCR ), sizeof pCL_Info->szDescr );
  }
}

void CALLBACK _export Open( WORD wBufSize, LP_DATA pvData )
{
  // Initializing "pvData" block
  pvData->b_size      = wBufSize - sizeof (CL_DATA);
  pvData->fAlways     = GetPrivateProfileInt( szSecADM_Conv, szEntAlways, 0, String( IDS_PROFILE ) );
  pvData->fAutoRename = GetPrivateProfileInt( szSecADM_Conv, szEntAutoRename, 0, String( IDS_PROFILE ) );
}

void CALLBACK _export Close( LP_DATA pvData )
{
  // Destroying all objects in "pvData" block
}

BOOL CALLBACK _export QueryName( WORD iConv, LPCL_FILE_INFO fi, LP_DATA pvData )
{
  if (
       iConv == CV_TF && fi->dest.os == IOS_DOS33
       && (
            ( fi->src.os == IOS_MSDOS && IsTXT( fi->src.name ) )
            ||
            ( fi->src.os == IOS_PRODOS && fi->src.type == IT_PRODOS_TXT )
          )
     )
     fi->dest.type = IT_DOS33_T;

  if ( pvData->fAutoRename && iConv == CV_DI3 &&
       fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS &&
       fi->src.size == DSK_SIZE && IsPO( fi->src.name ) )
     {
       LPSTR p = fi->dest.name + lstrlen( fi->dest.name ) - 3;
       if ( *p == '.' ) p++;
       p[0] -= 'P' - 'D'; // .po -> .dsk; .PO -> .DSK
       p[1] -= 'O' - 'S';
       p[2] = p[1] - 'S' + 'K';
       p[3] = 0;
     }

  return FALSE; // TRUE will discard all QueryName calls for all following converters
}

BOOL CALLBACK _export QueryConv( HWND hwndParent, WORD iConv, LPCL_FILE_INFO fi, LP_DATA pvData )
{
  // don't change fi->dest.* fields! changes to fi->dest.type, fi->dest.subtype, fi->dest.name only in QueryName()

  pvData->nConv = -1;
  pvData->f     = 0;

  switch ( iConv )
  {
    case CV_DI: // checking whether this file can be DSK/DO->A4W or A4W->DSK/DO converted
    if (
         fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS && fi->src.type == IT_FDIMAGE
         && IsDSK( fi->src.name ) && IsA4W( fi->dest.name )
       ) pvData->nConv = DCV_DI_DSK_A4W;

    if (
         fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS && fi->src.type == IT_FDIMAGE
         && IsA4W( fi->src.name ) && IsDSK( fi->dest.name )
       )
       {
         char szType[32];

         _GetPrivateProfileString( szSecApple, szEntType, "", szType, sizeof szType, fi->src.shortname );
         if ( !lstrcmpi( szType, szTypeDisk2 ) )
         {
           static char szDefFormat[] = "DOS3.3";
           char        szFormat[10];

           _GetPrivateProfileString( szSecOptions, szEntFormat, szDefFormat, szFormat, sizeof szFormat, fi->src.shortname );
           if ( !lstrcmpi( szFormat, szDefFormat ) )
           {
             pvData->len   = 0;
             pvData->nConv = DCV_DI_A4W_DSK;
             _fmemset( pvData->b, 0, pvData->b_size );
           }
         }
       }
    break;

    case CV_DI2: // checking whether this file can be HDV<->A4W converted
    if (
         fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS && fi->src.type == IT_HDIMAGE
         && IsHDV( fi->src.name ) && IsA4W( fi->dest.name )
       ) pvData->nConv = DCV_DI_HDV_A4W;

    if (
         fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS
         && IsA4W( fi->src.name ) && IsHDV( fi->dest.name )
       )
       {
         char szType[32];

         _GetPrivateProfileString( szSecApple, szEntType, "", szType, sizeof szType, fi->src.shortname );
         if ( !lstrcmpi( szType, szTypeHDisk ) )
         {
           pvData->len   = 0;
           pvData->nConv = DCV_DI_A4W_HDV;
           _fmemset( pvData->b, 0, pvData->b_size );
         }
       }
    break;

    case CV_DI3: // checking whether this file can be PO<->DSK/DO converted
    if ( fi->src.os == IOS_MSDOS && fi->dest.os == IOS_MSDOS && fi->src.size == DSK_SIZE )
    {
      if ( IsPO( fi->src.name ) && fi->src.size == DSK_SIZE && IsDSK( fi->dest.name ) )
      {
        pvData->nConv = DCV_DI_PO_DSK;
        fi->dest.type  = IT_FDIMAGE;
      }
      if ( IsDSK( fi->src.name ) && fi->src.type == IT_FDIMAGE && IsPO( fi->dest.name ) )
      {
        pvData->nConv = DCV_DI_DSK_PO;
        fi->dest.type  = IT_FILE;
      }
    }
    break;

    case CV_TF: // text file transfer MS-DOS <-> ProDOS/DOS3.3
    if (
         fi->src.os == IOS_MSDOS
         &&
         (
           fi->dest.os == IOS_PRODOS || fi->dest.os == IOS_DOS33
         )
         &&
         (
           fi->dest.type == IT_PRODOS_TXT || IsTXT( fi->src.name )
         )
       ) pvData->nConv = DCV_TF_M_P3;

    if (
         fi->dest.os == IOS_MSDOS
         &&
         (
           fi->src.os == IOS_PRODOS || fi->src.os == IOS_DOS33
         )
         &&
         (
           fi->src.type == IT_PRODOS_TXT || fi->src.type == IT_DOS33_T || IsTXT( fi->dest.name )
         )
       ) pvData->nConv = DCV_TF_P3_M;
    break;
  }

  return ( pvData->nConv != (BYTE)-1 );
}

BOOL CALLBACK _export OpenConv( HWND hwndParent, WORD iConv, LPCL_FILE_INFO fi, LP_DATA pvData )
{
  if ( ( pvData->nConv == DCV_DI_DSK_A4W || pvData->nConv == DCV_DI_HDV_A4W ) && pvData->fAlways )
  {
    SetupButton( hwndParent, iConv, pvData );
  }                                 

  return TRUE;
}

WORD CALLBACK _export RBlockSize( WORD nDefault, LP_DATA pvData )
{
  // PO<->DSK converters read 1 track (16*256 bytes) each time
  if ( pvData->nConv == DCV_DI_PO_DSK || pvData->nConv == DCV_DI_DSK_PO ) return 4096;

  return min( nDefault, pvData->b_size / 2 );
}

WORD CALLBACK _export Convert( LPBYTE pbSrc, WORD size, LPBYTE FAR * ppbDest, LP_DATA pvData )
{
  WORD NewSize = 0;

  switch ( pvData->nConv )
  {
    case DCV_TF_M_P3: // MS-DOS text file to ProDOS/DOS3.3
    {
      LPBYTE pb;

      for ( pb = *ppbDest = pbSrc; size--; pbSrc++ )
      {
        if ( *pbSrc == 10 && pvData->f )
        {
          pvData->f = FALSE;
          continue; // skip LF after CR
        }
        pvData->f = ( *pbSrc == 13 );
        *pb++     = ( *pbSrc == 10 ) ? 0x8D : *pbSrc | 128;

        NewSize++;
      }
    }
    break;

    case DCV_TF_P3_M: // ProDOS/DOS3.3 text file to MS-DOS
    {
      LPBYTE pb;

      for ( pb = *ppbDest = pvData->b; size--; pbSrc++ )
      {
        if ( !*pbSrc ) continue;

        *pb++ = *pbSrc & 127;
        NewSize++;
        if ( ( *pbSrc & 127 ) == 13 )
        {
          *pb++ = 10; // expand CR to CRLF
          NewSize++;
        }
      }
    }
    break;

    case DCV_DI_A4W_DSK:
    if ( size )
    {                      
      while ( pvData->f < 2 && size-- ) // Looking for sequence of 2 x 26(1Ah) bytes which is the end of A4W file header
      {
        if ( *pbSrc++ == 26 )
          pvData->f++;
        else
          pvData->f = 0;
      }

      if ( pvData->f == 2 ) // the file header has been skipped and now the A4W data "section" is being transferred
      {
        *ppbDest     = pbSrc;
        pvData->len += ( NewSize = size );
      }
    }
    else // size=0 -> flush request. A4W disk image length can be less than 143,360 bytes. DSK must be exactly 143,360
      if ( pvData->len < DSK_SIZE )
      {
        *ppbDest     = pvData->b;
        NewSize      = DSK_SIZE >= pvData->len + pvData->b_size ? pvData->b_size : (WORD)( DSK_SIZE - pvData->len );
        pvData->len += NewSize;
      }
    break;

    case DCV_DI_DSK_A4W:
    if ( pvData->f )
    {
      *ppbDest = pbSrc;
      NewSize  = size;
    }
    else
    {
      char szDiskName[70];
      int  l;

      lstrcpy( szDiskName, "\r\nName=" );
      lstrlen( szDiskName );
      if ( !GetPrivateProfileString( szSecADM_Conv, szEntDiskName, "", szDiskName + lstrlen( szDiskName ), sizeof szDiskName - lstrlen( szDiskName ), String( IDS_PROFILE ) ) )
        *szDiskName = 0;
      wsprintf( (LPSTR)pvData->b, "[%s]\r\n%s=%s%s\r\n\032\032", (LPSTR)szSecApple, (LPSTR)szEntType, (LPSTR)szTypeDisk2, (LPSTR)szDiskName );
      l = lstrlen( (LPSTR)pvData->b );
      _fmemcpy( pvData->b + l, pbSrc, size );
      NewSize   = size + l;
      *ppbDest  = pvData->b;
      pvData->f = 1;
    }
    break;

    case DCV_DI_A4W_HDV:
    if ( size )
    {                      
      while ( pvData->f < 2 && size-- ) // see "case DCV_DI_A4W_DSK" above
      {
        if ( *pbSrc++ == 26 )
          pvData->f++;
        else
          pvData->f = 0;
      }

      if ( pvData->f == 2 )
      {
        *ppbDest     = pbSrc;
        pvData->len += ( NewSize = size );
      }
    }
    break;

    case DCV_DI_HDV_A4W:
    if ( pvData->f )
    {
      *ppbDest = pbSrc;
      NewSize  = size;
    }
    else
    {
      char szDiskName[70];
      int  l;

      lstrcpy( szDiskName, "\r\nName=" );
      lstrlen( szDiskName );
      if ( !GetPrivateProfileString( szSecADM_Conv, szEntDiskName, "", szDiskName + lstrlen( szDiskName ), sizeof szDiskName - lstrlen( szDiskName ), String( IDS_PROFILE ) ) )
        *szDiskName = 0;
      wsprintf( (LPSTR)pvData->b, "[%s]\r\n%s=%s%s\r\n\r\n[Options]\r\nSize=32768\r\n\032\032", (LPSTR)szSecApple, (LPSTR)szEntType, (LPSTR)szTypeHDisk, (LPSTR)szDiskName );
      l = lstrlen( (LPSTR)pvData->b );
      _fmemcpy( pvData->b + l, pbSrc, size );
      NewSize   = size + l;
      *ppbDest  = pvData->b;
      pvData->f = 1;
    }
    break;
                       
    case DCV_DI_PO_DSK:
    case DCV_DI_DSK_PO:
    *ppbDest = pbSrc;
    if ( size == 4096 )
    {
      BYTE tmp[256];                
      int  i, po[8] = { 0, 14, 13, 12, 11, 10, 9, 8 };

      for ( i = 1; i < 8; i++ )
      {
        _fmemcpy( tmp, pbSrc + i * 256, sizeof tmp );
        _fmemcpy( pbSrc + i * 256, pbSrc + po[i] * 256, sizeof tmp );
        _fmemcpy( pbSrc + po[i] * 256, tmp, sizeof tmp );
      }
      NewSize = size;
    }
    break;
  }

  return NewSize;
}

void CALLBACK _export CloseConv( LP_DATA pvData )
{
}
