// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: gxscomm.cpp
// C++ Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 09/20/1999
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.
 
You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The gxSerialComm class is a base class used to open a serial 
port for bidirectional communication. It includes several 
low-level functions needed by derived classes to initalize a 
serial port, transmit, and receive data.
*/
// ----------------------------------------------------------- // 
#if defined(__UNIX__)
#include <fcntl.h>   // File control definitions
#include <string.h>
#endif

#include "gxscomm.h" 

// NOTE: This array must contain the same number of exceptions as the
// internal error code enumeration. 
const int gxsMaxSCommExceptionMessages = 17;
const char *gxsSerialCommExceptionMessages[gxsMaxSCommExceptionMessages] = {
  "Serial port exception: No errors reported",               // NO_ERROR
  "Serial port exception: Invalid error code",               // INVALID_CODE
  "Serial port exception: Invalid baud rate",                // BAUDRATE_ERROR
  "Serial port exception: Invalid character size",           // CS_ERROR
  "Serial port exception: Invalid flow control",             // FLOWCONTROL
  "Serial port exception: Cannot initialize serial device",  // INIT_ERROR
  "Serial port exception: Invalid initialization parameter", // INVALIDPARM
  "Serial port exception: Cannot open serial device",        // OPEN_ERROR
  "Serial port exception: Invalid parity",                   // PARITY_ERROR
  "Serial port exception: Serial device receive error",      // RECEIVE_ERROR
  "Serial port exception: Invalid stop bit",                 // STOPBIT_ERROR
  "Serial port exception: Serial device transmit error",     // TRANSMIT_ERROR

  // Variable block exception messages
  "Serial port exception: Variable block acknowledgment error", // BLOCKACK
  "Serial port exception: Bad database block header",           // BLOCKHEADER
  "Serial port exception: Bad database block size",             // BLOCKSIZE
  "Serial port exception: Variable block synchronization error" // BLOCKSYNC
};

gxSerialComm::gxSerialComm()
{
  // Default setting 
  baud_rate = 9600;
  parity = 'N';
  character_size = 8;
  stop_bits = 1;
  flow_control = gxSerialComm::scommNO_FLOW_CONTROL;
  binary_mode = 1; 
  scomm_error = gxSerialComm::scomm_NO_ERROR;

#if defined (__WIN32__)
  memset(&dcb, 0, sizeof(dcb));
  ZeroMemory(&ov, sizeof(ov));
#endif

#if defined (__UNIX__)
  memset(&options, 0, sizeof(options));
#endif
}

gxSerialComm::~gxSerialComm()
{
  Close();
}

int gxSerialComm::OpenSerialPort(char *device_name)
// Open a serial device for read/write operations. 
{
#if defined (__WIN32__)
  // Open com ports 1 through 9 for read/write access.
  // Valid device names for Windows are: COM1, COM2, etc.
  device_handle = CreateFile(device_name,
			     GENERIC_READ | GENERIC_WRITE,
			     FILE_SHARE_READ | FILE_SHARE_WRITE,
			     NULL,
			     OPEN_EXISTING,
			     FILE_FLAG_OVERLAPPED,
			     NULL);
  
  if(device_handle == INVALID_HANDLE_VALUE) {
    scomm_error = gxSerialComm::scomm_OPEN_ERROR;
    return -1;
  }
  return scommREAD_WRITE; // Device open for read/write access

#elif defined (__UNIX__)
  // NOTE: UNIX systems will not allow you to open tty devices
  // for both reading and writing unless run the program as root
  // or the owner of that device file. The permissions for TTY
  // devices defaults to crw--w--w- or crw-r--r--. This function
  // will try to open the device for read/write, read, and then write.
  // The O_NOCTTY bit is set so that no controlling tty is assigned.
  // The O_NDELAY bit is set to enable nonblocking mode for this
  // device. This will allow all read operations to return if there
  // is no input immediately available. NOTE: If the O_NONBLOCK bit 
  // is set write operations will not work properly under HPUX 10.20.
  device_handle = open(device_name, O_RDWR | O_NOCTTY | O_NDELAY);

  if(device_handle < 0) 
    device_handle = open(device_name, O_RDONLY | O_NOCTTY | O_NDELAY);
  else 
    return scommREAD_WRITE;
  
  if(device_handle < 0) 
    device_handle = open(device_name, O_WRONLY | O_NOCTTY | O_NDELAY);
  else 
    return scommREAD_ONLY;
  
  if(device_handle < 0) {
    scomm_error = gxSerialComm::scomm_OPEN_ERROR;
    return -1;
  }
  else
    return scommWRITE_ONLY;

#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
}

int gxSerialComm::InitSerialPort()
{
#if defined (__WIN32__)
  // Set timeouts
  COMMTIMEOUTS cto = { 2, 1, 1, 0, 0 };

  // SetCommTimeouts failed
  if(!SetCommTimeouts(device_handle, &cto)) {
    scomm_error = gxSerialComm::scomm_INIT_ERROR;
    return -1;
  }
  
  // Set the device-control block structure
  dcb.DCBlength = sizeof(dcb);

  // Set the baud rates
  switch(baud_rate) {
    case 110: 
      dcb.BaudRate = CBR_110;
      break;
    case 300:
      dcb.BaudRate = CBR_300;
      break;
    case 600: 
      dcb.BaudRate = CBR_600;
      break;
    case 1200: 
      dcb.BaudRate = CBR_1200;
      break;
    case 2400: 
      dcb.BaudRate = CBR_2400;
      break;
    case 4800: 
      dcb.BaudRate = CBR_4800;
      break;
    case 9600: 
      dcb.BaudRate = CBR_9600;
      break;
    case 14400: 
      dcb.BaudRate = CBR_14400;
      break;
    case 19200: 
      dcb.BaudRate = CBR_19200;
      break;
    case 38400: 
      dcb.BaudRate = CBR_38400;
      break;
    case 56000: 
      dcb.BaudRate = CBR_56000;
      break;
    case 57600: 
      dcb.BaudRate = CBR_57600;
      break;
    case 115200: 
      dcb.BaudRate = CBR_115200;
      break;
    case 128000: 
      dcb.BaudRate = CBR_128000;
      break;
    case 256000: 
      dcb.BaudRate = CBR_256000;
      break;
    default:
      scomm_error = gxSerialComm::scomm_BAUDRATE_ERROR;
      return -1; // Invalid baud rate
  }

  // Set the parity
  switch(parity) {
    case 'n' : case 'N' :
      dcb.Parity = NOPARITY;
      break;
    case 'e' : case 'E' :
      dcb.Parity = EVENPARITY;
      break;
    case 'o' : case 'O' :
      dcb.Parity = ODDPARITY;
      break;
    case 'm' : case 'M' :
      dcb.Parity = MARKPARITY;
      break;
    case 's' : case 'S' :
      // Space parity is setup the same as no parity 
      dcb.Parity = NOPARITY;
      break;
    default:
      scomm_error = gxSerialComm::scomm_PARITY_ERROR;
      return -1; // Invalid parity
  }
    
  // Set the stop bits
  switch(stop_bits) {
    case 1 :
      dcb.StopBits = ONESTOPBIT;
      break;
    case 15 :
      dcb.StopBits = ONE5STOPBITS;
      break;
    case 2 :
      dcb.StopBits = TWOSTOPBITS;
      break;
    default:
      scomm_error = gxSerialComm::scomm_STOPBIT_ERROR;
      return -1; // Invalid stop bit
  }
  
  // Set the character size
  switch(character_size) {
    case 7 :
    dcb.ByteSize = 7;
      break;
    case 8 :
    dcb.ByteSize = 8;
      break;
    default:
      scomm_error = gxSerialComm::scomm_CS_ERROR;
      return -1; // Invalid character size
  }
  
  // NOTE: The Win32 API does not support non-binary mode transfers
  dcb.fBinary = binary_mode;

  switch(flow_control) {
    case scommHARD_FLOW :
      dcb.fDtrControl = DTR_CONTROL_ENABLE;
      dcb.fOutxCtsFlow = 1;
      dcb.fRtsControl = DTR_CONTROL_HANDSHAKE;
      break;
    case scommXON_XOFF: {
      int DC1 = 17; char xon = DC1;
      int DC2 = 19; char xoff = DC2;
      dcb.XonChar = xon;
      dcb.XoffChar = xoff;
      dcb.XonLim = 1024;
      dcb.XoffLim = 1024;
      dcb.fDtrControl = DTR_CONTROL_DISABLE;
      dcb.fOutxCtsFlow = 0;
      dcb.fRtsControl = RTS_CONTROL_DISABLE;
      dcb.fOutxDsrFlow = 0;
    }
    break;
    case scommSOFT_FLOW :
      dcb.fDtrControl = DTR_CONTROL_DISABLE;
      dcb.fOutxCtsFlow = 0;
      dcb.fRtsControl = RTS_CONTROL_DISABLE;
      dcb.fOutxDsrFlow = 0;
      break;
    case scommNO_FLOW_CONTROL :
      dcb.fDtrControl = DTR_CONTROL_DISABLE;
      dcb.fOutxCtsFlow = 0;
      dcb.fRtsControl = RTS_CONTROL_DISABLE;
      dcb.fOutxDsrFlow = 0;
      break;
    default:
      scomm_error = gxSerialComm::scomm_FLOWCONTROL_ERROR;
      return -1; // Invalid flow control
  }

  // SetCommState failed
  if(!SetCommState(device_handle, &dcb)) {
    scomm_error = gxSerialComm::scomm_INIT_ERROR;
    return -1;
  }
  
#elif defined (__UNIX__)
  // Reset the port options
  memset(&options, 0, sizeof(options));

  // Set the baud rates
  switch(baud_rate) {
    case 0: // 0 baud (drop DTR)
      cfsetispeed(&options, B0);
      cfsetospeed(&options, B0);
      break;
    case 50: 
      cfsetispeed(&options, B50);
      cfsetospeed(&options, B50);
      break;
    case 75: 
      cfsetispeed(&options, B75);
      cfsetospeed(&options, B75);
      break;
    case 110: 
      cfsetispeed(&options, B110);
      cfsetospeed(&options, B110);
      break;
    case 134: 
      cfsetispeed(&options, B134);
      cfsetospeed(&options, B134);
      break;
    case 150: 
      cfsetispeed(&options, B150);
      cfsetospeed(&options, B150);
      break;
    case 200: 
      cfsetispeed(&options, B200);
      cfsetospeed(&options, B200);
      break;
    case 300: 
      cfsetispeed(&options, B300);
      cfsetospeed(&options, B300);
      break;
    case 600: 
      cfsetispeed(&options, B600);
      cfsetospeed(&options, B600);
      break;
    case 1200: 
      cfsetispeed(&options, B1200);
      cfsetospeed(&options, B1200);
      break;
    case 1800: 
      cfsetispeed(&options, B1800);
      cfsetospeed(&options, B1800);
      break;
    case 2400: 
      cfsetispeed(&options, B2400);
      cfsetospeed(&options, B2400);
      break;
    case 4800: 
      cfsetispeed(&options, B4800);
      cfsetospeed(&options, B4800);
      break;
    case 9600: 
      cfsetispeed(&options, B9600);
      cfsetospeed(&options, B9600);
      break;
    case 19200: 
      cfsetispeed(&options, B19200);
      cfsetospeed(&options, B19200);
      break;
    case 57600: 
      cfsetispeed(&options, B57600);
      cfsetospeed(&options, B57600);
      break;
    case 115200: 
      cfsetispeed(&options, B115200);
      cfsetospeed(&options, B115200);
      break;
      // Removed 07/16/2000 to maintain compatibility with all UNIX
      // variants that due no defined baud rates higher then 115K
      // case 230400: 
      // cfsetispeed(&options, B230400);
      // cfsetospeed(&options, B230400);
      // break;
      // case 460800: 
      // cfsetispeed(&options, B460800);
      // cfsetospeed(&options, B460800);
      // break;
    default:
      scomm_error = gxSerialComm::scomm_BAUDRATE_ERROR;
      return -1; // Invalid baud rate
  }

  // Set the character size, parity and stop bits
  if((character_size == 8) && (parity == 'N') && (stop_bits == 1)) {
    // No parity (8N1) 
    options.c_cflag &= ~PARENB;
    options.c_cflag &= ~CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS8;
    options.c_iflag = IGNPAR;
  }
  else if((character_size == 7) && (parity == 'E') && (stop_bits == 1)) {
    // Even parity (7E1) 
    options.c_cflag |= PARENB;
    options.c_cflag &= ~PARODD;
    options.c_cflag &= ~CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS7;
    options.c_iflag |= (INPCK | ISTRIP);
  }
  else if((character_size == 7) && (parity == 'O') && (stop_bits == 1)) {
    // Odd parity (7O1) 
    options.c_cflag |= PARENB;
    options.c_cflag |= PARODD;
    options.c_cflag &= ~CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS7;
    options.c_iflag |= (INPCK | ISTRIP);
  }
  else if((character_size == 7) && (parity == 'M') && (stop_bits == 1)) {
    // Mark parity is simulated by using 2 stop bits (7M1)
    options.c_cflag &= ~PARENB;
    options.c_cflag |= CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS7;
    options.c_iflag |= (INPCK | ISTRIP);
  }
  else if((character_size == 7) && (parity == 'S') && (stop_bits == 1)) {
    // Space parity is setup the same as no parity (7S1)
    options.c_cflag &= ~PARENB;
    options.c_cflag &= ~CSTOPB;
    options.c_cflag &= ~CSIZE;
    options.c_cflag |= CS7;
    options.c_iflag |= (INPCK | ISTRIP);
  }
  else {
    scomm_error = gxSerialComm::scomm_INVALIDPARM;
    return -1; // Invalid character size, parity and stop bits combination
  }
  
  switch(flow_control) {
    case scommHARD_FLOW :
#if defined(CRTSCTS)
      options.c_cflag |= CRTSCTS;
      break;
#else
      break; // Hard flow control is not supported
#endif

    case scommXON_XOFF :
      options.c_iflag |= (IXON | IXOFF | IXANY);
#if defined(CRTSCTS)
      options.c_cflag &= ~CRTSCTS;
#endif
      break;

    case scommSOFT_FLOW :
#if defined(CRTSCTS)
      options.c_cflag &= ~CRTSCTS;
      break;
#else
      break; // Hard flow control is not supported
#endif

    case scommNO_FLOW_CONTROL :
#if defined(CRTSCTS)
      options.c_cflag &= ~CRTSCTS;
      break;
#else
      break; // Hard flow control is not supported
#endif

    default:
      scomm_error = gxSerialComm::scomm_FLOWCONTROL_ERROR;
      return -1; // Invalid flow control
  }

  if(!binary_mode) { // Set the port for canonical input (line-oriented) 
    // Input characters are put into a buffer until a CR (carriage return)
    // or LF (line feed) character is received.
    options.c_lflag |= (ICANON | ECHOE);

    // Postprocess the output.
    // The ONLCR flag will map NL (linefeeds) to CR-NL on output.
    // The OLCUC flag will map characters to upper case for tty output.
    options.c_oflag = OPOST | ONLCR | OLCUC;
  }
  else { // Use raw input/output
    // Input characters are passed through exactly as they are received,
    // when they are received.
    options.c_lflag = 0;
    options.c_oflag = 0;
  }
  
  // Enable the receiver and set local mode
  options.c_cflag |= (CLOCAL | CREAD);
  
  // Initialize control characters if needed.
  // Default values can be found in /usr/include/termios.h, and are given
  // in the comments.

  options.c_cc[VTIME]    = 0;     // inter-character timer unused 
  options.c_cc[VMIN]     = 1;     // blocking read until 1 character arrives 

  // options.c_cc[VEOF]     = 4;     // Ctrl-d  
  // options.c_cc[VINTR]    = 0;     // Ctrl-c 
  // options.c_cc[VQUIT]    = 0;     // Ctrl-\ 
  // options.c_cc[VERASE]   = 0;     // del 
  // options.c_cc[VKILL]    = 0;     // @ 
  // options.c_cc[VSTART]   = 0;     // Ctrl-q 
  // options.c_cc[VSTOP]    = 0;     // Ctrl-s 
  // options.c_cc[VSUSP]    = 0;     // Ctrl-z 
  // options.c_cc[VEOL]     = 0;     // '\0' 
  // options.c_cc[VWERASE]  = 0;     // Ctrl-w 
  // options.c_cc[VLNEXT]   = 0;     // Ctrl-v 
  // options.c_cc[VEOL2]    = 0;     // '\0' 

  // May not be defined in every UNIX variant
  // options.c_cc[VSWTC]    = 0;     // '\0' 
  // options.c_cc[VREPRINT] = 0;     // Ctrl-r 
  // options.c_cc[VDISCARD] = 0;     // Ctrl-u 
  
#if defined (FASYNC) // ASYNC I/O is a BDS feature
  // Make the file descriptor asynchronous
  fcntl(device_handle, F_SETFL, FASYNC);
#endif

  // Set the new options for the port. The TCSANOW constant specifies
  // that all changes should occur immediately without waiting for
  // output data to finish sending or input data to finish receiving. 
  tcflush(device_handle, TCIFLUSH); // Clean the serial line
  tcsetattr(device_handle, TCSANOW, &options);

#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif

return 1; // No errors reported
}

void gxSerialComm::Close()
{
// Type definitions
#if defined (__WIN32__)
  CloseHandle(device_handle);
#elif defined (__UNIX__)
  close(device_handle);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
}

int gxSerialComm::RawRead(void *buf, int bytes)
// Read a specified number of bytes from the serial port
// and return whether or not the read was completed.
// Returns the number of bytes received or -1 if an
// error occurred.
{
#if defined (__WIN32__)
  if(!ReadFile(device_handle, (char *)buf, bytes, &bytes_read, &ov)) {
    scomm_error = gxSerialComm::scomm_RECEIVE_ERROR;
    return -1;
  }
#elif defined (__UNIX__)
  bytes_read = read(device_handle, (char *)buf, bytes);
  if(bytes_read < 0) {
    scomm_error = gxSerialComm::scomm_RECEIVE_ERROR;
    return -1;
  }
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  return (int)bytes_read;
}

int gxSerialComm::RawWrite(const void *buf, int bytes)
// Write a specified number of bytes to a serial port
// and return whether or not the write was complete.
// Returns the total number of bytes moved or -1 if
// an error occurred.
{
#if defined (__WIN32__)
  if(!WriteFile(device_handle, (char *)buf, bytes, &bytes_moved, &ov)) {
    scomm_error = gxSerialComm::scomm_TRANSMIT_ERROR;
    return -1;
  }
#elif defined (__UNIX__)
  bytes_moved = write(device_handle, (char *)buf, bytes);
  if(bytes_moved < 0) {
    scomm_error = gxSerialComm::scomm_TRANSMIT_ERROR;
    return -1;
  }
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  return (int)bytes_moved;
}

int gxSerialComm::Recv(void *buf, int bytes)
// Receive a specified number of bytes from a serial port
// and do not return until all the byte have been read.
// Returns the total number of bytes read or -1 if an
// error occurred.
{
  int br = 0;               // Byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(br < bytes) { // Loop unitl the buffer is full
    if((num_read = RawRead(p, num_req-br)) > 0) {
      br += num_read;   // Increment the byte counter
      p += num_read;    // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      scomm_error = gxSerialComm::scomm_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  
  bytes_read = br; // Undate the object's byte counter
  return bytes_read;
}

int gxSerialComm::Send(const void *buf, int bytes)
// Write a specified number of bytes to a serial port and do
// not return until all the bytes have been written. Returns
// the total number of bytes written or -1 if an error occurred.
{
  int bm = 0;                // Byte counter
  int num_moved = 0;         // Actual number of bytes written
  int num_req = (int)bytes;  // Number of bytes requested 
  char *p = (char *)buf;     // Pointer to the buffer

  while(bm < bytes) { // Loop unitl the buffer is empty
    if((num_moved = RawWrite(p, num_req-bm)) > 0) {
      bm += num_moved;  // Increment the byte counter
      p += num_moved;   // Move the buffer pointer for the next read
    }
    if(num_moved < 0) {
      scomm_error = gxSerialComm::scomm_TRANSMIT_ERROR;
      return -1; // An error occurred during the read
    }
  }
  
  bytes_moved = bm; // Update the object's byte counter
  return bytes_moved;
}

int gxSerialComm::InitSerialPort(char *device_name, int sp, char pr, int cs,
				 int sb, int flow, int bin_mode)
// Initialize a serial device using the specified parameters. Returns
// -1 if an error occurred during initialization or the current access
// mode of the port (scommREAD_WRITE, scommREAD_ONLY, scommREAD_WRITE).
{
  int status = OpenSerialPort(device_name); 
  if(status < 0) return -1;

  SetBaudRate(sp);
  SetCharacterSize(cs);
  SetParity(pr);
  SetStopBits(sb);
  SetFlowControl(flow);
  if(bin_mode) BinaryMode(); else CharacterMode();

  if(InitSerialPort() < 0) return -1;
  
  return status;
}

const char *gxSerialComm::SerialCommExceptionMessage()
// Returns a null terminated string that can
// be use to log or print a serial port exeception.
{
  if(scomm_error > gxsMaxSCommExceptionMessages)
    scomm_error = gxSerialComm::scomm_INVALID_ERROR_CODE;
  
  // Find the corresponding message in the exception array
  return gxsSerialCommExceptionMessages[scomm_error];
}
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
