// ------------------------------- //
// -------- start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: membuf.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 09/20/1999
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The MemoryBuffer class is used to create create and manipulate resizable
memory buffers. The MemoryBuffer class guarantees that each object is 
unique by storing a unique copy of each buffer with each object. This 
ensures that MemoryBuffer objects can be safely copy constructed, 
assigned, resized, and deleted by multiple threads. NOTE: Multiple 
threads accessing shared memory segments must be handled by the 
application.
*/
// ----------------------------------------------------------- // 
#include "membuf.h"
      
MemoryBuffer::MemoryBuffer(const void *buf, unsigned bytes)
{
  mptr = 0;
  l_length = d_length = 0;
  if(Alloc(bytes)) memmove(mptr, buf, bytes);
}

MemoryBuffer::MemoryBuffer(const MemoryBuffer &buf)
{
  mptr = 0;
  l_length = d_length = 0;
  if(Alloc(buf.l_length)) memmove(mptr, buf.mptr, buf.l_length);
}

MemoryBuffer &MemoryBuffer::operator=(const MemoryBuffer &buf)
{
  if(this != &buf) { // Prevent self assignment
    if(Alloc(buf.l_length)) memmove(mptr, buf.mptr, buf.l_length);
  }
  return *this;
}

void *MemoryBuffer::Alloc(unsigned bytes)
// Allocate a specified number of bytes for this memory buffer.
// This function will try to re-use the current memory segment
// allocated for this buffer before re-allocating memory for
// the buffer. Returns a void pointer to the buffer or a null
// value if a memory allocation error occurs.
{
  if(mptr) { // Memory was previously allocated for this object
    if(d_length >= bytes) { // Try to reuse this space
      l_length = bytes;
      return (void *)mptr;
    }
    else { // Must resize the buffer
      delete mptr;// Delete the previous copy and re-allocate memory
      l_length = d_length = 0;
    }
  }

  // Allocate the specified number of bytes
  mptr = new unsigned char[bytes];
  if(!mptr) return 0; // Memory allocation error

  // Set the logical and dimensioned length of the buffer
  l_length = d_length = bytes;
  return (void *)mptr; // Return a pointer to the buffer
}

void *MemoryBuffer::Realloc(unsigned bytes, int keep, int reuse)
// Resize the logical length of the buffer. If the
// "keep" variable is true the old data will be
// copied into the new space. By default the old
// data will not be deleted. Returns a pointer to the
// buffer or a null value if an error occurs.
{
  // Try to reuse the memory allocated for this object
  if((reuse == 1) && (mptr != 0)) { 
    if(d_length >= bytes) { // No additional memory has to be allocated
      l_length = bytes;
      return (void *)mptr;
    }
  }
  
  unsigned char *nmptr = new unsigned char[bytes];
  if(!nmptr) return 0;

  if((keep == 1) && (mptr != 0)) { // Copy old data into the new memory segment
    if(bytes < l_length) l_length = bytes;
    memmove(nmptr, mptr, l_length);
  }

  if(mptr) delete mptr;        // Free the previously allocated memory
  mptr = nmptr;                // Point to new memory buffer
  l_length = d_length = bytes; // Record new allocated length
  return (void *)mptr;
}

void MemoryBuffer::Destroy()
// Frees the memory allocated for the buffer and resets the
// length variables.
{
  if(mptr) delete mptr;
  mptr = 0;
  l_length = d_length = 0;
}

void *MemoryBuffer::FreeBytes()
// Free any unused bytes allocated for this buffer. Returns
// a pointer to the re-allocated memory buffer or a null
// value if an error occurs.
{
  // Check for unused bytes
  if(d_length == l_length) return (void *)mptr;
  return Realloc(l_length, 1, 0);
}

int MemoryBuffer::resize(unsigned bytes, int keep)
// Resize the logical length of the buffer. If the
// "keep" variable is true the old data will be
// copied into the new space. By default the old
// data will not be deleted. Returns true if
// successful or false if an error occurs.
{
  if(!Realloc(bytes, keep)) return 0;
  return 1;
}

unsigned MemoryBuffer::Find(void *buf, unsigned offset)
// Returns index of first occurrence of pattern void *buf.
// Returns -1 if the pattern is not found.
{
  unsigned char *start = mptr + offset;          // Start of buffer data
  unsigned char *next = start;                   // Next buffer element
  unsigned char *Pattern = (unsigned char *)buf; // Next pattern element
  unsigned i = offset;                           // Next buffer element index
  
  while(i < l_length && *Pattern) {
    if (*next == *Pattern) {
      Pattern++;
      if(*Pattern == 0) return i; // Pattern was found
      next++;
    }
    else {
      i++;
      start++;
      next = start;
      Pattern = (unsigned char *)buf;
    }
  }
  return -1; // No match was found
}

unsigned MemoryBuffer::Find(void *buf, unsigned bytes, unsigned offset) const
// Returns index of first occurrence of pattern void *buf.
// Returns -1 if the pattern is not found.
{
  char *start = (char *)mptr + offset; // Start of buffer data
  char *next = start;          // Next buffer element
  char *Pattern = (char *)buf; // Next pattern element
  unsigned i = offset, j = 1;  // Buffer and pattern indexes
  
  while(i < l_length && j <= bytes) {
    if (*next == *Pattern) { // Matching character
      if(j == bytes) return i; // Entire match was found
      next++; Pattern++; j++;
    }
    else { // Try next spot in buffer
      i++;
      start++;
      next = start;
      Pattern = (char *)buf; j = 1;
    }
  }
  return -1; // No match was found
}

unsigned MemoryBuffer::DeleteAt(unsigned position, unsigned bytes)
{
  unsigned long buf; // long is used to prevent overflows
  unsigned end;

  if(position < l_length && bytes !=0) {
    buf = position + bytes;
    if(buf > l_length) buf = l_length;
    end = unsigned(buf);
    bytes = end - position;
    // Move the elements up to take their place
    memmove(mptr+position, mptr+end, l_length-end);
    l_length -= bytes;
  }
  else
    bytes = 0;

  return bytes;
}

unsigned MemoryBuffer::InsertAt(unsigned position, const void *buf,
				unsigned bytes)
// Insert a specified number of bytes a the current position, keeping
// the current buffer intact. Returns the number of bytes inserted or
// zero if an error occurs.
{
  unsigned old_length = l_length; // Record the current buffer length
  
  // Ensure that there are enough bytes to hold the insertion
  if(!resize(bytes+l_length)) return 0;

  if(position < old_length) { // Move the data in the middle of the buffer
    memmove(mptr+position+bytes, mptr+position, old_length-position);
  }
  
  if(position > l_length) position = l_length; // Stay in bounds
  memmove(mptr+position, buf, bytes);
  return bytes;
}

unsigned MemoryBuffer::ReplaceAt(unsigned position, const void *buf,
				 unsigned bytes)
// Replace a specified number of bytes at the specified position. Returns
// the number of bytes replaced or zero if an error occurs.
{
  if(position > l_length) position = l_length; // Stay in bounds
  unsigned end = l_length-position;
  if(bytes > end) { // There are not enough bytes to hold the replacement
    unsigned needed = bytes-end;
    if(!resize(l_length + needed)) return 0;
  }
  memmove(mptr+position, buf , bytes);
  return bytes;
}

int BufferCompare(const MemoryBuffer &a, const MemoryBuffer &b)
// Returns -1 if a < b, 0 if a == b, and 1 if a > b
{

  unsigned an = a.l_length;
  unsigned bn = b.l_length;
  unsigned sn = (an < bn) ? an : bn;
  unsigned char *ap = (unsigned char *)a.mptr;
  unsigned char *bp = (unsigned char *)b.mptr;

  for(unsigned i = 0; i < sn; i++) {
    if(*ap < *bp) return -1;
    if(*ap++ > *bp++) return 1;
  }

  if(an == bn) return 0;
  if(an < bn) return -1;
  return 1;
}

int MemoryBuffer::Load(const void *buf, unsigned bytes) 
// Load this object with a unique copy of the specified buffer.
// Returns true if successful or false if an error occurs.
{   
  if(!mptr) { // Ensure that this buffer has been initialized
    if(!Alloc(bytes)) return 0;
  }

  if(d_length < bytes) { // Ensure enough byte have been allocated
    if(!Realloc(bytes, 0, 0)) return 0;
  }
  else { // Reuse the current memory segment
    l_length = bytes;
  }

  memmove(mptr, (unsigned char *)buf, l_length);
  return 1;
}

void MemoryBuffer::Clear() 
// Clear the buffer
{ 
  if(l_length > 0) DeleteAt(0, l_length); 
} 

MemoryBuffer operator+(const MemoryBuffer &a, const MemoryBuffer &b)
{
  MemoryBuffer buf(a.l_length + b.l_length);
  buf += a;
  buf += b;
  return buf;
}

unsigned char &MemoryBuffer::operator[](unsigned i) 
{
  if(i >= l_length) i = l_length; // If not in bounds truncate to l_length
  return mptr[i];
}

unsigned char &MemoryBuffer::operator[](unsigned i) const
{
  if(i >= l_length) i = l_length; // If not in bounds truncate to l_length
  return mptr[i];
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //
