/** \file DS_Stack.h
    Stack data structure

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.
Permission is granted to use and copy this file for non-commercial use only.
Please contact the author concerning commercial usage.
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/
#ifndef H_DS_STACK
#define H_DS_STACK

#include <DS_Enumeration.h>

/** A classic LIFO data structure.  */
class Stack : public Vector  // Vector should normally be protected, but
{                            // we make it public so that Stack, Queue and 
public:                      // derivations be used as generic Containers.
  /** Default Constructor: Construct an empty stack. */
  Stack(const unsigned flags=0) : Vector(flags) {}

  /** Copy Constructor: Construct a copy of a stack. */
  Stack(const Stack& s)         : Vector(s)     {}

  /** Construct a stack from any container.  The elements
      are pushed in the index order of the container. */
  Stack(const Container& c)     : Vector(c)     {}

  /** Construct a stack from an enumeration.  The elements
      are pushed in the order they are enumerated. */
  Stack(const Enumeration& e)   : Vector(e)     {}

  /** Construct a stack from a vector.  The elements
      are pushed in index order. */
  Stack(const Vector& v)        : Vector(v)     {}

  /** Destructor. */
  ~Stack()                                      {}

  /** Assignment: Copy a stack. */
  Stack& operator=(const Stack& s)              { return *this = s; }

 
  /** Returns non-zero if the stack is empty. */
  DS_BOOL empty()                               { return isEmpty(); } 

  /** Returns the top element in the stack. */
  Data peek()                                   { return lastElement(); }

  /** Returns the top element in the stack. */
  Data top()                                    { return peek(); }

  /** Returns the top element in the stack and removes it.
      Note: pop() returns garbage in a destructive stack. */
  Data pop()                                    { return Vector::pop(); }

  /** Push an item to the top of the stack. */
  void push(Data item)                          { addElement(item); }

  /** Search for an item by utilizing the equality function of Containable. */
  Data search(Data item)                        { return elementAt(lastIndexOf(item)); }

  virtual void clear()                          { Vector::clear(); }

  /** Dip into the stack and retrieve an item */
  Data peek(const int index)
  {
    int count = Vector::size();
    if (index < 0 || index >= count) throw IndexOutOfBounds(index, count);
    return Vector::get(count-index-1);
  }
};



#endif // H_DS_STACK