/* Quat - A 3D fractal generation program */ 
/* Copyright (C) 1997,98 Dirk Meyer */ 
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */ 
/* mail:  Dirk Meyer */ 
/*        Marbacher Weg 29 */ 
/*        D-71334 Waiblingen */ 
/*        Germany */ 
/* */ 
/* This program is free software; you can redistribute it and/or */ 
/* modify it under the terms of the GNU General Public License */ 
/* as published by the Free Software Foundation; either version 2 */ 
/* of the License, or (at your option) any later version. */ 
/* */ 
/* This program is distributed in the hope that it will be useful, */ 
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */ 
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */ 
/* GNU General Public License for more details. */ 
/* */ 
/* You should have received a copy of the GNU General Public License */ 
/* along with this program; if not, write to the Free Software */ 
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */ 
 
#include <math.h>
#include <qpainter.h>
//#include "MandelPreview.h"
#include "MandelPreview.moc"
#include "common.h"
#include "qmath.h"

MandelPreview::MandelPreview(QWidget *parent, const char *name) : QWidget(parent, name)
{
   LButtonPressed = FALSE;
   len_x = 80; len_y = 60; mandel_x = -1; mandel_y = -1.5;
   mandel_lx = 4; mandel_ly = 3;
   Pixmap = new QPixmap((int)len_x, (int)len_y, -1);
   updated = FALSE;
   CHECK_PTR(Pixmap);
}

void MandelPreview::setCx(QString s)
{
   int x, y; 
//   QRect qr(0,0,(int)len_x,(int)len_y);
//   QPaintEvent qp(qr);

   cx = s.toDouble(NULL);
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
//   paintEvent(&qp);
   repaint(FALSE);
}

void MandelPreview::setCy(QString s)
{
   int x, y;
//   QRect qr(0,0,(int)len_x,(int)len_y);
//   QPaintEvent qp(qr);

   cy = s.toDouble(NULL);
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
//   paintEvent(&qp);
   repaint(FALSE);
}

void MandelPreview::setCk(QString s)
{
   ck = s.toDouble(NULL);
   CheckUpdate();
}

void MandelPreview::setCl(QString s)
{ 
   cl = s.toDouble(NULL);
   CheckUpdate();
}

void MandelPreview::setMaxiter(QString s)
{
   Maxiter = s.toInt(NULL);
   CheckUpdate();
}

void MandelPreview::setBailout(QString s)
{
   Bailout = s.toDouble(NULL);
   Bailout *= Bailout;
   CheckUpdate();
}

void MandelPreview::setFormula(int i)
{
   Formula = i;
   CheckUpdate();
}

void MandelPreview::MoveLeft()
{
   int x, y;

   mandel_x -= mandel_lx/3;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::MoveRight()
{
   int x, y;

   mandel_x += mandel_lx/3;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::MoveUp()
{
   int x, y;

   mandel_y -= mandel_ly/3;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::MoveDown()
{
   int x, y;

   mandel_y += mandel_ly/3;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::ZoomIn()
{
   int x, y;

   mandel_x += mandel_lx/4;
   mandel_y += mandel_ly/4;
   mandel_lx /= 2;
   mandel_ly /= 2;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::ZoomOut()
{
   int x, y;

   mandel_x -= mandel_lx/2;
   mandel_y -= mandel_ly/2;
   mandel_lx *= 2;
   mandel_ly *= 2;
   C2Coo(cx, cy, &x, &y);
   SetMyCursor(x, y);
   CalcImage();
   repaint();
}

void MandelPreview::SetMyCursor(int x, int y)
{
   if (x>len_x || x<0) x = -10;
   if (y>len_y || y<0) y = -10;
   CursorX = x; CursorY = y;
}

void MandelPreview::C2Coo(double cx, double cy, int *x, int *y)
{
   *x = (int)floor((cx - mandel_x) / mandel_lx * len_x + 0.5);
   *y = (int)floor((cy - mandel_y) / mandel_ly * len_y + 0.5);
   if (CursorY>len_y || CursorY<0) CursorY = -10;
   if (CursorX>len_x || CursorX<0) CursorX = -10;
   return;
}
void MandelPreview::Coo2C(int x, int y, double *cx, double *cy)
{
   *cx = mandel_x + (double)x/len_x * mandel_lx;
   *cy = mandel_y + (double)y/len_y * mandel_ly;
}

void MandelPreview::paintEvent(QPaintEvent *QP)
{
   QPainter painter(this);
   QRect R;
   R = QP->rect();
   if (R.right()>(int)len_x-1) R.setRight((int)len_x-1);
   if (R.bottom()>(int)len_y-1) R.setBottom((int)len_y-1);
   painter.setClipRect(R);
   painter.drawPixmap(0, 0, *Pixmap, 0, 0, -1, -1);
   painter.setPen(QColor(255,0,0));
   painter.drawLine(CursorX-1, CursorY, CursorX-3, CursorY);
   painter.drawLine(CursorX+1, CursorY, CursorX+3, CursorY);
   painter.drawLine(CursorX, CursorY-1, CursorX, CursorY-3);
   painter.drawLine(CursorX, CursorY+1, CursorX, CursorY+3);
   if (!updated) painter.drawRect(0, 0, (int)len_x, (int)len_y);
   painter.end();
   painter.flush();
}

void MandelPreview::mousePressEvent(QMouseEvent *QM)
{
   QString s;

   if (QM->button() == LeftButton) 
   {
      LButtonPressed = TRUE;
      setCursor(blankCursor);
      if (QM->x()>=0 && QM->x()<len_x && QM->y()>=0 && QM->y()<len_y)
      {
         Coo2C(QM->x(), QM->y(), &cx, &cy);
         s.setNum(cx, 'g', 12);
         emit cxChanged((const char *)s);
         s.setNum(cy, 'g', 12);
         emit cyChanged((const char *)s);
         SetMyCursor(QM->x(), QM->y());
      }
   }
   else if (QM->button() == RightButton)
   {
      CalcImage();
   }
}

void MandelPreview::mouseMoveEvent(QMouseEvent *QM)
{
   QString s;
   int x, y;

   if (!LButtonPressed) return;
   x = QM->x(); y = QM->y();
   if (x>=0 && x<(int)len_x && y>=0 && y<(int)len_y)
      setCursor(blankCursor);
   else setCursor(arrowCursor);
   if (x<0) x = 0; if (x>=(int)len_x) x = (int)len_x-1;
   if (y<0) y = 0; if (y>=(int)len_y) y = (int)len_y-1;
   Coo2C(x, y, &cx, &cy);
   s.setNum(cx, 'g', 12);
   emit cxChanged((const char *)s);
   s.setNum(cy, 'g', 12);
   emit cyChanged((const char *)s);
   SetMyCursor(x, y);
}

void MandelPreview::mouseReleaseEvent(QMouseEvent *QM)
{
   if (QM->button() == LeftButton) 
   {
      LButtonPressed = FALSE;
      setCursor(arrowCursor);
   }
}

double MandelPreview::GetCursorX()
{
   return mandel_x + (double)CursorX / len_x * mandel_lx;
}

double MandelPreview::GetCursorY()
{
   return mandel_y + (double)CursorY / len_y * mandel_ly;
}

int MandelPreview::CalcMPixel(int x, int y)
{
   double xr, yr, p[4], it, q[4]; 
   point x1, x2, x3, x4, x5, c;  
   point one = { 1, 0, 0, 0 }; 
   int i; 
    
   if (Formula != 0 && Formula != 1) return(0); 
   Coo2C(x, y, &xr, &yr);
   it = 0;  
   if (Formula==0) 
   { 
      forall(i,4) p[i] = 0; 
      forall(i,4) q[i] = 0;  
   } 
   else if (Formula==1) 
   { 
      forall(i,4) 
      { 
         x1[i] = 0; 
         x1[0] = 0.5; 
         q[i] = x1[i]*x1[i]; 
      }                     
      c[0] = xr; c[1] = yr; c[2] = ck; c[3] = cl; 
   } 
   while (it<Maxiter && q[0]+q[1]+q[2]+q[3]<Bailout) 
   { 
      if (Formula==0) 
      {  
         p[1] = 2*p[0]*p[1] - yr; 
         p[2] = 2*p[0]*p[2] - ck; 
         p[3] = 2*p[0]*p[3] - cl; 
         p[0] = q[0] - q[1] - q[2] - q[3] - xr; 
         forall(i,4) q[i] = p[i] * p[i]; 
      } 
      else if (Formula==1) 
      { 
         forall(i,4) x3[i] = x1[i]; 
         q_sub(x2, one, x1);      /* 1 - x[n] */ 
         q_mul(x4, c, x3);        /* c*x[n] */ 
         q_mul(x5, x4, x2);       /* c*x[n]*(1-x[n]) */ 
         forall(i,4) x1[i] = x5[i]; 
         forall(i,4) q[i] = x1[i]*x1[i]; 
      }  
      it++; 
   } 
   if (it<Maxiter) return(0); else return(1); 
}

void MandelPreview::CalcImage()
{
   int x, y;
   QPainter paint;
   QWidget *w;

   w = topLevelWidget();
   w->setCursor(waitCursor);
   setCursor(waitCursor);
   Pixmap->fill(QColor(255,255,255));
   paint.begin(Pixmap);
   for (y=0; y<(int)len_y; y++)
      for (x=0; x<(int)len_x; x++)
         if (CalcMPixel(x, y)) paint.drawPoint(x, y);
   pic_ck = ck; pic_cl = cl; pic_Maxiter = Maxiter; pic_Bailout = Bailout;
   pic_Formula = Formula;
   w->setCursor(arrowCursor);
   setCursor(arrowCursor);
   updated = TRUE;
   paint.end();
   paint.flush();
   repaint();
}

void MandelPreview::CheckUpdate()
{
   bool old;

   old = updated;
   if (pic_ck != ck || pic_cl != cl || pic_Maxiter != Maxiter
      || pic_Bailout != Bailout || pic_Formula != Formula)
   updated = FALSE; else updated = TRUE;
   if (old != updated) repaint();
}
