unit GLE2000G;
{ Graphs Unit. Version 3.2 12th May 2000.
  Covers operations involved in drawing a graph}
interface
uses
  Windows, SysUtils, WinTypes, WinProcs,
  Messages, Classes, Graphics, Controls, Forms, Dialogs,
  Tabs, ExtCtrls, StdCtrls, Menus, GLE2000D, GLE2000T;

const  MaxNames= 50;

type
   DataItem=record
     DataStatus:char;
     Value:double;
   end;
   tAxis=record        {Data for linear axes or curved angle axes}
     xstart,ystart,xend,yend,  {Coordinates of end points}
     xcentre,ycentre,
     InnerRadius, OuterRadius, { Inner and out limits of aaxis}
     AxisStartAngle,AxisEndAngle, {used only by polar and angle axes}
     StartAt,StepBy,EndAt, {Parameters used for multiple axes}
     AxisLength,AxisAngle,    { The actual length in cms for linear axes
      or the angular length in degrees for curved axes.}
     ScaleHi,ScaleLo, {Scale limits. in case of angle axis these are in
      degrees or radians. Basic axis measures always in degrees.}
     {For angular axes tick postions etc are in degrees or radians.}
     TickStart,SubTickStart,MidTickStart,        {First tick positions}
     dTicks,dSubTicks,dMidTicks,                 {Tick spacings}
     TickLength,SubTickLength,MidTickLength,
     TickLineWidth,SubTickLineWidth,MidTickLineWidth,
     TickAngle,
     GridStart,SubGridStart,MidGridStart,        {Grid Starting Points}
     dGrid,dSubGrid,dMidGrid,         {Grid spacings}
     GridLength,MidGridLength,SubGridLength,
     GridLineWidth,SubGridLineWidth,MidGridLineWidth,
     GridAngle,TitleAngle,
     NameDist,NameShift,NameAngle, {Name shift in degs/rads for angle axes}
     AxisLineWidth,
     MidLabelShift,MidLabelDist,  {Label shift in degs/rads for angle axes}
     LabelShift,LabelDist,LabelAngle,
     TitleHei,MidLabelHei,LabelHei,NameHei,      {Text heights in cm}
     TitleShift,TitleDist:double;  {Title position, both in cms}

     nTicks,nSubTicks,nMidTicks,                 {Number of ticks}
     nGrids,nSubGrids,nMidGrids,                 {Number of Grid lines}
     AxisType,                                   {Axis Type}
     TickColor,SubTickColor,MidTickColor,        {Tick colours}
     GridColor,SubGridcolor,MidGridColor,        {Grid colours}
     AxisColor,NameColor,
     NumOfNames,NumOfPlaces,                     {Size of arrays}
     MidLabelSFigs,MidLabelDFigs,
     LabelSFigs,LabelDFigs,
     TitleColor,MidLabelColor,LabelColor:integer; {Title colours}
     TickStyle,SubTickStyle,MidTickStyle,        {Tick styles}
     GridStyle,SubGridStyle,MidGridStyle,        {Grid and tick line styles}
     AxisLineStyle:TPenStyle;
     Title,TitleFontName,LabelFontName,MidLabelFontName,NameFontName:string;
     AxisLabels,ANames:array[0 ..MaxNames] of string;
     ShowName,ShowPlace,ShowMidPlace:array[0 .. 50] of boolean;
     AxisOff,SideOn,Negate,PolarAxis,AngleAxis,                        {Flags}
     NoFirst,NoLast,LabelsOn,MidLabelsOn,
     GridOn,TicksOn,MidTicksOn,SubTicksOn,
     SymmetricTicks,SymmetricMidticks,SymmetricSubticks,
     MidGridOn,SubGridOn,NamesOn:Boolean;
   end; {tAxis}

   gdata=record
     name:string;
     dcol:array[0 ..99] of Byte;
   end;

   plotset=record
     xaxis,yaxis,zaxis,       {defines either x1axis or x2axis etc.}
     xcol,ycol,NumOfPoints,   {would be safe to use raxis and aaxis as well}
     mborder,mlcolor,mfcolor,msNumber,
     pcolor,pstyle,smoothvalue:integer;
     errwidth,errup,errdown,
     herrwidth,herrleft,herrright,
     pwidth,msize:double;
     mstyle:string;
     markeron,lineon,erron,
     smooth:boolean;
   end;

   PlotTransform=record
     gtype:integer;
     gta,gtb,gtc,     {Plot to paper coordinate transform }
     gtd,gte,gtf:double;
     xIsLog,yIsLog:Boolean;
   end;

   pdata= array[0 .. 99] of plotset;
   Datacol=array[ 0 .. 250] of DataItem;
   DataSet=array [0 .. 99] of DataCol;

var  gxsize,gysize,             {Graph Dimensions}
     gstartx,gstarty,           {Start point for graph box}
     gxoffset,gyoffset,         {Offset of graph origin }
     GTitleDist,GTitleOffset,   {Graph Title Position}
     gxorigin,gyorigin,         {Graph Origin}
     hscale,vscale,             {Axis scale factors}
     centreradius,              {Radius of central circle for polar plot}
     DataAtX,DataAtY,           {Position of data display}
     DColWidth,                 {Data display column width}
     GTitleHei:double;          {Title Height}
     GTitleColor,
     CentreLineColor, CentreFillColor, {Colours for central circle}
     DataColNum,NumOfDataCols,         {Data Column settings}
     CurrentDstart,CurrentDEnd,
     LastDStart,LastDEnd,
     DataRowNum,NumOfDataRows,
     DColStart,DColEnd,DColstep,
     DRowStart,DRowEnd,DRowStep:integer;   {Range of data display}
     x1axis,x2axis,y1axis,y2axis,z1axis,z2axis,r1axis,r2axis,
     r3axis,r4axis,CurrentAxis:tAxis;
     a1Axis,a2Axis:tAxis;
     NoBox,FullSize,DataDefined,ShowPlotData:Boolean;
     GTitle,GTitleFontName,
     DataSetName:string;             {Graph Title}
     Linc:array[0..19] of double;
     CurrentPenStyle:TPenStyle;
     CurrentBrushStyle:TBrushStyle;
     DataCol1:dataCol;
     Data:DataSet;
     PlotIsOn:array[0 .. 99] of Boolean;
     PlotData:pdata;       {The array of plotting data sets }
     CurrentPlot:plotset;  {The current set of plot data}
     GraphTransform,ScaleTransform:PlotTransform;

 procedure AutoScale(DataHi,DataLo:double;VAR ScaleHi,
                   ScaleLo:double;VAR NumOfSeg,ScaleType:integer);
{ Takes as input upper and lower data limits DataHi,DataLo and scale type
  ScaleType ( 1 for linear, 2 for log, 3 for probability scale ).
  Resets ScaleType if necessary and returns suitable upper and lower
  scale limits ScaleHi,ScaleLo. NumOfSeg is no. of segments in a linear scale
  or no. of decades*100 in a log. scale   }

  procedure DrawAxis(var Axis:tAxis);
  { Draws a linear or logarithmic axis specified by:
    ScaleLo - lower value of axis range
    ScaleHi - upper limit of axis range
    xstart,ystart - co-ordinates of start of axis
    xend,yend     - co-ordinates of end of axis,
    dStart - first point for ticks,
    dTicks - distance between ticks
    dSubStart,dSubticks - same for subticks
    dMidStart,dMidTicks - same for middle ticks
    TickAngle - Angle of ticks relative to axis
    Mark1Leng,Mark2Leng,Mark3Leng - scale marker lengths
    LogStyle - Submark style for a log axis
    AxisType - axis type: 1 = linear, 2 = logarithmic
     3 = probability,  4 = sinh,  9 = Linear polar scale,
     10 = log polar scale, 11 = prob polar scale,
     12 = sinh polar scale, 17 = linear polar angle }

procedure DrawAngleAxis(var AngAxis:TAxis);
{Draws an angular axis}

procedure SetAxisDefaults;
{Resets all axis parameters to default values}

procedure SetAngleAxisDefaults;
{Resets all angle axis parameters to their default values}

procedure BSpline(StepSize:double;Fit,NumOfCpts:integer;
                 PlotLeft,PlotRight,PlotLo,PlotHi:double;
                  Cpoint:array of RealPoint);
  {Draws a B-spline curve starting from Cpoint[0] and ending at
   CPoint[NumOfPts]. The intermediate points in the array Cpoint
   serve as control points. The parameter StepSize sets the size
   of the steps between each control point. (e.g. StepSize = 0.05
   sets 20 steps between contro points.) Fit is a parameter which
   selects the fitting strategy. Fit = 0 uses a basic B-spline
   algorithm with end points not fixed. Fit = 1 forces the curve to
   start at Cpoint[0] and end at Cpoint[NumOfCpts]. Increasing Fit
   up to 10 forces closer fitting to the straight lines through the
   control points. Fit between 11 and 20 forces curve through the
   control points. Fit betqween 21 and 30 does the same using a
   different algorithm. The curve is drawn using the current pen
   color, style etc. and using the current coordinate transform
   settings.}

procedure PlotToPaper(xd,yd:double;CurrentGraphTransform:PlotTransform;
                      var xp,yp:double);
{Converts plotter data co-ordinates xd,yd to paper co-ordinates
 xp,yp using the conversion parameters gca,gcb,gcc,gcd,gce,gcf,
 xIsLog,yIslog from the current data transform. }

function ClipMode(xy, lo, hi:double):integer;
{ Determines location of value relative to clipping limits}

 procedure SetPlotDefaults(PlotNumber:integer);
{Resets  a plot data record to default settings }

procedure SetUpGraph;
{At the start of a graph block this procedure
 performs initialisation operations for the graph}

procedure SetUpPolar;
{At the start of a Polar plot block this procedure
 performs initialisation operations for the plot}

procedure SetGraphTransform(xaxis,yaxis:integer;
                  var CurrentGraphTransform:PlotTransform;xofs,yofs:double);
{sets up Transform parameters for Graph to Paper.
 xaxis & yaxis define axes used.}

procedure ConstructGraph;
{At end f a graphics block this procedure uses the
 input to the block to construct the graph and plot
 the data.}

procedure ConstructPolarGraph;

Procedure InsertPlots(PlotHi,PlotLo,PlotLeft,PlotRight:double;
                       PolarPlot:Boolean);
        {Insets plots as set up in constructGraph or ConstructpolarGraph}

 procedure Inputdata(var Colnum:integer;DataSetName:string);
{Organises input of data to DataSet Data}

 procedure CheckData(xs,ys,cwidth:double;Rstart,Rend,Rstep,Cstart,Cend,Cstep:integer);
 { Display of a section of the data array
  Locate top left at point x,y.  Display from Rstart to Rend in steps of
  Rstep. From Cstart to Cend in steps of Cstep . Columns are spaced by cwidth}

implementation

var ig,jg,kg:integer;
xg,yg,x1g,y1g:double;

function Entier(x:double):integer;
{ Returns the largest integer less than X }
begin
 if x>=0 then Entier:=trunc(x) else Entier:=trunc(x)-1;
end;  {Entier}

procedure Mant(x:double;VAR Mantissa:DOUBLE;VAR Exponent:INTEGER);
{ Extracts the mantissa and exponent of a real number X
  Version 1. D. Ingram.  3/8/87                         }

var zz:double;

begin
 if x=0 then
 begin
  Mantissa:=0;Exponent:=0;
 end else begin
  if x>0 then Exponent:=1 else Exponent:=-1;
  zz:=LogE*LN(Abs(x));  {Log10 of x}
  Mantissa:=Exponent*exp(Frac(zz)/logE); {Antilog of fractional part of zz}
  Exponent:=Entier(zz);  {Rounded down}
 end; {if x=0}
end;  {Mant}

procedure AutoScale(DataHi,DataLo:double;VAR ScaleHi,
                   ScaleLo:double;VAR NumOfSeg,ScaleType:integer);

{ Takes as input upper and lower data limits DataHi,DataLo and scale type
  ScaleType ( 1 for linear, 2 for log, 3 for probability scale ).
  Resets ScaleType if necessary and returns suitable upper and lower
  scale limits ScaleHi,ScaleLo. NumOfSeg is no. of segments in a linear scale
  or no. of decades*100 in a log. scale   }

var Seg,y,z,DBase:double;
    ISclHi,ISclLo,ix,iy,iz:integer;
    Scale:array[1..9] of real;

begin
{ First set up array Scale and default for reference point.}
 Scale[1]:=0.1;Scale[2]:=0.2;Scale[3]:=0.5;Scale[4]:=0.5;Scale[5]:=0.5;
 Scale[6]:=1.0;Scale[7]:=1.0;Scale[8]:=1.0;Scale[9]:=1.0;
 DBase:=0.0;
{ Set ScaleType to linear scale if range small or if the range
  goes  through zero. Also, if the ratio DataHi/DataLo is close to
  1 set a reference point DBase for later operations in order
  avoid integer overflows.  }
 if DataLo<=0 then ScaleType:=1 else if DataHi/DataLo<5 then ScaleType:=1;
 if ScaleType=1 then    { Linear scale }
 begin
  if DataLo<>0 then     { Set a lower limit to scale range  }
  if (DataHi/DataLo<1.001) and (DataHi/DataLo>0.999) then
  begin
   DataHi:=DataHi*1.001;DataLo:=DataLo*0.999;
   { Now set reference point below DataLo  }
   Mant(2*DataLo-DataHi,z,iz);
   DBase:=Entier(z)*exp(iz/logE);
  end;
  { Check settings for DataHi and DataLo  }
  if (DataLo<=0.0) and (DataHi<=DataLo) then
  begin
   DataHi:=DataLo+0.01;
   DataLo:=DataLo-0.01;
  end;
  Mant(DataHi-DataLo,z,iz);
  iy:=Entier(z);
  { iy sets number of units in range. Now choose a suitable
    rounded scale interval to match this. }
  Seg:=Scale[iy]*exp(1/LogE);
  { Find number of segments from reference level DBase to upper
    and lower limits of range. Round these off to the next
    highest and lowest integers respectively. }
  ISclHi:=-Entier(-(DataHi-DBase)/Seg);
  ISclLo:=Entier((DataLo-DBase)/Seg);
  { Find upper and lower scale limits corresponding to the
    rounded off scale lengths.  }
  ScaleHi:=ISclHi*Seg+DBase;ScaleLo:=ISclLo*Seg+DBase;
  { Check rounding against original values of DataHi and DataLo }
  while (ScaleHi-DataHi)>0.95*Seg do
   begin
     ScaleHi:=ScaleHi-Seg;
     ISclHi:=pred(ISclHi);
   end;
  while (DataLo-ScaleLo)>0.95*Seg do
   begin
     ScaleLo:=ScaleLo+Seg;
     ISclLo:=succ(ISclLo);
   end;
  { Make sure these are not equal which would give zero scale
    size  }
  if (ISclHi=ISclLo) then
   begin
     ISclLo:=pred(ISclLo);
     ScaleLo:=ScaleLo-Seg;
   end;
  { Finally set up number of segments. }
  NumOfSeg:=ISclHi-ISclLo;
 end else
 begin
  { Set up of logarithmic axis and round off to nearest
    power of 10 for upper and lower limits of range   }
  Mant(DataHi,z,iz);
  ISclHi:=-Entier(-z);
  { Round up upper scale limit if just above or below 10 }
  if (z<1.1) then ISclHi:=1;
  if (ISclHi>8) then
  begin
    ISclHi:=1;iz:=iz+1;
  end;
  ScaleHi:=ISclHi*exp(iz/LogE);
  { Now set lower axis limit }
  Mant(DataLo,y,iy);
  ISclLo:=Entier(y);
  if (ISclLo>8) then
   begin
     ISclLo:=1;iy:=succ(iy);
   end;
  ScaleLo:=ISclLo*exp(IY/LogE);
  { Multiply integer NumOfSeg by 100 to allow 2 decimal places
    in later processing using this value. }
  NumOfSeg:=round(LogE*100*Ln(ScaleHi/ScaleLo));
 end;
end;  {Autoscale}

procedure DrawAxis(var Axis:tAxis);
{ Draws a linear or logarithmic axis specified by:
  ScaleLo - lower value of axis range
  ScaleHi - upper limit of axis range
  xstart,ystart - co-ordinates of start of axis
  xend,yend     - co-ordinates of end of axis,
  dStart - first point for ticks,
  dTicks - distance between ticks
  dSubStart,dSubticks - same for subticks
  dMidStart,dMidTicks - same for middle ticks
  TickAngle - Angle of Ticks relative to axis
  Mark1Leng,Mark2Leng,Mark3Leng - scale marker lengths
  LogStyle - Submark style for a log axis
  AxisType - axis type: 1 = linear, 2 = logarithmic  3 = probability
      4 = sinh,  5 = polar angle      }

VAR  x,x1,x2,y2,x3,r1,r2,rx,ry,ScaleAxis,ScaleRatio,LogHi,LogLo,LogBottom,
     scale1,scalelength,xscale,yscale,
     xmark1,ymark1,xmark2,ymark2,xmark3,ymark3,xmark4,ymark4,xGrid,yGrid,
     xdstart,ydstart,xdSubStart,ydSubStart,xdMidStart,ydMidStart,
     GridStartAngle, GridEndAngle,  { ************CHECK************}
     xdGridStart,ydGridStart,xdMidGridStart,ydMidGridStart,
     xdSubGridStart,ydSubGridStart,xdGrid,ydGrid,xdSubGrid,ydSubGrid,
     xdMidGrid,ydMidGrid,xdTicks,ydTicks,xdSubTicks,ydSubTicks,
     xdMidTicks,ydMidTicks,LoLimit,HiLimit,LastAngle,MidTickVal:double;
     i,inc1,inc2,istart,istop,TickNum,MidTickNum,SExp:integer;
     SText:String;
     TitleFont,LabelFont,MidLabelFont,NameFont:TFont;
begin
 with Axis do
 begin
  GridStartAngle:= startat; GridEndAngle:= EndAt;
  if (Abs(EndAt-StartAt)>=360) then GridEndAngle:= GridStartAngle+359.999;
  AxisAngle:= atan2(xend-xstart,yend-ystart);
  AxisLength:= Sqrt(Sqr(xend-xstart)+Sqr(yend-ystart));

 {Set Marker increments}
  xscale:=  cos(AxisAngle+TickAngle);  {N.B. xscale, yscale set for  }
  yscale:=  sin(AxisAngle+TickAngle);  { ticks, reset for other cases}
  xmark1:=  yscale*TickLength;                  { in later sections.          }
  ymark1:=  xscale*TickLength;
  xmark2:=  yscale*SubTickLength;
  xmark4:=  0.6*xmark2;
  ymark2:=  xscale*SubTickLength;
  ymark4:=  0.6*ymark2;
  xmark3:=  yscale*MidTickLength;
  ymark3:=  xscale*MidTickLength;
  xGrid :=  sin(AxisAngle+GridAngle)*GridLength;
  yGrid :=  cos(AxisAngle+GridAngle)*GridLength;

  if AxisType=1 then      {Linear Axis}
  begin
  ScaleLength:= ScaleHi-ScaleLo;
  HiLimit:= ScaleHi+Abs(ScaleLength)*0.0001; {Used to avoid roundoff errors}
  LoLimit:=ScaleLo-Abs(ScaleLength)*0.0001;  {when drawing scale end ticks.}
  ScaleAxis:= AxisLength/ScaleLength;
 {Set scale increments}
  xscale:=  ScaleAxis*cos(AxisAngle);   {xscale and yscale reset for axes }
  yscale:=  ScaleAxis*sin(AxisAngle);
  if nticks>0 then  {Setting nTicks overrides dTicks setting}
  begin
    Tickstart:= ScaleLo;
    dTicks:= ScaleLength/nTicks;
  end;
  xdStart:= xscale*(TickStart-ScaleLo)+xStart;
  ydStart:= yscale*(TickStart-ScaleLo)+ystart;
  xdTicks:= xscale*dTicks;
  ydTicks:= yscale*dTicks;

  if GridOn then
  begin
  { The grid is drawn before the axis so that axis overwrites it.
    First draw subgrid lines, then Midgrid line, and then grid lines.
    Finally add a border in case other axes not present }
   if nGrids>0 then  {Setting nGrids overrides dGrid setting}
   begin
    GridStart:= ScaleLo;
    dGrid:= ScaleLength/nGrids;
   end;
   xdGridStart:= xscale*(GridStart-ScaleLo)+xstart;
   ydGridStart:= yscale*(GridStart-scaleLo)+ystart;
   xdGrid:= xscale*dGrid;
   ydGrid:= yscale*dGrid;
   if ((dSubGrid<>0.0) or (nSubGrids>0)) and SubGridOn then  {Draw subGrids}
   begin
    DisplayForm.Image1.canvas.pen.color:= SubGridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*SubGridLinewidth);
    DisplayForm.Image1.canvas.pen.style:= SubGridStyle;
    if nSubGrids>0 then    {nSubGrids overrides setting of dSubgrid}
    begin
      dSubGrid:= dGrid/nSubGrids;
      SubGridStart:= GridStart;
    end;
    xdSubGridStart:=  xscale*(SubGridStart-ScaleLo)+xstart;
    ydSubGridStart:=  yscale*(SubGridStart-ScaleLo)+ystart;
    xdSubGrid:=  xscale*dSubGrid;
    ydSubGrid:=  yscale*dSubGrid;
    x:=SubGridStart;
    if PolarAxis then begin
     rx:= xdSubGridstart-xcentre;
     ry:= ydSubGridStart-ycentre;
     amove(xcentre,ycentre);
    end else amove(xdSubGridstart,ydSubGridstart);
    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      if polaraxis then
      begin
       r1:= sqrt(sqr(rx)+sqr(ry));
       amove(xcentre,ycentre);
       arcat(r1,r1,GridStartAngle,GridEndAngle);
       rx:= rx+xdSubGrid;
       ry:= ry+ydSubGrid;
      end else begin
       rline(xGrid,yGrid);
       rmove(-xGrid,-yGrid);
      end;  {if Ploaraxis}
     end;  {if x>=LoLimit ...}
     rmove(xdSubGrid,ydSubGrid);
     x:=x+dSubGrid;
    until (x>HiLimit) or (x<LoLimit);
   end;
   if ((dMidGrid<>0.0) or (nMidGrids>0)) and MidGridOn then
           {Draw MidGrids similar to SubGrids}
   begin
    DisplayForm.Image1.canvas.pen.color:= MidGridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*MidGridLineWidth);
    DisplayForm.Image1.canvas.pen.style:= MidGridStyle;
    if (nMidgrids>0) then
    begin
      dMidGrid:=dGrid/nMidGrids;
      MidGridstart:=GridStart;
    end;
    xdMidGridStart:=  xscale*(MidGridStart-ScaleLo)+xstart;
    ydMidGridStart:=  yscale*(MidGridStart-ScaleLo)+ystart;
    xdMidGrid:=  xscale*dMidGrid;
    ydMidGrid:=  yscale*dMidGrid;
    x:=MidGridStart;
    if PolarAxis then begin
     rx:= xdMidGridstart-xcentre;
     ry:= ydMidGridStart-ycentre;
     amove(xcentre,ycentre);
    end else amove(xdMidGridstart,ydMidGridstart);
    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      if polaraxis then
      begin
       r1:= sqrt(sqr(rx)+sqr(ry));
       amove(xcentre,ycentre);
       arcat(r1,r1,GridStartAngle,GridEndAngle);
       rx:= rx+xdMidGrid;
       ry:= ry+ydMidGrid;
      end else begin
       rline(xGrid,yGrid);
       rmove(-xGrid,-yGrid);
      end;
     end;
     rmove(xdMidGrid,ydMidGrid);
     x:=x+dMidGrid;
    until (x>HiLimit) or (x<LoLimit);
   end; {if dMidGrid}
   DisplayForm.Image1.canvas.pen.color:= GridColor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLineWidth);
   DisplayForm.Image1.canvas.pen.style:= GridStyle;
   x:= GridStart;      {Draw Grids - same for both linear and log axes}
   if PolarAxis then begin
     rx:= xdgridstart-xcentre;
     ry:= ydGridStart-ycentre;
     amove(xcentre,ycentre);
    end else amove(xdGridstart,ydGridstart);
   repeat
    if (x>=LoLimit) and (x<=HiLimit) then
    begin
     if polaraxis then
     begin
      r1:= sqrt(sqr(rx)+sqr(ry));
      amove(xcentre,ycentre);
      arcat(r1,r1,GridStartAngle,GridEndAngle);
      rx:= rx+xdGrid;
      ry:= ry+ydGrid;
     end else begin
      rline(xGrid,yGrid);
      rmove(-xGrid,-yGrid);
     end;
    end;
    rmove(xdGrid,ydGrid);
    x:=x+dGrid;
   until (x>=HiLimit)or (x<LoLimit);
   { Draw axis line  }
   DisplayForm.Image1.canvas.pen.color:= GridColor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLineWidth);
   DisplayForm.Image1.canvas.pen.style:= GridStyle;
   Amove(xstart,ystart);
   if polaraxis then begin
     Aline(xend,yend);
     Amove(xstart,ystart);
   end else begin   {Do not draw grid lines for polar axes}
     Aline(xstart+xGrid,ystart+yGrid);
     Aline(xend+xGrid,yend+yGrid);
     Aline(xend,yend);
     Aline(xstart,ystart);
     DisplayForm.Image1.canvas.pen.color:=Black;
   end;
  end;  {grid}

  if (not AxisOff) then
  begin
  {First draw sub ticks, then Midticks, ticks, grids and finally axis line.
   This sequence ensures colours, styles etc are not overwritten. }
   if  SubTicksOn and ((dSubTicks<>0.0) or (nSubTicks>0)) then   {Draw subticks}
   begin
    DisplayForm.Image1.canvas.pen.color:= SubTickColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*SubTickLinewidth);
    DisplayForm.Image1.canvas.pen.style:= SubTickStyle;
    if nSubTicks>0 then
    begin
     dSubTicks:= dTicks/nSubTicks;
     SubTickStart:= TickStart;
    end;
    xdSubStart:=  xscale*(SubTickStart-ScaleLo)+xstart;
    ydSubStart:=  yscale*(SubTickStart-ScaleLo)+ystart;
    xdSubTicks:=  xscale*dSubTicks;
    ydSubTicks:=  yscale*dSubTicks;
    amove(xdSubStart,ydSubStart);
    x:=SubTickStart;
    repeat
    if (x>=LoLimit) and (x<=HiLimit) then
    begin
     if SymmetricSubticks then
     begin
      rline(xmark2,-ymark2);
      rmove(-xmark2,ymark2);
     end;
     rline(-xmark2,ymark2);
     rmove(xmark2,-ymark2);
    end;
     rmove(xdSubTicks,ydSubTicks);
     x:=x+dSubTicks;
    until (x>HiLimit) or (x<LoLimit);
   end;
   if (dMidTicks<>0.0) or (nMidTicks>0) then   {Draw midpoint ticks}
   begin
    MidTickNum:=0;
    if NoFirst then  ShowMidPlace[0]:=False;
    DisplayForm.Image1.canvas.pen.color:= MidTickColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPercm*MidTickLineWidth);
    DisplayForm.Image1.canvas.pen.style:= MidTickStyle;
    if (nMidTicks>0) then
    begin
      dMidticks:= dticks/nMidTicks;
    end;
    xdMidStart:=  xscale*(MidTickStart-ScaleLo)+xstart;
    ydMidStart:=  yscale*(MidTickStart-ScaleLo)+ystart;
    xdMidTicks:=  xscale*dMidTicks;
    ydMidTicks:=  yscale*dMidTicks;
    amove(xdMidStart,ydMidStart);
    CurrentFont.Name:=MidLabelFontName;
    CurrentFont.size:=Round(MidLabelHei);
    CurrentFont.Color:=MidLabelColor;
    Lastangle:= Textangle; TextAngle:= LabelAngle;
    x:=MidTickStart;
    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
     if MidTicksOn then
     begin
      if SymmetricMidticks then
      begin
      rline(xmark3,-ymark3);
      rmove(-xmark3,ymark3);
      end;
      rline(-xmark3,ymark3);
     end;
     if (x>Hilimit) or (x<LoLimit) then Showmidplace[MidTickNum]:= False;
     if ShowMidPlace[MidTickNum] then
     begin
       Str(x:MidLabelSfigs:MidLabelDFigs,SText);
       x2:= (MidLabelShift);
       y2:= (MidLabelDist);
       rmove(x2,y2);
       Text(SText,TextAngle);
       rmove(-x2,-y2);
     end;
     if MidTicksOn then rmove(xmark3,-ymark3);
     end;
     rmove(xdMidTicks,ydMidTicks);
     x:=x+dMidTicks;
     If MidTickNum<50 then MidTickNum:=succ(MidTickNum);
    until (x>HiLimit) or (x<LoLimit);
     CurrentFont.Name:='Times New Roman';
     CurrentFont.size:=Round(0.2*PixelsPerCm);       {**** Check This ***** }
     CurrentFont.Color:=black;
     TextAngle:= Lastangle;
   end; {if dMidTicks}
   TickNum:=0;
   If NoFirst then Showplace[0]:= False;
   DisplayForm.Image1.canvas.pen.color:= TickColor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*TickLineWidth);
   DisplayForm.Image1.canvas.pen.style:= TickStyle;
   x:=TickStart; {Draw ticks - same for both linear and log axes}
   amove(xdstart,ydstart);
   repeat
    if (x>=LoLimit) and (x<=HiLimit) then
    begin
     if TicksOn then
     begin
      if SymmetricTicks then
      begin
       rline(xmark1,-ymark1);
       rmove(-xmark1,ymark1);
      end;
      rline(-xmark1,ymark1);
     end;
     if NoLast and (x>=Hilimit-dticks) then ShowPlace[TickNum]:= False;
     if ShowPlace[TickNum] then
     begin
       CurrentFont.Name:=LabelFontName;
       CurrentFont.size:=Round(LabelHei);
       CurrentFont.Color:=LabelColor;
       Lastangle:= Textangle; TextAngle:= LabelAngle;
       Str(x:LabelSfigs:LabelDFigs,SText);
       x2:= (LabelShift);
       y2:= (LabelDist);
       rmove(x2,y2);
       Text(SText,TextAngle);
       rmove(-x2,-y2);
       CurrentFont.Name:='Times New Roman';
       CurrentFont.size:=Round(0.2*PixelsPerCm);      { *** Check This **** }
       CurrentFont.Color:=black;
       TextAngle:= Lastangle;
     end;
     if ShowName[TickNum] then
     begin
       CurrentFont.Name:=NameFontName;
       CurrentFont.size:=Round(NameHei);
       CurrentFont.Color:=NameColor;
       Lastangle:= Textangle; TextAngle:= NameAngle;
       x2:= (NameShift);
       y2:= (NameDist);
       rmove(x2,y2);
       Text(ANames[TickNum],TextAngle);
       rmove(-x2,-y2);
       CurrentFont.Name:='Times New Roman';
       CurrentFont.size:=Round(0.2*PixelsPerCm);      { **** Check This *** }
       CurrentFont.Color:=black;
       TextAngle:=Lastangle;
     end;
     if TicksOn then rmove(xmark1,-ymark1);
    end;
    rmove(xdTicks,ydTicks);
    x:=x+dTicks;
    if TickNum<50 then TickNum:=succ(TickNum)
   until (x>HiLimit) or (x<LoLimit);
   if LabelsOn then
    begin
     { NOT SET}
    end;
   { Draw axis line  }
   DisplayForm.Image1.canvas.pen.color:= AxisColor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*AxisLineWidth);
   DisplayForm.Image1.canvas.pen.style:= AxisLineStyle;
   Amove(xstart,ystart);
   if SideOn then Aline(xend,yend);
  end; {Axisoff}
   DisplayForm.Image1.canvas.pen.color:=Black;
  end;{if AxisType=1}

  if AxisType=2 then   {Log Scales}
  { Dticks or nticks determine the decade intervals. The
    primary scale subtick points for log scales are set
    in array Linc during initialization. The subticks actually
    displayed are determined by nSubTicks. dSubTicks, dMidTicks
    nMidTicks are ignored for log scales.}
  begin
   {Set scale limits }
   LogHi:= LogE*Ln(Abs(ScaleHi)); {Log10 of ScaleHi}
   LogLo:= LogE*Ln(Abs(ScaleLo)); {Log10 of ScaleLo}
   LogBottom:=Entier(logLo);       {Log on next lowest decade point}
   dTicks:= AxisLength/(LogHi-LogLo);  {Length of a decade}
   TickStart:= dTicks*(LogBottom-LogLo);{Increment to 1st tick below axis start}
   xscale:=  cos(AxisAngle);
   yscale:=  sin(AxisAngle);
   xdStart:= xstart+TickStart*xscale;
   ydstart:= ystart+TickStart*yscale;

  if GridOn then
  begin
  { The grid is drawn before the axis so that axis overwrites it.
    First draw subgrid lines, then Midgrid line, and then grid lines.
    Finally add a border in case other axes not present }
   dGrid:=dTicks;GridStart:=Tickstart;
   xdGridStart:=xdStart; ydGridStart:=ydstart;
   if nSubGrids>0 then  {Sets up subGrid spacings according to number of
                         subGrids specified. This overrides any actual
                         specification of subGrid spacing.}
   begin
    case nsubGrids of
         1:begin
            istart:=8;
            inc1:=10;
           end;
       2,3:begin
            istart:=3;
            inc1:=5;
           end;
         4:begin
            istart:=2;
            inc1:=4;
           end;
      6..9:begin
            istart:=2;
            inc1:=2;
           end;
    10..20:begin
            istart:=1;
            inc1:=1;
           end;
      else begin     {Default is to only show midpoint}
             istart:=8;
             inc1:=10;
           end;
    end; {case nsubGrids}
   end else if dSubGrid>0 then {evaluates log increments if number of
                                subGridss = 0 or ifsubGrid spacing defined }
   begin
    i:=round(2*dsubGrid);
    case i of
         1:begin       {DSubGrids=0.5 : nSubGrids > 9}
            istart:=1;
            inc1:=1;
            nSubGrids:= 10;
           end;
         2:begin       {DSubGrids=1.0 : nSubGrids = 9}
            istart:=2;
            inc1:=2;
            nSubGrids:=  9;
           end;
       3,4:begin       {DSubGrids=2.0 : nSubGrids = 4}
            istart:=2;
            inc1:=4;
            nSubGrids:=  4;
           end;
         5:begin      {DSubGrids=2.5 : nSubGrids = 3}
            istart:=3;
            inc1:=5;
            nSubGrids:=  3;
           end;
     6..10:begin      {DSubGrids=5.0 : nSubGrids = 1}
            istart:=8;
            inc1:=10;
            nSubGrids:=  1;
           end;
     else begin     {Default is 10 subdivisions}
            istart:=2;
            inc1:=2;
            nSubGrids:=  9;
          end;
    end; {case i}
   end else begin
      istart:=2;   {Overall default is a ten division axis }
      inc1:=2;
      nSubGrids:=  9;
   end;  {if nsubGrids or dsubGrids}
   xdGrid:= xscale*dGrid;
   ydGrid:= yscale*dGrid;
   x1:=exp(LogBottom/LogE);  {Reference point for scale}
   if PolarAxis then begin
     rx:= xdgridstart-xcentre;  {Sets initial radius}
     ry:= ydGridStart-ycentre;
     r1:= sqrt(sqr(rx)+sqr(ry))*Sign(rx); {N.B. Only works for ry = 0 }
    end else amove(xdGridstart,ydGridstart);
   {Insert first decade Grid if this is on the scale}
   if x1>=ScaleLo*0.99 then
   begin   {Draw Decade Grid}
    DisplayForm.Image1.canvas.pen.color:= GridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLineWidth);
    DisplayForm.Image1.canvas.pen.style:= GridStyle;
     if polaraxis then
     begin
      amove(xcentre,ycentre);
      arcat(r1,r1,GridStartAngle,GridEndAngle);
     end else begin
      rline(xGrid,yGrid);
      rmove(-xGrid,-yGrid);
     end;
   end;
   repeat {External loop for Grid drawing }
      x:=x1; x3:=x1; r2:= r1;
      i:=istart;
      inc2:=inc1;
      istop:=20;
      x:= x+x1*(istart/2); {Starting point for subGrids}
      if SubGridOn then begin{Condition for adding sub grids and midgrids}
      repeat  {Inner loop for drawing subGrids}
       DisplayForm.Image1.canvas.pen.color:= SubGridColor;
       DisplayForm.Image1.canvas.pen.width:=
                                Round(PixelsPerCm*SubGridLineWidth);
       DisplayForm.Image1.canvas.pen.style:= SubGridStyle;
       x2:= dGrid*LogE*Ln(Abs(x/x3));  {Calculate subGrid spacing}
       xdSubGrid:=  xscale*x2;         {Set subGrid coordinats}
       ydSubGrid:=  yscale*x2;
       if polaraxis then begin
         r2:= r2+x2;
         amove(xcentre,ycentre);
       end else rmove(xdSubGrid,ydSubGrid);
       if (x>=0.999*ScaleLo) and (x<=1.001*ScaleHi) then {and draw it}
       begin      {Select Grid length as appropriate}
        if NSubGrids>=0 then begin
         If (i mod 2 <> 0) and (i<8) then begin  {half subGrids}
          If polaraxis then  arcat(r2,r2,GridStartAngle,GridEndAngle)
          else begin
           rline(xGrid,yGrid);
           rmove(-xGrid,-yGrid);
          end;
         end;
         if (i=8) then begin   {MidGrid}
          DisplayForm.Image1.canvas.pen.color:= MidGridColor;
          DisplayForm.Image1.canvas.pen.width:=
                                   Round(PixelsPerCm*MidGridLineWidth);
          DisplayForm.Image1.canvas.pen.style:= MidGridStyle;
          If polaraxis then  arcat(r2,r2,GridStartAngle,GridEndAngle)
          else begin
           rline(xGrid,yGrid);
           rmove(-xGrid,-yGrid);
          end;
         end;
         if (i mod 2 = 0) or (i>8) then begin  {subGrids}
          DisplayForm.Image1.canvas.pen.color:= SubGridColor;
          DisplayForm.Image1.canvas.pen.width:=
                          Round(PixelsPerCm*SubGridLineWidth);
          DisplayForm.Image1.canvas.pen.style:= SubGridStyle;
          If polaraxis then  arcat(r2,r2,GridStartAngle,GridEndAngle)
          else begin
           rline(xGrid,yGrid);
           rmove(-xGrid,-yGrid);
          end;
         end;
        end; {NSubGrids}
       end; {drawing subGrid}
       if (istart=1) and (i=8) then inc2:=2; {Stops drawing half interval }
       i:=i+inc2;                            { Grids at 5}
       x3:=x;
       x:=x+x1*(inc2/2);  {Running check of position on axis }
      until i>=istop;     {end of inner subGrids loop}
      end; {if SubGridOn}
      x1:=x1*10;x3:=x1;   {Reset for next decade}
      xdGridstart:=xdGridStart+xdGrid;
      ydGridStart:=ydGridStart+ydGrid;
      r1:= r1+dGrid;
      x:=x1;
      amove(xdGridStart,ydGridStart);
      if (x<=1.001*ScaleHi) and (x>=0.999*ScaleLo) then
      begin  {Draw next decade Grid}
       DisplayForm.Image1.canvas.pen.color:= GridColor;
       DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLineWidth);
       DisplayForm.Image1.canvas.pen.style:= GridStyle;
       If polaraxis then  begin
       amove(xcentre,ycentre);
       arcat(r1,r1,GridStartAngle,GridEndAngle)
       end else begin
        rline(xGrid,yGrid);
        rmove(-xGrid,-yGrid);
       end;
      end;
   until x1>=1.001*scaleHi; {End outer loop}
   { Draw axis line  }
   DisplayForm.Image1.canvas.pen.color:= Gridcolor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLineWidth);
   DisplayForm.Image1.canvas.pen.style:= GridStyle;
   Amove(xstart,ystart);
   if polaraxis then begin
     Aline(xend,yend);
     Amove(xstart,ystart);
   end else begin    {Do not draw grid lines for polar axes}
     Aline(xstart+xGrid,ystart+yGrid);
     Aline(xend+xGrid,yend+yGrid);
     Aline(xend,yend);
     Aline(xstart,ystart);
   end;
   DisplayForm.Image1.canvas.pen.color:=Black;
  end; {If GridOn }

  if (not AxisOff) then
  begin
  {Reset scale starting point}
   xdStart:= xstart+TickStart*xscale;
   ydstart:= ystart+TickStart*yscale;
   if nSubTicks>0 then  {Sets up subTick spacings according to number of
             subticks specified. This overrides any actual specification
             of subtick spacing.}
   begin
    case nsubticks of
         1:begin
            istart:=8;
            inc1:=10;
           end;
       2,3:begin
            istart:=3;
            inc1:=5;
           end;
         4:begin
            istart:=2;
            inc1:=4;
           end;
      6..9:begin
            istart:=2;
            inc1:=2;
           end;
    10..20:begin
            istart:=1;
            inc1:=1;
           end;
      else begin     {Default is to only show midpoint}
             istart:=8;
             inc1:=10;
           end;
    end; {case nsubsticks}
   end else if dSubTicks>0 then {evaluates log increments if number of
                subticks = 0 or ifsubtick spacing defined }
   begin
    i:=round(2*dsubTicks);
    case i of
         1:begin       {DSubticks=0.5 : nSubTicks > 9}
            istart:=1;
            inc1:=1;
            nSubTicks:= 10;
           end;
         2:begin       {DSubticks=1.0 : nSubTicks = 9}
            istart:=2;
            inc1:=2;
            nSubTicks:=  9;
           end;
       3,4:begin       {DSubticks=2.0 : nSubTicks = 4}
            istart:=2;
            inc1:=4;
            nSubTicks:=  4;
           end;
         5:begin      {DSubticks=2.5 : nSubTicks = 3}
            istart:=3;
            inc1:=5;
            nSubTicks:=  3;
           end;
     6..10:begin      {DSubticks=5.0 : nSubTicks = 1}
            istart:=8;
            inc1:=10;
            nSubTicks:=  1;
           end;
     else begin     {Default is 10 subdivisions}
            istart:=2;
            inc1:=2;
            nSubTicks:=  9;
          end;
    end; {case i}
   end else begin
      istart:=2;   {Overall default is a ten division axis }
      inc1:=2;
      nSubTicks:=  9;
   end;  {if nsubticks or dsubticks}
   TickNum:=0;
   If NoFirst then  begin  {Mod 20/4/2000}
   {If Scalelo<1.0  log axes start at position 1, otherwise at position 0}
     If ScaleLo<1.0 then Showplace[1]:= False
     else ShowPlace[0]:= False;
   end;
   xdTicks:= xscale*dTicks;
   ydTicks:= yscale*dTicks;
   x1:=exp(LogBottom/LogE);  {Reference point for scale}
   LastAngle:= Textangle; {Remember labels and subLabels take the same angle!}
   repeat {Outer loop for decades}
    amove(xdstart,ydstart);
    if (x1>=ScaleLo*0.999) and (x1<= 1.001*ScaleHi) then
    begin
    { if LabelsOn then   }
     begin
      CurrentFont.Name:=LabelFontName;
      CurrentFont.size:=Round(LabelHei);
      CurrentFont.Color:=LabelColor;
      TextAngle:= LabelAngle;
     end; {if LabelsOn }
    { Draw Decade Tick}
     DisplayForm.Image1.canvas.pen.color:= TickColor;
     DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*TickLineWidth);
     DisplayForm.Image1.canvas.pen.style:= TickStyle;
     if TicksOn then
     begin
     if SymmetricTicks then
     begin
      rline(xmark1,-ymark1);
      rmove(-xmark1,ymark1);
     end;
      rline(-xmark1,ymark1);
     end;
     if ShowPlace[TickNum]  then
     begin
       Str(x1:LabelSfigs:LabelDfigs,SText);
       x2:= (LabelShift);
       y2:= (LabelDist);
       rmove(x2,y2);
       Text(SText,TextAngle);
       rmove(-x2,-y2);
     end;
     if TicksOn then rmove(xmark1,-ymark1);
    end; {Draw Decade Tick }
     {Settings for inner loop }
      x:=x1; x3:=x1;
      i:=istart;
      inc2:=inc1;
      istop:=20;
      x:= x+x1*(istart/2); {Starting point for subticks}
      repeat  {Inner loop for drawing subticks}
        MidTickVal:= (i+2)/2;   {MidTickVal is the actual scale value at
              the tick. This is later converted to an integer index
              MidTickNum used in the arrays. i gives a running count of
              all possible place settings including half spaces. }
        DisplayForm.Image1.canvas.pen.color:= SubTickColor;
        DisplayForm.Image1.canvas.pen.width:=
                           Round(PixelsPerCm*SubTickLineWidth);
        DisplayForm.Image1.canvas.pen.style:= SubTickStyle;
        CurrentFont.Name:=MidLabelFontName;
        CurrentFont.size:=Round(MidLabelHei);
        CurrentFont.Color:=MidLabelColor;
        TextAngle:= LabelAngle;
        x2:= dTicks*LogE*Ln(Abs(x/x3));  {Calculate subtick spacing}
        xdSubTicks:=  xscale*x2;         {Set subtick coordinates}
        ydSubTicks:=  yscale*x2;
        rmove(xdSubTicks,ydSubTicks);       {Check for start or end of scale}
        if (x>=0.999*ScaleLo) and (x<=1.001*ScaleHi) then {If O.K. draw tick}
        begin      {Select tick length as appropriate}
        if NSubticks>=0 then begin
         If (i mod 2 <> 0) and (i<8) then begin  {Test for half subticks}
          if SymmetricSubticks then
          begin
           rline(xmark4,-ymark4);
           rmove(-xmark4,ymark4);
          end;
           rline(-xmark4,ymark4);
           rmove(xmark4,-ymark4);
         end;
         if (i=8) then begin   {Test for Midtick}
          DisplayForm.Image1.canvas.pen.color:= MidTickColor;
          DisplayForm.Image1.canvas.pen.width:=
                                    Round(PixelsPerCm*MidTickLineWidth);
          DisplayForm.Image1.canvas.pen.style:= MidTickStyle;
          if MidTicksOn then
          begin
            if SymmetricMidticks then
            begin
             rline(xmark3,-ymark3);
             rmove(-xmark3,ymark3);
            end;
            rline(-xmark3,ymark3);
            rmove(xmark3,-ymark3);
          end;
         end;  {MidTicks}
         if (i mod 2 = 0) or (i>8) then begin  {Test for subTicks}
          DisplayForm.Image1.canvas.pen.color:= SubTickColor;
          DisplayForm.Image1.canvas.pen.width:=
                                 Round(PixelsPerCm*SubTickLineWidth);
          DisplayForm.Image1.canvas.pen.style:= SubTickStyle;
          if SymmetricSubticks then
           begin
            rline(xmark2,-ymark2);
            rmove(-xmark2,ymark2);
          end;
          rline(-xmark2,ymark2);
          rmove(xmark2,-ymark2);
         end; {Drawing SubTicks}
         MidTickNum:= Trunc(MidTickVal);
        {Next insert tick labels}
         if (MidTickNum <10)  and ((nSubTicks<2) or (nSubTicks>3))
           and ShowMidPlace[MidTickNum] and ( (i mod 2)=0) then
         begin  {Unless nSubticks = 2 or 3 labels are single digits}
           Str(MidTickNum:1,SText);
           x2:= (MidLabelShift)-xmark2; {Position relative to end of tick}
           y2:= (MidLabelDist)+ymark2;
           rmove(x2,y2);
           Text(SText,TextAngle);
           rmove(-x2,-y2);
         end else if (MidTickNum <10)  and (nSubTicks>1) and (nSubTicks<4)
           and ShowMidPlace[MidTickNum] {and ( (i mod 2)=0)} then
         begin  {Special case of nSubticks = 2 or 3 }
           Str(MidTickval:3:1,SText);
           x2:= (MidLabelShift)-xmark2;
           y2:= (MidLabelDist)+ymark2;
           rmove(x2,y2);
           Text(SText,TextAngle);
           rmove(-x2,-y2);
         end;  {end of label insertion}
        end; { if NsubTicks>0}
       end; {if x> - Scale check}
       if (istart=1) and (i=8) then inc2:=2; {Stops drawing half interval }
       i:=i+inc2;                            { ticks at 5}
       x3:=x;
       x:=x+x1*(inc2/2);  {Running check of position on axis }
      until i>=istop;  {end of inner subticks loop}
      x1:=x1*10;x3:=x1;  {Reset for next decade}
      xdstart:=xdStart+xdTicks;
      ydStart:=ydStart+ydTicks;
      x:=x1;
      TickNum:=succ(tickNum);
      if Nolast and (x>0.8*ScaleHi) then ShowPlace[TickNum]:= False;
   until x1>=1.001*scaleHi; {End outer loop}
   if LabelsOn then
   begin
    CurrentFont.Name:='Times New Roman';
    CurrentFont.size:=Round(0.2*PixelsPerCm);      { **** Check This *** }
    CurrentFont.Color:=black;
    TextAngle:= LastAngle;
   end;
   { Draw axis line  }
   DisplayForm.Image1.canvas.pen.color:= Axiscolor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPercm*AxisLineWidth);
   DisplayForm.Image1.canvas.pen.style:= AxisLineStyle;
   Amove(xstart,ystart);
   Aline(xend,yend);
{  end; }
  end; {AxisType=2}
  end; {AxisOff}
  {Finally inset axis title string}
   amove( (xstart+xend)/2 + TitleShift, (ystart+yend)/2 + TitleDist);
    Lastangle:= Textangle;TextAngle:= TitleAngle;
    CurrentFont.Name:=TitleFontName;
    CurrentFont.size:=Round(TitleHei);
    CurrentFont.Color:=TitleColor;
    Text(title,TitleAngle);
    CurrentFont.Name:='Times New Roman';
    CurrentFont.size:=Round(0.2*pixelsPerCm);    { i.e.  10* PixelsperCm/50 }
    CurrentFont.Color:=black;
    TextAngle:= LastAngle;
    amove(xstart,ystart);
    if PolarAxis then
    begin   {Turns off grids etc for multiple copies of polar axes}
     PolarAxis:= False;
     GridOn:= False;
     LabelsOn:= False;
     MidLabelsOn:= False;
     NamesOn:= False;
     for i:= 0 to 50 do
     begin
       ShowName[i]:= False;
       ShowPlace[i]:= False;
       ShowMidPlace[i]:= False;
     end;
    end;
 end; {With Axis Do}
 DisplayForm.Image1.canvas.pen.color:= clBlack;
 DisplayForm.Image1.canvas.pen.width:=1;
 DisplayForm.Image1.canvas.pen.style:= psSolid;
end;   {DrawAxis}

procedure DrawAngleAxis(var AngAxis:TAxis);
{ Draws a curved linear axis consisting of a circular arc with
  radial scale marker and grid lines.
  The axis is specified by:
  ScaleLo - lower value of axis range in arbitrary user units
  ScaleHi - upper limit of axis range in arbitrary user units
  xstart  - the absolute angle of the axis start point. Absolute
            angles start at zero on ahorizontal line, positive
            direction to the right, and are positive in the
            anticlockwise direction.
  ystart  - the axis radius in cms.
  xend    - the absolute angle of the axis end point
  centreradius  - the radius of a circle surrounding the centre point
  xcentre, ycentre - the coordinates of the centre of the arc
  dStart - first point for ticks, in user scale units.
  dTicks - distance between ticks in user scale units.
  nTicks - number of ticks in scale
  dSubStart,dSubticks nsubticks etc.- same for subticks
  dMidStart,dMidTicks etc. - same for middle ticks
  dGridStart etc.    - same for major grid lines.
  dMidGridStart etc. - same for midgrid lines
  dsubgridStart etc. - same for subgrid lines.
  TickAngle - Angle of Ticks relative to axis (not normally used)
  Ticklength, Gridlength etc. - Line lengths in cms. Unlike rectalinear
            axes, midgridlengths and subgridlenths may be independently
            set.                                   }

VAR  x,x1,x2,x3,y1,y2,y3,ScaleAxis,AxisRadius,
     scalelength,xscale,yscale,CentreArcRadius,    {added 12/4/2000}
     {xmark1,ymark1,xmark2,ymark2,xmark3,ymark3,xmark4,ymark4,}xGrid,yGrid,
     xdstart,ydstart,xdSubStart,ydSubStart,xdMidStart,ydMidStart,
     xdend,ydend,xdSubEnd,ydSubEnd,xdMidEnd,ydMidEnd,
     xdGridStart,ydGridStart,xdMidGridStart,ydMidGridStart,
     xdSubGridStart,ydSubGridStart,
     xdGridEnd,ydGridEnd,xdMidGridEnd,ydMidGridEnd,
     xdSubGridEnd,ydSubGridEnd,
     xdGrid,ydGrid,xdSubGrid,ydSubGrid,
     xdMidGrid,ydMidGrid,xdTicks,ydTicks,xdSubTicks,ydSubTicks,
     xdMidTicks,ydMidTicks,LoLimit,HiLimit,LastAngle,MidTickVal,
     LabelHeight,LabelWidth,NameHeight,NameWidth,
     MidLabelheight,MidlabelWidth:double;
     i,inc1,inc2,istart,istop,TickNum,MidTickNum,SExp:integer;
     SText:String;
     TitleFont,LabelFont,MidLabelFont,NameFont:TFont;
begin
 with AngAxis do
 begin
  If xStart>xEnd then
  begin
   if Messagedlg('G30: Axis end angle is less than Axis Start Angle. Leaving this'
        + ' unchanged will give erratic results and may crash the system.',
        mtWarning,[mbOK,mbAbort],0)= mrAbort then
   begin {Halt to check on error}
     Disaster:= True;
     exit;
   end;
  end;
  AxisRadius:= yStart;CentreArcRadius:= yend;{Change names to ease identification}
  AxisStartAngle:= xstart/AngleMeasure; {Convert angles to radians}
  AxisEndangle:= xend/AngleMeasure;
  AxisLength:= AxisEndAngle-AxisStartAngle;
  {Now shift start angle to between 0 and two pi while retaining the
   difference between the start angle and end angle.}
  while AxisStartAngle>TwoPi do
  begin
     AxisStartAngle:= AxisStartAngle - TwoPi;
     AxisEndAngle:= AxisEndAngle - TwoPi;
  end;
  while AxisStartAngle<0.0 do
  begin
    AxisStartAngle:= AxisStartAngle + TwoPi;
    AxisEndAngle:=   AxisEndAngle + TwoPi;
  end;
 { If axis length is close to 2 pi round it up to a complete circle.}
  if (AxisLength>0.98*TwoPi) and (Axislength<1.02*TwoPi)
    then AxisLength:= TwoPi;
  if (AxisLength<-0.98*TwoPi) and (Axislength>-1.02*TwoPi)
    then AxisLength:= -TwoPi;

 {In angular axes marker and grid legths must evaluated for each angle}

  ScaleLength:= ScaleHi-ScaleLo;
  HiLimit:= ScaleHi+ScaleLength*0.0001; {Used to avoid roundoff errors}
  LoLimit:=ScaleLo-ScaleLength*0.0001;  {when drawing scale end ticks.}
  {Set coversion between absolute angles and user scale}
  ScaleAxis:= AxisLength/ScaleLength;
   {Set HiLimit and LoLimit to allow proper stepping in repeat loops.}
  If ScaleAxis<0 then
  begin
    x:= HiLimit; HiLimit:= LoLimit; LoLimit:= x;
  end;
  if GridOn then
  begin
   y3:= GridAngle/AngleMeasure; {Used for angled grids}
   if nGrids>0 then  {Setting nGrids overrides dGrid setting}
   begin
    GridStart:= ScaleLo;
    dGrid:= ScaleLength/nGrids;
   end;
  {  First draw subgrid lines, then Midgrid line, and then grid lines.
     The grid is drawn before the axis so that axis overwrites it. }
   if ((dSubGrid<>0.0) or (nSubGrids>0)) and SubGridOn then  {Draw subGrids}
   begin   {Set Subgrid style, width and colour}
    DisplayForm.Image1.canvas.pen.color:= SubGridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*SubGridLinewidth);
    DisplayForm.Image1.canvas.pen.style:= SubGridStyle;
    if nSubGrids>0 then    {nSubGrids overrides setting of dSubgrid}
    begin
      dSubGrid:= dGrid/nSubGrids;
      SubGridStart:= GridStart;
    end;
    { y1 is the running index of absolute angular position}
    y1:= (SubGridStart-ScaleLo)*ScaleAxis+AxisStartAngle;
    xdSubGridStart:=  xcentre+AxisRadius*cos(y1);
    ydSubGridStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdSubGridstart,ydSubGridstart);
    x:=SubGridStart; {x is the running index of user scale position}
    xdSubgridEnd:= -SubgridLength*cos(y1+y3)*hscale;  {These are relative moves}
    ydSubgridEnd:= -SubGridLength*sin(y1+y3)*hscale;

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      amove(xdsubGridStart,ydSubGridStart);
      rline(xdSubGridEnd,ydSubGridEnd);  {Draw Subgridline}
     end;
     y1:= y1+dSubgrid*ScaleAxis;  {Calculate parameters for next Subgrid line}
     xdSubGridStart:=  xcentre+AxisRadius*cos(y1);
     ydSubGridStart:=  ycentre+AxisRadius*sin(y1);
     xdSubgridEnd:= -SubgridLength*cos(y1+y3)*hscale; {These are relative moves}
     ydSubgridEnd:= -SubGridLength*sin(y1+y3)*hscale;
     x:= x+dSubGrid;    {Increment Scale}
    until (x>HiLimit) or (x<LoLimit);
   end;

   if ((dMidGrid<>0.0) or (nMidGrids>0)) and MidGridOn then  {Draw MidGrids}
   begin  {Set style width and colour for midgrid lines}
    DisplayForm.Image1.canvas.pen.color:= MidGridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*MidGridLinewidth);
    DisplayForm.Image1.canvas.pen.style:= MidGridStyle;
    if nMidGrids>0 then    {nMidGrids overrides setting of dMidGrid}
    begin                  {and sets them to align with grid lines}
      dMidGrid:= dGrid/nMidGrids;
      MidGridStart:= GridStart;
    end;
    y1:= (MidGridStart-ScaleLo)*ScaleAxis+AxisStartAngle;
    xdMidGridStart:=  xcentre+AxisRadius*cos(y1);
    ydMidGridStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdMidGridstart,ydMidGridstart);
    x:=MidGridStart;
    xdMidGridEnd:= -MidGridLength*cos(y1+y3)*hscale;  {These are relative moves}
    ydMidGridEnd:= -MidGridLength*sin(y1+y3)*hscale;

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      amove(xdMidGridStart,ydMidGridStart);
      rline(xdMidGridEnd,ydMidGridEnd);  {Draw MidGridline}
     end;
     y1:= y1+dMidGrid*ScaleAxis;  {Calculate parameters for next MidGrid line}
     xdMidGridStart:=  xcentre+AxisRadius*cos(y1);
     ydMidGridStart:=  ycentre+AxisRadius*sin(y1);
     xdMidGridEnd:= -MidGridLength*cos(y1+y3)*hscale;  {These are relative moves}
     ydMidGridEnd:= -MidGridLength*sin(y1+y3)*hscale;
     x:= x+dMidGrid;    {Increment Scale}
    until (x>HiLimit) or (x<Lolimit);
   end;
   {Finally draw the Main Grid lines}
   if ((dGrid<>0.0) or (nGrids>0)) and GridOn then  {Draw Grids}
   begin
    DisplayForm.Image1.canvas.pen.color:= GridColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*GridLinewidth);
    DisplayForm.Image1.canvas.pen.style:= GridStyle;
    y1:= (GridStart-ScaleLo)*ScaleAxis+AxisStartAngle;  {the Grid start angle}
    xdGridStart:=  xcentre+AxisRadius*cos(y1);
    ydGridStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdGridstart,ydGridstart);
    x:=GridStart;
    xdGridEnd:= -GridLength*cos(y1+y3)*hscale;  {These are relative moves}
    ydGridEnd:= -GridLength*sin(y1+y3)*hscale;

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      amove(xdGridStart,ydGridStart);
      rline(xdGridEnd,ydGridEnd);  {Draw Gridline}
     end;
     y1:= y1+dGrid*ScaleAxis;  {Calculate parameters for next Grid line}
     xdGridStart:=  xcentre+AxisRadius*cos(y1);
     ydGridStart:=  ycentre+AxisRadius*sin(y1);
     xdGridEnd:= -GridLength*cos(y1+y3)*hscale;  {These are relative moves}
     ydGridEnd:= -GridLength*sin(y1+y3)*hscale;
     x:= x+dGrid;    {Increment Scale}
    until (x>HiLimit) or (x<LoLimit);
   end;
   DisplayForm.Image1.canvas.pen.color:=Black;
  end;  {grid on}

  if (not AxisOff) then
  begin
   y3:= TickAngle/AngleMeasure;
   if nticks>0 then  {Setting nTicks overrides dTicks setting}
   begin             {The scale then starts at the axis start}
    Tickstart:= ScaleLo;
    dTicks:= ScaleLength/nTicks;
   end;
  {First draw sub ticks, then Midticks, ticks, grids and finally axis line.
   This sequence ensures colours, styles etc are not overwritten. }
   if  SubTicksOn and ((dSubTicks<>0.0) or (nSubTicks>0)) then   {Draw subticks}
   begin
    DisplayForm.Image1.canvas.pen.color:= SubTickColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*SubTickLinewidth);
    DisplayForm.Image1.canvas.pen.style:= SubTickStyle;

    if nSubTicks>0 then    {nSubTicks overrides setting of dSubTick}
    begin                  {and aligns them with primary grid}
      dSubTicks:= dTicks/nSubTicks;
      SubTickStart:= TickStart;
    end;
   {y1 is the running index of absolute angular position}
    y1:= (SubTickStart-ScaleLo)*ScaleAxis+AxisStartAngle;
    xdSubStart:=  xcentre+AxisRadius*cos(y1);
    ydSubStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdSubStart,ydSubStart);
    x:=SubTickStart; {x is the running index of user scale position}
    xdSubEnd:= -SubTickLength*cos(y1+y3);  {These are relative moves}
    ydSubEnd:= -SubTickLength*sin(y1+y3);

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin
      amove(xdsubStart,ydSubStart);
      rline(xdSubEnd,ydSubEnd);  {Draw SubTickline}
      if SymmetricSubticks then  {Makes ticks symmetric about axis}
      begin
      amove(xdSubStart,ydSubStart);
      rline(-xdSubEnd,-ydSubEnd);
      end;
     end;
     y1:= y1+dSubTicks*ScaleAxis;  {Calculate parameters for next SubTick line}
     xdSubStart:=  xcentre+AxisRadius*cos(y1);
     ydSubStart:=  ycentre+AxisRadius*sin(y1);
     xdSubEnd:= -SubTickLength*cos(y1+y3);  {These are relative moves}
     ydSubEnd:= -SubTickLength*sin(y1+y3);
     x:= x+dSubTicks;    {Increment Scale}
    until (x>HiLimit) or (x<LoLimit);
   end;
   {Now draw Mid ticks}
   if  MidTicksOn and ((dMidTicks<>0.0) or (nMidTicks>0)) then
   begin
    MidTickNum:=0; {A counter used when adding labels}
    if NoFirst then  ShowMidPlace[0]:=False; {Suppress the first label}
    DisplayForm.Image1.canvas.pen.color:= MidTickColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*MidTickLinewidth);
    DisplayForm.Image1.canvas.pen.style:= MidTickStyle;
    if nMidTicks>0 then    {nMidTicks overrides setting of dMidTick}
    begin                  {and forces alignment with primary ticks}
      dMidTicks:= dTicks/nMidTicks;
      MidTickStart:= TickStart;
    end;
   {In this case we can also add labels to the midticks, so set up
    the font for this}
    CurrentFont.Name:=MidLabelFontName;
    CurrentFont.size:=Round(MidLabelHei);
    CurrentFont.Color:=MidLabelColor;
    Lastangle:= Textangle; TextAngle:= LabelAngle; {Sets angle for labels}
    {As for subticks y1 is the running index of absolute angular position}
    y1:= (MidTickStart-ScaleLo)*ScaleAxis+AxisStartAngle;
    xdMidStart:=  xcentre+AxisRadius*cos(y1);
    ydMidStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdMidStart,ydMidStart);
    x:=MidTickStart; {x is the running index of user scale position}
    xdMidEnd:= -MidTickLength*cos(y1+y3);  {These are relative moves}
    ydMidEnd:= -MidTickLength*sin(y1+y3);

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin   {draw the Midtick}
      amove(xdMidStart,ydMidStart);
      rline(xdMidEnd,ydMidEnd);  {Draw MidTickline}
      if SymmetricMidticks then
      begin
       amove(xdMidStart,ydMidStart);
       rline(-xdMidEnd,-ydMidEnd);
      end;
     end;
     {ShowMidPlace is the flag determining if label is added}
     if (x>Hilimit)or (x<loLimit) then ShowMidPlace[MidTickNum]:= False;
     if ShowMidPlace[MidTickNum] then
     begin
       Str(x:MidLabelSfigs:MidLabelDFigs,SText);
       MidLabelheight:= DisplayForm.Image1.canvas.TextHeight(SText)/pixelspercm;
       MidLabelWidth:=  DisplayForm.image1.canvas.TextWidth(SText)/pixelspercm;
       {The following offsets the labels according to their angular position}
       If abs(cos(y1))<0.1 then
        begin
        y2:=0; {Top or bottom label} x2:= -0.5*MidlabelWidth;
        end else if cos(y1)>0 then
        begin
           y2:= MidLabelShift; {Left side}  x2:=0;
        end else begin {right Side}
          y2:= MidlabelShift; x2:= -MidLabelWidth;
        end;
       {Now set starting point for label string}
        xdMidStart:=  xcentre+(AxisRadius+MidLabelDist)*cos(y1+y2);
        ydMidStart:=  ycentre+(AxisRadius+MidlabelDist)*sin(y1+y2);
        amove(xdMidStart+x2,ydMidStart);
        Text(SText,TextAngle);
     end; {show Midplace label}
     y1:= y1+dMidTicks*ScaleAxis;  {Calculate parameters for next MidTick line}
     xdMidStart:=  xcentre+AxisRadius*cos(y1);
     ydMidStart:=  ycentre+AxisRadius*sin(y1);
     xdMidEnd:= -MidTickLength*cos(y1+y3);  {These are relative moves}
     ydMidEnd:= -MidTickLength*sin(y1+y3);
     x:= x+dMidTicks;    {Increment Scale}
     if MidTickNum<50 then MidTickNum:= succ(MidTickNum);
    until (x>HiLimit) or (x<LoLimit);
   {Reset the font after adding labels}
     CurrentFont.Name:='Times New Roman';
     CurrentFont.size:=Round(0.2*PixelsPerCm);    {**** Check This ***** }
     CurrentFont.Color:=black;
     TextAngle:= Lastangle;
   { end;  }
   end;
   {Now Draw Ticks}
    TickNum:=0; {The counter for tick lables and names}
    if NoFirst then  ShowPlace[0]:=False; {Suppress first label}
    DisplayForm.Image1.canvas.pen.color:= TickColor;
    DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*TickLinewidth);
    DisplayForm.Image1.canvas.pen.style:= TickStyle;
   {Set font for Tick Labels}
    CurrentFont.Name:=LabelFontName;
    CurrentFont.size:=Round(LabelHei);
    CurrentFont.Color:=LabelColor;
    Lastangle:= Textangle; TextAngle:= LabelAngle;
    { y1 is the running index of absolute angular position}
    y1:= (TickStart-ScaleLo)*ScaleAxis+AxisStartAngle;
    xdStart:=  xcentre+AxisRadius*cos(y1);
    ydStart:=  ycentre+AxisRadius*sin(y1);
    amove(xdStart,ydStart);
    x:=TickStart; {x is the running index of user scale position}
    xdEnd:= -TickLength*cos(y1+y3);  {These are relative moves}
    ydEnd:= -TickLength*sin(y1+y3);

    repeat
     if (x>=LoLimit) and (x<=HiLimit) then
     begin   {draw the tick}
      if TicksOn then
      begin
       amove(xdStart,ydStart);
       rline(xdEnd,ydEnd);  {Draw Tickline}
       if Symmetricticks then
       begin
        amove(xdStart,ydStart);
        rline(-xdEnd,-ydEnd);
       end;
      end;
      {Suppress the last Label if NoLast flag set}
      if NoLast and (x>=Hilimit-dticks) then ShowPlace[TickNum]:= False;
     end;
     if (x>Hilimit) or (x<Lolimit) then Showplace[TickNum]:= False;
     if ShowPlace[TickNum] then   {Draw the label}
     begin
       Str(x:LabelSfigs:LabelDFigs,SText);
       Labelheight:= DisplayForm.Image1.canvas.TextHeight(Stext)/pixelspercm;
       LabelWidth:=  DisplayForm.image1.canvas.TextWidth(Stext)/pixelspercm;
       {Calculate displacement for label dependent on its angular position}
       If abs(cos(y1))<0.1 then
        begin
         y2:=0; {Top or bottom label} x2:= -0.5*LabelWidth;
        end else if cos(y1)>0 then
        begin
           y2:= LabelShift; {Left side}  x2:=0;
        end else begin {right Side}
          y2:= labelShift; x2:= -LabelWidth;
        end;
       {Now set starting point for label string}
        xdStart:=  xcentre+(AxisRadius+LabelDist)*cos(y1+y2);
        ydStart:=  ycentre+(AxisRadius+labelDist)*sin(y1+y2);
        amove(xdStart+x2,ydStart);
        Text(SText,TextAngle);
     end;
     {For main divisions we can also add text string names. The process
      is similar to that used for labels}
     if ShowName[TickNum] then
     begin
      CurrentFont.Name:=NameFontName;
      CurrentFont.size:=Round(NameHei);
      CurrentFont.Color:=NameColor;
      TextAngle:= NameAngle;
      Nameheight:= DisplayForm.Image1.canvas.TextHeight
                       (ANames[TickNum])/pixelspercm;
      NameWidth:=  DisplayForm.image1.canvas.TextWidth
                       (ANames[TickNum])/pixelspercm;
       If abs(cos(y1))<0.1 then
        begin
          y2:=0; {Top or bottom label} x2:= -0.5*NameWidth;
        end else if cos(y1)>0 then
        begin
           y2:= NameShift; {Left side}  x2:=0;
        end else begin {right Side}
          y2:= NameShift; x2:= -NameWidth;
        end;
       {Now set starting point for name string}
        xdStart:=  xcentre+(AxisRadius+NameDist)*cos(y1+y2);
        ydStart:=  ycentre+(AxisRadius+NameDist)*sin(y1+y2);
        amove(xdStart+x2,ydStart);
        Text(ANames[TickNum],TextAngle);
        {Now reset the label font}
        CurrentFont.Name:=LabelFontName;
        CurrentFont.size:=Round(LabelHei);
        CurrentFont.Color:=LabelColor;
        TextAngle:= LabelAngle;
     end; {ShowName}
     y1:= y1+dTicks*ScaleAxis;  {Calculate parameters for next Tick}
     xdStart:=  xcentre+AxisRadius*cos(y1);
     ydStart:=  ycentre+AxisRadius*sin(y1);
     xdEnd:= -TickLength*cos(y1+y3);  {These are relative moves}
     ydEnd:= -TickLength*sin(y1+y3);
     x:= x+dTicks;    {Increment Scale}
     if TickNum<50 then TickNum:= succ(TickNum);
    until (x>HiLimit) or (x<LoLimit);
{    if LabelsOn then  reset font  }
     CurrentFont.Name:='Times New Roman';
     CurrentFont.size:=Round(0.2*PixelsPerCm);       {**** Check This ***** }
     CurrentFont.Color:=black;
     TextAngle:= Lastangle;
   { Draw axis line  }
   DisplayForm.Image1.canvas.pen.color:= AxisColor;
   DisplayForm.Image1.canvas.pen.width:= Round(PixelsPerCm*AxisLineWidth);
   DisplayForm.Image1.canvas.pen.style:= AxisLineStyle;
   if SideOn then
   begin
     Amove(xcentre,ycentre);
     ArcAt(AxisRadius,AxisRadius,AxisStartAngle*AngleMeasure,
           AxisEndAngle*AngleMeasure);    {Mod 12/4/2000}
   end;
   if CentreArcRadius>0.0   then
   begin
     Amove(xcentre, ycentre);
     ArcAt(CentreArcRadius,CentreArcRadius,AxisStartAngle*AngleMeasure,
           AxisEndAngle*AngleMeasure);
   end;
  end; {Axisoff}
  DisplayForm.Image1.canvas.pen.color:=Black;
  {Finally inset axis title string}
  amove( {(xstart+xend)/2 +} TitleShift, {(ystart+yend)/2} + TitleDist);
  Lastangle:= Textangle;TextAngle:= TitleAngle;
  CurrentFont.Name:=TitleFontName;
  CurrentFont.size:=Round(TitleHei);
  CurrentFont.Color:=TitleColor;
  Text(title,TitleAngle);
  CurrentFont.Name:='Times New Roman';
  CurrentFont.size:=Round(0.2*pixelsPerCm);    { i.e.  10* PixelsperCm/50 }
  CurrentFont.Color:=black;
  TextAngle:= LastAngle;
  amove(AxisRadius*cos(AxisStartAngle),AxisRadius*sin(AxisStartAngle));
 end; {With Axis Do}
 DisplayForm.Image1.canvas.pen.color:= clBlack;
 DisplayForm.Image1.canvas.pen.width:=1;
 DisplayForm.Image1.canvas.pen.style:= psSolid;
end;   {DrawAngleAxis}


Procedure SetAxisDefaults;
{Resets all axis parameters to default values}
var i:integer;
begin   {Default axis settings}
   with x1Axis do  { For linear axis  }
   begin
      AxisstartAngle:= 0.0;AxisEndAngle:= 0.0;
      InnerRadius:= -1.0; OuterRadius:= -2.0;
      Polaraxis:= False; AngleAxis:= False;
      StartAt:=0.0;StepBy:=0.0;EndAt:=0.0;
      ScaleLo:=0.0;ScaleHi:=5.0;
      TickStart:=0.0;dTicks:=1.0;nTicks:=0;
      TickLength:=-0.25;TickLineWidth:=0.01;Tickstyle:=psSolid;
      TickAngle:=0;TickColor:=Black;
      SubTickStart:=-0.001;dSubTicks:=0.25;nSubTicks:=0;
      SubTickLength:=-0.15;SubTickLineWidth:=0.01;
      SubTickstyle:=psSolid;SubTickColor:=Black;
      MidTickStart:=0.5;dMidticks:=1.0;nMidTicks:=0;
      MidTickLength:=-0.2;MidTickLinewidth:=0.01;
      MidTickStyle:=psSolid;MidTickColor:=Black;
      Gridstart:=0.0;dGrid:=1.0;nGrids:=0;GridLineWidth:=0.01;
      GridStyle:=psDash;GridAngle:=0.0;GridColor:=Black;
      GridAngle:=0.0;
      SubGridStart:=0.0;dSubGrid:=0.25;nSubGrids:=0;
      SubGridLineWidth:=0.01;SubGridStyle:=psDot;
      SubGridColor:=Black;
      MidGridStart:=0.5;dMidGrid:=1.0;nMidGrids:=0;
      MidGridLineWidth:=0.01;MidGridStyle:=psDash;
      MidGridColor:=black;
      AxisLineWidth:=0.02;AxisLineStyle:=psSolid;
      AxisType:=1; {Linear}
      AxisColor:=Black;
      Title:= '';   {Leave blank }
      TitleFontName:= 'Times New Roman';TitleColor:=Black;
      TitleHei:=0.14*PixelsPerCm; {pt};
      TitleDist:= -0.7; TitleShift:= -0.6;
      MidLabelFontName:= 'Arial';MidLabelColor:=Blue;
      MidLabelHei:= 0.08*PixelsPerCm;
      Midlabelshift:= -0.25; MidLabelDist:= -0.08;
      MidLabelSFigs:= 5; MidLabelDFigs:= 2;
      LabelFontName:='Arial';LabelColor:=Black;LabelHei:=0.12*PixelsPerCm;
      LabelShift:= -0.3; LabelDist:= -0.1;
      LabelSFigs:= 5; LabelDFigs:= 2;
      AxisOff:=False; SideOn:=True;GridOn:=False;Negate:=False;
      PolarAxis:= False; MidGridOn:= False; SubGridOn:= False;
      TicksOn:=True; MidTicksOn:=True; SubTicksOn:=True;
      SymmetricTicks:= False; SymmetricMidTicks:= False;
      SymmetricSubTicks:= False;
      NoFirst:=False;NoLast:=False;
      LabelsOn:=True;MidLabelsOn:=False;NamesOn:=False;
      NameFontName:= 'Arial'; NameHei:= 0.12*PixelsPerCm; NameColor:= Purple;
      NameShift:= 0.3; NameDist:= -0.5;
      ANames[0]:=' Sun';
      ANames[1]:=' Mon';
      ANames[2]:='Tues';
      ANames[3]:=' Wed';
      ANames[4]:='Thurs';
      ANames[5]:=' Fri';
      ANames[6]:=' Sat';
      ANames[7]:=' Sun';
      ANames[8]:=' Mon';
      ANames[9]:='Tues';
    {  ANames[10]:=' Wed ';  }
      ANames[11]:='Thurs';
      ANames[12]:=' Fri';
      ANames[13]:=' Sat';
      ANames[14]:=' Sun';
      ANames[15]:='Mon';
      ANames[16]:='Tues';
      ANames[17]:=' Wed ';
      ANames[18]:='Thurs';
      ANames[19]:='Fri';
      ANames[20]:='Sat';
      for i:= 0 to 50 do
      begin
        ShowName[i]:=False;ShowPlace[i]:=True;
        ShowMidplace[i]:= False;
      end;
   end;
     y1Axis:=x1Axis;
   with y1Axis do   {For linear axis }
    begin
      ScaleLo:=0.0;ScaleHi:=8.0;
      SubTickStart:=0.00;dSubTicks:=0.25;
      MidTickStart:=0.5;dMidticks:=1.0;
      TickLength:=0.25;
      SubTickLength:=0.125;MidTickLength:=0.0;
      nSubTicks:=0;
      LabelShift:= -0.7; LabelDist:= 0.1;
      Title:= '';   {Leave Blank}
      Titledist := -0.5; TitleAngle:= 1.57; TitleShift := -1.5;
    end;
    x2Axis:=x1Axis;
    x2axis.Axisoff:=True; x2axis.Title := '';
    y2Axis:=y1Axis;
    y2axis.Axisoff:=True; y2axis.Title:= '';
    z1Axis:=x1Axis;
    z1axis.Axisoff:=True; z1axis.Title:= '';
    z2Axis:=y1Axis;
    z2axis.Axisoff:=True; z2axis.Title:= '';
    r1axis:= x1axis;
    with r1axis do
    begin
     Polaraxis:= True;
     xcentre:= 8.0;
     ycentre:= 7.0;
     Axisstartangle:= 0.0;
     AxisEndangle:= 360;
     InnerRadius:= 0; OuterRadius:= 0.0;
     AxisType:= 1;Scalehi:=5.0; Scalelo:= 0.0;
     Symmetricticks:= True; Symmetricsubticks:= true;
     LabelDist:= -0.4; LabelShift:= -0.2; LabelHei:=0.15*PixelsPerCm;
     LabelColor:= Black; LabelSFigs:= 3; LabelDFigs:= 1;
     Midlabelson:= False; MidTickStart:= 0.5;
     LabelsOn:= True;
     for i:= 0 to 5 do ShowMidPlace[i]:= False;
     Nofirst:= True;  NoLast:= False;
     Startat:= 0.0; StepBy:= 90; EndAt:= 360;
     GridOn:= True;
     nGrids:= 5;
     gridStart:= 0.0;
     dGrid:= 1.0;
     dTicks:= 1.0;
     Ticklength:= 0.2;
     SubTicklength:= 0.1;
     Midtickson:= False;
     SubTicksOn:= True;
     SubTickStart:= 0.2;
     dSubTicks := 0.2;
     GridStyle:= psSolid;
     GridColor:= Blue;
     SubGridOn:= False;
     SubGridColor:= Red;
     MidGridOn:= False;
     MidGridColor:= Green;
     SubGridstart:= 1.0;
    end;
    r2axis:= r1axis;
    r2axis.Axisoff:=True; r2axis.Title := '';
    r2axis.GridOn:= False;
    r3axis:= r2axis;
    r4axis:= r3axis;
    a1axis:=x1axis;
    with a1axis do
    begin
       Angleaxis:= True;
       InnerRadius:= 0.0; OuterRadius:= 0.0; {Mod added 11/4/2000}
       xcentre:= 8;
       ycentre:= 7;
       xstart:=0.0;
       ystart:= 5.0;
       xend:=360;
       yend:= 0.5;
       ScaleLo:= 0.0;
       ScaleHi:= 360.0;
       GridOn:= True;
       dGrid:=30.0;
       Gridlength:= 3.5;
       Gridcolor:= blue;
       GridStyle:= psSolid;
       MidGridOn:= True;
       MidGridlength:= 3.0;
       MidGridColor:= Green;
       MidGridStart:= 10.0;
       DmidGrid:=10.0;
       SubGridOn:=False;
       Subgridcolor:= red;
       SubGridStyle:= psSolid;
       dSubGrid:= 10.0;
       SubGridLength:= 1.0;
       TicksOn:=True;
       TickLength:= -0.25;
       nTicks:= 36;
       LabelsOn:= True;
       LabelDist:= 0.5;
       LabelShift:= 0.0;
       LabelHei:= 0.15*Pixelspercm;
       LabelSfigs:= 3;
       LabelDFigs:= 0;
       NamesOn := False;
       NameDist:= 0.75;
       Namehei:= 8.0;
       NameShift:= 0.4;
       NameColor:= Fuschia;
       MidTickson:=True;
       MidTickStart:= 0.0;
       MidTickLength:= -0.1;
       MidLabelsOn:= False;
       MidlabelColor:= Red;
       MidLabelDist:= 0.2;
       MidlabelShift:= 0.0;
       MidlabelFontName:= 'Arial';
       MidLabelhei:= 6.0;
       MidLabelColor:= red;
       dMidTicks:= 1;
       SubTicksOn:= False;
       SymmetricSubTicks:= False;
       AxisOff:= False;
       for i:= 0 to 50 do
       begin
        ShowName[i]:=False;ShowPlace[i]:=True;
        ShowMidplace[i]:= False;
       end;
       NoLast:= True;
    end;
    a2axis:=x1axis;
    a2axis.Axisoff:=True; a2axis.AngleAxis:= False; a2axis.Title := '';
    Gtitle:= '     ';    { Leave Blank }
    GtitleFontName:= 'Times New Roman'; GtitleColor:= Black;
    GTitlehei:= 0.2*PixelsPerCm; GtitleOffset:= -1.5; GtitleDist:= -0.5;
end; {SetAxisDefaults}

procedure SetAngleAxisDefaults;
{Resets all angle axis parameters to their default values}
begin
 {Still to be set up}
end;

procedure PlotToPaper(xd,yd:double;CurrentGraphTransform:PlotTransform;
         var xp,yp:double);
{Converts plotter data co-ordinates xd,yd to paper co-ordinates
 xp,yp using the conversion parameters gta,gtb,gtc,gtd,gte,gtf,
 xIsLog,yIslog from the current graph data set. }

begin
 with CurrentGraphTransform do
 begin
  if xIsLog then
   if xd>0 then xd:= ln(xd)
   else begin
     Messagedlg('G31: Attempt to plot zero or negative x value on a log axis.',
      mtError,[mbabort],0);
      Disaster:=True;
      exit;
   end;
  if yIsLog then
   if yd>0 then yd:= ln(yd)
   else begin
     Messagedlg('G32: Attempt to plot zero or negative y value on a log axis.',
      mtError,[mbabort],0);
      Disaster:=True;
      exit;
   end;
   xp:= gta*xd + gtb*yd + gtc;
   yp:= gtd*xd + gte*yd + gtf;
 end;   {with CurrentGraphTransform}
end;  {PlotToPaper}

function ClipMode(xy, lo, hi:double):integer;
{ Determines location of value relative to clipping limits}

begin
  if xy<lo then ClipMode:= -1
  else if xy>hi then ClipMode:=1
  else Clipmode:=0;
end; {ClipMode}

procedure ClipDraw( x1,y1,x2,y2,PlotLeft,PlotRight,
                    PlotHi,PlotLo:double);
{Draws a line from x1,y1 to x2,y2 clipped to plot rectangle defined
 by PlotHi, PlotLo, Plotleft, PlotRight. x1,y1 are reset }

 var ix1, ix2, iy1, iy2:integer;

 begin
  ix1:= ClipMode(x1,PlotLeft,PlotRight);
  ix2:= ClipMode(x2,PlotLeft,PlotRight);
  iy1:= ClipMode(y1,PlotLo,PlotHi);
  iy2:= ClipMode(y2,PlotLo,PlotHi);
  if (ix1*ix2<=0) and (iy1*iy2<=0) then
  begin
   case iy1 of
    -1:begin
        x1:=x1 + (x2-x1)*(PlotLo-y1)/(y2-y1);
        y1:=PlotLo;
       end;
     1:begin
        x1:= x1 + (x2-x1)*(PlotHi-y1)/(y2-y1);
        y1:=PlotHi;
       end;
   end; {case iy1}
   ix1:= ClipMode(x1,Plotleft,Plotright);
   case ix1 of
    -1:begin
        y1:= y1 + (y2-y1)*(PlotLeft-x1)/(x2-x1);
        x1:= PlotLeft;
       end;
     1:begin
        y1:= y1 +(y2-y1)*(PlotRight-x1)/(x2-x1);
        x1:= Plotright;
       end;
   end; {case ix1}

   case iy2 of
    -1:begin
        x2:=x1 + (x2-x1)*(PlotLo-y1)/(y2-y1);
        y2:=PlotLo;
       end;
     1:begin
        x2:= x1 + (x2-x1)*(PlotHi-y1)/(y2-y1);
        y2:=PlotHi;
       end;
   end; {case iy1}
   ix2:= ClipMode(x2,Plotleft,Plotright);
   case ix2 of
    -1:begin
        y2:= y1 + (y2-y1)*(PlotLeft-x1)/(x2-x1);
        x2:= PlotLeft;
       end;
     1:begin
        y2:= y1 +(y2-y1)*(PlotRight-x1)/(x2-x1);
        x2:= Plotright;
       end;
   end; {case ix1}
   if SetPolar then begin
     apmove(y1,x1);
     apline(y2,x2);
   end else begin
     amove(x1,y1);
     aline(x2,y2);
   end;
  end; {if}
 end;   {ClipDraw}

  procedure BSpline(StepSize:double;Fit,NumOfCpts:integer;
               PlotLeft,PlotRight,PlotLo,PlotHi:double;
                     Cpoint:array of RealPoint);
  {Draws a B-spline curve starting from Cpoint[0] and ending at
   CPoint[NumOfPts]. The intermediate points in the array Cpoint
   serve as control points. The parameter StepSize sets the size
   of the steps between each control point. (e.g. StepSize = 0.05
   sets 20 steps between contro points.) Fit is a parameter which
   selects the fitting strategy. Fit = 0 uses a basic B-spline
   algorithm with end points not fixed. Fit = 1 forces the curve to
   start at Cpoint[0] and end at Cpoint[NumOfCpts]. Increasing Fit
   up to 10 forces closer fitting to the straight lines through the
   control points. Fit between 11 and 20 forces curve through the
   control points. Fit between 21 and 30 does the same using a
   different algorithm. The curve is drawn using the current pen
   colour, style etc. and using the current coordinate transform
   settings. PlotLeft to PlotHi set the plot clipping boundaries.}

   const NSA=1.0/6;
         NSB=4*NSA;

   var  i,is0,is1,j,lastj,xxs:integer;
        x0,y0,x1,x2,y1,y2,x3,y3,m1,m2,c1,c2,Denom,
        tp,t1,t2,t2a,t3,t3a,nc1,nc2,nc3,nc4:double;
        xp,yp:array[0 .. 100] of double;
        Cp:Array[0 .. 127] of Realpoint;
   begin
     if Fit=0 then   {Basic BSpline}
       for i:=0 to NumOfCpts do Cp[i]:= Cpoint[i]
     else if Fit=1 then begin  {Forces fit at ends}
      Cp[0]:= Cpoint[0];
      Cp[1]:= Cpoint[0];
      for i:= 0 to NumOfCpts do Cp[i+2]:= Cpoint[i];
      Cp[NumOfCpts+3]:= Cpoint[NumOfCpts];
      Cp[NumOfCpts+4]:= Cpoint[NumOfCpts];
      NumOfCpts:=NumOfCpts+4;
     end else begin  {other cases}
      x0:= Cpoint[NumOfCpts].x; y0:= Cpoint[NumOfCpts].y;
      i:=NumOfCpts;
      if (Fit>=2) and (Fit<=10) then begin  {Start and end points}
        Denom:=Fit+2;   {Sets offset for new control points}
        Cp[0]:= Cpoint[0];
        Cp[1]:= Cpoint[0];
        Cp[2]:= Cpoint[0];
        Cp[3].x:= Cpoint[0].x + (Cpoint[1].x-Cpoint[0].x)/Denom;
        Cp[3].y:= Cpoint[0].y + (Cpoint[1].y-Cpoint[0].y)/Denom;
        x1:= x0 - (x0-Cpoint[i-1].x)/Denom;
        y1:= y0 - (y0-Cpoint[i-1].y)/Denom;
        Cp[2*i+2].x:= x1; Cp[2*i+2].y:= y1;
        Cp[2*i+3]:= Cpoint[i];
        Cp[2*i+4]:= Cpoint[i];
        Cp[2*i+5]:= Cpoint[i];
      end else if (Fit>10) then begin  {Forced through original control points}
        Denom:= Fit-8;
        Cp[0]:= Cpoint[0];
        Cp[1]:= Cpoint[0];
        Cp[2]:= Cpoint[0];
        Cp[3*i]:= Cpoint[i];
        Cp[3*i+1]:= Cpoint[i];
        Cp[3*i+2]:= Cpoint[i];
      end;
      If Fit>20 then Denom:= Denom-10;  {Reset Denominator for last set}
      {Now process intermediate values. All use similar algorithms.}
      for i:= 1 to NumOfCpts-1 do
      begin
        x0:= Cpoint[i].x; y0:= Cpoint[i].y;
        x1:= x0 + (Cpoint[i-1].x-x0)/Denom; {x1,y1,x2,y2 are added}
        y1:= y0 + (Cpoint[i-1].y-y0)/Denom; {control points}
        x2:= x0 + (Cpoint[i+1].x-x0)/Denom;
        y2:= y0 + (Cpoint[i+1].y-y0)/Denom;
        if (y1=y2) then begin  {Special cases of vertical and }
         y3:= y1; x3:= x0;     {horizontal lines }
        end else if (x1=x2) then begin
         x3:= x1; y3:= y0;
        end else if Fit<=20 then begin  {First algorithm}
         m1:= (y2-y1)/(x2-x1); m2:= -1/m1;
         c1:= y1-m1*x1; c2:= y0-m2*x0;
         x3:= (c2-c1)/(m1-m2);  {x3,y3 is mean intercept of a }
         y3:= m1*x3+c1;         {vertical from original c. p. }
        end else begin           {to line joining new c.p.'s}
         x3:= (x1+x2)/2;    {For 2nd algorithm x3,y3 is centre }
         y3:= (y1+y2)/2;    {point on line joining c.p.'s}
        end; {x3,y3 now set for all conditions}
        x1:= x1+x0-x3;
        y1:= y1+y0-y3;
        x2:= x2+x0-x3;
        y2:= y2+y0-y3;
       {Now check for pathalogical cases - x1 and x2 on same side of x0}
       {This should only be possible for 1t algorithm.}
        if Abs(x2-x0)>Abs(x2-x1) then begin
         x1:=2*x0-x1;y1:=2*y0-y1;
        end else if Abs(x1-x0)>Abs(x2-x1) then begin
         x2:=2*x0-x2;y2:=2*y0-y2;
        end; {x1,y1, x2,y2 now set for all conditions}
        {Now set outputs according to choice of smoothing}
        if (Fit>1) and (Fit<=10) then begin
         Cp[2*i+2].x:= x1; {Original c.p.replaced by two}
         Cp[2*i+2].y:= y1; {offset c.p.'s}
         Cp[2*i+3].x:= x2;
         Cp[2*i+3].y:= y2;
        end else if (Fit>10) then begin
         Cp[3*i].x:=x1; Cp[3*i].y:= y1;     {Offset c.p.'s added on either }
         Cp[3*i+1].x:=x0; Cp[3*i+1].y:= y0; {side of original c.p.}
         Cp[3*i+2].x:=x2; Cp[3*i+2].y:= y2;
        end;
      end; {for i:= 1 to NumOfCpts-1}
      {Finally set the new value for the array length}
      If (Fit>1) and (Fit<=10) then NumOfCpts:= 2*NumOfCpts+5
        else if Fit>10 then NumOfCpts:=3*NumOfCpts+2;
     end; {of other cases}
      {Now evaluate Bspline}
     xxs:= NumOfCpts-2;
     for i:= 1 to xxs do
     begin
      is0:=1;
      tp:=0;
      repeat
       t1:=tp*0.5;
       t2:=sqr(tp);
       t2a:=0.5*t2;
       t3:=t2*tp;
       t3a:=t3*0.5;
       nc1:=nsa-nsa*t3+t2a-t1;
       nc2:=nsb+t3a-t2;
       nc3:=nsa+t1+t2a-t3a;
       nc4:=nsa*t3;
       xp[is0]:= nc1*Cp[i-1].x + nc2*Cp[i].x + nc3*Cp[i+1].x + nc4*Cp[i+2].x;
       yp[is0]:= nc1*Cp[i-1].y + nc2*Cp[i].y + nc3*Cp[i+1].y + nc4*Cp[i+2].y;
       is0:=succ(is0);
       tp:=tp+StepSize;
      until tp>=1.0;
      if i=1 then Lastj:=1;
      for j:=1 to is0-1 do
      begin
         ClipDraw( xp[lastj],yp[lastj],xp[j],yp[j],PlotLeft,PlotRight,
                    PlotHi,PlotLo);
         lastj:=j;
      end; {for j }
       xp[0]:=xp[is0-1];yp[0]:=yp[is0-1];
       lastj:=0;
     end; {for i:=1 to ..}
   end; {bSpline}

 procedure SetPlotDefaults(PlotNumber:integer);
 {Resets  a plot data record to default settings }
 begin
  with Plotdata[PlotNumber] do
  {Plotdata  defines the columns to be used for this plot,
   the plot style, colour etc., the axes used for this plot,
   and the conversion parameters used to convert from
   plot coordinates topaper coordinates.}
  begin
     xcol:= 0;  {default x is first column}
     ycol:= PlotNumber;  {Plot numbers go from 1 to 99 and first y
                                column is col. 1.}
     xaxis:=1; {Define reference axes}
     yaxis:=1;
     zaxis:=1;
     pstyle:= 1;         { *** CHECK THIS ****}
     pwidth:= 0.02;
     pcolor:= black;
     mlcolor:= black;
     mfcolor:= white;
     mborder:= 1;
     msize:= 3;
     msNumber:=0;
     mstyle:= 'circle';
     smoothvalue:= 0;
     errwidth:= 0.1;
     errup:= 0.05;
     errdown:= 0.05;
     herrwidth:= 0.1;
     herrleft:= 0.05;
     herrright:= 0.05;
     markeron:= False;
     LineOn:= False;
     smooth:= False;
  end;
  with GraphTransform do
  begin
     gta:=1;gtb:=0;gtc:=0;   {defaults}
     gtd:=0;gte:=1;gtf:=0;
     xIsLog:=False;
     yIsLog:=False;
     gtype:= 0;
  end;
 end; {setPlotDefaults}

procedure SetUpGraph;
 var i,j:integer;
 begin
  {This will initiate graph parameters}
     SetAxisDefaults;
     gxsize:=15;            {Default Graph Dimensions}
     gysize:=10;
     hscale:= 0.7;          {Default Axis scale factors}
     vscale:= 0.7;
     NoBox:=False;
     Fullsize:=False;
     ShowPlotData:= False;           {Defaults for data display}
     DataAtX:= 1.0; DataAtY:= 10.0;
     DRowStart:= 0; DRowEnd:= 20; DRowStep:=1;
     DColStart:= 0; DColEnd:= 5;  DcolStep:=1;DColwidth:= 1.0;
     SetData:=False;
     DataColNum:=0;
     NumOfDataCols:=0;
     DataRowNum:=0;
     NumOfDatarows:=0;
     CurrentDstart:=0;
     CurrentDEnd:=0;
     LastDStart:=0;
     LastDEnd:=0;
     for i:= 0 to 99 do
     begin
       PlotIsOn[i]:= False;  {Resets all plots prior to input of graph data.}
       SetPlotDefaults(i);
       for j:=0 to 250 do
       begin
        Data[i][j].DataStatus:= 'x';    { Note that because we use column vectors}
        Data[i][j].Value:= j;       { for individual plots this data array is}
       end;                         {in column/row format rather than the    }
     end;                           {normal row/column format.}
   {  gstartx,gstarty,  - Calculated - Start point for graph box}
   {  gxorigin,gyorigin - Calculated - Graph Origin}
   {  x1length,y1length - Calculated - Axes lengths}
   {  x2length,y2length - Calculated - Axes lengths}

   {  UseX1Axis:=True; UseY1Axis:=True; UseX2Axis:=False; UseY2Axis:=False; }
     {set the graph origin}
    gstartx:=CurrentPoint.x;gstarty:=CurrentPoint.y;
    x1axis.NumOfNames:=0;  {Resets counters for Names and places}
    x1axis.NumOfPlaces:=0;
    x2axis.NumOfNames:=0;
    x2axis.NumOfPlaces:=0;
    y1axis.NumOfNames:=0;
    y1axis.NumOfPlaces:=0;
    y2axis.NumOfNames:=0;
    y2axis.NumOfPlaces:=0;
    z1axis.NumOfNames:=0;
    z1axis.NumOfPlaces:=0;
    z2axis.NumOfNames:=0;
    z2axis.NumOfPlaces:=0;
 end; {SetUpGraph}

 procedure SetupPolar;
 var i,j:integer;
 begin
  {This will initiate graph parameters}
     SetAxisDefaults;
     gxsize:=10;            {Default Graph Dimensions}
     gysize:=10;
     hscale:= 0.7;          {Default Axis scale factors}
     vscale:= 0.7;
     NoBox:=False;
     Fullsize:=False;
     ShowPlotData:= False;           {Defaults for data display}
     DataAtX:= 1.0; DataAtY:= 10.0;
     DRowStart:= 0; DRowEnd:= 20; DRowStep:=1;
     DColStart:= 0; DColEnd:= 5;  DcolStep:=1;DcolWidth:= 1.0;
     SetData:=False;
     DataColNum:=0;
     NumOfDataCols:=0;
     DataRowNum:=0;
     NumOfDatarows:=0;
     CurrentDstart:=0;
     CurrentDEnd:=0;
     LastDStart:=0;
     LastDEnd:=0;
     for i:= 0 to 99 do
     begin
       PlotIsOn[i]:= False;  {Resets all plots prior to input of graph data.}
       SetPlotDefaults(i);
       for j:=0 to 250 do
       begin
        Data[i][j].DataStatus:= 'x'; {Note that because we use column vectors}
        Data[i][j].Value:= j;        {for individual plots this data array is}
       end;                          {in column/row format rather than the   }
     end;                            {normal row/column format.}
   {  gstartx,gstarty,  - Calculated - Start point for graph box}
   {  gxorigin,gyorigin - Calculated - Graph Origin}
   {  x1length,y1length - Calculated - Axes lengths}
   {  x2length,y2length - Calculated - Axes lengths}
   {set the graph origin}
    gstartx:=CurrentPoint.x;gstarty:=CurrentPoint.y;
    x1axis.NumOfNames:=0;  {Resets counters for Names and places}
    x1axis.NumOfPlaces:=0;
    x2axis.NumOfNames:=0;
    x2axis.NumOfPlaces:=0;
    y1axis.NumOfNames:=0;
    y1axis.NumOfPlaces:=0;
    y2axis.NumOfNames:=0;
    y2axis.NumOfPlaces:=0;
    z1axis.NumOfNames:=0;
    z1axis.NumOfPlaces:=0;
    z2axis.NumOfNames:=0;
    z2axis.NumOfPlaces:=0;
 end; {SetUpPolar}


procedure SetGraphTransform(xaxis,yaxis:integer;var CurrentGraphTransform:PlotTransform;
               xofs,yofs:double);
 {sets up Transform parameters for Graph to Paper
  xaxis & yaxis define axes used.}
var xlo,xhi,xoffset,ylo,yhi,yoffset:double;
begin
 { ***** ERROR DOES NOT WORK WITH CURRENTGRAPHTRANSFORM ***}
    with CurrentGraphTransform do
     begin
     if xofs>=0 then xoffset:= xofs;
     if yofs>=0 then yoffset:= yofs;
     gta:=1;gtb:=0;gtc:=0;   {defaults}
     gtd:=0;gte:=1;gtf:=0;
     xIsLog:=False;
     yIsLog:=False;
     gtc:= gstartx + xoffset;   {this needs checking !!}
     gtf:= gstarty + yoffset;
     case   xaxis of
      1:CurrentAxis:= x1axis;
      2:CurrentAxis:= x2axis;
      3:CurrentAxis:= a1axis;
      4:CurrentAxis:= a2axis;
     end; {case }
     with CurrentAxis do
     begin
      If AxisType=2  then begin
        xlo:= ln(ScaleLo);xhi:= ln(ScaleHi);
        xIslog:=True;
      end else begin
        xlo:= Scalelo; xhi:=Scalehi;
        xIslog:=False;
      end;
       gta:= AxisLength/(xhi-xlo);
       gtb:= 0; {Currently restricted to vertical y axis}
       gtc:= gtc-gta*xlo;
       if xaxis in [3,4] then
       begin
          gta:= gta*Degree;  {Converts to degree measure}
          if gta>0.0 then gtc:= AxisStartangle*Degree-gta*Xlo
          else gtc:= AxisEndAngle*Degree-gta*Xhi; {Startingpoint in degrees}
          gtype:= 1;  {Check this as a suitable key to switch type}
       end;
     end; {with CurrentAxis do}
     case   yaxis of
      1:CurrentAxis:= y1axis;
      2:CurrentAxis:= y2axis;
      3:CurrentAxis:= r1axis;
      4:CurrentAxis:= r2axis;
      5:CurrentAxis:= r3axis;
      6:CurrentAxis:= r4axis;
     end; {case }
     with CurrentAxis do
     begin
      If AxisType=2  then begin
        ylo:= ln(ScaleLo);yhi:= Ln(ScaleHi);
        yIsLog:=True;
      end else begin
        ylo:= Scalelo; yhi:=Scalehi;
        yIsLog:=False;
      end; {if}
      gtd:= 0; {Currently restricted to horizontal x axis}
      gte:= AxisLength/(yhi-ylo);
      gtf:= gtf-gte*ylo;
      if yaxis in [3 .. 6] then
      begin
        gtf:= innerradius-gte*ylo;
      end;
      xlo:= xlo+0;
     end;  {with CurrentAxis do}
    end; {With GraphTransform}
 end; {SetGraphTransform}


Procedure ConstructGraph;
{Sets up a graph using parameters as set in this module
 This is currently a dummy version for development}
var PlotLo,PlotHi,PlotLeft,PlotRight,LastAngle:double;
    i:integer;
    PolarPlot:Boolean;
begin
  LastFont.assign(CurrentFont);
 {First set up graph box. Note gstartx and gstarty have
  been already set in SetUpgraph.}
 if (not Nobox) and (not Fullsize) then box(gxsize,gysize);
 if Fullsize then
 begin
   hscale:=1.0; vscale:=1.0;
 end;
 if hscale<=0.1 then hscale:=0.7; {otherwise use value set up in data input}
 if vscale<=0.1 then vscale:=0.7;
 gxoffset:= gxsize*(1-hscale )/2; gyoffset:=gysize*(1-vscale)/2;
 {Next set start and end points for each axis.
  We currently assume x axis is horizontal and
  y axis is vertical - but provision for axes at
  arbitrary angles could be included later. }
 with x1axis do
 begin
  xstart:= gStartx+gxoffset; ystart:=gStarty+gyoffset;
  AxisLength:= gxsize*hscale ;   {Note this assumes horizontal axis}
  xend:= xstart +AxisLength; yend:=ystart;
  plotLeft:=xstart; Plotright:= xend;
  If Gridlength=0 then Gridlength:= gysize*vscale;
 end;
 DrawAxis(x1axis);
 with y1axis do
 begin
   xstart:= gStartx+gxoffset; ystart:=gStarty+gyoffset;
   AxisLength:= gysize*vscale;  {assumes vertical axis}
   xend:= xstart; yend:=ystart+AxisLength;
   PlotLo:= ystart; PlotHi:= yend;
   If Gridlength=0 then Gridlength:= gxsize*hscale ;
 end;
 DrawAxis(y1axis);
 with x2axis do
 begin {xstart & ystart are actual axis starting points
        xend & yend are axis end points}
   xstart:= gStartx+gxoffset; ystart:=gStarty+gyoffset+gySize*vscale;
   AxisLength:= gxsize*hscale ;   {assumes horizontal axis}
   xend:= xstart +AxisLength; yend:=ystart;
   If Gridlength=0 then Gridlength:= -gysize*vscale;
 end;
 DrawAxis(x2axis);
 with y2axis do
  begin   {see above for xstart, ystart etc.}
   xstart:= gStartx+gxoffset+gxsize*hscale ; ystart:=gStarty+gyoffset;
   AxisLength:= gysize*vscale;   {Assumes vertical axis }
   xend:= xstart; yend:=ystart+AxisLength;
   If Gridlength=0 then Gridlength:= -gxsize*hscale ;
 end;
 DrawAxis(y2Axis);
 {Finally add graph title}
 amove( gstartx+gxsize/2 + GTitleOffset, gstarty+gysize + GTitleDist);
 LastAngle:= TextAngle;

 CurrentFont.Name:=GTitleFontName;
 CurrentFont.size:=Round(GTitleHei);
 CurrentFont.Color:=GTitleColor;
 Text(Gtitle,TextAngle);
 CurrentFont.Name:='Times New Roman';  {Temporary settings}
 CurrentFont.size:= Round(0.2*PixelsPerCm);        { ***  Check This ***}
 CurrentFont.Color:=black;
 TextAngle:= LastAngle;
 amove(gstartx,gstarty);
 Polarplot:= False;
 InsertPlots(PlotHi,PlotLo,PlotLeft,PlotRight,PolarPlot);
 CurrentFont.assign(LastFont);
end;{ ConstructGraph}

procedure ConstructPolarGraph;
{Sets up a polar graph using parameters as set in this module,
 gstartx, gstarty define centre of  the graph. }
var PlotLo,PlotHi,PlotLeft,PlotRight,LastAngle,
    theta,rout,rin,angstart,angend,
    goffset,xs,ys,xe,ye:double;
    i,color1,color2:integer;
    BStyle1:TBrushStyle;
    PolarPlot:Boolean;

begin
  LastFont.assign(CurrentFont);
 {First set up graph box. Note gstartx and gstarty have
  been already set in SetUpgraph.}
 gxoffset:= gxsize/2; gyoffset:=gysize/2; {sets start at centre of box}
 If gxoffset<gyoffset then goffset:= gxoffset else goffset:= gyoffset;
 if (not Nobox) and (not Fullsize) then
 begin
   rmove(-gxoffset,-gyoffset); {Start for Polar is at centre of box.}
   box(gxsize,gysize);
 end;
 if Fullsize then
 begin
   hscale:=1.0; vscale:=1.0;
 end;
 if hscale<=0.1 then hscale:=0.7; {otherwise use value set up in data input}
 if vscale<=0.1 then vscale:=0.7;
 {Next set start and end points for each axis. Initial value of
  xstart, ystart etc. must be saved rix etc and restored after multiple
  axes have been drawn.}
    with a1axis do
    begin    {Draw this first so the raxis labels will overwrite it.}
     Angleaxis:= True;
     theta:= startat;
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend; {Save values}
     xcentre:= gstartx;
     ycentre:= gstarty;
     if OuterRadius>0.0 then ystart:= hscale*OuterRadius {mod 11/4/2000}
     else ystart:= hscale*goffset;
     if InnerRadius>0.0 then yend:= hscale*InnerRadius
     else yend:= hscale*yend; {goffset fixes outer radius}
     DrawangleAxis(a1axis);
     PlotLeft:= AxisStartAngle*Degree;  {Moved here & Modified  16/4/2000}
     PlotRight:= AxisEndAngle*Degree;   {Must be in degrees. Anticlockwise
                                         rotation assumed.}
     if (PlotRight-PlotLeft)>359.9 then
     begin      {Extends angular ange of plot to cover multiple rotations}
        Plotright:= Plotright+1800;  {Currently set to =/- five rotations}
        PlotLeft:=  Plotleft-1800;
     end;
     xstart:= xs; ystart:= ys;  {Restores original values}
     xend:= xe; yend:= ye;
    end;
    with a2axis do
    begin    {Draw this first so the raxis labels will overwrite it.}
     Angleaxis:= True;
     theta:= startat;
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend; {Save values}
     xcentre:= gstartx;
     ycentre:= gstarty;
     if OuterRadius>0.0 then ystart:= hscale*OuterRadius  {mod 11/4/2000}
     else ystart:= hscale*goffset;
     if InnerRadius>0.0 then yend:= hscale*InnerRadius
     else yend:= hscale*yend; {goffset fixes outer radius}
     DrawangleAxis(a2axis);
     xstart:= xs; ystart:= ys;  {Restores original values}
     xend:= xe; yend:= ye;
    end;
    with r1axis do
    begin
     Polaraxis:= True;
     theta:= startat;
     if (InnerRadius>0.0) then rin:= InnerRadius*hscale
     else rin:= 0.0;
  {   OuterRadius:= GOffset;  }
     if  (outerRadius>0.0) then rout:= OuterRadius*hscale
     else rout:= GOffset*hscale; {Set by graph size }      {mod 11/4/2000}
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend;
     xcentre:= gstartx;
     ycentre:= gstarty;
     PlotLo:= Rin;
     PlotHi:= Rout;
     repeat    {draw multiple copies of axis at defined angles.}
       xstart:= xcentre+rin*cos(theta*DegtoRad);
       ystart:= Ycentre+rin*sin(theta*degTorad);
       xend:= xcentre+rout*cos(theta*DegTorad);
       yend:= ycentre+rout*sin(theta*degToRad);
       DrawAxis(r1axis);
       Theta:= Theta+Stepby;
     until Theta>EndAt;
     xstart:= xs; ystart:= ys; {Restores original values}
     xend:= xe; yend:= ye;
    amove(xcentre,ycentre);
    end;
    with r2axis do
    begin
     Polaraxis:= True;
     theta:= startat;
     if (InnerRadius>0.0) then rin:= InnerRadius*hscale
     else rin:= 0.0;
     if  (outerRadius>0.0) then rout:= OuterRadius*hscale
     else rout:= GOffset*hscale; {Set by graph size }  {mod 11/4/2000}
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend;
     xcentre:= gstartx;
     ycentre:= gstarty;
     repeat    {draw multiple copies of axis at defined angles.}
       xstart:= xcentre+rin*cos(theta*DegtoRad);
       ystart:= Ycentre+rin*sin(theta*degTorad);
       xend:= xcentre+rout*cos(theta*DegTorad);
       yend:= ycentre+rout*sin(theta*degToRad);
       DrawAxis(r2axis);
       Theta:= Theta+Stepby;
     until Theta>EndAt;
     xstart:= xs; ystart:= ys; {Restores original values}
     xend:= xe; yend:= ye;
    amove(xcentre,ycentre);
    end;
    with r3axis do
    begin
     Polaraxis:= True;
     theta:= startat;
     if (InnerRadius>0.0) then rin:= InnerRadius*hscale
     else rin:= 0.0;
     if  (outerRadius>0.0) then rout:= OuterRadius*hscale
     else rout:= GOffset*hscale; {Set by graph size }      {mod 11/4/2000}
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend;
     xcentre:= gstartx;
     ycentre:= gstarty;
     repeat    {draw multiple copies of axis at defined angles.}
       xstart:= xcentre+rin*cos(theta*DegtoRad);
       ystart:= Ycentre+rin*sin(theta*degTorad);
       xend:= xcentre+rout*cos(theta*DegTorad);
       yend:= ycentre+rout*sin(theta*degToRad);
       DrawAxis(r3axis);
       Theta:= Theta+Stepby;
     until Theta>EndAt;
     xstart:= xs; ystart:= ys; {Restores original values}
     xend:= xe; yend:= ye;
    amove(xcentre,ycentre);
    end;
    with r4axis do
    begin
     Polaraxis:= True;
     theta:= startat;
     if (InnerRadius>0.0) then rin:= InnerRadius*hscale
     else rin:= 0.0;
     if  (outerRadius>0.0) then rout:= OuterRadius*hscale
     else rout:= GOffset*hscale; {Set by graph size }      {mod 11/4/2000}
     xs:= xstart;  ys:= ystart; xe:=xend; ye:= yend;
     xcentre:= gstartx;
     ycentre:= gstarty;
     repeat    {draw multiple copies of axis at defined angles.}
       xstart:= xcentre+rin*cos(theta*DegtoRad);
       ystart:= Ycentre+rin*sin(theta*degTorad);
       xend:= xcentre+rout*cos(theta*DegTorad);
       yend:= ycentre+rout*sin(theta*degToRad);
       DrawAxis(r4axis);
       Theta:= Theta+Stepby;
     until Theta>EndAt;
     xstart:= xs; ystart:= ys; {Restores original values}
     xend:= xe; yend:= ye;
    amove(xcentre,ycentre);
    end;
   {Finally add graph title}
     amove( gstartx + GTitleOffset, gstarty+gyoffset + GTitleDist);
     LastAngle:= TextAngle;{TextAngle:=GTitleAngle; }
     CurrentFont.Name:=GTitleFontName;
     CurrentFont.size:=Round(GTitleHei);
     CurrentFont.Color:=GTitleColor;
     Text(Gtitle,TextAngle);
     CurrentFont.Name:='Times New Roman';  {Temporary settings}
     CurrentFont.size:= Round(0.2*PixelsPerCm);        { ***  Check This ***}
     CurrentFont.Color:=black;
     TextAngle:= LastAngle;
     amove(gstartx,gstarty);
   { Polarplot:= False;   }
     InsertPlots(PlotHi,PlotLo,PlotLeft,PlotRight,PolarPlot);
     Displayform.canvas.Brush.Color:= white;
     amove(a1axis.xcentre,a1axis.ycentre);
     If CentreRadius>0.0 then begin
       Color1:= DisplayForm.Image1.canvas.pen.color;
       Color2:= DisplayForm.Image1.Canvas.brush.color;
       BStyle1:= DisplayForm.Image1.Canvas.brush.style;
       try
        try
         DisplayForm.Image1.canvas.pen.color:= CentreLineColor;
         DisplayForm.Image1.Canvas.brush.color:= CentreFillColor;
         DisplayForm.Image1.Canvas.brush.style:= bsSolid;
         Circle(CentreRadius); {Check fill for this}
        except
          on EOverflow do
            Messagedlg('G33: Overflow error in setting central circle ' +
              'for angle axis. Central Circle not set.',mtWarning,[mbOK],0);
          else
            Messagedlg('G34: Error in setting central circle for aaxis. '
            + 'Central Circle not set.',mtWarning,[mbOK],0);
        end;
       finally
        DisplayForm.Image1.canvas.pen.color:= Color1;
        DisplayForm.Image1.Canvas.brush.color:= Color2;
        DisplayForm.Image1.Canvas.brush.style:= BStyle1;
       end;
     end; {if centreradius}
     CurrentFont.assign(LastFont);
  end; {Construct Polar Graph}


Procedure InsertPlots(PlotHi,PlotLo,PlotLeft,PlotRight:double;
                       PolarPlot:Boolean);
{Inputs plots as set up}
var x,y,xp,yp,xp1,yp1,x1,y1:double;
    i,j,k,StartRow,EndRow:Integer;
    Cpt: array[0 .. 250] of RealPoint;
begin
 for  i:= 0 to 99 do
 begin
  if PlotIsOn[i] and DataDefined then    {i.e. This data set is set up}
  begin
   with PlotData[i] do
   begin
    k:= -1;
   repeat
    k:= succ(k);
   until data[xcol][k].DataStatus =' '; { Must start with a number }
   StartRow:= k;
   repeat
    k:= succ(k)
   until (k = 251) or not ( data[xcol][k].DataStatus = ' ');
   EndRow:= pred(k);
    numofpoints:= EndRow-StartRow;
   {First set co-ordinate transform from plot data to paper.
    This uses data from axis settings. - see PlotToPaper}
    SetGraphTransform(xaxis,yaxis,GraphTransform,gxoffset,gyoffset);
   {sets up Transform parameters for Graph to Paper according
    to choice of x & y axes}
    if Lineon then
     begin
      DisplayForm.Image1.canvas.pen.style:= Setpenstyle(pstyle);
      Displayform.Image1.canvas.pen.width:= Round(PixelsPerCm*pwidth);
      DisplayForm.Image1.canvas.pen.color:= pcolor;
      if not smooth then
      begin
       j:= StartRow;
       x:= data[xcol][j].Value; y:= data[ycol][j].Value;
       PlotToPaper(x,y,GraphTransform,xp,yp);
       j:= succ(j);
       repeat
        if data[ycol][j].DataStatus= '_' then
        begin   {Check for a plot termination }
           repeat
             j:= succ(j);
           until (data[ycol][j].DataStatus = '_') or (j = EndRow);
           if j<= EndRow then
           begin   {Reset for start of next plot section }
            j:= succ(j);
            x:= data[xcol][j].Value; y:= data[ycol][j].Value;
            PlotToPaper(x,y,GraphTransform,xp,yp);
            j:= succ(j);
           end;
        end;
        if (data[ycol][j].DataStatus <> '*') then
        begin  {No skipped entries }
         x:=Data[xcol][j].Value; y:=data[ycol][j].Value;
         PlotToPaper(x,y,GraphTransform,xp1,yp1);
         ClipDraw(xp,yp,xp1,yp1,plotLeft,PlotRight,PlotHi,PlotLo);
         xp:=xp1; yp:=yp1;
        end;
        j:= succ(j);  {If entry skipped merely increment j }
       until j> EndRow;
      end else begin  {Case of a smoothed spline curve }
       k:= 0;  {Used as running variable for spline curves}
       j:= StartRow;
       x:= data[xcol][j].Value; y:= data[ycol][j].Value;
       PlotToPaper(x,y,GraphTransform,Cpt[k].x,Cpt[k].y);
       k:= succ(k); j:= succ(j);
       repeat
        if data[ycol][j].DataStatus= '_' then
        begin  {If a termination plot sp[line up to this point. }
         BSpline(0.05,SmoothValue,k-1,PlotLeft,PlotRight,
                PlotLo,PlotHi,Cpt);
         k:= 0; {reset for next curve}
           repeat {Step to start of next plotted section of data.}
             j:= succ(j);
           until (data[ycol][j].DataStatus = '_') or (j = EndRow);
           if j<= EndRow then
           begin  {Start of next plot }
            j:= succ(j);
            x:= data[xcol][j].Value; y:= data[ycol][j].Value;
            PlotToPaper(x,y,GraphTransform,Cpt[k].x,Cpt[k].y);
            k:= succ(k); j:= succ(j);
           end;
        end;
        if (data[ycol][j].DataStatus <> '*') then
        begin
         x:=Data[xcol][j].Value; y:=data[ycol][j].Value;
         PlotToPaper(x,y,GraphTransform,Cpt[k].x,Cpt[k].y);
         k:= succ(k);;
        end;
        j:= succ(j);
       until j> EndRow;
       if k>2 then  {draw final spline curve if there are enough points. }
         BSpline(0.05,SmoothValue,k-1,PlotLeft,PlotRight,
                PlotLo,PlotHi,Cpt);
       k:= 0; {Always reset for next curve}
      end; {Smooth}
     end; {if lineon}

     if MarkerOn then
     begin
      LineOn:= False;
      {set Marker borders}
      if (mborder=1) or (msNumber in [0..7,16..31,40..42]) then
      DisplayForm.Image1.canvas.pen.style:= pSsolid
      else  DisplayForm.Image1.canvas.pen.style:= pSclear;
      Displayform.Image1.canvas.pen.width:= 1;
      DisplayForm.Image1.canvas.pen.color:= mlcolor;
      {Set white backgrounds}
      if msNumber in [32 .. 39] then
      begin
           DisplayForm.Image1.canvas.pen.style:= pSsolid;
           DisplayForm.Image1.canvas.brush.color:= clWhite;
      {set coloured backgrounds}
      end else DisplayForm.Image1.canvas.brush.color:= mfColor;
      if msNumber in [8..15,21,24 .. 31] then
           DisplayForm.Image1.canvas.brush.style:= bsSolid
      {set rest to clear}     
      else DisplayForm.Image1.canvas.brush.style:= bsClear;
      j:= StartRow; {Plot first marker }
      x:= data[xcol][j].Value; y:= data[ycol][j].Value;
      PlotToPaper(x,y,GraphTransform,xp,yp);
      if (Clipmode(xp,PlotLeft,PlotRight)=0)
          and (ClipMode(yp,PlotLo,PlotHi)=0) then
      begin {Select operation according to type of plot}
        if SetPolar then apmove(yp,xp) else amove(xp,yp);
        Marker(mstyle,msize);
      end; {if }
      j:= succ(j);
       repeat
        if data[ycol][j].DataStatus= '_' then
        begin   {Check for a plot termination }
           repeat
             j:= succ(j);
           until (data[ycol][j].DataStatus = '_') or (j = EndRow);
           if j<= EndRow then
           begin   {Reset for next marker }
            j:= succ(j);
            x:= data[xcol][j].Value; y:= data[ycol][j].Value;
            PlotToPaper(x,y,GraphTransform,xp,yp);
            if (Clipmode(xp,PlotLeft,PlotRight)=0)
               and (ClipMode(yp,PlotLo,PlotHi)=0) then
            begin
             if SetPolar then apmove(yp,xp) else amove(xp,yp);
             Marker(mstyle,msize);
            end; {if }
            j:= succ(j);
           end;
        end;
        if (data[ycol][j].DataStatus <> '*') then
        begin  {No skipped entries }
         x:=Data[xcol][j].Value; y:=data[ycol][j].Value;
         PlotToPaper(x,y,GraphTransform,xp,yp);
         if (Clipmode(xp,PlotLeft,PlotRight)=0)
             and (ClipMode(yp,PlotLo,PlotHi)=0) then
         begin
          if SetPolar then apmove(yp,xp) else amove(xp,yp);
          Marker(mstyle,msize);
         end; {if }
        end;
        j:= succ(j);  {If entry skipped merely increment j }
       until j> EndRow;
       DisplayForm.Image1.Canvas.Brush.Style:= bsClear;
     end; {if MarkerOn}
   end; {with PlotData}
  end; {If PlotIsOn}
 end; {for i := }
 CurrentFont.assign(LastFont);
 if ShowPlotData then  CheckData(DataAtX,DataAtY,DColWidth,DRowStart,
          DRowEnd,DRowStep,DColStart,DColEnd,DColStep);
end; {Insert Plots}


procedure InputData(var Colnum:integer;DataSetName:String);
{Organises input of plot data from a file.
 Colnum is the current last data column.}

var DText :TextFile;
    i,j,NumOfEntries,codea,codeb,codec:integer;
    x:double;
    s,s1,s2,s3,s4:string;      {mod Inserted s2,s3 20/4/2000}
    IsText:Boolean;
begin {Open File}
  AssignFile(DText,DataSetName);
  FileMode:= 0;  {Read only}
  {$I-}
    Reset(DText);
  {$I+}
  {File now open - we hope!}
  SetData:=True;
  If Colnum=0 then LastDend:=0 else LastDEnd:= Colnum-1;
  CurrentDstart:=Colnum;
  Readln(DText,s);
  repeat
      s:= s+'  '; {Needed to avoid loss of last character in string }
      Analyse(s,NumOfEntries);
      if StringType[1]<>7 then  {Not a comment line}
      begin
       NumOfEntries:=pred(NumOfEntries); {Remove return at end of line}
       if DataRowNum=0 then    {The first line of data}
       begin
         NumOfDataCols:=NumOfEntries;  {Set number of columns}
       end else if NumOfDataCols<>NumofEntries then
       begin  {For subsequent rows first check number of columns is the same }
          Messagedlg('G35: Error in number of entries at row ' +
           IntToStr(DataRowNum) + ' of ' + DataSetName,mtError,[mbAbort],0);
           CloseFile(DText);
           SetData:= False;
           Disaster:=True;
           exit;
       end;
       for i:= 1 to  NumOfEntries do  {if all O.K. enter data
           Remember that first entry is reserved for labels!}
       begin
        DataColNum:= CurrentDStart+pred(i); {First data entry i in column 0}
        with  Data[DataColNum][DataRowNum] do {Change for multiple data sets! }
        begin
         if (StringType[i]=46) then begin  {A special data symbol}
          case  StringText[i][1] {Round(StringValue[i])} of
           '*'{2}:begin  {A missing entry}
               DataStatus:= '*';  Value := 1.0e-50;
             end;
           '>'{3}:begin {Copy the first column}
               DataStatus:= Data[0][DataRowNum].DataStatus;
               Value:= Data[0][DataRowNum].Value;
             end;
           '<'{4}:begin {Copy the last column of the previous data set}
               DataStatus:= Data[LastDend][DataRowNum].DataStatus;
               Value:= Data[LastDend][DataRowNum].Value;
             end;
           '|'{5}:begin {Copy corresponding column of the previous data set.}
               DataStatus:= Data[LastDstart+pred(i)][DataRowNum].DataStatus;
               Value:= Data[LastDStart+pred(i)][DataRowNum].Value;
             end;
           '_':begin {Start or end of data}
               DataStatus:= '_';  Value := StringValue[i];
             end;
          end; {case}
          {Local variables may have been changed - so reset these.}
          vv[i+29]:= Value;
         end else begin  {Simple data input.}
          {First evaluate outstanding expressions.}
          if StringType[i]=6 then
          begin
           codea:=0;codeb:=0;codec:=0;
           Value:=Expression(StringText[i],s4,IsText,
                              codea,codeb,codec);
           if codea=0 then  begin
              vv[i+29]:= Value;
              StringValue[i]:= Value;
           end else begin { Errors! }
            Str(DataRowNum:4,S2);      {Mod 20/4/2000}
            Str(i-1: 4,S3);
            CloseFile(DText);         { ** Added 25/1/99  **}
            Messagedlg('G36: Error during data input from dataset ' +
            DatasetName + '   Check line ' +  s2 + ' at column ' + s3,
            mtError,[mbAbort],0);
            Disaster:=True;
            Exit;
           end; { if codea }
          end;  { if StringType[i]=6 }
          DataStatus:= ' ';  Value:= StringValue[i];
         end; {if StringType}
        end; {with Data }
        DataColNum:=Succ(DataColNum);
       end; {for i}
       DataRowNum:= succ(DataRowNum);
      end; {If not a comment}
      Readln(DText,s);
     until Eof(Dtext);
     CloseFile(DText);
     SetData:=False; {Reset for next data input}
     if Colnum=0 then NumOfDataRows:= DataRowNum
     else if NumOfDataRows<>DataRowNum then
     begin   {I think this message may not be needed!}
       Messagedlg('G37 Number of rows in ' + DataSetName +
       ' differs from that of last data set.',
       mtError,[mbAbort],0);
       SetData:= False;
       Disaster:=True;
       exit;
     end;
     DataRowNum:=0;
     Colnum:= CurrentDstart+NumOfEntries;
end; {input Data}

procedure CheckData(xs,ys,cwidth:double;
                    Rstart,Rend,Rstep,Cstart,Cend,Cstep:integer);
{ Display of a section of the data array. Locate top left at point x,y.
  Display from Rstart to Rend in steps of Rstep and from Cstart to Cend
  in steps of Cstep. Columns are spaced by cwidth. As currently set, the
  font, text colour,  number filed etc. are those currently set before the
  command begin graph.  }

  var i,j:integer;
      x,y:double;
      s:string;
  begin
     amove(xs,ys);
     i:= Rstart;
     y:= ys;
     repeat
       j:= Cstart;
       repeat
        x:= data[j][i].Value;
        Str(x:SigFigs:Decfigs,s);
        Text(s,0);
        rmove(cwidth,0);
        j:= j + Cstep;
       until (j >= (CEnd+1));
       i:= i + Rstep;
       y:= y - 0.4;
       amove(xs,y);
     until (i >= (Rend+1));
  end;{checkData}


initialization
 Setaxisdefaults;
end.
