/* Quat - A 3D fractal generation program */
/* Copyright (C) 1998 Dirk Meyer */
/* (email: dirk.meyer@studbox.uni-stuttgart.de) */
/* mail:  Dirk Meyer */
/*        Marbacher Weg 29 */
/*        D-71334 Waiblingen */
/*        Germany */
/* */
/* This program is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2 */
/* of the License, or (at your option) any later version. */
/* */
/* This program is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */
/* */
/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software */
/* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>    
#include <math.h> /* to calc. angle in view editor, floor */
#include <windows.h>
#include <commdlg.h>
#include <time.h>
#include "quat.h"            
#include "qmath.h"
#include "iter.h" /* calcbase */ 
#include "colors.h"
#include "win2.h"
#include "calculat.h"  /* IsNumber */
#include "winver.h"
#include "resource.h"
   
#ifdef WIN32
#define WM_CTLCOLOR	   WM_CTLCOLOREDIT
#endif

int CalcMPixel(int x, int y, double bailout, int maxiter, int formula, double ck, double cl);
int CalcJPixel(int x, int y, double bailout, double kvalue, double lvalue, int maxiter, int form,
   double cx, double cy, double ck, double cl);
int MSWIN_SetColors(struct disppal_struct *disppal);
int MSWIN_SetColors2(struct disppal_struct *disppal);
int MSWIN_update_bitmapDLG(long x1, long x2, long xres, int y, unsigned char *Buf, int which);
void MSWIN_eol2(int line);

/* Defined in quat.c */
extern VER_ReturnVideoInfo ReturnVideoInfo;
extern VER_SetColors SetColors;
extern VER_Initialize Initialize;
extern VER_Done Done;
extern VER_update_bitmap update_bitmap;
extern VER_getline getline;
extern VER_putline putline;
extern VER_check_event check_event;
extern VER_Change_Name Change_Name;
extern VER_Debug Debug;
extern VER_eol eol;

struct ViewSelectorStruct
{                    
   HWND hWnd;   /* to identify this structure */
   int used;  
   HDC hDC;
   HBITMAP hBitmap, hOldBitmap;
   int x, y;   /* which axis is mapped in x, y directions: 0=e, 1=j, 2=k */
   int dx, dy;  /* direction: pos or neg */
   int vx, vy;   /* viewpoint in bitmap coordinates */      
   int state;   /* 0.. normal, 1.. dragging */
} ;

HDC hDC1;
HDC hDC2;   
HBITMAP hBitmap1, hOldBitmap1, hBitmap2, hOldBitmap2;
HWND hBar;   /* ScrollBar in ObjectEditor */
HPALETTE hOldEdPal, hEdPal;  /* palette set by MSWIN_SetColors2 (for Obj & Col Editor) */

extern unsigned char imginmem, zbufinmem;
extern HINSTANCE hInst;
extern struct frac_struct *frac;
extern struct view_struct *view;
extern struct realpal_struct *realpal;
extern double *cut_buf;
extern char *colscheme;  
extern char *act_file;
extern char *act_name;  
extern HDC hMemoryDC;
extern HWND myhWnd; 
extern int picx, picy, pixx, pixy;  
extern time_t calc_time;   
extern int pixperevent;
extern char TrueCol;

double mandel_x, mandel_y, mandel_lx, mandel_ly;
int old_x, old_y;
HWND hObjDlg, hViewDlg, hJulPrev, hColDlg, hHereSetColors;   
HCURSOR hCur;
struct ViewSelectorStruct VS[3];  
struct view_struct *vtmp, *v;  
struct realpal_struct *tmppal;
double *tmpcut; 
int numcut;

int flag;

int DoReadINI(HWND hWnd, char *givenfile)
{
   OPENFILENAME ofn;
   char chReplace, *filter, *file, *file_t, *Error, c=0;
   struct frac_struct f;
   struct view_struct v;
   struct realpal_struct r;
   char cscheme[251];
   double cbuf[140];
   int i;
                           
   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(256);
   if (!filter || !file || !file_t || !Error)
   {
      free(filter); free(file); free(file_t); free(Error);
      MessageBox(hWnd, "DoReadINI: Memory allocation failed.", "Quat: Error", MB_OK);
      return 0;
   }
   file[0] = 0;
   if (!givenfile)
   {
      strcpy(filter, "INI Files (*.ini)|*.ini|COL Files (*.col)|*.col|All files (*.*)|*.*|");
      chReplace = filter[strlen(filter) - 1];
      for (i = 0; filter[i] != '\0'; i++) 
         if (filter[i] == chReplace) filter[i] = '\0';

      memset(&ofn, 0, sizeof(OPENFILENAME));
                           
      ofn.lStructSize = sizeof(OPENFILENAME);
      ofn.hwndOwner = hWnd;
      ofn.lpstrFilter = filter;
      ofn.nFilterIndex = 1;
      ofn.lpstrFile = file;                       
      ofn.lpstrTitle = "Read parameters from INI";
      ofn.nMaxFile = 256;
      ofn.lpstrFileTitle = file_t;
      ofn.nMaxFileTitle = 256;
      ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
                           
      c = GetOpenFileName(&ofn);
   }
   else
   {
      strncpy(file, givenfile, 256);
      strncpy(file_t, strrchr(givenfile, '\\')+1, 256);
   }
   if (givenfile || c)
   {
      f = *frac; v = *view; r = *realpal;
      memcpy(cbuf, cut_buf, 140*sizeof(double));
      strncpy(cscheme, colscheme, 251);
      i = ParseINI(file, Error, &f, &v, &r, cscheme, cbuf);
      free(filter); free(file_t); free(file);
      if (i!=0) 
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         free(Error);
         return(-1); 
      }
      else 
      {
         /* When ZBuffer in memory, only read light or colors or rendering parameters */
         if (zbufinmem)
         {
            for (i=0; i<3; i++) view->light[i] = v.light[i];
            view->ambient = v.ambient;
            view->phongmax = v.phongmax;
            view->phongsharp = v.phongsharp;
        /*    view->antialiasing = v.antialiasing;   */
            *realpal = r;
            strncpy(colscheme, cscheme, 251);
            MessageBox(hWnd, "Those parameters that can be modified for a Z Buffer have been read successfully.", "Quat: Message", MB_OK);
         }                                   
         else
         {
            *frac = f; *view = v; *realpal = r;
            memcpy(cut_buf, cbuf, 140*sizeof(double));
            strncpy(colscheme, cscheme, 251);
            MessageBox(hWnd, "INI file read successfully. Parameters added to current ones. Maybe you wanted to reset"
               " them before reading", "Quat: Message", MB_OK);
         }
         free(Error);
         return(0);
      }
   }
   else 
   {
      free(filter); free(file_t); free(file); free(Error);
      return(0);
   }
}                           

int DoReadPNG(HWND hWnd)
{
   OPENFILENAME ofn;
   char chReplace, *filter, *file, *file_t, *Error;
   int i;
                           
   filter = malloc(256); file = malloc(256); file_t = malloc(256); Error = malloc(300);
   file[0] = 0;
   strcpy(filter, "PNG images (*.png)|*.png|Z-Buffers (*.zpn)|*.zpn|All files (*.*)|*.*|"); 
   chReplace = filter[strlen(filter) - 1];
   for (i = 0; filter[i] != '\0'; i++) 
      if (filter[i] == chReplace) filter[i] = '\0';

   memset(&ofn, 0, sizeof(OPENFILENAME));
   
   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hWnd;
   ofn.lpstrFilter = filter;
   ofn.nFilterIndex = 1;
   ofn.lpstrFile = file;    
   ofn.lpstrTitle = "Read parameters from PNG";
   ofn.nMaxFile = 256;
   ofn.lpstrInitialDir = "";
   ofn.lpstrFileTitle = file_t;
   ofn.nMaxFileTitle = 256;
   ofn.Flags = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;
                           
   if (GetOpenFileName(&ofn))
   {                           
      HDC hDC;
      
      if (hMemoryDC!=0)
      {
         hDC = GetDC(myhWnd);
         Rectangle(hMemoryDC, 0, 0, picx, picy);
         ReleaseDC(myhWnd, hDC);        
         InvalidateRect(myhWnd, NULL, FALSE);
         UpdateWindow(myhWnd);
      }
      i = ReadParametersPNG(file, Error, frac, view, realpal, colscheme, cut_buf);
      free(file_t); free(filter); free(file);
      if (i!=0) 
      {
         MessageBox(hWnd, Error, "Quat: Error", MB_OK);
         free(Error);
         return(-1); 
      }
      else 
      {
         free(Error);
         MessageBox(hWnd, "Parameters read successfully.", "Quat: Message", MB_OK);
         return(0);
      }
   }
   else 
   {
      free(file); free(Error); free(file_t); free(filter);
      return(-1);
   }
}                             

int DoNew(HWND hWnd)
{
   if (MessageBox(hWnd, "This will set the parameters to their default values.", 
      "Quat: Notification", MB_OKCANCEL) == IDOK)
   {
      SetDefaults(frac, view, realpal, colscheme, cut_buf);
      if (hMemoryDC!=0)
      {
         HDC hDC;
         hDC = GetDC(myhWnd);
         Rectangle(hMemoryDC, 0, 0, picx, picy);
         ReleaseDC(myhWnd, hDC);        
         InvalidateRect(myhWnd, NULL, FALSE);
         UpdateWindow(myhWnd);
      }             
      calc_time = 0;
      MessageBox(hWnd, "Parameters reset to default.", "Quat: Message", MB_OK);
   }
   return(0);
}   

int DoParamSave(HWND hWnd, char *ftype, const char *def_suffix, unsigned char mode)
{
   OPENFILENAME ofn;
   char chReplace, *filter, *file, *file_t;
   char Error[256];          
   int i;
                           
   filter = malloc(256); file = malloc(256); file_t = malloc(256);
   if (!filter || !file || !file_t)
   {
      free(filter); free(file); free(file_t);
      MessageBox(hWnd, "DoParamSave: Memory allocation failed.", "Quat: Error", MB_OK);
      return 0;
   }
   if ((mode & PS_USEFILE)!=0) 
   {
      strncpy(file, act_name, 256);  
      BuildName(file, NULL, ".ini", file, Error);
   }
   else *file = 0;
   strncpy(filter, ftype, 256);
   strncat(filter, "|All files (*.*)|*.*|", 21);
   chReplace = filter[strlen(filter) - 1];
   for (i = 0; filter[i] != '\0'; i++) 
      if (filter[i] == chReplace) filter[i] = '\0';

   memset(&ofn, 0, sizeof(OPENFILENAME));
                           
   ofn.lStructSize = sizeof(OPENFILENAME);
   ofn.hwndOwner = hWnd;
   ofn.lpstrFilter = filter;
   ofn.nFilterIndex = 1;
   ofn.lpstrFile = file;
   ofn.nMaxFile = 256;
   ofn.lpstrFileTitle = file_t;
   ofn.nMaxFileTitle = 256;
   ofn.lpstrDefExt = def_suffix;
   ofn.Flags = OFN_OVERWRITEPROMPT | OFN_HIDEREADONLY;
                           
   if (!GetSaveFileName(&ofn))
   {
      free(filter); free(file); free(file_t);
      return 0;
   }
   
   if (!strchr(file, '.')) strcat(file, def_suffix);
   if (!WriteINI(Error, file, mode, frac, view, realpal, colscheme, cut_buf))
      MessageBox(hWnd, "File written.", "Quat: Message", MB_OK);
   else MessageBox(hWnd, Error, "Quat: Error", MB_OK);
   
   free(filter); free(file); free(file_t);
   return(0);
}

int CalcMPixel(int x, int y, double bailout, int maxiter, int formula, double ck, double cl)
{
   double xr, yr, p[4], it, q[4];
   point x1, x2, x3, x4, x5, c; 
   point one = { 1, 0, 0, 0 };
   int i;
   
   if (formula != 0 && formula != 1) return(0);
   yr = mandel_y+(double)y*mandel_ly/60;
   xr = mandel_x+(double)x*mandel_lx/80;
   it = 0; 
   if (formula==0)
   {
      forall(i,4) p[i] = 0;
      forall(i,4) q[i] = 0; 
   }
   else if (formula==1)
   {
      forall(i,4)
      {
         x1[i] = 0;
         x1[0] = 0.5;
         q[i] = x1[i]*x1[i];
      }                    
      c[0] = xr; c[1] = yr; c[2] = ck; c[3] = cl;
   }
   while (it<maxiter && q[0]+q[1]+q[2]+q[3]<bailout)
   {
      if (formula==0)
      { 
         p[1] = 2*p[0]*p[1] - yr;
         p[2] = 2*p[0]*p[2] - ck;
         p[3] = 2*p[0]*p[3] - cl;
         p[0] = q[0] - q[1] - q[2] - q[3] - xr;
         forall(i,4) q[i] = p[i] * p[i];
      }
      else if (formula==1)
      {
         forall(i,4) x3[i] = x1[i];
         q_sub(x2, one, x1);      /* 1 - x[n] */
         q_mul(x4, c, x3);        /* c*x[n] */
         q_mul(x5, x4, x2);       /* c*x[n]*(1-x[n]) */
         forall(i,4) x1[i] = x5[i];
         forall(i,4) q[i] = x1[i]*x1[i];
      } 
      it++;
   }
   if (it<maxiter) return(1); else return(0);
}
   
int CalcJPixel(int x, int y, double bailout, double kvalue, double lvalue, int maxiter, int form,
   double cx, double cy, double ck, double cl)
{
   double tmp, p[4], q[4]; 
   point x1, x2, x3, x4, x5, c;
   point one = { 1, 0, 0, 0 };
   int i, it;
   
   if (form != 0 && form != 1) return(0);
   p[0] = -2+(double)x/20; /* *4/80; */
   p[1] = -1.5+(double)y/20; /* *3/60; */
   p[2] = kvalue; p[3] = lvalue;  
   it = 0;
   if (form == 0) forall(i,4) q[i] = p[i]*p[i];
   else if (form == 1) 
   {
      forall(i,4) x1[i] = p[i];
      c[0] = cx; c[1] = cy; c[2] = ck; c[3] = cl;
      forall(i,4) q[i] = p[i]*p[i];
   }
   while (it<maxiter && q[0]+q[1]+q[2]+q[3]<bailout)
   {
      if (form == 0)
      {
         tmp = p[0]*p[1]; p[1] = tmp + tmp - cy;
         tmp = p[0]*p[2]; p[2] = tmp + tmp - ck;
         tmp = p[0]*p[3]; p[3] = tmp + tmp - cl;
         p[0] = q[0] - q[1] - q[2] - q[3] - cx;
         forall(i,4) q[i] = p[i]*p[i];
      }
      else if (form == 1)
      {
         forall(i,4) x3[i] = x1[i];
         q_sub(x2, one, x1);      /* 1 - x[n] */
         q_mul(x4, c, x3);        /* c*x[n] */
         q_mul(x5, x4, x2);       /* c*x[n]*(1-x[n]) */
         forall(i,4) x1[i] = x5[i];
         forall(i,4) q[i] = x1[i]*x1[i];
      }
      it++;
   }
   if (it<maxiter) return(1); else return(0);
}
   
int UpdateParameters(struct frac_struct *f, struct view_struct *v, struct realpal_struct *r, char *col,
   double *cut)
/* Called by quat.c, if a function wants to tell its locally stored parameters */
{                                                                       
   int i;
   
   *frac = *f; *view = *v; *realpal = *r; strncpy(colscheme, col, 251); 
   for (i=0; i<140; i++) cut_buf[i] = cut[i];
   
   return(0);
}

LONG APIENTRY MandelSelector(hWnd, message, wParam, lParam)
register HWND hWnd;                   
UINT message;                
WPARAM wParam;               
LPARAM lParam;               
{      
   switch (message)
   {
      case WM_CREATE:      
      {
         HDC hDC;
         HBRUSH hBrush, hOldBrush;
         HPEN hPen, hOldPen;

         old_x = -1; old_y = -1;
         hDC = GetDC(hWnd);
         hBitmap1 = CreateCompatibleBitmap(hDC, 80, 60);
         hDC1 = CreateCompatibleDC(hDC);
         hOldBitmap1 = SelectObject(hDC1, hBitmap1);
         hBrush = CreateSolidBrush(RGB(0, 0, 0));
         hOldBrush = SelectObject(hDC1, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(hDC1, hPen);
         Rectangle(hDC1, 0, 0, 80, 60);
         SelectObject(hDC1, hOldBrush);
         SelectObject(hDC1, hOldPen);
         DeleteObject(hBrush); DeleteObject(hPen);
         ReleaseDC(hWnd, hDC);        
                                       
         flag = 0;
         mandel_x = -1; mandel_y = -1.5; mandel_lx = 4; mandel_ly = 3;
         break;
      }
      case WM_DESTROY:
         if (hOldBitmap1!=0) SelectObject(hDC1, hOldBitmap1);
         hOldBitmap1 = 0;
         DeleteObject(hBitmap1);
         DeleteDC(hDC1); 
         break;
      case WM_PAINT:
      {
         RECT R;
         HDC hDC;
         PAINTSTRUCT ps;
         
         GetUpdateRect(hWnd, &R, FALSE);
         {       
             hDC = BeginPaint(hWnd, &ps);

             BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, 
                hDC1, R.left, R.top, SRCCOPY);
             EndPaint(hWnd, &ps);
             ReleaseDC(hWnd, hDC);
         }
         break;
      }
      case WM_LBUTTONDOWN:
      {
         int x, y;
         char s[40];
         
         if (imginmem || zbufinmem) break;
         x = LOWORD(lParam); y = HIWORD(lParam);
         if (x>=0 && x<80 && y>=0 && y<60)
         {                                      
            SetCapture(hWnd); flag = 1;    
            hCur = SetCursor((HCURSOR)NULL);
            sprintf(s, "%.15f", mandel_x+(double)x/80*mandel_lx); CleanNumberString(s);
            SetDlgItemText(hObjDlg, IDC_EDIT4, s);
            sprintf(s, "%.15f", mandel_y+(double)y/60*mandel_ly); CleanNumberString(s);
            SetDlgItemText(hObjDlg, IDC_EDIT5, s);
         }
         break;
      }         
      case WM_LBUTTONUP:
      {        
         int x, y;
         
         x = LOWORD(lParam);
         y = HIWORD(lParam);
         if (flag == 1)
         {
            SetCursor(hCur);
            flag = 0; ReleaseCapture();
          /*  SendMessage(hObjDlg, WM_USER+1, 0, 0); */
         }
         break;
      }
      case WM_MOUSEMOVE:                    
      {
         int x, y;
         char s[40]; 
         
         x = LOWORD(lParam);
         y = HIWORD(lParam);
         if (x<0) x = 0; if (x>80) x = 80; if (y<0) y=0; if (y>60) y = 60;
         if (flag == 1)
         {                                                                
            sprintf(s, "%.15f", mandel_x+(double)x/80*mandel_lx); CleanNumberString(s);
            SetDlgItemText(hObjDlg, IDC_EDIT4, s);
            sprintf(s, "%.15f", mandel_y+(double)y/60*mandel_ly); CleanNumberString(s);
            SetDlgItemText(hObjDlg, IDC_EDIT5, s);
            /* SendMessage(hViewDlg, WM_USER, 0, 0); */
         }
         break;
      }
      case WM_RBUTTONDOWN:
      {
         SendMessage(hWnd, WM_USER, 0, 0);
      }
      case WM_USER:   /* mandelbrot changed */
      {                  
         int x, y, form;               
         char s[40];
         RECT R;  
         double bailout, ck, cl;
         int maxiter;
         HCURSOR hcurSave; 

         hcurSave = SetCursor(LoadCursor((HANDLE)NULL, IDC_WAIT));
         SetCapture(hWnd);
         GetDlgItemText(hObjDlg, IDC_EDIT1, s, 40);
         bailout = atof(s);     
         bailout *= bailout;
         GetDlgItemText(hObjDlg, IDC_EDIT2, s, 40);
         maxiter = atoi(s);
         GetDlgItemText(hObjDlg, IDC_EDIT6, s, 40);
         ck = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT7, s, 40);
         cl = atof(s);
         form = (int)SendDlgItemMessage(hObjDlg, IDC_LIST1, LB_GETCURSEL, 0, 0L);
         /* Calculate fractal (only for ObjectDialog) */
         for (y=0; y<60; y++)
         {
            for (x=0; x<80; x++)
            {
               if (CalcMPixel(x,y,bailout,maxiter,form,ck,cl)==0) SetPixel(hDC1, x, y, RGB(0,0,0));
               else SetPixel(hDC1, x, y, RGB(255,255,255));
            }
                 
         }
         SetCursor(hcurSave);
         ReleaseCapture();
         SendMessage(hWnd, WM_USER+1, 1, 0);
         R.left = 0; R.top = 0; R.right = 80; R.bottom = 59;
         InvalidateRect(hWnd, &R, TRUE);          
      }                                         
      case (WM_USER+1):       /* Cursor changed */
      {        
         /* wParam: 0: No repaint done */
         
         int x, y, form;
         double bailout, cx, cy,ck,cl;
         int maxiter;   
         char s[40];
         RECT R;
         
         GetDlgItemText(hObjDlg, IDC_EDIT4, s, 40);
         cx = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT5, s, 40);
         cy = atof(s);
         x = (int)((cx - mandel_x)/mandel_lx*80); y = (int)((cy - mandel_y)/mandel_ly*60);
         /* Cursor not changed and no move/resize? */
         if (x==old_x && y==old_y && wParam==0) break;
         GetDlgItemText(hObjDlg, IDC_EDIT6, s, 40);
         ck = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT7, s, 40);
         cl = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT1, s, 40);
         bailout = atof(s); 
         bailout *= bailout;
         form = (int)SendDlgItemMessage(hObjDlg, IDC_LIST1, LB_GETCURSEL, 0, 0L);
         GetDlgItemText(hObjDlg, IDC_EDIT2, s, 40);
         maxiter = atoi(s);
         if (wParam==0)
         {
            if (old_x-2>=0)
            {
               if (CalcMPixel(old_x-2, old_y,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x-2, old_y, RGB(0,0,0)); 
                  else SetPixel(hDC1, old_x-2, old_y, RGB(255,255,255));  
            }
            if (old_x-1>=0)
            {
               if (CalcMPixel(old_x-1, old_y,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x-1, old_y, RGB(0,0,0));
                  else SetPixel(hDC1, old_x-1, old_y, RGB(255,255,255));  
            }
            if (old_x+1<80)
            {
               if (CalcMPixel(old_x+1, old_y,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x+1, old_y, RGB(0,0,0));
                  else SetPixel(hDC1, old_x+1, old_y, RGB(255,255,255));  
            }
            if (old_x+2<80)
            {
               if (CalcMPixel(old_x+2, old_y,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x+2, old_y, RGB(0,0,0));
                  else SetPixel(hDC1, old_x+2, old_y, RGB(255,255,255));  
            }
            if (old_y-2>=0)
            {
               if (CalcMPixel(old_x, old_y-2,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x, old_y-2, RGB(0,0,0)); 
                  else SetPixel(hDC1, old_x, old_y-2, RGB(255,255,255));  
            }
            if (old_y-1>=0)
            {
               if (CalcMPixel(old_x, old_y-1,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x, old_y-1, RGB(0,0,0)); 
                  else SetPixel(hDC1, old_x, old_y-1, RGB(255,255,255));  
            }
            if (old_y+1<=60)
            {
               if (CalcMPixel(old_x, old_y+1,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x, old_y+1, RGB(0,0,0)); 
                  else SetPixel(hDC1, old_x, old_y+1, RGB(255,255,255));  
            }
            if (old_y+2<=60)
            {
               if (CalcMPixel(old_x, old_y+2,bailout,maxiter,form,ck,cl)==0)
                  SetPixel(hDC1, old_x, old_y+2, RGB(0,0,0)); 
                  else SetPixel(hDC1, old_x, old_y+2, RGB(255,255,255));  
            }
         }
         old_x = x; old_y = y;
         SetPixel(hDC1, x-2, y, RGB(255,0,0));
         SetPixel(hDC1, x-1, y, RGB(255,0,0));
         SetPixel(hDC1, x+1, y, RGB(255,0,0));
         SetPixel(hDC1, x+2, y, RGB(255,0,0));
         SetPixel(hDC1, x, y-2, RGB(255,0,0));
         SetPixel(hDC1, x, y-1, RGB(255,0,0));
         SetPixel(hDC1, x, y+1, RGB(255,0,0));
         SetPixel(hDC1, x, y+2, RGB(255,0,0));
         R.left = 0; R.top = 0; R.right = 79; R.bottom = 59;
         InvalidateRect(hWnd, &R, TRUE);
         break;
      }
   }
   return(TRUE);
}

int MSWIN_SetColors2(struct disppal_struct *disppal)
{            
   LOGPALETTE *plgpl;
   HDC hDC;
   int i, bits;
   HPALETTE hP1, hP2;
                                             
   plgpl = (LOGPALETTE *) malloc(sizeof(LOGPALETTE) + disppal->maxcol*sizeof(PALETTEENTRY));
   if (!plgpl) return(-1);
   /*hDC = GetDC(NULL);*/
   bits = GetDeviceCaps(hDC2, COLORRES)/3;
   plgpl->palNumEntries=disppal->maxcol;
   plgpl->palVersion = 0x300;
   for (i=0; i<disppal->maxcol; i++)
   {
      plgpl->palPalEntry[i].peRed = disppal->cols[i].r<<(8-bits);
      plgpl->palPalEntry[i].peGreen = disppal->cols[i].g<<(8-bits);
      plgpl->palPalEntry[i].peBlue = disppal->cols[i].b<<(8-bits);
      plgpl->palPalEntry[i].peFlags = 0;
   }
   if (hEdPal)
   {
      DeleteObject(hEdPal);
      hEdPal = 0;
   }
   hEdPal = CreatePalette(plgpl);
   free(plgpl);
   /* ReleaseDC(NULL, hDC);*/
   hDC = GetDC(hHereSetColors);
   hP1 = SelectPalette(hDC, hEdPal, 0); 
   hP2 = SelectPalette(hDC2, hEdPal, 0);
   /* hOldPal2 = SelectPalette(hMemoryDC, hPal, 0); */
   i = RealizePalette(hDC);
   /*hP2 = SelectPalette(hDC, hP1, 0);*/
   ReleaseDC(hHereSetColors, hDC);
   if (i>0) InvalidateRect(hHereSetColors, (LPRECT)NULL, 1);
   return(0);
}                

int MSWIN_update_bitmapDLG(long x1, long x2, long xres, int y, unsigned char *Buf, int which)
{
   long i;
   unsigned int idx;

   for (i=x1; i<=x2; i++) 
   {
      if (!TrueCol)
      {
         idx = GetNearestPaletteIndex(hEdPal, RGB(Buf[3*i], Buf[3*i+1], Buf[3*i+2]));        
         SetPixel(hDC2, i+1, y+1, PALETTEINDEX(idx));
      }	
      else
         SetPixel(hDC2, i, y, RGB(Buf[3*i], Buf[3*i+1], Buf[3*i+2]));  
   }
   return(0);
}

void MSWIN_eol2(int line)
{
   RECT R;
   R.left = 0; R.top = line-1; R.right = 80; R.bottom = line+1;
   InvalidateRect(hJulPrev, &R, TRUE);
 /*  UpdateWindow(hJulPrev); */
   return;
}

LONG APIENTRY JuliaPreview(hWnd, message, wParam, lParam)
register HWND hWnd;                   
UINT message;                
WPARAM wParam;               
LPARAM lParam;                           
{
   switch (message)
   {
      case WM_CREATE:      
      {
         HDC hDC;    
         HBRUSH hBrush, hOldBrush;
         HPEN hPen, hOldPen;

         hJulPrev = hWnd;
         old_x = -1; old_y = -1;
         hDC = GetDC(hWnd);
         hBitmap2 = CreateCompatibleBitmap(hDC, 80, 60);
         hDC2 = CreateCompatibleDC(hDC);
         hOldBitmap2 = SelectObject(hDC2, hBitmap2);
         hBrush = CreateSolidBrush(RGB(0, 0, 0));
         hOldBrush = SelectObject(hDC2, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(hDC2, hPen);
         Rectangle(hDC2, 0, 0, 80, 60);              
         SelectObject(hDC2, hOldBrush);
         SelectObject(hDC2, hOldPen);
         DeleteObject(hPen); DeleteObject(hBrush);
         ReleaseDC(hWnd, hDC);        
         break;
      }                      
      case WM_QUERYNEWPALETTE:
      {               
         int i;
         HDC hDC;
                         
         if (hEdPal!=0)
         {
            hDC = GetDC(hWnd);
            i = RealizePalette(hDC); 
            if (i > 0) InvalidateRect(hWnd, NULL, TRUE);
            ReleaseDC(hWnd, hDC);
         }
         return(TRUE);
         break;
      }
      case WM_DESTROY:
      {        
         if (hOldBitmap2!=0) SelectObject(hDC2, hOldBitmap2);
         hOldBitmap2 = 0;
         DeleteObject(hBitmap2);
         SelectObject(hDC2, hOldEdPal);
         DeleteDC(hDC2); 
         DeleteObject(hEdPal);
         break;
      }
      case WM_PAINT:
      {                        
         RECT R;
         HDC hDC;
         PAINTSTRUCT ps;
         
         GetUpdateRect(hWnd, &R, FALSE);
         {       
             hDC = BeginPaint(hWnd, &ps);

             BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, 
                hDC2, R.left, R.top, SRCCOPY);
             EndPaint(hWnd, &ps);
             ReleaseDC(hWnd, hDC);
         }
         break;
      }    
      case (WM_USER+3):			 /* 2D and 3D Preview */
      {
         int x, y, form;               
         char s[250];
         double bailout, kvalue=0, lvalue, cx, cy, ck, cl;
         int maxiter;
         HCURSOR hcurSave; 
		 int useflag;

         hcurSave = SetCursor(LoadCursor((HANDLE)NULL, IDC_WAIT));
         SetCapture(hWnd);
         GetDlgItemText(hObjDlg, IDC_EDIT1, s, 40);
         bailout = atof(s);         
         bailout *= bailout;
         GetDlgItemText(hObjDlg, IDC_EDIT2, s, 40);
         maxiter = atoi(s);
         if (wParam==0) kvalue = (double)GetScrollPos(hBar, SB_CTL)/20;
         GetDlgItemText(hObjDlg, IDC_EDIT3, s, 40);
         lvalue = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT4, s, 40);
         cx = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT5, s, 40);
         cy = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT6, s, 40);
         ck = atof(s);
         GetDlgItemText(hObjDlg, IDC_EDIT7, s, 40);
         cl = atof(s);
         form = (int)SendDlgItemMessage(hObjDlg, IDC_LIST1, LB_GETCURSEL, 0, 0L);
         if (wParam==0)  /* 2D Preview */
		 {
            /* Calculate fractal (only for ObjectDialog) */
            for (y=0; y<60; y++)
            {
               for (x=0; x<80; x++)
               {
                  if (CalcJPixel(x,y,bailout,kvalue,lvalue,maxiter,form,cx,cy,ck,cl)==0) 
                     SetPixel(hDC2, x, y, RGB(0,0,0));
                  else SetPixel(hDC2, x, y, RGB(255,255,255));
               }
            }
         }
         else     /* 3D Preview */
         {
            struct frac_struct f;
            struct view_struct v;
            struct realpal_struct r;
            char cscheme[251];
            double cbuf[140];
            time_t old_time;
            int xs=0, ys=0;

            ReturnVideoInfo = MSWIN_ReturnVideoInfo;
            SetColors = MSWIN_SetColors2;
            Initialize = DUMMY_Initialize;
            Done = DUMMY_Done;
            update_bitmap = MSWIN_update_bitmapDLG;
            getline = DUMMY_getline;
            putline = DUMMY_putline;
            check_event = MSWIN_check_event;
            Change_Name = DUMMY_Change_Name;
            Debug = MSWIN_Debug;
            eol = MSWIN_eol2;
            old_time = calc_time;
            useflag = IsDlgButtonChecked(hObjDlg, IDC_CHECK1);
            SetDefaults(&f, &v, &r, cscheme, cbuf);
            v.LXR = 4;
            r.cols[0].col1[0] = 1;
            r.cols[0].col1[1] = 1;
            r.cols[0].col1[2] = 0;
            if (useflag) 
            {
               v = *view;
               strncpy(cscheme, colscheme, 250);
               for (x=0; x<140; x++) cbuf[x] = cut_buf[x];
               r = *realpal;
            }
            f.bailout = bailout;
            f.maxiter = maxiter;
            f.lvalue = lvalue;
            f.formula = form;
            v.interocular = 0;
            f.c[0] = cx; f.c[1] = cy; f.c[2] = ck; f.c[3] = cl;
            v.xres = 80; v.yres = 60; v.zres = 60;
            v.antialiasing = 1;
            hHereSetColors = hWnd;
            if (CreateImage(s, &xs, &ys, &f, &v, &r, cscheme, cbuf, 40, 0))
               MessageBeep(MB_ICONEXCLAMATION);
            calc_time = old_time;
         }
         InvalidateRect(hJulPrev, NULL, TRUE);          
         SetCursor(hcurSave);
         ReleaseCapture();
         break;
      }
   } 
   return(TRUE);
}

LRESULT CALLBACK DoObjectEditor(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{        
   switch (message)
   {
      case WM_INITDIALOG:
      {                       
         char s[40];
         
         hObjDlg = hDlg;
         sprintf(s, "%.15f", frac->bailout); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT1, s);
         sprintf(s, "%i",  frac->maxiter);
         SetDlgItemText(hDlg, IDC_EDIT2, s);
         sprintf(s, "%.15f", frac->lvalue); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT3, s);
         sprintf(s, "%.15f", frac->c[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);
         sprintf(s, "%.15f", frac->c[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         sprintf(s, "%.15f", frac->c[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT6, s);
         sprintf(s, "%.15f", frac->c[3]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT7, s);
         /* Initialize Julia 3rd-dim.Intersec.-ScrollBar */
         hBar = GetDlgItem(hDlg, IDC_SCROLLBAR1);
         SetScrollRange(hBar, SB_CTL, -40, 40, FALSE);
         SetScrollPos(hBar, SB_CTL, 0, TRUE);
         SetDlgItemText(hDlg, IDC_JULIA, "0");
         /* Initialize Formula List Box */
         SendDlgItemMessage(hDlg, IDC_LIST1, LB_ADDSTRING, 0, (LPARAM) ((LPSTR) "x[n+1] = x[n]^2 - c"));
         SendDlgItemMessage(hDlg, IDC_LIST1, LB_ADDSTRING, 0, (LPARAM) ((LPSTR) "x[n+1] = c*x[n]*(1-x[n])"));
         SendDlgItemMessage(hDlg, IDC_LIST1, LB_SETCURSEL, frac->formula, 0L);

         SendDlgItemMessage(hDlg, IDC_EDIT2, WM_ACTIVATE, WA_ACTIVE, 0); 
         SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
         if (imginmem || zbufinmem)
         {                       
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT7), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_LIST1), FALSE);
         }
/*         SendDlgItemMessage(hDlg, IDC_EDIT1, EM_SETSEL, 0, -1);    */
         return(TRUE);
         break;                   
      }
      case WM_CTLCOLOR:
      {
         SendDlgItemMessage(hDlg, IDC_USER1, WM_USER+1, 0, 0);
         break;
      }       
      case WM_HSCROLL:
      {        
         int nPos, pos, min, max;
         char s[40];
         
         nPos = (short int)HIWORD(wParam);
         if ((HWND)lParam==hBar)
         {
            pos = GetScrollPos(hBar, SB_CTL);
            GetScrollRange(hBar, SB_CTL, &min, &max);
            switch((int)LOWORD(wParam))
            {                   
               case SB_LEFT: pos = min; break;
               case SB_RIGHT: pos = max; break;
               case SB_LINELEFT: pos--; break;
               case SB_LINERIGHT: pos++; break;
               case SB_PAGELEFT: pos -= 5; break;
               case SB_PAGERIGHT: pos += 5; break;
               case SB_THUMBPOSITION: pos = nPos; break;
               case SB_THUMBTRACK: pos = nPos; break;
             }                                 
             if (pos<min) pos = min;
             if (pos>max) pos = max;
             SetScrollPos(hBar, SB_CTL, pos, TRUE);
             sprintf(s, "%.15f", (double)pos/20); CleanNumberString(s);
             SetDlgItemText(hDlg, IDC_JULIA, s);
         }
         break;
      }
      case (WM_USER+2):
      {            
         char s[40];
                       
         sprintf(s, "%.15f", frac->c[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);
         sprintf(s, "%.15f", frac->c[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         break;
      }
      case WM_COMMAND:
         if (wParam == IDOK)
         {              
            char s[40];
            /* IN ZUKUNFT: BEREICHS-PRFUNG BER KEYWORD-STRUKTUR */
            
            EndDialog(hDlg, TRUE);
            GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
            if (IsNumber(s)) frac->bailout = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
            if (IsNumber(s)) frac->maxiter = atoi(s);
            GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
            if (IsNumber(s)) frac->lvalue = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
            if (IsNumber(s)) frac->c[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
            if (IsNumber(s)) frac->c[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
            if (IsNumber(s)) frac->c[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT7, s, 40);
            if (IsNumber(s)) frac->c[3] = atof(s);
            frac->formula = (int)SendDlgItemMessage(hDlg, IDC_LIST1, LB_GETCURSEL, 0, 0L);

            return(TRUE);
         }       
         if (wParam == IDCANCEL)
         {
            EndDialog(hDlg, FALSE);
            return(TRUE);
         }
         if (wParam == IDC_BUTTON2)
         {
            mandel_x += mandel_lx/4;
            mandel_y += mandel_ly/4;
            mandel_lx /= 2;
            mandel_ly /= 2;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }        
         if (wParam == IDC_BUTTON3)
         {        
            mandel_x -= mandel_lx/2;
            mandel_y -= mandel_ly/2;
            mandel_lx *= 2;
            mandel_ly *= 2;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }        
         if (wParam == IDC_BUTTON4)
         {
            mandel_y -= mandel_ly/3;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }
         if (wParam == IDC_BUTTON5)
         {
            mandel_y += mandel_ly/3;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }                                                     
         if (wParam == IDC_BUTTON6)
         {                         
            mandel_x -= mandel_lx/3;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }        
         if (wParam == IDC_BUTTON7)
         {
            mandel_x += mandel_lx/3;
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
            break;
         }       
         if (wParam == IDC_2DPREVIEW)
         {
            SendDlgItemMessage(hDlg, IDC_USER2, WM_USER+3, 0, 0);
            break;
         }
         if (wParam == IDC_3DPREVIEW)
         {
            SendDlgItemMessage(hDlg, IDC_USER2, WM_USER+3, 1, 0);
            break;
         }
         break;
   }           
   return(FALSE);
}

LONG APIENTRY ViewSelector(register HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{                
   switch(message)
   {
      case WM_NCCREATE:      
      {
         HDC hDC; 
         HBRUSH hBrush, hOldBrush;
         HPEN hPen, hOldPen;        
         int i;
               
         i = 0;
         while (VS[i].used != 0 && i<3) i++;
         if (i<3)
         {
            VS[i].hWnd = hWnd; VS[i].used = 1; VS[i].state = 0;
         }
         /* else error in code (>3 selectors) */
         VS[0].x = 0; VS[0].dx = 1; VS[0].y = 1; VS[0].dy = 1;   /* front */
         VS[1].x = 2; VS[1].dx = -1; VS[1].y = 1; VS[1].dy = 1;  /* above */
         VS[2].x = 0; VS[2].dx = 1; VS[2].y = 2; VS[2].dy = -1;   /* side */
         hDC = GetDC(hWnd);
         VS[i].hBitmap = CreateCompatibleBitmap(hDC, 80, 60);
         VS[i].hDC = CreateCompatibleDC(hDC);
         VS[i].hOldBitmap = SelectObject(VS[i].hDC, VS[i].hBitmap);
         hBrush = CreateSolidBrush(RGB(255, 255, 255));
         hOldBrush = SelectObject(VS[i].hDC, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(VS[i].hDC, hPen);
         Rectangle(VS[i].hDC, 0, 0, 80, 60);              
         SelectObject(VS[i].hDC, hOldBrush);
         SelectObject(VS[i].hDC, hOldPen);
         DeleteObject(hPen); DeleteObject(hBrush);
         ReleaseDC(hWnd, hDC);        
         break;
      }                      
      case WM_NCDESTROY:
      {       
         int i;
                
         i = 0;                
         while (VS[i].hWnd != hWnd && i<3) i++;
         if (VS[i].hBitmap!=0) SelectObject(VS[i].hDC, VS[i].hOldBitmap);
         VS[i].hOldBitmap = 0;
         DeleteObject(VS[i].hBitmap);
         DeleteDC(VS[i].hDC); 
         VS[i].hWnd = 0; VS[i].used = 0;
         break;
      }
      case WM_PAINT:
      {                        
         RECT R;
         HDC hDC;
         PAINTSTRUCT ps;
         int i;
         
         GetUpdateRect(hWnd, &R, FALSE);
         hDC = BeginPaint(hWnd, &ps);
             
         i = 0;
         while (VS[i].hWnd != hWnd && i<3) i++;
         BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, 
            VS[i].hDC, R.left, R.top, SRCCOPY);
         EndPaint(hWnd, &ps);
         ReleaseDC(hWnd, hDC);
         break;
      }
      case WM_LBUTTONDOWN:
      {               
         int i = 0, x, y;
         
         if (imginmem || zbufinmem) break;
         while (VS[i].hWnd != hWnd && i<3) i++;   
         x = LOWORD(lParam);
         y = HIWORD(lParam);
         if (x>VS[i].vx-2 && x<VS[i].vx+2 && y>VS[i].vy-2 && y<VS[i].vy+2)
         {
            VS[i].state = 1; SetCapture(hWnd);
            SendMessage(hViewDlg, WM_USER, 0, 0);
         }                                    
         break;
      }
      case WM_LBUTTONUP:
      {        
         int i=0, x, y;
         
         while (VS[i].hWnd != hWnd && i<3) i++;
         x = LOWORD(lParam);
         y = HIWORD(lParam);
         if (VS[i].state == 1)
         {
            VS[i].state = 0; ReleaseCapture();
            SendMessage(hViewDlg, WM_USER, 0, 0);
         }
         break;
      }
      case WM_MOUSEMOVE:                    
      {
         int i=0;
         short int x, y;
         char s[40]; 
         int idControl=0;
         
         while (VS[i].hWnd != hWnd && i<3) i++;
         x = LOWORD(lParam);
         y = HIWORD(lParam);
         if (x<0) x = 0; if (x>79) x = 79; if (y<0) y=0; if (y>59) y = 59;
         if (VS[i].state == 1)
         {                                                                
            flag = 1;
            x -= 40; y -= 30;
            if (VS[i].dx < 0) x *= -1;
            if (VS[i].dy < 0) y *= -1;
            sprintf(s, "%.15f", ((double)x)/10); CleanNumberString(s);
            if (VS[i].x == 0) idControl = IDC_EDIT1;
            if (VS[i].x == 1) idControl = IDC_EDIT2;
            if (VS[i].x == 2) idControl = IDC_EDIT3;
            SetDlgItemText(hViewDlg, idControl, s);
            sprintf(s, "%.15f", ((double)y)/10); CleanNumberString(s);
            if (VS[i].y == 0) idControl = IDC_EDIT1;
            if (VS[i].y == 1) idControl = IDC_EDIT2;
            if (VS[i].y == 2) idControl = IDC_EDIT3;
            SetDlgItemText(hViewDlg, idControl, s);
            flag = 0;
            SendMessage(hViewDlg, WM_USER, 0, 0);
         }
         break;
      }
      case WM_USER:   /* parameter has changed, total redraw */    
      {
         int i, e, j, k, x=0, y=0, n;
         int posx, posy, arrowadd;
         RECT R; 
         double er, jr, kr;
         HPEN hPen, hOldPen;
         HBRUSH hBrush, hOldBrush;  
         struct view_struct *v;
         struct basestruct base, sbase;     
         int planex[4], planey[4];  
         POINT p[5];
                   
         i = 0;                   
         while (VS[i].hWnd != hWnd && i<3) i++;   
         /* Clear area */
         hBrush = CreateSolidBrush(RGB(255, 255, 255));
         hOldBrush = SelectObject(VS[i].hDC, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(VS[i].hDC, hPen);
         Rectangle(VS[i].hDC, 0, 0, 80, 60);              
         SelectObject(VS[i].hDC, hOldBrush);
         SelectObject(VS[i].hDC, hOldPen);
         DeleteObject(hPen); DeleteObject(hBrush);
         /* Draw "x" axis */
         /* MoveToEx(VS[i].hDC, 40, 30, NULL);*/
         posy = 30;
         if (VS[i].dx>0) { posx = 60; arrowadd = -6; }
         else { posx = 20; arrowadd = 6; }
         p[0].x = 40; p[0].y = 30;      
         p[1].x = posx; p[1].y = posy;
         p[2].x = posx+arrowadd; p[2].y = posy-3;
         Polyline(VS[i].hDC, p, 3);
         /*LineTo(VS[i].hDC, posx, posy); 
         LineTo(VS[i].hDC, posx+arrowadd, posy-3);*/
         p[0].x = posx; p[0].y = posy;
         p[1].x = posx+arrowadd; p[1].y = posy+3;     
         Polyline(VS[i].hDC, p, 2);
         /*MoveToEx(VS[i].hDC, posx, posy,NULL);
         LineTo(VS[i].hDC, posx+arrowadd, posy+3);  */
         if (VS[i].x==0) TextOut(VS[i].hDC, posx-arrowadd, posy, "e", 1);
         if (VS[i].x==1) TextOut(VS[i].hDC, posx-arrowadd, posy, "j", 1);
         if (VS[i].x==2) TextOut(VS[i].hDC, posx-arrowadd, posy, "k", 1);
         /* Draw "y" axis */
         posx = 40;
         if (VS[i].dy>0) { posy = 50; arrowadd = -6; }
         else { posy = 10; arrowadd = 6; }
         p[0].x = 40; p[0].y = 30;
         p[1].x = posx; p[1].y = posy;
         p[2].x = posx-3; p[2].y = posy+arrowadd;
         Polyline(VS[i].hDC, p, 3);
         /*MoveToEx(VS[i].hDC, 40, 30, NULL);
         LineTo(VS[i].hDC, posx, posy);
         LineTo(VS[i].hDC, posx-3, posy+arrowadd);*/
         p[0].x = posx; p[0].y = posy;
         p[1].x = posx+3; p[1].y = posy+arrowadd;
         Polyline(VS[i].hDC, p, 2);
         /*MoveToEx(VS[i].hDC, posx, posy,NULL);
         LineTo(VS[i].hDC, posx+3, posy+arrowadd); */
         if (VS[i].dy>0) posy = 40;
         if (VS[i].y==0) TextOut(VS[i].hDC, posx+4, posy, "e", 1);
         if (VS[i].y==1) TextOut(VS[i].hDC, posx+4, posy, "j", 1);
         if (VS[i].y==2) TextOut(VS[i].hDC, posx+4, posy, "k", 1);
         /* Draw viewpoint and view plane */       
         v = (struct view_struct *)lParam;
         if (!calcbase(&base, &sbase, *v, 0))
         {
            er = v->s[0]; jr = v->s[1]; kr = v->s[2];
            e = (int)(er*10); j = (int)(jr*10); k = (int)(kr*10);
            if (VS[i].x==0) x = e;
            if (VS[i].x==1) x = j;
            if (VS[i].x==2) x = k;  
            planex[0] = (int)(base.O[VS[i].x]*10);
            planex[1] = (int)((base.O[VS[i].x]+v->LXR*base.x[VS[i].x])*10);
            planex[2] = (int)((base.O[VS[i].x]+v->LXR*base.x[VS[i].x]+v->LXR*3/4*base.y[VS[i].x])*10);
            planex[3] = (int)((base.O[VS[i].x]+v->LXR*3/4*base.y[VS[i].x])*10);
            if (VS[i].y==0) y = e;
            if (VS[i].y==1) y = j;
            if (VS[i].y==2) y = k;                                                
            planey[0] = (int)(base.O[VS[i].y]*10);
            planey[1] = (int)((base.O[VS[i].y]+v->LXR*base.x[VS[i].y])*10);
            planey[2] = (int)((base.O[VS[i].y]+v->LXR*base.x[VS[i].y]+v->LXR*3/4*base.y[VS[i].y])*10);
            planey[3] = (int)((base.O[VS[i].y]+v->LXR*3/4*base.y[VS[i].y])*10);
            if (VS[i].dx<0) { x *= -1; for (n=0; n<4; n++) planex[n] *= -1; }
            if (VS[i].dy<0) { y *= -1; for (n=0; n<4; n++) planey[n] *= -1; }
            x += 40; y += 30;
            for (n=0; n<4; n++) { planex[n] += 40; planey[n] += 30; }
            hPen = CreatePen(PS_SOLID, 1, RGB(0, 255, 0));
            hOldPen = SelectObject(VS[i].hDC, hPen);
            p[0].x = planex[0]; p[0].y = planey[0];
            p[1].x = planex[1]; p[1].y = planey[1];
            p[2].x = planex[2]; p[2].y = planey[2];
            p[3].x = planex[3]; p[3].y = planey[3];
            p[4].x = planex[0]; p[4].y = planey[0];
            /*MoveToEx(VS[i].hDC, planex[0], planey[0],NULL);
            for (n=1; n<4; n++) LineTo(VS[i].hDC, planex[n], planey[n]);
            LineTo(VS[i].hDC, planex[0], planey[0]);*/
            Polyline(VS[i].hDC, p, 5);
            SelectObject(VS[i].hDC, hOldPen);
            DeleteObject(hPen);
            hPen = CreatePen(PS_SOLID, 1, RGB(255, 128, 0));
            hOldPen = SelectObject(VS[i].hDC, hPen);
            Ellipse(VS[i].hDC, planex[0]-1, planey[0]-1, planex[0]+2, planey[0]+2);
            SelectObject(VS[i].hDC, hOldPen);
            DeleteObject(hPen);
            if (VS[i].state == 1)
               hPen = CreatePen(PS_SOLID, 1, RGB(255,0,0));
            else hPen = CreatePen(PS_SOLID, 1, RGB(0,0,255));
            hOldPen = SelectObject(VS[i].hDC, hPen);
            Rectangle(VS[i].hDC, x-2, y-2, x+3, y+3);
            SelectObject(VS[i].hDC, hOldPen);
            DeleteObject(hPen);
            VS[i].vx = x; VS[i].vy = y;
         }
         else TextOut(VS[i].hDC, 1, 1, "not def.", 8);
         
         R.left = 0; R.top = 0; R.right = 80; R.bottom = 60;
         InvalidateRect(hWnd, &R, TRUE);
         break;
      }
   }
   return(TRUE);
}

LRESULT CALLBACK DoViewEditor(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
   char s[40];
   
   switch (message)
   {
      case WM_INITDIALOG:
      {                       
         flag = 1;
         vtmp = malloc(sizeof(struct view_struct));
         if (vtmp == NULL)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON2), FALSE);
         }       
         else *vtmp = *view;
         v = malloc(sizeof(struct view_struct));
         if (v == NULL)
         {
            free(vtmp);
            EndDialog(hDlg, FALSE);
            return(FALSE);
         }
         hViewDlg = hDlg;
         sprintf(s, "%.15f", view->s[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT1, s);
         sprintf(s, "%.15f", view->s[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT2, s);
         sprintf(s, "%.15f", view->s[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT3, s);
         sprintf(s, "%.15f", view->up[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);
         sprintf(s, "%.15f", view->up[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         sprintf(s, "%.15f", view->up[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT6, s);
         sprintf(s, "%.15f", view->light[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT7, s);
         sprintf(s, "%.15f", view->light[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT8, s);
         sprintf(s, "%.15f", view->light[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT9, s);
         sprintf(s, "%.15f", view->Mov[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT10, s);
         sprintf(s, "%.15f", view->Mov[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT11, s);
         sprintf(s, "%.15f", view->LXR); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT12, s);    
         sprintf(s, "%.15f", view->interocular); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT13, s);
         SendMessage(hDlg, WM_USER, 0, 0);
         SendDlgItemMessage(hDlg, IDC_EDIT1, WM_ACTIVATE, WA_ACTIVE, 0); 
         if (imginmem || zbufinmem)
         {                       
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT10), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT11), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT12), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT13), FALSE);
         }
         if (imginmem)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT7), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT8), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT9), FALSE);
         }
         flag = 0;
         return(TRUE);
         break;                   
      }
      case WM_CTLCOLOR:
      {           
         if (flag==0)
         {   
            flag = 1;
            SendMessage(hDlg, WM_USER, 0, 0);
            flag = 0;
         }
         break;
      }                      
      case WM_USER:
      {        
         double angle; 
         
         *v = *view;
         GetDlgItemText(hViewDlg, IDC_EDIT1, s, 40);
         if (IsNumber(s)) v->s[0] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT2, s, 40);
         if (IsNumber(s)) v->s[1] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT3, s, 40);
         if (IsNumber(s)) v->s[2] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT4, s, 40);
         if (IsNumber(s)) v->up[0] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT5, s, 40);
         if (IsNumber(s)) v->up[1] = atof(s); 
         GetDlgItemText(hViewDlg, IDC_EDIT6, s, 40);
         if (IsNumber(s)) v->up[2] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT10, s, 40);
         if (IsNumber(s)) v->Mov[0] = atof(s);
         GetDlgItemText(hViewDlg, IDC_EDIT11, s, 40);
         if (IsNumber(s)) v->Mov[1] = atof(s); 
         GetDlgItemText(hViewDlg, IDC_EDIT12, s, 40);
         if (IsNumber(s)) v->LXR = atof(s);           
         GetDlgItemText(hViewDlg, IDC_EDIT13, s, 40);
         if (IsNumber(s)) v->interocular = atof(s);
         angle = v->s[0]*v->s[0];
         angle += v->s[1]*v->s[1];
         angle += v->s[2]*v->s[2];
         angle = sqrt(angle);
         angle = 2*atan(0.5*v->interocular/angle); 
         angle *= 360/6.28318;
         sprintf(s, "%.4f", angle); CleanNumberString(s);
         strcat(s, "");
         SetDlgItemText(hViewDlg, IDC_ANGLE, s);
         
         SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, (LPARAM)v);
         SendDlgItemMessage(hDlg, IDC_USER2, WM_USER, 0, (LPARAM)v);
         SendDlgItemMessage(hDlg, IDC_USER3, WM_USER, 0, (LPARAM)v);
         break;
      }                     
      case WM_COMMAND:
         if (wParam == IDC_BUTTON1)
         {
            GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
            if (IsNumber(s)) vtmp->s[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
            if (IsNumber(s)) vtmp->s[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
            if (IsNumber(s)) vtmp->s[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
            if (IsNumber(s)) vtmp->up[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
            if (IsNumber(s)) vtmp->up[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
            if (IsNumber(s)) vtmp->up[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT7, s, 40);
            if (IsNumber(s)) vtmp->light[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT8, s, 40);
            if (IsNumber(s)) vtmp->light[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT9, s, 40);
            if (IsNumber(s)) vtmp->light[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT10, s, 40);
            if (IsNumber(s)) vtmp->Mov[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT11, s, 40);
            if (IsNumber(s)) vtmp->Mov[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT12, s, 40);
            if (IsNumber(s)) vtmp->LXR = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT13, s, 40);
            if (IsNumber(s)) vtmp->interocular = atof(s);
         }
         if (wParam == IDC_BUTTON2)
         {
            sprintf(s, "%.15f", vtmp->s[0]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT1, s);
            sprintf(s, "%.15f", vtmp->s[1]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT2, s);
            sprintf(s, "%.15f", vtmp->s[2]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT3, s);
            sprintf(s, "%.15f", vtmp->up[0]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT4, s);
            sprintf(s, "%.15f", vtmp->up[1]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT5, s);
            sprintf(s, "%.15f", vtmp->up[2]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT6, s);
            sprintf(s, "%.15f", vtmp->light[0]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT7, s);
            sprintf(s, "%.15f", vtmp->light[1]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT8, s);
            sprintf(s, "%.15f", vtmp->light[2]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT9, s);
            sprintf(s, "%.15f", vtmp->Mov[0]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT10, s);
            sprintf(s, "%.15f", vtmp->Mov[1]); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT11, s);
            sprintf(s, "%.15f", vtmp->LXR); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT12, s);    
            sprintf(s, "%.15f", vtmp->interocular); CleanNumberString(s);
            SetDlgItemText(hDlg, IDC_EDIT13, s);
            SendMessage(hDlg, WM_USER, 0, 0);
         }
         if (wParam == IDOK)
         {              
            /* IN ZUKUNFT: BEREICHS-PRFUNG BER KEYWORD-STRUKTUR */
            
            free(v); free(vtmp);
            EndDialog(hDlg, TRUE);
            GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
            if (IsNumber(s)) view->s[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
            if (IsNumber(s)) view->s[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
            if (IsNumber(s)) view->s[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
            if (IsNumber(s)) view->up[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
            if (IsNumber(s)) view->up[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
            if (IsNumber(s)) view->up[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT7, s, 40);
            if (IsNumber(s)) view->light[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT8, s, 40);
            if (IsNumber(s)) view->light[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT9, s, 40);
            if (IsNumber(s)) view->light[2] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT10, s, 40);
            if (IsNumber(s)) view->Mov[0] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT11, s, 40);
            if (IsNumber(s)) view->Mov[1] = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT12, s, 40);
            if (IsNumber(s)) view->LXR = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT13, s, 40);
            if (IsNumber(s)) view->interocular = atof(s);

            return(TRUE);
         }       
         if (wParam == IDCANCEL)
         {                      
            free(v); free(vtmp);
            EndDialog(hDlg, FALSE);
            return(TRUE);
         }
   }
   return(FALSE);
}

LONG APIENTRY PalettePreview(register HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   static int sizex, sizey;
   switch (message)
   {
      case WM_CREATE:      
      {
         HDC hDC;    
         HBRUSH hBrush, hOldBrush;
         HPEN hPen, hOldPen;  
         /* DWORD lDlgBaseUnits; */
         RECT R;

         hDC = GetDC(hWnd);
         /*lDlgBaseUnits = GetDialogBaseUnits();*/
         R.top = 0; R.left = 0; R.right = 235; R.bottom = 12;
         MapDialogRect(hColDlg, &R);
         /*old_x = (172*LOWORD(lDlgBaseUnits)) / 4;*/
         /*old_y = (5*HIWORD(lDlgBaseUnits)) / 4;*/
         sizex = R.right - R.left;
         sizey = R.bottom - R.top;
         /* This _shouldnt_ be neccessary ... */
         if (GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE)
         {
            sizex *= 2; sizey *= 2;
         }
         else
         {
            sizex = (int)(floor((double)sizex*1.5)); 
            sizey = (int)(floor((double)sizey*1.65));
         }
         hBitmap2 = CreateCompatibleBitmap(hDC, sizex, sizey);
         hDC2 = CreateCompatibleDC(hDC);
         hOldBitmap2 = SelectObject(hDC2, hBitmap2);
         hBrush = CreateSolidBrush(RGB(255, 255, 255));
         hOldBrush = SelectObject(hDC2, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(hDC2, hPen);
         Rectangle(hDC2, 0, 0, sizex, sizey);
         SelectObject(hDC2, hOldBrush);
         SelectObject(hDC2, hOldPen);
         DeleteObject(hPen); DeleteObject(hBrush);
         ReleaseDC(hWnd, hDC);    
            
         break;
      }                      
      case WM_DESTROY:
      {        
         if (hOldBitmap2!=0) SelectObject(hDC2, hOldBitmap2);
         hOldBitmap2 = 0;
         DeleteObject(hBitmap2);
         SelectObject(hDC2, hOldEdPal);
         DeleteObject(hEdPal);
         ReleaseDC(hWnd, hDC2);
         DeleteDC(hDC2); 
         break;
      }
      case WM_QUERYNEWPALETTE:
      {               
         int i;
         HDC hDC;
         HPALETTE hOldPal;
                         
         if (hEdPal!=0)
         {
            hDC = GetDC(hWnd);
            hOldPal = SelectPalette(hDC, hEdPal, 0);
            i = RealizePalette(hDC); 
            SelectPalette(hDC, hOldPal, 0);
            ReleaseDC(hWnd, hDC);
            if (i > 0) 
               InvalidateRect(hWnd, (LPRECT) NULL, 1);
            return 1;
         }
         return 0;
         break;
      }
      case WM_PAINT:
      {                        
         RECT R;
         HDC hDC;
         PAINTSTRUCT ps;
         
         GetUpdateRect(hWnd, &R, FALSE);
         {       
            hDC = BeginPaint(hWnd, &ps);

            BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, 
               hDC2, R.left, R.top, SRCCOPY);
            EndPaint(hWnd, &ps);
            ReleaseDC(hWnd, hDC);
         }
         break;
      }                           
      case (WM_USER):
      {
         RECT R;
         int i, j, zero;  
         float CBuf, BBuf;
         double offs, weightsum;
         unsigned char col[3];
         int pos, idx;
         COLORREF idxc;
         int PalC;
         int depth;
                
         pos = GetScrollPos(hBar, SB_CTL); idx = pos-1;
         zero = 0;
         CalcWeightsum(tmppal);
         for (i=0; i<tmppal->colnum; i++) if (tmppal->cols[i].weight==0) zero = 1;
         if (zero==1) TextOut(hDC2, 1, 1, "Weight is zero!", 15);
         else 
         {
            PalC = GetDeviceCaps(hDC2, RASTERCAPS) & RC_PALETTE;
            
            if (PalC)  /* Palette Mode */
            {
               struct disppal_struct dp;

               depth = GetDeviceCaps(hDC2, COLORRES)/3;
               dp.maxcol = GetDeviceCaps(hDC2, SIZEPALETTE)-GetDeviceCaps(hDC2, NUMRESERVED);
               dp.btoc = 0.3;
               CreateDispPal(&dp, tmppal, dp.maxcol, 0.2, depth, depth, depth);   
               hHereSetColors = hWnd;
               if (MSWIN_SetColors2(&dp)) MessageBeep(MB_ICONASTERISK);
            }

            BBuf = (float)1;
            for (i=1; i<sizex-1; i++) 
            {
               CBuf = ((float)i-1)/((float)sizex-2);
               PixelvalueTrueMode(0, 0, 255, 255, 255, tmppal, &col[0], &CBuf, &BBuf);
               idxc = RGB(col[0],col[1],col[2]);
               if (PalC) 
                  idxc = PALETTEINDEX(GetNearestPaletteIndex(hEdPal,idxc));
               for (j=1; j<sizey-1; j++) 
                  SetPixel(hDC2, i, j, idxc);
            }
         }
         weightsum = 0;
         for (i=0; i<tmppal->colnum; i++) weightsum += tmppal->cols[i].weight;
         offs = 0;                                                             
         for (i=0; i<idx; i++) offs += tmppal->cols[i].weight;
         R.left = (int)(offs/weightsum*(float)(sizex-2)+0.5)+1; 
         R.right = (int)((offs+tmppal->cols[idx].weight)/weightsum*(float)(sizex-2)+0.5)+1;
         R.top = 1; R.bottom = sizey-1;
         FrameRect(hDC2, &R, GetStockObject(BLACK_BRUSH));
         InvalidateRect(hWnd, NULL, TRUE); 
         break;
      }
   } 
   return(TRUE);
}

LONG APIENTRY ColorClipboard(register HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   static int sizex, sizey;
   switch (message)
   {
      case WM_CREATE:      
      {
         HDC hDC;    
         HBRUSH hBrush, hOldBrush;
         HPEN hPen, hOldPen;  
         RECT R;

         hDC = GetDC(hWnd);
         R.top = 0; R.left = 0; R.right = 10; R.bottom = 10;
         MapDialogRect(hColDlg, &R);
         sizex = R.right - R.left; 
         sizey = R.bottom - R.top;
         /* This _shouldnt_ be neccessary ... */
         if (GetDeviceCaps(hDC, RASTERCAPS) & RC_PALETTE)
         {
            sizex *= 2; sizey *= 2;
         }
         else
         {
            sizex = (int)(floor((double)sizex*1.5)); 
            sizey = (int)(floor((double)sizey*1.65));
         }
         hBitmap1 = CreateCompatibleBitmap(hDC, sizex, sizey);
         hDC1 = CreateCompatibleDC(hDC);
         hOldBitmap1 = SelectObject(hDC1, hBitmap1);
         hBrush = CreateSolidBrush(RGB(255, 255, 255));
         hOldBrush = SelectObject(hDC1, hBrush);
         hPen = CreatePen(PS_SOLID, 1, RGB(0, 0, 0));
         hOldPen = SelectObject(hDC1, hPen);
         Rectangle(hDC1, 0, 0, sizex, sizey);
         SelectObject(hDC1, hOldBrush);
         SelectObject(hDC1, hOldPen);
         DeleteObject(hPen); DeleteObject(hBrush);
         ReleaseDC(hWnd, hDC);        
         break;
      }                      
      case WM_DESTROY:
      {        
         if (hOldBitmap1!=0) SelectObject(hDC1, hOldBitmap1);
         hOldBitmap1 = 0;
         DeleteObject(hBitmap1);
         DeleteDC(hDC1);
         break;
      }
      case WM_PAINT:
      {                        
         RECT R;
         HDC hDC;
         PAINTSTRUCT ps;
         
         GetUpdateRect(hWnd, &R, FALSE);
         {       
             hDC = BeginPaint(hWnd, &ps);

             BitBlt(hDC, R.left, R.top, R.right-R.left, R.bottom-R.top, 
                hDC1, R.left, R.top, SRCCOPY);
             EndPaint(hWnd, &ps);
             ReleaseDC(hWnd, hDC);
         }
         break;
      }    
      case (WM_USER+4):
      {
         RECT R;
         HBRUSH brush;                 

         brush = CreateSolidBrush(RGB(LOWORD(lParam),HIWORD(lParam),LOWORD(wParam)));
         R.left = 1; R.right = sizex-1; R.top = 1; R.bottom = sizey-1;
         FillRect(hDC1, &R, brush);
         R.left = 0; R.top = 0; R.right = sizex; R.bottom = sizey;
         InvalidateRect(hWnd, &R, TRUE);          
         break;
      }
   } 
   return(TRUE);
}

LRESULT CALLBACK DoColorEditor(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
   static double Rs, Gs, Bs;
   
   switch (message)
   {
      case WM_DESTROY:
      {
         free(tmppal);
         break;
      }
      case WM_INITDIALOG:
      {         
         int i;
         HWND Combo = GetDlgItem(hDlg, IDC_COMBO1);
         
         hColDlg = hDlg;
         tmppal = malloc(sizeof(struct realpal_struct)); 
         if (!tmppal) 
         {                      
            MessageBox(hDlg, "Memory allocation failed.", "Quat: Error", MB_OK);
            DestroyWindow(hDlg);
         }
         *tmppal = *realpal;
         for (i=tmppal->colnum; i<30; i++) tmppal->cols[i].weight = 1;
         /* Initialize ScrollBar */
         hBar = GetDlgItem(hDlg, IDC_SCROLLBAR1);
         SetScrollRange(hBar, SB_CTL, 1, realpal->colnum, FALSE);
         SetScrollPos(hBar, SB_CTL, 1, TRUE);
         SetDlgItemText(hDlg, IDC_COLNUM, "1");

         SetDlgItemText(hDlg, IDC_COMBO1, colscheme);

         /* Initialize Combo Box */
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)colscheme), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"0.36*x-0.5   # linear in x dir."), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"0.48*y-0.5   # linear in y dir."), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"0.25*z-0.5   # linear in z dir."), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"0.7*sqrt(x^2+y^2+z^2)   # spherical"), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"x+y   # diagonal striped"), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"(sin(7*x)+sin(7*y)+sin(7*z)+3)/6   # big speckles") , 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"y+0.2*sin(15*x)   # spikes"), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"# These are only examples."), 0);
         SendMessage(Combo, CB_SETITEMDATA, 
            (LPARAM) SendMessage(Combo, CB_ADDSTRING, 0, (LPARAM)(LPCSTR)"# You can also enter your own formula, of course."), 0);



         SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
         SendDlgItemMessage(hDlg, IDC_EDIT1, WM_ACTIVATE, WA_ACTIVE, 0);  /* select first edit c. */
         if (imginmem)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT7), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_COMBO1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON6), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_SEL1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_COPY1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_PASTE1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_SEL2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_COPY2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_PASTE2), FALSE);
         }
         flag = 0;
         SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);  /* Redraw color preview */
         return(TRUE);
         break;
      }                                   
      case WM_CTLCOLOR:
      {
         if (flag!=1) SendMessage(hDlg, WM_USER+3, 0, 0);    /* update tmppal from edit controls */
         break;
      }                      
      case WM_USER+2:   /* Update edit controls */
      {        
         char s[40];
         int pos, idx;
                 
         pos = GetScrollPos(hBar, SB_CTL); idx = pos-1;
         sprintf(s, "%i", pos); SetDlgItemText(hDlg, IDC_COLNUM, s);
         sprintf(s, "%.15f", tmppal->cols[idx].weight); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT1, s);
         sprintf(s, "%.15f", tmppal->cols[idx].col1[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT2, s);
         sprintf(s, "%.15f", tmppal->cols[idx].col1[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT3, s);
         sprintf(s, "%.15f", tmppal->cols[idx].col1[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);             
         sprintf(s, "%.15f", tmppal->cols[idx].col2[0]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         sprintf(s, "%.15f", tmppal->cols[idx].col2[1]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT6, s);
         sprintf(s, "%.15f", tmppal->cols[idx].col2[2]); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT7, s);             
         break;
      }
      case WM_USER+3:     /* write changed edit control to tmppal */
      {        
         int pos, idx;
         int changed = 0;
         char s[40];               
         double d = 0.0;
                 
         pos = GetScrollPos(hBar, SB_CTL); idx = pos-1;
         GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].weight) { changed = 1; tmppal->cols[idx].weight = d; }
         GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col1[0]) { changed = 1; tmppal->cols[idx].col1[0] = d; }
         GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col1[1]) { changed = 1; tmppal->cols[idx].col1[1] = d; }
         GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col1[2]) { changed = 1; tmppal->cols[idx].col1[2] = d; }
         GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col2[0]) { changed = 1; tmppal->cols[idx].col2[0] = d; }
         GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col2[1]) { changed = 1; tmppal->cols[idx].col2[1] = d; }
         GetDlgItemText(hDlg, IDC_EDIT7, s, 40);
         if (IsNumber(s)) d = atof(s);
         if (d != tmppal->cols[idx].col2[2]) { changed = 1; tmppal->cols[idx].col2[2] = d; }
         if (changed)
         {
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
         }         
         break;
      }
      case WM_VSCROLL:
      {        
         int nPos, pos, min, max;
         char s[40];
         
         nPos = (short int) HIWORD(wParam);
         if ((HWND)lParam==hBar)
         {
            pos = GetScrollPos(hBar, SB_CTL);
            GetScrollRange(hBar, SB_CTL, &min, &max);
            switch((int)LOWORD(wParam))
            {                   
               case SB_LEFT: pos = min; break;
               case SB_RIGHT: pos = max; break;
               case SB_LINELEFT: pos--; break;
               case SB_LINERIGHT: pos++; break;
               case SB_PAGELEFT: pos -= 5; break;
               case SB_PAGERIGHT: pos += 5; break;
               case SB_THUMBPOSITION: pos = nPos; break;
               case SB_THUMBTRACK: pos = nPos; break;
             }                                 
             if (pos<min) pos = min;
             if (pos>max) pos = max;
             flag = 1;  /* This is no change entered by user */
             /* if flag wouldnt be set, the WM_CTLCOLOR would result in */
             /* an update of act. color with values of previous color */
             SetScrollPos(hBar, SB_CTL, pos, TRUE);
             sprintf(s, "%i", pos); CleanNumberString(s);
             SetDlgItemText(hDlg, IDC_COLNUM, s);
             SendMessage(hDlg, WM_USER+2, 0, 0);               
             SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
             flag = 0;
             
         }
         break;
      }
      case WM_COMMAND:
      {
         if (wParam == IDC_BUTTON6)   /* Add */
         {
            if (tmppal->colnum<30)
            {
               int z, idx;
               
               tmppal->colnum++; 
               idx = GetScrollPos(hBar, SB_CTL)-1;
               for (z=tmppal->colnum-1; z>idx; z--) tmppal->cols[z] = tmppal->cols[z-1];
               if (idx>0) for (z=0; z<3; z++)
                  tmppal->cols[idx].col1[z] = tmppal->cols[idx-1].col2[z];
               else for (z=0; z<3; z++) tmppal->cols[idx].col1[z] = tmppal->cols[idx].col2[z];
               for (z=0; z<3; z++) tmppal->cols[idx].col2[z] = tmppal->cols[idx+1].col1[z];
               SetScrollRange(hBar, SB_CTL, 1, tmppal->colnum, TRUE);
               /* SetScrollPos(hBar, SB_CTL, tmppal->colnum, TRUE); */
               /* sprintf(s, "%i", tmppal->colnum);
               SetDlgItemText(hDlg, IDC_COLNUM, s); */
               SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
               SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);   /* Redraw color preview */
            }        
            break;
         }
         if (wParam == IDC_BUTTON2)   /* Del */
         {
            if (tmppal->colnum>1)
            {
               int i, idx;
               
               idx = GetScrollPos(hBar, SB_CTL)-1;
               for (i=idx; i<tmppal->colnum-1; i++) tmppal->cols[i] = tmppal->cols[i+1];
               tmppal->colnum--;                                        
               flag = 1;
               if (idx>=tmppal->colnum) SetScrollPos(hBar, SB_CTL, idx+1, FALSE);
               SetScrollRange(hBar, SB_CTL, 1, tmppal->colnum, TRUE);
               SendMessage(hDlg, WM_USER+2, 0, 0);
               SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
               flag = 0;
               break;
            }
         }
         if (wParam == IDC_SEL1)
         {
            CHOOSECOLOR cc;
            COLORREF clr;
            COLORREF aclrCust[16];
            int i, pos;

            pos = GetScrollPos(hBar, SB_CTL) - 1;
            for (i = 0; i < 16; i++) aclrCust[i] = RGB(255, 255, 255);
            clr = RGB((long)floor(tmppal->cols[pos].col1[0]*255+0.5), 
               (long)floor(tmppal->cols[pos].col1[1]*255+0.5), 
               (long)floor(tmppal->cols[pos].col1[2]*255+0.5));
            memset(&cc, 0, sizeof(CHOOSECOLOR));
            cc.lStructSize = sizeof(CHOOSECOLOR);
            cc.hwndOwner = hDlg;
            cc.rgbResult = clr;
            cc.lpCustColors = aclrCust;
            cc.Flags = CC_FULLOPEN + CC_RGBINIT;
            if (ChooseColor(&cc))
            {  
               tmppal->cols[pos].col1[0] = (double)GetRValue(cc.rgbResult)/255;
               tmppal->cols[pos].col1[1] = (double)GetGValue(cc.rgbResult)/255;
               tmppal->cols[pos].col1[2] = (double)GetBValue(cc.rgbResult)/255;   
               flag = 1;
               SendMessage(hDlg, WM_USER+2, 0, 0);
               SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);   /* Redraw color preview */
               flag = 0;
            }
         }
         if (wParam == IDC_SEL2)
         {
            CHOOSECOLOR cc;
            COLORREF clr;
            COLORREF aclrCust[16];
            int i, pos;

            pos = GetScrollPos(hBar, SB_CTL) - 1;
            for (i = 0; i < 16; i++) aclrCust[i] = RGB(255, 255, 255);
            clr = RGB((long)floor(tmppal->cols[pos].col2[0]*255+0.5), 
               (long)floor(tmppal->cols[pos].col2[1]*255+0.5),
               (long)floor(tmppal->cols[pos].col2[2]*255+0.5));
            memset(&cc, 0, sizeof(CHOOSECOLOR));
            cc.lStructSize = sizeof(CHOOSECOLOR);
            cc.hwndOwner = hDlg;
            cc.rgbResult = clr;
            cc.lpCustColors = aclrCust;
            cc.Flags = CC_FULLOPEN + CC_RGBINIT;
            if (ChooseColor(&cc))
            {  
               tmppal->cols[pos].col2[0] = (double)GetRValue(cc.rgbResult)/255;
               tmppal->cols[pos].col2[1] = (double)GetGValue(cc.rgbResult)/255;
               tmppal->cols[pos].col2[2] = (double)GetBValue(cc.rgbResult)/255;   
               flag = 1;
               SendMessage(hDlg, WM_USER+2, 0, 0);
               SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);   /* Redraw color preview */
               flag = 0;
            }
         }          
         if (wParam == IDC_BUTTON5)
         {     
            struct realpal_struct tmppal2;
            
            tmppal2 = *realpal;
            *realpal = *tmppal;
            DoParamSave(hDlg, "COL Files (*.col)|*.col", "col", PS_COL);
            *realpal = tmppal2;
         }
         if (wParam == IDC_COPY1)
         {
            int pos;

            pos = GetScrollPos(hBar, SB_CTL) - 1;
            Rs = tmppal->cols[pos].col1[0];
            Gs = tmppal->cols[pos].col1[1];
            Bs = tmppal->cols[pos].col1[2];
            SendDlgItemMessage(hDlg, IDC_USER2, WM_USER+4, MAKELONG((WORD)(Bs*255),0),
               MAKELONG((WORD)(Rs*255),(WORD)(Gs*255))); 
         }
         if (wParam == IDC_COPY2)
         {
            int pos;
    
            pos = GetScrollPos(hBar, SB_CTL) - 1;
            Rs = tmppal->cols[pos].col2[0];
            Gs = tmppal->cols[pos].col2[1];
            Bs = tmppal->cols[pos].col2[2];
            SendDlgItemMessage(hDlg, IDC_USER2, WM_USER+4, MAKELONG((WORD)(Bs*255),0),
               MAKELONG((WORD)(Rs*255),(WORD)(Gs*255))); 
         }
         if (wParam == IDC_PASTE1)
         {
            int pos;

            pos = GetScrollPos(hBar, SB_CTL) - 1;
            tmppal->cols[pos].col1[0] = Rs;
            tmppal->cols[pos].col1[1] = Gs;
            tmppal->cols[pos].col1[2] = Bs;
            SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
/*            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER+2, 0, 0);  */
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
         }
         if (wParam == IDC_PASTE2)
         {
            int pos;

            pos = GetScrollPos(hBar, SB_CTL) - 1;
            tmppal->cols[pos].col2[0] = Rs;
            tmppal->cols[pos].col2[1] = Gs;
            tmppal->cols[pos].col2[2] = Bs;
            SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
/*            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER+2, 0, 0);*/
            SendDlgItemMessage(hDlg, IDC_USER1, WM_USER, 0, 0);
         }
         if (wParam == IDOK)
         {              
            EndDialog(hDlg, TRUE);
            *realpal = *tmppal;
            GetDlgItemText(hDlg, IDC_COMBO1, colscheme, 251);
/*            comment = strchr(colscheme, '#');
            if (comment!=NULL) *comment = 0;  */
            return(TRUE);
         }       
         if (wParam == IDCANCEL)
         {
            EndDialog(hDlg, FALSE);
            return(TRUE);
         }
      }
   }
   return(FALSE);
}

LRESULT CALLBACK DoIntersecEditor(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
   switch (message)
   {
      case WM_DESTROY:
      {
         free(tmppal);
         break;
      }
      case WM_INITDIALOG:
      {         
         int i, j;
         
         tmpcut = malloc(sizeof(double)*140); 
         if (!tmpcut)
         {                      
            MessageBox(hDlg, "Memory allocation failed.", "Quat: Error", MB_OK);
            DestroyWindow(hDlg);
         }       
         /* count objects */
         i = 0; numcut = 0;
         while (*(char *)&cut_buf[i]!=CUT_TERMINATOR && numcut<20) { i += 7; numcut++; }
         /* Initialize ScrollBar */
         hBar = GetDlgItem(hDlg, IDC_SCROLLBAR1);
         j = numcut; if (j==0) j = 1;
         SetScrollRange(hBar, SB_CTL, 1, j, FALSE);
         SetScrollPos(hBar, SB_CTL, 1, TRUE);
         if (numcut==0 || imginmem || zbufinmem)
         {
            if (numcut==0)
            {
               SetDlgItemText(hDlg, IDC_NUMCUT, "0");
               EnableWindow(GetDlgItem(hDlg, IDC_SCROLLBAR1), FALSE);
            }
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
         }
         /* for (i=0; i<140; i++) tmpcut[i] = 0; */
         memcpy(tmpcut, cut_buf, sizeof(double)*140);
         /* for (i=0; i<140; i++) *(tmpcut+i) = *(cut_buf+i); */
         SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
         if (numcut>0)
            SendDlgItemMessage(hDlg, IDC_EDIT1, WM_ACTIVATE, WA_ACTIVE, 0);  /* select first edit c. */
         else SendDlgItemMessage(hDlg, IDC_BUTTON1, WM_ACTIVATE, WA_ACTIVE, 0);
         if (imginmem || zbufinmem)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON2), FALSE);
         }
         flag = 0;
         return(TRUE);
         break;
      }                                   
      case WM_CTLCOLOR:
      {
         if (flag!=1) SendMessage(hDlg, WM_USER+3, 0, 0);    /* update tmpcut from edit controls */
         break;
      }                      
      case WM_USER+2:   /* Update edit controls */
      {        
         char s[40];
         int idx;
                 
         if (numcut==0) 
         {
            SetDlgItemText(hDlg, IDC_NUMCUT, "0");
            idx = 0;
         }
         else
         {
            idx = GetScrollPos(hBar, SB_CTL) - 1;
            sprintf(s, "%i", idx+1); SetDlgItemText(hDlg, IDC_NUMCUT, s);
            *(char *)&tmpcut[idx*7] = CUT_PLANE;
         }
         sprintf(s, "%.15f", *(tmpcut+idx*7+1)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT1, s);
         sprintf(s, "%.15f", *(tmpcut+idx*7+2)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT2, s);
         sprintf(s, "%.15f", *(tmpcut+idx*7+3)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT3, s);
         sprintf(s, "%.15f", *(tmpcut+idx*7+4)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);             
         sprintf(s, "%.15f", *(tmpcut+idx*7+5)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         sprintf(s, "%.15f", *(tmpcut+idx*7+6)); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT6, s);
         break;
      }
      case WM_USER+3:     /* write changed edit control to tmpcut */
      {        
         int idx;
         char s[40];               
                 
         idx = GetScrollPos(hBar, SB_CTL) - 1;
         GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+1) = atof(s);
         GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+2) = atof(s);
         GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+3) = atof(s);
         GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+4) = atof(s);
         GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+5) = atof(s);
         GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
         if (IsNumber(s)) *(tmpcut+idx*7+6) = atof(s);
         break;
      }
      case WM_VSCROLL:
      {        
         int nPos, pos, min, max;
         char s[40];
         
         nPos = (short int)HIWORD(wParam);
         if ((HWND)lParam==hBar)
         {
            pos = GetScrollPos(hBar, SB_CTL);
            GetScrollRange(hBar, SB_CTL, &min, &max);
            switch((int)LOWORD(wParam))
            {                   
               case SB_LEFT: pos = min; break;
               case SB_RIGHT: pos = max; break;
               case SB_LINELEFT: pos--; break;
               case SB_LINERIGHT: pos++; break;
               case SB_PAGELEFT: pos -= 5; break;
               case SB_PAGERIGHT: pos += 5; break;
               case SB_THUMBPOSITION: pos = nPos; break;
               case SB_THUMBTRACK: pos = nPos; break;
             }                                 
             if (pos<min) pos = min;
             if (pos>max) pos = max;
             flag = 1;  /* This is no change entered by user */
             /* if flag wouldnt be set, the WM_CTLCOLOR would result in */
             /* an update of act. object with values of previous object */
             SetScrollPos(hBar, SB_CTL, pos, TRUE);
             sprintf(s, "%i", pos); CleanNumberString(s);
             SetDlgItemText(hDlg, IDC_NUMCUT, s);
             SendMessage(hDlg, WM_USER+2, 0, 0);               
             flag = 0;
         }
         break;
      }
      case WM_COMMAND:
      {
         if (wParam == IDC_BUTTON1)   /* Add an object */
         {
            char s[40];
            
            if (numcut<20)
            {
               numcut++; 
               SetScrollRange(hBar, SB_CTL, 1, numcut, FALSE);
               SetScrollPos(hBar, SB_CTL, numcut, TRUE);
               sprintf(s, "%i", numcut);         
               EnableWindow(GetDlgItem(hDlg, IDC_SCROLLBAR1), TRUE);                                
               if (!imginmem && !zbufinmem)
               {
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), TRUE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), TRUE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), TRUE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), TRUE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), TRUE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), TRUE);
               }
               SetDlgItemText(hDlg, IDC_NUMCUT, s);
               SendMessage(hDlg, WM_USER+2, 0, 0);    /* update edit controls */ 
            }        
            break;
         }
         if (wParam == IDC_BUTTON2)   /* Del */
         {
            if (numcut>0)
            {
               int i, idx;
               
               idx = GetScrollPos(hBar, SB_CTL) - 1;
               for (i=idx; i<numcut-1; i++)       
               {                                                 
                  *(tmpcut+i*7) = *(tmpcut+(i+1)*7);
                  *(tmpcut+i*7+1) = *(tmpcut+(i+1)*7+1);
                  *(tmpcut+i*7+2) = *(tmpcut+(i+1)*7+2);
                  *(tmpcut+i*7+3) = *(tmpcut+(i+1)*7+3);
                  *(tmpcut+i*7+4) = *(tmpcut+(i+1)*7+4);
                  *(tmpcut+i*7+5) = *(tmpcut+(i+1)*7+5);
                  *(tmpcut+i*7+6) = *(tmpcut+(i+1)*7+6);
               }
               numcut--;                                        
               flag = 1;
               if (numcut!=0)
               {
                  if (idx>=numcut) SetScrollPos(hBar, SB_CTL, numcut, FALSE);
                  SetScrollRange(hBar, SB_CTL, 1, numcut, TRUE);
               }
               else
               {
                  EnableWindow(GetDlgItem(hDlg, IDC_SCROLLBAR1), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
                  EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
               }
               SendMessage(hDlg, WM_USER+2, 0, 0);
               flag = 0;
               break;
            }
         }
         if (wParam == IDOK)
         {              
            EndDialog(hDlg, TRUE);
            /* for (i=0; i<140; i++) *(cut_buf+i) = *(tmpcut+i); */
            *(char *)&tmpcut[numcut*7] = CUT_TERMINATOR;
            memcpy(cut_buf, tmpcut, sizeof(double)*140);
            return(TRUE);
         }       
         if (wParam == IDCANCEL)
         {
            EndDialog(hDlg, FALSE);
            return(TRUE);
         }
      }
   }
   return(FALSE);
}

LRESULT CALLBACK DoOtherEditor(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
   char s[40];
   static int tmpev;
   
   switch (message)
   {
      case WM_INITDIALOG:
      {
         sprintf(s, "%i", view->xres); SetDlgItemText(hDlg, IDC_EDIT1, s);
         sprintf(s, "%i", view->yres); SetDlgItemText(hDlg, IDC_EDIT2, s);
         sprintf(s, "%i", view->zres); SetDlgItemText(hDlg, IDC_EDIT3, s);
         sprintf(s, "%.15f", view->phongmax); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT4, s);
         sprintf(s, "%.15f", view->phongsharp); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT5, s);
         sprintf(s, "%.15f", view->ambient); CleanNumberString(s);
         SetDlgItemText(hDlg, IDC_EDIT6, s);
         sprintf(s, "%i", view->antialiasing); SetDlgItemText(hDlg, IDC_EDIT7, s); 
         sprintf(s, "%li", calc_time); strcat(s, " sec.");
         SetDlgItemText(hDlg, IDC_CALCTIME, s);
         tmpev = pixperevent;
         sprintf(s, "Check events every %i pixels", pixperevent); 
         SetDlgItemText(hDlg, IDC_EVENTS, s);
         SendDlgItemMessage(hDlg, IDC_EDIT1, WM_ACTIVATE, WA_ACTIVE, 0); 
         if (imginmem || zbufinmem)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON1), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON2), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON3), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_BUTTON5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT7), FALSE);
         }
         if (imginmem)
         {
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT4), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT5), FALSE);
            EnableWindow(GetDlgItem(hDlg, IDC_EDIT6), FALSE);
         }
         return(TRUE);
         break;
      }
      case WM_COMMAND:
      {
         if (wParam == IDC_BUTTON1)
         {
            SetDlgItemText(hDlg, IDC_EDIT1, "80");
            SetDlgItemText(hDlg, IDC_EDIT2, "60");
            SetDlgItemText(hDlg, IDC_EDIT3, "60");
         }
         if (wParam == IDC_BUTTON2)
         {
            SetDlgItemText(hDlg, IDC_EDIT1, "80");
            SetDlgItemText(hDlg, IDC_EDIT2, "60");
            SetDlgItemText(hDlg, IDC_EDIT3, "120");
         }
         if (wParam == IDC_BUTTON3)
         {
            SetDlgItemText(hDlg, IDC_EDIT1, "160");
            SetDlgItemText(hDlg, IDC_EDIT2, "120");
            SetDlgItemText(hDlg, IDC_EDIT3, "120");
         }
         if (wParam == IDC_BUTTON4)
         {
            SetDlgItemText(hDlg, IDC_EDIT1, "320");
            SetDlgItemText(hDlg, IDC_EDIT2, "240");
            SetDlgItemText(hDlg, IDC_EDIT3, "240");
         }       
         if (wParam == IDC_BUTTON5)
         {
            SetDlgItemText(hDlg, IDC_EDIT1, "640");
            SetDlgItemText(hDlg, IDC_EDIT2, "480");
            SetDlgItemText(hDlg, IDC_EDIT3, "480");
         }       
         if (wParam == IDC_PLUSBUTTON)
         {
            if (tmpev<50)
            {
               tmpev++;
               sprintf(s, "Check events every %i pixels", tmpev); 
               SetDlgItemText(hDlg, IDC_EVENTS, s);
            }
         }
         if (wParam == IDC_MINUSBUTTON)
         {
            if (tmpev>1)
            {
               tmpev--;
               sprintf(s, "Check events every %i pixels", tmpev); 
               SetDlgItemText(hDlg, IDC_EVENTS, s);
            }
         }
         if (wParam == IDOK)
         {              
            char s[40];
            /* IN ZUKUNFT: BEREICHS-PRFUNG BER KEYWORD-STRUKTUR */
            
            EndDialog(hDlg, TRUE);
            GetDlgItemText(hDlg, IDC_EDIT1, s, 40);
            if (IsNumber(s)) view->xres = atoi(s);
            GetDlgItemText(hDlg, IDC_EDIT2, s, 40);
            if (IsNumber(s)) view->yres = atoi(s);
            GetDlgItemText(hDlg, IDC_EDIT3, s, 40);
            if (IsNumber(s)) view->zres = atoi(s);
            GetDlgItemText(hDlg, IDC_EDIT4, s, 40);
            if (IsNumber(s)) view->phongmax = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT5, s, 40);
            if (IsNumber(s)) view->phongsharp = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT6, s, 40);
            if (IsNumber(s)) view->ambient = atof(s);
            GetDlgItemText(hDlg, IDC_EDIT7, s, 40);
            if (IsNumber(s)) view->antialiasing = atoi(s);
            pixperevent = tmpev;
            return(TRUE);
         }       
         if (wParam == IDCANCEL)
         {
            EndDialog(hDlg, FALSE);
            return(TRUE);
         }
      }
   }
   return(FALSE);
}
