/* PhotoOrganizer 
 * Copyright (C) 1999 Dmitry Rogatkin.  All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 *  ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *  ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package photoorganizer.formats;
import java.io.File;
import java.util.Date;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.awt.Image;
import java.awt.Dimension;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.text.DecimalFormat;
import javax.swing.Icon;

import photoorganizer.Resources;

public abstract class AbstractImageInfo extends BasicIo {
    final static DateFormat dateformat = new SimpleDateFormat("yyyy:MM:dd HH:mm:ss");
    final static DecimalFormat fnumberformat = new DecimalFormat("F1:#0.0#");
    static final String NA = "n/a";
    final static byte [] BMP_SIG = { 0x42, 0x4D };
    final static int BMP24_HDR_SIZE = 54;
    // conversions
    public final static double[] AV_TO_FSTOP = 
	{1, 1.4, 2, 2.8, 4, 5.6, 8, 11, 16, 22, 32 };
    public final static Rational[] TV_TO_SEC =
	{new Rational(1,1), new Rational(1,2), new Rational(1,4), new Rational(1,8),
	 new Rational(1,15), new Rational(1,30), new Rational(1,60), new Rational(1,125),
	 new Rational(1,250), new Rational(1,500), new Rational(1,1000), new Rational(1,2000),
	 new Rational(1,4000), new Rational(1,8000), new Rational(1,16000) };
    
    public AbstractImageInfo() {
    }

    public AbstractImageInfo(InputStream is, byte[] data, int offset, String name) throws FileFormatException {
	this.is = is;
	this.data = data;
	this.offset = offset;
	this.name = name;
	readInfo();
    }

    public String getName() {
	return name;
    }

    public void setName(String name) {
	this.name = name;
    }

    public abstract void readInfo() throws FileFormatException;

    public abstract String getFormat();
    
    public abstract int getResolutionX();

    public abstract int getResolutionY();
    
    public abstract String getMake();

    public abstract String getModel();
    
    public abstract String getDataTimeOriginalString();

    public abstract float getFNumber();

    public abstract Rational getShutter();

    public abstract boolean isFlash();

    public abstract String getQuality();

    public abstract float getFocalLength();

    public abstract int getMetering(); // matrix, dot, CenterWeightedAverage..
    
    public abstract int getExpoProgram(); // full automatic, ...

    public abstract String getReport();

    public abstract Icon getThumbnailIcon(BasicJpeg im, Dimension size);

    public Icon getThumbnailIcon(BasicJpeg im) { 
		return getThumbnailIcon(im, null);
	}
	
    public String getThumbnailExtension() {
	return Resources.EXT_JPEG;
    }


    public void saveThumbnailImage(BasicJpeg im, OutputStream os) throws IOException {
        if (os == null || im == null)
            return;
	if (photoorganizer.Controller.getAdvancedImage() != null) {
	    try {
		// try advanced image API
		photoorganizer.Controller.getAdvancedImage().saveThumbnailImage(im.getLocationName(), os, null);
                return;
	    } catch(Throwable e) {
	    }
        }
        os.close();
    }

    public Date getDateTimeOriginal() {
	try {
	    return dateformat.parse(getDataTimeOriginalString());
	} catch (NullPointerException e) {
	} catch (ParseException e) {
	    System.err.println(""+e);
	}
	return new Date();
    }

    // conversions
    public float apertureToFnumber(float aperture) {
	try {
    	    int si = (int)aperture;
	    float result = (float)AV_TO_FSTOP[si];
	    aperture -= si;
	    if (aperture != 0)
		result += (AV_TO_FSTOP[si+1]-AV_TO_FSTOP[si])*aperture;
            return result;
        } catch(ArrayIndexOutOfBoundsException e) {
        }
        return -1;
    }

    InputStream is;
    int offset;
    String name;
}