
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;

import java.util.*;
import org.webmacro.util.*;
import java.io.*;


/**
  *  Conditions evaluate() to Boolean.TRUE or Boolean.FALSE
  *  depending on its contents and the supplied context.
  *  You can build expressions from a combination of the
  *  operators ! && || == and Terms; you can also group 
  *  sub-expressions  with parentheses. An expression is true if 
  *  it is non-null; and false if it is Boolean.FALSE or null. 
  *  Beware that null values incur a slight performance penalty; 
  *  therefore we recommend you use boolean values in expressions 
  *  wherever possible.
  */
abstract public class Condition implements Macro
{

   /** 
     * Parse the expression and return a macro that will evaluate
     * to Boolean.TRUE or Boolean.FALSE depending on the condition.
     * <p>
     * @return boolean
     * @exception ParseException if the sytax was invalid and we could not recover
     * @exception IOException if we could not successfullly read the parseTool
     */
   public final static Object parse(ParseTool in) 
      throws ParseException, IOException
   {
      return parseCond(in);
   }

   /**
     * Exactly the same as parse(), but return type is Condition
     */
   public static Condition parseCond(ParseTool in) 
      throws ParseException, IOException
   {

      Condition value;
      boolean parens;

      // get parens
      parens = in.parseChar('(');

      // get lhs and possibly only sub-expression
      in.parseWhitespace();
      switch(in.ttype)
      {
         case '!': value = NotCondition.parseNot(in); break;
         case '(': value = Condition.parseCond(in); break;
         default : value = TermCondition.parseTerm(in); break;
      }
      if (null == value) {
         throw new ParseException(in,"Expected term/expression, got: " + (char) in.ttype);
      }

      // read left-associative operators
      boolean moreTerms = true;
      while(moreTerms) {
         Condition oper;
         in.parseWhitespace();
         switch(in.ttype) {
            case '!': oper = NotEqualCondition.parseNotEqual(value,in); break;
            case '=': oper = EqualCondition.parseEqual(value,in); break;
            case '&': oper = AndCondition.parseAnd(value,in); break;
            case '|': oper = OrCondition.parseOr(value,in); break;
            default : oper = null;
         }
         if (oper == null) {
            moreTerms = false;
         } else {
            value = oper;
         }
      }
     
      // check for close paren
      in.parseWhitespace();
      if (parens && !in.parseChar(')')) {
         throw new ParseException(in,"Mismatched braces around expression.");
      }
      
      return value;
   }

   /**
     * Utility function to parse the operator and right term of a binary cond
     */
   final static Condition parseBinOp(char[] opChars, ParseTool in) 
      throws IOException, ParseException
   {
      for (int i = 0; i < opChars.length; i++){
         if (! in.parseChar(opChars[i])) {
            if (i == 0) {
               return null;
            } else {
               throw new ParseException(in, "Expected character " + opChars[i] 
                     + " after " + opChars[i - 1] + " but got " 
                     + (char) in.ttype);
            } 
         }
      }
      in.parseWhitespace();

      Condition right = Condition.parseCond(in); 
      if (null == right) {
         throw new ParseException(in,"Expected term/expression after operator "
                +" but got: " + (char) in.ttype);
      }
      return right;
   }

   /**
     * Evaluate the condition and write it out
     */
   final public void write(Writer out, Object context) 
      throws InvalidContextException, IOException
   {
      out.write(evaluate(context).toString());
   }

   /**
     * Return an object representing this condition
     */
   public Object evaluate(Object context) 
      throws InvalidContextException
   {
      if (test(context)) {
         return Boolean.TRUE;
      } else {
         return Boolean.FALSE;
      }
   }

   /**
     * Returns true if the condition is logically true; returns false if the
     * condition is either logically false or undefined.
     */
   public abstract boolean test(Object context);

}


/**
  * Utility class
  */
final class NotCondition extends Condition {

   private final Condition _cond;

   final static Condition parseNot( ParseTool in ) 
      throws ParseException, IOException
   {
      if (! in.parseChar('!')) {
         return null;
      }
      in.parseWhitespace();
      Condition cond = Condition.parseCond(in);
      return new NotCondition(cond);
   }

   NotCondition(Condition c) {
      _cond = c;
   }

   final public boolean test(Object context) {
      return (! _cond.test(context) );
   }
}

/**
  * Utility class
  */
final class TermCondition extends Condition {

   final private Object _term;
   final private boolean _macro;

   final static Condition parseTerm(ParseTool in) 
      throws ParseException, IOException
   {
      Object term = Term.parse(in);
      return new TermCondition(term);
   }

   TermCondition(Object term) {
      _term = term;
      _macro = (term instanceof Macro);
   }

   final public Object evaluate(Object context) 
      throws InvalidContextException
   {
      Object ret = _term;
      if (_macro) {
         ret = ((Macro) ret).evaluate(context);
      }
      return ret;
   }

   final public boolean test(Object context) {
      try {
         Object ret = evaluate(context);
         return ((ret != null) && !ret.equals(Boolean.FALSE));
      } catch (Exception e) {
         return false;
      }
   }
}



/**
  * Utility class
  */
final class EqualCondition extends Condition {

   static final char[] _opChars = { '=','=' };
   private final Condition _l,_r;
   EqualCondition(Condition l, Condition r) { _l = l; _r = r; }

   final static Condition parseEqual(Condition left, ParseTool in)
      throws ParseException, IOException
   {
      Condition right = parseBinOp(_opChars, in);
      if (null == right) {
         return null;
      }
      return new EqualCondition(left,right);
   }

   public final boolean test(Object context) {
      Object left, right;
      try { left = _l.evaluate(context); }
      catch (Exception e) { left = null; }

      try { right = _r.evaluate(context); }
      catch (Exception e) { right = null; }

      if ((left == null) || (right == null)) {
         return (left == right);
      }
      return left.equals(right);
   }
}

/**
  * Utility class
  */
final class NotEqualCondition extends Condition {

   static final char[] _opChars = { '!','=' };
   private final Condition _l,_r;
   NotEqualCondition(Condition l, Condition r) { _l = l; _r = r; }

   final static Condition parseNotEqual(Condition left, ParseTool in)
      throws ParseException, IOException
   {
      Condition right = parseBinOp(_opChars, in);
      if (null == right) {
         return null;
      }
      return new NotEqualCondition(left,right);
   }

   public final boolean test(Object context) {
      Object left, right;
      try { left = _l.evaluate(context); }
      catch (Exception e) { left = null; }

      try { right = _r.evaluate(context); }
      catch (Exception e) { right = null; }

      if ((left == null) || (right == null)) {
         return (left != right);
      }
      return (!left.equals(right));
   }
}
/**
  * Utility class
  */
final class AndCondition extends Condition {
  
   static final char[] _opChars = { '&','&' };
   private final Condition _l,_r;
   AndCondition(Condition l, Condition r) { _l = l; _r = r; }

   final static Condition parseAnd(Condition left, ParseTool in)
      throws ParseException, IOException
   {
      Condition right = parseBinOp(_opChars, in);
      if (null == right) {
         return null;
      }
      return new AndCondition(left,right);
   }
   
   public final boolean test(Object context) {
      return (_l.test(context) && _r.test(context));
   }
}


/**
  * Utility class
  */
final class OrCondition extends Condition {
   
   private final Condition _l,_r;
   static final char[] _opChars = { '|','|' };
   OrCondition(Condition l, Condition r) { _l = l; _r = r; }

   final static Condition parseOr(Condition left, ParseTool in)
      throws ParseException, IOException
   {
      Condition right = parseBinOp(_opChars, in);
      if (null == right) {
         return null;
      }
      return new OrCondition(left,right);
   }

   public final boolean test(Object context) {
      return (_l.test(context) || _r.test(context));
   }
}



