
/*
 * Copyright (C) 1994,95 David Stes.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "fab.h"
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h> /* stat() in makepath */

id fab; /* unique instance of fabricator */

/* if given a directory, make name ourselves
 * otherwise, put output in indicated file
 */

static void makepath(char *buffer,char *path,char *rootname,char *suffix)
{
    int makefile;
    struct stat statbuf;

    if (stat(path,&statbuf)) {
	makefile = 0; /* case when user indicates a file to be created */
    } else {
	makefile = (statbuf.st_mode & S_IFDIR); /* case user indicates dir */
    }

    if (makefile) {
	sprintf(buffer,"%s/%s.%s",path,rootname,suffix);
    } else {
	sprintf(buffer,"%s",path);
    }
}

FILE *fabfopen(const char *name,const char *type)
{
    FILE *f = fopen(name,type);
    if (!f) { fprintf(stderr,"Couldn't open %s\n",name);exit(1); }
    return f;
}

FILE *fabfreopen(const char *name,const char *type,FILE *s)
{
    FILE *f = freopen(name,type,s);
    if (!f) { fprintf(stderr,"Couldn't open %s\n",name);exit(1); }
    return f;
}

@implementation Fabricator

static id setupsearchpath()
{
    /* current directory, not used for Computer Algebra Kit... */
    return [[Fablist new] add:[Fabstring str:"."]];
} 

- setUp
{
    searchpath = setupsearchpath();
    filenames  = [Fablist new];
    rootnames  = [Fablist new];
    return self;
}

+ new
{
    return [[super new] setUp];
}

- (STR)searchfile:(STR)name
{
    FILE *f;int i;
    
    for(i=0;i<[searchpath size];i++) {
	char aBuffer[FAB_FILENAME_MAX + 1];
	sprintf(aBuffer,"%s/%s",[[searchpath at:i] str],name);
	if ((f = fopen(aBuffer,"r"))) {
	    STR res;
	    fclose(f);
            res = (STR)malloc(strlen(aBuffer)+1);
            strcpy(res,aBuffer);
            return res;
	}
    }
    
    fprintf(stderr,"Couldn't find %s\nSearch Path:\n",name);
    for(i=0;i<[searchpath size];i++) {
	fprintf(stderr,"%s\n",[[searchpath at:i] str]);
    }
    exit(1);return NULL;
}

- addfile:(STR)string
{
    char *suffix;
    char rootname[1024];
    
    if (strrchr(string,'.')) {
	if (strrchr(string,'/')) {
	    strcpy(rootname,strrchr(string,'/') + 1);
	} else {
	    strcpy(rootname,string);
	}
	suffix = strrchr(rootname,'.');*suffix = 0; /* delete .fab suffix */
	[filenames add:[Fabstring str:string]];
	[rootnames add:[Fabstring str:rootname]];
    } else {
	fprintf(stderr,"%s is not a valid filename\n",string);
    }
    
    return self;
}

- addsearchpath:(STR)dir
{
    [searchpath add:[Fabstring str:dir]];
    return self;
}

- setheaderdir:(STR)string	 { headerdir = string;return self; }
- setsourcedir:(STR)string	 { sourcedir = string;return self; }
- setpublicheaderdir:(STR)string { publicheaderdir = string;return self; }
- setpublicsourcedir:(STR)string { publicsourcedir = string;return self; }
- setmandir:(STR)string		 { mandir = string;return self; }
- sethtmldir:(STR)string	 { htmldir = string;return self; }
- setrtfdir:(STR)string	         { rtfdir = string;return self; }
- setasciidir:(STR)string	 { asciidir = string;return self; }

- dumpheader:file to:(STR)dir:(STR)rootname
{
    FILE *f;char aBuffer[FAB_FILENAME_MAX + 1];
    makepath(aBuffer,dir,rootname,"h");
    f = fabfopen(aBuffer,"w");[file dumpheader:f];fclose(f);
    return self;
}

- dumppublicheader:file to:(STR)dir:(STR)rootname
{
    FILE *f;char aBuffer[FAB_FILENAME_MAX + 1];
    makepath(aBuffer,dir,rootname,"h");
    f = fabfopen(aBuffer,"w");[file dumppublicheader:f];fclose(f);
    return self;
}

- dumpsource:file to:(STR)dir:(STR)rootname
{
    FILE *f;char aBuffer[FAB_FILENAME_MAX + 1];
    makepath(aBuffer,dir,rootname,"m");
    f = fabfopen(aBuffer,"w");[file dumpsource:f];fclose(f);
    return self;
}

- dumppublicsource:file to:(STR)dir:(STR)rootname
{
    FILE *f;char aBuffer[FAB_FILENAME_MAX + 1];
    makepath(aBuffer,dir,rootname,"m");
    f = fabfopen(aBuffer,"w");[file dumppublicsource:f];fclose(f);
    return self;
}

- dumprtf:text to:(STR)dir:(STR)rootname
{
    char aBuffer[FAB_FILENAME_MAX + 1];

    makepath(aBuffer,dir,rootname,"rtf");
    fabfreopen(aBuffer,"w",stdout);
    writertf(text);

    fclose(stdout);

    return self;
}

- dumphtml:text to:(STR)dir:(STR)rootname
{
    char aBuffer[FAB_FILENAME_MAX + 1];

    makepath(aBuffer,dir,rootname,"html");
    fabfreopen(aBuffer,"w",stdout);

    printf("<HTML>\n");
    printf("<TITLE>%s Specification Sheet</TITLE>\n",rootname);
    printf("<H1>%s Specification Sheet</H1>\n",rootname);
    printf("\n<HR>\n");
    writehtml(text);

    fclose(stdout);

    return self;
}

- dumpman:text to:(STR)dir:(STR)rootname
{
    time_t now;
    char buf[1024];
    struct tm *tm;
    char aBuffer[FAB_FILENAME_MAX + 1];

    makepath(aBuffer,dir,rootname,"3");/* .3 suffix for section 3 */
    fabfreopen(aBuffer,"w",stdout);

    now = time(NULL);tm = localtime(&now);strftime(buf,256,"%b %d, %Y",tm);
    printf(".TH \"%s\" 3 \"%s\"\n",rootname,buf);
    writenroff(text);

    fclose(stdout);

    return self;
}

- dumpascii:text to:(STR)dir:(STR)rootname
{
    char aBuffer[FAB_FILENAME_MAX + 1];

    makepath(aBuffer,dir,rootname,"txt");
    fabfreopen(aBuffer,"w",stdout);
    writeascii(text);
    fclose(stdout);

    return self;
}

- run
{
    int i;
    id file;
    
    for(i=0;i<[filenames size];i++) {
	STR rootname = [[rootnames at:i] str];
	STR filename = [[filenames at:i] str];

	if ((file = [Unit name:filename:rootname])) {
	    if (headerdir) [self dumpheader:file to:headerdir:rootname];
	    if (sourcedir) [self dumpsource:file to:sourcedir:rootname];
	    if (publicheaderdir) [self dumppublicheader:file to:publicheaderdir:rootname];
	    if (publicsourcedir) [self dumppublicsource:file to:publicsourcedir:rootname];
	    if ((mandir)||(htmldir)||(asciidir)||(rtfdir)) {
	      id text = [file paragraphs:[OrdCltn new]];
	      if (mandir) [self dumpman:text to:mandir:rootname];
	      if (htmldir) [self dumphtml:text to:htmldir:rootname];
	      if (asciidir) [self dumpascii:text to:asciidir:rootname];
	      if (rtfdir) [self dumprtf:text to:rtfdir:rootname];
	    }
	} else {
	    exit(1);
	}
    }
    
    return self;
}

@end

