#ifndef _ISOURCEH
#define _ISOURCEH

/***************************************************************

	ImgSource Version 2 
							  
	Win32 Image File Processing DLL

	Copyright (c) 2000 Smaller Animals Software, Inc.

	Contact Information : 
		smallest@smalleranimals.com
		http://www.smalleranimals.com/index.htm

		This DLL is freely distributable. The full-access key is 
	not. If you have a full-access key without paying for it, 
	you are using this without permission, and that is illegal. 
		
		Companies who distribute ImgSource as part of an application
	without first registering ImgSource are in violation of the 
	terms-of-use (see ImgSource.rtf) and will be prosecuted.

	------

	There is no GIF support. This is due to Unisys licensing concerns.

	------

	This DLL, and the supporting static library, ISrc*.lib, is 
	thread-safe. All error and state data (PNG/JPG Text, DPI info, 
	error codes etc.) are stored in thread-local data. Each thread 
	gets its own copy of the error and state data. It will appear 
	to each thread that it is the only thread using the DLL.

	------

	Tech Credit:

		Color quantization code modified from DL1Quant, by 
		Dennis Lee (denlee@ecf.utoronto.ca)

		JPEG code from IJG v6b.
		PNG code from LibPng 1.0.3
		TIFF code from LibTiff v3.5.4
		TGA code from Timothy Bish
		PCX code from Bob Johnson's PCXLib
		PAX code from Smaller Animals Software's PALib
		TLA code from Smaller Animals Software's TLALib
		Crypto from Wei Dai's CryptLib 3.1
		Compression from ZLib

		All other code written by Smaller Animals Software. 

	Usage notes :

		0)	DLL must be initialized with _ISInitialize. The value 
	passed to this function may be the key to unlock the full 
	version of the DLL. Registered users will be given the real key. 
	Everyone else is free to try any value they like.

		1)	All parameters designated const char * are C-style 
	character strings. They are to be zero-terminated - some languages 
	don't do this by default. 

		2)	UINT32 is a 32-bit unsigned integer. This is a "Long" in VB. 
	VB's "Integer" is only 2 Bytes - this is too small.

		3)	BOOL is a 32-bit unsigned integer, zero for FALSE, one 
	for TRUE. Again, this is a "Long" in VB.

		4)	RGBQUAD is a structure defined as follows :

		typedef struct tagRGBQUAD { 
			BYTE    rgbBlue; 
			BYTE    rgbGreen; 
			BYTE    rgbRed; 
			BYTE    rgbReserved; 		  // rarely used
		} RGBQUAD; 

		5)	"__int32" is a 32-bit signed integer.

		6)	HGLOBAL is a HANDLE to memory allocated by GlobalAlloc.
	If a routine returns a non-NULL HGLOBAL value, it is up to
	the caller to call GlobalFree to release this memory back
	to the system. See ImgSource.rtf for more info.

		7)  RGB means 24-bit (3 BYTEs per pixel) Red, Green, Blue, 
	one BYTE each. If you have an RGB buffer of an image that 
	is 100 x 100 pixels, you MUST have a buffer of 100 x 3 x 100 
	BYTEs. All calls that says RGB assume this arrangement. RGB 
	is the same as "packed RGB".
	
		8)	"double" designates a 64-bit floating-point number.

		9)	"colormapped" or "8-bit" means an image which is made 
	up of 8-bit pixels each of which is an index into a palette 
	of 24-bit RGB values. This is not the same as a 24-bit 
	(RGB) buffer.

***************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

////////////////////////////////////////////////////////
// UINT32 is a 32-bit unsigned integer
// VC6 defines its own version of UINT32, using 
// 'unsigned int'. very dumb, given that 'int' will be 64
// bits when MS makes the jump to 64-bit Windows.
#if   _MSC_VER < 1200
typedef unsigned __int32 UINT32;
#else
// just use VC6's version of UINT32...
#endif

////////////////////////////////////////////////////////
// source manager object
typedef HGLOBAL HISSRC;

////////////////////////////////////////////////////////
// destination manager object
typedef HGLOBAL HISDEST;

////////////////////////////////////////////////////////
// callback function definition
typedef BOOL (CALLBACK* ISCALLBACKPTR)(const UINT32 uCurRow, const UINT32 uHeight, const UINT32 uUserData);

////////////////////////////////////////////////////////
// single-line file input objects
typedef HGLOBAL HDBMP;	// HDBMP = Handle to a Decompress BMP object
typedef HGLOBAL HDPCX;
typedef HGLOBAL HDPNG;
typedef HGLOBAL HDJPG;
typedef HGLOBAL HDPAX;
typedef HGLOBAL HDTLA;

////////////////////////////////////////////////////////
// single-line file output objects
typedef HGLOBAL HCBMP;	// HCBMP = Handle to a Compress BMP object
typedef HGLOBAL HCPCX;
typedef HGLOBAL HCTIFF;
typedef HGLOBAL HCPNG;
typedef HGLOBAL HCJPG;
typedef HGLOBAL HCTGA;
typedef HGLOBAL HCPAX;
typedef HGLOBAL HCTLA;

////////////////////////////////////////////////////////
// User source / dest function types
typedef BOOL	(CALLBACK* ISSourceOpenFn)		(const UINT32 uUserInfo);
typedef UINT32	(CALLBACK* ISSourceReadFn)		(void *pData, const UINT32 uDataLen, const UINT32 uUserInfo);
typedef BOOL	(CALLBACK* ISSourceWriteFn)		(const void *pData, const UINT32 uDataLen, const UINT32 uUserInfo);
typedef BOOL	(CALLBACK* ISSourceSeekFn)		(const UINT32 uOffset, const UINT32 uOrigin, const UINT32 uUserInfo);
typedef UINT32	(CALLBACK* ISSourceGetPosFn)	(const UINT32 uUserInfo);
typedef UINT32	(CALLBACK* ISSourceGetSizeFn)	(const UINT32 uUserInfo);
typedef BOOL	(CALLBACK* ISSourceEOFFn)		(const UINT32 uUserInfo);
typedef UINT32	(CALLBACK* ISSourceCloseFn)		(const UINT32 uUserInfo);

////////////////////////////////////////////////////////
// Channel mask values
#define CHRED		1
#define CHGREEN		2
#define CHBLUE		4

////////////////////////////////////////////////////////
// ImgSource Error values

// _ISGetLastError will return one of these enum values
// they are numbered consecutively, starting with -1.
#ifndef IS_ERROR_VALUES
#define IS_ERROR_VALUES
enum {
	IS_ERR_TRIALVERSION = -1,	// LIB was not initialized with a registered key
	IS_ERR_OK = 0,				// no err                                                                        
	IS_ERR_MEM = 1,				// out of memory                                                                 
	IS_ERR_FILEOPEN,			// error on file open                                                            
	IS_ERR_FILEREAD,			// error on file read                                                            
	IS_ERR_FILEWRITE,			// error on file write                                                           
	IS_ERR_BADPARAM = 5,		// bad user param                                                                
	IS_ERR_INVALIDBMP,			// bad BMP file                                                                  
	IS_ERR_BMPRLE,				// some RLE variations are not supported                                         
	IS_ERR_RESERVED1,			// reserved value
	IS_ERR_INVALIDJPG,			// bad JPG file                                                                  
	IS_ERR_DC = 10,				// error with device context                                                     
	IS_ERR_DIB,					// problem with a GetDIBits call                                                 
	IS_ERR_RESERVED2,			// reserved value
	IS_ERR_NORESOURCE,			// resource not found                                                            
	IS_ERR_CALLBACKCANCEL,		// callback returned FALSE - operation aborted                                   
	IS_ERR_INVALIDPNG = 15,		// bad PNG file                                                                  
	IS_ERR_PNGCREATE,			// internal PNG lib behavior - contact smaller animals s.w.                      
	IS_ERR_INTERNAL,			// misc unexpected behavior error - contact smaller animals s.w.                 
	IS_ERR_FONT,				// trouble creating a font object                                                
	IS_ERR_INTTIFF,				// misc internal TIFF error                                                      
	IS_ERR_INVALIDTIFF = 20,	// invalid TIFF file                                                             
	IS_ERR_NOTIFFLZW,			// this will not read TIFF-LZW images (note, unused error message)
	IS_ERR_INVALIDPCX,			// invalid PCX image                                                             
	IS_ERR_CREATEBMP,			// a call to the fn CreateCompatibleBitmap failed                                
	IS_ERR_NOLINES,				// end of an image while using single-line de/compression                        
	IS_ERR_GETDIB = 25,			// error during a call to GetDIBits                                              
	IS_ERR_NODEVOP,				// device does not support an operation required by this function                
	IS_ERR_INVALIDWMF,			// invalid windows metafile                                                      
	IS_ERR_DEPTHMISMATCH,		// the file was not of the requested bit-depth                                   
	IS_ERR_BITBLT,				// a call to BitBlt failed.                                                      
	IS_ERR_BUFTOOSMALL = 30,	// output buffer is too small for this operation                                 
	IS_ERR_TOOMANYCOLORS,		// not enough room in the output palette to store the colors from this image 
	IS_ERR_INVALIDTGA,			// Invalid TGA File                                                              
	IS_ERR_NOTGATHUMBNAIL,		// No TGA Thumbnail in the file                                                  
	IS_ERR_RESERVED3,			// reserved value
	IS_ERR_CREATEDIB = 35,		// a call to the fn CreateDIBitmap failed                                        
	IS_ERR_NOLZW,				// LZW de/compression is not permitted
	IS_ERR_SELECTOBJ,			// a call to SelectObject has failed (DC does not support this operation?)
	IS_ERR_BADMANAGER,			// the HISSRC or HISDEST object passed into the function does appear to be valid
	IS_ERR_OBSOLETE,			// the function is obsolete
	IS_ERR_CREATEDIBSECTION=40,	// a call to CreateDIBSection failed
	IS_ERR_SETWINMETAFILEBITS,	// a call to SetWinMetaFileBits failed (95/98 only)
	IS_ERR_GETWINMETAFILEBITS,	// a call to GetEnhMetaFileBits	or GetWinMetaFileBits failed
	IS_ERR_PAXPWD,				// apparently invalid PAX password
	IS_ERR_INVALIDPAX,			// invalid PAX file
	IS_ERR_NOSUPPORT = 45,		// this function is not supported in this build (see DLL build options)
	IS_ERR_INVALIDPSD,			// invalid PSD (Photoshop) file
	IS_ERR_PSDNOTSUPPORTED,		// this Photoshop sub-format is not supported
	IS_ERR_DECRYPT,				// decryption error - possible bad password for encrypted files
	IS_ERR_ENCRYPT,				// encryption failed
	IS_ERR_COMPRESSION = 50,	// compression failed
	IS_ERR_DECOMPRESSION,		// decompression error - possible bad password for encrypted files
	IS_ERR_INVALIDTLA,			// invalid TLA file. may indicate incorrect password
	IS_ERR_INVALIDWBMP,			// invalid or unsupported WBMP (Wireless Bitmap) image.
	IS_ERR_NOTIFFTAG,			// ImgSource does not support reading this TIFF tag
};
#endif

#pragma pack(push, 1)
// point structure, used in some ImgSource functions
#ifndef IS_XYDOUBLE_STRUCT_DEF
#define IS_XYDOUBLE_STRUCT_DEF
typedef struct XY_tag 
{
	double x; 
	double y;
} XYdouble;
#endif

#ifndef IS_DPI_STRUCT_DEF
#define IS_DPI_STRUCT_DEF
typedef struct ISDPIStruct_t
{
	UINT32 uDPIX;
	UINT32 uDPIY;
	UINT32 uDPIUnits;
} ISDPIStruct;
#endif

#pragma pack(pop, 1)


#define _ISEXP_  WINAPI

/*******************************************************
	_ISGetDLLVersion

	Purpose : 
	Get DLL version string - same as _ISGetLibVersion

	Param					Use
	------------------------------------
	pVers					receives version string	
							must be at least 20 chars!

*******************************************************/
extern void	_ISEXP_ _ISGetDLLVersion(char *pVers);

/*******************************************************
	_ISGetLibVersion

	Purpose : 
	Get DLL version string - same as _ISGetDLLVersion

	Param					Use
	------------------------------------
	pVers					receives version string	
							must be at least 20 chars!

*******************************************************/
extern void	_ISEXP_ _ISGetLibVersion(char *pVers);

/*******************************************************
	_ISGetLastError

	Purpose : 
	Get error code set by last function.

	All functions in ImgSource set the global error value.
	It is recommended that you test the error value often.

	Return
	------
	Error value
*******************************************************/
extern UINT32	_ISEXP_ _ISGetLastError();

/*******************************************************
	_ISInitialize

	Purpose : 
	Initialize the library

	Param					Use
	------------------------------------
	pKey					library unlock key
*******************************************************/
extern void		_ISEXP_ _ISInitialize(const char *pKey);

/*******************************************************
	_ISSetCallback

	Purpose : 
	Set global callback function. This function is called
	by all file read / write operations, and most image
	processing calls. It is not called by image display
	functions.

	Param					Use
	------------------------------------
	pCallback			pointer to callback function

	The callback function must be a function of this type :
	
		BOOL CALLBACK MyCallbackFunction(const UINT32 uCurRow, 
										const UINT32 uTotalRows,
										const UINT32 uUserData)
		{
			if (time to quit)
			{
				return FALSE;
			}

			return TRUE;
		}

	Set it as :
		_ISSetCallback(MyCallbackFunction);

	The library will also call the callback function once 
	per row for all image processing calls, except where 
	stated otherwise. 
	
	Operations which do not operate on image buffers are not 
	"image processing calls" and so, these won't use the callback 
	function. This includes calls such as JPG and PNG text calls, 
	PNG gamma and background operations, DPI queries, error 
	queries, etc.. 
	
	Basically, if there is no loop over pixels, there is no 
	callback call. However, this does not mean that *all* 
	operations which do have a pixel loop use the callback 
	function: some don't. Check the function comments in this 
	file, to be sure.

	If the callback function returns FALSE the operation aborts and 
	_ISGetLastError will return IS_ERR_CALLBACKCANCEL

	You can use this to monitor the progress of ImgSource operations.

	It is not neccessary for you to use this function. If you 
	do not set the callback function or pass a NULL to this call, 
	no callback will be performed.

	The callback is thread safe. Each thread can have its own 
	callback function.

	Parameter				Use
	------------------------------------
	pCallback				pointer to callback function
	uUserData				32-bit user data. this data is thread safe.

	Return
	------
	Error value
*******************************************************/
extern void		_ISEXP_ _ISSetCallback(ISCALLBACKPTR pCallback, const UINT32 uUserData);

/*******************************************************
	_ISOpenFileSource

	Purpose : 
	Create a file source object.

	Param					Use
	------------------------------------
	pFileName			input file path

	Return
	------
	NULL on failure, else a valid ImgSource source object.
	You must call _ISCloseSource to delete this object!
*******************************************************/
extern HISSRC	_ISEXP_ _ISOpenFileSource(const char * pFileName);

/*******************************************************
	_ISOpenMemorySource

	Purpose : 
	Create a memory-based source object.

	Param					Use
	------------------------------------
	pInBuf				input buffer
	uBufLen				length of input buffer

	Return
	------
	NULL on failure, else a valid ImgSource source object.
	You must call _ISCloseSource to delete this object!
*******************************************************/
extern HISSRC	_ISEXP_ _ISOpenMemorySource(const BYTE *pInBuf, const UINT32 uBufLen);

/*******************************************************
	_ISOpenUserSource

	Purpose : 
	Create a user-defined source object. Call this with
	pointers to caller-defined functions. 

	Param				Use
	------------------------------------
	pOpenFn			source open function
						called during ISOpenUserSource!!
						BOOL bOk = Open(uUserData);

	pReadFn			file read	
						UINT32 uReadLen = Read(pBuffer, uNumBytes, uUserData);

	pSeekFn			file seek. not all formats use this (PCX, TIFF and TGA do)

						BOOL bOK = Seek(uOffset, uOrigin, uUserData);
						uOrigin is one of :
                        0: Set absolute position
						1: Offset from current position
						2: From EOF. Offset is negative

	pPosFn			get file position
						UINT32 uPos = GetPos(uUserData);

	pSizeFn			get file size
						UINT32 uSize = GetSize(uUserData);

	pEOFFn			is end of file
						BOOL bEOF = EOF(uUserData);

	pCloseFn			source close
						UINT32 pos = Close(uUserData);
					
	uUserInfo		user-defined value. This will be
						passed to all of the functions in this
						ImgSource source object. The value can
						be used to identify which source object is
						being used, if you have multiple source
						objects open.

	Return
	------
	NULL on failure, else a valid ImgSource source object.
	You must call _ISCloseSource to delete this object!
*******************************************************/
extern HISSRC	_ISEXP_ _ISOpenUserSource(const ISSourceOpenFn pOpenFn,
											 const ISSourceReadFn pReadFn,
											 const ISSourceSeekFn pSeekFn,
											 const ISSourceGetPosFn pPosFn,
											 const ISSourceGetSizeFn pSizeFn,
											 const ISSourceEOFFn pEOFFn,
											 const ISSourceCloseFn pCloseFn,
											 const UINT32 uUserInfo);

/*******************************************************
	_ISCloseSource

	Purpose : 
	Close the source object.

	Param					Use
	------------------------------------
	hSource				valid 

	Return
	------
	Final position reached by the source data pointer. 
   Usually equal to the number of BYTEs read.
*******************************************************/
extern UINT32	_ISEXP_ _ISCloseSource(HISSRC hSource);

/*******************************************************
	_ISOpenFileDest

	Purpose : 
	Create a file destination object.

	Param					Use
	------------------------------------
	pFileName			output file path

	Return
	------
	NULL on failure, else a valid ImgSource destination object.
	You must call _ISCloseDest to delete this object!
*******************************************************/
extern HISDEST	_ISEXP_ _ISOpenFileDest(const char *pFileName);

/*******************************************************
	_ISOpenMemoryDest

	Purpose : 
	Create a memory destination object.

	Param					Use
	------------------------------------
	pOutBuf				output buffer. allocated by called
	uBufLen				length of output buffer

	Return
	------
	NULL on failure, else a valid ImgSource destination object.
	You must call _ISCloseDest to delete this object!
*******************************************************/
extern HISDEST	_ISEXP_ _ISOpenMemoryDest(BYTE *pOutBuf, const UINT32 uBufLen);

/*******************************************************
	_ISOpenUserDest

	Purpose : 
	Create a user-defined destination object. Call this with
	pointers to caller-defined functions. 

	Param				Use
	------------------------------------
	pOpenFn			source open function
					called during ISOpenUserDest!!
						BOOL bOk = Open(uUserData);

	pWriteFn		file write
						BOOL bOK = Write(pBuffer, uNumBytes, uUserData);

	pSeekFn			file seek. not all formats use this (TIFF does)
						BOOL bOK = Seek(uOffset, uOrigin, uUserData);
						uOrigin is one of :
                        0: Set absolute position
						1: Offset from current position
						2: From EOF. Offset is negative

	pPosFn			get file position
						UINT32 uPos = GetPos(uUserData);

	pSizeFn			get file size
						UINT32 uSize = GetSize(uUserData);

	pEOFFn			is end of file
						BOOL bEOF = EOF(uUserData);

	pCloseFn			source close
						UINT32 pos = Close(uUserData);
					
	uUserInfo		user-defined value. This will be
					passed to all of the functions in this
					ImgSource destination object. The value can
					be used to identify which destination object is
					being used, if you have multiple destination
					objects open.

	Return
	------
	NULL on failure, else a valid ImgSource destination object.
	You must call _ISCloseDest to delete this object!
*******************************************************/
extern HISDEST	_ISEXP_ _ISOpenUserDest(const ISSourceOpenFn pOpenFn,
											 const ISSourceWriteFn pWriteFn,
											 const ISSourceSeekFn pSeekFn,
											 const ISSourceGetPosFn pPosFn,
											 const ISSourceGetSizeFn pSizeFn,
											 const ISSourceEOFFn pEOFFn,
											 const ISSourceCloseFn pCloseFn,
											 const UINT32 uUserInfo);

/*******************************************************
	_ISCloseDest

	Purpose : 
	Close the destination object. 

	Param					Use
	------------------------------------
	hDest					valid 

	Return
	------
	Highest position reached by the destination data pointer. 
   Usually equal to the number of BYTEs read.
*******************************************************/
extern UINT32	_ISEXP_ _ISCloseDest(HISDEST hDest);

/*******************************************************
	_ISRead

	Purpose : 
	Read from the source manager object.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hSource				valid source manager
   pData             buffer to hold the data
   uDataLen          length of data to read

	Return
	------
	The number of bytes read.
*******************************************************/
extern UINT32	_ISEXP_ _ISRead(HANDLE hSource, BYTE * pData, const UINT32 uDataLen);

/*******************************************************
	_ISWrite

	Purpose : 
	Write from a destination manager object.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hDest 					valid destination manager
	pData					data to write
	uDataLen				length of data to write

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWrite(HANDLE hSource, const BYTE * pData, const UINT32 uDataLen);

/*******************************************************
	_ISSeek

	Purpose : 
	Set the file pointer.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hMgr   					valid source or destination manager
   uOffset					offset bytes
   uOrigin					starting place for repositioning :
							0: Set absolute position (uOffset >= 0)
							1: Offset from current position 
							2: From EOF. Offset is negative (uOffset <= 0)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSeek(HANDLE hMgr, UINT32 uOffset, UINT32 uOrigin);

/*******************************************************
	_ISGetPos

	Purpose : 
	Return the file pointer position.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hMgr   					valid source or destination manager

	Return
	------
	The current position of the file pointer
*******************************************************/
extern UINT32	_ISEXP_ _ISGetPos(HANDLE hMgr);

/*******************************************************
	_ISGetSize

	Purpose : 
	Return the size of the current file.

   If this is used on a memory destination object, it 
   returns the highest position reached by a write 
   operation.

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hMgr   					valid source or destination manager

	Return
	------
	The size of the current file.
*******************************************************/
extern UINT32	_ISEXP_ _ISGetSize(HANDLE hMgr);

/*******************************************************
	_ISEOF

	Purpose : 
	Test end of file condition

   This can be used for testing custom managers, or for
   providing extra functionality for your application.

	Param					Use
	------------------------------------
	hMgr   					valid source or destination manager

	Return
	------
	TRUE if EOF
*******************************************************/
extern BOOL	_ISEXP_ _ISEOF(HANDLE hMgr);

/*******************************************************
	ISWriteBMP

	Purpose : 
	Write 1,4,8 or 24 bits per pixel BMP file.

	Note:
	The format of your input image data depends on the 
	value you use for uBitsPerPixel. If you set uBitsPerPixel 
	to 24, pImg must be 24-bit RGB. If 8, pImg must be 8-bit 
	colormapped and pPal must contain a	valid palette. If 4, 
	pImg must be four bits per pixel with input rows ((w + 1) / 2)
	bytes wide. If 1, the image data must be 1 bit per pixel, 
	a row must be ((w + 7) / 8) bytes wide.

	In all cases, the pixel rows must start on whole byte boundaries.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image. 

	uWidth					image width (in pixels)
	iHeight					image height (in pixels). may be 
							negative for top-down BMPs

	uBitsPerPixel			1, 4, 8 or 24
	uColors					colors in pPal. 0, if uBitsPerPixel
							is 24.

	pPal					array of uColors RGBQUAD entries. the 
							image palette. NULL, if uBitsPerPixel
							is 24.

	bRLE8					write a Run Length Encoded 8-bit file

	Return
	------
	BYTEs written

*******************************************************/
extern UINT32	_ISEXP_ _ISWriteBMP(HISDEST hDest, BYTE *pImg, UINT32 uWidth, __int32 iHeight, UINT32 uBitsPerPixel, UINT32 uColors, RGBQUAD * pPal, BOOL bRLE8);

/*******************************************************
	ISWriteTIFF

	Purpose : 
	Write an 8-bit or 24-bit TIFF file.

	If uBitsPerPixel = 24, input image must be 24-bit RGB.
	If uBitsPerPixel = 8, input image must be 8-bit.


	Note : If you try to write LZW-compressed TIFFs (Comp 
	Mode 5), ImgSource will switch to mode 1, no compression. 
	This is to comply with the Unisys patent on LZW compression.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uBitsPerPixel			8 = write 8-bit image, pPal must be valid
							24 = RGB image
	pPal					array of 256 RGBQUAD entries. the 
							image palette.
	uCompMode				1 : no compression (default)

							7 : JPEG (only for 24-bit writes!)
							uses "no compresion" if set for 8-bit writes

							32773 :  Macintosh PackBits

							32946 :  Deflate (using ZLib. same as PNG, etc..)

	pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	1 - no units specified in file
										2 - dots / inch
										3 - dots / cm

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteTIFF(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uCompMode, ISDPIStruct *pDPIStruct);
								  
/*******************************************************
	ISWriteMultiTIFF

	Purpose : 
	Write a multi-paged 8-bit or 24-bit TIFF file.

	If uBitsPerPixel = 24, input images must be 24-bit RGB.
	If uBitsPerPixel = 8, input images must be 8-bit.

	Note : If you try to write LZW-compressed TIFFs (Comp 
	Mode 5), ImgSource will switch to mode 1, no compression. 
	This is to comply with the Unisys patent on LZW compression.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImages					array of pointers to your images
	uWidthArray				array of images widths (in pixels)
	uHeightArray			array of image heights (in pixels)
	uPageCount				pages to write (plus size of previous 3 arrays)
	uBitsPerPixel			8 = write 8-bit images, pPal must be valid
							24 = write RGB images
	pPal					array of 256 RGBQUAD entries. the 
							image palette.
	uCompMode				1 : no compression

							7 : JPEG (only for 24-bit writes!)
							uses "no compresion" if set for 8-bit writes

							32773 :  Macintosh PackBits

							32946 :  Deflate (using ZLib. same as PNG, etc..)

	pDPIStruct				optional resolution info. applies to all images.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	1 - no units specified in file
										2 - dots / inch
										3 - dots / cm
	
	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteMultiTIFF(HISDEST hDest, BYTE **pImages, UINT32 *uWidthArray, UINT32 *uHeightArray, UINT32 uPageCount, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uCompMode, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWrite1BitToTIFFFax

	Purpose : 
	Write a FAX (Group 3 or Group 4 TIFF) file from 
	an array of 1 bit images.
	
	Note:
	Each pixel row must start on a byte boundary and it must
	be a whole number of bytes wide.

	These rows must be (int)((uWidth + 7) / 8) bytes wide.

	Zero is white, one is black. 

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImageArray				an array of 1 bit images
	uWidthArray				image width array (in pixels)
	uHeight					image height array (in pixels)

	uPageCount				number of pages. also number of
							entries in pImageArray, uWidthArray
							and uHeightArray

	uFaxType				0 = CCITT Group 3
							1 = CCITT Group 4

	pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	1 - no units specified in file
										2 - dots / inch
										3 - dots / cm
	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWrite1BitToTIFFFax(HISDEST hDest, BYTE **pImages, UINT32 *uWidthArray, UINT32 *uHeightArray, UINT32 uPageCount, UINT32 uFaxType, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWritePCX

	Purpose : 
	Write an 8-bit or 24-bit PCX file.

	If uBitsPerPixel = 24, input image must be 24-bit RGB.
	If uBitsPerPixel = 8, input image must be 8-bit.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uBitsPerPixel			8 = write 8-bit image, pPal must be valid
							24 = RGB image
	pPal					array of 256 RGBQUAD entries. the 
							image palette.

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWritePCX(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISWriteWBMP

	Purpose : 
	Write a Writeless Bitmap. These are images used on
	cell phones, PDAs, etc.. 

	The input image must be a two-color, 8-bit image.
	All non-zero pixels are written as white. All zero
	pixels are written as black. 

	Because of their intended use, these images are 
	generally very small. Exact dimension limits are 
	impossible to give, since every cell phone or PDA has 
	a different amount of display area available. 
	
	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to 8-bit image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32 _ISEXP_ _ISWriteWBMP(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISWriteTGA

	Purpose : 
	Write an 8-bit or 24-bit TGA file.

	If uBitsPerPixel = 24, input image must be 24-bit RGB.
	If uBitsPerPixel = 8, input image must be 8-bit.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uBitsPerPixel			8 = write 8-bit image, pPal must be valid
							24 = RGB image
	pPal					array of 256 RGBQUAD entries. the 
							image palette.

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteTGA(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISWritePNG

	Purpose : 
	Write a PNG file.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image data
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uWidthBytes				image row width, in bytes
	uComponentBitDepth		bits per component:
							8 for 8-bit, RGB, RGBA
							1 for 1 bpp

	uColors					colors in image. if between 1 and 256, 
							you must supply a valid palette at pPal.
							if 16 bits or higher, 0 is acceptable here.

	pPal					array of up to 256 RGBQUAD entries. 
							the image palette.

	uColorType				0 = gray, 2 = RGB, 3 = colormapped,
							4 = gray-alpha, 6 = RGBA

	fGamma					file gamma value. 0 if don't care
	
	uInterlaceType			1 for Adam7 interlaced

    pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	0 - no units specified in file
										1 - dots / meter

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWritePNG(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uWidthBytes, UINT32 uComponentBitDepth, UINT32 uColors, RGBQUAD * pPal, UINT32 uColorType, double fGammaVal, UINT32 uInterlaceType, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWriteJPG

	Purpose : 
	Write a JPG file.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image data
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uQuality				1<->100 . quality is the inverse of
							compression.
	uProgressive			1 for progressive JPG
	uBitsPerPixel			8 = input image is 8 bit grayscale
							24 = input image is 24-bit RGB

    pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	0 - no units specified in file
										1 - dots / inch
										2 - dots / cm
	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteJPG(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uQuality, UINT32 uProgressive, UINT32 uBytesPerPixel, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWritePAX

	Purpose: 
	Write a PAX file.

    Note:
    Use ISSetPAXOutputInfo to set the PAX ID
    for each output file - before calling this 
    function!

	If uBitsPerPixel = 24, input image must be 24-bit RGB.
	If uBitsPerPixel = 8, input image must be 8-bit.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image data
	uWidth				    image width (in pixels)
	uHeight				    image height (in pixels)
    pPassword               password for this file
    uEncType                PAX encryption type (1 - 6)
	uBitsPerPixel			8 = write 8-bit image, pPal must be valid
							24 = RGB image
	pPal					array of 256 RGBQUAD entries. the 
							image palette.

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWritePAX(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, const char *pPassword, UINT32 uEncType, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISWritePSD

	Purpose : 
	Write a PSD (Adobe Photoshop) file.

    Note:
	This does not write any layer, mask or effect data.
	When Photoshop reads this file, it will place all 
	image data into the background layer.

	If uBitsPerPixel = 24, input image must be 24-bit RGB.
	If uBitsPerPixel = 8, input image must be 8-bit.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image data
	uWidth				    image width (in pixels)
	uHeight				    image height (in pixels)
	uBitsPerPixel			8 = write 8-bit image, pPal must be valid
							24 = RGB image
	pPal					array of 256 RGBQUAD entries. the 
							image palette.

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWritePSD(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISWriteTLA

	Purpose : 
	Write a TLA image.

    Notes:
	TLA supports encrypted and un-encrypted varieties. These
	have different header signatures, so it should be possible
	for readers to tell if a file is encrypted or not before
	attempting to decode it. Other than the signature, the 
	encrypted and un-encrypted formats are identical. Encryption
	will have no affect on compression.

	The format of the image data you provide must
	agree with what you use for uBitsPerPixel. ImgSource
	does not verify that you are providing data in the 
	correct format.

	It is legal to write an 8-bit TLA image with 0 colors 
	in the palette. In this situation, the reading application
	should know what the palette is by means outside the scope
	of this library.

	uStripSize and uFilterType will have directly affect the
	compression obtained on your pixel data. Just what kind of
	effect they will have is difficult to determine. The standard
	values of uStripSize = 20 and uFilterType = 5 should give
	good results for most applications.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImage					pointer to image data
	uWidth				    image width (in pixels)
	uHeight				    image height (in pixels)

	pPassword				encryption password. if this is an 
							empty string (not NULL!), the file 
							will be written as an un-encrypted TLA.

	uBitsPerPixel:			8:	pPal must contain uPalColors RGBQUADs
								image data is 1 byte per pixel.
							16:	pPal is ignored
								image data is 16 bits per pixel 
								A-RRRRR-GGGGG-BBBBB
							24:	pPal is ignored
								image data is 24-bit RGB
							32:	pPal is ignored
								image data is 32-bit RGBA

	uStripSize				pixel strip size. this may affect
							the compression ratio. 20 is a good choice.

	uFilterType				set compression pre-filter.
							0 = no filter
							1 = horizontal subtractive filter
							2 = vertical subtractive filter
							4 = Paeth filter
							these may be OR'd together. 
							5 is a good choice.

	uPalColors				number of colors in palette.

    pPal                    palette of uPalColors RGBQUADs
	
	pDataArray				array of pointers to tag data
	dataSizeArray			array of UNIT32s that provide the 
							lengths of each element in pDataArray

	uTags					number of elements in pDataArray and
							dataSizeArray.
	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteTLA(HISDEST hDest, BYTE *pImage, UINT32 uWidth, UINT32 uHeight, const char *pPassword, UINT32 uBitsPerPixel, UINT32 uStripSize, UINT32 uFilterType, UINT32 uPalColors, RGBQUAD * pPal, BYTE **pDataArray, UINT32 *dataSizeArray, UINT32 uTags);


/*******************************************************
	ISWriteRGBToWMF

	Purpose : 
	Write a Windows Metafile from an RGB image.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pRGB					pointer to RGB image
	uWidth				image width (in pixels)
	uHeight				image height (in pixels)
	bPlaceable			1 to write a placeable metafile 

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteRGBToWMF(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 bPlaceable);

/*******************************************************
	ISWriteRGBToEMF

	Purpose : 
	Write an Enhanced Windows Metafile from an RGB image.
   
	These files can be read by MS Office applications.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pRGB					pointer to RGB image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)

	Return
	------
	BYTEs written
*******************************************************/
extern UINT32	_ISEXP_ _ISWriteRGBToEMF(HISDEST hDest, BYTE *pImg, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISInitWriteBMPLines

	Purpose : 
	Set up for single line BMP file writing.
	Use with ISWriteNextBMPLine and ISDestroyWriteBMPLines.

	If pPal==NULL, ISReadNextBMPLine expects RGB lines
	else 8 bit lines

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid dest object
	uWidth				image width (in pixels)
	uHeight				image height (in pixels)
	pPal					palette

	Return
	------
	HCBMP - BMP single line write object
*******************************************************/
extern HCBMP	_ISEXP_ _ISInitWriteBMPLines(HISDEST hDest, UINT32 uWidth, __int32 iHeight, UINT32 uBitsPerPixel, __int32 iColors, RGBQUAD * pPal);

/*******************************************************
	ISWriteNextBMPLine

	Purpose : 
	Write a single BMP line

	Param					Use
	------------------------------------
	hSingle				single line HCBMP object
	pRow					image data. See ISInitWriteBMPLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextBMPLine(HCBMP hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWriteBMPLines

	Purpose : 
	Destroy the HCBMP object

	Param					Use
	------------------------------------
	hSingle				single line BMP object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWriteBMPLines(HCBMP hSingle);

/*******************************************************
	ISInitWritePCXLines

	Purpose : 
	Set up for single line PCX file writing.
	Use with ISWriteNextPCXLine and ISDestroyWritePCXLines.

	If pPal==NULL, ISReadNextPCXLine expects RGB lines
	else 8 bit lines

  Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid dest object
	uWidth				image width (in pixels)
	uHeight				image height (in pixels)
	pPal					palette

	Return
	------
	HCPCX - PCX single line write object
*******************************************************/
extern HCPCX	_ISEXP_ _ISInitWritePCXLines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal);

/*******************************************************
	ISWriteNextPCXLine

	Purpose : 
	Write a single PCX line

	Param					Use
	------------------------------------
	hSingle				single line HCPCX object
	pRow					image data. See ISInitWritePCXLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextPCXLine(HCPCX hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWritePCXLines

	Purpose : 
	Destroy the HCPCX object

	Param					Use
	------------------------------------
	hSingle				single line PCX object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWritePCXLines(HCPCX hSingle);

/*******************************************************
	ISInitWriteTGALines

	Purpose : 
	Set up for single line TGA file writing.
	Use with ISWriteNextTGALine and ISDestroyWriteTGALines.

	If pPal==NULL, ISReadNextTGALine expects RGB lines
	else 8 bit lines

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid dest object
	uWidth				image width (in pixels)
	uHeight				image height (in pixels)
	pPal					palette

	Return
	------
	HCTGA - TGA single line write object
*******************************************************/
extern HCTGA	_ISEXP_ _ISInitWriteTGALines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal);

/*******************************************************
	ISWriteNextTGALine

	Purpose : 
	Write a single TGA line

	Param					Use
	------------------------------------
	hSingle				single line HCTGA object
	pRow					image data. See ISInitWriteTGALines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextTGALine(HCTGA hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWriteTGALines

	Purpose : 
	Destroy the HCTGA object

	Param					Use
	------------------------------------
	hSingle				single line HCTGA object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWriteTGALines(HCTGA hSingle);

/*******************************************************
	ISInitWriteTIFFLines

	Purpose : 
	Set up for single line TIFF file writing.
	Use with ISWriteNextTIFFLine and ISDestroyWriteTIFFLines.

	If pPal==NULL, ISWriteNextTIFFLine expects RGB lines
	else 8 bit lines

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid dest object
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uBitsPerPixel			8 = write 8-bit images, pPal must be valid
							24 = write RGB images

	pPal					palette - when uBitsPerPixel = 8 

	uCompMode				1 : no compression (default)

							7 : JPEG (only for 24-bit writes!)
							uses "no compresion" if set for 8-bit writes

							32773 :  Macintosh PackBits

							32946 :  Deflate (using ZLib. same as PNG, etc..)

	pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	1 - no units specified in file
										2 - dots / inch
										3 - dots / cm


	Return
	------
	HCTIFF - TIFF single line write object
*******************************************************/
extern HCTIFF	_ISEXP_ _ISInitWriteTIFFLines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD * pPal, UINT32 uCompMode, ISDPIStruct* pDPIStruct);

/*******************************************************
	ISWriteNextTIFFLine

	Purpose : 
	Write a single TIFF line

	Param					Use
	------------------------------------
	hSingle				single line TIFF object
	pRow					image data. See ISInitWriteTIFFLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextTIFFLine(HCTIFF hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWriteTIFFLines

	Purpose : 
	Destroy the HCTIFF object

	Param					Use
	------------------------------------
	hSingle				single line HCTIFF object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWriteTIFFLines(HCTIFF hSingle);

/*******************************************************
	ISInitWritePNGLines

	Purpose : 
	Set up for single line PNG file writing.
	Use with ISWriteNextPNGLine and ISDestroyWritePNGLines.

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid destination object
	p8Bit					pointer to 8 bit image
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uWidthBytes				image row width, in bytes (ex. w *3 for RGB)
	uBitDepth				bits per component (ex. 8 for 8-bit and RGB)
	uColors					colors in image. if between 1 and 256, 
							you must supply a valid palette at pPal.
							if 16 bits or higher, 0 is acceptable here.

	pPal					array of up to 256 RGBQUAD entries. 
							the image palette.

	uColorType				0 = gray, 2 = RGB, 3 = palettized,
							4 = gray-alpha, 6 = RGBA

	fGamma					file gamma value. 0 if don't care

    pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	0 - no units specified in file
										1 - dots / meter

	Return
	------
	BYTEs written
*******************************************************/
extern HCPNG	_ISEXP_ _ISInitWritePNGLines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uWidthBytes, UINT32 uBitDepth, UINT32 uColors, RGBQUAD * pPal, UINT32 uColorType, double fGammaVal, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWriteNextPNGLine

	Purpose : 
	Write a single PNG line

	Param					Use
	------------------------------------
	hSingle				single line PNG object
	pRow					image data. See ISInitWritePNGLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextPNGLine(HCPNG hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWritePNGLines

	Purpose : 
	Destroy the HCPNG object

	Param					Use
	------------------------------------
	hSingle				single line HCPNG object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWritePNGLines(HCPNG hSingle);

/*******************************************************
	ISInitWriteJPGLines

	Purpose : 
	Set up for single line JPG file writing.
	Use with ISWriteNextJPGLine and ISDestroyWriteJPGLines.

	If bGrayscale = 0, ISWriteNextJPGLine expects RGB lines
	else 8 bit lines.

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hDest					valid destination object
	pImg					pointer to image data
	uWidth					image width (in pixels)
	uHeight					image height (in pixels)
	uQuality				1<->100 . quality is the inverse of
							compression.
	uProgressive			1 for progressive JPG
	bGrayscale				1 = input image is 8 bit grayscale
							0 = input image is 24-bit RGB

    pDPIStruct				optional resolution info.
							(pass NULL if you don't care)
							uDPIX		X pixel density
							uDPIY		Y pixel density
							uDPIUnits	0 - no units specified in file
										1 - dots / inch
										2 - dots / cm


	Return
	------
	HCJPG - JPG single line write object
*******************************************************/
extern HCJPG	_ISEXP_ _ISInitWriteJPGLines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, UINT32 uQuality, UINT32 uProgressive, UINT32	bGrayscale, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISWriteNextJPGLine

	Purpose : 
	Write a single JPG line

	Param					Use
	------------------------------------
	hSingle					single line JPG object
	pRow					image data. See ISInitWriteJPGLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextJPGLine(HCJPG hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWriteJPGLines

	Purpose : 
	Destroy the HCJPG object

	Param					Use
	------------------------------------
	hSingle					single line HCJPG object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWriteJPGLines(HCJPG hSingle);

/*******************************************************
	ISInitWritePAXLines

	Purpose : 
	Set up for single line PAX file writing.
	Use with ISWriteNextPAXLineRGB, ISWriteNextPAXLine8Bit
    and ISDestroyWritePAXLines.

  	Note: Single line input/output functions are not 
	available for unregistered users.

    Note:
    Use ISSetPAXOutputInfo to set the PAX ID
    for each output file - before calling this 
    function!

	Param					Use
	------------------------------------
	hDest					valid dest object
	uWidth				    image width (in pixels)
	uHeight				    image height (in pixels)
	pPal					palette of 256 RGBQUADs 
    uBitsPerPixel           BYTEs per pixel (1 or 3)
                            8 : format is 8-bit PAX. pPal
                            must point to a palette. lines must 
                            be written with ISWriteNextPAXLine8Bit.
                            24 : format is 24-bit PAX. pPal
                            may be NULL. lines must be written 
                            with ISWriteNextPAXLineRGB.

    pPassword               password for this file
    uEncType                PAX encryption type (1 - 6)

	Return
	------
	HCPAX - PAX single line write object
*******************************************************/
extern HCPAX	_ISEXP_ _ISInitWritePAXLines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, RGBQUAD * pPal, UINT32 uBitsPerPixel, const char *pPassword, UINT32 uEncType);

/*******************************************************
	ISWriteNextPAXLine

	Purpose : 
	Write a single PAX line

	Param					Use
	------------------------------------
	hSingle				    single line HCPAX object
	pRow					image data. See ISInitWritePAXLines 
							for image format notes.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextPAXLine(HCPAX hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWritePAXLines

	Purpose : 
	Destroy the HCPAX object

	Param					Use
	------------------------------------
	hSingle				    single line PAX object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWritePAXLines(HCPAX hSingle);

/*******************************************************
	ISInitWriteTLALines

	Purpose : 
	Set up for single line TLA file writing.
	Use with ISWriteNextTLALine and ISDestroyWriteTLALines.

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	See comments for ISWriteTLA for an explanation of
	the parameters.

	Return
	------
	HCTLA - TLA single line write object
*******************************************************/
extern HCTLA	_ISEXP_ _ISInitWriteTLALines(HISDEST hDest, UINT32 uWidth, UINT32 uHeight, const char *pPassword, UINT32 uBitsPerPixel, UINT32 uStripSize, UINT32 uFilterType, UINT32 uPalColors, RGBQUAD * pPal, BYTE **pDataArray, UINT32 *dataSizeArray, UINT32 uTags);

/*******************************************************
	ISWriteNextPAXLine

	Purpose : 
	Write a single TLA line

	Param					Use
	------------------------------------
	hSingle				    single line HCTLA object
	pRow					image data. See comments for 
							ISWriteAllToTLA for an explanation 
							of the image data format.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISWriteNextTLALine(HCTLA hSingle, BYTE *pRow);

/*******************************************************
	ISDestroyWriteTLALines

	Purpose : 
	Destroy the HCTLA object

	Param					Use
	------------------------------------
	hSingle				    single line TLA object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyWriteTLALines(HCTLA hSingle);

/*******************************************************
	ISReadImageToDIB

	Purpose : 
	Read an image file to a DIB.
	
	Operation:
	This function attempts to determine the file type
	by using ISGuessFileType, if this fails, it examines
	the HISSRC object to see if the manager is associated
	with a file (from ISOpenFileSource). If there is a file,
	the file extension is used to determine file type.

	Once file type has been determined, it determines the 
	bit depth of the input image, then uses the appropriate
	ImgSource file reading function to read the image to 
	read the image its native bit depth, or the closest
	supported bit-depth for that file format.

	This image is then converted to a DIB and returned to the
	caller.

	Param					Use
	------------------------------------
	hSource					valid source object. if this is
							not a file source, this function
							may not be able to determine file 
							type, if the format is not one 
							supported by ISGuessFileType.

	Return
	------
	HGLOBAL handle to DIB. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL _ISEXP_ _ISReadImageToDIB(HISSRC hSrc);

/*******************************************************
	ISReadBMP

	Purpose : 
	Read a BMP file.
	
	If the file bit depth is greater than the requested 
	bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

	Param					Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uBitsPerPixel		desired bits per pixel (1, 8 or 24)
						if 1, returned image lines will 
						be ((w + 7) / 8) bytes wide

	pPal				256 RGBQUADs. receives palette, 
						if image is colormapped

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadBMP(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISReadPCX

	Purpose : 
	Read a PCX file.

	Note:
	If the file bit depth is greater than the requested 
	bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

	Note:
	For custom source managers writers : this operation
	requires that the input source manager is able to seek 
	from the end of the file. This is because the PCX palette 
	is stored after the pixel data, but must be read first.

	Param					Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uBitsPerPixel		desired bits per pixel.
						8 - read to 8-bit 
						24 - read to RGB
	pPal				256 RGBQUADs. receives palette, 
						if image is colormapped

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadPCX(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISReadTGA

	Purpose : 
	Read a TGA file to an RGB buffer. Files with 
	less than 24 bits per pixel are promoted to RGB.

	Note:
	If the file bit depth is greater than the requested 
	bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

	Note:
	For custom source managers writers : this operation
	requires that the input source manager is able to seek 
	from the end of the file.

	Param					Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uBitsPerPixel		desired bits per pixel.
						8 - read to 8-bit 
						24 - read to RGB
	pPal				256 RGBQUADs. receives palette, 
						if image is colormapped


	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadTGA(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISReadPNG

	Purpose : 
	Read a PNG file. 
	
	Note: 
	Files with less than the requested bit depth 
	are promoted to the requested depth.

	A PNG file may contain any number of key/text pairs.
	When ImgSource reads a PNG file, it stores these strings 
	in the ImgSource PNG text buffer. These strings remain 
	valid until :

		1) the next PNG file is read
		2) ISClearPNGInputText is called

	See ISGetPNGInputTextCount, ISGetPNGInputText,
	ISClearPNGInputText.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	uBitsPerPixel			desired bits per pixel
							8 - read to 8-bit colormapped
							24 - read to 24-bit RGB
							32 - read to 32-bit RGBA

	pPal					256 RGBQUADs. receives palette

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadPNG(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISReadJPG

	Purpose : 
	Read a JPG file. 
	
	A JPG file may contain any number text fields :
	JPEG_COM markers. These strings can be any length. 
	When ImgSource reads a JPG file, it stores these strings 
	in the ImgSource JPG text buffer. These strings remain 
	valid until :

		1) the next JPG file is read
		2) ISClearJPGInputText is called

	See ISGetJPGInputTextCount, ISGetJPGInputText,
	ISClearJPGInputText.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	uBitsPerPixel			desired bit depth:
							8 - image will be returned as 8-bit grayscale
							24 - image will be returned as 24-bit RGB

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadJPG(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel);

/*******************************************************
	ISReadWBMP

	Purpose : 
	Read a WBMP file. This is NOT a "Windows Bitmap", this is
	a "Wireless Bitmap", intended for use on cell phones, PDAs,
	etc..
	
	This supports WBMP Level-0, uncompressed, two color images.
	Extension Headers are not supported.

	All images are read to 8-bit image buffers. 

	Param					Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	pPal				256 RGBQUADs. receives palette

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL _ISEXP_ _ISReadWBMP(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal);

/*******************************************************
	ISReadMetafileToRGB

	Purpose : 
	Read a metafile (WMF, EMF, AMF) to an RGB buffer.

   Due to the way metafiles are handled by the OS, metafiles
   can only be read from a disk-based file.

	Param				Use
	------------------------------------
	pFileName			path to input file
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uDefWidth			default width for non-sized files
	uDefHeight			default height for non-sized files
	clrBackground		background color to render the image onto

	Return
	------
	HGLOBAL handle to RGB image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadMetafileToRGB(const char *pFileName, UINT32 *puWidth, UINT32 *puHeight, UINT32 uDefWidth, UINT32 uDefHeight, COLORREF clrBackground);

/*******************************************************
	ISGuessFileType

	Purpose : 
	Try to guess at the format of the input image 
	
	Param					Use
	------------------------------------
	hSource				valid source object

	Return
	------
	file type :
		0 - unable to identify or file read error
		1 - BMP
		2 - GIF
		3 - JPG
		4 - PNG
		5 - PCX
		6 - TIFF
		7 - WMF
		8 - EMF
		9 - PAX
		10 - PSD (Photoshop)
		11 - TLA (encrypted)
		12 - TLA (unencrypted)
		13 - WBMP	
		14 - TGA

   Note : 
   This function can confuse certain varieties of TGA
   files for WBMP files. The TGA specification does not provide
   for a identifying "signature" in the TGA file, so there is
   no sure way to tell a TGA file from a random stream of BYTEs.
   However, WBMP does provide such a signature. Unfortunately, 
   the WBMP signature is simply two zero BYTEs at the start of 
   the file and it is quite common for TGA files to start with two
   zero bytes. 

   To get around the problem, this function will examine the 
   file extension of the source file (if the source manager was
   created with ISOpenFileSource). If the signature says WBMP
   but the file extension says "TGA", this will return 14 (TGA).

   Note : This function moves the "file pointer" for the
   source manager used. So you can't do this : 

   HISSRC hSrc = ISOpenFileSource(...);
   ISGuessFileType(hSrc);   // moves file pointer
   ISReadJPGToRGB(hSrc...); // not at start of file!! read will fail
   ISCloseSource(hSrc);

   Instead, do this : 

   HISSRC hSrc = ISOpenFileSource(...);
   ISGuessFileType(hSrc);   // moves file pointer
   ISSeek(hSrc, 0,0);       // moves file pointer back to the start of the file
   ISReadJPGToRGB(hSrc...); // succeeds
   ISCloseSource(hSrc);

*******************************************************/
extern UINT32	_ISEXP_ _ISGuessFileType(HISSRC hSource);

/*******************************************************
	ISReadPAX

	Purpose : 
	Read a PAX file.

	Note:
	If the file bit depth is greater than the requested 
	bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

	Param				Use
	-----------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uBitsPerPixel		desired bit depth
						8 - read to 8-bit 
						24 - read to RGB
	pPal				256 RGBQUADs. receives palette, 
						if image is colormapped

    pPassword           password for this file. if the correct
                        password is not supplied, this operation
                        will fail.

	Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadPAX(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, const char *pPassword);

/*******************************************************
	ISReadPSD

	Purpose : 
	Read a PSD (Adobe Photoshop) file.

	Reads the following Photoshop sub-formats:
		Bitmap (compressed only)
		Grayscale
		Indexed
		RGB
		CMYK
		Duotone (without color info)

	Note:
	If the file bit depth is greater than the requested 
	bit depth, this function will fail with IS_ERR_DEPTHMISMATCH

    Note:
	When Photoshop writes a PSD file, it combines all 
	layers, masks, effects, etc. into a single-layered 
	image. This combination is saved into every PSD file. 
	This is what ImgSource reads. The actual layer, mask, 
	etc. data is ignored by the PSD reader.
	
	Param				Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	uBitsPerPixel		desired bit depth
						8 - read to 8-bit 
						24 - read to RGB
	pPal				256 RGBQUADs. receives palette, 
						if image is colormapped

    Return
	------
	HGLOBAL handle to image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadPSD(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal);

/*******************************************************
	ISReadTLA

	Purpose : 
	Read a TLA file.

	Note:
	Use ISGuessFileType to determine the encryption
	state (encrypted/unencrypted) of this file before
	attempting to read. Attempting to read an encrypted
	TLA file with the wrong (or no) password will fail.
	
	1 byte per pixel images will usually have a palette.

	Param					Use
	------------------------------------
	hSource				    valid source object
	puWidth				    receives image width (in pixels)
	puHeight				receives image height (in pixels)
    pPassword               password for this file. if you are
							reading an unencrypted TLA file,
							this is ignored. otherwise, only the
							password that was used to create the 
							file will work for reading the file.

	uBytesPerPixel			format to return the data in.
							data is returned in one of these formats:

							1 = 8-bit pixels, with palette, if any.
							2 = 16 bit RGBA (A-RRRRR-GGGGG-BBBBB)
							3 = 24-bit RGB
							4 = 32-bit RGBA
							
							* Can not read 2, 3 or 4 bpp to 1 byte per pixel.
							* Can read all images to higher bit depth (promotion).
							* Can read 3 and 4 bytes per pixel to 16 bytes per pixel.
							
	pPal					receives palette, if any. should be 
							256 RGBQUADs, though fewer colors may actually be 
							in use.
	
	Return
	------
	HGLOBAL handle to the image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadTLA(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, const char *pPassword, UINT32 uBytesPerPixel, RGBQUAD *pPal);

/*******************************************************
	_ISReadTIFF

	Purpose : 
	Read a single page from a TIFF file. 

	For single page TIFFs, use page index 0.

	Use ISGetTIFFPageCount to determine the number of
	pages in the TIFF file.

 	All TIFFs are promoted to 24 or 32 bits per pixel as
	requested.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	uBitsPerPixel			desired bit depth
							24 - RGB 
							32 - RGBA
	pPal					receives image palette (not
							currently used)
	uFrameIndex				page number to read

	Return
	------
	HGLOBAL handle to RGB image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadTIFF(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uFrameIndex);

/*******************************************************
	ISReadTIFFFaxPageTo8Bit

	Purpose : 
	Read a single page from a CCITT Group 3 or 4 TIFF file 
	to an 8 bit buffer. 

    There is no palette returned. The pixels in the image are
	either 0 or 1. 0 = white, 1 = black.
	
	This will only work with G3 and G4 TIFFs. If you try to read 
	any other type, this function will fail.

	Use ISGetTIFFPageCount to determine the number of
	pages in the TIFF file.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	uFrameIndex				page number to read

	Return
	------
	HGLOBAL handle to RGB image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadTIFFFaxPageTo8Bit(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uFrameIdx);

/*******************************************************
	ISReadTIFFFaxPageTo1Bit

	Purpose : 
	Read a single page from a CCITT Group 3 or 4 TIFF file 
	to a 1 bit buffer. Pixel rows start on whole BYTE boundaries.

    There is no palette returned. The pixels in the image are
	either 0 or 1. 0 = white, 1 = black.
	
	This will only work with G3 and G4 TIFFs. If you try to read 
	any other type, this function will fail.

	Use ISGetTIFFPageCount to determine the number of
	pages in the TIFF file.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	uFrameIndex				page number to read

	Return
	------
	HGLOBAL handle to RGB image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadTIFFFaxPageTo1Bit(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uFrameIdx);

/*******************************************************
	ISGetTIFFPageCount

	Purpose : 
	Determine the number of frames/pages in the current TIFF file.

	Param					Use
	------------------------------------
	hSource					valid source object

	Return
	------
	Number of frames.
*******************************************************/
extern UINT32	_ISEXP_ _ISGetTIFFPageCount(HISSRC hSource);

/*******************************************************
	ISGetTIFFTag

	Get a single TIFF tag data from a file.

	
	Tag name				ID			Variant data type
	-----------------------------------------------------
	(tags and note from tiff.h, part of LibTiff)

	TIFFTAG_IMAGEWIDTH		256			VT_UI4
	TIFFTAG_IMAGELENGTH		257			VT_UI4
	TIFFTAG_BITSPERSAMPLE	258			VT_UI2
	TIFFTAG_COMPRESSION		259			VT_UI2
		Values:
			COMPRESSION_NONE		1		dump mode 
			COMPRESSION_CCITTRLE	2		CCITT modified Huffman RLE 
			COMPRESSION_CCITTFAX3	3		CCITT Group 3 fax encoding 
			COMPRESSION_CCITTFAX4	4		CCITT Group 4 fax encoding 
			COMPRESSION_LZW			5		Lempel-Ziv  & Welch 
			COMPRESSION_OJPEG		6		6.0 JPEG 
			COMPRESSION_JPEG		7		JPEG DCT compression 
			COMPRESSION_NEXT		32766	NeXT 2-bit RLE 
			COMPRESSION_CCITTRLEW	32771	#1 w/ word alignment 
			COMPRESSION_PACKBITS	32773	Macintosh RLE 
			COMPRESSION_THUNDERSCAN	32809	ThunderScan RLE 
			
			codes 32895-32898 are reserved for 
			ANSI IT8 TIFF/IT <dkelly@etsinc.com) 
			COMPRESSION_IT8CTPAD	32895   IT8 CT w/padding 
			COMPRESSION_IT8LW		32896   IT8 Linework RLE 
			COMPRESSION_IT8MP		32897   IT8 Monochrome picture 
			COMPRESSION_IT8BL		32898   IT8 Binary line art 
			
			compression codes 32908-32911 are reserved for Pixar 
			COMPRESSION_PIXARFILM	32908   Pixar companded 10bit LZW 
			COMPRESSION_PIXARLOG	32909   Pixar companded 11bit ZIP 
			COMPRESSION_DEFLATE		32946	Deflate compression 
			COMPRESSION_ADOBE_DEFLATE   8   Deflate compression, 
											as recognized by Adobe 
			
			compression code 32947 is reserved for 
			Oceana Matrix <dev@oceana.com> 
			COMPRESSION_DCS         32947   Kodak DCS encoding 
			
			COMPRESSION_JBIG		34661	ISO JBIG 
			COMPRESSION_SGILOG		34676	SGI Log Luminance RLE 
			COMPRESSION_SGILOG24	34677	SGI Log 24-bit packed 

	TIFFTAG_PHOTOMETRIC		262			VT_UI2
		Values:
			PHOTOMETRIC_MINISWHITE	0	 min value is white 
			PHOTOMETRIC_MINISBLACK	1	 min value is black 
			PHOTOMETRIC_RGB			2	 RGB color model 
			PHOTOMETRIC_PALETTE		3	 color map indexed 
			PHOTOMETRIC_MASK		4	 holdout mask 
			PHOTOMETRIC_SEPARATED	5	 color separations 
			PHOTOMETRIC_YCBCR		6	 CCIR 601 
			PHOTOMETRIC_CIELAB		8	 1976 CIE L*a*b* 
			PHOTOMETRIC_LOGL		32844	 CIE Log2(L) 
			PHOTOMETRIC_LOGLUV		32845	 CIE Log2(L) (u',v') 

	TIFFTAG_RESOLUTIONUNIT	296			VT_UI2
		Values:
			RESUNIT_NONE			1	no meaningful units
			RESUNIT_INCH			2	english
			RESUNIT_CENTIMETER		3	metric

	TIFFTAG_XRESOLUTION		282			VT_R4
	TIFFTAG_YRESOLUTION		283			VT_R4
	TIFFTAG_NUMBEROFINKS	334			VT_UI2
	TIFFTAG_DOCUMENTNAME	269			VT_BSTR
	TIFFTAG_IMAGEDESCRIPTION 270		VT_BSTR
	TIFFTAG_MAKE			271			VT_BSTR
	TIFFTAG_MODEL			272			VT_BSTR
	TIFFTAG_PAGENAME		285			VT_BSTR
	TIFFTAG_SOFTWARE		305			VT_BSTR
	TIFFTAG_DATETIME		306			VT_BSTR
	TIFFTAG_ARTIST			315			VT_BSTR
	TIFFTAG_HOSTCOMPUTER	316			VT_BSTR
	TIFFTAG_INKNAMES		333			VT_BSTR
	TIFFTAG_TARGETPRINTER	337			VT_BSTR

	Note:
	It is the caller's responsibility to call SysFreeString
	to release any string data that was allocated by this
	function.

	ex.
	ISGetTIFFTag(hSrc, 256, &varData);
 	if (varData.vt == VT_BSTR)
	{
		if (varData.bstrVal!=NULL)
		{
			::SysFreeString(varData.bstrVal);
		}
	}

	Note:
	The caller must call ISSeek(hSrc, 0, 0) after
	each call to reset the file pointer, if you are
	getting multiple tags, or if you plan to read the
	file using any functions that assume the file pointer
	is at the start of the file.

	Param					Use
	------------------------------------
	hSource					source manager
	uFrameIdx				frame index of the TIFF (0 for first)
	uTiffTagID				ID value for the tag you want (see table above)
	pVarData				receives data. 

	returns FALSE if not a TIFF FAX file
*******************************************************/
extern BOOL _ISEXP_ _ISGetTIFFTag(HISSRC hSource, UINT32 uFrameIdx, UINT32 uTiffTagID, VARIANT *pVarData);

/*******************************************************
	ISInitReadBMPLines

	Purpose : 
	Set up for single line BMP file reading.
	Use with ISReadNextBMPLine and ISDestroyReadBMPLines.

	If pPal==NULL, ISReadNextBMPLine returns RGB lines
	else 8 bit lines

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hSource					valid source object
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	pPal					receives palette

	Return
	------
	HDBMP - BMP single line read object
*******************************************************/
extern HDBMP	_ISEXP_ _ISInitReadBMPLines(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal);

/*******************************************************
	ISReadNextBMPLine

	Purpose : 
	Read a single line from a BMP file

	Param					Use
	------------------------------------
	hSingle					valid HDBMP object
	pRow					receives image data, see 
							ISInitReadBMPLines for format notes

	puRowIndex				receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextBMPLine(HDBMP hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISDestroyReadBMPLines

	Purpose : 
	Destroy the HDBMP object and any memory associated 
	with it.

	Param					Use
	------------------------------------
	hSingle				valid HDBMP object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyReadBMPLines(HDBMP hSingle);

/*******************************************************
	ISInitReadPCXLines

	Purpose : 
	Set up for single line PCX file reading.
	Use with ISReadNextPCXLine and ISDestroyReadPCXLines.

	If pPal==NULL, ISReadNextPCXLine returns RGB lines
	else 8 bit lines

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Note for custom source managers writers : this operation
	requires that the input source manager is able to seek 
	from the end of the file. This is because the PCX palette 
	is stored after the pixel data, but must be read first.

	Param				Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	pPal				receives palette

	Return
	------
	HDPCX - PCX single line read object
*******************************************************/
extern HDPCX	_ISEXP_ _ISInitReadPCXLines(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal);

/*******************************************************
	ISReadNextPCXLine

	Purpose : 
	Read a single line from a PCX file

	Param					Use
	------------------------------------
	hSingle				valid HDPCX object
	pRow					receives image data, see 
							ISInitReadPCXLines for format notes

	puRowIndex			receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextPCXLine(HDPCX hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISDestroyReadPCXLines

	Purpose : 
	Destroy the HDPCX object and any memory associated 
	with it.

	Param				Use
	------------------------------------
	hSingle				valid HDPCX object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyReadPCXLines(HDPCX hSingle);

/*******************************************************
	ISInitReadJPGLines

	Purpose : 
	Set up for single line JPG file reading.
	Use with ISReadNextJPGLine and ISDestroyReadJPGLines.

	If bGrayscale==0, this returns RGB lines
	else 8 bit lines

	A JPG file may contain any number text fields :
	JPEG_COM markers. These strings can be any length. 
	When ImgSource reads a JPG file, it stores these strings 
	in the ImgSource JPG text buffer. These strings remain 
	valid until :

		1) the next JPG file is read
		2) ISClearJPGInputText is called

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param				Use
	------------------------------------
	hSource				valid source object
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	pPal				receives palette

	Return
	------
	HDJPG - JPG single line read object
*******************************************************/
extern HDJPG	_ISEXP_ _ISInitReadJPGLines(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 bGrayscale);

/*******************************************************
	ISReadNextJPGLine

	Purpose : 
	Read a single line from a JPG file

	Param				Use
	------------------------------------
	hSingle				valid HDJPG object
	pRow				receives image data, see 
						ISInitReadJPGLines for format notes

	puRowIndex			receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextJPGLine(HDJPG hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISDestroyReadJPGLines

	Purpose : 
	Destroy the HDJPG object and any memory associated 
	with it.

	Param				Use
	------------------------------------
	hSingle				valid HDJPG object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyReadJPGLines(HDJPG hSingle);

/*******************************************************
	ISInitReadPAXLines

	Purpose : 
	Set up for single line PAX file reading.
	Use with ISReadNextPAXLineRGB, ISReadNextPAXLine8Bit
    and ISDestroyReadPAXLines.

  	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	hSource			    	valid source object
	puWidth			    	receives image width (in pixels)
	puHeight				receives image height (in pixels)
	pPal					receives palette (for 8-bit images)
    pPassword               image password. if the correct
                            password is not supplied, this operation
                            will fail.

	Return
	------
	HDPAX - PAX single line read object
*******************************************************/
extern HDPAX	_ISEXP_ _ISInitReadPAXLines(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, RGBQUAD *pPal, const char *pPassword);

/*******************************************************
	ISReadNextPAXLineRGB

	Purpose : 
	Read a single RGB line from a PAX file

	Param					Use
	------------------------------------
	hSingle				    valid HDPAX object
	pRow					receives RGB image data, see 
							ISInitReadPAXLines for format notes
	puRowIndex			    receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextPAXLineRGB(HDPAX hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISReadNextPAXLine8Bit

	Purpose : 
	Read a single 8-bit line from a PAX file

	Param					Use
	------------------------------------
	hSingle				    valid HDPAX object
	pRow					receives 8-bit image data, see 
							ISInitReadPAXLines for format notes
	puRowIndex			    receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextPAXLine8Bit(HDPAX hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISDestroyReadPAXLines

	Purpose : 
	Destroy the HDPAX object and any memory associated 
	with it.

	Param					Use
	------------------------------------
	hSingle			    	valid HDPAX object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyReadPAXLines(HDPAX hSingle);

/*******************************************************
	ISInitReadTLALines

	Purpose : 
	Set up for single line TLA file reading.
	Use with ISReadNextTLALine
    and ISDestroyReadTLALines.

	Note: Single line input/output functions are not 
	available for unregistered users.

	Param					Use
	------------------------------------
	See ISReadTLA for parameter notes.

	Return
	------
	HDTLA - TLA single line read object
*******************************************************/
extern HDTLA	_ISEXP_ _ISInitReadTLALines(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 uBytesPerPixel, RGBQUAD *pPal, const char *pPassword);

/*******************************************************
	ISReadNextTLALine

	Purpose : 
	Read a single line from a PAX file

	Param					Use
	------------------------------------
	hSingle				    valid HDPAX object
	pRow					receives image data, see 
							ISReadTLA for format notes
	puRowIndex			    receives index of row that was read

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISReadNextTLALine(HDTLA hSingle, BYTE *pRow, UINT32 *puRowIndex);

/*******************************************************
	ISDestroyReadTLALines

	Purpose : 
	Destroy the HDTLA object and any memory associated 
	with it.

	Param					Use
	------------------------------------
	hSingle			    	valid HDTLA object

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDestroyReadTLALines(HDTLA hSingle);

/*******************************************************
	ISGetJPGDims

	Purpose : 
	Find dimensions of the JPG file.

	Param					Use
	------------------------------------
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	puBitDepth			receives image bit depth
	pDPIStruct			receives resolution info
						uDPIUnits is one of:
							0 - no units specified in file
							1 - dots / inch
							2 - dots / cm


	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetJPGDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISGetBMPDims

	Purpose : 
	Find dimensions of the BMP file.

	Param					Use
	------------------------------------
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	puBitDepth				receives image bit depth

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetBMPDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth);

/*******************************************************
	ISGetPCXDims

	Purpose : 
	Find dimensions of the PCX file.

	Param					Use
	------------------------------------
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
	puBitDepth				receives image bit depth 

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetPCXDims(HISSRC hSource,	UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth);

/*******************************************************
	ISGetPNGDims

	Purpose : 
	Find dimensions of the PNG file, along with color depth info/

	Also sets Input DPI fields.

	A PNG file may contain any number key/text pairs.
	When ImgSource reads a PNG file, it stores these strings 
	in the ImgSource PNG text buffer. These strings remain 
	valid until :

		1) the next PNG file is read
		2) ISClearPNGInputText is called

	See ISGetPNGInputTextCount, ISGetPNGInputText,
	ISClearPNGInputText.

	Param					Use
	------------------------------------
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)

	puBitDepth			receives image bits per component.
						this is not the same as bits per pixel!!

	puColorType			receives PNG color type. 
 						0 = gray, 2 = RGB, 3 = colormapped,
						4 = gray-alpha, 6 = RGBA

	puInterlaceType		receives interlace type 
						0 = not interlaced
						1 = Adam7 interlaced

	pDPIStruct			receives resolution info
						uDPIUnits is one of:
							0 - no units specified in file
							1 - dots / meter

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetPNGDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, UINT32 *puColorType, UINT32 *puInterlaceType, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISGetWBMPDims

	Purpose : 
	Find dimensions of the WBMP file.

	Param					Use
	------------------------------------
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetWBMPDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight);

/*******************************************************
	ISGetTGADims

	Purpose : 
	Find dimensions of the TGA file.

	Param					Use
	------------------------------------
	puWidth				receives image width (in pixels)
	puHeight				receives image height (in pixels)
	puBitDepth			receives image bit depth

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetTGADims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth);

/*******************************************************
	ISGetTIFFDims

	Purpose : 
	Find dimensions of the TIFF file.

	Param				Use
	-----------------------------------
	puWidth				receives image width (in pixels)
	puHeight			receives image height (in pixels)
	puBitDepth			receives image bit depth
	pDPIStruct			receives resolution info
						uDPIUnits is one of:
						1 - no units specified in file
						2 - dots / inch
						3 - dots / cm

	Return
	------
	TRUE if successful
*******************************************************/
extern	BOOL	_ISEXP_ _ISGetTIFFDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, ISDPIStruct *pDPIStruct);

/*******************************************************
	ISGetPAXDims

	Purpose : 
	Get dimension and bit depth information about a PAX file

	Param					Use
	------------------------------------
	hSource				    valid source object
	puWidth				    receives image width (in pixels)
	puHeight				receives image height (in pixels)
    puBitDepth              receives image bits per pixel (8, 24)
    pPassword               password for this file. if the correct
                            password is not supplied, this operation
                            will fail.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPAXDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, const char *pPassword);

/*******************************************************
	ISGetPSDDims

	Purpose : 
	Get dimension and color mode information about a PSD 
	(Adobe Photoshop) file

	Param					Use
	------------------------------------
	hSource				    valid source object
	puWidth				    receives image width (in pixels)
	puHeight				receives image height (in pixels)
    puMode					Bitmap = 0
							Grayscale = 1
							Indexed = 2
							RGB = 3
							CMYK = 4
							* MultiChannel = 7
							DuoTone = 8
							* Lab = 9

    * = not supported by ImgSource PSD Read functions

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPSDDims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puMode);

/*******************************************************
	ISGetTLADims

	Purpose : 
	Get dimension and bit depth information about a TLA file

	Param					Use
	------------------------------------
	hSource				    valid source object
	puWidth				    receives image width (in pixels)
	puHeight				receives image height (in pixels)
    puBitDepth              receives image bits per pixel (8, 24)
    pPassword               password for this file. if the file is 
							encrypted and the correct password is 
							not supplied, this operation will fail.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetTLADims(HISSRC hSource, UINT32 *puWidth, UINT32 *puHeight, UINT32 *puBitDepth, const char *pPassword);

/*******************************************************
	ISGetPNGPalette

	Purpose : 
	Get the PNG palette. Only valid for PNGs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPNGPalette(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetBMPPalette

	Purpose : 
	Get the BMP palette. Only valid for BMPs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL		_ISEXP_ _ISGetBMPPalette(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetPCXPalette

	Purpose : 
	Get the PCX palette. Only valid for PCXs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPCXPalette(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetTGAPalette

	Purpose : 
	Get the TGA palette. Only valid for TGAs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL	_ISEXP_ _ISGetTGAPalette(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetPAXPalette

	Purpose : 
	Get the PAX palette. Only valid for PAXs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
    pPassword               password for this file. if the correct
                            password is not supplied, this operation
                            will fail.
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPAXPalette(HISSRC hSource, const char *pPassword, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetPSDPalette

	Purpose : 
	Get the PSD palette. Only valid for PSDs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource					open source manager
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	TRUE if successful
*******************************************************/
extern BOOL	 _ISEXP_ _ISGetPSDPalette(HISSRC hSource, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetTLAPalette

	Purpose : 
	Get the TLA palette. Only valid for TLAs with 8bpp or less.

	Param					Use
	------------------------------------
	hSource				    valid source object
    pPassword               password for this file. if the file is 
							encrypted and the correct password is 
							not supplied, this operation will fail.
	pPalColors				receives number of colors in the palette
	pPal					256 RGBQUADs . receives palette

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetTLAPalette(HISSRC hSource, const char *pKey, UINT32 *pPalColors, RGBQUAD *pPal);

/*******************************************************
	ISGetJPGInputTextCount

	Purpose :
	Get count of text strings read by last JPG file read.
	
	Return
		number of strings 
*******************************************************/
extern UINT32	_ISEXP_ _ISGetJPGInputTextCount();

/*******************************************************
	ISGetJPGInputText

	Purpose :
	Fetch string read by last JPG read.
	
	Param					Use
	------------------------------------
	uIndex				index of string to fetch

	Return
		HGLOBAL pointer to string.
		caller must use GlobalFree to delete this object.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISGetJPGInputText(UINT32 uIndex);

/*******************************************************
	ISClearJPGInputText

	Purpose :
	Clear all JPG input strings
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearJPGInputText();

/*******************************************************
	ISGetJPGInputMarkerCount

	Purpose :
	Get count of JPG special markers read by last JPG file 
	(or dimension) read.
	These are application-specific data chunks.

	Return
		number of markers
*******************************************************/
extern UINT32	_ISEXP_ _ISGetJPGInputMarkerCount();

/*******************************************************
	ISGetJPGInputMarker

	Purpose :
	Fetch JPG marker data read by last JPEG read. 
	
	Param					Use
	------------------------------------
	uIndex				index of marker to fetch
	phData				receives HGLOBAL to data
	puDataLen			receives data length in bytes
	puDataType			marker type value : 
							JPEG_APP1 = 0xE1
							JPEG_APP2 = 0xE2
							...up to 
							JPEG_APP14

   Note : This data is application-specific. 
   ImgSource does not and cannot interpret the data in 
   any way.

	Return
		TRUE on success
		caller must use GlobalFree to delete the phData 
      memory.
   
*******************************************************/
extern BOOL	_ISEXP_ _ISGetJPGInputMarker(UINT32 uIndex, HGLOBAL *phData, UINT32 *puDataLen, UINT32 *puDataType);

/*******************************************************
	ISClearJPGInputMarkers

	Purpose :
	Clear all JPG input marker data
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearJPGInputMarkers();

/*******************************************************
	ISAddJPGOutputText

	Purpose :
	Add a string to be written to all subsequent JPG files
	as a JPEG_COM comment.
	
	Param					Use
	------------------------------------
	pText					zero-terminated string to write

	Return
		FALSE on error
*******************************************************/
extern BOOL	_ISEXP_ _ISAddJPGOutputText(const char *pText);

/*******************************************************
	ISClearJPGOutputText

	Purpose :
	Clear all JPG output strings
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearJPGOutputText();

/*******************************************************
	ISAddJPGOutputMarker

	Purpose :
	Add JPG app marker to be written on next JPG write. 
	
	Param					Use
	------------------------------------
	pData				   data
	uDataLen			   data length in bytes
	uDataType			marker type value : 
							JPEG_APP1 = 0xE1
							JPEG_APP2 = 0xE2
							...up to 
							JPEG_APP14

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISAddJPGOutputMarker(BYTE *pData, UINT32 uDataLen, UINT32 uDataType);

/*******************************************************
	ISClearJPGOutputMarkers

	Purpose :
	Clear all JPG output marker data
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearJPGOutputMarkers();

/*******************************************************
	ISSetJPGDCT

	Switch the compression method : dct is one of :

	0 - default (usually JDCT_ISLOW)
	1 - JDCT_ISLOW: slow but accurate integer algorithm
	2 - JDCT_IFAST: faster, less accurate integer method
	3 - JDCT_FLOAT: floating-point method

	It is not necessary to use this function to be able 
	save JPG files. This is only for people who need more 
	speed or higher quality in their output.

	All JPG files written will be written using this setting.
	
	Param					Use
	------------------------------------
	uDct					level to set
*******************************************************/
extern void	_ISEXP_ _ISSetJPGDCT(UINT32 uDCT);

/*******************************************************
	ISJPGTransform

	Purpose :
	Perform a lossless transformation on the image in a JPG file. This 
	can be used to perform one of a set of simple transformations without 
	decompressing the JPG data. By not decompressing, you don't have to 
	re-compress, thus avoiding adding additional JPG artifacts to the output
	image.

	This works by rearranging the compressed data (DCT coefficients), 
	without ever fully decoding the image.  Therefore, its transformations 
	are lossless: there is no image degradation at all, which is not the case
	if you were to read the image to RGB, transform, and save again as JPG.

	The transpose transformation has no restrictions regarding image 
	dimensions. The other transformations operate rather oddly if the 
	image dimensions are not a multiple of the iMCU size (usually 8 or 16 
	pixels, though sometimes 32 or 48), because they can only transform 
	complete blocks of DCT coefficient data in the desired way.

	The default behavior is to discard these "extra" blocks. This prevents 
	ugly strips of unmodified data along one edge. So, strictly speaking, these
	transformations are not lossless for all images: they are lossless only for 
	images with dimensions that are multiples of 8 or 16. You can override this
	discarding of extra blocks, though it is not recommended.

	Another option is to discard the chrominance channels in standard 
	(YCbCr encoded) JPEGs. This leaves just the luminance which gives you
	a grayscale JPEG.

	All JPEG markers (JPEG_APPx and JPEG_COM) are copied from source to 
	destination.

	Base values for uFlags :
	0		no transform
	1		horizontal flip
	2		vertical flip
	3		transverse (transpose across UR-to-LL axis)
	4		transpose (transpose across UL-to-LR axis)
	5		rotate 90 deg (CW)
	6		rotate 180 deg
	7		rotate 270 deg (CW)

	extra flags. Logical "OR" these with the base values:
	bit		value	purpose
	---------------------
	8		0x0100	grayscale output
	9		0x0200	do not trim extra MCU blocks
	10		0x0400	progressive output

	Param					Use
	------------------------------------
	hSource					source JPEG file
	hDest					destination JPEG file
	uFlags					operations to perform:

	Return
	------
	FALSE on failure

*******************************************************/
extern BOOL _ISEXP_ _ISJPGTransform(HISSRC hSource, HISDEST hDest, UINT32 uFlags);

/*******************************************************
	ISPNGSetScreenGamma

	Purpose :
	Set the screen gamma value for PNG reads and writes

	Param					Use
	------------------------------------
	fScreenGamma		value to set as gamma. 

*******************************************************/
extern void	_ISEXP_ _ISPNGSetScreenGamma(double fScreenGamma);

/*******************************************************
	ISAllowPNGGammaCorrectInput

	Purpose :
	En/disable PNG gamma correction on file input.
	Gamma correction is enabled by default.

	Param					Use
	------------------------------------
	bAllow					TRUE : Allow gamma correction

*******************************************************/
extern void	_ISEXP_ _ISAllowPNGGammaCorrectInput(BOOL bAllow);

/*******************************************************
	ISPNGSetAllowAlphaPalette

	Purpose :
   Instruct the PNG writer to write alpha information 
   for 8-bit images.

	If TRUE, the PNG writer will use the rgbReserved 
   entries of the input palette as alpha values 
   (0 = transparent, 255 = opaque).

   Default value is FALSE

	Param					Use
	------------------------------------
	bAllowAlpha		   TRUE = allow, FALSE = disallow

*******************************************************/
extern void	_ISEXP_ _ISPNGSetAllowAlphaPalette(BOOL bAllowAlpha);

/*******************************************************
	ISPNGSetDefBackground

	Purpose :
		Set the background colors to be used in blending the
	alpha channel from PNG images that have one. This is used
	for both reading and writing of PNG images - if the input image
	uses an alpha channel but has no background specified, this
	value is used. When writing a file, this value is written
	as the default background.


	Param					Use
	------------------------------------
	uIndex				palette index to use for background
							(if you know the incoming image has a 
							certain palette)

	uRed					red component of RGB background value
	uGreen				green component of RGB background value
	uBlue					blue component of RGB background value
	uGray					gray value to be used for grayscale images

	Return
		none
*******************************************************/
extern void	_ISEXP_ _ISPNGSetDefBackground(UINT32 uIndex, UINT32 uRed, UINT32 uGreen, UINT32 uBlue, UINT32 uGray);

/*******************************************************
	ISGetPNGInputTextCount

	Purpose :
	Get count of text strings read by last PNG file read.
	
	Return
		number of strings 
*******************************************************/
extern UINT32	_ISEXP_ _ISGetPNGInputTextCount();

/*******************************************************
	ISGetPNGInputText

	Purpose :
	Fetch string read by last PNG read.
	
	Param					Use
	------------------------------------
	uIndex				index of string to fetch
	hKey					receives handle to key text
	hText					receives handle to string text
	puCompressed		receives compression flag

	Caller must use GlobalFree to delete key and string!

	Return
		FALSE on error
		
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPNGInputText(UINT32 uIndex, HGLOBAL *pKey, HGLOBAL *pText, UINT32 *puCompression);

/*******************************************************
	ISClearPNGInputText

	Purpose :
	Clear all PNG input strings
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearPNGInputText();

/*******************************************************
	ISGetPNGOutputTextCount

	Purpose :
	Get count of text strings that have been added with
	ISAddPNGOutputText
	
	Return
		number of strings 
*******************************************************/
extern UINT32	_ISEXP_ _ISGetPNGOutputTextCount();

/*******************************************************
	ISAddPNGOutputText

	Purpose :
	Add a text field to the ImgLib global PNG text buffer.
	PNG text fields consist of a mandatory 1-79 character key and 
	an optional text buffer of any size. The text can be written 
	compressed or uncompressed. It is not recommended that you 
	compress text fields of length < 1000 bytes.

	Keys should be plain ASCII, no control or non-printable chars.

	You may add as many fields as you wish. 

	The keywords that are given in the PNG Specification are:

      Title            Short (one line) title or caption for image
      Author           Name of image's creator
      Description      Description of image (possibly long)
      Copyright        Copyright notice
      Creation Time    Time of original image creation
      Software         Software used to create the image
      Disclaimer       Legal disclaimer
      Warning          Warning of nature of content
      Source           Device used to create the image
      Comment          Miscellaneous comment; conversion from other
                       image format

	Param					Use
	------------------------------------
	pText					zero-terminated string to write
	pKey					zero-terminated string to use as the key
	bCompress			1 to compress (for strings > 1K)

	Return
		FALSE on error
*******************************************************/
extern BOOL	_ISEXP_ _ISAddPNGOutputText(const char *pText, const char *pKey, UINT32 bCompress);

/*******************************************************
	ISGetPNGOutputText

	Purpose :
	Fetch string previously added with ISAddPNGOutputText
	
	Param					Use
	------------------------------------
	uIndex				index of string to fetch
	hKey					receives handle to key text
	hText					receives handle to string text
	puCompressed		receives compression flag

	Caller must use GlobalFree to delete key and string!

	Return
		FALSE on error
		
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPNGOutputText(UINT32 uIndex, HGLOBAL *pKey, HGLOBAL *pText, UINT32 *puCompression);

/*******************************************************
	ISClearPNGOutputText

	Purpose :
	Clear all PNG output strings
	
	Return
		TRUE if ok
*******************************************************/
extern BOOL	_ISEXP_ _ISClearPNGOutputText();

/*******************************************************
	ISSetTIFFCompressionMode

	Purpose :
	Set compression mode settings for all subsequent TIFF 
	file write operations.
	
	Param					Use
	------------------------------------
	uCompMode				one of :
                     
							1 : no compression (default)

							7 : JPEG (only for 24-bit writes!)
							uses "no compresion" if set for 8-bit writes

							32773 :  Macintosh PackBits

							32946 :  Deflate (using ZLib. same as PNG, etc..)

   Note : If you try to write LZW-compressed TIFFs (Comp 
   Mode 5), ImgSource will switch to mode 1, no compression. 
   This is to comply with the Unisys patent on LZW compression.

*******************************************************/
extern void	_ISEXP_ _ISSetTIFFCompressionMode(UINT32 uCompMode);

/*******************************************************
	ISIsTIFFFAX

	Determine if a given file is a CCITT TIFF FAX file

	Param					Use
	------------------------------------
	hSource					source manager
	uFrameIdx				frame index of the TIFF (0 for first)
	puCompType				receives compression type
							1: CCITT G3
							2: CCITT G4
							3: CCITT RLE
							4: CCITT RLEW

	returns FALSE if not a TIFF FAX file
*******************************************************/
extern BOOL	_ISEXP_ _ISIsTIFFFAX(HISSRC hSource, UINT32 uFrameIdx, UINT32 *puCompType);

/*******************************************************
	ISGetPAXID

	Purpose :
	Get the ID for a PAX file.
	
	Param					Use
	------------------------------------
	hSource					source manager
	pID                     PAX file ID (128 BYTEs)

	Return
	------
	FALSE on failure

*******************************************************/
extern BOOL	_ISEXP_ _ISGetPAXID(HISSRC hSource, BYTE *pID);

/*******************************************************
	ISGetPAXInputTagCount

	Purpose :
	Get count of PAX data tags read by last PAX file 
    (or dimension) read.
	
	Return
	------
	tag count

*******************************************************/
extern UINT32	_ISEXP_ _ISGetPAXInputTagCount();

/*******************************************************
	ISGetPAXInputTag

	Purpose :
	Retrieve a PAX data tag as read by the last PAX file
    read or dimension read.

	Param					Use
	------------------------------------
    uIndex                  index of tag to retrieve
    pName                   receives tag name. this will always
                            be 4 characters plus a trailing NULL BYTE.
    phData                  receives HGLOBAL to tag data
    puDataLen               receives length of tag data

	Return
	------
	FALSE on failure

    caller must use GlobalFree to free the tag data memory

*******************************************************/
extern BOOL	_ISEXP_ _ISGetPAXInputTag(UINT32 uIndex, BYTE *pName, HGLOBAL *phData, UINT32 *puDataLen);
											  
/*******************************************************
	ISAddPAXOutputTag

	Purpose :
	Add a PAX data tag to be written to the next PAX file.

	Param					Use
	------------------------------------
    pName                   tag name. 4 characters, 
                            plus a zero byte
    pData                   tag data
    DataLen                 length of tag data

	Return
	------
	FALSE on failure

*******************************************************/
extern BOOL	_ISEXP_ _ISAddPAXOutputTag(const char *pName, const BYTE *pData, UINT32 uDataLen);

/*******************************************************
	ISClearPAXOutputTags

	Purpose :
	Clear all PAX output tags. 

*******************************************************/
extern void	_ISEXP_ _ISClearPAXOutputTags();
										  // 32
/*******************************************************
	ISSetPAXOutputInfo

	Purpose :
	Set output application, author and PAX file ID
    for the next PAX file.

	Param					Use
	------------------------------------
    pAppName                name of application (max 32 chars)
    pAuthor                 author name (max 32 chars)
    pID                     PAX file ID (128 BYTEs)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSetPAXOutputInfo(const char *pAppName, const char *pAuthor, BYTE *pID);

/*******************************************************
	ISGetPAXInputInfo

	Purpose :
	Get application, author and PAX file ID
    from the last read PAX file. 
    
    This data is set on PAX file read or dimension read.

	Param					Use
	------------------------------------
    pAppName                name of application (max 32 chars)
    pAuthor                 author name (max 32 chars)
    pID                     PAX file ID (128 BYTEs)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetPAXInputInfo(char *pAppName, char *pAuthor, BYTE *pID);

/*******************************************************
	ISMakePAXID

	Purpose :
	Generate a new PAX ID.

    Each PAX file should have a unique PAX ID. You can
    use this function, with two seed strings of your 
    choice, to generate PAX IDs.

    It is recommended that you change at least one of 
    these seed strings for _every_ PAX file your application
    creates.

    Some good choices for seed strings are:
        ** Windows user name:
        HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\RegisteredOwner

        ** output file name

        ** current time as a string

	Param					Use
	------------------------------------
    pSeedString1            seed strings
    pSeedString2            
    pOutID                  receives PAX file ID 
                            (128 BYTEs, allocated by the caller)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISMakePAXID(const char *pSeedString1, const char *pSeedString2, BYTE *pOutID);

/*******************************************************
	ISGetTLAInputTagCount

	Purpose :
	Get count of TLA data tags read by last PAX file 
    (or dimension) read.
	
	Return
	------
	tag count

*******************************************************/
extern UINT32	_ISEXP_ _ISGetTLAInputTagCount();

/*******************************************************
	ISGetTLAInputTag

	Purpose :
	Retrieve a TLA data tag as read by the last TLA file
    read or dimension read.

	Param					Use
	------------------------------------
    uIndex                  index of tag to retrieve
    phData                  receives HGLOBAL to tag data
    puDataLen               receives length of tag data

	Return
	------
	FALSE on failure

    caller must use GlobalFree to free the tag data memory

*******************************************************/
extern BOOL	_ISEXP_ _ISGetTLAInputTag(UINT32 uIndex, HGLOBAL *phData, UINT32 *puDataLen);

/*******************************************************
	ISDrawHBITMAP

	Purpose :
	Draw an HBITMAP to a device context

	The device referred to by hDC must support 
	the following RASTERCAPS values : 
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
	Param					Use
	------------------------------------
	hDC					output device context
	hBmp					HBITMAP to draw
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth			X pixels to draw (may be less than 
							image width)

	uOutHeight			Y pixels to draw (may be less than 
							image width)

	hPal					optional palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawHBITMAP(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISDrawHBITMAPCropped

	Purpose :
	Draw an HBITMAP to a device context with cropping 

	The device referred to by hDC must support 
	the following RASTERCAPS values : 
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

  
	Param					Use
	------------------------------------
	hDC						output device context
	hBmp					HBITMAP to draw
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uSrcXStart				start drawing from this X position 
							in the source

	uSrcYStart				start drawing from this Y position 
							in the source

	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	hPal					optional palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawHBITMAPCropped(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth,	UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISDrawRGB

	Purpose :
	Draw an RGB image to a device context

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.

	Param					Use
	------------------------------------
	hDC						output device context
	pRGB					RGB buffer to draw
	uWidth					image width
	uHeight					image height
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	hPal					optional palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawRGB(HDC hDC, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISDrawRGBCrop

	Purpose :
	Draw an RGB image to a device context with cropping

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.
  
	Param					Use
	------------------------------------
	hDC						output device context
	pRGB					RGB buffer to draw
	uWidth					image width
	uHeight					image height
	uSrcXStart				start drawing from this X position 
							in the source

	uSrcYStart				start drawing from this Y position 
							in the source

	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	hPal					optional palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawRGBCrop(HDC hDC, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISStretchDrawRGB

	Purpose :
	Draw an RGB image to a device context using
	StretchDIBits. This will stretch the image if the
	output dimensions do not match the image dimensions.
	
   Note that the stretching results are usually less than
   satisfactory. This is because the StretchDIBits function
   is really not very good at stretching. It is recommended
   that you use a function like ISResizeRGB to do any image 
   resizing.
   
   This function is 10-20% faster than ISDrawRGB.

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_STRETCHDIB

	Param					Use
	------------------------------------
	hDC						output device context
	pRGB					RGB buffer to draw
	uWidth					image width
	uHeight					image height
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth				X pixels to draw
	uOutHeight				Y pixels to draw

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISStretchDrawRGB(HDC hDC, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISStretchDrawRGBCropped

	Purpose :
	Draw an RGB image to a device context using
	StretchDIBits. This does NOT stretch the image.

   Note that the stretching results are usually less than
   satisfactory. This is because the StretchDIBits function
   is really not very good at stretching. It is recommended
   that you use a function like ISResizeRGB to do any image 
   resizing.
   
   This function is 10-20% faster than ISDrawRGB.
   
   The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_STRETCHDIB
						
	Param					Use
	------------------------------------
	hDC						output device context
	pRGB					RGB buffer to draw
	uWidth					image width
	uHeight					image height
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uSrcXStart				start drawing from this X position 
							in the source

	uSrcYStart				start drawing from this Y position 
							in the source

	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISStretchDrawRGBCropped(HDC hDC, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISDraw8Bit

	Purpose :
	Draw an 8-bit image to a device context

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  


   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.
  
	Param					Use
	------------------------------------
	hDC						output device context
	p8Bit					8-bit buffer to draw
	uWidth					image width
	uHeight					image height
	pPal					image palette
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	hPal					optional screen palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDraw8Bit(HDC hDC, BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISDraw8BitCrop

	Purpose :
	Draw an 8 bit image to a device context with cropping

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  

   When drawing on a 256 color display, the palette is 
   used in the creation of a temporary HBITMAP. This 
   will give better color accuracy than is possible with 
   the default Windows palette.
  
	Param					Use
	------------------------------------
	hDC						output device context
	p8Bit					8-bit buffer to draw
	uWidth					image width
	uHeight					image height
	pPal					image palette
	uSrcXStart				start drawing from this X position 
							in the source

	uSrcYStart				start drawing from this Y position 
							in the source

	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	hPal					optional palette

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDraw8BitCrop(HDC hDC, BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth, UINT32 uOutHeight, HPALETTE hPal);

/*******************************************************
	ISStretchDraw8Bit

	Purpose :
	Draw an 8 bit image to a device context using
	StretchDIBits. This will stretch the image if the
	output dimensions do not match the image dimensions.

   Note that the stretching results are usually less than
   satisfactory. This is because the StretchDIBits function
   is really not very good at stretching. It is recommended
   that you use a function like ISResizeRGB to do any image 
   resizing.
   
   This function is 10-20% faster than ISDrawRGB.
   
   The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_STRETCHDIB
  
	Param					Use
	------------------------------------
	hDC						output device context
	p8Bit					8-bit buffer to draw
	uWidth					image width
	uHeight					image height
	pPal					image palette
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth				X pixels to draw
	uOutHeight				Y pixels to draw

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISStretchDraw8Bit(HDC hDC, BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISStretchDraw8BitCropped

	Purpose :
	Draw an 8 bit image to a device context using
	StretchDIBits. This will NOT stretch the image. The
   'Stretch' refers to the StretchDIBits function.

   Note that the stretching results are usually less than
   satisfactory. This is because the StretchDIBits function
   is really not very good at stretching. It is recommended
   that you use a function like ISResizeRGB to do any image 
   resizing.
   
   This function is 10-20% faster than ISDrawRGB.
	
   The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_STRETCHDIB
  
	Param					Use
	------------------------------------
	hDC						output device context
	p8Bit					8-bit buffer to draw
	uWidth					image width
	uHeight					image height
	pPal					image palette
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uSrcXStart				start drawing from this X position 
							in the source

	uSrcYStart				start drawing from this Y position 
							in the source

	uOutWidth				X pixels to draw (may be less than 
							image width)

	uOutHeight				Y pixels to draw (may be less than 
							image width)

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISStretchDraw8BitCropped(HDC hDC, BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, UINT32 uSrcXStart, UINT32 uSrcYStart, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISDrawTransparentBitmap

	Purpose :
	Draw an HBITMAP to a device context with one color
	transparent

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
	Param					Use
	------------------------------------
	hDC						output device context
	hBmp					HBITMAP to draw
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	clrTransparent			transparent color

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawTransparentBitmap(HDC hDC, HBITMAP hBmp, __int32 iXPos, __int32 iYPos, COLORREF clrTransparent);

/*******************************************************
	ISDrawTransparentRGB

	Purpose :
	Draw an RGB image to a device context with one color
	transparent

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
		RC_BITBLT

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
	Param					Use
	------------------------------------
	hDC						output device context
	pRGB					RGB image
	uWidth					image width
	uHeight					image height
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	clrTransparent			transparent color

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDrawTransparentRGB(HDC hDC, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, __int32 iXPos, __int32 iYPos, COLORREF clrTransparent);

/*******************************************************
	ISStretchDrawDIB

	Purpose :
	Draw a DIB image to a device context. This will stretch the
   image if the uOutWidht or uOutHeight params do not match
   the width and height of the image in the DIB.


	The device referred to by hDC must support the 
	following RASTERCAPS values : 
      RC_STRETCHDIB  

	Param					Use
	------------------------------------
	hDC						output device context
	pDIB					DIB image
	iXPos					X position (may be negative)
	iYPos					Y position (may be negative)
	uOutWidth				output width
	uOutHeight				output height

	Return
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISStretchDrawDIB(HDC hDC, BYTE *pDIB, __int32 iXPos, __int32 iYPos, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISLoadResourceBitmap

	Purpose :
	Load a bitmap from a resource, using a caller-supplied
	palette. This generally gives better results than
	CBitmap::LoadBitmap.
	
	Param					Use
	------------------------------------
	hInstance				calling app's hInstance
	pName					resource name
	hInPalette				palette to use, NULL if none

	Return
		HBITMAP object.
		Caller must call DestroyObject to release this memory
*******************************************************/
extern HBITMAP	_ISEXP_ _ISLoadResourceBitmap(HINSTANCE hInstance, const char *pName, HPALETTE hInPalette);

/*******************************************************
	ISHBITMAPToRGB

	Purpose :
	Convert an HBITMAP to an RGB buffer

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
  
	Param					Use
	------------------------------------
	hBmp					input HBITMAP
	puWidth					receives image width
	puHeight				receives image height
	hDC						device context to use
	hPal					optional palette

	Return
		RGB image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISHBITMAPToRGB(HBITMAP hBmp, UINT32 *puWidth, UINT32 *puHeight, HDC hDC, HPALETTE hPal);

/*******************************************************
	ISHBITMAPToRGBA

	Purpose :
	Convert an HBITMAP to an RGBA buffer

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
  
	Param					Use
	------------------------------------
	hBmp					input HBITMAP
	puWidth					receives image width
	puHeight				receives image height
	hDC						device context to use
	hPal					optional palette
	uDefAlpha				value to use if the source bitmap is not 32 bits

	Return
		RGBA image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISHBITMAPToRGBA(HBITMAP hBmp, UINT32 *uWidth, UINT32 *uHeight, HDC hDC, HPALETTE hPal, UINT32 uDefAlpha);


/*******************************************************
	ISDCToRGB

	Purpose :
	Capture a device context to an RGB buffer

	The device referred to by hDC must support 
	the following RASTERCAPS values : 
		RC_BITBLT
		RC_DI_BITMAP

   If your device does not support RC_BITMAP64,
   this operation may fail with large images.  
  
	Param				Use
	------------------------------------
	hDC					device context to use
	uXStart				image rectangle left
	uYStart				image rectangle top
	uWidth				image rectangle width
	uHeight				image rectangle height
	
	Return
		RGB image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISDCToRGB(HDC hDC, UINT32 uXStart, UINT32 uYStart, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISReadMetafileHandleToRGB

	Purpose : 
	Given a metafile handle, generate an RGB image.

	Param					Use
	------------------------------------
	hEmf					metafile handle
	puWidth					receives image width (in pixels)
	puHeight				receives image height (in pixels)
    uDefWidth				default width for non-sized files
    uDefHeight				default height for non-sized files
    clrBackground			background color to render the image onto

	Return
	------
	HGLOBAL handle to RGB image. Caller must use GlobalFree 
	to release this memory back to the OS.
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISReadMetafileHandleToRGB(HENHMETAFILE hEmf, UINT32 *puWidth, UINT32 *puHeight, UINT32 uDefWidth, UINT32 uDefHeight, COLORREF clrBackground);

/*******************************************************
	ISRGBToHBITMAP

	Purpose :
	Create an HBITMAP based on an RGB buffer.

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
  
	Param					Use
	------------------------------------
	pRGB					input RGB image
	uWidth					image width
	uHeight					image height
	hPal					optional palette
	hDC						device context to use

	Return
		HBITMAP object.
		Caller must call DestroyObject to release this memory
*******************************************************/
extern HBITMAP _ISEXP_ _ISRGBToHBITMAP(BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, HPALETTE hPal, HDC hDC);

/*******************************************************
	ISDIBToHBITMAP

	Purpose :
	Create an HBITMAP based on a DIB.

	The device referred to by hDC must support the 
	following RASTERCAPS values : 
		RC_DI_BITMAP
  
	Param					Use
	------------------------------------
	pDIB					input DIB
	hPal					optional palette
	hDC						device context to use

	Return
		HBITMAP object.
		Caller must call DestroyObject to release this memory
*******************************************************/
extern HBITMAP	_ISEXP_ _ISDIBToHBITMAP(BYTE * pDIB, HPALETTE hPal, HDC hDC);

/*******************************************************
	ISHBITMAPToDIB

	Purpose :
	Create a 24-bit DIB from an HBITMAP
  
	Param					Use
	------------------------------------
   HBITMAP					input HBITMAP
	hPal					optional palette

	Return
      DIB
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISHBITMAPToDIB(HBITMAP hBmp, HPALETTE hPal);

/*******************************************************
	ISRGBQUADToHPALETTE

	Purpose :
	Create an HPALETTE based on an array of RGBQUADs
	
	Param					Use
	------------------------------------
	pPal					array of RGBQUADs
   uColors					number of entries in pPal
	
	Return
		DIB
		Caller must call DeleteObject to release this memory
*******************************************************/
extern HPALETTE _ISEXP_ _ISRGBQUADToHPALETTE(RGBQUAD *pPal, UINT32 uColors);

/*******************************************************
	ISDIBToRGB

	Purpose :
	Convert a DIB to an RGB buffer. 
	
	Param					Use
	------------------------------------
	pDIB					pointer to a DIB
	puWidth					receives image width
	puHeight				receives image height

	Return
		RGB image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISDIBToRGB(BYTE * pDIB, UINT32 *uWidth, UINT32 *uHeight);

/*******************************************************
	ISDIBPartsToRGB

	Purpose :
	Convert a DIB to an RGB buffer. 
	
	Param					Use
	------------------------------------
	pDIBBits				pointer to the DIB image bits
	pBMIH					pointer to the BITMAPINFOHEADER
	pPalette				pointer to the palette data
	puWidth					receives image width
	puHeight				receives image height

	Return
		RGB image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISDIBPartsToRGB(BYTE *pDIBBits, BITMAPINFOHEADER * pBMIH, RGBQUAD *pPalette, UINT32 *uWidth, UINT32 *uHeight);

/*******************************************************
	ISDIBToColormapped

	Purpose :
	Convert a DIB to a colormapped buffer. 
	
	Param					Use
	------------------------------------
	pDIB					pointer to a DIB
	uDesiredBitDepth		bit depth to return 
							(currently only 1 and 8 are supported)
	puWidth					receives image width
	puHeight				receives image height
	puColorsInPal			receives number of colors in palette
	pPal					receives palette (should be 256 RGBQUADs)

	Return
		image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISDIBToColormapped(BYTE * pDIB, UINT32 uDesiredBitDepth, UINT32 *uWidth, UINT32 *uHeight, UINT32 *uColorsInPal, RGBQUAD *pPal);

/*******************************************************
	ISRGBToDIB

	Purpose :
	Create a 24-bit DIB from an RGB buffer
	
	Param					Use
	------------------------------------
	pRGB					input RGB image
	uWidth					image width
	uHeight					image height
	
	Return
		DIB
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _ISRGBToDIB(BYTE * pRGB, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	IS8BitToDIB

	Purpose :
	Create an 8-bit DIB from an 8-bit buffer and palette
	
	Param					Use
	------------------------------------
	p8Bit					input RGB image
	uWidth					image width
	uHeight					image height
	pPal					image palette (256 colors)
	
	Return
		DIB
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _IS8BitToDIB(BYTE * p8Bit, UINT32 uWidth, UINT32 uHeight, RGBQUAD *pPal);

/*******************************************************
	IS8BitToDIB2

	Purpose :
	Create a DIB from an 8-bit buffer and palette
	
	If you specify 1 or 4 for uBpP, the DIB will be a
	1 or 4 (respectively) bit-per-pixel DIB.

	Param					Use
	------------------------------------
	p8Bit					input 8-bit image
	uWidth					image width
	uHeight					image height
	uColors					colors in pPal
	uBPP					1,4 or 8 - bits per pixel for output DIB
	pPal					image palette (256 colors)
	
	Return
		DIB
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _IS8BitToDIB2(BYTE * p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uColors, UINT32 uBPP, RGBQUAD *pPal);

/*******************************************************
	IS1BitToDIB

	Purpose :
	Create a DIB from a 1-bit buffer and palette. Input
	rows must start on whole BYTE boundaries
	

	Param					Use
	------------------------------------
	p1Bit					input 1-bit image
	uWidth					image width
	uHeight					image height
	pPal					image palette (2 colors)
	
	Return
		DIB
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL	_ISEXP_ _IS1BitToDIB(BYTE * p1Bit, UINT32 uWidthPix, UINT32 uHeight, RGBQUAD *pPal);

/*******************************************************
	ISSetHBITMAPCreateMethod

	Purpose :
	Set the method used to create HBITMAPs.

   This affects all internal and external HBITMAP creation 
   operations.
	
	Param					Use
	------------------------------------
	uMethod					0 = use CreateDIBitmap
							1 = use CreateDIBSection

   CreateDIBitmap is the standard method of creating an
   HBITMAP. 

   CreateDIBSection is much faster for large images, but
   the HBITMAPs created are subject to the following
   restriction and/or bugs by Windows:

   1) "Dithered brushes do not work in device contexts that 
   have DIB sections selected into them. If you create an 
   HBITMAP using CreateDIBSection(), select it into a memory 
   DC, and then attempt to use a dithered brush returned by
   CreateSolidBrush() to draw into the DIB section, the 
   brush will not be drawn with a dithered color. ... This
   behavior is by design." Microsoft Win32 Software 
   Development Kit (SDK) versions 3.5, 3.51, 4.0

   2) "When using BitBlt()with DIBSections to create a 
   monochrome mask, the target pixel color is chosen without 
   regard for the background color in the target device 
   context. The AND mask that results is not what the 
   programmer intended, and subsequent transparent blts using 
   that mask do not maintain proper transparency...This
   behavior is by design." Microsoft Windows NT versions 3.5, 
   3.51 Microsoft Windows 95

   3) "The program displays the wrong colors or causes a 
   general protection (GP) fault when using a DIB section and 
   a custom palette in a memory DC... There is a problem in 
   Windows 95 where an internal palette structure is not set up 
   until the palette is selected into a screen DC. "
  
   These are quotes from the Microsoft Developers Network 
   (MSDN).

	Return
		none
*******************************************************/
extern void	_ISEXP_ _ISSetHBITMAPCreateMethod(UINT32 uMethod);

/*******************************************************
	ISVertFlipBuf

	Purpose : 
	Vertically flip a buffer. This can be used to vertically
	flip any buffer, not just images.

	Param					Use
	------------------------------------
	pBuffer 			buffer to flip
	uWidthBytes 		bytes in a buffer row (not pixels!!!!)
	uHeight 			buffer rows

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISVertFlipBuf(BYTE * pBuffer, UINT32 uWidthBytes, UINT32 uHeight);

/*******************************************************
	ISRGBToGrayScale24

	Purpose : 
	Convert an RGB image to RGB grayscale. Sets all channels
	in a pixel to the luminosity value for that pixel.

	Param					Use
	------------------------------------
	pRGB					image
	uWidth				width in pixels
	uHeight 			height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISRGBToGrayScale24(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISRGBToGrayScale8Bit

	Purpose : 
	Convert an RGB image to 8 bit grayscale. Sets the output
	pixel to the luminosity value for the input pixel.

	Param					Use
	------------------------------------
	pRGB					image
	uWidth				width in pixels
	uHeight 			height
	p8Bit					output buffer

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISRGBToGrayScale8Bit(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *p8Bit);

/*******************************************************
	ISRGBToBGR

	Purpose : 
	Swap red and blue channels in an image

	Param					Use
	------------------------------------
	pRGB					image
	uWidth				width in pixels
	uHeight 			height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISRGBToBGR(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISResizeImage

	Purpose : 
	Resize an image using bi-linear interpolation. This
	should only be used on continuous-tone images (RGBA,
	grayscale, RGB, etc.) 
	
	DO NOT use this on images that require a palette 
	unless the palette is arranged so that the visual 
	distance between two colors is directly proportional 
	to the distance between their	color indecies, for all
	combinations of colors in the palette.

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image
	uOutWidth			output width
	uOutHeight			output height
	uBytesPerPixel		bytes in a pixel (3 for RGB, 1 for grayscale, etc)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISResizeImage(BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uOutWidth, UINT32 uOutHeight, UINT32 uBytesPerPixel);

/*******************************************************
	ISResizeImageBicubic

	Purpose : 
	Resize an image using bi-cubic interpolation.

	This is much slower than bi-linear interpolation (ISResizeImage),
	but the results are slightly "better". Your milage may vary.

	This should only be used on continuous-tone images (RGBA,
	grayscale, RGB, etc.) 

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height
	pRGBOut 				destination image
	uOutWidth				output width
	uOutHeight				output height			3
	uBytesPerPixel			bytes in a pixel (3 for RGB, 1 for grayscale, etc)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL		_ISEXP_ _ISResizeImageBicubic(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uOutWidth, UINT32 uOutHeight, UINT32 uBytesPerPixel);

/*******************************************************
	ISResizeFilterImage

	Purpose : 
	Resize an image using a filter-based resize.

	This should only be used on continuous-tone images (RGBA,
	grayscale, RGB, etc.). Do not use with colormapped images!

	Note: this function is much slower than the other ImgSource resizing
	functions.

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height
	pRGBOut 				destination image
	uOutWidth				output width
	uOutHeight				output height
	uBytesPerPixel			bytes in a pixel (3 for RGB, 1 for grayscale, etc)
	uFilter					one of :
 		0: box 
		1: triangle 
		2: Hamming 
		3: Gaussian 
		4: bell 
		5: B-spline 
		6: cubic 1
		7: cubic 2
		8: Lanczos3 
		9: Mitchell 
		10: sinc 
		11: Hermite 
		12: Hanning 
		13: Catrom

	As always, it's impossible to say that a specific algorithm will 
	be the best for all possible images. But, in general, filters 4 
	through 13 give good results for both enlarging and resizing. 
	Filters 0-3 may work well in some situations, especially for reduction 
	only. But you should test a wide range of possible input images 
	before committing to a single filter type.

	Note:
	Callback is called per column (not per row, as in most other routines)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL _ISEXP_ _ISResizeFilterImage(BYTE *pRGB, UINT32 uSrcWidth, UINT32 uSrcHeight, BYTE *pRGBOut, UINT32 uDestWidth, UINT32 uDestHeight, UINT32 uBytesPerPixel, UINT32 uFilter);

/*******************************************************
	ISResize8Bit

	Purpose : 
	Resize an 8-bit image using bi-linear interpolation
	and closest-color color matching.

	destination image must be uOutWidth * uOutHeight bytes
	
	p8BitSrc != p8BitDest

	Param					Use
	------------------------------------
	p8BitSrc			source image
	uWidth				width in pixels
	uHeight 			height
	p8BitDest 			destination image
	uOutWidth			output width
	uOutHeight			output height
	uColors				number of colors in pPal
	pPal				source palette. destination will use the
						same pallete.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISResize8Bit(BYTE * p8BitSrc, UINT32 uWidth, UINT32 uHeight, BYTE * p8BitDest, UINT32 uOutWidthPix, UINT32 uOutHeight, UINT32 uColors, RGBQUAD *pPal);

/*******************************************************
	ISResize1Bit

	Purpose : 
	Resize an 1-bit image using bi-linear interpolation.

	Destination image must be 
	
	  (int)((uOutWidth + 7) / 8) * uOutHeight

	bytes.

	Pixel rows start on whole byte boundaries.

	p1BitSrc != p1BitDest

	Param					Use
	------------------------------------
	p1BitSrc			source image
	uWidth				width in pixels
	uHeight 			height
	p1BitDest 			destination image
	uOutWidth			output width
	uOutHeight			output height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISResize1Bit(BYTE * p1BitSrc, UINT32 uSrcWidthPix, UINT32 uSrcHeight, BYTE * p1BitDest, UINT32 uDestWidthPix, UINT32 uDestHeight);

/*******************************************************
	ISResizeRGBMasked

	Purpose : 
	Resize an RGB image using bi-linear interpolation. 
	Prevent color dithering for one RGB value.

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image
	uOutWidth			output width
	uOutHeight			output height
	clrMask 			mask color

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISResizeRGBMasked(BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, BYTE * pRGBOut, UINT32 uOutWidth, UINT32 uOutHeight, COLORREF clrMask);

/*******************************************************
	ISSimpleResampleRGB

	Purpose : 
	Resize an RGB image using nearest-pixel resampling

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height
	pRGBOut 				destination image
	uOutWidth				output width
	uOutHeight				output height

	Note, this is identical to ISSimpleResampleImage with
	uBytesPerPixel = 3

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSimpleResampleRGB(BYTE  *pInRGB, UINT32 uInWidth, UINT32 uInHeight, BYTE *pOutRGB, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISSimpleResampleImage

	Purpose : 
	Resize an image using nearest-pixel resampling

	This will work with colormapped images.

	pInImage != pOutImage

	Param					Use
	------------------------------------
	pInImage				source image
	uWidth					width in pixels
	uHeight 				height
	uBytesPerPixel			number of bytes in a pixel
	pOutImage 				destination image
	uOutWidth				output width
	uOutHeight				output height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSimpleResampleImage(BYTE  *pInImage, UINT32 uInWidth, UINT32 uInHeight, UINT32 uBytesPerPixel, BYTE *pOutImage, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISSharpenRGB

	Purpose : 
	Sharpen an RGB image.

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image
	uLevel				1..100 sharpening level

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSharpenRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uLevel);

/*******************************************************
	ISBlurRGB

	Purpose : 
	Blur an RGB image using a 3 x 3 matrix

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image
	uLevel				1..100 blur level

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISBlurRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uLevel);

/*******************************************************
	ISMedianCutRGB

	Purpose : 
	Perform median cut on an RGB image using a 3 x 3 matrix

	pRGB != pRGBOut

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISMedianCutRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut);

/*******************************************************
	ISApplyMatrixToRGB

	Purpose : 
	Apply a convolution matrix to an RGB image.

	pRGB != pRGBOut

	The matrix is an array of _9_ doubles, arranged as :

			0  1  2
			3  4  5
			6  7  8

		where "4" is the source pixel.

	It is applied for pixel(x,y) as :

	Step 1
	double tempSum = (
			pMatrix[0] * pixel(x-1, y-1) +
			pMatrix[1] * pixel(x,	y-1) +
			pMatrix[2] * pixel(x+1, y-1) +
			pMatrix[3] * pixel(x-1, y )  +
			pMatrix[4] * pixel(x,	y )  +
			pMatrix[5] * pixel(x+1, y )  +
			pMatrix[6] * pixel(x-1, y+1) +
			pMatrix[7] * pixel(x,	y+1) +
			pMatrix[8] * pixel(x+1, y+1));

	Step 2
			tempSum = tempSum * k;

	Step 3
			tempSum = min(tempSum, 255);

	Step 4
			tempSum = max(tempSum, 0);

	Step 5
			outval = (BYTE) tempSum;


	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			destination image

	uChannelMask		select the channels to apply the matrix 
							to. use CHRED , CHGREEN and CHBLUE OR'd
							together.

	uZeroMasked 		if 1, the input channels excluded by the 
							uChannelMask will have all values set to zero.
							else, the input channels excluded by the 
							channelMask will retain the values from the 
							source image.

	uDoFeedback 		if the bDoFeedback options is 1, 
							Step 2 becomes :
							tempSum = pixel(x,y) * (1.0 - k) + 
							(k * tempSum);

	k						value used in uDoFeedback calc.

	pMatrix 			array of doubles

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISApplyMatrixToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uChannelMask, UINT32 uZeroMasked, UINT32 uDoFeedback, double k, double *pMatrix);

/*******************************************************
	ISApplyLUTToRGB

	Purpose : 
	Apply a LUT (look-up-table) to pixels in an RGB image.

	For each channel in each pixel the following function is
	performed :
		
		v = Val[pixel][channel]
		v = LUT[v]

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	uChannelMask		select the channels to apply the LUT 
							to. use CHRED , CHGREEN and CHBLUE OR'd
							together.
	pLUT					array of 255 BYTE values

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISApplyLUTToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelMask, BYTE *pLUT);

/*******************************************************
	ISApplyGammaToRGB

	Purpose : 
	Perform gamma correction on an RGB image.

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	fGamma			 gamma correction factor
					 
					 gamma must be >= 0.0
					 gamma < 1 , decrease brightness
					 gamma = 1 , no change
					 gamma > 1 , increase brightness

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISApplyGammaToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, double fGamma);

/*******************************************************
	IS8BitToRGB

	Purpose : 
	Make an RGB image from an 8 bit image and a palette

	Param					Use
	------------------------------------
	p8Bit					source image
	pRGB					detination image
	uWidth				width in pixels
	uHeight 			height
	uColors 			number of colors in image
	pPal					palette

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _IS8BitToRGB(BYTE *p8Bit, BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uColors, RGBQUAD *pPal);

/*******************************************************
	ISHistogramEqualizeRGB

	Purpose : 
	Perform histogram equalization on an RGB image

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth				width in pixels
	uHeight 			height
	uLoThresh			low threshold
	uHiThresh			high threshold
	uChannelMask		select the channels to apply this operation 
							to. use CHRED , CHGREEN and CHBLUE OR'd
							together.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISHistogramEqualizeRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uLoThresh, UINT32 uHiThresh, UINT32 uChannelMask);

/*******************************************************
	ISHistogramEqualizeGrayscale

	Purpose : 
	Perform histogram equalization on an 8 bit grayscale image

	Param					Use
	------------------------------------
	pRGB				8-bit grayscale image
	uWidth				width in pixels
	uHeight 			height
	uLoThresh			low threshold
	uHiThresh			high threshold

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISHistogramEqualizeGrayscale(BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, UINT32 uLoThresh, UINT32 uHiThresh);

/*******************************************************
	ISHistogramStretchRGB

	Purpose : 
	Perform histogram stretch on an RGB image

	Param					Use
	------------------------------------
	pRGB				RGB image
	uWidth				width in pixels
	uHeight 			height
	fLowLimit			percent of pixels to set to 0
	fHiLimit			percent of pixels to set to 255
	uChannelMask		select the channels to apply this operation 
						to. use CHRED , CHGREEN and CHBLUE OR'd
						together.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISHistogramStretchRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, double fLowLimit, double fHighLimit, UINT32 channelMask);

/*******************************************************
	ISHistogramStretchGrayscale

	Purpose : 
	Perform histogram stretch on an 8 bit grayscale image

	Param					Use
	------------------------------------
	pRGB				8-bit grayscale image
	uWidth				width in pixels
	uHeight 			height
	fLowLimit			percent of pixels to set to 0
	fHiLimit			percent of pixels to set to 255

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISHistogramStretchGrayscale(BYTE *p8Bit, UINT32 uWidth, UINT32 uHeight, double fLowLimit, double fHighLimit);

/*******************************************************
	ISGetRGBBrightnessHistogram

	Purpose : 
	Find the brightness histogram of an image. Should not
	be used on images larger than 2 ^ 32 pixels.

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	pHisto				array of 255 UINT32s

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetRGBBrightnessHistogram(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 *pHisto);

/*******************************************************
	ISGetRGBChannelHistogram

	Purpose : 
	Find the histogram for a channel in an image. Should not
	be used on images larger than 2 ^ 32 pixels.

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth				width in pixels
	uHeight 			height
	uChannelMask		image channel to test
							one of CHRED, CHGREEN or CHBLUE
							
	pHisto				array of 255 UINT32s

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGetRGBChannelHistogram(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uChannelMask, UINT32 *pHisto);


/*******************************************************
	ISRotateImage

	Purpose : 
	Rotate an image

	Param					Use
	------------------------------------
	pInBuf					source image
	uWidth					width in pixels
	uHeight 				height
	
	uBytesPerPixel			1 = 8 bit colormapped
							3 = RGB
	
	pPal					palette for 8-bit
							(not used when bFast = TRUE)

	uPalColors				number of colors in pPal
	
	fRadians				rotate angle

	uClrSpec 				color to use when filling uncovered areas
							of the new image. if 8-bit, this is the
							palette index of the color to fill.
							if RGB, this should be a COLORREF.

	puOutWidth				receives output width
	puOutHeight 			receives output height

	bFast					if TRUE, use a quick and dirty rotation.
							palette is ignored if 8-bit. if FALSE, 
							use bi-linear interpolation for better
							accuracy. 
							
							the "fast" method is approx. 6x faster (for 
							RGB, and approx. 50x faster for 8 bit) than 
							the interpolated method, but the results
							of the slow method are far better. fast or 
							accurate, choose one.

	Return
	------
		HGLOBAL handle to new image
		Caller must call GlobalFree to release this memory
*******************************************************/
extern HGLOBAL _ISEXP_ _ISRotateImage(BYTE *pInBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, RGBQUAD *pPal, UINT32 uPalColors, double fRadians, UINT32 uClrSpec, UINT32 *uOutWidth, UINT32 *uOutHeight, BOOL bFast);

/*******************************************************
	ISQuickRotateImage

	Purpose : 
	Rotate an image multiples of 90 degrees
	This performs the rotation in-place. Be sure to 
	note that the image dimensions will change when
	rotating 90 or 270 degrees! (width and height will 
	reverse)

	Param					Use
	------------------------------------
	pImage					source image
	uWidth					width in pixels
	uHeight 				height
	uRotate 				rotation selector
							0 - 90 degrees cw
							1 - 180 deg
							2 - 270 deg cw
	uBytesPerPixel			bytes per image pixel

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISQuickRotateImage(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uRotate, UINT32 uBytesPerPixel);

/*******************************************************
	ISQuickRotate1Bit

	Purpose : 
	Rotate a 1-bit image multiples of 90 degrees.

	Note:
	When rotating 90 or 270 degrees, output width
	and height are swapped, with respect to input
	width & height.

	Note:
	The source and destination image buffers will not
	be the same size! Output buffer must be (at least)
	((int)((uOutWidth + 7) / 8) * uOutHeight) bytes. 

	Param					Use
	------------------------------------
	pInBuf					source image
	pOutBuf					output image
	uWidth					source width in pixels
	uHeight 				source height
	uRotate 				rotation selector
							0 - 90 degrees cw
							1 - 180 deg
							2 - 270 deg cw

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL		_ISEXP_ _ISQuickRotate1Bit(BYTE *pInBuf, BYTE *pOutBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uRotate);

/*******************************************************
	ISShearRGBX

	Purpose : 
	Shear an RGB image. This slides the top (or bottom)
	of an image by xOffset pixels. (width increases)

	Param			Use
	------------------------------------
	pRGB			source image
	uWidth			width in pixels
	uHeight 		height
	pRGBOut 		output buffer - allocated by caller
					buffer must be (uWidth + abs(xOffset)) * uHeight * 3 BYTEs 

	xOffset			if > 0, the top of the image is offset and this value is
					the distance to offset the top row of the output 
					image from the top-left corner.
					if < 0, the bottom of the image is offset and this value 
					is the distance to offset the bottom row from the 
					bottom-left corner.
					
	clrBack			color to fill the background

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISShearRGBX(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, __int32 xOffset, COLORREF clrBack);

/*******************************************************
	ISShearRGBY

	Purpose : 
	Shear an RGB image. This slides the left (or right)
	of an image by yOffset pixels. (height increases)

	Param				Use
	------------------------------------
	pRGB				source image
	uWidth				width in pixels
	uHeight 			height
	pRGBOut 			output buffer - allocated by caller
						buffer must be uWidth * (uHeight + abs(yOffset)) * 3 BYTEs 

	yOffset				if > 0, the left of the image is offset and this value is
						 the distance to offset the leftmost column of the output 
						 image from the top-left corner.
						 if < 0, the right of the image is offset and this value 
						 is the distance to offset the rightmost column from the 
						 top-right corner.

	clrBack				color to fill the background

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISShearRGBY(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, __int32 yOffset, COLORREF clrBack);

/*******************************************************
	ISShear8BitX

	Purpose : 
	Shear an 8-bit image. This slides the top (or bottom)
	of an image by xOffset pixels. (width increases)

	Param			Use
	------------------------------------
	pImg			source image
	uWidth			width in pixels
	uHeight 		height
	pOut 			output buffer - allocated by caller
					buffer must be (uWidth + abs(xOffset)) * uHeight BYTEs 

	xOffset			if > 0, the top of the image is offset and this value is
					the distance to offset the top row of the output 
					image from the top-left corner.
					if < 0, the bottom of the image is offset and this value 
					is the distance to offset the bottom row from the 
					bottom-left corner.
					
	uBackIdx		palette index to fill the background

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISShear8BitX(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, BYTE *pOut, __int32 xOffset, UINT32 uBackIdx);

/*******************************************************
	ISShear8BitY

	Purpose : 
	Shear an 8-bit image. This slides the left (or right)
	of an image by yOffset pixels. (height increases)

	Param			Use
	------------------------------------
	pImg			source image
	uWidth			width in pixels
	uHeight 		height
	pOut 			output buffer - allocated by caller
					buffer must be uWidth * (uHeight + abs(yOffset)) BYTEs 

	xOffset			if > 0, the top of the image is offset and this value is
					the distance to offset the top row of the output 
					image from the top-left corner.
					if < 0, the bottom of the image is offset and this value 
					is the distance to offset the bottom row from the 
					bottom-left corner.
					
	uBackIdx		palette index to fill the background

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISShear8BitY(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, BYTE *pOut, __int32 yOffset, UINT32 uBackIdx);

/*******************************************************
	ISCropImage

	Purpose : 
	Crop an image . Assumes the 'image' is a rectangular array 
	of scanlines. Each scan line in the input image is composed
	of uWidth pixels. Each pixel is uPixelBytes bytes wide.

	Output image width must be ((uRight - uLeft) + 1) pixels.
	Output image height must be ((uBottom - uTop) + 1) pixels.

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height 
	pRGBOut 				cropped image - allocated by the caller
	uLeft					left-most pixel in cropped image
	uTop					top-most pixel in cropped image
	uCropWidth				width of cropped image
	uCropHeight 			height of cropped image
	uPixelBytes 			bytes per pixel

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISCropImage(BYTE *pIn, UINT32 uWidth, UINT32 uHeight, BYTE *pOut, UINT32 uLeft, UINT32 uTop, UINT32 uCropWidth, UINT32 uCropHeight, UINT32 uPixelBytes);

/*******************************************************
	ISCrop1Bit

	Purpose : 
	Crop a 1 bit image

	Destination image must be 
	
	  (int)(((uRight - uLeft) + 7) / 8) * (uBottom - uTop)

	bytes.

	Pixel rows start on whole byte boundaries.

	Output image width must be ((uRight - uLeft) + 1) pixels.
	Output image height must be ((uBottom - uTop) + 1) pixels.

	Param					Use
	------------------------------------
	pImg					source image
	uWidth					width in pixels
	uHeight 				height 
	pImgOut 				cropped image - allocated by the caller
	uLeft					left-most pixel in cropped image
	uTop					top-most pixel in cropped image
	uCropWidth				width of cropped image
	uCropHeight 			height of cropped image

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISCrop1Bit(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, BYTE *pImgOut, UINT32 uLeft, UINT32 uTop, UINT32 uCropWidth, UINT32 uCropHeight);

/*******************************************************
	ISZoomRGB

	Purpose : 
  	Zoom in on a section of an RGB image. This is
	a call to ISCropImage followed by a call to 
	ISResizeImage.


	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height 
	uLeft					left-most pixel in cropped image
	uTop					top-most pixel in cropped image
	uRight					right-most pixel in cropped image
	uBottom 				bottom-most pixel in cropped image
	pRGBOut 				zoomed image - allocated by the caller
	uOutWidth				output width
	uOutHeight				output height

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISZoomRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uLeft, UINT32 uTop, UINT32 uRight, UINT32 uBottom, BYTE *pRGBOut, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISGetBorder

	Purpose : 
	Using color matching, this function will attempt to 
	find the coordinates of the image within any image border.

	For instance, this can be used to do an "auto-crop" of a 
	scanned image. Scanned pictuers often come from the scanner
	slightly rotated and bordered with a border of scanner 
	backing. This function will attempt to find the picture 
	space vs. the border space. You can then use the ISCrop* 
	functions to remove the border area.

	If you do not provide the border color, this function
	will use it's own algorithm for determining the color.
	To determine the color of the border part of the image,
	this function examines the four corners. If three of the 
	four corners match, within the tolerance specified, 
	their color is chosen as the border color. If three of the
	four corners do not match, this function will fail.

	After determining the border color, this function scans 
	along the edges of the image, stopping in each direction 
	when a pixel is detected that does not match the border 
	color, given the specified tolerance.

	Param					Use
	------------------------------------
	pImg					image
	uWidth					width in pixels
	uHeight 				height 
	uBitsPerPixel			bits per pixel in image (8 or 24)
	pPal					palette (for 8-bit images)
	uTolerance				color matching tolerance
							0 = strict color matching
							3 * (255 * 255) = maximum tolerance
								(black matches white)

	uColorSpec				border color (if you want to force your own)
							COLORREF if 24-bit, palette index if 8-bit

	uFlags					if bit 0 is set, the function will use your
							border color instead of using the three-corners
							method.

	puLeft					receives first column of non-border image
	puTop					receives first row of non-border image
	puRight					receives last column of non-border image
	puBottom				receives last row of non-border image
	
	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL		_ISEXP_ _ISGetBorder(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uBitsPerPixel, RGBQUAD *pPal, UINT32 uTolerance, UINT32 uColorSpec, UINT32 uFlags, UINT32 *puLeft, UINT32 *puTop, UINT32 *puRight, UINT32 *puBottom);

/*******************************************************
	ISFlipHorizontalImage

	Purpose : 
	Horizontally flip an image

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height 
	uBytesPerPixel			bytes per pixel

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISFlipHorizontalImage(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel);

/*******************************************************
	ISFlipHorizontal1Bit

	Purpose : 
	Horizontally flip a 1 bit image

	Param					Use
	------------------------------------
	pImg					source image
	uWidth					width in pixels
	uHeight 				height 

	This function expects that the 1-bit image is
	((uWidth + 7) / 8) * uHeight pixels. Each
	row must be ((uWidth + 7) / 8) bytes and must
	begin on a BYTE.

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISFlipHorizontal1Bit(BYTE *pImg, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISOverlayRGB

	Purpose : 
	Overlay one RGB onto another

	Param					Use
	------------------------------------
	pRGB1					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	pRGB2					top image
	uWidth2 				top width in pixels
	uHeight2				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fOpacity				opacity of top on bottom (> 0, < 1)
							(0 = transparent, 1 = opaque)
	bFast					favor speed over accuracy

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISOverlayRGB(BYTE *pRGB1, UINT32 uWidth1, UINT32 uHeight1, BYTE *pRGB2, UINT32 uWidth2, UINT32 uHeight2, __int32 iXPos, __int32 iYPos, double fOpacity, BOOL bFast);

/*******************************************************
	ISOverlay8BitToRGB

	Purpose : 
	Overlay an 8-bit image onto an RGB image

	Param					Use
	------------------------------------
	pRGB					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	p8Bit					top image
	uWidth2 				top width in pixels
	uHeight2				top height
	pPal					palette of top image
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fOpacity				opacity of top on bottom (> 0, < 1)
							(0 = transparent, 1 = opaque)
	bFast					favor speed over accuracy

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL 	_ISEXP_ _ISOverlay8BitToRGB(BYTE *pRGB1, UINT32 uWidth1, UINT32 uHeight1, BYTE *p8Bit, UINT32 uWidth2, UINT32 uHeight2, RGBQUAD *pPal, __int32 iXPos, __int32 iYPos, double fOpacity, BOOL bFast);

/*******************************************************
	ISOverlayImage8BitAlphaOnRGB

	Purpose : 
	Overlay an 8-bit image onto an RGB image using
	the rgbReserved values in the palette as alpha
	values.

	Param					Use
	------------------------------------
	pRGB					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	p8Bit					top image
	uWidth2 				top width in pixels
	uHeight2				top height
	pPal					palette of top image
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL 	_ISEXP_ _ISOverlayImage8BitAlphaOnRGB(BYTE *pBottomImg, UINT32 uBottomImageWidth, UINT32 uBottomImageHeight, BYTE *pTopImg, UINT32 uTopImageWidth, UINT32 uTopImageHeight, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos);

/*******************************************************
	ISOverlayImage8BitAlpha

	Purpose : 
	Overlay one 8-bit image onto another using the
	rgbReserved values in the palette as alpha values.

	Note:
	The images are assumed to have the same palette.

	Param					Use
	------------------------------------
	pImageBottom			bottom image
	uWidthBottom 			bottom width in pixels
	uHeightBottom			bottom height
	pImageTop				top image
	uWidthTop 				top width in pixels
	uHeightTop				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	uPalColors				number of colors in the palette
	pPal					palette

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL _ISEXP_ _ISOverlayImage8BitAlpha(BYTE *pBottomImg, UINT32 uBottomImageWidth, UINT32 uBottomImageHeight, BYTE *pTopImg, UINT32 uTopImageWidth, UINT32 uTopImageHeight, __int32 iXPos, __int32 iYPos, UINT32 uPalColors, RGBQUAD *pPal);

/*******************************************************
	ISOverlay8Bit

	Purpose : 
	Overlay one 8-bit image onto another

	Note:
	Both palettes are NULL:
	The images are assumed to have the same palette and the 
	top image is simply copied onto the bottom image with 
	no color matching. Opacity is ignored!

	pPalBottom is not NULL but pPalTop is NULL:
	Both images use the bottom palette.
	Output pixels are determined by looking up the RGB values 
	for both top and bottom pixels in the bottom palette and 
	combining these according to the opacity value. The 
	closest color to the resulting ideal RGB value will be 
	used from the bottom palette.

	Both palettes are not NULL:
	Output pixels are determined by looking up the RGB from 
	the bottom pixel in the bottom palette, the top pixel in 
	the top palette and combining these according to the opacity 
	value. The closest color to the resulting ideal RGB value 
	will be used from the bottom palette.

	Param					Use
	------------------------------------
	pImageBottom			bottom image
	uWidthBottom 			bottom width in pixels
	uHeightBottom			bottom height
	uBottomPalColors		number of colors in bottom palette
	pPalBottom				palette for bottom image. 
	pImageTop				top image
	uWidthTop 				top width in pixels
	uHeightTop				top height 
	pPalTop					palette for bottom image
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fOpacity				opacity of top on bottom (> 0, < 1)
							(0 = transparent, 1 = opaque)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISOverlay8Bit(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, UINT32 uBottomPalColors, RGBQUAD *pPalBottom, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos, double fOpacity);

/*******************************************************
	ISOverlay8BitTrans

	Purpose : 
	Overlay one 8-bit image onto another with a transparent color

	Note:
	The palette rules from ISOverlay8Bit apply to this function.

	Param					Use
	------------------------------------
	pImageBottom			bottom image
	uWidthBottom 			bottom width in pixels
	uHeightBottom			bottom height
	uBottomPalColors		number of colors in bottom palette
	pPalBottom				palette for bottom image. 
	pImageTop				top image
	uWidthTop 				top width in pixels
	uHeightTop				top height 
	pPalTop					palette for bottom image
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fOpacity				opacity of top on bottom (> 0, < 1)
							(0 = transparent, 1 = opaque)
	uTransIdx				color index in top image to make transparent

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL		_ISEXP_ _ISOverlay8BitTrans(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, UINT32 uBottomPalColors, RGBQUAD *pPalBottom, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, RGBQUAD *pPalTop, __int32 iXPos, __int32 iYPos, double fOpacity, UINT32 uTransIdx);

/*******************************************************
	ISOverlay1Bit

	Purpose : 
	Overlay one one bit image onto another

	pixel rows in each image must be :
	(imageWidth + 7) / 8 bytes wide. 
	
	pixel rows must start on byte boundaries.

	Param					Use
	------------------------------------
	pImageBottom			bottom image
	uWidthBottom			bottom width in pixels
	uHeightBottom			bottom height 
	pImageTop				top image
	uWidthTop 				top width in pixels
	uHeightTop				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISOverlay1Bit(BYTE *pImageBottom, UINT32 uWidthBottom, UINT32 uHeightBottom, BYTE *pImageTop, UINT32 uWidthTop, UINT32 uHeightTop, __int32 iXPos, __int32 iYPos);

/*******************************************************
	ISOverlayRGBTrans

	Purpose : 
	Overlay one RGB onto another with a transparency
	color.

	Param					Use
	------------------------------------
	pRGB1					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	pRGB2					top image
	uWidth2 				top width in pixels
	uHeight2				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fOpacity				opacity of top on bottom (> 0, < 1)
	clrTrans				color in top to make transparent
	uTolerance				maximum color distance to match

							0 - only match specified color 
								(strict color matching)

							3 * (255 * 255) - maximum tolerance
								(black matches white)

	bFast					favor speed over accuracy

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL 	_ISEXP_ _ISOverlayRGBTrans(BYTE *pRGB1, UINT32 uWidth1, UINT32 uHeight1, BYTE *pRGB2, UINT32 uWidth2, UINT32 uHeight2, __int32 iXPos, __int32 iYPos, double fOpacity, COLORREF clrTrans, UINT32 uTolerance, BOOL bFast);

/*******************************************************
	ISOverlayRGBEdgeFaded

	Purpose : 
	Overlay one RGB onto another, fade the edges of the
	top image.

	Param					Use
	------------------------------------
	pRGB1					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	pRGB2					top image
	uWidth2 				top width in pixels
	uHeight2				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)
	fFadeWidthRatio			rate at which the top edges fade to 
							transparent

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISOverlayRGBEdgeFaded(BYTE *pRGB1, UINT32 uWidth1, UINT32 uHeight1, BYTE *pRGB2, UINT32 uWidth2, UINT32 uHeight2, __int32 iXPos, __int32 iYPos, double fFadeWidthRatio);

/*******************************************************
	ISAlphaBlendRGB

	Purpose : 
	Overlay one RGB onto another using an alpha mask to control
	transparency.

	Param					Use
	------------------------------------
	pRGB1					bottom image
	uWidth1 				bottom width in pixels
	uHeight1				bottom height 
	pRGB2					top image
	uWidth2 				top width in pixels
	uHeight2				top height 
	pAlphaMask				8 bit alpha mask (uWidth2 x uHeight2 pixels)
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISAlphaBlendRGB(BYTE *pRGB1, UINT32 uWidth1, UINT32 uHeight1, BYTE *pRGB2, UINT32 uWidth2, UINT32 uHeight2, BYTE *pAlphaMask, __int32 iXPos, __int32 iYPos);
				
/*******************************************************
	ISAlphaBlendRGBA

	Purpose : 
	Overlay an RGBA image onto an RGB image.

	Param					Use
	------------------------------------
	pRGB					bottom image
	uWidthRGB				bottom width in pixels
	uHeightRGB				bottom height 
	pRGBA					top image (RGBA)
	uWidthRGBA 				top width in pixels
	uHeightRGBA				top height 
	iXPos					position of top on bottom (may be < 0)
	iYPos					position of top on bottom (may be < 0)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISAlphaBlendRGBA(BYTE *pRGB, UINT32 uWidthRGB, UINT32 uHeightRGB, BYTE *pRGBA, UINT32 uWidthRGBA, UINT32 uHeightRGBA, __int32 iXPos, __int32 iYPos);

/*******************************************************
	ISMakeDropShadowOnRGB

	Purpose : 
	Create a drop-shadowed image. 

	Note : the caller must allocate the output image buffer

	Param					Use
	------------------------------------
	pRGB1					source image
	uInWidth				source width in pixels
	uInHeight				source height in pixels
	bkColor					color of shadow background
	uShadowSize				size of shadow (pixels from edge of source)
	pOutRGB					user-allocated image. 
							this must be (uInWidth + uShadowSize) *
							   (uInHeight + uShadowSize) * 3 pixels.
	uPosition				0 = bottom right shadow
							1 = bottom left shadow
							2 = top left shadow
							3 = top right shadow

	bBlur					apply blur effect to the shadow

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISMakeDropShadowOnRGB(BYTE *pRGB, UINT32 uInWidth, UINT32 uInHeight, COLORREF bkColor, UINT32 uShadowSize, BYTE *pOutRGB, UINT32 uPosition, BOOL bBlur);

/*******************************************************
	ISFillRGBSolidRect

	Purpose : 
	Color a rectangle in an RGB buffer with a solid RGB 
	color.

	Param					Use
	------------------------------------
	pRGB					image
	uWidth				width in pixels
	uHeight 			height
	uRectStartX 		left of fill rect
	uRectStartY 		top of fill rect
	uRectWidth			width of fill rect
	uRectHeight 		height of fill rect
	clrFill 			fill RGB color

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISFillRGBSolidRect(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uRectStartX, UINT32 uRectStartY, UINT32 uRectWidth, UINT32 uRectHeight, COLORREF clrFill);

/*******************************************************
	ISFill8BitSolidRect

	Purpose : 
	Fill a rectangle in an 8-bit buffer with a single color index.

	Param					Use
	------------------------------------
	p8Bit					image
	uWidth					width in pixels
	uHeight 				height
	uRectStartX 			left of fill rect
	uRectStartY 			top of fill rect
	uRectWidth				width of fill rect
	uRectHeight 			height of fill rect
	uIdxFill 				color index to fill with

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISFill8BitSolidRect(BYTE *pImg, UINT32 uWidth, UINT32 uHeight, UINT32 uRectStartX, UINT32 uRectStartY, UINT32 uRectWidth, UINT32 uRectHeight, UINT32 uIdxFill);

/*******************************************************
	ISFloodFillRGB

	Purpose : 
	Perform a flood fill in an RGB image. All pixels 
	contiguous to the start pixel are changed to
	clrFill. The extent of the fill is controlled by
	uFillMode.

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth 					width in pixels
	uHeight					height 
	uStartX					point at which to start the fill, X
	uStartY					point at which to start the fill, Y
	clrFill					color to fill with
	uFillMode				0 - fill area is bounded by 
							pixels matching clrFill
							1 - all contiguous pixels matching 
							the start pixel are set to clrFill.
	uTolerance				0 - only match specified color 
								(strict color matching)

							3 * (255 * 255) - maximum tolerance
								(black matches white)

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISFloodFillRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uStartX, UINT32 uStartY, COLORREF clrFill, UINT32 uFillMode, UINT32 uTolerance);

/*******************************************************
	ISPolygonFillImage

	Purpose : 
	Perform a polygon fill onto an image.

	Note:
	If filling with an image, the bottom image and the
	overlay image must have the same number of bytes per
	pixel. 

	Note:
	When filling an 8-bit image with another 8-bit image, the 
	fill image must use the same palette as the bottom image.

	Param					Use
	------------------------------------
	pImage					image
	uWidth 					width in pixels
	uHeight					height 
	uBytesPerPixel			bytes per pixel (1 or 3)
	pPoints					array of XYdouble values describing the 
							polygon to fill
	uNumPoints				points in the pPoints array

	pOverlayImage			image to be used as the fill pattern. NULL,
							if filling a solid color.

							image tiling starts at the minimum X,Y value
							in pPoints.

	uOverlayWidth			image width in pixels
	uOverlayHeight			image height 

	uColorSpec				fill color, if not filling with an image.
							if uBytesPerPixel = 1, the low byte of 
							this value is used as the output.

							if uBytesPerPixel = 3, this value is treated 
							as a COLORREF value.

	fOpacity				0 to 1. 1 = solid, 0 = transparent.	not supported
							when using an overlay image.

	pPal					use when filling. only applies when uBytesPerPixel = 1 
							and bit 0 of uFlags is set. 

	uPalColors				entries in pPal

	uFlags					bit 0 = 1 : use opacity value for filling. if
							uBytesPerPixel = 1, pPal must contain a palette!

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISPolygonFillImage(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, XYdouble *pPoints, UINT32 uNumPoints, BYTE *pOverImage, UINT32 uOverW, UINT32 uOverH, UINT32 uClrSpec);

/*******************************************************
	ISDrawEllipseOnImage

	Purpose : 
	Draw an ellipse on an image. 
	
	Param					Use
	------------------------------------
	pImage					image
	uWidth 					image width in pixels
	uHeight					image height 
	uBytesPerPixel			number of bytes in an image pixel

	uColorSpec				if uBytesPerPixel = 1, 
							the low byte of this value is used as
							the output.

							if uBytesPerPixel = 2, 
							the low word of this value is used as
							the output.

							if uBytesPerPixel = 3, 
							this value is treated as a COLORREF value.

							if uBytesPerPixel = 4, 
							this value is copied to the output.

	iXCenter, iYCenter		center point
	uHorizRadius			horizontal radius
	uVertRadius				vertical radius

	bFill					1 to fill, 0 for outline
	fOpacity				0 to 1. 1 = solid, 0 = transparent.
							does not apply when uBytesPerPixel = 2.

	pPal					use when filling. only applies when
							uBytesPerPixel = 1 and bit 0 of uFlags is set.

	uPalColors				entries in pPal

	uFlags					bit 0 - use opacity value for filling (1, 3, 
							4 bpp only).

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL _ISEXP_ _ISDrawEllipseOnImage(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uColorSpec, __int32 iXCenter,  __int32 iYCenter,  UINT32 uHorizRadius, UINT32 uVertRadius, BOOL bFill, double fOpacity, RGBQUAD *pPal, UINT32 uPalColors, UINT32 uFlags);

/*******************************************************
	ISDrawLineOnImage

	Purpose : 
	Draw a solid line on an image

	Param					Use
	------------------------------------
	pImage					image
	uWidth 					width in pixels
	uHeight					height 
	uBytesPerPixel			number of bytes in an image pixel

	uColorSpec				if uBytesPerPixel = 1, 
							the low byte of this value is used as
							the output.

							if uBytesPerPixel = 2, 
							the low word of this value is used as
							the output.

							if uBytesPerPixel = 3, 
							this value is treated as a COLORREF value.

							if uBytesPerPixel = 4, 
							this value is copied to the output.

	x0, y0					starting point
	x1, y1					end point

	uFlags					not currently used

	Return
	------
		FALSE on failure
*******************************************************/
extern BOOL		_ISEXP_ _ISDrawLineOnImage(BYTE *pImage, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, UINT32 uColorSpec, __int32 x0,  __int32 y0, __int32 x1, __int32 y1, UINT32 uFlags);

/*******************************************************
	ISCountRGBColors

	Purpose : 
	Count colors in an RGB image

	Param					Use
	------------------------------------
	pRGB					image
	uWidth					width in pixels
	uHeight 				height 
	bFast					1 = fast, uses 1MB, else
							uses .5 MB
	
	Return
	------
		number of colors in image
*******************************************************/
extern UINT32	_ISEXP_ _ISCountRGBColors(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 bFast);

/*******************************************************
	ISGetFirstNColorsInRGB

	Purpose : 
	Retrieve the first uColors colors from an
	RGB image.

	Stops when uColors unique colors are found,
	or when all pixels have been examined.

	Param					Use
	------------------------------------
	pRGB					image
	uWidth					width in pixels
	uHeight 				height 
	uColors					number of colors to get
	pPal					uColors RGBQUADs
	
	Return
	------
		number of colors in image
*******************************************************/
extern BOOL	_ISEXP_ _ISGetFirstNColorsInRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uColors, RGBQUAD *pPal);

/*******************************************************
	ISColorReplaceRGB

	Purpose : 
	Replace one color with another in an RGB image

	Param					Use
	------------------------------------
	pRGB					image
	uWidth					width in pixels
	uHeight 				height 
	clrOld					color to replace
	clrNew					color to replace with
	uTolerance				maximum color distance to match

							0 - only match specified color 
								(strict color matching)

							3 * (255 * 255) + 1 
							  - maximum tolerance
								(black matches white)
	
	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISColorReplaceRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, COLORREF clrOld, COLORREF clrNew, UINT32 uTolerance);

/*******************************************************
	ISRGBToRGBA

	Purpose : 
	Copy the RGB channels of an RGB image to the RGB
	channels of an RGBA image.

	Param					Use
	------------------------------------
	pRGB					image
	uWidth					width in pixels
	uHeight 				height 
	uFillAlpha				default alpha value
	pRGBAOut				output RGBA image
	
	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISRGBToRGBA(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uFillAlpha, BYTE* pRGBAOut);

/*******************************************************
	ISCombineRGBAndAToRGBA

	Purpose : 
	Copy the RGB channels of an RGB image and an alpha mask
	an RGBA image.

	Param					Use
	------------------------------------
	pRGBA					output image (caller allcoated. w * h * 4)
	pRGB					input RGB image
	pA						input 8-bit alpha image
	uWidth					width in pixels (of all buffers)
	uHeight 				height (of all buffers)
	uFillAlpha				default alpha value (0-255)
	
	if pAlpha is NULL, the alpha channel is set using uFillAlpha

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISCombineRGBAndAToRGBA(BYTE *pRGB, BYTE *pAlpha, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBA, UINT32 uFillAlpha);

/*******************************************************
	ISSetAlphaChannelInRGBA

	Purpose : 
	Copy an 8-bit image to the alpha channel of an RGBA image.

	Param					Use
	------------------------------------
	pRGBA					RGBA image
	uWidth					width in pixels
	uHeight 				height 
	p8BitAlphaMask			alpha mask image (uWidth x uHeight) 
	
	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISSetAlphaChannelInRGBA(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, BYTE *p8BitAlphaMask);

/*******************************************************
	ISDrawTextOnRGB

	Purpose : 
	Draw text on an RGB image

	Text transparency is controlled with ISSetTextTransparency.

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pFont					name of font
	uFontPixels 			height of font
	iXPos					x pos of text
	iYPos					y pos of text
	clrText 				color of text

    Callback is called once per font height line

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawTextOnRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const char *pFontName, UINT32 uFontPixels, __int32 iXPos, __int32 iYPos, COLORREF clrText);

/*******************************************************
	ISDrawTextOnRGB2

	Purpose : 
	Draw text on an RGB image

	Text transparency is controlled with ISSetTextTransparency.

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pLogFont				valid LOGFONT
	iXPos					x pos of text
	iYPos					y pos of text
	clrText 				color of text

    Callback is called once per font height line

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawTextOnRGB2(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const LOGFONT *pLogFont, __int32 iXPos, __int32 iYPos, COLORREF clrText);

/*******************************************************
	ISDrawTextOnRGBEx

	Purpose : 
	Draw text on an RGB image with extended formatting
	options.
	
	Text transparency is controlled with ISSetTextTransparency.

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pFont					name of font
	uFontPixels 			height of font
	iLeft					left of text rect
	iTop					top of text rect
	iRight					right of text rect
	iBottom 				bottom of text rect
	uFlags					DT_WORDBREAK : break words at edge of rect
							DT_WORD_ELLIPSIS :	Truncates text 
							that does not fit in the rectangle and 
							adds ellipses.	

	clrText 				color of text

    Callback is called once per font height line

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawTextOnRGBEx(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const char *pFontName, UINT32 uFontPixels, __int32 iLeft, __int32 iTop, __int32 iRight, __int32 iBottom, UINT32 uFlags, COLORREF clrText);

/*******************************************************
	ISDrawTextOnRGBEx

	Purpose : 
	Draw text on an RGB image with extended formatting
	options.

    Text transparency is controlled with ISSetTextTransparency.

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pLogFont				valid LOGFONT
	iLeft					left of text rect
	iTop					top of text rect
	iRight					right of text rect
	iBottom 				bottom of text rect
	uFlags					DT_WORDBREAK : break words at edge of rect
							DT_WORD_ELLIPSIS :	Truncates text 
							that does not fit in the rectangle and 
							adds ellipses.	

	clrText 				color of text

    Callback is called once per font height line

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawTextOnRGBEx2(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const LOGFONT *pLogFont, __int32 iLeft, __int32 iTop, __int32 iRight, __int32 iBottom, UINT32 uFlags, COLORREF clrText);

/*******************************************************
	ISGetTextLineSize

	Purpose : 
	Determine pixels required to draw a single line of text.
	This returns size of text as would be drawn by ISDrawTextOnRGB
	or ISDrawTextOnRGB2

	To get the size of text as drawn with ISDrawSmoothTextOnRGB
	or ISDrawSmoothTextOnRGB2, first, multiply the height of the
	font by the "smoothing factor", call this function, then divide
	the width and height returned by the smoothing factor. This is 
	required to account for the scaling performed by the DrawSmoothText
	functions.

	Param					Use
	------------------------------------
	drawText 				 text to measure
	fontName 				 name of font
	fontPixels				 height of font
	pWidth					 receives text width
	pHeight					 receives text height

	Callback is not called

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISGetTextLineSize(const char *drawText, const char *fontName, UINT32 fontPixels, UINT32 *pWidth, UINT32 *pHeight);

/*******************************************************
	ISGetTextLineSize2

	Purpose : 
	Determine pixels required to draw a single line of text.
	This returns size of text as would be drawn by ISDrawTextOnRGB
	or ISDrawTextOnRGB2

	To get the size of text as drawn with ISDrawSmoothTextOnRGB
	or ISDrawSmoothTextOnRGB2, first, multiply the height of the
	font by the "smoothing factor", call this function, then divide
	the width and height returned by the smoothing factor. This is 
	required to account for the scaling performed by the DrawSmoothText
	functions.

    Param					Use
	------------------------------------
	drawText 				text to measure
	pLogFont				a properly initialized LOGFONT structure
	pWidth					receives text width
	pHeight					receives text height

	Callback is not called

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISGetTextLineSize2(const char *drawText, const LOGFONT *pLogFont, UINT32 *pWidth, UINT32 *pHeight);

/*******************************************************
	ISDrawSmoothTextOnRGB

	Purpose : 
	Draw text on an RGB image. The text is smoothed with
	resizing and masking to simulate anti-aliasing.

    Text transparency is controlled with ISSetTextTransparency.

	The higher the smoothing factor, the more memory
	this funciton requires and the slower it will be.

	Callback is called 9 times

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pFont					name of font
	uFontPixels 			height of font
	uSmoothingFactor		amount of smoothing 
							  1 equals no smoothing
							  2 is good
							  4 is better (recommended)
							  higher values are possible, 
							  but not recommended

	iXPos					x pos of text
	iYPos					y pos of text
	clrText 				color of text

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawSmoothTextOnRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const char *pFontName, UINT32 uFontHeight, UINT32 uSmoothingFactor, __int32 iXPos, __int32 iYPos, COLORREF textColor);

/*******************************************************
	ISDrawSmoothTextOnRGB2

	Purpose : 
	Draw text on an RGB image using a LOGFONT. The text 
	is smoothed with resizing and masking to simulate 
	anti-aliasing.

    Text transparency is controlled with ISSetTextTransparency.

	The higher the smoothing factor, the more memory
	this funciton requires and the slower it will be.

	Callback is called 9 times

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pText					text to write
	pLogFont				properly initialized LOGFONT
	uSmoothingFactor		amount of smoothing 
							  1 equals no smoothing
							  2 is good
							  4 is better (recommended)
							  higher values are possible, 
							  but not recommended

	iXPos					x pos of text
	iYPos					y pos of text
	clrText 				color of text

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISDrawSmoothTextOnRGB2(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, const char *pText, const LOGFONT *pLogFont, UINT32 uSmoothingFactor, __int32 iXPos, __int32 iYPos, COLORREF textColor);

/*******************************************************
	ISSetTextTransparency

	Purpose : 
	Set the transparency that ISDrawText* and 
	ISDrawSmoothText* use.

	Param					Use
	------------------------------------
	fTransparency			text transparency (0 <-> 1)
							0 = completely transparent (invisible)
							1.0 = completely opaque (default)

*******************************************************/	
extern void	_ISEXP_ _ISSetTextTransparency(double fTransparency);

/*******************************************************
	ISApplyConvolutionFilterToRGB

	Purpose : 
	Apply a convolution filter to an RGB image

	The number of columns and the number of rows in the 
	matrix must both be odd.

	The filter is applied as follows :

		M = matrix ( 3 x 3, in this example)

			 1	1  1
		M = 1 -8  1 		this is a sharpening matrix
			 1	1  1

		the matrix is centered on the current pixel. in 
		this example, M[4] is the center of the matrix.

		k is the level 

		P = pixels . since the matrix is centered on the 
		current pixel, P(x,y), the pixels used in this 
		calculation will be :
		
			P(x-1,y-1), P(x, y-1),	P(x+1, y-1)
			P(x-1,y),	P(x, y),	P(x+1, y)
			P(x-1,y+1), P(x, y+1),	P(x+1, y+1)

		t = sum ( M[i] * P[i] )  [for all elements of the matrix]

		outPixel = curPixel - (k * t)

	Matrix M in this example demonstrates the sharpening filter 
	used by this library.

	besides the ability to use matricies of arbitrary size,
	this function differs from ISApplyMatrixToRGB in the way 
	the final caluclation is performed. 

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pRGBOut 				output image
	uFilterCols 			columns in filter matrix
	uFilterRows 			rows in filter matrix
	pFilter 				filter matrix
	k						filter level

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISApplyConvolutionFilterToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uFilterCols, UINT32 uFilterRows, double * pFilter, double k);

/*******************************************************
	ISApplyLUTAndCrossoverMatrixToRGB

	Purpose : 
	Apply a crossover matrix and LUT to an RGB image.

	The matrix is a 12 item (3 rows, 4 columns) array of
	floating point values. 

	The LUT is an array of 3 * 256 BYTEs. 

	LUT[0]	-> LUT[255] are the values for the red channel (LUT_rd).
	LUT[256] -> LUT[511] are the values for the green channel (LUT_gr).
	LUT[512] -> LUT[767] are the values for the blue channel (LUT_bl).

	If pLUT is NULL, an identity LUT will be used.


	The matrix and LUT are applied as follows :

	  -------------

	  For each pixel, "cur", 

	  // LUT
	  cur.rd	= LUT_rd[cur.rd]
	  cur.gr	= LUT_gr[cur.gr]
	  cur.bl	= LUT_bl[cur.bl]

	  // matrix
	  out.rd =(cur.rd * matrix[0, 0] + 
				cur.gr * matrix[0, 1] +
				cur.bl * matrix[0, 2])
			
	  out.rd = out.rd + matrix[0, 3]

	  out.gr =(cur.rd * matrix[1, 0] + 
				cur.gr * matrix[1, 1] +
				cur.bl * matrix[1, 2]) 

	  out.gr = out.gr + matrix[1, 3]

	  out.bl =(cur.rd * matrix[2, 0] + 
				cur.gr * matrix[2, 1] +
				cur.bl * matrix[2, 2])

	  out.bl = out.bl + matrix[2, 3]
	  
	  ------------

	  Matrix samples : 

	  This matrix has no effect on the image :
		 1. 0	0	 0
		 0	1.	0	 0
		 0	0	1.	 0

	  Increase the saturation :
		 1.20  -.1	 -.1	 0
		 -.1	1.20 -.1  0
		 -.1	-.1	1.20	 0

	  Increases the brightness :
		 1. 0	0	 20
		 0	1.	0	 20
		 0	0	1.	 20

	  Change to grayscale :
		 1/3	1/3	 1/3	0
		 1/3	1/3	 1/3	0
		 1/3	1/3	 1/3	0

	  General rule : the sum of the first 3 items in 
	  a row should add to 1. 

	Param					Use
	------------------------------------
	pRGB					RGB image
	uWidth					width in pixels
	uHeight 				height 
	pLUT 					look-up table (LUT) - 768 values
	pMatrix					12 element crossover matrix

	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL	_ISEXP_ _ISApplyLUTAndCrossoverMatrixToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pLUT, double *pMatrix);

/*******************************************************
	ISDecimateRGB

	Purpose : 
	Recude the size of an RGB image using pixel area 
	averaging. For photographic images, this often gives
	better results than ISResizeRGB.

	This will only reduce an image. It will not
	enlarge.

	pRGB != pRGBOut

	this will fail if:
	(uWidth/uOutWidth) * (uHeight/uOutHeight) > 4096 * 4096

	Param					Use
	------------------------------------
	pRGB					source image
	uWidth					width in pixels
	uHeight 				height
	pRGBOut 				destination image
	uOutWidth				output width
	uOutHeight				utput height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDecimateRGB(BYTE * pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRGBOut, UINT32 uOutWidth, UINT32 uOutHeight);

/*******************************************************
	ISPushImage

	Purpose : 
	Perform a point to point warp of an image.

	This does not resize the image. It just smears the
	pixels around.

	Param					Use
	------------------------------------
	pInBuf					input image
	pOutBuf					output image
	uWidth					width in pixels
	uHeight 				height 
	uBytesPerPixel			1 = colormapped 8-bit image
							3 = RGB image
	pPal					image palette (8-bit only)
	uPalColors				colors in pPal
	uToX					x coordinate of the "to" pixel
	uToY					y coordinate of the "to" pixel
	uFromX					x coordinate of the "from" pixel
	uFromY					y coordinate of the "from" pixel
	uFlags					unused, for now
	
	Return
	------
		FALSE on failure
*******************************************************/	
extern BOOL		_ISEXP_ _ISPushImage(BYTE *pInBuf, BYTE *pOutBuf, UINT32 uWidth, UINT32 uHeight, UINT32 uBytesPerPixel, RGBQUAD *pal, UINT32 uNUmPalColors, UINT32 uToX, UINT32 uToY, UINT32 uFromX, UINT32 uFromY, UINT32 uFlags);

/*******************************************************
	ISMakeRectFadeMask

	Purpose : 
	Create an 8 bit buffer to be used as an edge fade mask.
	Can be used in conjunction with ISSetAlphaChannelInRGBA
	to create an edge-faded alpha mask.

	Param					Use
	------------------------------------
	p8BitMask			output 8-bit image
	uWidth				width in pixels
	uHeight 			height
	fFadeWidthRatio		rate of fade

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISMakeRectFadeMask(BYTE *p8BitMask, UINT32 uWidth, UINT32 uHeight, double fFadeWidthRatio);

/*******************************************************
	ISApplyBrightnessContrastToRGB

	Purpose : 
	Modify the brightness and/or contrast of an RGB image.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	fBrightness 			brightness adjustment (-1..+1)
	fContrast				contrast adjustment (-1..+1)

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISApplyBrightnessContrastToRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, double fBrightness, double fContrast);

/*******************************************************
	ISAutoBrightnessRGB

	Purpose : 
	Automagically modify the brightness of an RGB image.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	uLowThreshold			pixels below this level are set to black
	uHighThreshold			pixels above this level are set to white

	uMode					0 : stretch brightness values to fit 
							the specified range (histogram stretch)

							1 : evenly distribute brightness values
							across the specified range (histogram
							equalization)
	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISAutoBrightnessRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uLowThreshold, UINT32 uHighThreshold, UINT32 uMode);

/*******************************************************
	ISDespeckleRGB

	Purpose : 
	Apply a despeckle filter to the image.
	
	Best results are acheived by repeated applying this
	filter and inspecting the results.

	Callback is called 14 times

	Param					Use
	------------------------------------
	pRGB					24-bit RGB image
	uWidth					width in pixels
	uHeight 				height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDespeckleRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISDespeckleGrayscale

	Purpose : 
	Apply a despeckle filter to the image.
	
	Best results are acheived by repeated applying this
	filter and inspecting the results.

	Callback is called 6 times

	Param					Use
	------------------------------------
	pGrayscale				8-bit grayscale image
	uWidth					width in pixels
	uHeight 				height

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISDespeckleGrayscale(BYTE *pGrayscale, UINT32 uWidth, UINT32 uHeight);

/*******************************************************
	ISQuantizeRGBTo8Bit

	Purpose : 
	Create an 8 bit image by performing color quantization
	and dithering on an RGB image.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	p8Bit					output image
	uColors 				colors in output image
	pPal					receives palette of 8 bit image
	uDitherStyle			0 - two-value dither
							1 - four-value (Floyd-Steinberg) dither

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISQuantizeRGBTo8Bit(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *p8Bit, UINT32 uColors, RGBQUAD *pPal, UINT32 uDitherStyle);

/*******************************************************
	ISRGBTo8BitBestFit

	Purpose : 
	Create an 8 bit image by doing a best-fit (Euclidean
	distance) match of the pixels in the RGB to the values
	in the palette. Errors in color matching are ignored.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	p8Bit					output 8-bit image
	uColors 				colors in palette
	pPal					target palette

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISRGBTo8BitBestFit(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *p8Bit, UINT32 uColors, RGBQUAD *pPal);

/*******************************************************
	ISRGBTo8BitDithered

	Purpose : 
	Create an 8 bit image by doing a best-fit (Euclidean
	distance) match of the pixels in the RGB to the values
	in the palette. Uses the dithering step from 
	ISQuantizeRGBTo8Bit to diffuse color matching errors.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	p8Bit					output 8-bit image
	uColors 				colors in palette
	pPal					target palette
	uDitherStyle			0 - two-value dither
							1 - four-value (Floyd-Steinberg) dither

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISRGBTo8BitDithered(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *p8Bit, UINT32 uColors, RGBQUAD *pal, UINT32 uDitherStyle);

/*******************************************************
	ISGet8BitPaletteFromRGB

	Purpose : 
	By using the color quantization step from
	ISQuantizeRGBTo8Bit, generate a palette from the 
	input RGB image.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels
	uHeight 				height
	uColors 				colors in output image
	pPal					receives palette

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISGet8BitPaletteFromRGB(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, UINT32 uColors, RGBQUAD *pPal);

/*******************************************************
	ISSplitRGBChannels

	Purpose : 
	Copy the red, green and blue channels into separate
	buffers.

	Caller must allocate all output buffers

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels (of all buffers)
	uHeight 				height (of all buffers)
	pRed					8 bit red buffer
	pGreen					8 bit green buffer
	pBlue					8 bit blue buffer

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSplitRGBChannels(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRed, BYTE *pGreen, BYTE *pBlue);

/*******************************************************
	ISCombineRGBChannels

	Purpose : 
	Combine 8-bit red, green and blue images into a single
	RGB image.

	Param					Use
	------------------------------------
	pRGB					source RGB image
	uWidth					width in pixels (of all buffers)
	uHeight 				height (of all buffers)
	pRed					8 bit red buffer
	pGreen					8 bit green buffer
	pBlue					8 bit blue buffer

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISCombineRGBChannels(BYTE *pRGB, UINT32 uWidth, UINT32 uHeight, BYTE *pRed, BYTE *pGreen, BYTE *pBlue);

/*******************************************************
	ISSplitRGBAToRGBPlusA

	Purpose : 
	Copy the RGB channels and the Alpha channel in an RGBA
	image to seperate buffers.

	Caller must allocate all output buffers.

	if pRGB is NULL, only the alpha channel is copied.
	if pAlpha is NULL, only the RGB channels are copied.

	Param					Use
	------------------------------------
	pRGBA					source RGBA image
	uWidth					width in pixels (of all buffers)
	uHeight 				height (of all buffers)
	pRGB					output RGB buffer
	pAlpha					output Alpha buffer

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISSplitRGBAToRGBPlusA(BYTE *pRGBA, UINT32 uWidth, UINT32 uHeight, BYTE *pRGB, BYTE *pAlpha);

/*******************************************************
	ISPack8BitBuffer

	Purpose : 
	Pack the pixels from an 8-bit buffer into a 
	buffer with 1-7 bits per pixel. 
	
	This function uses the low uBPP bits from each input pixel 
	for the output pixels.

	Multiple output pixels will share BYTEs and a pixel may 
	span multiple BYTEs. 
	
	This function assumes the output buffer is large enough to
	hold all output pixels. The size of the output buffer must 
	be at least ((uWidth * uHeight * uBPP / 8) + 1) BYTEs. The 
	caller must allocate this buffer. 
	
	Data in the last BYTE of the output buffer will be in the
	HIGH bits.

	Unused bits in the last BYTE (that contains pixel data) of 
	the output buffer will be set to 0.

	Param					Use
	------------------------------------
	pInput				source 8-bit image
	uWidth				width in pixels (of all buffers)
	uHeight 			height (of all buffers)
	pOutput 			output image
	uBPP				output bits per pixel (1..7)
	bPadRows			if TRUE, output rows are uOutRowWidth
						BYTEs wide. if FALSE, no padding is
						added at the end of a line; the last
						pixel on a line may share a BYTE with 
						the first pixel on the next line.
	uOutRowWidth		width, in bytes, of output rows.
						ignored if bPadRows is FALSe
	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL 	_ISEXP_ _ISPack8BitBuffer(BYTE *pInput, UINT32 uWidth, UINT32 uHeight, BYTE *pOutput, UINT32 uBPP, BOOL bPadRows, UINT32 uOutRowWidth);


/*******************************************************
	ISUnpackBufferTo8Bit

	Purpose : 
	Unpack packed pixels in a 1-8 bit buffer.
	
	Input rows must be packed pixels - no row end padding
	and no inter-pixel padding.	If you need padding at the
	end of your rows, call this function once for each line, 
	moving the output address to accommodate the desired row 
	padding.

	This function assumes the output buffer is large enough to
	hold all output pixels. 

	The size of the output buffer must be at least (uWidth * 
	uHeight) BYTEs. The caller must allocate this buffer. 

	Data in the last BYTE of the output buffer is expected to 
	be in the HIGH bits.

	Param				Use
	------------------------------------
	pInput				source packed image
	uWidth				width in pixels (of all buffers)
	uHeight 			height (of all buffers)
	uBPP				input bits per pixel (1..7)
	pOutput 			output 8-bit buffer 

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISUnpackBufferTo8Bit(BYTE *pInput, UINT32 uWidth, UINT32 uHeight, UINT32 uInBPP, BYTE *pOut8Bit);

/*******************************************************
	ISCopyBuffer

	Purpose : 
	Copy input buffer to output buffer, adding or removing 
	row padding as specified.

	if uInWidthBytes = uOutWidthBytes, the input buffer is
	copied directly to the output buffer.

	if uInWidthBytes < uOutWidthBytes, each row in the output 
	buffer will have (uOutWidthBytes - uInWidthBytes) BYTEs
	added to the end of it. 
	
	if uInWidthBytes > uOutWidthBytes, each row in the output rows
	will be the first uOutWidthBytes BYTEs of the input row.

	this function can be used to convert an RGB buffer to and from
	a DWORD-aligned buffer.

	the values of any padding BYTEs in the output rows 
	will be whatever data was there when this function was called.
	ie. it is not overwritten.

	Param					Use
	------------------------------------
	pPaddedBuf			source buffer
	uInWidthBytes		BYTEs in an input row
	uHeight 			height (of all buffers)
	uOutWidthBytes		BYTEs in an output row
	pOutputBuf			output buffer : (uOutWidthBytes * uHeight)
						BYTEs.

	Return
	------
	FALSE on failure
*******************************************************/
extern BOOL	_ISEXP_ _ISCopyBuffer(BYTE *pInputBuf, UINT32 uInWidthBytes, UINT32 uHeight, UINT32 uOutWidthBytes, BYTE *pOutputBuf); 


#ifdef __cplusplus
}
#endif


#endif