//
//                  ooo        ooooo
//                  `88.       .888'
//                   888b     d'888   .oooo.   oooo    ooo
//                   8 Y88. .P  888  `P  )88b   `88b..8P'
//                   8  `888'   888   .oP"888     Y888'
//                   8    Y     888  d8(  888   .o8"'88b
//                  o8o        o888o `Y888""8o o88'   888o
//
//                          Across the event horizon...
//
//                       Microsoft Windows 95/98/NT Version
//
//  Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Dan Higdon
//
// maxdevs.hpp
//
// Event Devices
//
//      This module defines the physical devices used by the event
//  system.  Each device has a unique interface, suited to the device.
//  Only three devices are currently supported.
//
//  Keyboard -- The keyboard device.  This is not a serial device, but
//          rather a matrix of on/off switches corresponding to the
//          individual keys.  Key indexes are scan codes, and "extended"
//          keys are the same as their scan codes + 256.
//  Mouse -- The mouse is a multi-button positional device.  The mouse
//          can be queried for its button's states, as well as its current
//          position, and delta movement.
//  Joystick -- The standard IBM Joystick.  Like the mouse, this device
//          can only be queried for its current state.
//
//  Additionally, a new class has been added to "contain" those devices, and
//  maintain a "current state", so that multiple event systems can query
//  the devices, and devices (such as the mouse) whose states depend
//  on how recently they were queried will return meaningful values.
//
//

#ifndef __MAXDEVS_HPP
#define __MAXDEVS_HPP    1

#ifdef _MSC_VER
#pragma pack(push,1)
#endif


//
//
//                                Includes
//
//

#include <portable.h>
#include <mmsystem.h>
#include <dinput.h>

//
//
//                                Pragmas
//
//

//
//
//                                Equates
//
//

//
//
//                               Structures
//
//

//
//
//                                Classes
//
//

class MaxMouse;
class MaxKeyboard;
class MaxJoystick;

//
// MaxDevices - ABSTRACT BASE CLASS:
//  The set of devices currently supported by Max.
//  Each devices may be enabled/disabled individually.  This interface is
//  the base class for talking to Max devices.
//
class MaxDevices
{
protected:
    //
    // Protected Data
    //

    // Mouse state info
    short   snapshot_mx;
    short   snapshot_my;
    short   snapshot_mdx;
    short   snapshot_mdy;
    byte    snapshot_mbuttons;

    // Joystick state info
    JOYINFOEX   snapshot_jstat;

public:
    //
    // Public Data
    //
    MaxMouse    *theMouse;
    MaxKeyboard *theKeyboard;
    MaxJoystick *theJoystick;

    //
    // Construtor/Destructor
    //
    MaxDevices();
    virtual ~MaxDevices();

    //
    // Operations
    //
    virtual int enable_keyboard (int);
    virtual int enable_mouse    (int);
    virtual int enable_joystick (int);

    // Update the internal values for the various devices
    virtual void update ();
    virtual void wm_clear ();

    // Values for the devices
    virtual void get_mouse_movement (short *, short *) const;
    virtual void get_mouse_pos (short *, short *) const;
    virtual void set_mouse_pos (short, short);
    virtual byte get_mouse_buttons () const;

    // Keyboard device
    virtual int  check_keyboard_key (int) const;

    // Joystick device
    virtual void get_joystick (JOYINFOEX *) const;

    // Device operations (spread over all devices)
    void init ();
    void uninit ();

    //
    // Accessors
    //
    bool is_mouse_present() const;
    bool is_keyboard_present() const;
    bool is_joystick_present() const;
    bool is_mouse_enabled () const;
    bool is_keyboard_enabled () const;
    bool is_joystick_enabled () const;
};


//
// MaxDevicesWin32:
//  This implementation of the MaxDevices interface uses the "standard"
//  Win32 interface to provide all input.  This interface requires some
//  "extra" work, such as feeding data in from WindowProcs, but it should
//  work under any circumstances.  The MaxDevicesDirectX class is suggested
//  for any realtime applications.
//
class MaxDevicesWin32: public MaxDevices
{
private:
    // The client's window handle
    HWND    hwndClient;

public:
    //
    // Construtor/Destructor
    //
    MaxDevicesWin32 (HWND);
    virtual ~MaxDevicesWin32();

    //
    // Operations
    //

    // Update the internal values for the various devices
//    virtual void update ();

    // Used to clear any special states when using the WM_* pump
    // Must be called if wm_ device members are used.
//    virtual void wm_clear ();
};


//
// MaxDevicesDirectX:
//  This implementation of the MaxDevices interface uses the DirectInput
//  interface to provide input.  This is more efficient, but may cause some
//  interaction problems, due to its reliance on DirectInput.
//
class MaxDevicesDirectX: public MaxDevices
{
    //
    // Private Data
    //
    IDirectInput    *pdi;
    HWND            hwndClient;

public:
    //
    // Construtor/Destructor
    //
    MaxDevicesDirectX (HWND);
    virtual ~MaxDevicesDirectX();

    //
    // Operations
    //

    // Update the internal values for the various devices
//    virtual void update ();

    // Used to clear any special states when using the WM_* pump
    // Must be called if wm_ device members are used.
//    virtual void wm_clear ();
};


//
//
//                               Prototypes
//
//

#ifdef _MSC_VER
#pragma pack(pop)
#endif

#endif

// End of header - maxdevs.hpp 
