/*
**  ECHO_S.C
**
**  Win32 CONSOLE mode "ECHO" server.
**
**  This is the standard "echo" diagnostic server.
**  Once the client connects, everthing sent to it
**  is echoed back until the client closes the connection.
**
**  Multiple simultaneous connections can be handled.
**  Connect using the client ECHO_C.
**
*/

#include <windows.h>
#include <winsock.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "wil.h"
#include "str.h"
#include "cons.h"

#define MAX_STR   50
#define MAX_BUF  256
#define ONE_SEC 1000

#define ECHO_PORT  7
#define REQUESTS   4


static char Temp[MAX_BUF+8];
static char InBuffer[MAX_BUF+1];
static char HostName[MAX_BUF+1];
static ULONG  HostAddr = 0;
static SOCKET ListenSock = 0;
static SOCKET DataSock = 0;
static int    Sessions = 0;
static SOCKET SockList[REQUESTS];
static HANDLE ConIn;
static HANDLE ConOut;
static DWORD  OldMode;
static DWORD  NewMode;

/* display error */

static void DisplayError(int Code, char * Msg)
{PutString(ConOut ,"ERROR: ");
 if(Msg) PutString(ConOut,Msg);
 if(Code)
   {wilErrorText(Code,(char *)Temp,MAX_BUF);
    PutLine(ConOut,(char *)Temp);
   }
}

/* display error & exit (before calling wilAttach) */

void SayError(char *Msg)
{printf("ERROR: %s: %d\n", Msg, GetLastError() );
 exit(1);
}

void ShutDown(void)
{PutLine(ConOut,"Shutting down...");
 wilRelease();
 SetConsoleMode(ConIn, OldMode);
 exit(0);
}

void DisplayString(char *String)
{
 PutString(ConOut, String);
}

void DisplayLine(char *String)
{
 PutLine(ConOut, String);
}

void DisplayChar(char Chr)
{
 PutChar(ConOut, Chr);
}

/* find available session slot */

int GetSessionSlot(void)
{int i;
 for(i=0;i<REQUESTS;i++)
   if(SockList[i]==(SOCKET)0) return i;
 return -1;
}

/*** main ***/

void main(void)
{int  i, n;
 int  Code;
 BOOL Success;
 /* init session socket list */
 for(i=0;i<REQUESTS;i++) SockList[i] = (SOCKET)0;
 /* get handles for stdin & stdout */
 ConIn  = GetStdHandle(STD_INPUT_HANDLE);
 ConOut = GetStdHandle(STD_OUTPUT_HANDLE);
 /* modify console mode: disable line input & line echo */
 Success = GetConsoleMode(ConIn, &OldMode);
 if(!Success) SayError("GetConsoleMode");
 NewMode = OldMode & ~ENABLE_LINE_INPUT & ~ENABLE_ECHO_INPUT;
 Success = SetConsoleMode(ConIn, NewMode);
 if(!Success) SayError("SetConsoleMode");
 /* attach WINSOCK */
 PutString(ConOut,"Attaching WINSOCK...");
 Code = wilAttach();
 PutLine(ConOut,"OK");
 if(Code<0) DisplayError(Code,"wilAttach fails:");
 else
   {PutString(ConOut," Description: ");
    wilGetDescription((char *)Temp, MAX_BUF);
    PutLine(ConOut,(char *)Temp);
    PutString(ConOut,"My Host Name: ");
    wilGetMyHostName((char *)Temp, MAX_BUF);
    PutLine(ConOut,(char *)Temp);
    PutString(ConOut,"My Host Addr: ");
    wilGetMyHostDotted(0,(char *)Temp, MAX_BUF);
    PutLine(ConOut,(char *)Temp);
   }
 PutLine(ConOut,"\r\nStarting WHOIS Server: Type any key to exit...");
 /* get local host domain name */
 if(!wilGetMyHostName((char *)HostName, MAX_BUF))
   {DisplayError(0, "Cannot get local host name");
    ShutDown();
   }
 /* get local host IP address */
 HostAddr = wilGetHostAddr((char *)HostName,0);
 if(HostAddr==0)
   {DisplayError(0, "Cannot get local IP addess");
    ShutDown();
   }
 /* create listener socket */
 ListenSock = wilTcpSocket();
 if((int)ListenSock<=0)
   {DisplayError((int)Code, "wilListen:");
    ShutDown();
   }
 /* bind address & port number */
 Code = wilBind(ListenSock, HostAddr, ECHO_PORT);
 if(Code<=0)
   {DisplayError((int)Code, "wilBind:");
    ShutDown();
   }
 /* 'REQUESTS' = # concurrent sessions */
 Code = wilListen(ListenSock,REQUESTS);
 if(Code<=0)
  {DisplayError(Code, NULL);
   ShutDown();
  }
#if 1
 wsprintf((char *)Temp,"ListenSock=%d MyHostAddr=%lx Port=%d",
                      ListenSock, HostAddr, ECHO_PORT);
 DisplayLine((char *)Temp);
#endif
 DisplayLine("Listening [type any key to quit]...");

 /* wait for connection attempt */
 while(TRUE)
   {/* any key press shuts us down */
    if(KeyPress(ConIn)) ShutDown();
    /* any incoming connection ? */
    if(wilDataIsReady(ListenSock,0))
      {DisplayLine("Connection attempt...");
       /* accept new connection */
       DataSock = wilAccept(ListenSock,500);
       if((int)DataSock<=0) continue;
       /* get session number (index into SockList[])*/
       n = GetSessionSlot();
       if(n==-1) continue;
       wsprintf((LPSTR)Temp,"Connection %d is accepted",n);
       DisplayLine((char *)Temp);
       /* save socket number */
       SockList[n] = DataSock;
       Sessions++;
      }
    /* process all open connections */
    for(i=0;i<REQUESTS;i++)
      {/* get data sock for this session */
       DataSock = SockList[i];
       /* session in progress ? */
       if(DataSock!=(SOCKET)0)
         {/* is data ready on open connection ? */
          if(wilDataIsReady(DataSock,0))
            {/* read socket */
             Code = wilReadString(DataSock,(char *)InBuffer,MAX_BUF);
             if(Code==WIL_EOF)
               {/* client has closed connection */
                wilCloseSocket(DataSock,500);
                SockList[i] = (SOCKET)0;
                wsprintf((LPSTR)Temp,"Connection %d closed",i);
                DisplayLine((char *)Temp);
                Sessions--;
               }
             else
               {/* echo back to client */
                Code = wilWriteString(DataSock,(char *)InBuffer);
                /* display here too */
                wsprintf((LPSTR)Temp,"%d: %s", i, InBuffer);
                DisplayLine((char *)Temp);
               }
            }
         } /*end: if(DataSock!=(SOCKET)0)*/
      }
   } /*end: while(TRUE)*/
}


