// =====================================================================
// talk.c - A talk client for MS-DOS. Uses the WATTCP library.
//
// (C) 1993, 1994 by Michael Ringe, Institut fuer Theoretische Physik,
// RWTH Aachen, Aachen, Germany (michael@thphys.physik.rwth-aachen.de)
//
// This program is free software; see the file COPYING for details.
// =====================================================================

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\include\tcp.h"
#include "talk.h"


// ---------------------------------------------------------------------
// Global data
// ---------------------------------------------------------------------

char version[] =
    "Talk version " VERSION ", written by Michael Ringe\r\n";
char usage[] =
"NAME\r\n"
"    Talk version " VERSION "\r\n"
"\r\n"
"USAGE\r\n"
"    talk [-lo] user@{host|alias} [tty]  Talk to another user\r\n"
"    talk [-aq]  Wait for talk requests\r\n"
"\r\n"
"OPTIONS\r\n"
"    -l   Log the talk session to " LOGFILENAME "\r\n"
"    -o   Use old-style talk protocol\r\n"
"    -a   Enter answer mode\r\n"
"    -q   Be quiet in answer mode\r\n"
"\r\n"
"AUTHOR\r\n"
"    (C) Copyright 1994 by Michael Ringe\r\n"
"    This program is free software; see the file COPYING for details.\r\n"
;

char aamsg1[] = "Hello %s, %60s\n";
char *aamsg2[11] =
	{"This talk is running in answer mode.",
	 "If you don't mind talking to a machine you may leave",
	 "a message. Finish by closing the talk connection.",
	 "Your message will be logged and read later.",
	 "-- %s",
	 NULL
	};

char
    *userarg = NULL,		// Command line arg, NULL = wait
    localuser[20] = "PC-User",	// My user name
    remoteuser[20] = "",	// Remote user name
    *remotehost = NULL;		// Remote host name
int answermode = 0;		// Operate as answering machine
int opt_o = 0;			// Use old protocol
tcp_Socket ds;			// Talk socket
longword remoteip = 0;		// Remote IP address
char remotetty[20] = "";	// Terminal to connect to


// ---------------------------------------------------------------------
// Initialization
// ---------------------------------------------------------------------

static void err_usage(void)

{
    cputs(usage);
    exit(1);
}

static void my_usr_init(char *key, char *val)

{
    if (*val == '$')
    {
#if (DEBUG)
	char *old = val;
#endif
	val = getenv(strupr(val+1));
	DB_printf(("%s: '%s' expands to '%s'\n",key,old,val));
	if (val == NULL) return;	// Use default
    }
    if (!stricmp(key,"talk_localname"))
    {
	mystrncpy(localuser,val,sizeof(localuser));
    }
    else if (!stricmp(key,"talk_screenmode"))
    {
	if (!strnicmp(val,"split",5))
	    scrnmode = 1;
	if (strstr(val,"autocr") != NULL)
	    autocr = 1;
    }
    else if (!stricmp(key,"talk_colors"))
    {
	unsigned short *x = a[0];
	int i,j;

	for (j = 0; j < 2 && *val != 0; ++j)
	{
	    for (i = PAL_SIZE; i > 0; --i)
	    {
		*x++ = xtoi(val);
		if (*++val != 0) ++val;
	    }
	    if (*val != 0) ++val;
	}
	if (j == 1)
	    memcpy(a+1,a,PAL_SIZE*sizeof(short));
    }
    else if (!stricmp(key,"talk_alias"))
    {
	char *c = strchr(val,':');
	if (c != NULL)
	{
	    *c++ = 0;
	    if (userarg != NULL && !strcmp(val,userarg))
		userarg = strdup(c);
	}
    }
    else if (!stricmp(key,"talk_logfile"))
    {
	logfilename = strdup(val);
    }
    else if (!stricmp(key,"talk_message"))
    {
	static int n = 0;

	if (n < 10)
	{
	    aamsg2[n++] = strdup(val);
	    aamsg2[n] = NULL;
	}
    }
}

static int init(int argc, char *argv[])

{
    int i;
    char *c;

    init_video();

    // Process options
    for (i = 1; i < argc && *(c = argv[i]) == '-'; ++i)
    {
	while (*++c != 0)
	{
	    if (*c == 'a') answermode = 1;
	    else if (*c == 'l') log = 1;
	    else if (*c == 'q') quiet = 1;
	    else if (*c == 'o') opt_o = 1;
	    else err_usage();
	}
    }
    if (!answermode) quiet = 0;	// Ignore -q if not in answer mode

    // Process remaining arguments
    switch (argc - i)
    {
	case 2:
	    mystrncpy(remotetty,argv[i+1],sizeof(remotetty));
	case 1:
	    userarg = argv[i];
	    if (answermode) err_usage();
	case 0:
	    break;
	default: err_usage();
    }

    // Start up wattcp
    usr_init = my_usr_init;
    sock_init();

    // Enter message
    if (answermode)
    {
	int n;
	char tmp[100];
	userarg = NULL;
	log = 1;
	cputs("Enter message (up to 10 lines, end with a "
		"single '.'),\r\n");
	cputs("or just press <ENTER> for a default message:\r\n");
	for (n = 0; n < 10; ++n)
	{
	    *tmp = 79;
	    cgets(tmp);
	    cputs("\r\n");
	    if (tmp[1] == 1 && tmp[2] == '.' || n == 0 && tmp[1] == 0)
		break;
	    aamsg2[n] = strdup(tmp+2);
	    aamsg2[n+1] = NULL;
	}
    }

    // Split argument into user and host name
    if (userarg != NULL)
    {
	char *c = strchr(userarg,'@');
	if (c == NULL) err_usage();
	*c++ = 0;
	mystrncpy(remoteuser,userarg,sizeof(remoteuser));
	remotehost = c;

	// Look up remote IP address
	if (!(remoteip = resolve(remotehost)))
	{
	    cprintf("Error: Cannot resolve '%s'\r\n",remotehost);
	    return 1;
	}
	DB_printf(("Remote IP: %d.%d.%d.%d\n",DB_ipsplit(remoteip)));

    }
    DB_printf(("Local user: %s\nRemote user: %s\nRemote host: %s\n",
	localuser,remoteuser,remotehost));
    return 0;
}

// ---------------------------------------------------------------------
// exchange_edit_chars() - Exchange the three edit characters with the
//                         remote talk.
// ---------------------------------------------------------------------

static void exchange_edit_chars(void)

{
    sock_write(&ds,my_ec,3);
    sock_read(&ds,his_ec,3);
    DB_printf(("Remote edit characters: ERASE=%x, KILL=%x, WERASE=%x\n",
	    his_ec[0],his_ec[1],his_ec[2]));
}


// ---------------------------------------------------------------------
// talk() - This function does the actual talking
// ---------------------------------------------------------------------

static int talk(void)

{
    static char buf[500];
    unsigned short ch;
    int i;
    int inactive = 0;

#if (!NONET)
    exchange_edit_chars();
#endif
    DB_puts("Entering talk(), press return:");
#if DEBUG
    gets(buf);
#endif


    if (!quiet)
    {
	sound(2000);
	delay(100);
	nosound();
	init_screen();
    }
    cprintf("Connected to %s@%s at %s.\r\n",remoteuser,remotehost,
	dostime());

    // If in answer mode, write the message to the remote screen
    // Using sock_printf here would use another 3kB memory.
    if (answermode)
    {
	int n;
	sprintf(buf,aamsg1,remoteuser,dostime());
	sock_puts(&ds,buf);
	for (n = 0; n < 10 && aamsg2[n] != NULL; ++n)
	{
	    sprintf(buf,aamsg2[n],localuser);
	    sock_puts(&ds,buf);
	    sock_putc(&ds,'\n');
	}
    }

    // Main loop, repeat until ESC pressed or socket closed
    while (tcp_tick(&ds))
    {

	// Process keyboard input
	if ((ch = readkey()) != 0)
	{
	    if (ch == ESC) return 1;	// User break
	    inactive = 0;

	    // If it's a normal character, write it to both
	    // local screen and remote host.
	    if (ch < 0x100)
	    {
		select(0);
		myputch(ch);

		// Translate edit characters
		for (i = 0; i < EC_SIZE; ++i)
		    if (ch == my_ec[i])
		    {
			ch = his_ec[i];
			break;
		    }
		sock_putc(&ds,ch);
	    }
	}

	// Process input from remote host
	i = sock_fastread(&ds,buf,sizeof(buf));
#if (NONET)
	i = 0;
	if (ch == ALT_C)
	{ i = 17;
	  strcpy(buf," 123 4567 199 8xxx");
	  ch = 0;
	}
#endif
	if (i > 0)
	{
	    int k;
	    select(1);
	    inactive = 0;
	    for (k = 0; k < i; ++k)
		myputch(buf[k]);
	}
	if (!inactive)
            ip_timer_init((udp_Socket *)&ds,answermode ?
		ANSWERMODE_TIMEOUT : TIMEOUT);
	inactive = 1;
	if (ip_timer_expired(&ds))
	{
	    sock_puts(&ds,"time-out");
	    return 1;
	}
    }
    return 2;		// Closed by remote party
}


// ---------------------------------------------------------------------
// main()
// ---------------------------------------------------------------------

void main(int argc, char *argv[])

{
    int status;

    if (init(argc,argv))
	exit(1);

#if (NONET)
    talk();
    openlog(0);
    cleanup(1);
    exit(0);
#endif

    if (userarg != NULL)
    {
	if (opt_o)
	    status = o_invite();
	else
	{
	    status = invite();
	    if (status == 1) status = o_invite();
	}
        if (status != 0 && status != 2) exit(1);
    }

    do
    {
	if (status != 2) 
	    status = wait_invite();
 	if (status == 27) break;
	if (status == 0 || status == 2)
	{
	    if (log) openlog(1);
	    status = talk();
	    openlog(0);
	    sock_abort(&ds);
	    sock_close(&ds);
	    cleanup(status);
	    status = 0;
	}
    }
    while (answermode);
    exit(0);
}


