; MODSTUF4.ASM
;
; This is the fourth part of the assembler "half" of the player module.
; Here we handle time-critical things like actual resampling and mixing of
; mod samples.  This file is broken into four parts to get around the
; assembler's 64k limit on source files.
;
; PATTERN, ROW, AND TICK ROUTINES ************************************
;
; These routines should leave DF clear.
;
; DOPAT routine, processes a new pattern table entry.  On entry, BX is the
; new table entry.
;
; Returns carry set if at the end of the song, carry clear otherwise.
; Destroys AX, BX, CX, DX.
;
	EVEN
DOPAT:
	;
	; If the new pattern is off the end of the pattern table, set the
	; new pattern to the song end jump position.
	;
	CMP	BX,_pattablesize
	JB	DOPAT_INTABLE
	MOV	BX,_songendjump
	;
	; If we set the new pattern to the song end jump position, verify
	; that the song end jump position is in fact in the table.  If not,
	; the mod is not looped, and we're at the end of it.
	;
	CMP	BX,_pattablesize
	JB	DOPAT_INTABLE
DOPAT_SONGEND:
	STC				; CF set = end of song
	RET
	;
	; We have a valid pattern table entry.  If we've been here before,
	; and if looping is disabled, stop playback.
	;
	EVEN
DOPAT_INTABLE:
	MOV	AL,_visiteds[BX]
	AND	AL,_loopdisable
	JNZ	DOPAT_SONGEND
	;
	; Playback is continuing.  Get set up for the new pattern.
	;
	MOV	_currentpat,BX		; current pattern = next pattern
	MOV	_nextpat,BX		; (may have been changed above)
	OR	BX,BX			; are we at pattern 0?
	JNZ	DOPAT_SETVISITED
	;
	; If at pattern table entry 0 (beginning of song), set speed and
	; global volume to the default.
	;
	PUSH	BX
	XOR	BH,BH			; set beats per minute to the default
	MOV	BL,_bpmdefl
	MOV	_beatspermin,BL
	SHL	BX,1
	MOV	AX,_samprate		; samples per tick = (5*samprate)
	MOV	DX,5			;   / (2*beatspermin)
	MUL	DX
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_samplespertick,AX
	MOV	AX,_ticksdefl		; set ticks per division to the default
	MOV	_ticksperrow,AX
	MOV	AL,_globalvoldefl	; set global volume to the default
	MOV	_globalvol,AL
	POP	BX
	;
	; Mark this pattern table entry visited, get the segment address of
	; the pattern data, and set the pattern loop start and loop count to
	; 0.
	;
DOPAT_SETVISITED:
	MOV	_visiteds[BX],1		; mark pattern visited
	SHL	BX,1			; get pattern number
	MOV	BX,_patterntable[BX]
	CMP	BX,254			; end of song if pattern # 255 (.s3m)
	JA	DOPAT_SONGEND
	JE	DOPAT_SKIPPAT		; skip to next if pattern # 254 (.s3m)
	CMP	BX,_nfilepatterns	; invalid if no such pattern
	JAE	DOPAT_SONGEND
	SHL	BX,1			; set pattern buffer segment address
	MOV	AX,_patternsegs[BX]
	MOV	_currentpatseg,AX
	MOV	_patloopstart,0		; set pattern loop start to 0
	MOV	_patloopcount,0		; set pattern loop count to 0
	CLC				; CF clear = not at end of song
	RET
	;
	; Pattern number 255, in an .s3m file, is just a place holder to
	; skip over.
	;
	EVEN
DOPAT_SKIPPAT:
	MOV	BX,_nextpat
	INC	BX
	JMP	SHORT DOPAT
;
; DOROW routine, processes a new row (division) in the pattern.
;
; Returns carry set if at the end of the song, carry clear otherwise.
; Destroys AX, BX, CX, DX, SI, DI, ES.
;
	EVEN
DOROW:
	MOV	BX,_nextpat		; see if new pattern
	CMP	BX,_currentpat
	JE	DOROW_OLDPAT
	CALL	DOPAT			; new pattern, process it (destroys
	JNC	DOROW_OLDPAT		;   AX, BX, CX, DX)
	RET				; end of song (CF already set)
	;
	; Playback is continuing, and we have _nextpat = _currentpat (if
	; we didn't before, DOPAT would set them equal).  Set the current
	; row equal to the next row, and determine the next pattern and
	; next row.
	;
	EVEN
DOROW_OLDPAT:
	MOV	AL,_nextrow
	MOV	_currentrow,AL
	INC	AL			; next = current + 1
	CMP	AL,64			; past end of pattern?
	JB	DOROW_ROWOK
	XOR	AL,AL			; next row is first row in new pattern
	INC	_nextpat
DOROW_ROWOK:
	MOV	_nextrow,AL		; save next row
	;
	; Loop over the channels, processing new notes, new samples, and
	; new effects.
	;
	MOV	CX,_nchannels		; CX is loop count
	MOV	BX,OFFSET _channelrecs	; BX addresses channel data
	MOV	AL,_currentrow
	MUL	_bytesperrow
	MOV	DI,AX			; ES:DI addresses note record
	MOV	ES,_currentpatseg
	;
	; Determine whether there is a new note on the channel.
	;
DOROW_CHANNELLOOP:
	MOV	NEWNOTE,0		; assume no new note
	CMP	ES:[DI].NOTEPERIOD,0	; if period = 0, no new note
	JE	DOROW_EFFEND
	JL	DOROW_STOPNOTE		; period < 0 means stop note now
	MOV	NEWNOTE,1		; period > 0, looks like new note
	MOV	AL,ES:[DI].NOTEEFFNUM	; get effect in AL
	CMP	AL,3			; if effect < 3, new note
	JB	DOROW_EFFEND
	JE	DOROW_NOTNEW		; if effect = 3, not a new note
	CMP	AL,5			; if effect = 4, new note
	JB	DOROW_EFFEND
	JE	DOROW_NOTNEW		; if effect = 5, not a new note
	CMP	AL,23			; if effect > 5 and effect <> 23, new
	JNE	DOROW_EFFEND		;   note
DOROW_NOTNEW:
	MOV	NEWNOTE,0		; effect is 3, 5 or 23, not a new note
	JMP	SHORT DOROW_EFFEND
	;
	; A period value of -1 (from an .s3m file) indicates that the current
	; note is to end, but a new note is not to begin.  Here we just mark
	; the sample as played through.
	;
	EVEN
DOROW_STOPNOTE:
	MOV	[BX].CHANSAMPDONE,1
	MOV	[BX].CHANLASTSAMP,0
	;
	; If not a new note, call the effect end routine for the channel.
	;
DOROW_EFFEND:
	CMP	NEWNOTE,1		; skip it if new note
	JE	DOROW_CHKSAM
	CALL	WORD PTR [BX].CHANEFFENDPROC
	;
	; Check for a new sample on the channel.
	;
DOROW_CHKSAM:
	MOV	AL,ES:[DI].NOTESAMPNUM	; get sample number in AX
	OR	AL,AL			; if sample number is 0, no new sample
	JZ	DOROW_CHKVOL
	CMP	AL,BYTE PTR _ninstruments ; if number is invalid, ignore it
	JA	DOROW_CHKVOL
	MOV	AH,TYPE SAMPREC		; new sample - set pointer in channel
	MUL	AH			;   data
	ADD	AX,OFFSET _samplerecs
	MOV	[BX].CHANSAMPPTR,AX
	MOV	SI,AX			; SI addresses data for sample
	MOV	AL,[SI].SAMPVOL		; set main volume on channel to sample
	MOV	[BX].CHANMAINVOL,AL	;   default
	CMP	NEWNOTE,1		; check if new note
	JE	DOROW_CHKVOL		; new sample w/o new note (!):
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume on
	MOV	[BX].CHANMIXVOL,AL	;   channel to sample default
	MOV	AX,[SI].SAMPC4SPD	; set C4SPD on channel to sample
	MOV	[BX].CHANC4SPD,AX	;   default
	MOV	AX,[SI].SAMPSEG		; set current sample segment on channel
	MOV	[BX].CHANSAMPSEG,AX	;   to segment of sample data
	XOR	AX,AX			; set offset in sample buffer to 0
	MOV	[BX].CHANSAMPFRAC,AX
	MOV	[BX].CHANSAMPOFFS,AX
	MOV	[BX].CHANSAMPDONE,AL	; set sample done flag on channel to 0
	MOV	[BX].CHANSAMPHALF,AL	; set 64k half of EMS sample to first
	;
	; If there is a volume setting on the channel, use it (S3M).
	;
DOROW_CHKVOL:
	MOV	AL,ES:[DI].NOTEVOL	; get note volume in AL
	CMP	AL,255			; if 255, no volume set
	JE	DOROW_CHKNOTE
	MOV	[BX].CHANMAINVOL,AL
	CMP	NEWNOTE,1		; check if new note
	JE	DOROW_SETNOTE		; volume set w/o new note:
	MOV	[BX].CHANEFFVOL,AL	; set effective and mixing volume on
	MOV	[BX].CHANMIXVOL,AL	;   channel to sample default
	JMP	SHORT DOROW_DOEFFECT
	;
	; Process a new note on the channel, if there is one.
	;
	EVEN
DOROW_CHKNOTE:
	CMP	NEWNOTE,0		; skip it if not a new note
	JE	DOROW_DOEFFECT
	MOV	AL,[BX].CHANMAINVOL	; set effective and mixing volume on
DOROW_SETNOTE:
	MOV	[BX].CHANEFFVOL,AL	;   channel to main volume
	MOV	[BX].CHANMIXVOL,AL
	MOV	AX,ES:[DI].NOTEPERIOD	; set effective period to note period
	MOV	[BX].CHANEFFPERIOD,AX
	MOV	SI,[BX].CHANSAMPPTR	; set segment to sample segment
	MOV	AX,[SI].SAMPSEG
	MOV	[BX].CHANSAMPSEG,AX
	MOV	AX,[SI].SAMPC4SPD	; set C4SPD on channel to sample
	MOV	[BX].CHANC4SPD,AX	;   default
	XOR	AX,AX			; set offset in sample buffer to 0
	MOV	[BX].CHANSAMPFRAC,AX
	MOV	[BX].CHANSAMPOFFS,AX
	MOV	[BX].CHANSAMPDONE,AL	; set sample done flag on channel to 0
	MOV	[BX].CHANSAMPHALF,AL	; set 64k half of EMS sample to first
	MOV	[BX].CHANDELAYCOUNT,AL	; set delay count to 0
	PUSH	BX			; save BX, CX, DI
	PUSH	CX
	PUSH	DI
	MOV	CX,[BX].CHANEFFPERIOD	; CX = period
	MOV	BX,[BX].CHANC4SPD	; BX = C4SPD
	CALL	GETSTEP			; DX.AX = sample step (BX, CX, DI gone)
	POP	DI			; restore DI, CX, BX
	POP	CX
	POP	BX
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step
	MOV	[BX].CHANSTEPINT,DX
	;
	; Process the effect on the channel.
	;
DOROW_DOEFFECT:
	MOV	[BX].CHANEFFTICKPROC,OFFSET DONOTHING ; assume no effect
	MOV	[BX].CHANEFFENDPROC,OFFSET DONOTHING
	MOV	AL,ES:[DI].NOTEEFFNUM	; AL = effect number
	CMP	AL,NEFFECTS		; skip if out of range
	JAE	DOROW_CHANLPEND
	MOV	DL,ES:[DI].NOTEEFFPARM	; DL = effect parameter
	OR	DL,AL			; if both zero, no effect
	JZ	DOROW_CHANLPEND
	XOR	AH,AH			; call effect setup routine
	SHL	AX,1
	MOV	SI,AX
	CALL	WORD PTR EFSETUPS[SI]
DOROW_CHANLPEND:
	ADD	DI,TYPE NOTEREC		; go to next channel in row
	ADD	BX,TYPE CHANREC		; go to next channel data record
	DEC	CX			; > 128 bytes, can't use LOOP
	JZ	DOROW_CHKSPEED
	JMP	SHORT DOROW_CHANNELLOOP
	;
	; All channels have been set up.  Check whether there was a set
	; speed effect, and if so, process it now.
	;
	EVEN
DOROW_CHKSPEED:
	CMP	_setspeedparm,0		; was there a set speed effect?
	JE	DOROW_SETTICKSLEFT	; skip it if not
	XOR	BL,BL			; get parameter in BL, set to 0 for
	XCHG	BL,_setspeedparm	;   next time
	CMP	BL,MAXTICKS		; if parameter <= cutoff, set ticks/row
	JA	DOROW_SETBEATS		;   (different people disagree on the
	XOR	BH,BH			;   cutoff point here - this is how
	MOV	_ticksperrow,BX		;   Multiplayer does it)
	JMP	SHORT DOROW_SETTICKSLEFT
	EVEN
DOROW_SETBEATS:
	XOR	BH,BH			; parameter > cutoff, set beats/min
	MOV	_beatspermin,BL
	SHL	BX,1
	MOV	AX,_samprate		; samples per tick = (5*samprate)
	MOV	DX,5			;   / (2*beatspermin)
	MUL	DX
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_samplespertick,AX
	;
	; Set the number of ticks remaining (plus 1) to the number of ticks
	; per division.
	;
DOROW_SETTICKSLEFT:
	MOV	AX,_ticksperrow
	MOV	_ticksleft,AX
	;
	; Check whether there was a pattern delay effect, and if so, process
	; it now.
	;
	CMP	_patdelayparm,0
	JE	DOROW_DONE
	XOR	AL,AL			; get parameter in AL, set to 0 for
	XCHG	AL,_patdelayparm	;   next time
	MUL	BYTE PTR _ticksperrow
	ADD	_ticksleft,AX
	;
	; All done.
	;
DOROW_DONE:
	CLC				; not end of song yet
	RET
;
; DOTICK routine, fills the partial-mix buffer with one tick's worth of
; samples.  Returns with carry clear if successful, set if at end of song.
;
; Destroys every register except DS.  Must be called with DF clear for
; incrementing, returns with DF clear.
;
	EVEN
DOTICK:
	DEC	_ticksleft		; see if end of division
	JNZ	DOTICK_DOCHANS
	CALL	DOROW			; new division, process it (destroys
	JNC	DOTICK_DOCHANS		;   AX, BX, CX, DX, SI, DI, ES)
	RET				; end of song (CF already set)
	;
	; Process channels.  Channels 0 and 1, the first left and first
	; right channel respectively, need special treatment.
	;
	EVEN
DOTICK_DOCHANS:
	MOV	BX,OFFSET _channelrecs	; BX addresses channel 0 data
	CMP	[BX].CHANDELAYCOUNT,0	; is channel 0 delayed?
	JNE	DOTICK_CHAN0DELAYED
	PUSH	BX			; save BX, DS
	PUSH	DS
	CALL	RESAMPLEA		; call "first channel" resampler
	POP	DS			; restore DS, BX
	POP	BX
	JMP	SHORT DOTICK_CHAN0DONE
	EVEN
DOTICK_CHAN0DELAYED:			; channel 0 is delayed - fill w/zeros
	MOV	ES,_partmixseg
	XOR	DI,DI
	MOV	AX,DI
	MOV	CX,_samplespertick
DOTICK_FILL0LOOP:
	STOSW
	ADD	DI,2
	LOOP	DOTICK_FILL0LOOP
DOTICK_CHAN0DONE:
	ADD	BX,TYPE CHANREC		; BX addresses channel 1 data
	CMP	[BX].CHANDELAYCOUNT,0	; is channel 1 delayed?
	JNE	DOTICK_CHAN1DELAYED
	PUSH	BX			; save BX, DS
	PUSH	DS
	CALL	RESAMPLEA		; call "first channel" resampler
	POP	DS			; restore DS, BX
	POP	BX
	JMP	SHORT DOTICK_CHAN1DONE
	EVEN
DOTICK_CHAN1DELAYED:			; channel 1 is delayed - fill w/zeros
	MOV	ES,_partmixseg
	XOR	DI,DI
	MOV	AX,DI
	MOV	CX,_samplespertick
DOTICK_FILL1LOOP:
	ADD	DI,2
	STOSW
	LOOP	DOTICK_FILL1LOOP
	;
	; If the delay counts for the rest of the channels are not zero,
	; resample them.
	;
DOTICK_CHAN1DONE:
	ADD	BX,TYPE CHANREC		; BX addresses channel 2 data
	MOV	CX,_nchannels		; loop count = _nchannels - 2
	SUB	CX,2
DOTICK_MORECHANLP:
	CMP	[BX].CHANDELAYCOUNT,0	; is the channel delayed?
	JNE	DOTICK_MORECHANLPEND	; if so, skip it
	PUSH	BX			; save BX, CX, DS
	PUSH	CX
	PUSH	DS
	CALL	RESAMPLEB		; call "subsequent channel" resampler
	POP	DS			; restore DS, CX, BX
	POP	CX
	POP	BX
DOTICK_MORECHANLPEND:
	ADD	BX,TYPE CHANREC		; go to next channel data record
	LOOP	DOTICK_MORECHANLP
	;
	; Call the effect tick routine for each channel.
	;
	MOV	BX,OFFSET _channelrecs	; BX addresses channel data
	MOV	CX,_nchannels		; loop count = _nchannels
DOTICK_EFFLOOP:
	CALL	WORD PTR [BX].CHANEFFTICKPROC
	ADD	BX,TYPE CHANREC		; go to next channel data record
	LOOP	DOTICK_EFFLOOP
	CLC				; not end of song yet
	RET
;
; INITIALIZATION ROUTINE *********************************************
;
; _initmodstuf routine, called at startup to initialize the mod player
; module.  Takes no parameters, uses C calling conventions.
;
; Returns 0 in AX if successful, -1 if not.  Destroys BX, CX, DX, ES.
;
	EVEN
_initmodstuf:
	;
	; Allocate a partial-mix buffer.
	;
#IF M_I286
	PUSH	(PARTMIXSIZE+15)/16
#ELSE
	MOV	AX,(PARTMIXSIZE+15)/16
	PUSH	AX
#ENDIF
	CALL	_getblock		; destroys BX, CX, DX, ES
	ADD	SP,2
	OR	AX,AX
	JZ	_initmodstuf_FAILED
	MOV	_partmixseg,AX
	MOV	BX,OFFSET _channelrecs
	MOV	CX,8
	;
	; Set the partial mix address in the channel structures.
	;
_initmodstuf_PARTMIXLOOP:
	MOV	[BX].CHANPARTOFFS,0	; channels 0, 4, 8, 12, 16, 20, 24
	MOV	[BX].CHANPARTSEG,AX	;   and 28 are on the left
	ADD	BX,TYPE CHANREC
	MOV	[BX].CHANPARTOFFS,2	; channels 1, 5, 9, 13, 17, 21, 25
	MOV	[BX].CHANPARTSEG,AX	;   and 29 are on the right
	ADD	BX,TYPE CHANREC
	MOV	[BX].CHANPARTOFFS,2	; channels 2, 6, 10, 14, 18, 22, 26
	MOV	[BX].CHANPARTSEG,AX	;   and 30 are on the right
	ADD	BX,TYPE CHANREC
	MOV	[BX].CHANPARTOFFS,0	; channels 3, 7, 11, 15, 19, 23, 27
	MOV	[BX].CHANPARTSEG,AX	;   and 31 are on the left
	ADD	BX,TYPE CHANREC
	LOOP	_initmodstuf_PARTMIXLOOP
	;
	; Allocate a buffer for the silence sample.
	;
#IF M_I286
	PUSH	1
#ELSE
	MOV	AX,1
	PUSH	AX
#ENDIF
	CALL	_getblock		; destroys BX, CX, DX, ES
	ADD	SP,2
	OR	AX,AX
	JZ	_initmodstuf_FREEPARTMIX
	MOV	_samplerecs[0].SAMPSEG,AX
	;
	; Initialize the sample to all zeros.
	;
	MOV	ES,AX
	XOR	AX,AX
	MOV	ES:[0],AX
	MOV	ES:[2],AX
	;
	; Set up silence sample.  Everything not set here should be zero
	; and already is.
	;
	MOV	_samplerecs[0].SAMPLEN,2
	MOV	_samplerecs[0].SAMPC4SPD,8363
	;
	; That's it!
	;
	XOR	AX,AX
	JMP	SHORT _initmodstuf_DONE
	EVEN
_initmodstuf_FREEPARTMIX:
	PUSH	_partmixseg
	CALL	_freeblock		; destroys BX, CX, DX, ES
	MOV	_partmixseg,0
_initmodstuf_FAILED:
	MOV	AX,-1
_initmodstuf_DONE:
	RET
;
; INPUT/OUTPUT ROUTINES **********************************************
;
; BYTE2HEX routine, converts a byte to 2 hex digits for display or saving
; to a file.  Takes the byte to convert and a pointer to the string where
; the digits should be stored.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, DX.
;
	EVEN
BYTE2HEX:
	POP	DX		; return address in DX
	POP	AX		; byte value in AX
	MOV	BX,OFFSET HEXDIGITS	; BX addresses hex digit table
	MOV	AH,AL
	AND	AL,0Fh		; AL = low nibble
	XLATB			; convert low nibble to digit
	XCHG	AL,AH		; AH = low digit
#IF M_I286
	SHR	AL,4		; AL = high nibble
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	XLATB			; convert high nibble to digit
	POP	BX		; BX addresses string
	MOV	[BX],AX		; store digits in string
	JMP	DX		; jump to return address
;
; SAMPSTRING routine, converts a sample number to 2 ASCII digits for
; display or saving to a file.  If the number is 0, 2 spaces are returned;
; if the number is greater than the number of instruments in the file, 2
; asterisks are returned; otherwise, the number is returned with a leading
; zero if necessary.  Takes the sample number to convert and a pointer to
; the string where the digits should be stored.  Uses Pascal calling
; conventions.
;
; Returns nothing.  Destroys AX, BX, CX.
;
	EVEN
SAMPSTRING:
	POP	CX		; return address in CX
	POP	AX		; sample number in AL
	OR	AL,AL		; sample number = 0?
	JZ	SAMPSTRING_ZERO
	CMP	AL,_ninstruments ; sample number > _ninstruments?
	JA	SAMPSTRING_TOOHI
	AAM			; divide by 10, AH = quotient, AL = remainder
	ADD	AX,3030h	; convert to ASCII digits
	XCHG	AL,AH		; AL = high digit, AH = low digit
	JMP	SHORT SAMPSTRING_STORE
	EVEN
SAMPSTRING_ZERO:
	MOV	AX,2020h	; two spaces
	JMP	SHORT SAMPSTRING_STORE
	EVEN
SAMPSTRING_TOOHI:
	MOV	AX,2A2Ah	; two asterisks
SAMPSTRING_STORE:
	POP	BX		; BX addresses string
	MOV	[BX],AX		; store digits in string
	JMP	CX		; jump to return address
;
; EFFSTRING routine, converts an effect number and parameter to 3 hex
; digits for display or saving to a file.  If both effect and parameter
; are 0, 3 spaces are returned; if the effect is too large to be displayed
; in a single character (36 or greater), 3 asterisks are returned;
; otherwise, the effect number is followed by the parameter in hex.  Takes
; the effect number, parameter, and a pointer to the string where the digits
; should be stored.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX.
;
	EVEN
EFFSTRING:
	POP	CX		; return address in CX
	POP	DX		; parameter in DL
	POP	AX		; effect number in AL
	MOV	AH,AL		; both zero?
	OR	AH,DL
	JZ	EFFSTRING_ZERO
	CMP	AL,36		; effect number >= 36?
	JAE	EFFSTRING_TOOHI	; too high to display if so
	MOV	BX,OFFSET HEXDIGITS	; BX addresses hex digit table
	XLATB			; convert effect number to digit
	XCHG	AX,DX		; effect number in DL, parameter in AL
	MOV	AH,AL		; AH = high 4 bits, AL = low 4 bits of parm
#IF M_I286
	SHR	AH,4
#ELSE
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
	SHR	AH,1
#ENDIF
	AND	AL,0Fh
	XLATB			; convert low 4 bits of parm to hex digit
	XCHG	AL,AH		; AL = high 4 bits, AH = low 4 bits of parm
	XLATB			; convert high 4 bits of parm to hex digit
	JMP	SHORT EFFSTRING_STORE
	EVEN
EFFSTRING_ZERO:
	MOV	AX,2020h	; three spaces
	MOV	DL,AL
	JMP	SHORT EFFSTRING_STORE
	EVEN
EFFSTRING_TOOHI:
	MOV	AX,2A2Ah	; three asterisks
	MOV	DL,AL
EFFSTRING_STORE:
	POP	BX		; BX addresses string
	MOV	[BX],DL		; store effect number in string
	MOV	[BX+1],AX	; store effect parameter in string
	JMP	CX		; jump to return address
;
; PERIODSTRING routine, converts a period number to a 3-character note
; name and octave number for display or saving to a file.  If the period
; is zero, 3 spaces are returned; otherwise, a string like this is returned:
; "C 2", "D#3".  Takes the period number and a pointer to the location
; where the string should be stored.  Uses Pascal calling conventions.
;
; Returns nothing.
;
	EVEN
PERIODSTRING:
	POP	CX		; return address in CX
	POP	AX		; period number in AX
	PUSH	SI		; save SI, DI for C
	PUSH	DI
	OR	AX,AX		; period = 0?
	JZ	PERIODSTRING_ZERO
	CALL	NEARHALF	; BX = offset of closest period in HALFSTEPS
				;   (AX, DX, SI, DI destroyed)
	SHR	BX,1		; BX = note number 0-167
	MOV	AX,BX		; divide by 12 to get note, octave
	MOV	DL,12
	DIV	DL		; AH = note (0-11), AL = octave (0-13)
	MOV	BL,AH		; BX = position in string table for note
	XOR	BH,BH
	SHL	BX,1
	MOV	DX,NOTENAMES[BX]	; DX = 2 characters for note
	MOV	BX,OFFSET HEXDIGITS	; AL = character for octave number
	XLATB
	JMP	SHORT PERIODSTRING_STORE
	EVEN
PERIODSTRING_ZERO:
	MOV	DX,2020h	; DX = 2 spaces
	MOV	AL,DL		; AL = another space
PERIODSTRING_STORE:
	POP	DI		; restore DI, SI
	POP	SI
	POP	BX		; BX addresses caller's string
	MOV	[BX],DX		; save note name
	MOV	[BX+2],AL	; save octave number
	JMP	CX		; jump to return address
;
; VOLUMESTRING routine, converts a volume to ASCII for display or saving
; to a file.  If the volume is 255, 2 spaces are returned; if between 65
; and 254, 2 asterisks are returned; otherwise, 2 decimal digits are
; returned.  Takes the volume and a pointer to the string where the string
; should be stored.  Uses Pascal calling conventions.
;
; Returns nothing.
;
	EVEN
VOLUMESTRING:
	POP	CX		; return address in CX
	POP	AX		; volume in AL
	CMP	AL,255		; volume = 255?
	JE	VOLUMESTRING_NOVOL
	CMP	AL,64		; volume > 64?
	JA	VOLUMESTRING_TOOHI
	AAM			; divide by 10, AH = quotient, AL = remainder
	ADD	AX,3030h	; convert to ASCII digits
	XCHG	AL,AH		; AL = high digit, AH = low digit
	JMP	SHORT VOLUMESTRING_STORE
	EVEN
VOLUMESTRING_NOVOL:
	MOV	AX,2020h	; two spaces
	JMP	SHORT VOLUMESTRING_STORE
	EVEN
VOLUMESTRING_TOOHI:
	MOV	AX,2A2Ah	; two asterisks
VOLUMESTRING_STORE:
	POP	BX		; BX addresses string
	MOV	[BX],AX		; store digits in string
	JMP	CX		; jump to return address
;
; FIL2MEMNOTES routine, converts note records from a pattern in the file
; into internal format.  Takes 3 parameters, a near pointer to an array
; of notes to be converted, a far pointer to the buffer where the converted
; notes should be stored, and the number of notes to convert.  Uses Pascal
; calling conventions.  This version for mod files sets the volume field
; in the internal note record to 255, which in ScreamTracker means "continue
; previous volume."
;
; Returns nothing.  Destroys AX, BX, CX, DX, ES.
;
	EVEN
FIL2MEMNOTES:
	PUSH	BP		; save caller's BP
	MOV	BP,SP		; BP addresses the parameters
	PUSH	SI		; save caller's SI,DI
	PUSH	DI
	LES	DI,[BP+8]	; ES:DI -> output buffer
	MOV	SI,[BP+6]	; SI -> input buffer
	MOV	CX,[BP+4]	; CX = number of notes to convert
	CLD			; DF set for incrementing
FIL2MEMNOTES_LOOP:
	LODSW			; AX = xxxx xxxx wwww XXXX
	MOV	BL,AL		; BL = wwww XXXX
	AND	BL,0F0h		; BL = wwww 0000
	AND	AL,0Fh		; AX = xxxx xxxx 0000 XXXX
	XCHG	AL,AH		; AX = 0000 XXXX xxxx xxxx
	SHL	AX,1		; AX = 000X XXXx xxxx xxx0
	SHL	AX,1		; AX = 00XX XXxx xxxx xx00
	MOV	DX,AX		; DX = 00XX XXxx xxxx xx00
	LODSW			; AX = ssss tttt yyyy zzzz
	MOV	BH,AL		; BH = yyyy zzzz
#IF M_I286
	SHR	BH,4		; BH = 0000 yyyy
#ELSE
	SHR	BH,1
	SHR	BH,1
	SHR	BH,1
	SHR	BH,1
#ENDIF
	OR	BL,BH		; BL = wwww yyyy
	MOV	BH,255		; BX = 1111 1111 wwww yyyy (volume = 255)
	AND	AL,0Fh		; AX = ssss tttt 0000 zzzz
	XCHG	AL,AH		; AX = 0000 zzzz ssss tttt
	XCHG	AX,BX		; AX = sample number, BX = effect & parameter
	STOSW			; store sample number
	MOV	AX,DX		; AX = period
	STOSW			; store period
	MOV	AX,BX		; AX = effect & parameter
	STOSW			; store effect & parameter
	LOOP	FIL2MEMNOTES_LOOP
	POP	DI		; restore caller's DI,SI
	POP	SI
	POP	BP		; restore caller's BP
	RET	8		; remove parameters from the stack
;
; NOTE2PERIOD routine, converts an octave+note from an .S3M file to a
; period number.  Takes one parameter, the octave+note byte.  Uses Pascal
; calling conventions.
;
; Returns the period value in AX.  Destroys BX, DX.
;
	EVEN
NOTE2PERIOD:
	POP	DX		; get return address in DX
	POP	BX		; get octave+note in BL
	CMP	BL,254		; 255 is an empty note; 254 stops note
	JA	NOTE2PERIOD_EMPTY
	JE	NOTE2PERIOD_STOP
	MOV	AL,BL		; AL = octave number
#IF M_I286
	SHR	AL,4
#ELSE
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
	SHR	AL,1
#ENDIF
	MOV	AH,12		; multiply by 12
	MUL	AH
	XCHG	AX,BX		; BX = octave*12, AL = note number
	AND	AL,0Fh
	CBW			; AX = note number
	ADD	BX,AX		; BX = octave*12 + note
	SHL	BX,1		; BX = index in HALFSTEPS table
	MOV	AX,HALFSTEPS[BX]
	JMP	DX		; jump to return address
	EVEN
NOTE2PERIOD_EMPTY:
	XOR	AX,AX
	JMP	DX
	EVEN
NOTE2PERIOD_STOP:
	MOV	AX,-1
	JMP	DX
;
; SETBEATS routine, sets the beats per minute and adjusts the number of
; samples per tick to match.  Takes one parameter, the requested number of
; beats per minute.  Uses Pascal calling conventions.
;
; Returns nothing.  Destroys AX, BX, CX, DX.
;
	EVEN
SETBEATS:
	POP	CX		; CX is return address
	POP	BX		; BX is parameter
	MOV	_beatspermin,BL
	XOR	BH,BH
	SHL	BX,1
	MOV	AX,_samprate	; samples per tick = (5*samprate)
	MOV	DX,5		;   / (2*beatspermin)
	MUL	DX
	DIV	BX
	SHR	BX,1
	CMP	BX,DX
	ADC	AX,0
	MOV	_samplespertick,AX
	JMP	CX		; jump to return address
;
; OUTPUT SAMPLE GENERATOR ********************************************
;
; _getsamples() function.  This is the main entry point to the file module.
; It fills a 2k DMA buffer segment with fully-mixed samples, returning 0 if
; there is another segment full of sound available, 1 if this is the last
; segment and there is no more, -1 if the last one has already been returned.
;     This function takes 2, 3, or 4 parameters, so it must use C calling
; conventions.  The first parameter is the 2k buffer segment number, 0-31.
; The second parameter is a flag.  If 1, the first 2k of the sound is
; being requested; if 0, this is a continuation of a previous call.  If the
; second parameter is 0, there are only 2 parameters.  Otherwise, the third
; parameter is the sound whose playback is beginning:  if 0, the song, and
; if 1-31, sample 1-31.  If the third parameter is not 0, there are only
; three parameters.  Otherwise, the fourth parameter is the number of the
; entry in the pattern table where song playback should begin.
;
		;
		; Parameters to the function.
		;
GS_PARM		STRUC	[BP]
		DB	4 DUP (?)	; caller's BP and return address
GS_SEGNUM	DW	?		; 2k buffer segment number 0-31
GS_FIRST	DW	?		; 1 if first segment
GS_ITEM		DW	?		; 0 for song, 1-31 for sample
GS_PATENTRY	DW	?		; pattern table entry number in song
		ENDS
	EVEN
_getsamples:
	PUSH	BP		; save caller's BP and address parameters
	MOV	BP,SP
	PUSH	SI		; save SI, DI, DS for C
	PUSH	DI
	PUSH	DS
	;
	; Check whether playback is being started.
	;
	CMP	GS_FIRST,1		; is this the start of a new item?
	JNE	_getsamples_GOSUB
	;
	; Starting playback on a new item - select a subroutine table for it.
	;
	MOV	GS_STATE,0		; set state to "first buffer"
	MOV	GS_ITEMTBL,OFFSET GS_SONGROUTINES	; assume song
	CMP	GS_ITEM,0
	JE	_getsamples_GOSUB
	MOV	GS_ITEMTBL,OFFSET GS_SAMPROUTINES	; playing sample
	;
	; Set ES:DI to the address where the samples should be stored, CX
	; to the number of samples to generate (2048), clear the direction
	; flag, and jump to the appropriate subroutine for the current
	; state and item.
	;
_getsamples_GOSUB:
	MOV	BX,GS_SEGNUM
	SHL	BX,1
	MOV	ES,_partbufs[BX]
	XOR	DI,DI
	MOV	CX,2048
	CLD
	MOV	BX,GS_ITEMTBL
	ADD	BX,GS_STATE
	CALL	[BX]
	POP	DS
	POP	DI
	POP	SI
	POP	BP
	RET
;
; *********************** SUBROUTINES OF _getsamples
;
; These routines may destroy any registers they wish but must return 0, 1
; or -1 in AX according to the return code that _getsamples should return.
; On entry:
;     DS      addresses DGROUP (default data segment)
;     ES:DI   addresses the buffer where samples should be stored
;     CX      is the number of samples to be generated
;     direction flag is clear
;     BP      addresses parameters to _getsamples
; These subroutines can call each other, using the above registers.
;
; GSSONGSTART routine, obtains the first 2k samples from the song.  Returns
; 0 in AX (always).
;
	EVEN
GSSONGSTART:
	MOV	AX,GS_PATENTRY		; get the pattern to start with
	MOV	_nextpat,AX		; make it the next pattern
	MOV	BX,_nchannels		; set pointer for final-mix routine
	SUB	BX,4
	MOV	AX,GS_MIXERS[BX]
	MOV	GS_MIXPROC,AX
	MOV	_ticksleft,1		; start new division
	MOV	_currentpat,128		; start new pattern
	MOV	_nextrow,0		; start at beginning of pattern
	MOV	_setspeedparm,0		; no set speed effects yet
	MOV	_patdelayparm,0		; no pattern delay effects yet
	PUSH	CX			; save CX, DI, ES
	PUSH	DI
	PUSH	ES
	MOV	AX,DS			; set visited flags for pattern table
	MOV	ES,AX			;   entries prior to the current one
	MOV	DI,OFFSET _visiteds
	MOV	CX,GS_PATENTRY
	MOV	AL,1
	REP	STOSB			; REP will do nothing if CX = 0
	MOV	CX,_pattablesize	; clear visited flags for pattern
	SUB	CX,GS_PATENTRY		;   table entries current & later
	XOR	AL,AL
	REP	STOSB			; REP will do nothing if CX = 0
	;
	; Initialize all channels.
	;
	MOV	CX,428		; compute sample step for period 428,
	MOV	BX,8363		;   C4SPD 8363
	CALL	GETSTEP		; sample step in DX.AX (BX,CX,DI destroyed)
	MOV	BX,OFFSET _channelrecs
	MOV	CX,_nchannels
	MOV	SI,_samplerecs[0].SAMPSEG ; SI = segment of sample 0
GSSONGSTART_CHANLOOP:
	PUSH	CX
	XOR	CX,CX
	MOV	[BX].CHANSAMPPTR,OFFSET _samplerecs[0] ; set sample to sample 0
	; last sample does not need to be set
	MOV	[BX].CHANSAMPDONE,CL	; sample not played through
	MOV	[BX].CHANSAMPFRAC,CX	; position in sample = start of sample
	MOV	[BX].CHANSAMPOFFS,CX
	MOV	[BX].CHANSAMPSEG,SI
	MOV	[BX].CHANSTEPFRAC,AX	; set sample step for period 428,
	MOV	[BX].CHANSTEPINT,DX	;   C4SPD 8363
	; offset and segment of partial-mix buffer already set
	MOV	[BX].CHANC4SPD,8363	; set C4SPD to 8363
	MOV	[BX].CHANEFFPERIOD,428	; set effective period to 428
	MOV	[BX].CHANEFFTICKPROC,OFFSET DONOTHING ; set tick procedure
	MOV	[BX].CHANEFFENDPROC,OFFSET DONOTHING ; set end procedure
	; arpeggio sample steps do not need to be set
	; arpeggio counter does not need to be set
	MOV	[BX].CHANGLISS,CL	; set glissando flag off
	MOV	[BX].CHANSLIDEINCR,CX	; set slide increment to 0
	MOV	[BX].CHANPERIODGOAL,428	; set period goal to 428
	MOV	[BX].CHANVIBWAVE,OFFSET SINETBL ; set vibrato wave to sine
	MOV	[BX].CHANTREMWAVE,OFFSET SINETBL ; set tremolo wave to sine
	MOV	[BX].CHANVIBDEPTH,CX	; set vibrato depth to 0
	MOV	[BX].CHANTREMDEPTH,CX	; set tremolo depth to 0
	MOV	[BX].CHANVIBINCR,CX	; set vibrato increment to 0
	MOV	[BX].CHANTREMINCR,CX	; set tremolo increment to 0
	MOV	[BX].CHANVIBPOS,CX	; set position in vibrato table to 0
	MOV	[BX].CHANTREMPOS,CX	; set position in tremolo table to 0
	MOV	[BX].CHANVIBNORETRG,CL	; enable vibrato retrigger
	MOV	[BX].CHANTREMNORETRG,CL	; enable tremolo retrigger
	MOV	[BX].CHANMAINVOL,CL	; set main volume to 0
	MOV	[BX].CHANEFFVOL,CL	; set effective volume to 0
	MOV	[BX].CHANMIXVOL,CL	; set mixing volume to 0
	; signed volume increment does not need to be set
	; retrigger speed and count do not need to be set
	; cut count does not need to be set
	MOV	[BX].CHANDELAYCOUNT,CL	; set delay count to 0
	; 64k half of EMS sample does not need to be set (default sample not
	;   in EMS)
	MOV	[BX].CHANINFOBYTE,CL	; set info byte to 0
	ADD	BX,TYPE CHANREC
	POP	CX
	LOOP	GSSONGSTART_CHANLOOP
	;
	; Fill up the partial-mix buffer for the first time.  This destroys
	; every register except DS and the direction flag.
	;
	PUSH	BP			; save BP
	CALL	DOTICK
	POP	BP			; restore BP
	POP	ES			; get ES, DI, CX back
	POP	DI
	POP	CX
	JNC	GSSONGSTART_SETSTATE	; If no samples:
	CALL	DACGETLAST		;   get last output sample
	MOV	GS_LASTSAMP,AL		;   save it
	CALL	GSRAMP			;   ramp to baseline (get 1 buffer)
	MOV	GS_LASTSAMP,80h		;   mark at baseline
	MOV	GS_STATE,8		;   set state to "all done"
	MOV	AX,1			;   return "last buffer"
	RET
	;
	; Mark the partial-mix buffer full, and set state to "next buffer."
	; That routine will do the actual work of getting samples for us.
	;
	EVEN
GSSONGSTART_SETSTATE:
	MOV	AX,_samplespertick
	MOV	_inpartmix,AX
	MOV	PARTMIXOFFS,0
	MOV	GS_STATE,2
	;
	; If we're not at the baseline already (user stopped a previous
	; sound before it finished), just ramp to the baseline for now.
	;
	CALL	DACGETLAST
	MOV	GS_LASTSAMP,AL
	CMP	AL,80h
	JNE	GSSONGSTART_RAMP
	JMP	GSSONGPLAY		; already there - go generate samples
	EVEN
GSSONGSTART_RAMP:
#IF M_I286
	IMUL	BX,GS_SEGNUM,TYPE BUFSEGREC	; set info for user
#ELSE
	MOV	AX,GS_SEGNUM
	MOV	DX,TYPE BUFSEGREC
	MUL	DX
	MOV	BX,AX
#ENDIF
	MOV	SI,_currentpat
	MOV	_bufsegrecs[BX].BUFSEGENTRY,SI
	SHL	SI,1
	MOV	AX,_patterntable[SI]
	MOV	_bufsegrecs[BX].BUFSEGPATNUM,AX
	MOV	AL,_currentrow
	MOV	_bufsegrecs[BX].BUFSEGROW,AL
	MOV	AL,_beatspermin
	MOV	_bufsegrecs[BX].BUFSEGBPM,AL
	MOV	AX,_ticksperrow
	MOV	_bufsegrecs[BX].BUFSEGTICKSPER,AX
	MOV	AL,_globalvol
	MOV	_bufsegrecs[BX].BUFSEGGVOL,AL
	CALL	GSRAMP
	MOV	GS_LASTSAMP,80h		; we won't ramp again
	MOV	GS_STATE,2		; set state to "next buffer" again
	RET				; AX already 0
;
; GSMIX4 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 4-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, CX, DX.
;
	EVEN
GSMIX4:
	LODSW			; range is -16384 to 16256
	MOV	DX,AX
	LODSW
	ADD	AX,DX		; range is -32768 to 32512
	MOV	AL,AH		; range is -128 to 127
	ADD	AL,80h		; range is 0 to 255
	STOSB
	LOOP	GSMIX4
	RET
;
; GSMIX6 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 6-, 12- and
; 24-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX6:
	MOV	BX,64*3
	EVEN
GSMIX6_LOOP:
	LODSW			; range is -24576 to 24384
	MOV	DX,AX
	LODSW
	ADD	AH,60h		; range is 0 to 48960
	ADD	DH,60h
	ADD	AX,DX		; range is 0 to 97920
	RCR	AX,1		; range is 0 to 48960
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX6_LOOP
	RET
;
; GSMIX8 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 8-, 16- and
; 32-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, CX, DX.
;
	EVEN
GSMIX8:
	LODSW			; range is -32768 to 32512
	MOV	DX,AX
	LODSW
	ADD	AH,80h		; range is 0 to 65280
	ADD	DH,80h
	ADD	AX,DX		; range is 0 to 130560
	RCR	AX,1		; range is 0 to 65280
	MOV	AL,AH		; range is 0 to 255
	STOSB
	LOOP	GSMIX8
	RET
;
; GSMIX10 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 10- and 20-
; channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX10:
	MOV	BX,32*5
	EVEN
GSMIX10_LOOP:
	LODSW			; range is -20480 to 20320
	MOV	DX,AX
	LODSW
	ADD	AH,50h		; range is 0 to 40800
	ADD	DH,50h
	ADD	AX,DX		; range is 0 to 81600
	RCR	AX,1		; range is 0 to 40800
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX10_LOOP
	RET
;
; GSMIX14 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 14- and 28-
; channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX14:
	MOV	BX,32*7
	EVEN
GSMIX14_LOOP:
	LODSW			; range is -28672 to 28448
	MOV	DX,AX
	LODSW
	ADD	AH,70h		; range is 0 to 57120
	ADD	DH,70h
	ADD	AX,DX		; range is 0 to 114240
	RCR	AX,1		; range is 0 to 57120
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX14_LOOP
	RET
;
; GSMIX18 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 18-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX18:
	MOV	BX,16*9
	EVEN
GSMIX18_LOOP:
	LODSW			; range is -18432 to 18288
	MOV	DX,AX
	LODSW
	ADD	AH,48h		; range is 0 to 36720
	ADD	DH,48h
	ADD	AX,DX		; range is 0 to 73440
	RCR	AX,1		; range is 0 to 36720
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX18_LOOP
	RET
;
; GSMIX22 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 22-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX22:
	MOV	BX,16*11
	EVEN
GSMIX22_LOOP:
	LODSW			; range is -22528 to 22352
	MOV	DX,AX
	LODSW
	ADD	AH,58h		; range is 0 to 44880
	ADD	DH,58h
	ADD	AX,DX		; range is 0 to 89760
	RCR	AX,1		; range is 0 to 44880
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX22_LOOP
	RET
;
; GSMIX26 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 26-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX26:
	MOV	BX,16*13
	EVEN
GSMIX26_LOOP:
	LODSW			; range is -26624 to 26416
	MOV	DX,AX
	LODSW
	ADD	AH,68h		; range is 0 to 53040
	ADD	DH,68h
	ADD	AX,DX		; range is 0 to 106080
	RCR	AX,1		; range is 0 to 53040
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX26_LOOP
	RET
;
; GSMIX30 routine, mixes samples from the partial-mix buffer to unsigned
; mono 8-bit and places the results in the DMA buffer.  On entry, DS:SI
; addresses samples in the partial-mix buffer, CX is the number of samples
; to mix, ES:DI addresses the 2k DMA buffer segment where the fully-mixed
; samples should go, and DF is clear for incrementing.  For 30-channel mods.
;
; Returns address of next partially-mixed sample in DS:SI, address of next
; position in buffer segment in ES:DI.  Destroys AX, BX, CX, DX.
;
	EVEN
GSMIX30:
	MOV	BX,16*15
	EVEN
GSMIX30_LOOP:
	LODSW			; range is -30720 to 30480
	MOV	DX,AX
	LODSW
	ADD	AH,78h		; range is 0 to 61200
	ADD	DH,78h
	ADD	AX,DX		; range is 0 to 122400
	RCR	AX,1		; range is 0 to 61200
	XOR	DX,DX
	DIV	BX		; range is 0 to 255
	STOSB
	LOOP	GSMIX30_LOOP
	RET
;
; GSSONGPLAY routine, obtains 2k samples from the song, assuming playback
; has been started by a previous call to GSSONGSTART.  Returns 0 in AX
; (always).
;
	EVEN
GSSONGPLAY:
#IF M_I286
	IMUL	BX,GS_SEGNUM,TYPE BUFSEGREC	; set info for user
#ELSE
	MOV	AX,GS_SEGNUM
	MOV	DX,TYPE BUFSEGREC
	MUL	DX
	MOV	BX,AX
#ENDIF
	MOV	SI,_currentpat
	MOV	_bufsegrecs[BX].BUFSEGENTRY,SI
	SHL	SI,1
	MOV	AX,_patterntable[SI]
	MOV	_bufsegrecs[BX].BUFSEGPATNUM,AX
	MOV	AL,_currentrow
	MOV	_bufsegrecs[BX].BUFSEGROW,AL
	MOV	AL,_beatspermin
	MOV	_bufsegrecs[BX].BUFSEGBPM,AL
	MOV	AX,_ticksperrow
	MOV	_bufsegrecs[BX].BUFSEGTICKSPER,AX
	MOV	AL,_globalvol
	MOV	_bufsegrecs[BX].BUFSEGGVOL,AL
	;
	; Generate and mix samples until we get enough or until we get to
	; the end of the song.
	;
GSSONGPLAY_MAINLOOP:
	MOV	AX,CX		; AX = count of samples to mix this time
	CMP	AX,_inpartmix
	JBE	GSSONGPLAY_HAVECOUNT
	MOV	AX,_inpartmix
GSSONGPLAY_HAVECOUNT:
	SUB	CX,AX		; adjust count of samples still needed
	SUB	_inpartmix,AX	; adjust count of samples in partial-mix buffer
	PUSH	CX		; save count still needed on stack
	MOV	CX,AX		; set count for this time
	PUSH	DS		; save DS
	LDS	SI,DWORD PTR PARTMIXOFFS ; DS:SI addresses partial-mix samples
	CALL	WORD PTR SS:GS_MIXPROC	; mix samples, put in DMA buffer
	POP	DS		; get back DS
	MOV	PARTMIXOFFS,SI	; save back address in partial-mix buffer
	CMP	_inpartmix,0	; is the partial-mix buffer empty?
	JNE	GSSONGPLAY_MAINLPEND
	PUSH	DI		; save DI, ES
	PUSH	ES
	CALL	DOTICK		; partial-mix buffer empty - get more
	POP	ES		; restore ES, DI (DOTICK destroys all but
	POP	DI		;   DS, DF)
	JC	GSSONGPLAY_NOMORE ; end of song if DOTICK returned w/CF set
	MOV	AX,_samplespertick
	MOV	_inpartmix,AX
	MOV	PARTMIXOFFS,0
GSSONGPLAY_MAINLPEND:
	POP	CX		; get back count still needed
	OR	CX,CX		; go again if not zero
	JNZ	GSSONGPLAY_MAINLOOP
	;
	; All done.  Set return code and exit.
	;
GSSONGPLAY_DONE:
	XOR	AX,AX
	RET
	;
	; We got to the end of the song.  Go ramp to the baseline.
	;
	EVEN
GSSONGPLAY_NOMORE:
	POP	CX		; get back count still needed
	MOV	AL,ES:[DI-1]
	MOV	GS_LASTSAMP,AL
	MOV	GS_STATE,4	; set state to "ramping to baseline"
	JCXZ	GSSONGPLAY_DONE	; see if we need any more
	JMP	GSRAMP		; we need more - get them
;
; GSSAMPSTART routine, obtains the first 2k samples from the sample.
; Returns 0 in AX (always).
;
	EVEN
GSSAMPSTART:
	;
	; Compute and store the address of the sample record.
	;
	MOV	AX,GS_ITEM
	MOV	AH,TYPE SAMPREC
	MUL	AH
	ADD	AX,OFFSET _samplerecs
	MOV	GS_SAMPREC,AX
	;
	; Set the current sample position to the start of the sample.
	;
	MOV	BX,AX
	MOV	AX,[BX].SAMPSEG
	MOV	GS_SAMPSEG,AX
	XOR	AX,AX
	MOV	GS_SAMPFRAC,AX
	MOV	GS_SAMPINT,AX
	MOV	GS_SAMPHALF,AL
	;
	; Compute and store the sample step.  We always play at period 1712
	; (C2).
	;
	PUSH	CX		; save CX, DI
	PUSH	DI
	MOV	CX,1712
	MOV	BX,[BX].SAMPC4SPD
	CALL	GETSTEP		; destroys BX, CX, DI
	SHR	DX,1		; convert to words
	RCR	AX,1
	MOV	GS_STEPFRAC,AX
	MOV	GS_STEPINT,DX
	POP	DI		; restore DI, CX
	POP	CX
	;
	; Set state to "next buffer."  That routine will do the actual
	; work of getting samples for us.
	;
	MOV	GS_STATE,2
	;
	; If we're not at the baseline already (user stopped a previous
	; sound before it finished), just ramp to the baseline for now.
	;
	CALL	DACGETLAST
	MOV	GS_LASTSAMP,AL
	CMP	AL,80h
	JNE	GSSAMPSTART_RAMP
	JMP	GSSAMPPLAY		; already there - go generate samples
	EVEN
GSSAMPSTART_RAMP:
	CALL	GSRAMP
	MOV	GS_LASTSAMP,80h		; we won't ramp again
	MOV	GS_STATE,2		; set state to "next buffer" again
	RET				; AX already 0
;
; GSSAMPPLAY routine, obtains 2k samples from the sample, assuming playback
; has been started by a previous call to GSSAMPSTART.  Returns 0 in AX
; (always).  We only have one resampler for sample playback, so it needs to
; be able to handle the worst possible case.
;
; We have two different versions of this routine, depending on the
; processor.  First, the 286 version.
;
#IF M_I286
	EVEN
GSSAMPPLAY:
	;
	; Patch the code (eek!) - Intel doesn't have enough registers.
	;
	MOV	BX,GS_SAMPREC		; BX addresses sample data
	MOV	AL,[BX].SAMPVOL		; put volume in the code
	MOV	BYTE PTR CS:GSSAMPPLAY_VOLPT1,AL
	MOV	AX,[BX].SAMPLEN		; put sample length in the code
	MOV	WORD PTR CS:GSSAMPPLAY_LENPT1,AX
	MOV	AX,[BX].SAMPLSTART	; put sample loop start in the code
	MOV	WORD PTR CS:GSSAMPPLAY_LSTARTPT1,AX
	MOV	WORD PTR CS:GSSAMPPLAY_LSTARTPT2,AX
	MOV	AX,[BX].SAMPLLEN	; put sample loop length in the code
	MOV	WORD PTR CS:GSSAMPPLAY_LLENPT1,AX
	MOV	DX,GS_STEPFRAC		; DX is fractional part of sample step
	MOV	WORD PTR CS:GSSAMPPLAY_STFRACPT1,DX ; ... and put in the code
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:GSSAMPPLAY_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	GSSAMPPLAY_LOOPED
	MOV	WORD PTR CS:GSSAMPPLAY_SAMPEND,GSSAMPPLAY_JUMPINSTR
	;
	; Determine the "limit" for DI - where we stop when we have enough
	; samples.  Patch the code with it.
	;
GSSAMPPLAY_LOOPED:
	ADD	CX,DI
	MOV	WORD PTR CS:GSSAMPPLAY_LIMITPT1,CX
	MOV	WORD PTR CS:GSSAMPPLAY_LIMITPT2,CX
	;
	; Now set up register values.  We have:
	;
	;    ES:DI    addresses buffer where samples go (already set)
	;    DS       current segment in sample
	;    SI.BX    current word offset in sample
	;    BP.DX    sample step in words (DX already set)
	;    CX       8000h if 64k or more past sample start, 0 otherwise
	;
	XOR	CX,CX			; assume first 64k
	MOV	AX,GS_SAMPSEG		; get segment in AX
	CMP	[BX].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	GSSAMPPLAY_INEMS
	MOV	BYTE PTR CS:GSSAMPPLAY_SEGJMP,GSSAMPPLAY_SEGJMP1
	CMP	[BX].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT GSSAMPPLAY_SETCX
	EVEN
GSSAMPPLAY_INEMS:
	MOV	BYTE PTR CS:GSSAMPPLAY_SEGJMP,GSSAMPPLAY_SEGJMP2
	MOV	SI,[BX].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:GSSAMPPLAY_EMSHANDLE,SI
	PUSH	SI			; save EMS handle for mapping routine
	CMP	GS_SAMPHALF,0		; are we in the lower half?
	JNE	GSSAMPPLAY_EMS2ND
	CALL	MAPEMS1ST		; lower 64k - map 1st half
	CLC				; clear CF (lower 64k)
	JMP	SHORT GSSAMPPLAY_SETCX
	EVEN
GSSAMPPLAY_EMS2ND:
	CALL	MAPEMS2ND		; upper 64k - map 2nd half
	STC				; set CF (upper 64k)
GSSAMPPLAY_SETCX:			; CF set if 2nd 64k
	RCR	CX,1
	MOV	BP,GS_STEPINT
	MOV	BX,GS_SAMPFRAC
	MOV	SI,GS_SAMPINT
	MOV	DS,AX
	;
	; Main loop.  Go until we get to the end of the (unlooped) sample,
	; or until we have enough samples.
	;
	EVEN
GSSAMPPLAY_MAINLOOP:
	DB	0B0h		; MOV AL,immed SAMPVOL
GSSAMPPLAY_VOLPT1:
	DB	0
	SHL	BX,1		; convert offset to bytes
	RCL	SI,1
	IMUL	BYTE PTR [SI]	; multiply volume by sample
	SHR	SI,1		; convert offset back to words
	RCR	BX,1
	SAL	AX,1		; AH = signed 8-bit sample
	SAL	AX,1
	MOV	AL,AH		; put in AL
	ADD	AL,80h		; convert to unsigned
	STOSB			; store in buffer
	OR	SI,CX		; SI.BX = word offset relative to sample start
	ADD	BX,DX		; add sample step in words
	ADC	SI,BP
	JC	GSSAMPPLAY_SAMPEND ; wrap past 128k?
	DB	81h,0FEh	; CMP SI,immed SAMPLEN
GSSAMPPLAY_LENPT1:
	DW	0
	JAE	GSSAMPPLAY_SAMPEND ; wrap past end of sample? (JAE = JNC)
GSSAMPPLAY_SEGTEST:
	XOR	SI,CX		; check for 64k segment change
	DB	78h		; JS segchange_code
GSSAMPPLAY_SEGJMP:
	DB	0
GSSAMPPLAY_MAINLPEND:
	DB	81h,0FFh	; CMP DI,immed word
GSSAMPPLAY_LIMITPT1:
	DW	0
	JNE	GSSAMPPLAY_MAINLOOP ; go again if more samples needed
	;
	; We got all the samples needed, and there is at least one more
	; output sample to generate.  Save back the current segment and
	; offset, and the flag to indicate whether it's the first or
	; second half of the EMS sample.
	;
	MOV	SS:GS_SAMPFRAC,BX
	MOV	SS:GS_SAMPINT,SI
	MOV	SS:GS_SAMPSEG,DS
	ROL	CH,1
	MOV	SS:GS_SAMPHALF,CH
	;
	; We're all done for now.
	;
	JMP	SHORT GSSAMPPLAY_DONE
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
GSSAMPPLAY_SEGJMP1	EQU	$ - (OFFSET GSSAMPPLAY_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	MOV	AX,DS			; DS value in AX for adjustment
	JZ	GSSAMPPLAY_SEGLOW	; if CX = 0, CX was 8000h, so low 64k
	ADD	AH,10h			; low to high, add 1000h to DS
	MOV	DS,AX
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	EVEN
GSSAMPPLAY_SEGLOW:
	SUB	AH,10h			; high to low, subtract 1000h from DS
	MOV	DS,AX
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
GSSAMPPLAY_SEGJMP2	EQU	$ - (OFFSET GSSAMPPLAY_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	DB	68h			; PUSH immed SAMPEMSHANDLE
GSSAMPPLAY_EMSHANDLE:
	DW	0
	JZ	GSSAMPPLAY_SEGLOW2	; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	EVEN
GSSAMPPLAY_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.BX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
GSSAMPPLAY_SAMPEND:
	DB	0EBh		; JMP SHORT GSSAMPPLAY_RAMP (or NOP, NOP)
	DB	GSSAMPPLAY_JUMPSIZE
	MOV	DX,0		; LSB of DX = CF
	RCL	DX,1
	MOV	AX,SI		; DX:AX = integer of offset in words
	DB	2Dh		; SUB AX,immed SAMPLSTART
GSSAMPPLAY_LSTARTPT1:
	DW	0
	SBB	DX,0
	DB	0BEh		; MOV SI,immed SAMPLLEN
GSSAMPPLAY_LLENPT1:
	DW	0
	DIV	SI
	MOV	SI,DX
	DB	81h,0C6h	; ADD SI,immed SAMPLSTART
GSSAMPPLAY_LSTARTPT2:
	DW	0
	DB	0BAh		; MOV DX,immed GS_STEPFRAC
GSSAMPPLAY_STFRACPT1:
	DW	0
	JMP	SHORT GSSAMPPLAY_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  We need
	; to get set up, then go ramp to the baseline.
	;
	EVEN
GSSAMPPLAY_RAMP:
GSSAMPPLAY_JUMPSIZE	EQU	$ - (OFFSET GSSAMPPLAY_SAMPEND + 2)
GSSAMPPLAY_JUMPINSTR	EQU	(GSSAMPPLAY_JUMPSIZE*256) + 0EBh
	MOV	AX,DGROUP	; DS addresses local data again
	MOV	DS,AX
	MOV	AL,ES:[DI-1]	; save the last output sample
	MOV	GS_LASTSAMP,AL
	MOV	GS_STATE,4	; state is "ramp to baseline" now
	DB	0B9h		; MOV CX,immed word
GSSAMPPLAY_LIMITPT2:
	DW	0
	SUB	CX,DI		; get back remaining count
	JZ	GSSAMPPLAY_DONE	; are we done?
	JMP	GSRAMP		; if not, start ramping
	;
	; All done for now.  Set return code and return.
	;
	EVEN
GSSAMPPLAY_DONE:
	XOR	AX,AX
	RET
;
; Next, the 8086 version.
;
#ELSE
	EVEN
GSSAMPPLAY:
	;
	; Patch the code (eek!) - Intel doesn't have enough registers.
	;
	MOV	BX,GS_SAMPREC		; BX addresses sample data
	MOV	AH,[BX].SAMPVOL		; get offset in MULTBL for volume
	XOR	AL,AL
	SHL	AX,1
	ADD	AX,OFFSET MULTBL
	MOV	WORD PTR CS:GSSAMPPLAY_VOLPT1,AX
	MOV	AX,[BX].SAMPLEN		; put sample length in the code
	MOV	WORD PTR CS:GSSAMPPLAY_LENPT1,AX
	MOV	WORD PTR CS:GSSAMPPLAY_LENPT2,AX
	MOV	AX,[BX].SAMPLLEN	; put sample loop length in the code
	MOV	WORD PTR CS:GSSAMPPLAY_LLENPT1,AX
	MOV	DX,GS_STEPFRAC		; DX is fractional part of sample step
	MOV	WORD PTR CS:GSSAMPPLAY_STFRACPT1,DX ; ... and put in the code
	;
	; If the sample is looped, patch the code so we won't jump over the
	; part where we go back to the start of the loop.  Otherwise, patch
	; it so that we will.
	;
	MOV	WORD PTR CS:GSSAMPPLAY_SAMPEND,9090h
	OR	AX,AX			; loop length still in AX from above
	JNZ	GSSAMPPLAY_LOOPED
	MOV	WORD PTR CS:GSSAMPPLAY_SAMPEND,GSSAMPPLAY_JUMPINSTR
	;
	; Determine the "limit" for DI - where we stop when we have enough
	; samples.  Patch the code with it.
	;
GSSAMPPLAY_LOOPED:
	ADD	CX,DI
	MOV	WORD PTR CS:GSSAMPPLAY_LIMITPT1,CX
	MOV	WORD PTR CS:GSSAMPPLAY_LIMITPT2,CX
	;
	; We have the following register values:
	;
	;    ES:DI    addresses buffer where samples go (already set)
	;    DS       current segment in sample
	;    SI.DX    current word offset in sample (relative to DS)
	;    BP       integer part of sample step in words (fraction in code)
	;    CX       8000h if 64k or more past sample start, 0 otherwise
	;
	XOR	CX,CX			; assume first 64k
	MOV	AX,GS_SAMPSEG		; get segment in AX
	CMP	[BX].SAMPEMSFLAG,1	; is the sample in EMS?
	JE	GSSAMPPLAY_INEMS
	MOV	BYTE PTR CS:GSSAMPPLAY_SEGJMP,GSSAMPPLAY_SEGJMP1
	CMP	[BX].SAMPSEG,AX		; set CF if 2nd 64k
	JMP	SHORT GSSAMPPLAY_SETCX
	EVEN
GSSAMPPLAY_INEMS:
	MOV	BYTE PTR CS:GSSAMPPLAY_SEGJMP,GSSAMPPLAY_SEGJMP2
	MOV	SI,[BX].SAMPEMSHANDLE	; get EMS handle and patch into code
	MOV	WORD PTR CS:GSSAMPPLAY_EMSHANDLE,SI
	PUSH	SI			; save EMS handle for mapping routine
	CMP	GS_SAMPHALF,0		; are we in the lower half?
	JNE	GSSAMPPLAY_EMS2ND
	CALL	MAPEMS1ST		; lower 64k - map 1st half
	CLC				; clear CF (lower 64k)
	JMP	SHORT GSSAMPPLAY_SETCX
	EVEN
GSSAMPPLAY_EMS2ND:
	CALL	MAPEMS2ND		; upper 64k - map 2nd half
	STC				; set CF (upper 64k)
GSSAMPPLAY_SETCX:			; CF set if 2nd 64k
	RCR	CX,1
	MOV	BP,GS_STEPINT
	MOV	DX,GS_SAMPFRAC
	MOV	SI,GS_SAMPINT
	MOV	DS,AX
	EVEN
GSSAMPPLAY_MAINLOOP:
	SHL	DX,1
	RCL	SI,1
	MOV	BL,[SI]
	SHR	SI,1
	RCR	DX,1
	XOR	BH,BH
	SHL	BX,1
	DB	36h,8Bh,87h	; MOV AX,SS:[BX+immed]
GSSAMPPLAY_VOLPT1:
	DW	0		; offset in MULTBL for volume
	SAL	AX,1
	SAL	AX,1
	MOV	AL,AH
	ADD	AL,80h
	STOSB
	OR	SI,CX
	DB	81h,0C2h	; ADD DX,immed STEPFRAC
GSSAMPPLAY_STFRACPT1:
	DW	0
	ADC	SI,BP
	JC	GSSAMPPLAY_SAMPEND
	DB	81h,0FEh	; CMP SI,immed SAMPLEN
GSSAMPPLAY_LENPT1:
	DW	0
	JAE	GSSAMPPLAY_SAMPEND ; wrap past end of sample? (JAE = JNC)
GSSAMPPLAY_SEGTEST:
	XOR	SI,CX
	DB	78h		; JS segchange_code
GSSAMPPLAY_SEGJMP:
	DB	0
GSSAMPPLAY_MAINLPEND:
	DB	81h,0FFh	; CMP DI,immed word
GSSAMPPLAY_LIMITPT1:
	DW	0
	JNE	GSSAMPPLAY_MAINLOOP
	;
	; We got all the samples needed, and there is at least one more
	; output sample to generate.  Save back the current segment and
	; offset, and the flag to indicate whether it's the first or
	; second half of the EMS sample.
	;
	MOV	SS:GS_SAMPFRAC,DX
	MOV	SS:GS_SAMPINT,SI
	MOV	SS:GS_SAMPSEG,DS
	ROL	CH,1
	MOV	SS:GS_SAMPHALF,CH
	;
	; We're all done for now.
	;
	JMP	SHORT GSSAMPPLAY_DONE
	;
	; Different 64k segment now.  Adjust DS, CX, and SI.  The most
	; significant bit of SI is set when we get here and needs to be
	; cleared in any case.  If CX is 0, we're going from the lower 64k
	; to the upper 64k, and we need to set DS to SAMPSEG+1000h and CX
	; to 8000h.  If CX is 8000h, we're going from the upper 64k to the
	; lower 64k (a looped sample longer than 64k, and we just looped).
	; In that case, we need to set DS to SAMPSEG and CX to 0.  This is
	; for a sample loaded in conventional RAM.
	;
	EVEN
GSSAMPPLAY_SEGJMP1	EQU	$ - (OFFSET GSSAMPPLAY_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	MOV	AX,DS			; DS value in AX for adjustment
	JZ	GSSAMPPLAY_SEGLOW	; if CX = 0, CX was 8000h, so low 64k
	ADD	AH,10h			; low to high, add 1000h to DS
	MOV	DS,AX
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	EVEN
GSSAMPPLAY_SEGLOW:
	SUB	AH,10h			; high to low, subtract 1000h from DS
	MOV	DS,AX
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	;
	; Different 64k segment now.  Adjust CX and SI, and map in the
	; correct half of the EMS sample.  The most significant bit of SI
	; is set when we get here and needs to be cleared in any case.  If
	; CX is 0, we're going from the lower 64k to the upper 64k, and we
	; need to set CX to 8000h and map in the upper 64k.  If CX is 8000h,
	; we're going from the upper 64k to the lower 64k (a looped sample
	; longer than 64k, and we just looped).  In that case, we need to
	; set CX to 0 and map in the lower 64k.  This is for a sample loaded
	; in EMS RAM.
	;
	EVEN
GSSAMPPLAY_SEGJMP2	EQU	$ - (OFFSET GSSAMPPLAY_MAINLPEND)
	MOV	AX,8000h
	XOR	SI,AX			; clear MSB of SI
	XOR	CX,AX			; invert MSB of CX
	DB	0B8h			; MOV AX,immed SAMPEMSHANDLE
GSSAMPPLAY_EMSHANDLE:
	DW	0
	PUSH	AX
	JZ	GSSAMPPLAY_SEGLOW2	; if CX = 0, CX was 8000h, so low 64k
	CALL	MAPEMS2ND		; low to high, map in the second 64k
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	EVEN
GSSAMPPLAY_SEGLOW2:
	CALL	MAPEMS1ST		; high to low, map in the first 64k
	JMP	SHORT GSSAMPPLAY_MAINLPEND
	;
	; We got to the end of the sample.  CF:SI.DX is the current offset
	; in words from the start of the sample.  If the sample is looped,
	; go back to the beginning of the loop.
	;
	EVEN
GSSAMPPLAY_SAMPEND:
	DB	0EBh		; JMP SHORT GSSAMPPLAY_RAMP (or NOP, NOP)
	DB	GSSAMPPLAY_JUMPSIZE
	MOV	AL,0		; LSB of AL = CF
	RCL	AL,1		; AL:SI = integer of offset in words
GSSAMPPLAY_ADJLOOP:
	DB	81h,0EEh	; SUB SI,immed SAMPLLEN
GSSAMPPLAY_LLENPT1:
	DW	0
        SBB     AL,0
        JNZ     GSSAMPPLAY_ADJLOOP
	DB	81h,0FEh	; CMP SI,immed SAMPLEN
GSSAMPPLAY_LENPT2:
	DW	0
        JAE     GSSAMPPLAY_ADJLOOP
	JMP	SHORT GSSAMPPLAY_SEGTEST
	;
	; We got to the end of the sample, and it is not looped.  We need
	; to get set up, then go ramp to the baseline.
	;
	EVEN
GSSAMPPLAY_RAMP:
GSSAMPPLAY_JUMPSIZE	EQU	$ - (OFFSET GSSAMPPLAY_SAMPEND + 2)
GSSAMPPLAY_JUMPINSTR	EQU	(GSSAMPPLAY_JUMPSIZE*256) + 0EBh
	MOV	AX,DGROUP	; DS addresses local data again
	MOV	DS,AX
	MOV	AL,ES:[DI-1]	; save the last output sample
	MOV	GS_LASTSAMP,AL
	MOV	GS_STATE,4	; state is "ramp to baseline" now
	DB	0B9h		; MOV CX,immed word
GSSAMPPLAY_LIMITPT2:
	DW	0
	SUB	CX,DI		; get back remaining count
	JZ	GSSAMPPLAY_DONE	; are we done?
	JMP	GSRAMP		; if not, start ramping
	;
	; All done for now.  Set return code and return.
	;
	EVEN
GSSAMPPLAY_DONE:
	XOR	AX,AX
	RET
#ENDIF
;
; GSRAMP routine, generates samples to ramp the song or sample to the
; baseline.  Returns 0 in AX (always).
;
	EVEN
GSRAMP:
	MOV	AL,GS_LASTSAMP
	MOV	AH,1		; assume below baseline, signed increment is 1
	CMP	AL,80h
	JB	GSRAMP_DORAMP
	JA	GSRAMP_ABOVE
	XOR	AH,AH		; if at baseline, signed increment is 0
	JMP	SHORT GSRAMP_DORAMP
	EVEN
GSRAMP_ABOVE:
	NEG	AH		; if above baseline, signed increment is -1
	;
	; Get set up to generate samples.  We will generate 16 of each sample
	; value so that the change in the speaker position will be too slow
	; to be audible.  CX is the number of distinct sample values to
	; generate.  DX is the number of "extras" at the end, i.e., the
	; number of output samples modulo 16.
	;
GSRAMP_DORAMP:
	MOV	DX,CX		; DX is number of "extra" samples to do at end
	AND	DX,0Fh
#IF M_I286
	SHR	CX,4		; output 16 samples of each value
#ELSE
	SHR	CX,1
	SHR	CX,1
	SHR	CX,1
	SHR	CX,1
#ENDIF
	JZ	GSRAMP_FINISH	; less than 16 samples to be output?
	;
	; Generate the samples, 16 of each one, until either we get to the
	; baseline or we run out of room in the buffer.
	;
GSRAMP_RAMPLOOP:
	MOV	BX,CX		; save loop count
	MOV	CX,16		; output samples
	REP	STOSB
	MOV	CX,BX		; restore loop count
	ADD	AL,AH		; go to next sample value
	CMP	AL,80h		; at baseline yet?
	LOOPNE	GSRAMP_RAMPLOOP
#IF M_I286
	SHL	CX,4		; CX is number of extra silence samples
#ELSE
	SHL	CX,1
	SHL	CX,1
	SHL	CX,1
	SHL	CX,1
#ENDIF
	;
	; Output the last sample to fill up the buffer.
	;
GSRAMP_FINISH:
	ADD	CX,DX
	REP	STOSB
	MOV	GS_LASTSAMP,AL	; save last sample for next time
	CMP	AL,80h		; at baseline?
	JNE	GSRAMP_DONE
	MOV	GS_STATE,6	; set state to "one more buffer"
GSRAMP_DONE:
	XOR	AX,AX
	RET
;
; GSONEMORE routine, generates baseline samples.  Returns 1 in AX (always).
;
	EVEN
GSONEMORE:
	MOV	AX,8080h
	SHR	CX,1
	JNC	GSONEMORE_EVEN
	STOSB
GSONEMORE_EVEN:
	REP	STOSW
	MOV	GS_STATE,8	; set state to "all done"
	MOV	AX,1
	RET
;
; GSALLDONE routine.  Called when playback has not started or is complete.
; Generates no samples and returns -1 in AX (always).
;
	EVEN
GSALLDONE:
	MOV	AX,-1
	RET
_TEXT	ENDS
	END
