/*
    Tantrakr.c

    This is the main program file for Tantrakr, a mod player for the Tandy 
    DAC.
*/

#include <stdio.h>      /* standard I/O functions header file */
#include <dos.h>        /* standard DOS functions header file */
#include <string.h>     /* standard string functions header file */
#include <stdlib.h>     /* standard library functions header file */
#include <ctype.h>      /* standard character functions header file */
#include "memmgr.h"     /* memory management routines */
#include "modstuf.h"    /* music module loading/playing routines */
#include "dacstuf.h"    /* Tandy DAC routines */
#include "video.h"      /* video display routines */
#include "input.h"      /* user input routine */


/*************************************************************************/
/*                                                                       */
/* Macros.                                                               */
/*                                                                       */
/*************************************************************************/

    /* Sampling rate to use when none has been specified on the command 
       line. */
#if M_I286
#define MAINRATE 22000
#else
#define MAINRATE 10000
#endif


/*************************************************************************/
/*                                                                       */
/* Types.                                                                */
/*                                                                       */
/*************************************************************************/



/*************************************************************************/
/*                                                                       */
/* Global variables.                                                     */
/*                                                                       */
/*************************************************************************/

    /* Sampling rates for various numbers of channels in the mod file. */
static unsigned rateforchans[] = {
    MAINRATE,   /* 4 channels */
    MAINRATE,   /* 6 channels */
    MAINRATE,   /* 8 channels */
    MAINRATE,   /* 10 channels */
    MAINRATE,   /* 12 channels */
    MAINRATE,   /* 14 channels */
    MAINRATE,   /* 16 channels */
    MAINRATE,   /* 18 channels */
    MAINRATE,   /* 20 channels */
    MAINRATE,   /* 22 channels */
    MAINRATE,   /* 24 channels */
    MAINRATE,   /* 26 channels */
    MAINRATE,   /* 28 channels */
    MAINRATE,   /* 30 channels */
    MAINRATE    /* 32 channels */
};
    /* Previous <control>-<break> handler. */
static void (interrupt cdecl far *oldbreakhdlr)( void );
    /* Pathname of currently-loaded file. */
static char currentfile[_MAX_PATH] = { '\0' };
    /* State of the player - 0 if the file module has not been initialized 
       yet; 1 if the file module is initialized but no file is loaded; 2 if 
       a file is loaded but not being played; 3 if the song is playing; 4 
       if a sample is playing. */
static char playerstate = 0;
    /* 2k buffer segment flags.  0 = valid, not last; 1 = valid, last; 
       -1 = invalid (segment does not contain valid sound). */
static int dmaflags[32] = { -1 };
    /* Current 2k buffer segment for sample generation. */
static unsigned currentseg;
static unsigned dmaseg;            /* 2k buffer segment DMA is in */
static unsigned dmawas;            /* 2k buffer segment DMA was in before */

    /* User messages. */
static char fastforwardmsg[] = "Fast-forwarding ...";
static char continuemsg[] = "Playback continuing.";
static char rewindmsg[] = "Rewinding ...";
static char needdacmsg[] = "This program requires a Tandy DAC.\n";
static char badmemmsg[] =
    "Insufficient memory or memory allocation error.\n";
static char nodmamsg[] =
    "Insufficient memory to allocate DMA buffer for sound.";
static char nomixmsg[] = "Insufficient memory for sound mixing buffers.";
static char loadfilemsg[] = "Loading file ...";
static char loaddonemsg[] = "File loaded.";
static char playdonemsg[] = "Playback complete.";
static char toofastmsg[] = "Sound underflow - sampling rate too high.";
static char badbufmsg[] = "Internal error - invalid sound buffer state %d.";
static char inprogressmsg[] = "Playback in progress - can't load.";
static char enternamemsg[] = "Enter filename:";
static char notloadedmsg[] = "File not loaded.";
static char noneloadedmsg[] = "<no file loaded>";
static char noneloadedmsg2[] = "No file loaded.";
static char fillbufmsg[] = "Filling sound buffer ...";
static char playsongmsg[] = "Playing song.";
static char enterposmsg[] = "Enter new position (0-%u):";
static char badposmsg[] = "Invalid position.";
static char playpausemsg[] = "Playback paused.";
static char playhaltmsg[] = "Playback halted.";
static char nounloadmsg[] = "Playback in progress - can't unload.";
static char unloadmsg[] = "File unloaded.";
static char noripmsg[] = "Playback in progress - can't rip.";
static char enterripmsg[] = "Enter sample to rip (1-%u):";
static char noripmsg2[] = "No sample ripped.";
static char badsamnummsg[] = "Invalid sample number.";
static char samsavemsg[] = "Sample (*.SAM) and parameters (*.NOT) saved.";
static char samwritemsg[] = "Sample %u written out.";
static char samsavemsg2[] = "Samples (*.SAM) and parameters (*.NOT) saved.";
static char entersammsg[] = "Enter sample to play (1-%u):";
static char playsammsg[] = "Playing sample.";
static char nodumpmsg[] = "Playback in progress - can't dump.";
static char enterdumpmsg[] = "Enter name for dumpfile:";
static char nodumpmsg2[] = "File not dumped.";
static char savemodmsg[] = "Saving module data ...";
static char savemodmsg2[] = "Module data saved.";
static char volsetmsg[] = "Volume set.";
static char loopsetmsg[] = "Looping set.";
static char noratemsg[] =
    "Playback in progress - can't change sampling rate.";
static char enterratemsg[] =
    "Enter new sampling rate in Hertz (4000-44100):";
static char noratemsg2[] = "Sampling rate not changed.";
static char badratemsg[] = "Invalid sampling rate.";
static char ratesetmsg[] = "Sampling rate set.";
static char ticksetmsg[] = "Ticks/note set to %u.";
static char bpmsetmsg[] = "Bpm set to %u.";
static char gvolsetmsg[] = "Global volume set to %d.";
static char emssetmsg[] = "EMS usage set for next file.";
static char savememonmsg[] =
    "Save memory enabled - will not load unused patterns in next file.";
static char savememoffmsg[] =
    "Save memory disabled - will load all patterns in next file.";
static char optimizsetmsg[] = "Loading optimization set for next file.";


/*************************************************************************/
/*                                                                       */
/* Function prototypes.                                                  */
/*                                                                       */
/*************************************************************************/

void panic( char *msg, int retcod );
static void interrupt cdecl far breakhdlr( void );
static void interrupt cdecl far crithdlr( unsigned _es, unsigned _ds,
    unsigned _di, unsigned _si, unsigned _bp, unsigned _sp, unsigned _bx,
    unsigned _dx, unsigned _cx, unsigned _ax, unsigned _ip, unsigned _cs,
    unsigned _flags );
static void hookints( void );
static void unhookints( void );
static void process_parm( char *parm );
static void process_env( void );


/*************************************************************************/
/*                                                                       */
/* panic() routine.  This function displays a message to the user and    */
/* aborts the program, cleanly we hope.                                  */
/*                                                                       */
/*************************************************************************/

void panic( char *msg, int retcod )
{
    enddac();               /* set sound hardware back to default */
    endvid();               /* set video back to default */
    unhookints();           /* unhook interrupts */
    endmem();               /* release allocated memory */
    printf( "%s\n", msg );  /* display the error message */
    exit( retcod );         /* return the error code */
}


/*************************************************************************/
/*                                                                       */
/* breakhdlr() replacement handler for the <control>-C and <control>-    */
/* <break> interrupts.  Does nothing, ignoring the keystroke.            */
/*                                                                       */
/*************************************************************************/

static void interrupt cdecl far breakhdlr( void )
{
    /* do nothing */
}


/*************************************************************************/
/*                                                                       */
/* crithdlr() replacement handler for the critical error interrupt.      */
/* Causes the system call to fail rather than issuing the "Abort, Retry, */
/* Ignore, Fail?" prompt (DOS 3.1 or later).                             */
/*                                                                       */
/*************************************************************************/

static void interrupt cdecl far crithdlr( unsigned _es, unsigned _ds,
    unsigned _di, unsigned _si, unsigned _bp, unsigned _sp, unsigned _bx,
    unsigned _dx, unsigned _cx, unsigned _ax, unsigned _ip, unsigned _cs,
    unsigned _flags )
{
    _ax = (_ax & 0xff00) + 3;
}


/*************************************************************************/
/*                                                                       */
/* hookints() routine, hooks the <control>-C, <control>-<break>, and     */
/* critical error interrupts.                                            */
/*                                                                       */
/*************************************************************************/

static void hookints( void )
{
    oldbreakhdlr = _dos_getvect( 0x1b );
    _dos_setvect( 0x1b, breakhdlr );
    _dos_setvect( 0x23, breakhdlr );
    _dos_setvect( 0x24, crithdlr );
}


/*************************************************************************/
/*                                                                       */
/* unhookints() routine.  Unhooks the <control>-<break> interrupt at     */
/* program termination.  DOS takes care of the <control>-C and critical  */
/* error interrupts for us.                                              */
/*                                                                       */
/*************************************************************************/

static void unhookints( void )
{
    _dos_setvect( 0x1b, oldbreakhdlr );
}


/*************************************************************************/
/*                                                                       */
/* process_parm() routine.  Processes a command-line argument.           */
/*                                                                       */
/*************************************************************************/

static void process_parm( char *parm )
{
    int parmlen;            /* length of parameter string */
    unsigned long testrate; /* sampling rate from command line */
    char *endptr;           /* pointer for verifying rate string */
    int i;                  /* for looping over characters in the filename */
    int index;              /* index in sampling rate table */

    parmlen = strlen( parm );
    if ( *parm == '/' )
    {
        switch ( parm[1] )
        {
            /* /V: set volume. */
            case 'v':
            case 'V':
                if ( parmlen == 3 )
                    if ( parm[2] >= '0' && parm[2] <= '7' )
                        volume = parm[2] - '0';
                break;

            /* /L: set loop enable. */
            case 'l':
            case 'L':
                if ( parmlen == 3 )
                {
                    if ( parm[2] == '0' )
                        loopdisable = 1;
                    else if ( parm[2] == '1' )
                        loopdisable = 0;
                }
                break;

            /* /X: set EMS enable. */
            case 'x':
            case 'X':
                if ( parmlen == 3 )
                {
                    if ( parm[2] == '0' )
                        emsenable = 0;
                    else if ( parm[2] == '1' )
                        emsenable = 1;
                }
                break;

            /* /S: set "save memory" flag. */
            case 's':
            case 'S':
                if ( parmlen == 3 )
                {
                    if ( parm[2] == '0' )
                        savemem = 0;
                    else if ( parm[2] == '1' )
                        savemem = 1;
                }
                break;

            /* /O: set "optimize loading" flag. */
            case 'o':
            case 'O':
                if ( parmlen == 3 )
                {
                    if ( parm[2] == '0' )
                        optimize = 0;
                    else if ( parm[2] == '1' )
                        optimize = 1;
                }
                break;

            /* /nnnnn: set sampling rates. */
            default:
                parm++;
                index = 0;
                parmlen = strspn( parm, "0123456789" );
                while ( parmlen == 4 || parmlen == 5 && index < 15 )
                {
                    testrate = strtoul( parm, &endptr, 10 );
                    if ( testrate >= 4000L && testrate <= 44100L )
                        rateforchans[index++] = testrate;
                    else
                        break;
                    if ( *endptr == NULL )
                        break;
                    parm = endptr + 1;
                    parmlen = strspn( parm, "0123456789" );
                }
                if ( index > 0 )
                {
                    testrate = rateforchans[index-1];
                    while ( index < 15 )
                        rateforchans[index++] = testrate;
                }
                break;
        }
    }
    else
    {
        strcpy( currentfile, parm );
        strupr( currentfile );
    }
}


/*************************************************************************/
/*                                                                       */
/* process_env() routine.  Checks the environment for a variable named   */
/* TTPARMS and, if found, uses the variable to set command-line parame-  */
/* ters.  Returns nothing.                                               */
/*                                                                       */
/*************************************************************************/

static void process_env( void )
{
    char envvar[256];           /* TTPARMS string */
    char *envptr;               /* pointer to string from environment */

    envptr = getenv( "TTPARMS" );
    if ( *envptr == NULL )
        return;
    if ( strlen( envptr ) > 255 )
        return;
    strcpy( envvar, envptr );
    envptr = strtok( envvar, " \t" );
    while ( envptr != NULL )
    {
        process_parm( envptr );
        envptr = strtok( NULL, " \t" );
    }
}


/*************************************************************************/
/*                                                                       */
/* fast_forward() routine.  Implements the fast-forward function,        */
/* including handling repeated keystrokes.                               */
/*                                                                       */
/*************************************************************************/

void fast_forward( void )
{
    int patentry;               /* current pattern entry number */
    unsigned long timewas;      /* "before" time */
    unsigned long timeis;       /* "after" time */
    unsigned entryincr;         /* increment to next pattern entry */

    /* Suspend playback. */
    dacstop();

    /* Get the current time, set previous = current. */
    timewas = timeis = getsystime();

    /* Tell the user what's going on. */
    vidputmsg( fastforwardmsg );

    /* Find out where DMA is, and from that, what pattern entry we're 
       playing now.  If we're already playing the last entry in the pattern 
       table, wait until the user lets go of the ">" key, then resume 
       playback and return. */
    if ( (patentry = bufsegrecs[dmaseg].bufsegentry) >= pattablesize-1 )
    {
        /* Wait 11 timer ticks (about 0.6 seconds) for the typematic delay 
           (i.e., wait while current-previous < 11), filling buffers if 
           possible.  We won't return until the user lets go of the ">" 
           key.  Note that we don't need any special test for midnight 
           rollover since if that happens timeis-timewas will be negative, 
           or rather very large when construed as an unsigned number. */ 
        while ( timeis-timewas < 11 )
        {
            /* If there is room, fill a 2k buffer segment. */
            if ( currentseg != dmaseg )
            {
                dmaflags[currentseg] = getsamples( currentseg, 0 );
                currentseg = (currentseg+1) & 0x1f;
            }

            /* Get the current time. */
            timeis = getsystime();
        }

        /* Set previous time = current. */
        timewas = timeis;

        /* The typematic delay time has elapsed.  Wait until the user lets 
           go of the key, filling buffers. */
        while ( vidpollchar( GREATERTHAN ) )
        {
            /* Wait 3 timer ticks (about 0.16 seconds) for the typematic 
               repeat (i.e., wait while current-previous < 3), filling 
               buffers if possible. */
            while ( timeis-timewas < 3 )
            {
                /* If there is room, fill a 2k buffer segment. */
                if ( currentseg != dmaseg )
                {
                    dmaflags[currentseg] = getsamples( currentseg, 0 );
                    currentseg = (currentseg+1) & 0x1f;
                }

                /* Check the time. */
                timeis = getsystime();
            }

            /* Set previous time = current. */
            timewas = timeis;
        }

        /* Resume playback. */
        vidputmsg( continuemsg );
        dacresume();
        return;
    }

    /* We're not already at the end, so some actual fast-forwarding is to 
       be done.  We'll restart DMA at the first buffer segment, ultimately. 
       */
    dmawas = 0;

    /* Increment the pattern table entry and "start playback" at the new 
       pattern. */
    dmaflags[0] = getsamples( 0, 1, 0, ++patentry );

    /* Display the new pattern table entry number and pattern number.  Set 
       the note number onscreen to 0. */
    vidputentry( patentry );
    vidputpatnum( patterntable[patentry] );
    vidputnotenum( 0 );

    /* Fill buffer segments until 11 timer ticks (about 0.6 seconds) have 
       elapsed.  The default typematic delay is 0.5 seconds, and we need to 
       wait that long to see if the user is holding down the ">" key.  The 
       time might as well be used profitably if possible. */
    currentseg = 1;
    while ( timeis-timewas < 11 )
    {
        /* If there is room, fill a 2k buffer segment. */
        if ( currentseg )
        {
            dmaflags[currentseg] = getsamples( currentseg, 0 );
            currentseg = (currentseg+1) & 0x1f;
        }

        /* Check the time. */
        timeis = getsystime();
    }

    /* Set previous time = current. */
    timewas = timeis;

    /* If we're not already at the end of the pattern table, continue to 
       advance the pattern number while the user is holding down the ">" 
       key and while we're not at the end of the pattern table yet. */
    entryincr = vidpollchar( GREATERTHAN );
    if ( patentry < pattablesize )
    {
        if ( (patentry += entryincr) >= pattablesize )
            patentry = pattablesize-1;
        while ( entryincr )
        {
            /* "Start playback" at the new pattern. */
            dmaflags[0] = getsamples( 0, 1, 0, patentry );

            /* Display the new pattern table entry number and pattern 
               number. */
            vidputentry( patentry );
            vidputpatnum( patterntable[patentry] );

            /* Fill buffer segments until 3 timer ticks (about 0.16 
               seconds) have elapsed.  The default typematic rate is 10.9 
               per second, so we need to wait at least 0.1 seconds here.  
               Again, the time might as well be used profitably. */
            currentseg = 1;
            while ( timeis-timewas < 3 )
            {
                /* If there is room, fill a 2k buffer segment. */
                if ( currentseg )
                {
                    dmaflags[currentseg] = getsamples( currentseg, 0 );
                    currentseg = (currentseg+1) & 0x1f;
                }

                /* Check the time. */
                timeis = getsystime();
            }

            /* Set previous time = current. */
            timewas = timeis;

            /* Check whether the user is continuing to hold down the ">" 
               key. */
            entryincr = vidpollchar( GREATERTHAN );

            /* If at the end of the pattern table, stop fast-forwarding. */
            if ( patentry == pattablesize-1 )
                break;

            /* Go to next entry in pattern table. */
            if ( (patentry += entryincr) >= pattablesize )
                patentry = pattablesize-1;
        }
    }

    /* Wait until the user lets go of the ">" key, filling buffers if 
       possible. */
    while ( entryincr )
    {
        /* Wait 3 timer ticks for the typematic delay, filling buffers 
           if possible. */
        while ( timeis-timewas < 3 )
        {
            /* If there is room, fill a 2k buffer segment. */
            if ( currentseg )
            {
                dmaflags[currentseg] = getsamples( currentseg, 0 );
                currentseg = (currentseg+1) & 0x1f;
            }

            /* Check the time. */
            timeis = getsystime();
        }

        /* Set previous time = current. */
        timewas = timeis;

        /* See if the user is still holding the key down. */
        entryincr = vidpollchar( GREATERTHAN );
    }

    /* Restart song playback at the new position.  Note, the DMA buffer is 
       probably not full at this point, we are behind, so the next user 
       keystroke won't be processed for a few seconds (that was the whole 
       point of handling typematic here). */
    vidputmsg( continuemsg );
    vidputnums( 0 );
    dacstart();
}


/*************************************************************************/
/*                                                                       */
/* rewind() routine.  Implements the rewind function, including handling */
/* repeated keystrokes.                                                  */
/*                                                                       */
/*************************************************************************/

void rewind( void )
{
    int patentry;               /* current pattern entry number */
    unsigned long timewas;      /* "before" time */
    unsigned long timeis;       /* "after" time */
    unsigned entryincr;         /* increment to next pattern entry */

    /* Suspend playback. */
    dacstop();

    /* Get the current time, set previous = current. */
    timewas = timeis = getsystime();

    /* Tell the user what's going on. */
    vidputmsg( rewindmsg );

    /* Find out where DMA is, and from that, what pattern entry we're 
       playing now.  If we're already playing the first entry in the 
       pattern table, wait until the user lets go of the "<" key, then 
       resume playback and return. */
    if ( !(patentry = bufsegrecs[dmaseg].bufsegentry) )
    {
        /* Wait 11 timer ticks (about 0.6 seconds) for the typematic delay 
           (i.e., wait while current-previous < 11), filling buffers if 
           possible.  We won't return until the user lets go of the "<" 
           key.  Note that we don't need any special test for midnight 
           rollover since if that happens timeis-timewas will be negative, 
           or rather very large when construed as an unsigned number. */ 
        while ( timeis-timewas < 11 )
        {
            /* If there is room, fill a 2k buffer segment. */
            if ( currentseg != dmaseg )
            {
                dmaflags[currentseg] = getsamples( currentseg, 0 );
                currentseg = (currentseg+1) & 0x1f;
            }

            /* Get the current time. */
            timeis = getsystime();
        }

        /* Set previous time = current. */
        timewas = timeis;

        /* The typematic delay time has elapsed.  Wait until the user lets 
           go of the key, filling buffers. */
        while ( vidpollchar( LESSTHAN ) )
        {
            /* Wait 3 timer ticks (about 0.16 seconds) for the typematic 
               repeat (i.e., wait while current-previous < 3), filling 
               buffers if possible. */
            while ( timeis-timewas < 3 )
            {
                /* If there is room, fill a 2k buffer segment. */
                if ( currentseg != dmaseg )
                {
                    dmaflags[currentseg] = getsamples( currentseg, 0 );
                    currentseg = (currentseg+1) & 0x1f;
                }

                /* Check the time. */
                timeis = getsystime();
            }

            /* Set previous time = current. */
            timewas = timeis;
        }

        /* Resume playback. */
        vidputmsg( continuemsg );
        dacresume();
        return;
    }

    /* We're not already at the beginning, so some actual rewinding is to 
       be done.  We'll restart DMA at the first buffer segment, ultimately. 
       */
    dmawas = 0;

    /* Decrement the pattern table entry and "start playback" at the new 
       pattern. */
    dmaflags[0] = getsamples( 0, 1, 0, --patentry );

    /* Display the new pattern table entry number and pattern number.  Set 
       the note number onscreen to 0. */
    vidputentry( patentry );
    vidputpatnum( patterntable[patentry] );
    vidputnotenum( 0 );

    /* Fill buffer segments until 11 timer ticks (about 0.6 seconds) have 
       elapsed.  The default typematic delay is 0.5 seconds, and we need to 
       wait that long to see if the user is holding down the "<" key.  The 
       time might as well be used profitably if possible. */
    currentseg = 1;
    while ( timeis-timewas < 11 )
    {
        /* If there is room, fill a 2k buffer segment. */
        if ( currentseg )
        {
            dmaflags[currentseg] = getsamples( currentseg, 0 );
            currentseg = (currentseg+1) & 0x1f;
        }

        /* Check the time. */
        timeis = getsystime();
    }

    /* Set previous time = current. */
    timewas = timeis;

    /* If we're not already at the beginning of the song, continue to 
       decrement the pattern number while the user is holding down the "<" 
       key and while we're not at the beginning of the song yet. */
    entryincr = vidpollchar( LESSTHAN );
    if ( patentry )
    {
        if ( (patentry -= entryincr) < 0 )
            patentry = 0;
        while ( entryincr )
        {
            /* "Start playback" at the new pattern. */
            dmaflags[0] = getsamples( 0, 1, 0, patentry );

            /* Display the new pattern table entry number and pattern 
               number. */
            vidputentry( patentry );
            vidputpatnum( patterntable[patentry] );

            /* Fill buffer segments until 3 timer ticks (about 0.16 
               seconds) have elapsed.  The default typematic rate is 10.9 
               per second, so we need to wait at least 0.1 seconds here.  
               Again, the time might as well be used profitably. */
            currentseg = 1;
            while ( timeis-timewas < 3 )
            {
                /* If there is room, fill a 2k buffer segment. */
                if ( currentseg )
                {
                    dmaflags[currentseg] = getsamples( currentseg, 0 );
                    currentseg = (currentseg+1) & 0x1f;
                }

                /* Check the time. */
                timeis = getsystime();
            }

            /* Set previous time = current. */
            timewas = timeis;

            /* Check whether the user is continuing to hold down the "<" 
               key. */
            entryincr = vidpollchar( LESSTHAN );

            /* If at the beginning of the song, stop rewinding. */
            if ( !patentry )
                break;

            /* Go to the previous entry in the pattern table. */
            if ( (patentry -= entryincr) < 0 )
                patentry = 0;
        }
    }

    /* Wait until the user lets go of the "<" key, filling buffers if 
       possible. */
    while ( entryincr )
    {
        /* Wait 3 timer ticks for the typematic delay, filling buffers 
           if possible. */
        while ( timeis-timewas < 3 )
        {
            /* If there is room, fill a 2k buffer segment. */
            if ( currentseg )
            {
                dmaflags[currentseg] = getsamples( currentseg, 0 );
                currentseg = (currentseg+1) & 0x1f;
            }

            /* Check the time. */
            timeis = getsystime();
        }

        /* Set previous time = current. */
        timewas = timeis;

        /* See if the user is still holding the key down. */
        entryincr = vidpollchar( LESSTHAN );
    }

    /* Restart song playback at the new position.  Note, the DMA buffer is 
       not full at this point, we are behind, so the next user keystroke 
       won't be processed for a few seconds (that was the whole point of 
       handling typematic here). */
    vidputmsg( continuemsg );
    vidputnums( 0 );
    dacstart();
}


/*************************************************************************/
/*                                                                       */
/* Main program.                                                         */
/*                                                                       */
/*************************************************************************/

int main( int argc, char *argv[] )
{
    char *errmsg;               /* error message returned by subroutines */
    int parm;                   /* command-line parameter number */
    int keepgoing;              /* set to 0 when the user wants to exit */
    unsigned charcode;          /* user keystroke */
    char *inbuf;                /* input string returned by getinput() */
    static char tempstr[80];    /* for formatting string output */
    unsigned long ulong;        /* for numeric input */
    char *endptr;               /* for numeric input */
    unsigned u;                 /* for looping */
    int error;                  /* for error control */

    /* Verify the presence of a Tandy DAC. */
    if ( dacdetect() )
    {
        printf( needdacmsg );
        exit( 1 );
    }

    /* Initialize the memory management module. */
    if ( initmem() )
    {
        printf( badmemmsg );
        exit( 2 );
    }

    /* Hook the <control>-C, <control>-<break>, and critical error 
       interrupts.  Note:  from this point forward, we must be sure to 
       unhook <control>-<break> before we exit, ex., by exiting through 
       the panic() routine. */
    hookints();

    /* Process the TTPARMS variable, if set. */
    process_env();

    /* Process command-line paramters. */
    for ( parm = 1; parm < argc; parm++ )
        process_parm( argv[parm] );

    /* Set up our screen.  From this point forward, we must be sure to 
       reset the video to the default before we exit, ex., by exiting 
       through panic().  Command-line parameters must be processed before 
       we initialize video so that the correct settings will be displayed 
       to the user. */
    initvid();

    /* Set up the sound hardware.  This needs to be done after the memory 
       module has been initialized, and after the command-line parameters 
       have been processed, but before the player module is set up (we need 
       the volume setting (to ramp up the sound chip) and the output 
       sampling rate (to compute the initial DAC divider) from the command 
       line, and we need to use the memory module to allocate a DMA buffer, 
       which must be allocated before the player module allocates any 
       memory).  From this point forward, we must be sure to reset the 
       sound hardware and unhook the sound chip interrupt before we exit. 
       */
    if ( initdac() )
        panic( nodmamsg, 2 );

    /* Set up the player module.  This needs to be done after the memory 
       module has been initialized and after the sound hardware has been 
       set up.  Note that the player module does not need to be finalized 
       at termination. */
    if ( initmodstuf() )
        panic( nomixmsg, 2 );
    playerstate = 1;

    /* Load the file given on the command line.  The player module needs to 
       be set up before we can load a file. */
    if ( currentfile[0] != '\0' )
    {
        vidputmsg( loadfilemsg );
        if ( loadfile( currentfile, &errmsg ) )
        {
            vidputerrmsg( errmsg );
            currentfile[0] = '\0';
        }
        else
        {
            vidputmsg( loaddonemsg );
            vidputfilnam( currentfile );
            vidputentry( 0 );
            vidputnentries();
            vidputpatnum( patterntable[0] );
            vidputnpatterns();
            vidputnotenum( 0 );
            vidputticksper( ticksdefl );
            vidputnchans();
            vidputninstrs();
            vidputbpm( bpmdefl );
            vidputglobalvol( globalvoldefl );
            dacsetspeed( rateforchans[(nchannels-4) / 2] );
            vidputrate();
            playerstate = 2;
        }
    }

    /* Main program loop. */
    keepgoing = 1;
    while ( keepgoing )
    {
        /* If we're playing sound, check whether it's time to stop or to 
           generate more. */
        if ( playerstate > 2 )
        {
            switch ( dmaflags[dmaseg = dacgetpart()] )
            {
                /* Sound continuing.  If DMA is not in progress where we 
                   want to put new samples, put them there. */
                case 0:
                    if ( playerstate == 3 )
                    {
                        vidupdatenums( dmawas, dmaseg );
                        dmawas = dmaseg;
                    }
                    while ( dmaseg != currentseg )
                    {
                        dmaflags[currentseg] = getsamples( currentseg, 0 );
                        currentseg = (currentseg+1) & 0x1f;
                    }
                    break;

                /* End of sound. */
                case 1:
                    dacstop();
                    playerstate = 2;
                    vidputmsg( playdonemsg );
                    vidputentry( 0 );
                    vidputpatnum( patterntable[0] );
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    break;

                /* Invalid - buffer underflow. */
                case -1:
                    dacstop();
                    playerstate = 2;
                    vidputerrmsg( toofastmsg );
                    vidputentry( 0 );
                    vidputpatnum( patterntable[0] );
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    break;

                /* Invalid value. */
                default:
                    dacstop();
                    playerstate = 2;
                    sprintf( tempstr, badbufmsg, dmaflags[dmaseg] );
                    vidputerrmsg( tempstr );
                    vidputentry( 0 );
                    vidputpatnum( patterntable[0] );
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    break;
            }
        }

        /* Check for a keystroke from the user and process it if there is 
           one. */
        if ( charcode = vidgetnchar() )
            switch ( charcode )
            {
                /* Load file. */
                case LOWERL:
                    /* Can't load if currently playing. */
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( inprogressmsg );
                        break;
                    }

                    /* Get filename from user. */
                    vidputmsg( enternamemsg );
                    if ( getinput( &inbuf ) )
                    {
                        vidputmsg( notloadedmsg );
                        break;
                    }

                    /* If a file is loaded now, unload it. */
                    if ( playerstate == 2 )
                    {
                        unloadfile();
                        vidputfilnam( noneloadedmsg );
                        currentfile[0] = '\0';
                        vidputentry( 0 );
                        vidputnentries();
                        vidputpatnum( 0 );
                        vidputnpatterns();
                        vidputnotenum( 0 );
                        vidputticksper( 6 );
                        vidputbpm( 125 );
                        vidputglobalvol( 0 );
                        vidputnchans();
                        vidputninstrs();
                        playerstate = 1;
                    }

                    /* Try to load the file. */
                    vidputmsg( loadfilemsg );
                    if ( loadfile( inbuf, &errmsg ) )
                    {
                        vidputerrmsg( errmsg );
                        break;
                    }

                    /* File loaded.  Update the screen. */
                    strcpy( currentfile, inbuf );
                    strupr( currentfile );
                    vidputmsg( loaddonemsg );
                    vidputfilnam( currentfile );
                    vidputentry( 0 );
                    vidputnentries();
                    vidputpatnum( patterntable[0] );
                    vidputnpatterns();
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    vidputnchans();
                    vidputninstrs();
                    dacsetspeed( rateforchans[(nchannels-4) / 2] );
                    vidputrate();
                    playerstate = 2;
                    break;

                /* Play file. */
                case LOWERP:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }

                    /* If playback is already in progress, just stop it. */
                    if ( playerstate > 2 )
                        dacstop();

                    /* Fill the DMA buffer with samples, at least as far as 
                       possible.  The first call to getsamples() always returns 
                       0. */
                    vidputmsg( fillbufmsg );
                    dmaflags[0] = getsamples( 0, 1, 0, 0 );
                    for ( u=1; u<32; u++ )
                        dmaflags[u] = getsamples( u, 0 );

                    /* Set player state to "playing song," and start DMA at 
                       the beginning of the DMA buffer. */
                    vidputmsg( playsongmsg );
                    vidputnums( 0 );
                    playerstate = 3;
                    currentseg = 0;
                    dmawas = 0;
                    dacstart();
                    break;

                /* Jump to pattern entry. */
                case LOWERJ:
                    /* If not playing a song, just ignore the keystroke. */
                    if ( playerstate != 3 )
                        break;

                    /* Suspend playback. */
                    dacstop();

                    /* Ask the user which position to jump to. */
                    sprintf( tempstr, enterposmsg, pattablesize-1 );
                    vidputmsg( tempstr );
                    if ( getinput( &inbuf ) )
                    {
                        /* If the user cancelled out of the box, resume 
                           playback. */
                        vidputmsg( continuemsg );
                        dacresume();
                        break;
                    }

                    /* Convert and verify the pattern table entry number. */
                    ulong = strtoul( inbuf, &endptr, 10 );
                    if ( *endptr || ulong >= pattablesize )
                    {
                        /* If the user entered an invalid pattern table 
                           entry, leave playback stopped. */
                        vidputerrmsg( badposmsg );
                        playerstate = 2;
                        vidputentry( 0 );
                        vidputpatnum( patterntable[0] );
                        vidputnotenum( 0 );
                        vidputticksper( ticksdefl );
                        vidputbpm( bpmdefl );
                        vidputglobalvol( globalvoldefl );
                        break;
                    }

                    /* Fill the DMA buffer with samples, at least as far as 
                       possible.  The first call to getsamples() always returns 
                       0. */
                    vidputmsg( fillbufmsg );
                    u = ulong;
                    dmaflags[0] = getsamples( 0, 1, 0, u );
                    for ( u=1; u<32; u++ )
                        dmaflags[u] = getsamples( u, 0 );

                    /* Set player state to "playing song," and start DMA at 
                       the beginning of the DMA buffer. */
                    vidputmsg( continuemsg );
                    vidputnums( 0 );
                    currentseg = 0;
                    dmawas = 0;
                    dacstart();
                    break;

                /* Rewind. */
                case LESSTHAN:
                    /* If not playing a song, just ignore the keystroke. */
                    if ( playerstate != 3 )
                        break;

                    /* Call a subroutine to do this; it's fairly involved. 
                       */
                    rewind();
                    break;

                /* Fast forward. */
                case GREATERTHAN:
                    /* If not playing a song, just ignore the keystroke. */
                    if ( playerstate != 3 )
                        break;

                    /* Call a subroutine to do this; it's fairly involved. 
                       */
                    fast_forward();
                    break;

                /* Pause playback. */
                case UPPERP:
                    if ( playerstate > 2 )
                    {
                        dacstop();
                        vidputmsg( playpausemsg );
                    }
                    break;

                /* Resume playback. */
                case UPPERR:
                    if ( playerstate > 2 )
                    {
                        dacresume();
                        vidputmsg( continuemsg );
                    }
                    break;

                /* Stop playback. */
                case LOWERS:
                    if ( playerstate > 2 )
                    {
                        dacstop();
                        playerstate = 2;
                        vidputmsg( playhaltmsg );
                        vidputentry( 0 );
                        vidputpatnum( patterntable[0] );
                        vidputnotenum( 0 );
                        vidputticksper( ticksdefl );
                        vidputbpm( bpmdefl );
                        vidputglobalvol( globalvoldefl );
                    }
                    break;

                /* Unload file. */
                case LOWERU:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( nounloadmsg );
                        break;
                    }
                    unloadfile();
                    vidputfilnam( noneloadedmsg );
                    currentfile[0] = '\0';
                    vidputentry( 0 );
                    vidputnentries();
                    vidputpatnum( 0 );
                    vidputnpatterns();
                    vidputnotenum( 0 );
                    vidputticksper( 6 );
                    vidputbpm( 125 );
                    vidputglobalvol( 0 );
                    vidputnchans();
                    vidputninstrs();
                    playerstate = 1;
                    vidputmsg( unloadmsg );
                    break;

                /* Rip sample to file. */
                case LOWERR:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( noripmsg );
                        break;
                    }
                    sprintf( tempstr, enterripmsg, ninstruments );
                    vidputmsg( tempstr );
                    if ( getinput( &inbuf ) )
                    {
                        vidputmsg( noripmsg2 );
                        break;
                    }
                    ulong = strtoul( inbuf, &endptr, 10 );
                    if ( *endptr || ulong < 1 || ulong > ninstruments )
                    {
                        vidputerrmsg( badsamnummsg );
                        break;
                    }
                    if ( dumpsample( (unsigned) ulong, currentfile, &errmsg ) )
                    {
                        vidputerrmsg( errmsg );
                        break;
                    }
                    vidputmsg( samsavemsg );
                    break;

                /* Rip all samples to files. */
                case CTRLR:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( noripmsg );
                        break;
                    }
                    for ( u = 1, error = 0; u <= ninstruments; u++ )
                    {
                        /* Skip samples that are only copies of the default 
                           silence sample. */
                        if ( samplerecs[u].sampseg == samplerecs[0].sampseg )
                            continue;

                        /* Dump the sample and tell the user about it. */
                        if ( dumpsample( u, currentfile, &errmsg ) )
                        {
                            vidputerrmsg( errmsg );
                            error = 1;
                            break;
                        }
                        sprintf( tempstr, samwritemsg, u );
                        vidputmsg( tempstr );
                    }
                    if ( !error )
                        vidputmsg( samsavemsg2 );
                    break;

                /* Play sample. */
                case UPPERS:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }

                    /* If playback is already in progress, just stop it. */
                    if ( playerstate > 2 )
                        dacstop();

                    /* Ask the user which sample to play. */
                    sprintf( tempstr, entersammsg, ninstruments );
                    vidputmsg( tempstr );
                    if ( getinput( &inbuf ) )
                    {
                        /* If the user cancelled out of the box, resume 
                           playback if in progress before. */
                        if ( playerstate > 2 )
                            dacresume();
                        break;
                    }

                    /* Convert and verify the sample number. */
                    ulong = strtoul( inbuf, &endptr, 10 );
                    if ( *endptr || ulong < 1 || ulong > ninstruments )
                    {
                        /* If the user entered an invalid sample number, leave 
                           playback stopped. */
                        vidputerrmsg( badsamnummsg );
                        playerstate = 2;
                        vidputentry( 0 );
                        vidputpatnum( patterntable[0] );
                        vidputnotenum( 0 );
                        vidputticksper( ticksdefl );
                        vidputbpm( bpmdefl );
                        vidputglobalvol( globalvoldefl );
                        break;
                    }

                    /* Fill the DMA buffer with samples, at least as far as 
                       possible.  The first call to getsamples() always returns 
                       0. */
                    vidputmsg( fillbufmsg );
                    u = ulong;
                    dmaflags[0] = getsamples( 0, 1, u );
                    for ( u=1; u<32; u++ )
                        dmaflags[u] = getsamples( u, 0 );

                    /* Set player state to "playing sample," and start DMA at 
                       the beginning of the DMA buffer. */
                    vidputmsg( playsammsg );
                    vidputentry( 0 );
                    vidputpatnum( patterntable[0] );
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    playerstate = 4;
                    currentseg = 0;
                    dacstart();
                    break;

                /* Dump module data to file. */
                case CTRLD:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( nodumpmsg );
                        break;
                    }
                    vidputmsg( enterdumpmsg );
                    if ( getinput( &inbuf ) )
                    {
                        vidputmsg( nodumpmsg2 );
                        break;
                    }
                    vidputmsg( savemodmsg );
                    if ( dumpmod( inbuf, &errmsg ) )
                    {
                        vidputerrmsg( errmsg );
                        break;
                    }
                    vidputmsg( savemodmsg2 );
                    break;

                /* Decrease volume. */
                case LOWERV:
                    if ( volume )
                    {
                        dacsetvol( volume-1 );
                        vidputvolume();
                        vidputmsg( volsetmsg );
                    }
                    break;

                /* Increase volume. */
                case UPPERV:
                    if ( volume < 7 )
                    {
                        dacsetvol( volume+1 );
                        vidputvolume();
                        vidputmsg( volsetmsg );
                    }
                    break;

                /* Toggle loop disable. */
                case UPPERL:
                    loopdisable ^= 1;
                    vidputloopen();
                    vidputmsg( loopsetmsg );
                    break;

                /* Set output sampling rate. */
                case CTRLO:
                    if ( playerstate < 2 )
                    {
                        vidputerrmsg( noneloadedmsg2 );
                        break;
                    }
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( noratemsg );
                        break;
                    }
                    vidputmsg( enterratemsg );
                    if ( getinput( &inbuf ) )
                    {
                        vidputmsg( noratemsg2 );
                        break;
                    }
                    ulong = strtoul( inbuf, &endptr, 10 );
                    if ( *endptr || ulong < 4000 || ulong > 44100 )
                    {
                        vidputerrmsg( badratemsg );
                        break;
                    }
                    u = ulong;
                    dacsetspeed( u );
                    vidputrate();
                    vidputmsg( ratesetmsg );
                    break;

                /* Exit program. */
                case ESC:
                    keepgoing = 0;
                    break;

                /* Decrease ticks/note. */
                case LOWERT:
                    if ( playerstate >= 2 )
                        if ( ticksperrow > 1 )
                        {
                            sprintf( tempstr, ticksetmsg, --ticksperrow );
                            ticksdefl = ticksperrow;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Increase ticks/note. */
                case UPPERT:
                    if ( playerstate >= 2 )
                        if ( ticksperrow < MAXTICKS )
                        {
                            sprintf( tempstr, ticksetmsg, ++ticksperrow );
                            ticksdefl = ticksperrow;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Decrease bpm. */
                case LOWERB:
                    if ( playerstate >= 2 )
                        if ( beatspermin > MAXTICKS+1 )
                        {
                            setbeats( beatspermin-1 );
                            sprintf( tempstr, bpmsetmsg, beatspermin );
                            bpmdefl = beatspermin;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Increase bpm. */
                case UPPERB:
                    if ( playerstate >= 2 )
                        if ( beatspermin < 255 )
                        {
                            setbeats( beatspermin+1 );
                            sprintf( tempstr, bpmsetmsg, beatspermin );
                            bpmdefl = beatspermin;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Select file. */
                case LOWERF:
                    /* Can't load if currently playing. */
                    if ( playerstate > 2 )
                    {
                        vidputerrmsg( inprogressmsg );
                        break;
                    }

                    /* Get filename from user. */
                    error = fileselect( &inbuf, &errmsg );
                    if ( error == 1 )
                    {
                        vidputmsg( notloadedmsg );
                        break;
                    }
                    else if ( error == -1 )
                    {
                        vidputerrmsg( errmsg );
                        break;
                    }

                    /* If a file is loaded now, unload it. */
                    if ( playerstate == 2 )
                    {
                        unloadfile();
                        vidputfilnam( noneloadedmsg );
                        currentfile[0] = '\0';
                        vidputentry( 0 );
                        vidputnentries();
                        vidputpatnum( 0 );
                        vidputnpatterns();
                        vidputnotenum( 0 );
                        vidputticksper( 6 );
                        vidputbpm( 125 );
                        vidputglobalvol( 0 );
                        vidputnchans();
                        vidputninstrs();
                        playerstate = 1;
                    }

                    /* Try to load the file. */
                    vidputmsg( loadfilemsg );
                    if ( loadfile( inbuf, &errmsg ) )
                    {
                        vidputerrmsg( errmsg );
                        break;
                    }

                    /* File loaded.  Update the screen. */
                    strcpy( currentfile, inbuf );
                    strupr( currentfile );
                    vidputmsg( loaddonemsg );
                    vidputfilnam( currentfile );
                    vidputentry( 0 );
                    vidputnentries();
                    vidputpatnum( patterntable[0] );
                    vidputnpatterns();
                    vidputnotenum( 0 );
                    vidputticksper( ticksdefl );
                    vidputbpm( bpmdefl );
                    vidputglobalvol( globalvoldefl );
                    vidputnchans();
                    vidputninstrs();
                    dacsetspeed( rateforchans[(nchannels-4) / 2] );
                    vidputrate();
                    playerstate = 2;
                    break;

                /* Decrease global volume. */
                case LOWERG:
                    if ( playerstate >= 2 )
                        if ( globalvol > 0 )
                        {
                            sprintf( tempstr, gvolsetmsg, --globalvol );
                            globalvoldefl = globalvol;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Increase global volume. */
                case UPPERG:
                    if ( playerstate >= 2 )
                        if ( globalvol < 64 )
                        {
                            sprintf( tempstr, gvolsetmsg, ++globalvol );
                            globalvoldefl = globalvol;
                            vidputmsg( tempstr );
                        }
                    break;

                /* Toggle EMS enable. */
                case ALTX:
                    emsenable ^= 1;
                    vidputemsen();
                    vidputmsg( emssetmsg );
                    break;

                /* Toggle save memory. */
                case ALTS:
                    if ( savemem ^= 1 )
                        vidputmsg( savememonmsg );
                    else
                        vidputmsg( savememoffmsg );
                    vidputsavemem();
                    break;

                /* Toggle loading optimization. */
                case ALTO:
                    optimize ^= 1;
                    vidputloadopt();
                    vidputmsg( optimizsetmsg );
                    break;

                /* Redraw command screen (undocumented). */
                case F1:
                    vidputcmds();
                    break;

                default:
                    break;
            }
    }

    /* Reset the sound hardware. */
    enddac();

    /* Reset the video back to the default. */
    endvid();

    /* Unhook the <control>-<break> interrupt. */
    unhookints();

    /* Finalize the memory management module. */
    endmem();

    /* Return "success." */
    return( 0 );
}
