/*
	Basic Soundblaster access & detection routines
	==============================================
*/

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <conio.h>
#include "mtypes.h"
#include "sbio.h"
#include "mdriver.h"




void SB_MixerStereo(void)
/*
	Enables stereo output for DSP versions 3.00 >= ver < 4.00
*/
{
	outportb(MIXER_ADDRESS,0xe);
	outportb(MIXER_DATA,inportb(MIXER_DATA)|2);
}



void SB_MixerMono(void)
/*
	Disables stereo output for DSP versions 3.00 >= ver < 4.00
*/
{
	outportb(MIXER_ADDRESS,0xe);
	outportb(MIXER_DATA,inportb(MIXER_DATA)&0xfd);
}



BOOL SB_WaitDSPWrite(void)
/*
	Waits until the DSP is ready to be written to.

	returns FALSE on timeout
*/
{
	UWORD timeout=32767;

	while(timeout--){
		if(!(inportb(DSP_WRITE_STATUS)&0x80)) return 1;
	}
	return 0;
}



BOOL SB_WaitDSPRead(void)
/*
	Waits until the DSP is ready to read from.

	returns FALSE on timeout
*/
{
	UWORD timeout=32767;

	while(timeout--){
		if(inportb(DSP_DATA_AVAIL)&0x80) return 1;
	}
	return 0;
}



BOOL SB_WriteDSP(UBYTE data)
/*
	Writes byte 'data' to the DSP.

	returns FALSE on timeout.
*/
{
	if(!SB_WaitDSPWrite()) return 0;
	outportb(DSP_WRITE_DATA,data);
	return 1;
}



UWORD SB_ReadDSP(void)
/*
	Reads a byte from the DSP.

	returns 0xffff on timeout.
*/
{
	if(!SB_WaitDSPRead()) return 0xffff;
	return(inportb(DSP_READ_DATA));
}



void SB_SpeakerOn(void)
/*
	Enables DAC speaker output.
*/
{
	SB_WriteDSP(0xd1);
}



void SB_SpeakerOff(void)
/*
	Disables DAC speaker output
*/
{
	SB_WriteDSP(0xd3);
}



void SB_ResetDSP(void)
/*
	Resets the DSP.
*/
{
	int t;
	// reset the DSP by sending 1, (delay), then 0
	outportb(DSP_RESET,1);
	for(t=0;t<8;t++) inportb(DSP_RESET);
	outportb(DSP_RESET,0);
}



BOOL SB_Ping(void)
/*
	Checks if a SB is present at the current baseport by
	resetting the DSP and checking if it returned the value 0xaa.

	returns: TRUE   => SB is present
			 FALSE  => No SB detected
*/
{
	SB_ResetDSP();
	return(SB_ReadDSP()==0xaa);
}



UWORD SB_GetDSPVersion(void)
/*
	Gets SB-dsp version. returns 0xffff if dsp didn't respond.
*/
{
	UWORD hi,lo;

	if(!SB_WriteDSP(0xe1)) return 0xffff;

	hi=SB_ReadDSP();
	lo=SB_ReadDSP();

	return((hi<<8)|lo);
}
