/*
  gdb2txt <hp200lx gdb file>

  Convert a hp200lx gdb file to a text file.
  One line per field, same format as print from the database application in
  hp200lx.

  Copyright (C) 1998-2001  Ulf Nordquist <un@pobox.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  Or available at ftp://prep.ai.mit.edu/pub/gnu/COPYING.

  todo

  Using information in the file DBDOC.TXT by
  Andrew J. Gryc
  Hewlett-Packard
  Corvallis, OR  USA

  by un@pobox.com
  980509 : read database header
           handle TYPE_CARDDEF TYPE_CATEGORY
  980510 :        TYPE_FIELDDEF TYPE_VIEWPTDEF TYPE_LINKDEF TYPE_NOTE TYPE_DATA
                  TYPE_VIEWPTTABLE TYPE_LOOKUPTABLE
           dump data fields
  980511 : remove '&' from name
           empty line after each data record
	   correct WORDBOOL_FIELD, checkmark before name
           correct RADIO_FIELD, find parent name
  980512 : handle NOTE_FIELD
  980630 : Ver 1.1
           -fc
	   rdrec() - end TYPE_DATA and TYPE_NOTE with '\0'
  980727 : print version to stdout along with usage
           set carddef in rdrec()
  010119 : Ver 1.2
           handle NOTE_FIELD with negative id
*/

#include <malloc.h>
#include <stdio.h>

#define FIELDDEF_NODATA    0x80 /* The fielddef does not have any associated
				   data */
#define FIELDDEF_RESERVED  0x40 /* The fielddef record is reserved
				   (don't use) */
#define FIELDDEF_RELATIVE  0x20 /* The dataoffset does not point to the actual
				   data, but points to a word.  This word is
				   the offset to the data. */
#define FIELDDEF_NULLTITLE 0x10 /* The window related to the fielddef has no
				   title. */

#define BYTEBOOL_FIELD  0 /* Checkbox: points to a bit in a byte */
#define WORDBOOL_FIELD  1 /* points to a bit in a word */
#define STRING_FIELD    2 /* Points to ASCIIZ string */
#define PHONE_FIELD     3 /* Treated as a String by the database */
#define NUMBER_FIELD    4 /* Numbers are stored as ASCIIZ strings,
			     but compare as numbers */
#define CURRENCY_FIELD  5 /* Treated as a number by the database */
#define CATEGORY_FIELD  6 /* Stored as an ASCIIZ string with ';' 
			     separating each selection */
#define TIME_FIELD      7 /* int: Minutes since midnight */
#define DATE_FIELD      8 /* points to char array[3]; 
			     array[0] is Year: 0 starts at 1900
			     array[1] is Month from 0..11
			     array[2] is Day from 0..30 */
#define RADIO_FIELD     9 /* dataoffset points to actual offset of 
			     radio button byte */
#define NOTE_FIELD      10 /* The field is a note (which resides in a
			      separate record); dataoffset points to the
			      record number of the note.  If no note is
			      attached, the record number pointed to should
			      be -1. */
#define GROUP_FIELD     11 /* Used for Groupboxes; no data is attached */
#define STATIC_FIELD    12 /* Used for Static Text; no data is attached */
#define MULTILINE_FIELD 13 /* Same as a string, 
			      but the field can have CR/LFs */
#define LIST_FIELD      14 /* Doubles as a STATIC_FIELD */
#define COMBO_FIELD     15 /* Doubles as a STRING_FIELD */
#define USER_FIELD      16 /* The applications can define their own
			      field types starting at USER_FIELD NOTE:
			      The FIELDDEF for user defined fields
			      MUST have the status bit set to
			      FIELDDEF_CALLBACK and a FieldCallback
			      function provided. */

#define TYPE_DBHEADER     0 /* Header for database.  First record in db. */
#define TYPE_CARDDEF      4 /* Information on the card layout. */
#define TYPE_CATEGORY     5 /* List of categories for the database. */
#define TYPE_FIELDDEF     6 /* Definition record for each field */
#define TYPE_VIEWPTDEF    7 /* Viewpoint (sort and subset) definition */
#define TYPE_NOTE         9 /* Note record */
#define TYPE_VIEWPTTABLE 10 /* Table of viewpoint entries */
#define TYPE_DATA        11 /* Data record */
#define TYPE_LINKDEF     12 /* Smart clip record */
#define TYPE_CARDPAGEDEF 13 /* Info struct for multiple card database */
/*      14..30 TYPE_USER       Defined by the application */
#define TYPE_LOOKUPTABLE 31 /* Pointers to all other records in the db. */

#define STYLE_NOSHADOW      0x00004000L
#define STYLE_COMBOEDIT     0x00008000L
#define STYLE_DATETIME      0x00010000L
#define STYLE_CHECKBOX      0x00020000L  /* Should be set for a check box */
#define STYLE_NO_PARENT_KEY 0x00040000L
#define STYLE_PUSHB_WIDTH   0x00080000L
#define STYLE_PUSHBUTTON    0x00100000L
#define STYLE_COMBOLIST     0x00200000L
#define STYLE_RADIO         0x00400000L  /* Should be set for a radio button */
#define STYLE_GRAY          0x00800000L
#define STYLE_XYRELATIVE    0x01000000L
#define STYLE_WHCHAR        0x02000000L
#define STYLE_NOBORDER      0x04000000L
#define STYLE_LEAF          0x08000000L
#define STYLE_NOTIFY        0x10000000L
#define STYLE_NOFOCUS       0x20000000L
#define STYLE_SAVEUNDER     0x40000000L
#define STYLE_INVISIBLE     0x80000000L

struct dbcategory
{
   char *string;
};

struct dbdata
{
   int  bytes;
   void *data;
};

struct dbheader
{
   int  release;
   char	type;
   char status;
   int  viewpt;
   int  records;
   int  lookup;
   int  year;
   char month;
   char date;
   int  minutes;
   int  hash;
};

struct fielddef
{
   char type;
   char id;
   int  offset;
   int  flags;
   int  reserved;
   char name[25];
};

struct packedwindow
{
   int count;
   struct windowdata *windows;
};

struct rechdr
{
   char type;
   char status;
   int  length;
   int  number;
   void *data;
};

struct windowdata
{
   int x;
   int y;
   int w;
   int h;
   int size;
   int style;
   int parent;
};

void dumprec(void);
void *myrealloc(void *ptr,int size);

int debug;

struct rechdr *records;  /* array of all records */
int *fields;             /* array of field definition records index */
int *notes;              /* array of note records index */
int fcnt,ncnt,rcnt;      /* counters for the arrays */
int format;              /* output format */
struct packedwindow *carddef;

main(int argc, char **argv)
{
   int ac;
   FILE *fp;
   char buf[500];

   /* decode switches */

   ac=1;
   debug=0;
   format=0;
   while((ac<argc)&&(argv[ac][0]=='-'))
   {
      switch(argv[ac][1])
      {
	 case 'd':
	    debug++;
	    break;
	 case 'f':
	    format=argv[ac][2];
	    if(format!='c')
	       fprintf(stderr,
		       "gdb2txt: Warning! Ignoring unknown format '%c'\n",
		       format);
	    break;
	 default:
	    fprintf(stderr,"gdb2txt: Warning! Ignoring unknown switch '%c'\n",
		    argv[ac][1]);
	    break;
      }
      ac++;
   }

   /* open gdb file */

   if(ac>=argc)
   {
      fprintf(stderr,"gdb2txt: Error! No input file given.\n");
      fprintf(stderr,"Usage: gdb2txt <hp200lx gdb file>\n");
      fprintf(stderr,"Version %s - Date %s\n",VERSION,DATE);
      fprintf(stderr,
	      "Copyright (C) 1998-2001  Ulf Nordquist <un@pobox.com>\n");
      fprintf(stderr,"Distributed under the terms of the ");
      fprintf(stderr,"GNU General Public License\n");
      return 1;
   }
   if(!(fp=fopen(argv[ac],"r")))
   {
      fprintf(stderr,"gdb2txt: Error! Cannot open %s for input.\n",argv[ac]);
      return 1;
   }

   /* convert it to text */
   /* check signature */

   if(fread(buf,4,1,fp)!=1)
   {
      fprintf(stderr,"gdb2txt: Error! Cannot read signature\n");
      return 1;
   }
   if(strcmp(buf,"hcD"))
   {
      fprintf(stderr,"gdb2txt: Error! %s is not a gdb file.\n",argv[ac]);
      return 1;
   }

   /* read records */

   fcnt=ncnt=rcnt=0;
   records=NULL;
   fields=NULL;
   carddef=NULL;
   while(!feof(fp))
   {
      int c;

      records=(struct rechdr*)myrealloc(records,
					(rcnt+1)*sizeof(struct rechdr));
      if(!rdrec(fp))
      {
	 fprintf(stderr,"gdb2txt: Error! Aborted in record %d at byte %d\n",
		 rcnt,ftell(fp));
	 rcnt++;
	 break;
      }
      if(!rcnt)
      {
	 /* first record must be database header */

	 if(records->type!=TYPE_DBHEADER)
	 {
	    fprintf(stderr,"gdb2txt: Error! First record is not a header\n");
	    exit(1);
	 }
      }

      /* remember index of field defs */

      if((records+rcnt)->type==TYPE_FIELDDEF)
      {
	 fields=(int*)myrealloc(fields,(fcnt+1)*sizeof(int));
	 *(fields+fcnt)=rcnt;
	 fcnt++;
      }

      /* remember index of notes */

      if((records+rcnt)->type==TYPE_NOTE)
      {
	 notes=(int*)myrealloc(notes,(ncnt+1)*sizeof(int));
	 *(notes+ncnt)=rcnt;
	 ncnt++;
      }

      /* next record */

      rcnt++;

      /* force EOF if end of file */

      c=fgetc(fp);
      ungetc(c,fp);
   }
   fclose(fp);
   dumprec();
   return 0;
}

int
rdrec(FILE *fp)
{
   int pos,r;
   char buf[500];
   struct rechdr *rhp;

   rhp=(records+rcnt);
   r=1;

   /* header */

   rhp->type=fgetc(fp);
   rhp->status=fgetc(fp);
   rhp->length=fgetshort(fp);
   rhp->number=fgetshort(fp);

   /* type specific data */

   pos=ftell(fp);
   switch(rhp->type)
   {
      case TYPE_CARDDEF:

	 /* TYPE_CARDDEF Information on the card layout. */

	 {
	    struct packedwindow *pwp;
	    struct windowdata *wdp;
	    int bytes;

	    if(carddef)
	    {
	       fprintf(stderr,"gdb2txt: Error! More than one TYPE_CARDDEF\n");
	       r=0;
	    }
	    pwp=(struct packedwindow*)myrealloc(NULL,
						sizeof(struct packedwindow));
	    carddef=pwp;
	    rhp->data=pwp;
	    bytes=rhp->length-6;
	    pwp->count=0;
	    pwp->windows=NULL;
	    while(bytes>0)
	    {
	       pwp->windows=(struct windowdata*)
		  myrealloc(pwp->windows,
			    ((pwp->count)+1)*sizeof(struct windowdata));
	       wdp=(pwp->windows)+(pwp->count);
	       fgetint(fp);  /* window class handler unused */
	       wdp->x=fgetshort(fp);
	       wdp->y=fgetshort(fp);
	       wdp->w=fgetshort(fp);
	       wdp->h=fgetshort(fp);
	       wdp->size=fgetshort(fp);
	       wdp->style=fgetint(fp);
	       wdp->parent=fgetshort(fp);
	       pwp->count++;
	       bytes=bytes-(4+5*2+4+2);
	    }
	 }
	 break;
      case TYPE_CATEGORY:

	 /* TYPE_CATEGORY List of categories for the database. */

	 {
	    struct dbcategory *dbcp;

	    dbcp=(struct dbcategory*)myrealloc(NULL,sizeof(struct dbcategory));
	    rhp->data=dbcp;
	    if(fread(buf,rhp->length-6,1,fp)!=1)
	    {
	       fprintf(stderr,"gdb2txt: Error! Unexpected end of file\n");
	       r=0;
	    }
	    dbcp->string=(char*)myrealloc(NULL,rhp->length-6);
	    if(buf[rhp->length-6-1])
	    {
	       fprintf(stderr,"gdb2txt: Error! Category string is not ");
	       fprintf(stderr,"NUL terminated\n");
	       r=1;
	    }
	    strcpy(dbcp->string,buf);
	 }
	 break;
      case TYPE_DATA:
      case TYPE_NOTE:

	 /* TYPE_DATA Data record */
	 /* TYPE_NOTE Note record */

	 {
	    struct dbdata *dbdp;

	    dbdp=(struct dbdata*)myrealloc(NULL,sizeof(struct dbdata));
	    rhp->data=dbdp;
	    dbdp->bytes=rhp->length-6;
	    dbdp->data=myrealloc(NULL,dbdp->bytes+1);
	    fread(dbdp->data,dbdp->bytes,1,fp);
	    ((char*)(dbdp->data))[dbdp->bytes]='\0';
	 }
	 break;
      case TYPE_DBHEADER:

	 /* TYPE_DBHEADER Header for database.  First record in db. */

	 {
	    struct dbheader *dbhp;

	    dbhp=(struct dbheader*)myrealloc(NULL,sizeof(struct dbheader));
	    rhp->data=dbhp;
	    dbhp->release=fgetshort(fp);
	    dbhp->type=fgetc(fp);
	    dbhp->status=fgetc(fp);
	    dbhp->viewpt=fgetshort(fp);
	    dbhp->records=fgetshort(fp);
	    dbhp->lookup=fgetint(fp);
	    dbhp->year=fgetc(fp);
	    dbhp->month=fgetc(fp);
	    dbhp->date=fgetc(fp);
	    dbhp->minutes=fgetshort(fp);
	    dbhp->hash=fgetshort(fp);
	 }
	 break;
      case TYPE_FIELDDEF:

	 /* TYPE_FIELDDEF Definition record for each field */

	 {
	    struct fielddef *fdp;
	    int bytes,i,ni;

	    fdp=(struct fielddef*)myrealloc(NULL,sizeof(struct fielddef));
	    rhp->data=fdp;
	    fdp->type=fgetc(fp);
	    fdp->id=fgetc(fp);
	    fdp->offset=fgetshort(fp);
	    fdp->flags=fgetc(fp);
	    fdp->reserved=fgetshort(fp);
	    bytes=rhp->length-6-2-2-1-2;
	    if(bytes>21)
	    {
	       fprintf(stderr,"gdb2txt: Error! Field def name is too long ");
	       fprintf(stderr,", %d bytes\n",bytes);
	       r=1;
	    }
	    fread(buf,bytes,1,fp);
	    for(i=0,ni=0;i<bytes;i++)
	    {
	       if(buf[i]!='&')
		  fdp->name[ni++]=buf[i];
	       if(!buf[i])
		  break;
	    }
	    if(i>=bytes)
	    {
	       fprintf(stderr,"gdb2txt: Error! Field def name is not ");
	       fprintf(stderr,"NUL terminated\n");
	       r=1;
	    }
	 }
	 break;
      case TYPE_LINKDEF:
      case TYPE_LOOKUPTABLE:
      case TYPE_VIEWPTDEF:
      case TYPE_VIEWPTTABLE:

	 /* TYPE_LINKDEF Smart clip record */
	 /* TYPE_LOOKUPTABLE Pointers to all other records in the db. */
	 /* TYPE_VIEWPTDEF Viewpoint (sort and subset) definition */
	 /* TYPE_VIEWPTTABLE Table of viewpoint entries */
	 /* skip it */

	 fseek(fp,pos+rhp->length-6,SEEK_SET);
	 if(rhp->type==TYPE_LOOKUPTABLE)
	 {
	    /* skip TypeFirst Table */

	    fseek(fp,64,SEEK_CUR);
	    pos=pos+64;
	 }
	 break;
      default:
	 fprintf(stderr,"gdb2txt: Warning! Unknown record type %d\n",
		 rhp->type);
	 r=0;
	 break;
   }

   /* check file position */

   if(ftell(fp)-pos-(rhp->length-6))
   {
      int i;

      i=ftell(fp);
      fprintf(stderr,"gdb2txt: Warning! Record size error\n");
      fseek(fp,pos+rhp->length-6,SEEK_SET);
      r=0;
   }
   return r;
}

void
dumprec(void)
{
   int i,r;
   struct rechdr *rhp;

   rhp=records;
   for(r=0;r<rcnt;r++)
   {
      if(debug)
      {
	 printf("=== Record %d ===\n",r);
	 printf("  Type:   %d - ",rhp->type);
	 switch(rhp->type)
	 {
	    case TYPE_CARDDEF:
	       printf("Information on the card layout");
	       break;
	    case TYPE_CATEGORY:
	       printf("List of categories");
	       break;
	    case TYPE_DATA:
	       printf("Data record");
	       break;
	    case TYPE_DBHEADER:
	       printf("Header for database");
	       break;
	    case TYPE_FIELDDEF:
	       printf("Field definition record");
	       break;
	    case TYPE_LINKDEF:
	       printf("Smart clip record");
	       break;
	    case TYPE_LOOKUPTABLE:
	       printf("Pointers to all other records in the db.");
	       break;
	    case TYPE_NOTE:
	       printf("Note record");
	       break;
	    case TYPE_VIEWPTDEF:
	       printf("Viewpoint (sort and subset) definition");
	       break;
	    case TYPE_VIEWPTTABLE:
	       printf("Table of viewpoint entries");
	       break;
	    default:
	       printf("Unknown record type");
	       break;
	 }
	 printf("\n");
	 printf("  Status: %d\n",rhp->status);
	 printf("  Length: %d\n",rhp->length);
	 printf("  Number: %d\n",rhp->number);
	 printf("  --- Data\n");
      }
      switch(rhp->type)
      {
	 case TYPE_CARDDEF:

	    /* TYPE_CARDDEF Information on the card layout. */

	    {
	       struct packedwindow *pwp;
	       struct windowdata *wdp;
	       int w;

	       if(carddef!=rhp->data)
	       {
		  fprintf(stderr,"gdb2txt: Error! carddef is corrupted\n");
		  exit(1);
	       }
	       pwp=carddef;
	       wdp=pwp->windows;
	       for(w=0;w<pwp->count;w++)
	       {
		  if(debug)
		  {
		     printf("  --- Window %d ---\n",w);
		     printf("  X pos:       %d\n",wdp->x);
		     printf("  Y pos:       %d\n",wdp->y);
		     printf("  Width:       %d\n",wdp->w);
		     printf("  Height:      %d\n",wdp->h);
		     if(wdp->style&STYLE_RADIO)
		     {
			printf("  Radio index: %d\n",wdp->size);
		     }
		     else
		     {
			if(wdp->style&STYLE_CHECKBOX)
			{
			   printf("  Check on:    ");
			   for(i=0;i<32;i++)
			      if(((wdp->size)>>i)&1)
				 printf("%d ",i);
			}
			else
			{
			   printf("  Size:        %d\n",wdp->size);
			}
		     }
		     printf("  Style:       0x%08x\n",wdp->style);
		     if(wdp->style&STYLE_NOSHADOW)
			printf("               STYLE_NOSHADOW\n");
		     if(wdp->style&STYLE_COMBOEDIT)
			printf("               STYLE_COMBOEDIT\n");
		     if(wdp->style&STYLE_DATETIME)
			printf("               STYLE_DATETIME\n");
		     if(wdp->style&STYLE_CHECKBOX)
			printf("               STYLE_CHECKBOX\n");
		     if(wdp->style&STYLE_NO_PARENT_KEY)
			printf("               STYLE_NO_PARENT_KEY\n");
		     if(wdp->style&STYLE_PUSHB_WIDTH)
			printf("               STYLE_PUSHB_WIDTH\n");
		     if(wdp->style&STYLE_PUSHBUTTON)
			printf("               STYLE_PUSHBUTTON\n");
		     if(wdp->style&STYLE_COMBOLIST)
			printf("               STYLE_COMBOLIST\n");
		     if(wdp->style&STYLE_RADIO)
			printf("               STYLE_RADIO\n");
		     if(wdp->style&STYLE_GRAY)
			printf("               STYLE_GRAY\n");
		     if(wdp->style&STYLE_XYRELATIVE)
			printf("               STYLE_XYRELATIVE\n");
		     if(wdp->style&STYLE_WHCHAR)
			printf("               STYLE_WHCHAR\n");
		     if(wdp->style&STYLE_NOBORDER)
			printf("               STYLE_NOBORDER\n");
		     if(wdp->style&STYLE_LEAF)
			printf("               STYLE_LEAF\n");
		     if(wdp->style&STYLE_NOTIFY)
			printf("               STYLE_NOTIFY\n");
		     if(wdp->style&STYLE_NOFOCUS)
			printf("               STYLE_NOFOCUS\n");
		     if(wdp->style&STYLE_SAVEUNDER)
			printf("               STYLE_SAVEUNDER\n");
		     if(wdp->style&STYLE_INVISIBLE)
			printf("               STYLE_INVISIBLE\n");
		     printf("  Parent:      %d\n",wdp->parent);
		  }
		  wdp++;
	       }
	    }
	    break;
	 case TYPE_CATEGORY:

	    /* TYPE_CATEGORY List of categories for the database. */

	    {
	       struct dbcategory *dbcp;

	       dbcp=rhp->data;
	       if(debug)
		  printf("  Categories: %s\n",dbcp->string);
	    }
	    break;
	 case TYPE_DATA:

	    /* TYPE_DATA Data record */

	    {
	       struct dbdata *dbdp;
	       struct fielddef *fdp;
	       int f,off;
	       void *data;

	       dbdp=rhp->data;
	       if(debug)
		  printf("  Bytes: %d\n",dbdp->bytes);
	       data=dbdp->data;
	       for(f=0;f<fcnt;f++)
	       {
		  fdp=records[fields[f]].data;
		  if((fdp->flags&FIELDDEF_RELATIVE)||(fdp->type==NOTE_FIELD))
		     off=getshort(((char*)data)+fdp->offset);
		  else
		     off=fdp->offset;
		  switch(fdp->type)
		  {
		     case STRING_FIELD:
		     case NUMBER_FIELD:
		     case CATEGORY_FIELD:
			switch(format)
			{
			   case 'c':
			      printf("\"%s\",",((char*)data)+off);
			      break;
			   default:
			      printf("%s: %s\n",fdp->name,((char*)data)+off);
			      break;
			}
			break;
		     case RADIO_FIELD:
			{
			   struct windowdata *wdp;
			   char buf[1],*name;
			   struct fielddef *gfdp;

			   buf[0]='\0';
			   name=buf;
			   if(carddef)
			   {
			      if(f<carddef->count)
			      {
				 wdp=carddef->windows+f;
				 if(wdp->parent<fcnt)
				 {
				    gfdp=(records+(*(fields+wdp->
						     parent)))->data;
				    name=gfdp->name;
				 }
			      }
			   }
			   if(fdp->reserved==(*(((char*)data)+off)))
			   {
			      switch(format)
			      {
				 case 'c':
				    printf("\"%s\",",fdp->name);
				    break;
				 default:
				    printf("%s: %s",name,fdp->name);
				    if(debug)
				       printf(" <radio on>");
				    printf("\n");
				    break;
			      }
			   }
			   else
			   {
			      if(debug)
				 printf("%s <radio off>\n",fdp->name);
			   }
			}
			break;
		     case WORDBOOL_FIELD:
			if(*(((char*)data)+off))
			   printf("[X] ");
			else
			   printf("[ ] ");
			switch(format)
			{
			   case 'c':
			      printf(",");
			      break;
			   default:
			      printf("%s\n",fdp->name);
			      break;
			}
			break;
		     case NOTE_FIELD:
			{
			   struct dbdata *ndbdp;
			   struct rechdr *nrhp;
			   int comp,mcnt,n;

			   switch(format)
			   {
			      case 'c':
				 break;
			      default:
				 printf("%s: ",fdp->name);
				 break;
			   }
			   comp=0;
			   if(off<0)
			   {
			      /* compensating negative note id */

			      off=off+256;
			      comp=1;
			   }
			   if(off>=0)
			   {
			      ndbdp=NULL;
			      for(n=0,mcnt=0;n<ncnt;n++)
			      {
				 nrhp=records+(*(notes+n));
				 if(nrhp->number==off)
				 {
				    ndbdp=nrhp->data;
				    mcnt++;
				 }
			      }
			      if(ndbdp)
			      {
				 switch(format)
				 {
				    case 'c':
				       printf("\"%s\"",ndbdp->data);
				       break;
				    default:
				       printf("%s",ndbdp->data);
				       break;
				 }
			      }
			      else
			      {
				 if(debug)
				    printf("<No note was found>");
			      }
			   }
			   switch(format)
			   {
			      case 'c':
				 printf(",");
				 break;
			      default:
				 printf("\n");
				 break;
			   }
			   if(debug)
			   {
			      printf(" Note id: %d",off);
			      if(comp)
				 printf(" (compensated)");
			      if(mcnt>1)
				 printf(" multiple note id match (%d)",mcnt);
			   }
			}
			break;

			/* skip */

		     case GROUP_FIELD:
		     case LIST_FIELD:
			break;

			/* unkown */

		     case BYTEBOOL_FIELD:
		     case PHONE_FIELD:
		     case CURRENCY_FIELD:
		     case TIME_FIELD:
		     case DATE_FIELD:
		     case STATIC_FIELD:
		     case MULTILINE_FIELD:
		     case COMBO_FIELD:
		     case USER_FIELD:
		     default:
			printf("  Unknown field type %d\n",fdp->type);
			break;
		  }
	       }
	       printf("\n");
	    }
	    break;
	 case TYPE_DBHEADER:

	    /* TYPE_DBHEADER Header for database.  First record in db. */

	    {
	       struct dbheader *dbhp;

	       dbhp=rhp->data;
	       if(debug)
	       {
		  printf("  Release ver:    0x%04x\n",dbhp->release);
		  printf("  Type:           '%c' - ",dbhp->type);
		  switch(dbhp->type)
		  {
		     case '2':
			printf("Appointment book");
			break;
		     case 'D':
			printf("General database and phone book");
			break;
		     case 'N':
			printf("Notetaker");
			break;
		     case 'W':
			printf("World Time");
			break;
		     default:
			printf("Unknown");
			break;
		  }
		  printf("\n");
		  printf("  Status:         %d - ",dbhp->status);
		  switch(dbhp->status)
		  {
		     case 1:
			printf("Database already open");
			break;
		     case 2:
			printf("A record modified since last reconcile");
			break;
		     default:
			printf("Unknown");
			break;
		  }
		  printf("\n");
		  printf("  Viewpoint:      %d\n",dbhp->viewpt);
		  printf("  No of records:  %d\n",dbhp->records);
		  printf("  Lookup offset:  0x%x\n",dbhp->records);
		  printf("  Last reconcile: %d%02d%02d ",dbhp->year+1900,
			 dbhp->month+1,dbhp->date+1);
		  printf("%02d:%02d\n",dbhp->minutes/60,dbhp->minutes%60);
		  printf("  Magic hash:     0x%x\n",dbhp->hash);
	       }
	    }
	    break;
	 case TYPE_FIELDDEF:

	    /* TYPE_FIELDDEF Definition record for each field */

	    {
	       struct fielddef *fdp;

	       fdp=rhp->data;
	       if(debug)
	       {
		  printf("  Field type : %d - ",fdp->type);
		  switch(fdp->type)
		  {
		     case BYTEBOOL_FIELD:
			printf("BYTEBOOL_FIELD\n");
			break;
		     case WORDBOOL_FIELD:
			printf("WORDBOOL_FIELD\n");
			break;
		     case STRING_FIELD:
			printf("STRING_FIELD\n");
			break;
		     case PHONE_FIELD:
			printf("PHONE_FIELD\n");
			break;
		     case NUMBER_FIELD:
			printf("NUMBER_FIELD\n");
			break;
		     case CURRENCY_FIELD:
			printf("CURRENCY_FIELD\n");
			break;
		     case CATEGORY_FIELD:
			printf("CATEGORY_FIELD\n");
			break;
		     case TIME_FIELD:
			printf("TIME_FIELD\n");
			break;
		     case DATE_FIELD:
			printf("DATE_FIELD\n");
			break;
		     case RADIO_FIELD:
			printf("RADIO_FIELD\n");
			break;
		     case NOTE_FIELD:
			printf("NOTE_FIELD\n");
			break;
		     case GROUP_FIELD:
			printf("GROUP_FIELD\n");
			break;
		     case STATIC_FIELD:
			printf("STATIC_FIELD\n");
			break;
		     case MULTILINE_FIELD:
			printf("MULTILINE_FIELD\n");
			break;
		     case LIST_FIELD:
			printf("LIST_FIELD\n");
			break;
		     case COMBO_FIELD:
			printf("COMBO_FIELD\n");
			break;
		     case USER_FIELD:
			printf("USER_FIELD\n");
			break;
		     default:
			printf("unknown\n");
			break;
		  }
		  printf("  Field id:    %d\n",fdp->id);
		  printf("  Data offset: %d\n",fdp->offset);
		  printf("  Flags:       0x%02x\n",fdp->flags);
		  if(fdp->flags&FIELDDEF_NODATA)
		     printf("               FIELDDEF_NODATA\n");
		  if(fdp->flags&FIELDDEF_RESERVED)
		     printf("               FIELDDEF_RESERVED\n");
		  if(fdp->flags&FIELDDEF_RELATIVE)
		     printf("               FIELDDEF_RELATIVE\n");
		  if(fdp->flags&FIELDDEF_NULLTITLE)
		     printf("               FIELDDEF_NULLTITLE\n");
		  switch(fdp->type)
		  {
		     case BYTEBOOL_FIELD:
		     case WORDBOOL_FIELD:
			printf("  Reserved:    0x%04x - check bitmask\n",
			       fdp->reserved);
			break;
		     case RADIO_FIELD:
			printf("  Reserved:    0x%04x - value\n",
			       fdp->reserved);
			break;
		     case CATEGORY_FIELD:
			printf("  Reserved:    %d - category rec no\n",
			       fdp->reserved);
			break;
		     default:
			printf("  Reserved:    %d - not used\n",
			       fdp->reserved);
			break;
		  }
		  printf("  Name:        %s\n",fdp->name);
	       }
	    }
	    break;
	 case TYPE_NOTE:

	    /* TYPE_NOTE Note record */

	    {
	       struct dbdata *dbdp;
	       void *data;

	       dbdp=rhp->data;
	       if(debug)
	       {
		  printf("  Bytes: %d\n",dbdp->bytes);
		  printf("  %s\n",dbdp->data);
	       }
	    }
	    break;
	 case TYPE_LINKDEF:
	 case TYPE_LOOKUPTABLE:
	 case TYPE_VIEWPTDEF:
	 case TYPE_VIEWPTTABLE:
	    if(debug)
	       printf("  Data skipped\n");
	    break;
	 default:
	    if(debug)
	       printf("  Unknown record type\n");
	    break;
      }
      rhp++;
   }
}

int
fgetshort(FILE *fp)
{
   int i;

   i=fgetc(fp);
   i=i+fgetc(fp)*256;
   return i;
}

int
getshort(char *p)
{
   int i;

   i= *(p++);
   i=i+(*p)*256;
   return i;
}

int
fgetint(FILE *fp)
{
   int i;

   i=fgetc(fp);
   i=i+fgetc(fp)*256;
   i=i+fgetc(fp)*256*256;
   i=i+fgetc(fp)*256*256*256;
   return i;
}

void *
myrealloc(void *ptr,int size)
{
   if(!(ptr=realloc(ptr,size)))
   {
      fprintf(stderr,"gdb2txt: Error! Cannot allocate memory\n");
      exit(1);
   }
   return ptr;
}
