/*
 * Copyright (c) 2008 Luigi Rizzo, Riccardo Panicucci, Marta Carbone,
 *	Dipartimento di Ingegneria dell'Informazione, Universita` di Pisa.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: app-wrapper.c 242 2008-05-23 15:07:37Z luigi $
 *
 * This is scriptable plugin for Mozilla.
 * 
 * See myplugin.h for generic information on how to write a plugin.
 *
 * This particular plugin acts as a wrapper to run generic host
 * application within a firefox object.
 */

#define WANT_X11
#include "myplugin.h"

#include <string.h>	/* local stuff */
#include <pthread.h>
#include <fcntl.h>	/* open */
#include <unistd.h>	/* fork() */
#include <signal.h>	/* SIGTERM */
#include <pwd.h>	/* getpwuid */
#include <errno.h>	/* errno */

/* XXX don't trust too much the documentation below, it might be stale */
/*
 * The struct _instance describes the plugin instance. It contains data
 * belonging to a specific instance, e.g. local variables,
 * and possibly a scriptable object associated with the plugin instance.
 * A pointer to this struct is stored in the 'data' field of the NPP struct,
 * which is passed to the plugin when the browsers call the function New().
 */

/*
 * A scriptable object is seen by javascript through an NPObject struct,
 * which is just a pointer to a table (NPClass) of methods supported, and a
 * reference count.
 * We overlay the NPObject with our own struct, where we have additional
 * fields to store the extra information we need for our purposes,
 * e.g. internal variables, pointers to supported functions, and so on.
 * To allocate this extended data structure we must provide the
 * allocate() function in the NPClass API for this object, so the browser
 * will use it, instead of the default, to allocate the NPObject.
 * We can rely on the deallocate() function in the NPClass API if we
 * do not need special operation on destroy.
 *
 * So the linking is the following:
 *
 *   NPObject (overlaid by a 'struct _my_obj'):
 *	extended with a pointer to our struct _instance.
 *	extended with private data for the scriptable object.
 *
 *   NPP:
 *	the 'pdata' field points to our struct _instance.
 *
 *   struct _instance:
 *	contains a pointer to the struct _my_obj
 *	contains plugin instance-specific variables.
 */

struct _my_obj {
	NPObject me;	/* the actual object. */

	/*
	 * A pointer to the plugin instance, so we can access
	 * data belonging to the plugin instance
	 */
	struct _instance * instance;

	/*
	 * Method names are strings, but internally they are
	 * represented using unique identifiers
	 * (I guess this way we can use direct comparisons
	 * instead of strcmp).
	 * f1 in this example is the identifier for the string
	 * used for our method name.
	 */
	NPIdentifier f1;
};

struct _instance {
	struct _my_obj *objS; /* a pointer to scriptable object*/
	
	/* Here we store local variables */
	int local_href;	/* set if href is file:// ... */
	int stream_on_stdin;	/* can handle stream on stdin */
	int pipe_fd[2];	/* in case we open a pipe on a stream... */
	pid_t child_pid;
	long window;	/* the window used by setwindow */
	char cmdbuf[256]; /* a buffer to copy the external command */
	int width, height; /* width and height of plugin window */
};

static char * check_config(char *cmd,
	struct _instance *instance /* NULL if get_mime */,
	char *buf, int bufsize);

/*
 * A global variable. This is shared between all plugin instance.
 */
static int gInt=20;

static NPError my_Initialize(NPNetscapeFuncs* npn, NPPluginFuncs* npp)
{
	// printf("this is my initialize function\n");
	return NPERR_NO_ERROR;
}

/*
 * Called a starting point by the browser when loads each library.
 *
 * Should return a string containing the type,
 * extension list, and type description.
 * Separate multiple entries with a ';'
 */
char *NP_GetMIMEDescription(void)
{
	static char mime_buf[2048];
	char *s;

	bzero(mime_buf, sizeof(mime_buf));
	s = check_config(NULL, NULL, mime_buf, sizeof(mime_buf));
	if (s == NULL)
		s = "";

	my_NP_Initialize = my_Initialize;
	printf("calling %s returns %s\n", __FUNCTION__, s);
	return s;
}


/*
 * Below we implement the methods of the NPClass.
 */

/*
 * hasMethod returns whether the obj supports a given method.
 */
static bool _hasMethod(NPObject *o, NPIdentifier name)
{
	struct _my_obj *x = (struct _my_obj *)o;
	printf("calling %s [%s] %s\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name),
		name == x->f1 ? "found" : "not found");
	return (name == x->f1);
}

/*
 * _invoke does the actual invocation of the method, with arguments.
 * The result must be returned in *result.
 * Also return true in case of success, false in case of error.
 * It is expected that we can never call a non-existing method,
 * but we better check anyways...
 */
static bool _invoke (NPObject *o, NPIdentifier name,
	const NPVariant *argv, uint32_t argc, NPVariant *result)
{
	struct _my_obj *x = (struct _my_obj *)o;
	printf("calling %s [%s]\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name));
	/* find out which one of the supported methods we are calling */
	if (name == x->f1) {
		gInt++;
		//printf(" argc %d\n", argc); 
		/* XXX do we need to call strdup() ? who frees it ? */
		STRINGZ_TO_NPVARIANT(strdup("A return value"), *result);
		return TRUE;
	}
	printf ("Error calling _invoke\n");
	return FALSE;
}

/*
 * _invalidate() is called on live object thet belongs to a plugin
 * instance that is being destroyed. This function is always followed
 * by a call to the deallocate() function (or free()). XXX
 */
static void _invalidate (NPObject *npobj)
{
	// printf("calling %s\n", __FUNCTION__);
}

/*
 * _invokeDefault() calls the default method on the object, if any.
 */
static bool _invokeDefault(NPObject *o, const NPVariant *argv, uint32_t argc,
	NPVariant *result)
{
	printf("calling %s\n", __FUNCTION__);
	return false;
}

/*
 * These are the equivalent of hasMethod() for fields of the object.
 * verify if a property exists, get the value, set the value,
 * remove the property altogether.
 * XXX are they all needed ?
 */
static bool _hasProperty (NPObject *o, NPIdentifier name)
{
	printf("calling %s [%s]\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name));
	return false;
}

static bool _getProperty (NPObject *o, NPIdentifier name, NPVariant *result)
{
	printf("calling %s [%s]\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name));
	return false;
}

static bool _setProperty (NPObject *o, NPIdentifier name, const NPVariant *value)
{
	printf("calling %s [%s]\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name));
	return false;
}

static bool _removeProperty (NPObject *o, NPIdentifier name)
{
	printf("calling %s [%s]\n",
		__FUNCTION__, NPN_UTF8FromIdentifier(name));
	return false;
}

static NPObject *_allocate(NPP instance, NPClass *cl)
{
	struct _my_obj * o = malloc(sizeof(struct _my_obj));
	//printf("calling %s\n", __FUNCTION__);
	o->instance = (struct _instance *) instance->pdata;
	return (NPObject *)o;
}

static void _deallocate(NPObject *o)
{
	struct _my_obj * obj = (struct _my_obj *) o;
	//printf("calling %s, counter is %d\n", __FUNCTION__, o->referenceCount);
	obj->instance->objS = NULL;
	/* do local freeing ? */
	free(o);
}

/*
 * The descriptor for the API of the object, which is passed
 * as an argument to the .createobject() call.
 * XXX find out which methods are required.
 */
static NPClass xyz_scriptable =
{
	.structVersion = NP_CLASS_STRUCT_VERSION,
	.allocate = _allocate, /*allocate*/
	.deallocate = _deallocate, /*deallocate*/
	.invalidate = _invalidate, /*invalidate*/
	.hasMethod = _hasMethod,
	.invoke = _invoke,
	.hasProperty = _hasProperty, /*hasProperty*/
	.getProperty = _getProperty, /*getProperty*/
	.setProperty = _setProperty, /*setProperty*/
	.removeProperty = _removeProperty, /*removeProperty*/
	.invokeDefault = _invokeDefault, /*invokeDefault*/
};

/*
 * Create the scriptable object if it isn't already created.
 * Return a pointer to this object.
 * This is not part of the api - we can implement it the way we want.
 */
NPObject *my_GetScriptableObject (NPP instance)
{
	struct _instance * my_instance = (struct _instance *) instance->pdata;
	
	if (!my_instance->objS)
	{
		//printf("New scriptableObject\n");
		my_instance->objS = (struct _my_obj*)
				NPN_CreateObject(instance, &xyz_scriptable);
		if (my_instance->objS) { /* do local initialization */
			
			/* set the pointer to the function we implement */
			my_instance->objS->f1 =
				NPN_GetStringIdentifier("my_fn");
		}
	} else {
		//printf("Incrementing scriptableObject counter\n");
		/* Incrementing reference count  */
		NPN_RetainObject ((NPObject*)my_instance->objS);
	}
	return (NPObject*)(my_instance->objS);
}

/*
 * Resize the window plugin. We modify the value of "width" and "height"
 * attributed passed in the html tag object (or embed)
 */
bool doResize(NPP instance, int w, int h)
{
	NPError err;
	bool retval = FALSE;
	NPObject *idObj = NULL; /* the id object */
	NPVariant width, height; /* New width and height to pass to browser */

	printf("calling %s\n", __FUNCTION__);

	/*
	 * Retrieve the NPObject corresponding to this plugin.
	 * Instead of using GetElementById("foo") or window.document.foo
	 * where 'foo' is the id associated to this object, we just use
	 * NPN_GetValue on NPNVPluginElementNPObject
	 */
	err = NPN_GetValue(instance, NPNVPluginElementNPObject, &idObj);
	if (err != NPERR_NO_ERROR) {
		fprintf(stderr, "%s GetValue NPNVPluginElementNPObject fails\n",
			__FUNCTION__);
		goto done;
	}
	/*
	 * Now that idObj references the object, we can set width and height
	 * by creating two NPVariant structures and calling NPN_SetProperty
	 * XXX see if we can replace the macros with functions.
	 */
	INT32_TO_NPVARIANT(w, width);
	INT32_TO_NPVARIANT(h, height);

	/* Set new dimensions */
	retval = NPN_SetProperty(instance, idObj,
			NPN_GetStringIdentifier("width"), &width);
	if (!retval) {
		fprintf(stderr, "%s SetProperty width fails, %d\n",
			__FUNCTION__, w);
		goto done;
	}
	retval = NPN_SetProperty(instance, idObj,
			NPN_GetStringIdentifier("height"), &height);
	if (!retval) {
		fprintf(stderr, "%s SetProperty height fails, %d\n",
			__FUNCTION__, h);
		goto done;
	}

done:	/* error or exit */
	if (idObj)
		NPN_ReleaseObject(idObj);
	return retval;
}

/*
 * The handler for the X11 events we care about.
 * We manage only the event ConfigureNotify, which is generally
 * related to window resize and move.
 *
 * XXX Note that by default, the SDL library does not allow window
 * resize when running in an external window.
 * To allow resize we either need to modify the SDL library to allow resizes,
 * or as a workaround we can modify the application to generate
 * a ConfigureNotify event (with XSendEvent()) so the plugin will catch it
 * and run this handler.
 */
void xt_event_handler(Widget xtwidget, void *instanc, XEvent *event, Boolean *b)
{
	struct _instance *me;
	XConfigureRequestEvent *e;
	NPP instance = (NPP) instanc;
	me = instance->pdata;
	switch (event->type) {
	default:
		printf ("%s Unknown event type %d\n", __FUNCTION__, event->type);
		break;
	case ConfigureNotify:
		e = (XConfigureRequestEvent*) event;
		if (me->width != e->width || me->height != e->height) {
			fprintf (stderr, "%s ConfigureNotify Old: %dx%d, New: %dx%d\n",
				__FUNCTION__,
				me->width, me->height, e->width, e->height);
			doResize(instance, e->width, e->height);
		}
	}
}

static char *skip_blanks(char *d)
{
	if (d) {
		while (*d && index(" \t\r\n", *d))
			d++;
	}
	return d;
}

/*
 * Read the configuration file and extract information.
 * 'mode' determines the actual behaviour:
 *  - GET_MIME returns the mime-types line(s);
 *  - CHECK_CMD	checks if cmd is allowed,
 *	only returns success on 'safe' entries
 *  - CHECK_CMD_ALL	checks if cmd is allowed,
 *	returns success on all entries.
 *
 *
 * compare the program name with one of the allowed ones,
 * and return a pointer to the command.
 * The ~/.mozilla/plugins/app-wrapper.conf contains one mapping
 * per line, with the usual # delimiting comments.
 * The left side (case-insensitive) is the command name,
 * the right side is the real command that we run.
 * If there is no right side we use the original command name.
 * There is no default matching.
 *
 * commands are only allowed for local (file:// ) urls unless
 * they are marked as safe: in the config file.
 */
static char * check_config(char *mime, struct _instance *me,
		char *buf, int bufsize)
{
	FILE *f = NULL;
	struct passwd *pwe = NULL, pwe_buf;
	char *fname = NULL;
	char *ret = NULL;
	int i, l;
	uid_t my_uid;
	char *bufp;

	if (mime == NULL)
		mime = "GET_MIME";

	/* retrieve uid, home directory, and try to open the config file. */
	my_uid = getuid();
	bzero(&pwe_buf, sizeof(pwe_buf));
	if (my_uid <= 0)
		goto done;
	i = getpwuid_r(my_uid, &pwe_buf, buf, bufsize, &pwe);
	endpwent();
	if (i != 0 || pwe == NULL || pwe->pw_dir == NULL)
		goto done;
	asprintf(&fname, "%s/.mozilla/plugins/app-wrapper.conf", pwe->pw_dir);
	if (fname == NULL)
		goto done;
	f = fopen(fname, "r");
	if (f == NULL)
		goto done;
	/* read from the file and look for a match in the mime type. */
	l = strlen(mime);
	buf[0] = '\0';	/* terminate the response */
	bufp = buf;	/* append the result here */
	for (;;) {
		char *d, *cmd, *end = NULL, line[1024];
		int is_safe = 0;
		d = fgets(line, sizeof(line), f);
		if (d == NULL)
			break;
		/* skip leading whitespace */
		d = skip_blanks(d);
		if (index("#\r\n", *d))
			continue;	/* empty line */
		/* The third ':' is the end of the mime type. */
		cmd = index(d, ':');
		if (cmd)
			cmd = index(cmd + 1, ':');
		if (cmd)
			cmd = index(cmd + 1, ':');
		if (cmd) {
			for (end = cmd; *end && !index("\r\n#", *end); end++)
				;
			*end = '\0';
		}
		if (me == NULL) { /* GetMIMETypes */
			/* copy the mime line in the buffer,
			 * replace ':' with ',' after cmd
			 */
			for (; cmd && *cmd; cmd++) {
				if (*cmd == ':')
					*cmd = ',';
			}
			if (bufsize < 4) {
				fprintf(stderr, "no space, skip [%s]\n", d);
				continue;
			}
			l = strlen(d);
			if (l > bufsize - 2)
				l = bufsize - 2;
			strncat(bufp, d, l);
			bufp[l++] = ';';
			bufp[l] = '\0';
			bufp += l;
			bufsize -= l;
			continue;
		}
		// fprintf(stderr, "now line is [%s]\n", d);
		/* regular line, match mime and then parse options */
		if (strncasecmp(d, mime, l) || d[l] != ':')
			continue; /* mime type mismatch */
		if (!cmd)
			continue;
		cmd = skip_blanks(cmd+1);	/* point to the command */
		for (;;) {
			if (!strncasecmp(cmd, "safe:", 5)) {
				is_safe = 1;
				cmd = skip_blanks(cmd + 5);
			} else if (!strncasecmp(cmd, "stream:", 7)) {
				fprintf(stderr, "--- can do streaming\n");
				me->stream_on_stdin = 1;
				cmd = skip_blanks(cmd + 7);
			} else {
				break;
			}
		}
		/* skip empty lines */
		if (*cmd == '\0' || index("#\r\n", *cmd)) {
			ret = NULL;
		} else if (!is_safe && !me->local_href) {
			fprintf(stderr, "unsafe command %s\n", cmd);
			ret = NULL;
		} else {
			strncpy(buf, cmd, bufsize - 1);
			buf[bufsize-1] = '\0';
			ret = buf;
		}
		break;
	}
done:
	if (me == NULL && buf[0] != '\0')
		ret = buf;
	if (ret == NULL)
		fprintf(stderr, "Error checking config for %s uid %d\n",
			mime, my_uid);
	else
		fprintf(stderr, "[%s] maps to [%s]\n", mime, ret);
	if (f)
		fclose(f);
	if (fname)
		free(fname);
	return ret;
}

/* possibly run a command if instructed to do so
 */
static int run_cmd(struct _instance *me)
{
	/*
	 * Fork and run the command in the background.
	 */
	int argc, fd;
	char *argv[64];
	char *p, *src, *dst, cmdbuf[1024], *lim;

	if (!me->cmdbuf[0])
		return 0;
	sprintf(cmdbuf,"%ld", me->window);
	setenv("SDL_WINDOWID", cmdbuf, 1);

	bzero(cmdbuf, sizeof(cmdbuf));
	/*
	 * Replace keywords with specific parameters.
	 */
	src = me->cmdbuf;
	dst = cmdbuf;
	lim = cmdbuf + sizeof(cmdbuf) - 1;
	while (*src && dst < lim) {
		char c;
		int n = 0;
		c = *dst++ = *src++;
		if (c != '%')
			continue;
		dst--;	/* eat back the '%' */
		switch (*src++) {
		default:	/* ignore non-recognised chars */
			src--;
			continue;
		case 'w':
			/* print and advance by the number of chars printed */
			n = snprintf(dst, lim - dst, "%ld", me->window);
			break;
		}
		if (n > lim - dst)
			n = lim - dst;
		if (n > 0)
			dst += n;
	}

	fprintf(stderr, "-- object would like to run [%s]\n", cmdbuf);
	/* now split cmdbuf. p points to the rest of the string */
	argc = 0;
	p = cmdbuf;
	while (p && argc < sizeof(argv)/sizeof(argv[0]) - 1) {
		/* skip leading whitespace */
		while (*p && index(" \t\r\n", *p))
			p++;
		argv[argc] = strsep(&p, " \t\r\n");
		if (argv[argc] == NULL)
			break;
		if (argv[argc][0] == '&')
			continue;
		argc++;
	}
	argv[argc] = NULL;
	if (argc == 0)	/* nothing to do */
		return 0;
	if (me->stream_on_stdin) {
		int flag;
		fprintf(stderr, "/* open a pipe to pass stream data to the child */\n");
		if (pipe(me->pipe_fd)) {
			/* error opening the pipe, reset */
			fprintf(stderr, "-- cannot open pipe, give up on the stream\n");
			me->stream_on_stdin = 0;
		}
		flag = fcntl(me->pipe_fd[1], F_GETFL);
		fcntl(me->pipe_fd[1], F_SETFL, flag | O_NONBLOCK);
	}
	me->child_pid = fork();
	if (me->child_pid != 0)	{
		/* parent, close one end of the pipe and be done */
		if (me->stream_on_stdin)
			close(me->pipe_fd[0]);
		me->pipe_fd[0] = -1;
		return me->child_pid;
	}
	fd = open("/dev/null", O_RDWR);
	if (me->stream_on_stdin) {
		close(me->pipe_fd[1]);
		dup2(me->pipe_fd[0], 0);
	} else {
		/* map stdin to /dev/null */
		if (fd >= 0) {
			fprintf(stderr, "input from /dev/null\n");
			dup2(fd, 0);
		}
	}
	if (fd >= 0) {
		/* XXX should close all descriptors, but at least
		 * map stdout and stderr to /dev/null
		 */
		dup2(fd, 1);
		dup2(fd, 2);
	}
	/* use execvp so use PATH to find the location of programs */
	execvp(argv[0], argv);
	exit(0);	/* child */
}

/*
 * NPP_SetWindow is called by the browser on expose and other events
 * affecting the window.
 * The first time, we deselect the ButtonPressMask on the window
 * (XXX and perhaps later also resize, though we can get them through
 * NPP_SetWindow).
 * This lets the client that we run in the window to handle those events
 * by itself.
 *
 * HANDLING OF RESIZE AND MOVE:
 * When a window is resized by the application, we
 * we receive an event in the 'StructureNotify' set,
 * which we direct to xt_event_handler() and in turn goes to * doResize().
 * The latter calls the DOM routines in the browser to change
 * the width and height of the window where the application runs.
 * 
 */
NPError NPP_SetWindow(NPP instance, NPWindow* window)
{
	struct _instance *me;
	NPSetWindowCallbackStruct *ws_info;
	XWindowAttributes attr;

	//printf("calling %s\n", __FUNCTION__);
	if (window == NULL || instance == NULL) {
		printf("ERROR: null window or instance\n");
		return FALSE;
	}
	me = instance->pdata;
	ws_info = window->ws_info;

	fprintf(stderr,"%s: Window size received w=%d, h=%d; me:w=%d, h=%d\n", 
		__FUNCTION__, window->width, window->height,
		me->width, me->height);
	
    if (me->window == 0) {
	/* First time in the function, store all parameters and
	 * deselect the ButtonPressMask on the parent for this window
	 * so the application itself can handle buttons and other
	 * one-recipient events (XXX this probably includes also
	 * ResizeRedirectMask and SubstructureRedirectMask).
	 * Leave ResizeRedirectMask to the parent (i.e. myself)
	 * or the window is not resized properly by the browser.
	 */
	long private_events = ButtonPressMask |
		SubstructureRedirectMask ;
	me->window = (long)window->window;
	bzero(&attr, sizeof(attr));
        XGetWindowAttributes(ws_info->display, me->window, &attr);

	if (attr.your_event_mask & private_events) {
		// fprintf(stderr, "deselecting ev 0x%lx on the browser\n", private_events);
		attr.your_event_mask &= ~private_events;
		XSelectInput(ws_info->display, me->window,
			attr.your_event_mask);
        }

	/*
	 * Create event handlers for all X events we (as a plugin) might
	 * be interested in.
	 * Here we are interested in the ConfigureNotify event,
	 * which is sent when a window has some noticeable event e.g.
	 * a resize or a move.
	 * FIXME: can we avoid creation of a Xt widget?
	 */
	Widget xtwidget = XtWindowToWidget(ws_info->display, me->window);
	if (xtwidget) {
		XtAddEventHandler(xtwidget, StructureNotifyMask, False,
			(XtEventHandler)xt_event_handler, instance);
	}
	run_cmd(me);
    }

    /* This is the standard path on all executions. */

    // printf("WINDOWID=%ld\n", me->window);
	
    if (me->width != window->width || me->height != window->height) {
	    /*
	     * NPP_SetWindow send us new window dimensions.
	     * Here we should manage these difference
	     */
	    fprintf(stderr, "%s, Received new size: %dx%d\n", 
		   __FUNCTION__, window->width, window->height);
	    me->width = window->width;
	    me->height = window->height;

    }
    return NPERR_NO_ERROR;
}

/*
 * This function allow the browser to query the plugin for information.
 * In this example, we tell the browser that our plugin is scriptable,
 * and we create the scriptable object if it isn't already created.
 */
NPError NPP_GetValue(NPP instance, NPPVariable variable, void *value)
{
	//printf("calling %s with variable %d\n", __FUNCTION__, variable);
	if (instance == NULL) {
		printf("in %s instance is null\n", __FUNCTION__);
		return NPERR_INVALID_INSTANCE_ERROR;
	}
	switch (variable) {
	default:
		printf ("%s on unknown variable %d\n", __FUNCTION__, variable);
		break;

	case NPPVpluginNameString:
		*((char **)value) = "app-wrapper v.0.1 - generic application wrapper";
		break;

	case NPPVpluginDescriptionString:
		*((char **)value) = "app-wrapper v.0.1 $Id: app-wrapper.c 242 2008-05-23 15:07:37Z luigi $";
		break;
		
	case NPPVpluginScriptableNPObject:
		/*
		 * The browser wants to know if we have a scriptable object.
		 * We create or return the scriptable object.
		 */
		//printf ("Requesting scriptability\n");
		*(NPObject **)value = my_GetScriptableObject(instance);
		break;
	}
	return NPERR_NO_ERROR;
}

/*
 * As an example of the interaction, on a new we let the
 * plugin call into javascript and into dom.
 */

static bool plugin_calls_js(NPP instance, const char *js_code, NPVariant *ret);
static bool plugin_calls_dom(NPP instance);
/*
 * Called when a new istance is created.
 * In the struct NPP (in the field pdata) we can store local variable
 * belonging to a specific plugin instance. These data are not shared
 * with other instance.
 */
NPError NPP_New(NPMIMEType pluginType, NPP instance,
		uint16 mode, int16 argc, char *argn[],
                char *argv[], NPSavedData *saved)
{
	NPVariant res;
	struct _instance * my_instance;
	int i;

	//printf("calling %s\n", __FUNCTION__);

	/* Allocating space to store local data */
	my_instance = calloc (1, sizeof(struct _instance));
	if (my_instance == NULL)
		return NPERR_OUT_OF_MEMORY_ERROR;
	
	/* Initializing local plugin instance data */
	my_instance->objS = NULL;
	my_instance->width = 0;
	my_instance->height = 0;

	instance->pdata = (void*) my_instance;

	/*
	 * Check the URL we are called from, to see if we are in
	 * 'safe' mode or not.
	 */
	bzero(&res, sizeof(res));
	if (!plugin_calls_js(instance, "document.location.href", &res))
		printf("Error on javascript call\n");
	else {
		fprintf(stderr, "f returns [%s]\n",
			NPVARIANT_TO_STRING(res).utf8characters);
		if (NPVARIANT_IS_STRING(res)) {
			const char *x = NPVARIANT_TO_STRING(res).utf8characters;
			printf ("href is: %s\n", x);
			if (!strncasecmp(x, "file://", 7))
				my_instance->local_href = 1;
		}
		NPN_ReleaseVariantValue(&res);
	}
	for (i = 0; i < argc; i++) {
		int l = sizeof(my_instance->cmdbuf) - 1;
		char *s;
		fprintf(stderr, "Argument %d %s = '%s'\n",
			i, argn[i], argv[i]);
		/* store the translated type in a local string */
		if (!strcmp(argn[i], "type") &&
			(s = check_config(argv[i], my_instance,
				my_instance->cmdbuf, l)) ) {
			strncpy(my_instance->cmdbuf, s, l);
			my_instance->cmdbuf[l] = '\0';
		}
	}
    if (0) {

	if (!plugin_calls_dom(instance))
		printf("Error on dom call\n");
    }
	return NPERR_NO_ERROR;
}

/* 
 * Called when the plugin instance is destroyed.
 * We must free resources allocated by this instance,
 * and also signal termination to the process we create.
 */
NPError NPP_Destroy(NPP instance, NPSavedData **save)
{
	struct _instance * my_instance = (struct _instance *) instance->pdata;
	//fprintf(stderr, "-- %s called\n", __FUNCTION__);
	if (my_instance->child_pid)
		kill(my_instance->child_pid, SIGTERM);
	/* close the pipe if we opened one */
	if (my_instance->pipe_fd[1] > 0)
		close(my_instance->pipe_fd[1]);
	free(instance->pdata);
	//fprintf(stderr, "-- %s done\n", __FUNCTION__);
	return NPERR_NO_ERROR;
}

NPError NPP_NewStream(NPP instance, NPMIMEType type, NPStream* stream,
		NPBool seekable, uint16* stype)
{
	struct _instance *me = instance->pdata;

        printf("calling %s url %s seek %d type 0x%x stream %d\n",
                __FUNCTION__, stream->url, seekable, *type, me->stream_on_stdin);
        return NPERR_NO_ERROR;
}

/*
 * tell how much can we read, or -1 if not supported
 */
int32 NPP_WriteReady (NPP instance, NPStream *stream)
{
	struct _instance *me = instance->pdata;

	if (0) printf("calling %s url %s headers %s stream %d\n",
                __FUNCTION__, stream->url,
                stream->headers ? stream->headers : "NO_HDRS",
		me->stream_on_stdin);
	if (!me->stream_on_stdin) {
		NPN_DestroyStream(instance, stream, 0 /* reason */);
		return -1;      /* no data... */
	}
	return 8192;	/* a reasonable block size */
}

/*
 * Write the incoming stream to stdin. Return 0 if the pipe is full
 * (we have set it to non-blocking)
 */
int32 NPP_Write (NPP instance, NPStream *stream, int32 offset, int32 len, void *
buffer)
{
	struct _instance *me = instance->pdata;
	int ret;

	if (!me->stream_on_stdin || me->pipe_fd[1] <= 0)
		return -1; /* error */
	ret = write(me->pipe_fd[1], buffer, len);
	if (ret == -1 && errno == EAGAIN)	// wait for pipe to drain
		ret = 0;
	// printf("calling %s %d returns %d\n", __FUNCTION__, len, ret);
	// perror("error is");
	return ret;
}

NPError NPP_DestroyStream (NPP instance, NPStream *stream, NPError reason)
{
        printf("calling %s\n", __FUNCTION__);
        return NPERR_NO_ERROR;
}


/*
 * This is an example of how to run a piece of javascript code from the plugin.
 */
static bool plugin_calls_js(NPP instance, const char *js_code, NPVariant *res)
{
	NPError err;
	bool retval = FALSE;
	NPObject *winObj = NULL; /* the browser window */
	NPString str;	/* NS representation of the javascript code */

	//printf("calling %s\n", __FUNCTION__);

	err = NPN_GetValue(instance, NPNVWindowNPObject, &winObj);
	if (err != NPERR_NO_ERROR) {
		fprintf(stderr, "%s GetValue NPNVWindowNPObject fails\n",
			__FUNCTION__);
		goto done;
	}

	str.utf8characters = js_code;
	str.utf8length = strlen(str.utf8characters);

	bzero(res, sizeof(*res));
	/* Run the script in the browser window */
	retval = NPN_Evaluate(instance, winObj, &str, res);
done:
	if (winObj)
		NPN_ReleaseObject(winObj);
	// releasevariant res ?
	return retval;
}

/* This is an example of calling DOM from the plugin.
 * In this case, read the document location.
 */
static bool plugin_calls_dom(NPP instance)
{
	NPError err;
	bool retval = FALSE;
	NPObject *winObj = NULL; /* the browser window */
	NPObject *docObj = NULL; /* the document object */
	NPVariant ret; /* temporary return value */

	printf("calling %s\n", __FUNCTION__);

	bzero(&ret, sizeof(ret));	/* set a default */
	/* below is the equivalent of ret = window.document*/
	err = NPN_GetValue(instance, NPNVWindowNPObject, &winObj);
	if (err != NPERR_NO_ERROR) {
		fprintf(stderr, "%s GetValue NPNVWindowNPObject fails\n",
			__FUNCTION__);
		goto done;
	}
	retval = NPN_GetProperty(instance, winObj,
		NPN_GetStringIdentifier("document"), &ret);
	if (!retval) {
		fprintf(stderr, "%s GetProperty document fails\n",
			__FUNCTION__);
		goto done;
	}
	
	/* Make sure this is an object */
	if (NPVARIANT_IS_OBJECT(ret)) {
		docObj = NPVARIANT_TO_OBJECT(ret);
	} else {
		printf("Error in work2(): docVar is not a object\n");
		goto done;
	}

	bzero(&ret, sizeof(ret));	/* set a default */
	/* ret = window.document.title */
	retval = NPN_GetProperty(instance, docObj,
		NPN_GetStringIdentifier(""), &ret);
	if (!retval) {
		fprintf(stderr, "%s GetProperty location fails\n",
			__FUNCTION__);
		goto done;
	}
	if (NPVARIANT_IS_STRING(ret)) {
		printf ("The html title is: %s\n",
			NPVARIANT_TO_STRING(ret).utf8characters);
		retval = TRUE;
	} else {
		printf ("Error getting title in %s\n", __FUNCTION__);
		retval = FALSE;
	}

done:	/* error or exit */
	NPN_ReleaseVariantValue(&ret);
	if (docObj)
		NPN_ReleaseObject(docObj);
	if (winObj)
		NPN_ReleaseObject(winObj);
	return retval;
}
