/* sslsock.c
** Routines to use ssl sockets
** Originally based on a simple version by Al Globus <globus@nas.nasa.gov>.
** Debugged and prettified by Jef Poskanzer <jef@mail.acme.com>.  Also includes
** ifdefs to handle https via OpenSSL.
*/

#include "myhdr.h"

/* Some code to open a tcp or ssl socket */
#ifdef USE_SSL
#if defined(USE_XYSSL)	/* stubs for non-ssl */
#include "xyssl/net.h"
#include "xyssl/ssl.h"
#include "xyssl/havege.h"
typedef ssl_context SSL_CTX;
typedef ssl_session SSL;
#else	/* regular openssl */
#include <openssl/ssl.h>
#endif

#else	/* stubs for non-ssl */
typedef struct _a {}  SSL_CTX;
typedef struct _b {}  SSL;
#endif

/* a socket descriptor with room for ssl state */
struct my_sock {
	int sockfd;
	SSL_CTX *ssl_ctx;
	SSL *ssl;
};

void safe_close(struct my_sock *fd)
{
#ifdef USE_SSL
    if ( fd->ssl ) {
#ifdef USE_XYSSL
	ssl_close_notify( fd->ssl_ctx );
	close(fd->sockfd);
	ssl_free( fd->ssl_ctx );
#else
	SSL_free( fd->ssl );
	SSL_CTX_free( fd->ssl_ctx );
#endif
    }
#endif
    close( fd->sockfd );
    bzero(fd, sizeof(*fd));
    free(fd);
}

int safe_write(struct my_sock *fd, const char *buf, int bytes)
{
    int i;
    while (bytes > 0) {
#ifdef USE_SSL
	if (fd->ssl) {
#ifdef USE_XYSSL
	    while( ( i = ssl_write( fd->ssl_ctx, (char *)buf, bytes ) ) <= 0 ) {
		if ( i != XYSSL_ERR_NET_TRY_AGAIN )
		    break;
	    }
#else
	    i = SSL_write( fd->ssl, buf, bytes);
#endif
	}
	else
#endif
	    i = write( fd->sockfd, buf, bytes );
	if (i <= 0)
	    return -1;
	buf += i;
	bytes -= i;
    }
    return 0;
}

int safe_read(struct my_sock *fd, char *buf, int len)
{
#ifdef USE_SSL
    if ( fd->ssl ) {
#ifdef USE_XYSSL
	int i;
	do {
	    i = ssl_read( fd->ssl_ctx, buf, len );
	    if (i == XYSSL_ERR_NET_TRY_AGAIN )
		continue;
        } while (0);
	return i > 0 ? i : 0 ;
#else
	return SSL_read( fd->ssl, buf, len);
#endif
    } else
#endif
	return read( fd->sockfd, buf, len);
}

#if defined(AF_INET6) && defined(IN6_IS_ADDR_V4MAPPED)
#define USE_IPV6
#endif

struct my_sock *
open_client_socket( char* hostname, unsigned short port, int do_ssl )
{
#ifdef USE_IPV6
    struct addrinfo hints;
    char portstr[10];
    int gaierr;
    struct addrinfo* ai;
    struct addrinfo* ai2;
    struct addrinfo* aiv4;
    struct addrinfo* aiv6;
    struct sockaddr_in6 sa;
#else /* USE_IPV6 */
    struct hostent *he;
    struct sockaddr_in sa;
#endif /* USE_IPV6 */
    int sa_len, sock_family, sock_type, sock_protocol;

    struct my_sock *fd = calloc(1, sizeof(*fd));
    if (fd == NULL)
	return NULL;
    memset( (void*) &sa, 0, sizeof(sa) );

    //fprintf(stderr, "host %s port %d %s\n", hostname, port, do_ssl ? "SSL":"");
#ifdef USE_IPV6

    memset( &hints, 0, sizeof(hints) );
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    snprintf( portstr, sizeof(portstr), "%d", (int) port );
    if ( (gaierr = getaddrinfo( hostname, portstr, &hints, &ai )) != 0 ) {
	fprintf(
	    stderr, "getaddrinfo %s - %s\n", hostname,
	    gai_strerror( gaierr ) );
	exit( 1 );
    }

    /* Find the first IPv4 and IPv6 entries. */
    aiv4 = NULL;
    aiv6 = NULL;
    for ( ai2 = ai; ai2 != NULL; ai2 = ai2->ai_next ) {
	if ( ai2->ai_family == AF_INET && !aiv4 )
		aiv4 = ai2;
	else if ( ai2->ai_family == AF_INET6 && !aiv6 )
	    aiv6 = ai2;
    }

    /* If there's an IPv4 address, use that, otherwise try IPv6. */
    if ( aiv4 != NULL ) {
	if ( sizeof(sa) < aiv4->ai_addrlen ) {
	    fprintf(
		stderr, "%s - sockaddr too small (%lu < %lu)\n",
		hostname, (unsigned long) sizeof(sa),
		(unsigned long) aiv4->ai_addrlen );
	    exit( 1 );
	}
	sock_family = aiv4->ai_family;
	sock_type = aiv4->ai_socktype;
	sock_protocol = aiv4->ai_protocol;
	sa_len = aiv4->ai_addrlen;
	memmove( &sa, aiv4->ai_addr, sa_len );
    } else if ( aiv6 != NULL) {
	if ( sizeof(sa) < aiv6->ai_addrlen ) {
	    fprintf(
		stderr, "%s - sockaddr too small (%lu < %lu)\n",
		hostname, (unsigned long) sizeof(sa),
		(unsigned long) aiv6->ai_addrlen );
	    exit( 1 );
	}
	sock_family = aiv6->ai_family;
	sock_type = aiv6->ai_socktype;
	sock_protocol = aiv6->ai_protocol;
	sa_len = aiv6->ai_addrlen;
	memmove( &sa, aiv6->ai_addr, sa_len );
    } else {
	fprintf( stderr, "no valid address found for host %s\n", hostname );
	exit( 1 );
    }
    freeaddrinfo( ai );

#else /* USE_IPV6 */

    he = gethostbyname( hostname );
    if ( he == NULL ) {
	fprintf( stderr, "%s: unknown host - %s\n", argv0, hostname );
	exit( 1 );
    }
    sock_family = sa.sin_family = he->h_addrtype;
    sock_type = SOCK_STREAM;
    sock_protocol = 0;
    sa_len = sizeof(sa);
    memmove( &sa.sin_addr, he->h_addr, he->h_length );
    sa.sin_port = htons( port );

#endif /* USE_IPV6 */

    fd->sockfd = socket( sock_family, sock_type, sock_protocol );
    if ( fd->sockfd < 0 )
	perror( "socket" );

    if ( connect( fd->sockfd, (struct sockaddr*) &sa, sa_len ) < 0 )
	perror( "connect" );

#ifdef USE_SSL
    if ( do_ssl ) {
#ifdef USE_XYSSL
	int ret;
	havege_state *hs = calloc(1, sizeof(*hs));
	fd->ssl = calloc(1, sizeof (*fd->ssl));
	fd->ssl_ctx = calloc(1, sizeof (*fd->ssl_ctx));
	if (!hs || !fd->ssl || !fd->ssl_ctx) {
		fprintf(stderr, "cannot allocate xyssl context\n");
		exit(1);
	}
	havege_init( hs );
	ret = ssl_init(fd->ssl_ctx);
	if (ret) {
	    perror("ssl_init");
	    exit(1);
	}
	ssl_set_endpoint( fd->ssl_ctx, SSL_IS_CLIENT );
	ssl_set_authmode( fd->ssl_ctx, SSL_VERIFY_NONE );

	ssl_set_rng( fd->ssl_ctx, havege_rand, hs );
	//ssl_set_dbg( &ssl, my_debug, stdout );
	ssl_set_bio( fd->ssl_ctx, net_recv, &fd->sockfd, net_send, &fd->sockfd );

	ssl_set_ciphers( fd->ssl_ctx, ssl_default_ciphers );
	ssl_set_session( fd->ssl_ctx, 1, 600, fd->ssl );
#else
	/* Make SSL connection. */
	int r;
	SSL_load_error_strings();
	SSLeay_add_ssl_algorithms();
	fd->ssl_ctx = SSL_CTX_new( SSLv23_client_method() );
	if (fd->ssl_ctx == NULL)
	    perror("ssl_ctx");
	fd->ssl = SSL_new( fd->ssl_ctx );
	if (fd->ssl == NULL)
	    perror("ssl");
	SSL_set_fd( fd->ssl, fd->sockfd );
	r = SSL_connect( fd->ssl );
	if ( r <= 0 ) {
	    fprintf(stderr, "%s:%d SSL connection failed - %d\n",
		hostname, port, r );
	    exit( 1 );
	}
#endif
    }
#endif
    return fd;
}
