/*
 * PWCBSD - Philips USB webcam driver for FreeBSD 5.4 and higher
 *
 *  Copyright (C) 2006 Raaf 
 *
 * Based on the Linux pwc driver.
 *  
 *  Copyright (C) 1999-2003 Nemosoft Unv.
 *  Copyright (C) 2004-2006 Luc Saillard
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301  USA
 */

#include "pwc.h"

static void pwc_isoc_handler(usbd_xfer_handle xfer, usbd_private_handle addr,usbd_status status);
static void pwc_reset_buffers(struct pwc_softc *sc);
static void pwc_free_buffers(struct pwc_softc *sc, int detach);

#ifdef USB_DEBUG
int     pwcdebug = 0;
SYSCTL_NODE(_hw_usb, OID_AUTO, pwc, CTLFLAG_RW, 0, "USB pwc");
SYSCTL_INT(_hw_usb_pwc, OID_AUTO, debug, CTLFLAG_RW,&pwcdebug, 0, "pwc debug level");
#endif
#define MAX_ISOC_ERRORS	20

static d_open_t  pwc_open;
static d_close_t pwc_close;
static d_read_t  pwc_read;
static d_ioctl_t pwc_ioctl;
static d_mmap_t  pwc_mmap;
static d_poll_t  pwc_poll;

struct cdevsw pwc_cdevsw = {
	.d_version	= D_VERSION,
	.d_flags	= D_NEEDGIANT,
	.d_open		= pwc_open,
	.d_close	= pwc_close,
	.d_read		= pwc_read,
	.d_ioctl	= pwc_ioctl,
	.d_poll		= pwc_poll,
	.d_mmap		= pwc_mmap,
	.d_name		= "pwc",
};

static const struct pwc_info pwc_devs[] = {
	{{ 0x0471, 0x0302 }, 645, "Philips PCA645VC", &pwc_callbacks }, 
	{{ 0x0471, 0x0303 }, 646, "Philips PCA646VC" },
	{{ 0x0471, 0x0304 }, 646, "Askey VC010 type 2" },
	{{ 0x0471, 0x0307 }, 675, "Philips PCVC675K (Vesta)" },
	{{ 0x0471, 0x0308 }, 680, "Philips PCVC680K (Vesta Pro)" },
	{{ 0x0471, 0x030C }, 690, "Philips PCVC690K (Vesta Pro Scan)" },
	{{ 0x0471, 0x0310 }, 730, "Philips PCVC730K (ToUCam Fun)/PCVC830 (ToUCam II)" },
	{{ 0x0471, 0x0311 }, 740, "Philips PCVC740K (ToUCam Pro)/PCVC840 (ToUCam II)" },
	{{ 0x0471, 0x0312 }, 750, "Philips PCVC750K (ToUCam Pro Scan)" },
	{{ 0x0471, 0x0313 }, 720, "Philips PCVC720K/40 (ToUCam XS)" },
	{{ 0x0471, 0x0329 }, 740, "Philips SPC900NC" },

	{{ 0x069A, 0x0001 }, 645, "Askey VC010 type 1" }, 

	{{ 0x046D, 0x08B0 }, 740, "Logitech QuickCam Pro 3000" }, 
	{{ 0x046D, 0x08B1 }, 740, "Logitech QuickCam Notebook Pro" }, 
	{{ 0x046D, 0x08B2 }, 740, "Logitech QuickCam Pro 4000" }, 
	{{ 0x046D, 0x08B3 }, 740, "Logitech QuickCam Zoom" }, 
	{{ 0x046D, 0x08B4 }, 740, "Logitech QuickCam Zoom (new model)" }, 
	{{ 0x046D, 0x08B5 }, 740, "Logitech QuickCam Orbit/Sphere" }, 
	{{ 0x046D, 0x08B6 }, 730, "Logitech QuickCam (reserved ID)" }, 
	{{ 0x046D, 0x08B7 }, 730, "Logitech QuickCam (reserved ID)" }, 
	{{ 0x046D, 0x08B8 }, 730, "Logitech QuickCam (reserved ID)" }, 

	{{ 0x055D, 0x9000 }, 675, "Samsung MPC-C10" }, 
	{{ 0x055D, 0x9001 }, 675, "Samsung MPC-C30" },
	{{ 0x055D, 0x9002 }, 740, "Samsung SNC-35E" },

	{{ 0x041E, 0x400C }, 730, "Creative Labs Webcam 5" }, 
	{{ 0x041E, 0x4011 }, 740, "Creative Labs Webcam Pro Ex" }, 

	{{ 0x04CC, 0x8116 }, 730, "Sotec Afina Eye"  }, 

	{{ 0x06BE, 0x8116 }, 750, "AME Co. Afina Eye" }, 

	{{ 0x0d81, 0x1910 }, 740, "Visionite VCS-UC300" }, 
	{{ 0x0d81, 0x1900 }, 730, "Visionite VCS-UM100" },
};
#define pwc_lookup(v, p) ((const struct pwc_info *)usb_lookup(pwc_devs, v, p))
#define PWCUNIT(n) (minor(n))

USB_DECLARE_DRIVER(pwc);

static const struct pwc_info *
pwc_lookup1(int vendor, int product, struct pwc_softc *sc)
{
	const struct pwc_info *p;

	p = pwc_lookup(vendor, product);
	if (p != NULL) {
		bcopy(p, &sc->pwc_info, sizeof(struct pwc_info));
		if (sc->pwc_info.cb != NULL)
			bcopy(&sc->camera_cb, sc->pwc_info.cb, sizeof(struct camera_callbacks));
		else {
			/* XXX should be the default */
			sc->camera_cb = pwc_callbacks;
		}
	} else if (spcaDetectCamera(vendor, product, sc) == 0) {
		/* the descriptor is within the sc */
		p = &sc->pwc_info;
	}
	return p;
}

/*
 * Try to claim the device. Remember, the softc might be discarded
 * after the match routine so we cannot trust it to store anything.
 * However, it is a convenient argument for some of the other routines
 * e.g. those used in the attach routine - see DEVICE_PROBE(9)
 * and device_get_softc(9)
 */
static int
pwc_match(device_t dev)
{
	struct pwc_softc *sc = device_get_softc(dev);
	struct usb_attach_arg *uaa = device_get_ivars(dev);
	usb_interface_descriptor_t *id;

	Trace(TRACE_PROBE,"pwc_match: vendor=0x%x, product=0x%x release=%04x\n",uaa->vendor, uaa->product,uaa->release);
	
	if(pwc_lookup1(uaa->vendor, uaa->product, sc) == NULL)
		return UMATCH_NONE;

	/* Driver loaded when device was already plugged in, we have to claim all interfaces or get none... */
	if(uaa->usegeneric)
		return  UMATCH_VENDOR_PRODUCT;
	
	/* We don't want all interfaces */
	if(uaa->iface == NULL) 
		return UMATCH_NONE;
	
	id = usbd_get_interface_descriptor(uaa->iface);
	if(id == NULL) {
		printf("pwc: failed to get interface descriptor\n");
		return UMATCH_NONE;
	}
	
        Trace(TRACE_PROBE,"pwc_match: iface=%d\n",id->bInterfaceNumber);
	
	/* Interface 0 is the video interface
	 * Interface 1 is supposed to be audiocontrol
	 * Interface 2 is supposed to be audio
	 */
	printf("vend 0x%04x prod 0x%04x interface %d alt %d endpoints %d\n",
		uaa->vendor, uaa->product, id->bInterfaceNumber, id->bAlternateSetting, id->bNumEndpoints);
	if(id->bInterfaceNumber != 0)
		return UMATCH_NONE;
	return  UMATCH_VENDOR_PRODUCT;
}

static int
empty_consume(void)
{
	return 0;
}

static int
pwc_attach(device_t dev)
{
	struct pwc_softc *sc = device_get_softc(dev);
	struct usb_attach_arg *uaa = device_get_ivars(dev);
	usb_device_descriptor_t *dd;
	char devinfo[1024];
	const char *tmpstr;
	const struct pwc_info *info;
	int i, err;
	int unit = device_get_unit(dev);

	/* set up udev early so the lookup/attach routine can do more */
	sc->udev = uaa->device;
	sc->sc_dev = dev;
	sc->release = uaa->release;

	info = pwc_lookup1(uaa->vendor, uaa->product, sc);
	if(info == NULL) {
		device_printf(sc->sc_dev, 
            "attach error vendor/product mismatch (vendor=0x%x product=0x%x)\n",
			uaa->vendor,uaa->product);
		return ENXIO;
	}

	usbd_devinfo(uaa->device, 0, devinfo);
	device_set_desc_copy(dev, devinfo);
	device_printf(dev, "%s\n", devinfo);

	// sc->sc_iface = &dev->ifaces[0];
	err = usbd_device2interface_handle(uaa->device,0,&sc->sc_iface);
	if(err) {
		device_printf(sc->sc_dev, "failed to get interface handle\n");
		return ENXIO;
	}

	sc->power_save = 0;

	sc->spca50x.dev = sc->udev;
	sc->spca50x.pwc_softc = sc;
	/* XXX check callbacks ? */
	if (sc->camera_cb.cb_consume == NULL) {
		printf("*** Warning, no data consumer callback\n");
		sc->camera_cb.cb_consume = (consume_cb_t *)empty_consume;
	}

	if((dd = usbd_get_device_descriptor(sc->udev)) != NULL) {
#if __FreeBSD_version >= 600034
		char prod[USB_MAX_STRING_LEN];
		char manufact[USB_MAX_STRING_LEN];
		usbd_get_string(sc->udev,dd->iSerialNumber,sc->serial);
		usbd_get_string(sc->udev,dd->iProduct,prod);
		usbd_get_string(sc->udev,dd->iManufacturer,manufact);
		printf("prod <%s> manufacturer <%s> 0x%x 0x%x bcdev 0x%x\n",
			prod, manufact, UGETW(dd->idVendor), UGETW(dd->idProduct), UGETW(dd->bcdDevice));
#else
		strcpy(sc->serial,"serial only available under FreeBSD 6.0 and higher");
#endif
	} else {
		strcpy(sc->serial,"serial not found");
	}
	
	mtx_init(&sc->ptrlock,device_get_name(sc->sc_dev),NULL,MTX_DEF);

	tmpstr = "video";
	resource_string_value("pwc", unit, "devname", &tmpstr);
	sc->sc_dev_t = make_dev(&pwc_cdevsw, unit, UID_ROOT, GID_OPERATOR,
				0666, "%s%d", tmpstr, unit);
	
	resource_int_value("pwc", unit, "power_save", &sc->power_save);

	sc->stats = 0;
	resource_int_value("pwc", unit, "stats", &sc->stats);

	sc->led_on = 100;
	resource_int_value("pwc", unit, "led_on", &sc->led_on);
	if(sc->led_on < 0)
		sc->led_on = 0;
	
	sc->led_off = 0;
	resource_int_value("pwc", unit, "led_off", &sc->led_off);
	if(sc->led_off < 0)
		sc->led_off = 0;
	
	sc->vframes = 5;
	resource_int_value("pwc", unit, "fps", &sc->vframes);
	if(sc->vframes < 4 || sc->vframes > 30)
		sc->vframes = 5;

#ifdef USE_MMAP
	sc->pwc_mbufs = 2;	/* Default number of mmap() buffers */
#else
	sc->pwc_mbufs = 1;
#endif
	resource_int_value("pwc", unit, "mbufs", &sc->pwc_mbufs);
	if(sc->pwc_mbufs < 1 || sc->pwc_mbufs > MAX_IMAGES)
		sc->pwc_mbufs = 1;
	
	sc->pwc_fbufs = 3;	/* Default number of frame buffers */
	resource_int_value("pwc", unit, "fbufs", &sc->pwc_fbufs);
	if(sc->pwc_fbufs < 2 || sc->pwc_fbufs > MAX_FRAMES)
		sc->pwc_fbufs = 2;
	
	sc->vcompression = 2;	/* 0..3 = uncompressed..high */
	resource_int_value("pwc", unit, "compression", &sc->vcompression);
	if(sc->vcompression > 3)
		sc->vcompression = 3;
			
	sc->vsize = PSZ_CIF;	/* XXX default format, was PSZ_SIF */
	tmpstr = NULL;
	resource_string_value("pwc", unit, "size", &tmpstr);
	if(tmpstr != NULL) {
		for (i = 0; i < PSZ_MAX; i++) {
			if (!strcmp(pwc_image_sizes[i].name, tmpstr)) { /* Found! */
				sc->vsize = i;
				break;
			}
		}
	}
	if(sc->vsize == PSZ_VGA && sc->vframes > 15)
		sc->vframes = 15;
		
	sc->pwc_pad = 0;	/* XXX only exact formats */
	resource_int_value("pwc", unit, "pad", &sc->pwc_pad);

	if (sc->camera_cb.cb_attach)
		sc->camera_cb.cb_attach(sc);
	device_printf(sc->sc_dev, "%s %s (USB webcam) type %d\n",
		sc->pwc_info.name, sc->serial, sc->pwc_info.type);
	device_printf(sc->sc_dev, "  sensor %d (%s) bridge %d (%s)\n",
			sc->pwc_info.sensor, sensor_name(sc->pwc_info.sensor),
			sc->pwc_info.bridge, bridge_name(sc->pwc_info.bridge));

	usbd_add_drv_event(USB_EVENT_DRIVER_ATTACH, sc->udev,USBDEV(sc->sc_dev));
	return 0; /* success */
}

static void close_videopipe(struct pwc_softc *sc)
{
#ifdef NEW_USB
#warning using stack version 2
	usbd_transfer_unsetup(sc->xfer, MAX_ISOC_TRANSFERS);
#else
	if(sc->sc_videopipe != NULL) {
		usbd_abort_pipe(sc->sc_videopipe);
		usbd_close_pipe(sc->sc_videopipe);
		sc->sc_videopipe = NULL;
	}
#endif
}

static int
pwc_detach(device_t dev)
{
	struct pwc_softc *sc = device_get_softc(dev);

again:
	Trace(TRACE_PROBE,"pwc_detach: sc=%p\n",sc);
	
	close_videopipe(sc);

	sc->error_status = EPIPE;

	if(sc->vopen) {
		if(sc->state & PWC_ASLEEP) {
			wakeup(sc);
		}
		if(sc->state & PWC_POLL) {
			sc->state &= ~PWC_POLL;
			selwakeuppri(&sc->rsel,PZERO);
		}
		device_printf(sc->sc_dev, "Disconnected while webcam is in use!\n");
		usb_detach_wait(USBDEV(sc->sc_dev));
		goto again;
	}
	
	if(sc->sc_dev_t != NULL)
		destroy_dev(sc->sc_dev_t);
	
	mtx_destroy(&sc->ptrlock);
	pwc_free_buffers(sc,1);
	usbd_add_drv_event(USB_EVENT_DRIVER_DETACH, sc->udev,USBDEV(sc->sc_dev));
	return 0;
}

/*
 * open handler. Perform the following tasks:
 * - turn on the camera;
 * - allocate any necessary buffer (frame, decompressor, isoc buffers);
 * - initialize bridge and sensor as needed for the required format
 * - start ISOC transfers
 */
int
pwc_open(struct cdev *dev, int flag, int mode, usb_proc_ptr p)
{
	int i,err;
	int old_vframes;
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);

	if (sc == NULL)
		return ENXIO;

	Trace(TRACE_OPEN,"pwc_open: flag=%d, mode=%d, unit=%d\n",
		flag, mode, unit);

	if(sc->error_status == EPIPE)
		return sc->error_status;

	mtx_lock(&sc->ptrlock);
	if (sc->vopen) {	/* XXX could well use an atomic instruction */
		mtx_unlock(&sc->ptrlock);
		return EBUSY;
	}
	sc->vopen = 1;	/* protect against multiple opens */
	mtx_unlock(&sc->ptrlock);

	/* When opening, try the last video mode, whose id is saved in sc->vframes.
	 * Store it in a local variable as the callback will override it
	 * XXX the fallback mode should be in the try_video_mode handler.
	 */
	old_vframes = sc->vframes;

	/* Invoke the open callback if present */
	if (sc->camera_cb.cb_open && (err = sc->camera_cb.cb_open(sc)) )
		goto bad;

	err = ENOMEM;

	/* Allocate frame buffer structure */
	i = sc->pwc_fbufs * sizeof(struct pwc_frame_buf);
	sc->fbuf = malloc(i, M_USBDEV, M_WAITOK);
	if (sc->fbuf == NULL) {
		device_printf(sc->sc_dev, 
		    "Failed to allocate frame buffer structure.\n");
		goto bad;
	}
	memset(sc->fbuf, 0, i);
	
	/* create frame buffers, and make circular ring */
	for (i = 0; i < sc->pwc_fbufs; i++) {
		sc->fbuf[i].data = malloc(PWC_FRAME_SIZE,M_USBDEV,M_WAITOK);
		if (sc->fbuf[i].data == NULL) {
			device_printf(sc->sc_dev, "Failed to allocate frame buffer\n");
			goto bad;
		}
		memset(sc->fbuf[i].data, 128, PWC_FRAME_SIZE);
		sc->fbuf[i].filled = 0;
	}

	/* Allocate image buffer */
	if(sc->image_data == NULL) {
		sc->image_data = malloc(sc->pwc_mbufs * round_page(sc->len_per_image), M_USBDEV, M_WAITOK);
		if(sc->image_data == NULL) {
			device_printf(sc->sc_dev, "Failed to allocate image buffers\n");
			goto bad;
		}
	}
	for(i = 0; i < sc->pwc_mbufs; i++)	{
		sc->images[i].bufmem = ((char*)sc->image_data) + i * round_page(sc->len_per_image);
		sc->images[i].vma_use_count = 0;
	}
	for (; i < MAX_IMAGES; i++)
		sc->images[i].bufmem = NULL;
			
	for (i = 0; i < sc->pwc_mbufs; i++)
		sc->image_used[i] = 0;

#ifndef NEW_USB
	/* Allocate iso transfers */
	for (i = 0; i < MAX_ISOC_TRANSFERS; i++) {
		sc->sbuf[i].sc = sc;
		sc->xfer[i] = usbd_alloc_xfer(sc->udev); 
		
		if(sc->xfer[i] == NULL) {
			device_printf(sc->sc_dev, "Failed to allocate transfer\n");
			goto bad;
		}
		
		sc->sbuf[i].data = usbd_alloc_buffer(sc->xfer[i], ISO_BUFFER_SIZE);
		if(sc->sbuf[i].data == NULL) {
			device_printf(sc->sc_dev, "Failed to allocate transfer buffer %d\n", i);
			goto bad;
		}
	}
#endif

	sc->state = 0;
	sc->frames = 0;
	sc->start_ticks = ticks;
	sc->byte_count = 0;
	sc->bytes_skipped = 0;

	sc->vframe_count = 0;
	sc->vframes_dumped = 0;
	sc->vframes_error = 0;
	sc->visoc_errors = 0;
	sc->error_status = 0;
	sc->vsnapshot = 0;
	
	/* Try the last used video */
	sc->state |= PWC_INIT;	
	err = -pwc_try_video_mode(sc,
		pwc_image_sizes[sc->vsize].xy.x, pwc_image_sizes[sc->vsize].xy.y,
		old_vframes, sc->vcompression, 0 /* no snapshot */);
	sc->state &= ~PWC_INIT;

	init_jpeg_decoder(&sc->spca50x);
	Trace(TRACE_OPEN, "open complete with err %d\n", err);
	if(err == 0)
		return 0;

bad:
	pwc_free_buffers(sc,0);
	sc->vopen = 0;
	return err;
}

/* Important: close doesn't get called after a detach */
int
pwc_close(struct cdev *dev, int flag, int mode, usb_proc_ptr p)
{
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);	/* NULL check ? */
	int d = ticks - sc->start_ticks;

	if (d <= 0)
		d++;
	if (sc->frames == 0)
		sc->frames = 1;
	printf("%d bytes skipped\n", sc->bytes_skipped);
	printf("%d frames (%d dumped) in %d ticks, %d fps %d visoc_errors %d bytes %d bpf\n",
		sc->frames, sc->vframes_dumped, d,
		(sc->frames * hz * 1000) / d,
		sc->visoc_errors,
		sc->byte_count,
		sc->byte_count / sc->frames);
	
	Trace(TRACE_OPEN,"pwc_close: flag=%d, mode=%d, unit=%d\n", flag, mode, unit);
printf("called close routine\n");
	/* Dump statistics, but only if a reasonable amount of frames were
	   processed (to prevent endless log-entries in case of snap-shot
	   programs)
	 */
	if(sc->vframe_count > 20 && sc->stats) {
		device_printf(sc->sc_dev,
            "%d frames received, dumped %d frames, %d frames with errors.\n",
			sc->vframe_count, sc->vframes_dumped, sc->vframes_error);
	}
	close_videopipe(sc);
	if(sc->error_status != EPIPE) {
		/* if not unplugged, call the camera-specific shutdown */
		if (sc->camera_cb.cb_close)
			sc->camera_cb.cb_close(sc);
	}
	pwc_free_buffers(sc,0);

        sc->vopen = 0;
	usb_detach_wakeup(USBDEV(sc->sc_dev));	/* XXX ? */

        return 0;
}

/*
 * Read handler. Assume the ISOC engine is up and running, all we have to do
 * is possibly wait for a new frame, and return the data.
 * Once done, advance the frame pointers to the next frame.
 * At the moment there is no attempt here to do any error recovery,
 * in case this belongs to the interrupt service routine.
 */
int
pwc_read(struct cdev *dev, struct uio *uio, int flag)
{
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);
        int bytes_to_read;
	int count = uio->uio_resid;
	int err;

	Trace(TRACE_READ,"pwc_read: flag=%d, unit=%d\n", flag, unit);
	
	if (sc->error_status)
		return sc->error_status;

	/* In case we're doing partial reads, we don't have to wait for a frame */
	if(sc->image_read_pos == 0) {
		
		while ((err = pwc_handle_frame(sc)) == -EBUSY) {
		
			if(flag & O_NONBLOCK)
				return EWOULDBLOCK;
			
			sc->state |= PWC_ASLEEP;	
			err = tsleep(sc, PZERO | PCATCH, "pwcrd", 0);
			sc->state &= ~PWC_ASLEEP;
			
			if(sc->error_status)
				return sc->error_status;
			else if(err)
				return err;
		}
		if(err < 0)
			return -err;
	}

	if(sc->vpalette == VIDEO_PALETTE_RAW)
		bytes_to_read = sc->frame_size;
	else
 		bytes_to_read = sc->view.size;

	/* copy bytes to user space; we allow for partial reads */
	if(count + sc->image_read_pos > bytes_to_read)
		count = bytes_to_read - sc->image_read_pos;
	
	Trace(TRACE_READ_VERBOSE, "pwc_read: wants: %d bytes, reading: %d bytes\n",uio->uio_resid,count);

	err = uiomove(sc->images[sc->fill_image].bufmem + sc->image_read_pos,count,uio);
	if(err)
		return err;
	
	sc->image_read_pos += count;
	if(sc->image_read_pos >= bytes_to_read) { /* All data has been read */
		sc->frames++;
		sc->image_read_pos = 0;
		/* move to next image */
		sc->image_used[sc->fill_image] = 0;
		sc->fill_image = (sc->fill_image + 1) % sc->pwc_mbufs;
	}
	
	return 0;
}

int
pwc_ioctl(struct cdev *dev, u_long cmd, caddr_t addr, int flag, usb_proc_ptr p)
{
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);

	Trace(TRACE_READ,"pwc_ioctl: cmd 0x%lu addr %p\n", cmd, addr);

	if (sc->error_status)
		return sc->error_status;
	
	return -pwc_video_do_ioctl(sc,cmd,addr,unit);
}

int
pwc_poll(struct cdev *dev, int events, usb_proc_ptr p)
{
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);
	int revents = 0;
	
	Trace(TRACE_READ,"pwc_poll: events %d\n", events);

	if(sc->error_status)
		return sc->error_status;

	if (events & (POLLIN | POLLRDNORM)) {
		mtx_lock(&sc->ptrlock);
		if(sc->full_frames == NULL) {
			sc->state |= PWC_POLL;
			selrecord(p,&sc->rsel);
			Trace(TRACE_READ,"pwc_poll: will block\n");
		}
		else {
			revents |= events & (POLLIN | POLLRDNORM);
			Trace(TRACE_READ,"pwc_poll: not blocking\n");
		}
		mtx_unlock(&sc->ptrlock);
	}
	if (events & (POLLOUT | POLLWRNORM)) {
		/* write is not allowed, so no point blocking on it */
		revents |= events & (POLLOUT | POLLWRNORM);
	}
	return revents;
}

int
pwc_mmap(struct cdev *dev, vm_offset_t offset, vm_paddr_t *paddr, int nprot)
{
#ifdef USE_MMAP
	int unit = PWCUNIT(dev);
	struct pwc_softc *sc = devclass_get_softc(pwc_devclass, unit);

	if(sc == NULL)
		return ENXIO;
	
	if (sc->error_status)
		return sc->error_status;

	if(nprot & PROT_EXEC || sc->image_data == NULL || offset < 0 || offset >= sc->pwc_mbufs * round_page(sc->len_per_image))
		return -1;

	*paddr = vtophys(((char*)sc->image_data) + offset);
	Trace(TRACE_MEMORY,"pwc_mmap: phys: 0x%08x offset: 0x%08x\n",*paddr,offset);
	return 0;
#else
	return ENXIO;
#endif
}

#ifdef NEW_USB
/*
 * Called whenever there is something significant for a
 * read trasnfer, e.g. error, setup, or data ready
 */
static void
pwc_isoc_read_callback(struct usbd_xfer *xfer)
{
	struct pwc_softc *sc = xfer->priv_sc;
	int i;	/* buffer number */
	uint16_t isize = sc->vmax_packet_size;	/* each buffer is this big */

	USBD_CHECK_STATUS(xfer);	/* demux for the interrupt reason */
tr_transferred:
	for (i = 0; i < MAX_ISOC_TRANSFERS; i++) {
		if (xfer->frlengths[i] == 0)
			continue;
		sc->consume(sc, (unsigned char *)xfer->buffer + isize * i, xfer->frlengths[i]);
	}
 tr_setup:
 tr_error:
	for(i = 0; i < MAX_ISOC_TRANSFERS; i++) {
                /* setup size for next transfer */
                xfer->frlengths[i] = isize;
        } 
        usbd_start_hardware(xfer);
        return;
}
#else	/* !NEW_USB */
static void setup_xfer(struct pwc_iso_buf *req, usbd_xfer_handle xfer)
{
	/* setup size for next transfer */
	struct pwc_softc *sc = req->sc;
	int i;

	for (i = 0; i < ISO_FRAMES_PER_DESC; i++)
		req->sizes[i] = sc->vmax_packet_size;
	
	usbd_setup_isoc_xfer(xfer, sc->sc_videopipe, req, req->sizes,
				ISO_FRAMES_PER_DESC,USBD_NO_COPY, pwc_isoc_handler);

	usbd_transfer(xfer);
}
#endif	/* !NEW_USB */

int
pwc_try_video_mode(struct pwc_softc *sc, int width, int height, int new_fps, int new_compression, int new_snapshot)
{
	usb_endpoint_descriptor_t *found = NULL, *edesc = NULL;
	int i, err, ret;

	close_videopipe(sc);

	pwc_reset_buffers(sc);
	
	/* Try to set video mode... if that fails fallback to previous mode  */
	ret = pwc_set_video_mode(sc, width, height, new_fps, new_compression, new_snapshot);
	if(ret < 0) {
		Trace(TRACE_FLOW, "pwc_set_video_mode attempt 1 failed.\n");
		err = pwc_set_video_mode(sc,sc->view.x,sc->view.y,sc->vframes,sc->vcompression,sc->vsnapshot);
		if(err < 0) {
		        Trace(TRACE_FLOW, "pwc_set_video_mode attempt 2 failed.\n");
			return err;
		}
	}

	sc->drop_frames++; /* try to avoid garbage during switch */
	sc->vsync = VSYNC_NOCOPY;	

	err = set_alt_interface(sc->udev, sc->sc_iface,sc->valternate);
	if(err != USBD_NORMAL_COMPLETION) {
		device_printf(sc->sc_dev,
            "Failed to set alternate interface to: %d (error %d)\n",
            sc->valternate,err);
		return -err;
	}

        if(sc->sc_iface == NULL || sc->sc_iface->idesc == NULL) {
		device_printf(sc->sc_dev, "Failed to get endpoint count\n");
		return -EINVAL;
	}
	for (i = 0; i < sc->sc_iface->idesc->bNumEndpoints; i++) {
		edesc = usbd_interface2endpoint_descriptor(sc->sc_iface, i);
		if (edesc)
			Trace(TRACE_IOCTL, "desc %d endpoint %d size %d\n", i, UE_GET_ADDR(edesc->bEndpointAddress), UGETW(edesc->wMaxPacketSize));
		if(edesc != NULL && UE_GET_ADDR(edesc->bEndpointAddress) == sc->vendpoint)
			found = edesc;
	}
	if(found == NULL) {
		device_printf(sc->sc_dev, "Failed to find videoendpoint %d\n", sc->vendpoint);
		return -EINVAL;
	}
	edesc = found;
	
	i = UGETW(edesc->wMaxPacketSize); /* bit 0-10 are the miniframe, 11-12 the count+1 */
	sc->vmax_packet_size = (i & 0x7ff) * (1 + ((i>>11) & 3) );
	printf("endpoint %d frame size %d\n", sc->vendpoint, sc->vmax_packet_size);

	if(sc->vmax_packet_size < 0 || sc->vmax_packet_size > ISO_MAX_FRAME_SIZE) {
		device_printf(sc->sc_dev, "Invalid packetsize (%d) for endpoint %d\n",
			sc->vmax_packet_size,edesc->bEndpointAddress);
		return -EINVAL;
	}

	/* XXX for 2.0 (fast) pipes maybe i need to call usbd_setup_pipe directly ? */

#ifdef NEW_USB
        mtx_assert(&sc->sc_mtx, MA_OWNED);

        if(!(sc->state & UGEN_OPEN_IN) && sc->pipe_in) {
		struct usbd_config usbd_config[1] = { /* zero */ };
                usbd_config[0].type = ed->bmAttributes & UE_XFERTYPE;
                usbd_config[0].endpoint = ed->bEndpointAddress & UE_ADDR;
                usbd_config[0].direction = UE_DIR_IN;
                usbd_config[0].timeout = sc->in_timeout;
                         
                switch(ed->bmAttributes & UE_XFERTYPE) {
		default:
			printf("sorry can only work on ISOC pipes");
			return -EINVAL;
			break;

		case UE_ISOCHRONOUS:
			isize = UGETW(ed->wMaxPacketSize);
			sc->in_frames = ISO_FRAMES_PER_DESC;
			if(usbd_get_speed(sc->sc_udev) == USB_SPEED_HIGH)
				sce->in_frames *= 8;
			  if(ugen_allocate_blocks(sc, sce, UGEN_RD_CFG,
			      &sce->in_queue, sce->in_frames * 10, isize) == 0)
				return ENOMEM;
			 
			usbd_config[0].flags = USBD_SHORT_XFER_OK;
			usbd_config[0].bufsize = isize * sce->in_frames;
			usbd_config[0].frames = sce->in_frames;
			usbd_config[0].callback = ugenisoc_read_callback;
			usbd_config[0].timeout = 0;

			err = __usbd_transfer_setup(sc, sce, UGEN_RD_CFG,
				sc->sc_udev, sce->pipe_in->iface_index,
				&sce->xfer_in[0], &usbd_config[0], 1);

			if(!err) {     
				err = __usbd_transfer_setup(sc, sce, UGEN_RD_CFG,
				sc->sc_udev, sce->pipe_in->iface_index,
				&sce->xfer_in[1], &usbd_config[0], 1);
			}

			if(err) {
				usbd_transfer_unsetup(&sce->xfer_in[0], 1);
				usbd_transfer_unsetup(&sce->xfer_in[1], 1);
				ugen_free_blocks(&sce->in_queue);
				return (EIO);
			}
                         
			usbd_transfer_start(sce->xfer_in[0]);
			usbd_transfer_start(sce->xfer_in[1]);
			printf("isoc open done\n");
			break;
		}
	}
#else
	printf("speed %d attributes %d\n",
		sc->sc_iface->device->speed,
		edesc->bmAttributes);
	err = usbd_open_pipe(sc->sc_iface,edesc->bEndpointAddress, 0, &sc->sc_videopipe);
	if(err != USBD_NORMAL_COMPLETION) {
		device_printf(sc->sc_dev, "Failed to open videopipe (error %d)\n", err);
		return -err;
	}

	for (i = 0; i < MAX_ISOC_TRANSFERS; i++) {
		setup_xfer(&sc->sbuf[i], sc->xfer[i]);
	}
#endif
	Trace(TRACE_FLOW, "try_video_mode done\n");

	if(sc->state & PWC_INIT)
		return 0;
	
	return ret;
}

/*
 * Prepare for the next frame, appending the current one
 * to the full list, and grab one from the free list
 * (if available) or from the full list if nothing free.
 */
int
pwc_next_fill_frame(struct pwc_softc *sc)
{
	int ret = 0;
	
	mtx_lock(&sc->ptrlock);

	if(sc->fill_frame != NULL) { /* append to 'full' list */
		if (sc->full_frames == NULL)
			sc->full_frames = sc->fill_frame;
		else
			sc->full_frames_tail->next = sc->fill_frame;
		sc->full_frames_tail = sc->fill_frame;
	}
	if (sc->empty_frames != NULL) {
		/* We have empty frames available. That's easy */
		sc->fill_frame = sc->empty_frames;
		sc->empty_frames = sc->empty_frames->next;
	} else { /* Hmm. Take it from the full list */
		sc->fill_frame = sc->full_frames;
		sc->full_frames = sc->full_frames->next;
		ret = 1;	/* report the frame dumped above */
		sc->vframes_dumped++;
		if (sc->vframe_count > FRAME_LOWMARK && sc->vframes_dumped <= 20)
			Trace(TRACE_ISOC,"Dumping frame %d%s.\n",
				sc->vframe_count,
				sc->vframes_dumped == 20 ? " (last message)" : "");
	}
	sc->fill_frame->next = NULL;
	mtx_unlock(&sc->ptrlock);
	return ret;
}

static void
pwc_isoc_handler(usbd_xfer_handle xfer, usbd_private_handle addr,usbd_status status)
{
   	struct pwc_iso_buf *req = addr;
	struct pwc_softc *sc = req->sc;
	u_int32_t count;
	int awake = 0;
	int i;
	
	usbd_get_xfer_status(xfer, NULL, NULL, &count, NULL);
	Trace(TRACE_ISOC_VERBOSE, "pwc_isoc_handler: status=%d count=%u\n",status,count);

	if (status == USBD_CANCELLED) {
		Trace(TRACE_ISOC, "pwc_isoc_handler: status = cancelled\n");
		return;
	}

	if(status != USBD_NORMAL_COMPLETION) {
		Trace(TRACE_ISOC, "pwc_isoc_handler called with status: %d\n",status);
		if (++sc->visoc_errors > MAX_ISOC_ERRORS) {
			if(sc->error_status != EIO)
				device_printf(sc->sc_dev, "Too many ISOC errors, bailing out.\n");
			sc->error_status = EIO;
			awake = 1;
		}
	} else { /* normal completion */
		/* Reset ISOC error counter. We did get here, after all. */
		// sc->visoc_errors = 0;
		/* Consume data */
		for (i = 0; i < ISO_FRAMES_PER_DESC; i++) {
			sc->byte_count += req->sizes[i];
			awake |= sc->camera_cb.cb_consume(sc, req->data + sc->vmax_packet_size * i, req->sizes[i]);
		}
	}
	if(awake) {
		if(sc->state & PWC_ASLEEP) {
			wakeup(sc);
		}
		if(sc->state & PWC_POLL) {
			sc->state &= ~PWC_POLL;
			selwakeuppri(&sc->rsel, PZERO);
		}
	}
	/* setup size for next transfer */
	setup_xfer(req, xfer);
}

int
pwc_handle_frame(struct pwc_softc *sc)
{
	int ret = 0;
	unsigned char *yuv;
	char *image;

	mtx_lock(&sc->ptrlock);
	
	/* First grab our read_frame; this is removed from all lists, so
	   we can release the lock after this without problems */

	if (sc->full_frames == NULL) {
		mtx_unlock(&sc->ptrlock);
		return -EBUSY;
	}
	
	sc->read_frame = sc->full_frames;
	sc->full_frames = sc->full_frames->next;
	sc->read_frame->next = NULL;
	
	/* Decompression is a lenghty process, so it's outside of the lock.
	 * This gives the isoc_handler the opportunity to fill more frames
	 * in the mean time.
	*/
	mtx_unlock(&sc->ptrlock);
	image = sc->images[sc->fill_image].bufmem;
	if (image == NULL)
		return -EFAULT;
	yuv = ((char*)sc->read_frame->data) + sc->frame_header_size; /* skip header */
	/* Raw format; that's easy... */
        if (sc->vpalette == VIDEO_PALETTE_RAW) {
                memcpy(image, yuv, sc->frame_size);
        } else if (sc->camera_cb.cb_decompress) {
		ret = sc->camera_cb.cb_decompress(sc, yuv, image, sc->read_frame->filled);
	}
	mtx_lock(&sc->ptrlock);

	/* We're done with read_buffer, tack it to the end of the empty buffer list */
	if(sc->empty_frames == NULL) {
		sc->empty_frames = sc->read_frame;
		sc->empty_frames_tail = sc->empty_frames;
	} else {
		sc->empty_frames_tail->next = sc->read_frame;
		sc->empty_frames_tail = sc->read_frame;
	}
	
	sc->read_frame = NULL;
	mtx_unlock(&sc->ptrlock);
	return ret;
}

/*
 * Reset the full_frames list, link all frames in the
 * empty_frames list, and take the first one for fill_frame.
 */
static void 
pwc_reset_buffers(struct pwc_softc *sc)
{
	int i;

	mtx_lock(&sc->ptrlock);
	
	sc->full_frames = NULL;
	sc->full_frames_tail = NULL;
	for (i = 0; i < sc->pwc_fbufs; i++) {
		sc->fbuf[i].filled = 0;
		if (i > 0)
			sc->fbuf[i].next = &sc->fbuf[i - 1];
		else
			sc->fbuf->next = NULL;
	}
	sc->empty_frames = &sc->fbuf[sc->pwc_fbufs - 1];
	sc->empty_frames_tail = sc->fbuf;
	sc->read_frame = NULL;
	sc->fill_frame = sc->empty_frames;
	sc->empty_frames = sc->empty_frames->next;

	sc->image_read_pos = 0;
	sc->fill_image = 0;

	mtx_unlock(&sc->ptrlock);
}

static void
pwc_free_buffers(struct pwc_softc *sc, int detach)
{
	int i;
	Trace(TRACE_MEMORY, "Entering free_buffers(%p).\n", sc);
	if (sc->fbuf != NULL) {
		for (i = 0; i < sc->pwc_fbufs; i++) {
			if (sc->fbuf[i].data != NULL) {
				free(sc->fbuf[i].data,M_USBDEV);
				sc->fbuf[i].data = NULL;
			}
		}
		free(sc->fbuf,M_USBDEV);
		sc->fbuf = NULL;
	}

	if (sc->decompress_data != NULL) {
		free(sc->decompress_data,M_USBDEV);
		sc->decompress_data = NULL;
	}
#ifdef USE_MMAP
	if(sc->image_data != NULL && detach == 1)
#else
	if(sc->image_data != NULL)
#endif
	{
		free(sc->image_data,M_USBDEV);
		sc->image_data = NULL;
	}
#ifdef NEW_USB
#warning	must free xfer buffers
#else
	for (i = 0; i < MAX_ISOC_TRANSFERS; i++) {
		if (sc->xfer[i] != NULL) {
			usbd_free_xfer(sc->xfer[i]); /* implicit buffer free */
			sc->xfer[i] = NULL;
		}
	}
#endif
}

DRIVER_MODULE(pwc, uhub, pwc_driver, pwc_devclass, usbd_driver_load, 0);
