// File        : dmaplayf.c
//
// Description : Function to playback any format Covox file using dmaPlay().
//               File size is only limited to size of hard drive.
//
// Written by : Ryan Hanlon
// Date       : Feb 18, 1992
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <dos.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define   _DMA_PLAY_FILE_BUFFER_SIZE        0x8000
#define   _DMA_PLAY_FILE_BUFFER_SIZE_MAX    0xF000

#define   _DMA_PLAY_FILE_BUFFER_COUNT       4
#define   _DMA_PLAY_FILE_BUFFER_COUNT_MIN   2
#define   _DMA_PLAY_FILE_BUFFER_COUNT_MAX   5

extern BYTE _dmaInProgressFlag;

// Flag used to determine if a key is stuck during  i/o.
//
extern BYTE _ioStopFlag;


//
WORD dmaPlayFile( HANDLE fileHandle, BYTE fileFormat, BYTE playbackRate,
                  WORD   portOut,    WORD channel,    WORD irqNumber,  
                  BOOL   noiseFlag,  BOOL trebleFlag, BOOL hookKBFlag,
                  LONG   bufferSize, WORD bufferCount )
{

   static BYTE   tempRate;
   static BYTE   tempFormat;

   _CONVERT_DATA cnvrtData;  // Data structure defined in cvxdigi.h used for
                             // unpacking of sound data.

   WORD  portOutAddress;     // Base port of Covox DMA board.

   LPSTR dmaBuffer[ _DMA_PLAY_FILE_BUFFER_COUNT_MAX ];

   WORD  currentBuffer = 0;  // This variable is used to switch between the
                             // buffers being processed by dmaPlay().
   LONG  bytesAvailable;     // Set by cvxBufferAlloc as the bytes available.
   LONG  bytesRead;          // Actual number of bytes read with cvxFileRead
   LONG  sourceSize;         // Amount of source read from file.
   WORD  i;                  // Loop counter.

   WORD  returnValue = _ERROR_NONE;


   // If no buffer Size is given or bufferSize specified is too large
   // then use defaults.
   //
   if( !bufferSize || 
       (bufferSize > ( LONG )_DMA_PLAY_FILE_BUFFER_SIZE_MAX) )
      bufferSize = ( LONG )_DMA_PLAY_FILE_BUFFER_SIZE;

   // If no bufferCount is given or bufferCount is too large
   // then use defaults.
   //
   if( !bufferCount ||
       (bufferCount > _DMA_PLAY_FILE_BUFFER_COUNT_MAX) ||
       (bufferCount < _DMA_PLAY_FILE_BUFFER_COUNT_MIN) )
      bufferCount = _DMA_PLAY_FILE_BUFFER_COUNT;

   // Subtract one from bufferCount because one buffer is used for unpack.
   //
   if( bufferCount )
      bufferCount -= 1;

   // Allocate memory for buffers used during DMA playback.
   //
   for(i=0; ( (i<bufferCount) && !returnValue ) ; i++)
   {
      dmaBuffer[i]= cvxBufferAlloc( bufferSize, &bytesAvailable );

      if( dmaBuffer[i] == _NULL )
      {
         returnValue = _ERROR_MEMORY_ALLOC;
      }
   }

   if( !returnValue )
   {
      // Allocate memory for buffer used to unpack sound data.
      //
      cnvrtData.sourcePointer = cvxBufferAlloc( bufferSize, &bytesAvailable );

      if( cnvrtData.sourcePointer == _NULL )
      {
         returnValue = _ERROR_MEMORY_ALLOC;
      }

      if( !returnValue )
      {
         // Initialize DMA for playback.
         //
         returnValue = dmaInit(portOut, channel, irqNumber, &portOutAddress);
      
         if( !returnValue )
         {
            // Get header information, if it exists, from file.
            //
            returnValue =  cvxFileRead( fileHandle, cnvrtData.sourcePointer,
                                        (LONG)_HEADER_LENGTH_VMF, &sourceSize );
      
            if( !returnValue )
            {
               // Get file format and playback rate.
               //
               cvxHeaderParse( cnvrtData.sourcePointer, &tempFormat, &tempRate );
               
               // If no file format was specified use return from cvxHeaderParse.
               //
               if( !fileFormat )
                  fileFormat = tempFormat;
               else
                  // If specified file format is different from format found in
                  // header then return with error.
                  //
                  if ( fileFormat != tempFormat )
                    returnValue = _ERROR_INVALID_FORMAT; 
         
               // If no playback rate was specified then set rate to rate found
               // in header.
               //
               if( !playbackRate )
                  playbackRate = tempRate;

               // If rate is zero then use default of 132.
               //
               if( !playbackRate )
                  playbackRate = _CVX_RATE_DEFAULT;
         
               // Assign buffer for unpackFirst.
               //
               cnvrtData.destinationPointer  = dmaBuffer[ currentBuffer ];

               // If file type is VMF then bypass header and set format to .V8
               //
               if( fileFormat == _FORMAT_VMF )
               {
                  sourceSize -= ( LONG )_HEADER_LENGTH_VMF;
                  fileFormat = _FORMAT_V8;
                  cnvrtData.destinationPointer  += ( LONG )_HEADER_LENGTH_VMF;
               }

       
               cnvrtData.sourceLength = sourceSize; // Just convert the header information for now.
      
               cnvrtData.bufferFormat        = ( WORD )fileFormat;
               cnvrtData.sampleRate          = ( BYTE )playbackRate;
               cnvrtData.trebleFlag          = ( BOOL )trebleFlag;
               cnvrtData.noiseFlag           = ( BOOL )noiseFlag;
               cnvrtData.destinationLength   = (LONG)_HEADER_LENGTH_VMF;
      
               
               if( unpackFirst((_CONVERT_DATA far *) &cnvrtData) )
                  returnValue = _ERROR_DURING_UNPACK_FIRST;
      
               cnvrtData.destinationPointer = dmaBuffer[ currentBuffer ];
               cnvrtData.destinationLength = bufferSize;
      
               // Hook int 9h (keyboard interrupt).  This hook will set the flag
               // _ioStopFlag when a key has been struck.
               //
               if( hookKBFlag )
                  hookKeyboard();
      
               // Do the following loop until all data has been played or 
               // until a key has been struck.
               //
               while( !returnValue && !_ioStopFlag )
               {
                  do
                  {
                     _asm nop
      
                  }while( ( dmaNumberInQueue() == bufferCount ) && !_ioStopFlag );
      
                  // Check flag indicating exit playback.
                  //
                  if( !returnValue )
                  {
                     // Move left over source data to the begining of 
                     // the file buffer.
                     //
                     sourceSize -= cnvrtData.sourceUsed;
                     for(i = 0 ; (LONG)i < sourceSize ; i++)
                        *(cnvrtData.sourcePointer + i) = *(cnvrtData.sourcePointer + cnvrtData.sourceUsed + i);
      
                     returnValue = cvxFileRead( fileHandle,
                                                (LPSTR)((HPSTR)cnvrtData.sourcePointer + sourceSize),
                                                bufferSize - sourceSize, 
                                                &bytesRead );
                     if( !returnValue )
                     {
                        sourceSize += bytesRead;
            
                        cnvrtData.sourceLength = (LONG) sourceSize;
                        cnvrtData.destinationPointer = dmaBuffer[ currentBuffer ];
      
                        if(unpackNext((_CONVERT_DATA far *) &cnvrtData))
                        {
                           returnValue = _ERROR_DURING_UNPACK_NEXT;
                           break;
                        }
      
                        // If no more unpacking to be done then loop here while
                        // dma playback is still active and no keypress occurs.
                        //
                        if( cnvrtData.destinationFilled == 0 )
                        {
                           // Done now wait for end to be played out 
                           //
                           while( _dmaInProgressFlag && !_ioStopFlag )
                           {
                              _asm   nop
                           }

                           break;
                        }
                        else
                        {
                           if( !_ioStopFlag && !returnValue )
                           {
                              // Place a playback request into the DMA queue.
                              //
                              dmaPlay( dmaBuffer[currentBuffer],(LONG)cnvrtData.destinationFilled,playbackRate, 1 );
                           }
      
                           // Increment currentBuffer so next buffer is accessed.
                           //
                           if(currentBuffer == bufferCount - 1)
                           {
                              currentBuffer=0;
                           }
                           else
                           {
                              currentBuffer++;
                           }
                        }

                     } // end if( !returnValue )

                  } // end if( !returnValue )
      
               } // end while( !returnValue && !_ioStopFlag )
      
               dmaFlushQueue();
   
               // Remove the dma sub-system.
               //
               dmaUninit();
   
               // reset keyboard vector.
               //
               if( hookKBFlag )
                  unhookKeyboard();
   
            } // end if cvxFileRead ok.
   
         } // end if dmaInit() ok 
      
      } // end if cvxBufferAlloc for cnvrtData.sourcePointer ok

      // Free memory allocated for the source buffer.
      //
      if( cvxBufferFree( cnvrtData.sourcePointer ) )
      {
         returnValue = _ERROR_MEMORY_DEALLOC;
      }

   } // end if cvxBufferAlloc for dmaBuffer[] ok

   // Free allocated buffer memory used for the playback buffers.
   //
   for( i=(bufferCount-1); ( (i != -1) ); i-- )
   {
      if( cvxBufferFree( dmaBuffer[i]) )
      {
         returnValue = _ERROR_MEMORY_DEALLOC;
      }
   }

   return( returnValue );
}

