
{===========================================================================}
{ Konzept        : DATA BECKERs Sound Blaster Superbuch                     }
{ Prog. WAVE2VOC : Ein Konvertierungsprogramm, das WAVE-Dateien in VOC-Da-  }
{                  teien umwandelt.                                         }
{===========================================================================}
{ Autor          : Arthur Burda                                             }
{ Dateiname      : WAVE2VOC.PAS                                             }
{ entwickelt am  : 12.07.1993                                               }
{ letztes Update : 01.09.1993                                               }
{ Version        : 1.03                                                     }
{ Compiler       : Turbo Pascal 6.0 und hher                               }
{===========================================================================}

PROGRAM WAVE2VOC;

{$D-}                                        { keine Debugger-Informationen }
{$F-}                                        { FAR-Aufrufe sind nicht ntig }
{$G+}                                                   { 286-Code erzeugen }
{$I-}                                                   { keine I/O-Prfung }
{$R-}                                               { keine Bereichsprfung }
{$S-}                                                  { keine Stackprfung }
{$X+}                    { Behandlung von Funktionen wie Prozeduren mglich }

{$M 16384,65536,655360}         { 16 KB Stack, min. 64 KB Heap, max. 640 KB }

USES CRT, DOS;                                { CRT- und DOS-Unit einbinden }

TYPE

  IDString = ARRAY[0..$13] OF Char;                { Identifikations-String }
  String4  = ARRAY[0..3] OF Char;                    { String aus 4 Zeichen }

CONST

  VOC_IDStr : IDString = ('C', 'R', 'E', 'A', 'T', 'I', 'V', 'E', #32, 'V',
                          'O', 'I', 'C', 'E', #32, 'F', 'I', 'L', 'E', #26);

TYPE

  {=========================================================================}
  { TWAVERec: Struktur des WAVE-Header                                      }
  {=========================================================================}

  TWAVERec = RECORD
    RIFF      : String4;                                           { "RIFF" }
    FileLen   : LongInt;                         { Lnge der WAVE-Datei - 8 }
    WAVE      : String4;                                           { "WAVE" }
    fmt_      : String4;                                           { "fmt_" }
    BlocksLen : LongInt; { Lngenangabe bestehend aus Bytes 10h 00h 00h 00h }

    { Headerblock: wave-format }

    FormatType  : Word;                         { Art des benutzten Formats }
    Channels    : Word;   { Anzahl der benutzten Kanle (1: Mono, 2: Stereo }
    SampleRate  : LongInt;                                  { Sampling-Rate }
    BytesProSec : LongInt;   { durschnittl. Anz. Bytes pro Sek. bei Ausgabe }
    SampleType  : Word;                           { Anzahl Bytes pro Sample }

    { Headerblock: format-specific }

    SampleBits : Word;     { Anzahl Bits pro Sample (8: 8 Bits, 16: 16 Bits }

    { Danach folgt abhngig von der Lnge der beiden Headerblcke }
    { der String "data" und anschlieend die Lnge der Daten (ein }
    { Doppelwort).                                                }

  END;

  {=========================================================================}
  { TVOCRec: Struktur des VOC-Header                                        }
  {=========================================================================}

  TVOCRec = RECORD
    IDStr                : IDString;                        { Datei-Kennung }
    DataOffs, Ver, Check : Word;   { Datenblock-Offset, Version, Check-Code }
  END;

VAR
  WAVEName : String;                                  { Name der WAVE-Datei }
  WAVEFile : File;                                     { WAVE-Dateivariable }
  VOCName  : String;                                   { Name der VOC-Datei }
  VOCFile  : File;                                      { VOC-Dateivariable }

{===========================================================================}
{ Prozedur ShowHelp: Hilfe zum Programm auf dem Bildschirm anzeigen.        }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE ShowHelp;

BEGIN
  WriteLn('Konvertiert eine WAVE-Datei in eine VOC-Datei.');
  WriteLn;
  WriteLn('Syntax: WAVE2VOC [Ausgangsdatei].WAV [Zieldatei].VOC');
END;

{===========================================================================}
{ Funktion UpperString: Wandelt einen String beliebiger Lnge in Gro-      }
{                       schreibung um und liefert ihn an den Aufrufer der   }
{                       Funktion zurck.                                    }
{===========================================================================}
{ Eingabe: S = String, der in Groschreibung umgewandelt werden soll        }
{ Ausgabe: String in Groschreibung                                         }
{---------------------------------------------------------------------------}

FUNCTION UpperString(S : String) : String;

VAR
  Count : Word;                                                { ein Zhler }
  Upper : String;                  { in Groschreibung umgewandelter String }

BEGIN
  UpperString := '';
  IF S <> '' THEN                               { Ist S kein leerer String? }
    BEGIN                                          { nein, S ist nicht leer }
      Upper := '';
      FOR Count := 1 TO Length(S) DO                     { String umwandeln }
        Upper := Upper+UpCase(S[Count]);
      UpperString := Upper;                      { neuen String zurckgeben }
    END;
END;

{===========================================================================}
{ Prozedur Error: Zeigt einen Fehler an und beendet das Programm mit einem  }
{                 Halt-Befehl.                                              }
{===========================================================================}
{ Eingabe: Text = Fehlermeldung                                             }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Error(Text : String);

BEGIN
  WriteLn;
  WriteLn(Text);                                   { Fehlermeldung ausgeben }
  Halt;                                                  { Programm beenden }
END;

{===========================================================================}
{ Prozedur Convert: Kmmert sich um die Konvertierung WAVE -> VOC.          }
{===========================================================================}
{ Eingabe: keine                                                            }
{ Ausgabe: keine                                                            }
{---------------------------------------------------------------------------}

PROCEDURE Convert;

VAR
  WAVEHeader    : TWAVERec;                                   { WAVE-Header }
  VOCHeader     : TVOCRec;                                     { VOC-Header }
  Count         : Word;                                        { ein Zhler }
  Result        : Integer;                          { Fehlerstatus-Variable }
  WAVEBlockSize : LongInt;                    { Gre der WAVE-Sample-Daten }

  {=========================================================================}
  { Prozedur ReadError: Wird beim Auftreten eines Lesefehlers aufgerufen.   }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE ReadError(FileName : String);

  BEGIN
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Erase(VOCFile);                                     { VOC-Datei lschen }
    Error('Fehler: Lesefehler bei '+FileName);     { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur WriteError: Wird beim Auftreten eines Schreibfehlers aufgeru-  }
  {                      fen.                                               }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE WriteError(FileName : String);

  BEGIN
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Erase(VOCFile);                                     { VOC-Datei lschen }
    Error('Fehler: Schreibfehler bei '+FileName);  { Fehlermeldung ausgeben }
  END;

  {=========================================================================}
  { Prozedur SeekError: Wird beim Auftreten eines Positionierungsfehlers    }
  {                     aufgerufen.                                         }
  {=========================================================================}
  { Eingabe: FileName = Name der Datei, bei der der Fehler aufgetreten ist  }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE SeekError(FileName : String);

  BEGIN
    Close(WAVEFile);                                 { WAVE-Datei schlieen }
    Close(VOCFile);                                   { VOC-Datei schlieen }
    Erase(VOCFile);                                     { VOC-Datei lschen }
    Error('Fehler: Positionierungsfehler '+        { Fehlermeldung ausgeben }
      ' bei '+FileName);
  END;

  {=========================================================================}
  { Prozedur TransferData: bertrgt die Sample-Daten von einer WAVE-Datei  }
  {                        in eine VOC-Datei.                               }
  {=========================================================================}
  { Eingabe: keine                                                          }
  { Ausgabe: keine                                                          }
  {-------------------------------------------------------------------------}

  PROCEDURE TransferData;

  CONST
    BufSize = 60*1024;                                  { Puffergre 60 KB }

  VAR
    DataBuf      : Pointer;                                   { Datenpuffer }
    TransferSize : LongInt;  { Gre der noch zu bertragenden Sample-Daten }
    B            : Byte;                                    { ein Datenbyte }

  BEGIN
    TransferSize := WAVEBlockSize;
    GetMem(DataBuf, BufSize);              { Puffer auf dem Heap einrichten }

    { solange wiederholen, bis alle Sample-Daten bertragen sind }

    REPEAT
      IF TransferSize >= BufSize THEN     { Datenblockgre >= Puffergre? }
        BEGIN
          BlockRead(WAVEFile, DataBuf^, BufSize);          { Daten einlesen }
          Result := IOResult;                        { Fehlerstatus sichern }
          IF Result <> 0 THEN                       { Liegt ein Fehler vor? }
            BEGIN                                                      { ja }
              FreeMem(DataBuf, BufSize);            { Puffer deinstallieren }
              ReadError(WAVEName);      { Fehlerbehandlungsroutine aufrufen }
            END;
          BlockWrite(VOCFile, DataBuf^, BufSize);  { Pufferinhalt speichern }
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              WriteError(VOCName);
            END;
          Dec(TransferSize, BufSize);
        END
      ELSE
        BEGIN
          BlockRead(WAVEFile, DataBuf^, TransferSize);
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              ReadError(WAVEName);
            END;
          BlockWrite(VOCFile, DataBuf^, TransferSize);
          Result := IOResult;
          IF Result <> 0 THEN
            BEGIN
              FreeMem(DataBuf, BufSize);
              WriteError(VOCName);
            END;
          TransferSize := 0;
        END;
    UNTIL TransferSize = 0;

    FreeMem(DataBuf, BufSize);                  { Puffer vom Heap entfernen }
  END;

{ Convert }

VAR
  RIFFStr    : String;                                     { "RIFF"-Kennung }
  WAVEStr    : String;                                     { "WAVE"-Kennung }
  BlockType  : Byte;                                       { Datenblock-Typ }
  BlockSize  : LongInt;                             { Gre des Datenblocks }
  TC         : Word;     { Sample-Rate in komprimierter Form (16-Bits-Wert) }
  TC_        : Byte;      { Sample-Rate in komprimierter Form (8-Bits-Wert) }
  PackMethod : Byte;                                { Komprimierungsmethode }
  Mode       : Byte;                             { Modus (Mono oder Stereo) }
  B          : Byte;                                        { ein Datenbyte }
  W          : Word;                                        { ein Datenwort }

BEGIN
  WriteLn('Konvertierung ', WAVEName, ' -> ', VOCName);
  BlockRead(WAVEFile, WAVEHeader, SizeOf(TWAVERec));    { WAVE-Header lesen }
  Result := IOResult;                              { Fehlerstatus speichern }
  IF Result <> 0 THEN                                 { Fehler aufgetreten? }
    ReadError(WAVEName);
  RIFFStr := '';
  FOR Count := 0 TO 3 DO
    RIFFStr := RIFFStr+UpCase(WAVEHeader.RIFF[Count]);
  WAVEStr := '';
  FOR Count := 0 TO 3 DO
    WAVEStr := WAVEStr+UpCase(WAVEHeader.WAVE[Count]);
  IF (RIFFStr <> 'RIFF')                              { "RIFF"-Kennung oder }
  OR (WAVEStr <> 'WAVE') THEN              { "WAVE"-Kennung nicht gefunden? }
    BEGIN                                                            { nein }
      Close(WAVEFile);                               { WAVE-Datei schlieen }
      Close(VOCFile);               { bereits angelegte VOC-Datei schlieen }
      Erase(VOCFile);                                  { und danach lschen }
      Error('Fehler: Datei '+WAVEName+' nicht im WAVE-Format');
    END;
  WriteLn;
  WriteLn('Konvertierung luft ...');

  { VOC-Header mit Daten fllen }

  WITH VOCHeader DO
    BEGIN
      IDStr := VOC_IDStr;
      DataOffs := 26;
      IF WAVEHeader.SampleBits = 16 THEN              { 16 Bits pro Sample? }
        BEGIN                                                          { ja }
          Ver := 256+20;             { VOC-Dateiversion 1.20 (neues Format) }
          Check := 4383;                                       { Check-Code }
        END
      ELSE                                        { nein, 8 Bits pro Sample }
        BEGIN
          Check := 4393;             { VOC-Dateiversion 1.10 (altes Format) }
          Ver := 256+10;
        END;
    END;

  BlockWrite(VOCFile, VOCHeader, SizeOf(TVOCRec));   { VOC-Header schreiben }
  Result := IOResult;                                { Fehlerstatus sichern }
  IF Result <> 0 THEN                       { Schreibvorgang nicht korrekt? }
    WriteError(VOCName);                                             { nein }
  IF (WAVEHeader.SampleBits = 8)                        { 8 Bits pro Sample }
  AND (WAVEHeader.Channels = 2) THEN                           { in Stereo? }
    BEGIN   { ja, zustzlicher Datenblock erforderlich (SB Pro Erweiterung) }
      BlockType := 8;                        { Blocktyp "Zustzliche Daten" }
      BlockSize := 4;                                  { Blockgre 4 Bytes }
      BlockWrite(VOCFile, BlockType, SizeOf(Byte));
      BlockWrite(VOCFile, BlockSize, 3);
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
      TC := 65536-(256000000 DIV (WAVEHeader.SampleRate*2));
      PackMethod := 0;             { Art der Komprimierung: 8 Bit ungepackt }
      Mode := 1;                                            { Modus: Stereo }
      BlockWrite(VOCFile, TC, SizeOf(Word));
      BlockWrite(VOCFile, PackMethod, SizeOf(Byte));
      BlockWrite(VOCFile, Mode, SizeOf(Byte));
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
    END;
  Seek(WAVEFile,                    { 4 Positionen weiter in der WAVE-Datei }
    FilePos(WAVEFile)+4);
  Result := IOResult;
  IF Result <> 0 THEN
    SeekError(WAVEName);
  BlockRead(WAVEFile, WAVEBlockSize,    { Gre der WAVE-Sample-Daten lesen }
    SizeOf(LongInt));
  Result := IOResult;
  IF Result <> 0 THEN
    ReadError(WAVEName);

  { Falls ein 16-Bits-Sample vorliegt, Blocktyp 9 ("Neue Sample-Daten }
  { (neues VOC-Format)"), falls ein 8-Bits-Sample, Blocktyp 1 ("Neue  }
  { Sample-Daten") erzeugen und speichern                             }

  IF WAVEHeader.SampleBits = 16 THEN
    BEGIN
      BlockType := 9;
      BlockSize := WAVEBlockSize+12;
      BlockWrite(VOCFile, BlockType, SizeOf(Byte));
      BlockWrite(VOCFile, BlockSize, 3);
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
      TC := WAVEHeader.SampleRate;
      PackMethod := 0;             { Art der Komprimierung: 8 Bit ungepackt }
      B := 0;
      BlockWrite(VOCFile, TC, SizeOf(Word));
      BlockWrite(VOCFile, PackMethod, SizeOf(Byte));
      BlockWrite(VOCFile, B, SizeOf(Byte));
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
      W := 0;
      BlockWrite(VOCFile, WAVEHeader.SampleBits, SizeOf(Byte));
      BlockWrite(VOCFile, WAVEHeader.Channels, SizeOf(Byte));
      B := 4;
      BlockWrite(VOCFile, B, SizeOf(Byte));
      B := 0;
      BlockWrite(VOCFile, B, SizeOf(Byte));
      BlockWrite(VOCFile, W, SizeOf(Word));
      BlockWrite(VOCFile, W, SizeOf(Word));
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
    END
  ELSE
    BEGIN
      BlockType := 1;
      BlockSize := WAVEBlockSize+2;
      BlockWrite(VOCFile, BlockType, SizeOf(Byte));
      BlockWrite(VOCFile, BlockSize, 3);
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
      TC_ := 256-(1000000 DIV WAVEHeader.SampleRate);
      PackMethod := 0;
      BlockWrite(VOCFile, TC_, SizeOf(Byte));
      BlockWrite(VOCFile, PackMethod, SizeOf(Byte));
      Result := IOResult;
      IF Result <> 0 THEN
        WriteError(VOCName);
    END;

  TransferData;                            { Konvertierung der Sample-Daten }

  { Zum Schlu Blocktyp 0 ("Blockende") hinzufgen }

  BlockType := 0;
  BlockWrite(VOCFile, BlockType, SizeOf(Byte));
  Result := IOResult;
  IF Result <> 0 THEN
    WriteError(VOCName);

  { Dateien schlieen und Konvertierung beenden }

  Close(WAVEFile);                                   { WAVE-Datei schlieen }
  Close(VOCFile);                                     { VOC-Datei schlieen }
  WriteLn;
  WriteLn('Konvertierung beendet.');
END;

{---------------------------------------------------------------------------}
{ Hauptprogramm                                                             }
{---------------------------------------------------------------------------}

VAR
  Result : Integer;                                 { Fehlerstatus-Variable }
  Dir    : DirStr;                              { Verzeichnis der VOC-Datei }
  Name   : NameStr;                   { Name der VOC-Datei ohne Erweiterung }
  Ext    : ExtStr;                                       { Dateierweiterung }

BEGIN
  TextColor(LightGray);                                { Textfarbe hellgrau }
  WriteLn;
  WriteLn('VOC2WAVE  *  Version 1.03  *  (c) 1993 by Arthur Burda');
  WriteLn(''+
    '');
  IF ParamCount = 0 THEN               { Kommandozeilenparameter angegeben? }
    Error('Fehler: Keine Parameter angegeben.')         { nein, also Fehler }
  ELSE
    IF ParamCount = 1 THEN                     { nur 1 Parameter angegeben? }
      IF ParamStr(1) = '/?' THEN  { Ja. Ist der Parameter ein Fragezeichen? }
        ShowHelp                                       { ja, Hilfe anzeigen }
      ELSE                                                           { nein }
        Error('Fehler: Zu wenig Parameter')                 { Fehlermeldung }
    ELSE
      BEGIN
        WAVEName := UpperString(ParamStr(1));    { WAVE-Dateinamen auslesen }
        FSplit(WAVEName, Dir, Name, Ext);        { WAVE-Dateinamen zerlegen }
        IF Ext = '' THEN
          WAVEName := WAVEName+'.WAV';
        VOCName := UpperString(ParamStr(2));      { VOC-Dateinamen auslesen }
        FSplit(VOCName, Dir, Name, Ext);          { VOC-Dateinamen zerlegen }
        IF Ext = '' THEN
          VOCName := VOCName+'.VOC';
        Assign(WAVEFile, WAVEName);       { WAVE-Datei mit Namen verknpfen }
        Reset(WAVEFile, 1);                             { WAVE-Datei ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result <> 0 THEN                           { Fehler aufgetreten? }
          Error('Fehler: Datei '+WAVEName+     { ja, Fehlermeldung ausgeben }
            ' konnte nicht geffnet werden.');
        Assign(VOCFile, VOCName);          { VOC-Datei mit Namen verknpfen }
        Rewrite(VOCFile, 1);               { VOC-Datei zum Schreiben ffnen }
        Result := IOResult;                         { Fehlerstatus abfragen }
        IF Result <> 0 THEN                           { Fehler aufgetreten? }
          BEGIN                                                        { ja }
            Close(WAVEFile);       { bereits geffnete WAVE-Datei schlieen }
            Error('Fehler: Datei '+VOCName+        { Fehlermeldung ausgeben }
              ' konnte nicht geffnet werden.');
          END;
        Convert;                                    { Konvertierung starten }
      END;
END.
